get_data <- function(path_to_csv) {
  human_data <- read.csv(path_to_csv,numerals ="no.loss",stringsAsFactors=FALSE)
  # It's convenient to have a single identifier for each individual sequence; each participant sees
  # each sequence only once, so we can concatenate them to get a global sequence ID.
  human_data <- human_data %>% mutate(trial_id=paste(ID,Sequence,sep="_"))
  # Guarantee that within a given trial the judgments are ordered for easier
  # downstream model eval
  human_data <- human_data[order(human_data$trial_id,human_data$Number_in_seq),]
}

# Converts strings to sequences of integers
proc_seq_helper <- function(s,f) {
  len <- nchar(s)
  cs <- rep(-1,len)
  for(i in 1:len) {
    cs[i] <- f(strtoi(substr(s,i,i)))
  }
  return(cs)
}

# Returns a sequence of integers denoting whether a cause occurs (1) or
# or does not (0) in the corresponding index of the sequence.
cause_from_seq <- function(s) {
  proc_seq_helper(s,function(x) {(x+1) %% 2})
}

# As cause_from_seq, but for the effect variable.
eff_from_seq <- function(s) {proc_seq_helper(s,function(x) {
  if(x==0 || x==3) 1 else 0})
}

# Plots participant judgments.
plot_seq <- function(seq,sdf) {
  p<-ggplot(data=sdf, aes(x=Number_in_seq, y=Correlation_estimate)) +
    ggtitle(seq) + 
    theme_minimal() +
    theme(plot.title = element_text(hjust = 0.5)) +
    geom_bar(stat="identity") + ylim(-100,100) + geom_line(data=sdf,aes(x=Number_in_seq,y=Correlation_estimate))
  print(p)
}

# Plotting bit
plot_model_vs_people <- function(title_str,human_df,model_df) {
  p<-ggplot() + 
    theme_minimal() +
    # TODO/TIA: This paste appears to be necessary. Do you know why?
    ggtitle(paste(title_str)) +
    theme(plot.title = element_text(hjust = 0.5)) +
    geom_line(data=human_df,aes(x=Number_in_seq,y=Correlation_estimate)) + ylim(-100,100) +
    geom_point(data=model_df,aes(x=i,y=model_preds))
    print(p)
}

run_test <- function(test_results,test_name,test_bool,failure_message) {
  test_results <- rbind(test_results,data.frame(test_name=test_name,passed=test_bool,
                                  message=if(test_bool) {"SUCCESS"} else {failure_message}))
  test_results
}

check_automarker_answer_format <- function(var_name,
                                           allowed_values,
                                           allow_blank = TRUE,
                                           case_insensitive = TRUE,
                                           envir = if (isTRUE(getOption("knitr.in.progress"))) knitr::knit_global() else .GlobalEnv) {
  exists_flag <- exists(var_name, envir = envir, inherits = FALSE)
  is_char_flag <- FALSE
  all_allowed_flag <- FALSE
  invalid <- character(0)
  msg <- ""
  
  if (exists_flag) {
    obj <- get(var_name, envir = envir, inherits = FALSE)
    is_char_flag <- is.character(obj)
    
    if (is_char_flag) {
      obj_trim <- trimws(obj)
      
      if (allow_blank && length(obj_trim) >= 1 && all(obj_trim == "")) {
        msg <- sprintf("WARNING: %s is blank; treated as a non-answer.", var_name)
        return(data.frame(variable = var_name, exists = TRUE, is_character = TRUE,
                          all_allowed = TRUE, invalid_values = "", message = msg))
      }
      
      if (any(obj_trim == "")) {
        msg <- sprintf("WARNING: %s contains blank entries mixed with answers.", var_name)
        return(data.frame(variable = var_name, exists = TRUE, is_character = TRUE,
                          all_allowed = FALSE, invalid_values = "", message = msg))
      }
      
      obj_norm <- if (case_insensitive) toupper(obj_trim) else obj_trim
      allowed_norm <- if (case_insensitive) toupper(allowed_values) else allowed_values
      invalid <- obj_trim[!(obj_norm %in% allowed_norm)]
      
      if (length(invalid) > 0) {
        msg <- sprintf('!ERROR!: %s contains invalid values: %s',
                       var_name, paste(sprintf('"%s"', unique(invalid)), collapse = ", "))
        return(data.frame(variable = var_name, exists = TRUE, is_character = TRUE,
                          all_allowed = FALSE, invalid_values = paste(invalid, collapse = ", "),
                          message = msg))
      }
      
      msg <- sprintf("Good. %s answer conforms to the answer format", var_name)
      return(data.frame(variable = var_name, exists = TRUE, is_character = TRUE,
                        all_allowed = TRUE, invalid_values = "", message = msg))
    }
  }
  
  if (!exists_flag)
    msg <- sprintf("!ERROR!. %s does not exist in the workspace. Assign a value to this answer.", var_name)
  else if (!is_char_flag)
    msg <- sprintf("!ERROR!. %s exists but is not a character vector. Assign a correct value to this answer.", var_name)
  
  data.frame(variable = var_name, exists = exists_flag, is_character = is_char_flag,
             all_allowed = all_allowed_flag,
             invalid_values = paste(invalid, collapse = ", "),
             message = msg)
}