(* Word64.sml
 *
 * Peter Bertelsen
 * October 1997
 *)

open Word8Vector
exception Word64Overflow of string

datatype word = WORD of vector

(* for WORD bs, bs[0] is the most significant byte *)

local
    open Word
    infix >> << orb
in
    fun fromWord w =
	let fun h 0w0 w' l = l
	      | h i   w' l =
	        h (i-0w1) (w' >> 0w8) (Word8.fromLargeWord w' :: l)
	in
	    WORD (fromList(h 0w8 w []))
	end

    fun toWord' (_, b, w) = (w << 0w8) orb Word8.toLargeWord b

    fun toWord (WORD bs) = foldli toWord' 0w0 (bs, 4, NONE)
end

fun tohex2 n = let
    val s = Word8.toString n
in if n < 0w16 then "0" ^ s else s end

fun toHexString (WORD l) = "0x" ^ foldl (fn (x,s) => s ^ (tohex2 x)) "" l

fun fromInt k =
    let fun h 0 k' l = l
	  | h i k' l = h (i-1) (k' div 0x100) (Word8.fromInt k' :: l)
    in
	WORD (fromList(h 8 k []))
    end

fun toInt' k0 bs =
    let fun h (_, b, k) = k * 0x100 + Word8.toInt b
    in
	foldli h k0 (bs, 1, NONE)
    end handle Overflow => raise Word64Overflow (toHexString (WORD bs))

fun toInt (WORD bs) = toInt' (Word8.toInt(sub(bs, 0))) bs

fun toIntX (WORD bs) = toInt' (Word8.toIntX(sub(bs, 0))) bs

fun toString w = (Int.toString (toInt w))
    handle Word64Overflow s => s

fun fromBytes bs =
    if length bs = 8 then SOME (WORD bs)
    else NONE

fun toBytes (WORD bs) = bs

fun compare' x y i =
    let fun h 0 i' = Word8.compare(sub(x, i'), sub(y, i'))
	  | h k i' =
	    (case Word8.compare(sub(x, i'), sub(y, i')) of
		 EQUAL => h (k-1) (i'+1)
	       | res   => res)
    in
	h (7-i) i
    end

fun compare (WORD x, WORD y) = compare' x y 0

fun compareX (WORD x, WORD y) =
    let val x0 = sub(x, 0)
	val y0 = sub(y, 0)
    in
	if (Word8.andb(x0, 0wx80) = Word8.andb(y0, 0wx80)) then
	    (case Word8.compare(x0, y0) of
		 EQUAL => compare' x y 1
	       | res   => res)
	else Word8.compare(y0, x0)  (* sign(x) <> sign(y) *)
    end

fun emit out (WORD bs) = app out bs

fun scan src =
    (let fun h 0 l = l
	   | h i l = h (i-1) (valOf(src()) :: l)

	 (* NOTE: src is imperative; we must use its results in the
	  * correct order: #[byte 0, byte 1, ..., byte 7]. *)

	 val bs = fromList(rev(h 8 []))
     in
	 SOME (WORD bs)
     end) handle Option => NONE

