(* Word32.sml
 *
 * Peter Bertelsen
 * October 1997
 *)

exception Word32Overflow of string

open Word8Vector

datatype word = WORD of vector

fun toVec (WORD v) = v

fun fromVec v = WORD v

(* for WORD bs, bs[0] is the most significant byte *)

fun tohex2 n = let
    val s = Word8.toString n
in if n < 0w16 then "0" ^ s else s end

fun toHexString (WORD l) = "0x" ^ foldl (fn (x,s) => s ^ (tohex2 x)) "" l

local
    open Word
    infix >> << orb
in
    fun fromWord w =
	let fun h 0w0 w' l = l
	      | h i   w' l =
	        h (i-0w1) (w' >> 0w8) (Word8.fromLargeWord w' :: l)
	in
	    WORD (fromList(h 0w4 w []))
	end
    fun toWord' (b, w) = (w << 0w8) orb Word8.toLargeWord b

    fun toWord (WORD bs) = foldl toWord' 0w0 bs

    fun toList (WORD bs) = foldr (op ::) [] bs
    fun make ns = WORD (Word8Vector.fromList (List.map Word8.fromInt ns))
    (* To let us look at things *)
end

fun fromInt k =
    let fun h 0 k' l = l
	  | h i k' l = h (i-1) (k' div 0x100) (Word8.fromInt k' :: l)
    in
	WORD (fromList(h 4 k []))
    end

fun toInt' k0 bs =
    let fun h (_, b, k) = k * 0x100 + Word8.toInt b
    in
	foldli h k0 (bs, 1, NONE)
    end handle Overflow => raise Word32Overflow (toHexString (WORD bs))

fun toInt (WORD bs) =
    let val b0 = sub(bs, 0)
    in
	toInt' (Word8.toInt b0) bs
    end

fun toIntX (WORD bs) = (* Does this work correctly for eg [0xff, 4, 4, 4]? *)
    let val b0 = sub(bs, 0)
    in
	toInt' (Word8.toIntX b0) bs
    end

fun toString w = (Int.toString (toInt w))
    handle Word32Overflow s => s

fun fromBytes bs =
    if length bs = 4 then SOME (WORD bs)
    else NONE

fun toBytes (WORD bs) = bs

fun compare' x y i =
    let fun h 0 i' = Word8.compare(sub(x, i'), sub(y, i'))
	  | h k i' =
	    (case Word8.compare(sub(x, i'), sub(y, i')) of
		 EQUAL => h (k-1) (i'+1)
	       | res   => res)
    in
	h (3-i) i
    end

fun compare (WORD x, WORD y) = compare' x y 0

fun compareX (WORD x, WORD y) =
    let val x0 = sub(x, 0)
	val y0 = sub(y, 0)
    in
	if (Word8.andb(x0, 0wx80) = Word8.andb(y0, 0wx80)) then
	    (case Word8.compare(x0, y0) of
		 EQUAL => compare' x y 1
	       | res   => res)
	else Word8.compare(y0, x0)  (* sign(x) <> sign(y) *)
    end

fun emit out (WORD bs) = app out bs

fun scan src =
    (let fun h 0 l = l
	   | h i l = h (i-1) (valOf(src()) :: l)

	 (* NOTE: src is imperative; we must use its results in the
	  * correct order: #[byte 0, byte 1, byte 2, and byte 3]. *)

	 val bs = fromList(rev(h 4 []))
     in
	 SOME (WORD bs)
     end) handle Option => NONE



(* Arithmetic: not needed at the moment *)

local
fun iter oper (WORD a) (WORD b) =
    let
	fun h n prev (acc:int list) =
	    if n < 0 then (fromList (List.map Word8.fromInt (acc)), prev)
	    else let val p = Word8.toInt (sub (a,n))
		     val q = Word8.toInt (sub (b,n))
		     val i = oper (p,q) + prev div 256
		 in
		     h (n-1) i (i::acc)
		 end
    in
	h 3 0 []
    end


fun subtract (u, v) = (* Raises Word32Overflow if u < v *)
    let
	val (w, c) = iter (op-) u v
    in if c < 0 then raise Word32Overflow (toString u ^ " - " ^ toString v)
       else WORD w
    end

fun add (u,v) = (* Raises Word32Overflow if u+v > 0xffffffff *)
    let
	val (w, c) = iter (op+) u v
    in if c > 255 then raise Word32Overflow (toHexString u ^ " + " ^ toHexString v)
       else WORD w
    end

in

fun x+y = add (x,y)
fun x-y = subtract(x,y)

end
