(* Jvmtype.sml
 *
 * Peter Bertelsen
 * December 1997
 *)

exception InvalidType of string
exception InvalidTypeDescriptor of string
fun typeError s = raise InvalidType("Jvmtype." ^ s)

datatype jclass = CLASS of {pkgs: string list, name: string}

fun class (c as {pkgs, name}) =
    let fun h (#".", _) = false
	  | h (#"/", _) = false
	  | h (#"[", _) = false
	  | h (#"]", _) = false
	  | h (_,   ok) = ok

	fun validName s = Substring.foldl h true (Substring.all s)
    in
	if List.all validName pkgs andalso validName name then CLASS c
	else CLASS c (*typeError ("class: invalid class identifier " ^ (qualName (CLASS c)))*)
    end

and className (CLASS {name, ...}) = name

and packages (CLASS {pkgs, ...}) = pkgs

and revAppend s cs = CharVector.foldl (op ::) cs s

and revAppendDelim (s, cs) = #"/" :: (revAppend s cs)
and revQualName cs {pkgs, name} =
        revAppend name (List.foldl revAppendDelim cs pkgs)
and qualName (CLASS c) = String.implode(rev (revQualName [] c))

and qualNameToClass name =
    if name ="" then class{pkgs = [], name = ""} else
    let val l = String.tokens (fn c => (c = #"/" orelse c = #".")) name
	val n = List.length l
	val pkgs = List.take (l,n-1)
	val name' = List.nth (l,n-1)
    in
        class {pkgs = pkgs, name = name'}
    end


datatype jtype =
    Tboolean
  | Tchar
  | Tfloat
  | Tdouble
  | Tbyte
  | Tshort
  | Tint
  | Tlong
  | Tarray of jtype
  | Tclass of jclass

fun isSimple t =
    (case t of
	 Tboolean => true
       | Tchar    => true
       | Tfloat   => true
       | Tdouble  => true
       | Tbyte    => true
       | Tshort   => true
       | Tint     => true
       | Tlong    => true
       | _        => false)

type method_sig = jtype list * jtype option

fun arrayOf' (0, t) = t
  | arrayOf' (k, t) = arrayOf'(k-1, Tarray t)
fun arrayOf  (n, t) = if n >= 0 then arrayOf'(n, t)
		      else raise Domain

fun arrayDim' (Tarray t, res) = arrayDim'(t, res + 1)
  | arrayDim' (_,        res) = res
fun arrayDim (Tarray t) = arrayDim'(t, 1)
  | arrayDim   _        = raise Domain

fun arrayBase' (Tarray t) = arrayBase' t
  | arrayBase' t = t
fun arrayBase  (Tarray t) = arrayBase' t
  | arrayBase   _         = raise Domain

fun width t =
    (case t of
	 Tboolean => 1
       | Tchar    => 1       | Tfloat   => 1
       | Tdouble  => 2
       | Tbyte    => 1
       | Tshort   => 1
       | Tint     => 1
       | Tlong    => 2
       | Tarray _ => 1
       | Tclass _ => 1)

fun revDesc (t, cs) =
    (case t of
	 Tboolean     => #"Z" :: cs
       | Tchar        => #"C" :: cs
       | Tfloat       => #"F" :: cs
       | Tdouble      => #"D" :: cs
       | Tbyte        => #"B" :: cs
       | Tshort       => #"S" :: cs
       | Tint         => #"I" :: cs
       | Tlong        => #"J" :: cs
       | Tarray t'    => revDesc(t', #"[" :: cs)
       | Tclass (CLASS c) => #";" ::(revQualName (#"L" :: cs) c))

fun typeDesc t =
    let val cs = revDesc(t, [])
    in
	String.implode(rev cs)
    end

fun methodDesc (argTs, retT) =
    let val cs   = foldl revDesc [#"("] argTs
	val cs'  = #")" :: cs
	val cs'' = (case retT of
			NONE   => #"V" :: cs'
		      | SOME t => revDesc(t, cs'))
    in
	String.implode(rev cs'')
    end

local
    open Substring

    val splitQualName = splitl (fn c => c <> #"/" andalso c <> #";")

    fun scanQualName ids ss =
	let val (id, ss') = splitQualName ss
	    val id'  = string id
	    val ss'' = triml 1 ss'
	in
	    if isEmpty ss' then
		typeError "scanQualName: invalid qualified name"
	    else
		if sub(ss', 0) = #";" then (rev ids, id', ss'')
		else scanQualName (id' :: ids) ss''
	end

    fun scan' dim ss =
	let val ss' = triml 1 ss
	in
	    (case sub(ss, 0) of
		 #"Z" => (dim, Tboolean, ss')
	   | #"C" => (dim, Tchar,   ss')
	   | #"F" => (dim, Tfloat,  ss')
	   | #"D" => (dim, Tdouble, ss')
	   | #"B" => (dim, Tbyte,   ss')
	   | #"S" => (dim, Tshort,  ss')
	   | #"I" => (dim, Tint,    ss')
	   | #"J" => (dim, Tlong,   ss')
	   | #"[" => scan' (dim + 1) ss' (* NW: was chopping char off twice *)
	   | #"L" => let val (pkgs, name, ss'') = scanQualName [] ss'
			 val class = CLASS {pkgs = pkgs, name = name}
		     in
			 (dim, Tclass class, ss'')
		     end
	   | _ => typeError ("scan: invalid type descriptor: ..." ^ (string ss)))
	end

    fun scan ss =
	let val (dim, t, ss') = scan' 0 ss
	in
	    (arrayOf'(dim, t), ss')
	end

    fun scanArgs (ss, ts) =
	if sub(ss, 0) = #")" then
	    (rev ts, triml 1 ss)
	else
	    let val (t, ss')  = scan ss
	    in
		scanArgs(ss', t :: ts)
	    end
in
    fun scanTypeDesc s =
	let val (t, rest) = scan(all s)
	in
	     if isEmpty rest then SOME t
	     else NONE
	end

    fun scanMethodDesc s =
	let val ss = all s
	in
	     if sub(ss, 0) = #"(" then
	         let val (args, ss') = scanArgs(triml 1 ss, [])
		 in
		     if sub(ss', 0) = #"V" then SOME(args, NONE)
		     else
			 let val (ret, ss'') = scan ss'
			 in
			     if isEmpty ss'' then SOME(args, SOME ret)
			     else NONE (* NW: error? *)
			 end
		 end
	     else NONE
	end



end


fun parseType_aux charlist =  (* also used in parseClass_ref and while processing anewarray instruction *)
    let val h = hd charlist
	val t = tl charlist
	fun scan (#";"::cs,li) = (cs, li)
	  | scan (c::cs,li)    = scan (cs,c :: li)
	  | scan ([],li)       = ([],li)
    in  case h of #"B" => ( Tbyte, t )
		| #"C" => ( Tchar, t )
		| #"D" => ( Tdouble, t )
		| #"F" => ( Tfloat, t )
		| #"I" => ( Tint, t )
		| #"J" => ( Tlong, t )
		| #"S" => ( Tshort, t )
		| #"Z" => ( Tboolean, t )
		| #"L" => ( fn (junk, result) =>
			       (Tclass (class {pkgs=[],
							       name=implode (rev result)}) , junk)
			  )    (scan (t,[]))
		| #"[" => ( fn (result, junk) => ( Tarray result, junk )
			  )    (parseType_aux t)
		| t    => raise InvalidTypeDescriptor (implode charlist)
    end


fun parseType s =
    let val (result,junk) = (parseType_aux o explode) s
    in  if junk = []
	then result
	else raise Match (* raise InvalidTypeDescriptor (Substring.string s)*)
    end

fun parseType_option charlist = (* only used by next fn *)
    let val h = hd charlist
	val t = tl charlist
	fun scan (#";"::cs,li) = (cs, li)
	  | scan (c::cs,li)    = scan (cs,c :: li)
	  | scan ([],li)       = ([],li)
    in  case h of #"B" => ( SOME Tbyte, t )
		| #"C" => ( SOME Tchar, t )
		| #"D" => ( SOME Tdouble, t )
		| #"F" => ( SOME Tfloat, t )
		| #"I" => ( SOME Tint, t )
		| #"J" => ( SOME Tlong, t )
		| #"S" => ( SOME Tshort, t )
		| #"Z" => ( SOME Tboolean, t )
		| #"L" => ( fn (junk, result) =>
	                       ( SOME ( Tclass (class { pkgs = []
								      , name = implode (rev result) } ) )
			       , junk )
			  ) (scan (t,[]))
		| #"[" => ( fn (result, junk) => ( SOME (Tarray result), junk )
			  )    (parseType_aux t)
		| #"V" => ( NONE, t )
		| _    => raise InvalidTypeDescriptor (implode charlist)
    end

fun parseTypes s =
    let val clist = tl (explode s)
	fun somefunc (res,#")"::cs)    = (res,cs)
	  | somefunc (res,c::cs)       = ( fn (result, junk) => somefunc (result::res,junk)
					 ) (parseType_aux (c::cs))
	  | somefunc _                 = raise InvalidTypeDescriptor s
	val (result_list, rest) = somefunc ([],clist)
    in  ( rev result_list
	, (#1 o parseType_option) rest )
    end


