(* Int32.sml
 *
 * Peter Bertelsen
 * November 1997
 *)

datatype int = INT of Word32.word

exception Int32Overflow of string


local
    val min = Word8Vector.fromList [0wx80, 0w0, 0w0, 0w0]
    val max = Word8Vector.fromList [0wx7f, 0wxff, 0wxff, 0wxff]
in
    val minInt = SOME (INT (valOf(Word32.fromBytes min)))
    val maxInt = SOME (INT (valOf(Word32.fromBytes max)))
end

fun fromInt k = INT (Word32.fromInt k)

fun toInt (INT n) = Word32.toIntX n
    handle Word32.Word32Overflow s => raise Int32Overflow s

fun toString w = (Int.toString (toInt w))
    handle Int32Overflow s => s

fun fromBytes bs =
    (case Word32.fromBytes bs of
	 NONE   => NONE
       | SOME n => SOME (INT n))

fun toBytes (INT n) = Word32.toBytes n

fun compare (INT x, INT y) = Word32.compareX(x, y)

fun emit out (INT n) = Word32.emit out n

fun scan src =
    (case Word32.scan src of
	 NONE   => NONE
       | SOME n => SOME (INT n))

(* Arithmetic *)
open Word8Vector
local
    fun sign b = Word8.andb(0wx80, sub(b,0))

    fun add (u,v) =
	let
	    val a = toBytes u
	    val b = toBytes v
	    fun h n prev acc =
		if n < 0 then (fromList (List.map Word8.fromInt acc))
		else let val p = Word8.toInt (sub (a,n))
			 val q = Word8.toInt (sub (b,n))
			 val i = p+q + prev div 256
		     in
			 h (n-1) i (i::acc)
		     end
	    val r = h 3 0 []
	    val () = let val s = sign a
		     in if sign b <> s then () (* No overflow *)
			else if sign r = s then ()
			else raise Overflow
		     end
	in
	    valOf (fromBytes r)
	end


    fun subtract (u,v) =
	let
	    val a = toBytes u
	    val b = toBytes v
	    fun h n prev acc =
		if n < 0 then (fromList (List.map Word8.fromInt acc))
		else let val p = Word8.toInt (sub (a,n))
			 val q = Word8.toInt (Word8.notb (sub (b,n)))
			 val i = p+q + prev div 256
		     in
			 h (n-1) i (i::acc)
		     end
	    val r = h 3 256 []
	    val () = let val s = sign a
		     in if sign b = s then () (* No overflow *)
			else if sign r = s then ()
			else raise Overflow
		     end
	in
	    valOf (fromBytes r)
	end


in

fun u + v = add(u,v)      handle Overflow => raise Int32Overflow (toString u ^ " + " ^ toString v)
fun u - v = subtract(u,v) handle Overflow => raise Int32Overflow (toString u ^ " - " ^ toString v)

end

(* Alternatively ...

    fun negate u =  (* any problems with negate for boundary cases? *)
	let
	    val u' = valOf(fromBytes (map Word8.notb (toBytes u)))
	in
	    add(u', fromInt 1)  (* ie, (not u)+1 *)
	end

    fun sub2(u,v) = add(u, negate v)
*)
