(* Emitcode.sml
 *
 * Peter Bertelsen
 * December 1997
 *)

open Bytecode Jvmtype

(* NOTE: it is assumed that Constpool.indexValue and Localvar.toInt
 * return integers in the range [0, 0xffff].
 *)

exception InvalidCode of string

fun codeError s = raise InvalidCode("Emitcode." ^ s)

fun bug s = raise InternalError("Emitcode." ^ s)

type address = int

datatype placeholder =
    OFS16 of {addr: address, base: address}
  | OFS32 of {addr: address, base: address}

datatype target =
    PENDING of placeholder list ref
  | RESOLVED of address

fun typeKey t =
    case t of
	Tboolean => 0w4 : Word8.word
      | Tchar    => 0w5
      | Tfloat   => 0w6
      | Tdouble  => 0w7
      | Tbyte    => 0w8
      | Tshort   => 0w9
      | Tint     => 0w10
      | Tlong    => 0w11
      | _ => bug "typeKey: not a simple JVM type"

val exnFind = InvalidCode "Emitcode.emit: label not found"

val wide = 0w196 : Word8.word   (* opcode for JVM wide instruction prefix *)

fun isU1    k =      0 <= k andalso k <= 0xff
fun isU2    k =      0 <= k andalso k <= 0xffff
fun isByte  k =   ~128 <= k andalso k <= 127
fun isShort k = ~32768 <= k andalso k <= 32767

fun placeHolder16 addr = addr + 2
fun placeHolder32 addr = addr + 4

datatype method_kind = STATIC | NONSTATIC

fun argsWidth mkind (msig: method_sig) =
    let val extra =
	(case mkind of
	     STATIC    => 0
	   | NONSTATIC => 1)   (* include size of objectref (this) *)
    in
	List.foldl (fn (t, res) => width t + res) extra (#1 msig)
    end


fun emit cp returnTy code =
    let	val nextAddr = ref 0   (* the first instruction goes into code'[0] *)
	val targets  = Polyhash.mkPolyTable(1021, exnFind)
	val code'    = Word8Array.array(0xffff, 0w0)
	               (* max 0xffff bytes of code per method *)


	local
	    val shortjumps = ref (EstimateJumps.shortJumps code)
	in

	fun jumpIsShort() =
	    case !shortjumps of
		[] => bug "empty jumplist"
	      | h::t => (shortjumps := t; h)
	end

	local
	    open Constpool
	in
	    val insInt'        = insInt cp
	    val insFloat'      = insFloat cp
	    val insLong'       = insLong cp
	    val insDouble'     = insDouble cp
	    val insClass'      = insClass cp
	    val insArrayClass' = insArrayClass cp
	    val insString'     = insString cp
	    val insFieldref'   = insFieldref cp
	    val insMethodref'  = insMethodref cp
	    val insIMethodref' = insIMethodref cp
	end

	fun setU1 w addr = (Word8Array.update(code', addr, w);
			    addr + 1)

	(* NOTE: the set* functions store a value at a specific
	 * address in the code' array, and return the next address;
	 * the emit* functions also update the nextAddr counter, and
	 * return () : unit.
	 *)

	val setU1i = setU1 o Word8.fromInt

	fun emitU1 w = nextAddr:= setU1 w (!nextAddr)

	val setVec = Word8Vector.foldl (fn (w, a) => setU1 w a)

	fun setU2i k addr = setVec addr (Word16.toBytes(Word16.fromInt k))

	val setCpIndex = setU2i o Constpool.indexValue

	fun setInt k addr = setVec addr (Int32.toBytes k)

	val setInti = setInt o Int32.fromInt

	fun emitBipush k =   (* assuming isByte k holds *)
	    let val addr = setU1 0w16 (!nextAddr)
	    in
		nextAddr:= setU1i k addr
	    end

	fun emitSipush k =   (* assuming isShort k holds *)
	    let val addr = setU1 0w17 (!nextAddr)
	    in
		nextAddr:= setU2i k addr
	    end

	fun emitLdc index =
	    let val index' = Constpool.indexValue index
	    in
		if isU1 index' then
		    let val addr = setU1 0w18 (!nextAddr)   (* ldc *)
		    in
			nextAddr:= setU1i index' addr
		    end
		else
		    let val addr = setU1 0w19 (!nextAddr)   (* ldc_w *)
		    in
			nextAddr:= setU2i index' addr
		    end
	    end

	fun emitLdc2_w index =
	    let val addr = setU1 0w20 (!nextAddr)   (* ldc2_w *)
	    in
		nextAddr:= setCpIndex index addr
	    end

	fun emitIntConst i =
	    (let val i' = Int32.toInt i
	     in
		 case i' of
		     ~1 => emitU1 0w2   (* iconst_m1 *)
		   |  0 => emitU1 0w3   (* iconst_0 *)
		   |  1 => emitU1 0w4   (* iconst_1 *)
		   |  2 => emitU1 0w5   (* iconst_2 *)
		   |  3 => emitU1 0w6   (* iconst_3 *)
		   |  4 => emitU1 0w7   (* iconst_4 *)
		   |  5 => emitU1 0w8   (* iconst_5 *)
		   |  _ => if isByte i' then emitBipush i'
			   else if isShort i' then emitSipush i'
				else emitLdc(insInt' i)
	     end) handle Int32.Int32Overflow _ => emitLdc(insInt' i)

	local
	    open Real32
	 (* val bytes_0 = toBytes(fromReal 0.0) *)
	    val bytes_0 = Word8Vector.fromList[0w0, 0w0, 0w0, 0w0]
	    val bytes_1 = toBytes(fromReal 1.0)
	    val bytes_2 = toBytes(fromReal 2.0)
	in
	    fun emitFloatConst f =
		let val bytes = toBytes f
		in
		    if bytes = bytes_0 then
			emitU1 0w11   (* fconst_0 *)
		    else if bytes = bytes_1 then
			     emitU1 0w12   (* fconst_1 *)
			 else if bytes = bytes_2 then
			          emitU1 0w13   (* fconst_2 *)
			      else
				  emitLdc(insFloat' f)
		end
	end

	local
	    val zero = Int64.fromInt 0
	    val one  = Int64.fromInt 1
	in
	    fun emitLongConst l =
		if l = zero then
		    emitU1 0w9   (* lconst_0 *)
		else
		    if l = one then
			emitU1 0w10   (* lconst_1 *)
		    else
			emitLdc2_w(insLong' l)
	end

	local
	    open Real64
 	    val bytes_0 = toBytes(fromReal 0.0)
	    val bytes_1 = toBytes(fromReal 1.0)
	in
	    fun emitDoubleConst d =
		let val bytes = toBytes d
		in
		    if bytes = bytes_0 then
			emitU1 0w14   (* dconst_0 *)
		    else
			if bytes = bytes_1 then
			    emitU1 0w15   (* dconst_1 *)
			else
			    emitLdc2_w(insDouble' d)
		end
	end

	fun emitVarAccess instr j =   (* assuming isU2 j holds *)
	    let val addr = !nextAddr
	    in
		if isU1 j then
		    let val addr' = setU1 instr addr
		    in
			nextAddr:= setU1i j addr'
		    end
		else   (* emit wide instruction variant *)
		    let val addr'  = setU1 wide  addr
			val addr'' = setU1 instr addr'
		    in
			nextAddr:= setU2i j addr''
		    end
	    end

	fun immVarAccess {instr, imm_0, imm_1, imm_2, imm_3} index =
	    let val index' = Localvar.toInt index
	    in
		case index' of
		    0 => emitU1 imm_0
		  | 1 => emitU1 imm_1
		  | 2 => emitU1 imm_2
		  | 3 => emitU1 imm_3
		  | _ => emitVarAccess instr index'
	    end

	fun emitBranch16 instr lbl =
	    let val base = !nextAddr
		val addr = setU1 instr base
	    in
		case Polyhash.peek targets lbl of
		    NONE => let val phs  = [OFS16 {addr = addr, base = base}]
				val trgt = PENDING (ref phs)
			    in
				Polyhash.insert targets (lbl, trgt);
				nextAddr:= placeHolder16 addr
			    end
		  | SOME (PENDING phs) =>
			(phs:= OFS16 {addr = addr, base = base} :: (!phs);
			 nextAddr:= placeHolder16 addr)
		  | SOME (RESOLVED trgt) =>
			let val ofs = trgt - base
			in
			    if isShort ofs then nextAddr:= setU2i ofs addr
			    else codeError "emit: branch offset out of range"
			end
	    end

	fun setOffset32 base lbl addr =
	    (case Polyhash.peek targets lbl of
		 NONE => let val phs = [OFS32 {addr = addr, base = base}]
			 in
			     Polyhash.insert targets (lbl, PENDING (ref phs));
			     placeHolder32 addr
			 end
	       | SOME (PENDING phs) =>
		     (phs:= OFS32 {addr = addr, base = base} :: (!phs);
		      placeHolder32 addr)
	       | SOME (RESOLVED trgt) => setInti (trgt - base) addr)

	fun emitBranch16Or32 {instr, instr_w, lbl} =
	    let val base = !nextAddr
		val noDanger = jumpIsShort()
	    in
		case Polyhash.peek targets lbl of
		    NONE => if noDanger then
				let val addr = setU1 instr base
				    val phs  = [OFS16 {addr = addr, base = base}]
				in
				    Polyhash.insert targets (lbl, PENDING (ref phs));
				    nextAddr:= placeHolder16 addr
				end
			    else
				let val addr = setU1 instr_w base
				    val phs  = [OFS32 {addr = addr, base = base}]
				in
				    Polyhash.insert targets
				                    (lbl, PENDING (ref phs));
				    nextAddr:= placeHolder32 addr
				end
		  | SOME (PENDING phs) =>
		    if noDanger then
			let val addr = setU1 instr base
			in
			    phs:= OFS16 {addr = addr, base = base} :: (!phs);
			    nextAddr:= placeHolder16 addr
			end
		    else
			let val addr = setU1 instr_w base
			in
			    phs:= OFS32 {addr = addr, base = base} :: (!phs);
			    nextAddr:= placeHolder32 addr
			end
		  | SOME (RESOLVED trgt) =>
		        let val ofs = trgt - base
			in
			    if isShort ofs then
				let val addr = setU1 instr base
				in
				    nextAddr:= setU2i ofs addr
				end
			    else
				let val addr = setU1 instr_w base
				in
				    nextAddr:= setInti ofs addr
				end
			end
	    end

	fun backPatch lbl =
	    let val trgt = !nextAddr

		fun patch ofs =
		    (case ofs of
			 OFS16 {addr, base} =>
			     let val ofs = trgt - base
			     in
				 if isShort ofs then setU2i ofs addr
				 else codeError
				      "emit: branch offset out of range"
			     end
		       | OFS32 {addr, base} => setInti (trgt - base) addr;
		     ())
	    in
		(case Polyhash.peek targets lbl of
		     NONE => ()
		   | SOME (PENDING phs) => List.app patch (!phs)
		   | _ => codeError "emit.backPatch: label already resolved");
		Polyhash.insert targets (lbl, RESOLVED trgt)
	    end

	fun emitIinc {var, const} =
	    let val j    = Localvar.toInt var
		val addr = !nextAddr
	    in
		if isU1 j andalso isByte const then
		    let val addr'  = setU1 0w132 addr   (* iinc *)
			val addr'' = setU1i j addr'
		    in
			nextAddr:= setU1i const addr''
		    end
		else if isShort const then
		         let val addr'   = setU1 wide addr
			     val addr''  = setU1 0w132 addr'
			     val addr''' = setU2i j addr''
			 in
			     nextAddr:= setU2i const addr'''
			 end
		     else codeError "emit: iinc constant out of range"
	    end

	fun emitClassRef instr class =
	    let val addr  = setU1 instr (!nextAddr)
		val index =
		    (case class of
			 CLASS c    => insClass' c
		       | ARRAY elem => insArrayClass'(Tarray elem))
	    in
		nextAddr:= setCpIndex index addr
	    end

	fun emitLdc index =
	    let val index' = Constpool.indexValue index
	    in
		if isU1 index' then
		    let val addr = setU1 0w18 (!nextAddr)   (* ldc *)
		    in
			nextAddr:= setU1i index' addr
		    end
		else
		    let val addr = setU1 0w19 (!nextAddr)   (* ldc_w *)
		    in
			nextAddr:= setU2i index' addr
		    end
	    end

	fun emitLoadClass class =  (* CHECK THIS CAREFULLY *)
	    let
		val index =
		    (case class of
			 CLASS c    => insClass' c
		       | ARRAY elem => insArrayClass'(Tarray elem))
		val index' = Constpool.indexValue index
	    in
		if isU1 index' then
		    let val addr = setU1 0w18 (!nextAddr)   (* ldc *)
		    in
			nextAddr:= setU1i index' addr
		    end
		else
		    let val addr = setU1 0w19 (!nextAddr)   (* ldc_w *)
		    in
			nextAddr:= setU2i index' addr
		    end
	    end

	fun emitFieldAccess instr args =
	    let val addr  = setU1 instr (!nextAddr)
	    in
		nextAddr:= setCpIndex (insFieldref' args) addr
	    end

	fun emitInvokeinterface (mref as {msig, ...}) =
	    let val nargs = argsWidth NONSTATIC msig
	    in
		if isU1 nargs then
		    let val addr   = setU1 0w185 (!nextAddr)
			val addr'  = setCpIndex (insIMethodref' mref) addr
			val addr'' = setU1i nargs addr'
		    in
			nextAddr:= setU1 0w0 addr''
		    end
		else codeError "emit [Jinvokeinterface]: nargs out of range"
	    end

	fun emitInvokeMethod mkind instr (mref as {msig, ...}) =
	    if isU1 (argsWidth mkind msig) then
		let val addr  = setU1 instr (!nextAddr)
		in
		    nextAddr:= setCpIndex (insMethodref' mref) addr
		end
	    else codeError
		 "emit [Jinvoke{special,static,virtual}]: nargs out of range"

	fun emitNewarray {elem, dim} =
	    if dim = 1 then
		if isSimple elem then
		    let val addr = setU1 0w188 (!nextAddr)   (* newarray *)
		    in
			nextAddr:= setU1 (typeKey elem) addr
		    end
		else
		    let val addr = setU1 0w189 (!nextAddr)   (* anewarray *)
		    in
			nextAddr:= setCpIndex (insArrayClass' elem) addr
		    end
	    else
		if isU1 dim then   (* multinewarray *)
		    let val addr  = setU1 0w197 (!nextAddr)
			val index = insArrayClass'(Tarray elem)
			val addr' = setCpIndex index addr
		    in
			nextAddr:= setU1i dim addr'
		    end
		else codeError "emit [Jnewarray]: dim out of range"

	fun wordAlign addr = addr + (~addr mod 4)

	fun emitLookupswitch {default, cases} =
	    let val base   = !nextAddr
		val addr   = setU1 0w171 base
		val addr'  = setOffset32 base default (wordAlign addr)
		val addr'' = setInti (length cases) addr'

		fun addCase ((key, lbl), map) = Binarymap.insert(map, key, lbl)

		val caseMap =
		    List.foldl addCase (Binarymap.mkDict Int32.compare) cases

		fun setCase (key, lbl, a) =
		    setOffset32 base lbl (setInt key a)
	    in
		nextAddr:= Binarymap.foldl setCase addr'' caseMap
	    end

	fun emitTableswitch {default, offset, targets} =
	    let val base    = !nextAddr
		val addr    = setU1 0w170 base
		val addr'   = setOffset32 base default (wordAlign addr)
		val addr''  = setInt offset addr'
		val len     = Vector.length targets
		val i32     = Int32.fromInt
		val high    = Bytecode.make_key (offset, len)
		val addr''' = setInt high addr''
		val setOffsets =
		    Vector.foldl (fn (lbl, a) => setOffset32 base lbl a)
	    in
		nextAddr:= setOffsets addr''' targets
	    end

	fun returnInstr () : Word8.word =
	    (case returnTy of
		 NONE   => 0w177   (* return *)
	       | SOME t =>
		     (case t of
			  Tboolean => 0w172   (* ireturn *)
			| Tchar    => 0w172   (* ireturn *)
			| Tfloat   => 0w174   (* freturn *)
			| Tdouble  => 0w175   (* dreturn *)
			| Tbyte    => 0w172   (* ireturn *)
			| Tshort   => 0w172   (* ireturn *)
			| Tint     => 0w172   (* ireturn *)
			| Tlong    => 0w173   (* lreturn *)
			| Tarray _ => 0w176   (* areturn *)
			| Tclass _ => 0w176   (* areturn *)))


	fun emitInstr instr =
	    (case instr of
		 Jlabel lbl         => backPatch lbl
	       | Jsconst s          => emitLdc(insString' s)
	       | Jaaload            => emitU1 0w50
	       | Jaastore           => emitU1 0w83
	       | Jaconst_null       => emitU1 0w1
	       | Jaload j           => immVarAccess
		                           {instr = 0w25,
					    imm_0 = 0w42, imm_1 = 0w43,
					    imm_2 = 0w44, imm_3 = 0w45} j
	       | Jarraylength       => emitU1 0w190
	       | Jastore j          => immVarAccess
					   {instr = 0w58,
					    imm_0 = 0w75, imm_1 = 0w76,
					    imm_2 = 0w77, imm_3 = 0w78} j
	       | Jathrow            => emitU1 0w191
	       | Jbaload            => emitU1 0w51
	       | Jbastore           => emitU1 0w84
	       | Jcaload            => emitU1 0w52
	       | Jcastore           => emitU1 0w85
	       | Jcheckcast class   => emitClassRef 0w192 class
	       | Jclassconst class  => emitLoadClass class
	       | Jd2f               => emitU1 0w144
	       | Jd2i               => emitU1 0w142
	       | Jd2l               => emitU1 0w143
	       | Jdadd              => emitU1 0w99
	       | Jdaload            => emitU1 0w49
	       | Jdastore           => emitU1 0w82
	       | Jdcmpg             => emitU1 0w152
	       | Jdcmpl             => emitU1 0w151
	       | Jdconst d          => emitDoubleConst d
	       | Jddiv              => emitU1 0w111
	       | Jdload j           => immVarAccess
		                           {instr = 0w24,
					    imm_0 = 0w38, imm_1 = 0w39,
					    imm_2 = 0w40, imm_3 = 0w41} j
	       | Jdmul              => emitU1 0w107
	       | Jdneg              => emitU1 0w119
	       | Jdrem              => emitU1 0w115
	       | Jdstore j          => immVarAccess
		                           {instr = 0w57,
					    imm_0 = 0w71, imm_1 = 0w72,
					    imm_2 = 0w73, imm_3 = 0w74} j
	       | Jdsub              => emitU1 0w103
	       | Jdup               => emitU1 0w89
	       | Jdup_x1            => emitU1 0w90
	       | Jdup_x2            => emitU1 0w91
	       | Jdup2              => emitU1 0w92
	       | Jdup2_x1           => emitU1 0w93
	       | Jdup2_x2           => emitU1 0w94
	       | Jf2d               => emitU1 0w141
	       | Jf2i               => emitU1 0w139
	       | Jf2l               => emitU1 0w140
	       | Jfadd              => emitU1 0w98
	       | Jfaload            => emitU1 0w48
	       | Jfastore           => emitU1 0w81
	       | Jfcmpg             => emitU1 0w150
	       | Jfcmpl             => emitU1 0w149
	       | Jfconst f          => emitFloatConst f
	       | Jfdiv              => emitU1 0w110
	       | Jfload j           => immVarAccess
		                           {instr = 0w23,
					    imm_0 = 0w34, imm_1 = 0w35,
					    imm_2 = 0w36, imm_3 = 0w37} j
	       | Jfmul              => emitU1 0w106
	       | Jfneg              => emitU1 0w118
	       | Jfrem              => emitU1 0w114
	       | Jfstore j          => immVarAccess
		                           {instr = 0w56,
					    imm_0 = 0w67, imm_1 = 0w68,
					    imm_2 = 0w69, imm_3 = 0w70} j
	       | Jfsub              => emitU1 0w102
	       | Jgetfield  a       => emitFieldAccess 0w180 a
	       | Jgetstatic a       => emitFieldAccess 0w178 a
	       | Jgoto lbl          => emitBranch16Or32
					   {instr = 0w167, instr_w = 0w200,
					    lbl   = lbl}
	       | Ji2b               => emitU1 0w145
	       | Ji2c               => emitU1 0w146
	       | Ji2d               => emitU1 0w135
	       | Ji2f               => emitU1 0w134
	       | Ji2l               => emitU1 0w133
	       | Ji2s               => emitU1 0w147
	       | Jiadd              => emitU1 0w96
	       | Jiaload            => emitU1 0w46
	       | Jiand              => emitU1 0w126
	       | Jiastore           => emitU1 0w79
	       | Jiconst i          => emitIntConst i
	       | Jidiv              => emitU1 0w108
	       | Jif_acmpeq lbl     => emitBranch16 0w165 lbl
	       | Jif_acmpne lbl     => emitBranch16 0w166 lbl
	       | Jif_icmpeq lbl     => emitBranch16 0w159 lbl
	       | Jif_icmpne lbl     => emitBranch16 0w160 lbl
	       | Jif_icmplt lbl     => emitBranch16 0w161 lbl
	       | Jif_icmpge lbl     => emitBranch16 0w162 lbl
	       | Jif_icmpgt lbl     => emitBranch16 0w163 lbl
	       | Jif_icmple lbl     => emitBranch16 0w164 lbl
	       | Jifeq lbl          => emitBranch16 0w153 lbl
	       | Jifne lbl          => emitBranch16 0w154 lbl
	       | Jiflt lbl          => emitBranch16 0w155 lbl
	       | Jifge lbl          => emitBranch16 0w156 lbl
	       | Jifgt lbl          => emitBranch16 0w157 lbl
	       | Jifle lbl          => emitBranch16 0w158 lbl
	       | Jifnull lbl        => emitBranch16 0w198 lbl  (* kwxm: was 0x160 *)
	       | Jifnonnull lbl     => emitBranch16 0w199 lbl  (* kwxm: was 0x159 *)
	       | Jiinc a            => emitIinc a
	       | Jiload j           => immVarAccess
		                           {instr = 0w21,
					    imm_0 = 0w26, imm_1 = 0w27,
					    imm_2 = 0w28, imm_3 = 0w29} j
	       | Jimul              => emitU1 0w104
	       | Jineg              => emitU1 0w116
	       | Jinstanceof class  => emitClassRef 0w193 class
	       | Jinvokeinterface a => emitInvokeinterface a
	       | Jinvokespecial   a => emitInvokeMethod NONSTATIC 0w183 a
	       | Jinvokestatic    a => emitInvokeMethod STATIC 0w184 a
	       | Jinvokevirtual   a => emitInvokeMethod NONSTATIC 0w182 a
	       | Jior               => emitU1 0w128
	       | Jirem              => emitU1 0w112
	       | Jishl              => emitU1 0w120
	       | Jishr              => emitU1 0w122
	       | Jistore j          => immVarAccess
					   {instr = 0w54,
					    imm_0 = 0w59, imm_1 = 0w60,
					    imm_2 = 0w61, imm_3 = 0w62} j
	       | Jisub              => emitU1 0w100
	       | Jiushr             => emitU1 0w124
	       | Jixor              => emitU1 0w130
	       | Jjsr lbl           => emitBranch16Or32
					   {instr = 0w168, instr_w = 0w201,
					    lbl   = lbl}
	       | Jl2d               => emitU1 0w138
	       | Jl2f               => emitU1 0w137
	       | Jl2i               => emitU1 0w136
	       | Jladd              => emitU1 0w97
	       | Jlaload            => emitU1 0w47
	       | Jland              => emitU1 0w127
	       | Jlastore           => emitU1 0w80
	       | Jlcmp              => emitU1 0w148
	       | Jlconst l          => emitLongConst l
	       | Jldiv              => emitU1 0w109
	       | Jlload j           => immVarAccess
					   {instr = 0w22,
					    imm_0 = 0w30, imm_1 = 0w31,
					    imm_2 = 0w32, imm_3 = 0w33} j
	       | Jlmul              => emitU1 0w105
	       | Jlneg              => emitU1 0w117
	       | Jlookupswitch a    => emitLookupswitch a
	       | Jlor               => emitU1 0w129
	       | Jlrem              => emitU1 0w113
	       | Jlshl              => emitU1 0w121
	       | Jlshr              => emitU1 0w123
	       | Jlstore j          => immVarAccess
					   {instr = 0w55,
					    imm_0 = 0w63, imm_1 = 0w64,
					    imm_2 = 0w65, imm_3 = 0w66} j
	       | Jlsub              => emitU1 0w101
	       | Jlushr             => emitU1 0w125
	       | Jlxor              => emitU1 0w131
	       | Jmonitorenter      => emitU1 0w194
	       | Jmonitorexit       => emitU1 0w195
	       | Jnew class         => emitClassRef 0w187 (CLASS class)
	       | Jnewarray a        => emitNewarray a
	       | Jnop               => emitU1 0w0
	       | Jpop               => emitU1 0w87
	       | Jpop2              => emitU1 0w88
	       | Jputfield  a       => emitFieldAccess 0w181 a
	       | Jputstatic a       => emitFieldAccess 0w179 a
	       | Jret j             => emitVarAccess 0w169 (Localvar.toInt j)
	       | Jreturn            => emitU1(returnInstr())
	       | Jsaload            => emitU1 0w53
	       | Jsastore           => emitU1 0w86
	       | Jswap              => emitU1 0w95
	       | Jtableswitch a     => emitTableswitch a
	     )

	val codeVec = (List.app emitInstr code;
		       Word8Array.extract(code', 0, SOME (!nextAddr)))

	fun labelMap lbl =
	    (case Polyhash.peek targets lbl of
		 NONE => codeError ("emit.labelMap: unknown label " ^ Int.toString (Label.toInt lbl))
	       | SOME (RESOLVED trgtAddr) => trgtAddr
	       | _ => codeError "emit.labelMap: unresolved label")
    in
	(codeVec, labelMap)
    end



