(* Convert Classfile.class_file to Classdecl.class_decl *)
(* This is the inverse of Classfile.fromClassDecl;  it should really go
   in Classfile.sml,  but this leads to circular dependencies unless we put
   almost everything in one file. *)


open Bytecode Classfile Constpool Jvmtype Localvar

local

(* Stuff for decompiling bytecode *)

exception DecompileError of string
fun error s = raise DecompileError s
exception InvalidOpcode of string

val i16 = Word16.toInt



fun extract_name (pool:pool) idx =
case (lookup pool idx) of CPutf8 s => s
	                | n        => raise DecompileError "extract_name"

fun extract_class (pool:pool) idx =
    case (lookup pool idx) of
	CPclass idx2 => extract_name pool idx2
  | n => raise DecompileError "extract_class 2"


fun getField_ref pool idx =
    let val fld = lookup pool idx
	val (cl_idx,naty_idx) =
	    case fld of
		CPfieldref {class, nameType} => (class, nameType)
	      | n => raise InvalidEntry "FieldExpectedError"
	val naty = lookup pool naty_idx
	val cl_name = extract_class pool cl_idx
	val cl = Jvmtype.class {name=cl_name, pkgs=[]}
	val (na_idx,ty_idx) = case naty of CPnametype {name,desc} => (name, desc)
			                 | n            => raise InvalidEntry "NameTypeExpectedError"
	val na_val = extract_name pool na_idx
	val ty_val = parseType (extract_name pool ty_idx)
    in  { class = cl,
	  name  = na_val,
	  ty    = ty_val }
    end

fun getMethod_ref pool idx =
    let val fld = lookup pool idx
	val (cl_idx,naty_idx) =
	    case fld of
		CPmethodref {class, nameType} => (class, nameType)
	      | n => raise InvalidEntry "MethodExpectedError"
	val naty    = lookup pool naty_idx
	val cl_name = extract_class pool cl_idx
	val cl      = class {name=cl_name, pkgs=[]}
	val (na_idx,ty_idx) = case naty of
				  CPnametype {name,desc} => (name, desc)
				| n => raise InvalidEntry "NameTypeExpectedError"
	val na_val      = extract_name pool na_idx
	val type_string = extract_name pool ty_idx
	val ty_val      = parseTypes type_string
    in  { class = cl,
	  name  = na_val,
	  msig = ty_val }
    end

fun getIMethod_ref pool idx =
    let val fld = lookup pool idx
	val (cl_idx,naty_idx) =
	    case fld of
		CPimethodref {class, nameType} => (class, nameType)
	      | n => raise InvalidEntry "InterfaceMethodExpectedError"
	val naty    = lookup pool naty_idx
	val cl_name = extract_class pool cl_idx
	val cl      = class {name=cl_name, pkgs=[]}
	val (na_idx,ty_idx) = case naty of
				  CPnametype {name,desc} => (name, desc)
				| n => raise InvalidEntry "NameTypeExpectedError"
	val na_val      = extract_name pool na_idx
	val type_string = extract_name pool ty_idx
	val ty_val      = parseTypes type_string
    in  { class = cl,
	  name  = na_val,
	  msig = ty_val }
    end


fun crefToClass name =
let
    val char_list = explode name
    fun choose clist = case (hd clist) of
			   #"[" => ARRAY ((#1 o parseType_aux) (tl clist))
			 | x    => CLASS (Jvmtype.class {name=name,pkgs=[]})
in
    choose char_list
end

fun parseClass_ref pool idx =
    let val name = extract_class pool idx
    in
	crefToClass name
    end


(* The following function converts an array of bytecode to a list of
   constructors from Bytecode.sml,  building a table which maps offsets
   in the code array to positions in the list.  *)


fun decompileBytecode codeAttr pool =
    let
	fun translateCode code =
	    let
		val code_size = Word8Vector.length code

		val counter = ref 0
		val offset_map = ref (Intmap.empty())

		fun decompileOpcode v i0  = (* v is always 'code'; could lift it out *)
		    let
			val () = offset_map:= Intmap.insert(!offset_map, i0, !counter)
			(* Note that this still works for wide instructions.  Maybe. *)
			val () = counter:=(!counter+1)

			val (instr,i) = parseU1_int v i0

			val i32 = Int32.fromInt
			val i64 = Int64.fromInt
			val r32 = Real32.fromReal
			val r64 = Real64.fromReal
		    in
			case instr of
			    0  => ( Jnop        , i )
			  | 1  => ( Jaconst_null, i )
			  | 2  => ( Jiconst (i32 ~1)  , i )
			  | 3  => ( Jiconst (i32 0)   , i )
			  | 4  => ( Jiconst (i32 1)   , i )
			  | 5  => ( Jiconst (i32 2)   , i )
			  | 6  => ( Jiconst (i32 3)   , i )
			  | 7  => ( Jiconst (i32 4)   , i )
			  | 8  => ( Jiconst (i32 5)   , i )
			  | 9  => ( Jlconst (i64 0)   , i )
			  | 10 => ( Jlconst (i64 1)   , i )
			  | 11 => ( Jfconst (r32 0.0) , i )
			  | 12 => ( Jfconst (r32 1.0) , i )
			  | 13 => ( Jfconst (r32 2.0) , i )
			  | 14 => ( Jdconst (r64 0.0) , i )
			  | 15 => ( Jdconst (r64 1.0) , i )
			  | 16 => ( fn (byte,j) => ( Jiconst (i32 byte), j )
				  ) (parseU1_intX v i)
			  | 17 => ( fn (bytes2,j) => ( Jiconst (i32 bytes2), j )
				  ) (parseU2_intX v i)
			  | 18 => let val (idx, i) = (parseU1_int v i)  (* ldc *)
				      val cp_item = Constpool.lookup pool (Constpool.makeIndex idx)
				  in  case cp_item of
					  Constpool.CPint integer    => (Jiconst integer, i)
					| Constpool.CPfloat float    =>
					  let val r_opt = (Real32.fromBytes float)
					  in  case r_opt of
						  SOME r => (Jfconst r, i)
						| NONE   => raise InvalidOpcode "pCode 18a"
					  end
					| Constpool.CPstring str_ref =>
					  let val cp_item = (Constpool.lookup pool str_ref)
					  in  case cp_item of
						  Constpool.CPutf8 s => (Jsconst s, i)
						| _                  => raise InvalidOpcode "pCode 18b"
					  end
					| Constpool.CPclass c_ref =>  (* New in Java 5 *)
                                          let
					      val n = extract_name pool c_ref
					      val cl = crefToClass n
					  in
					      (Jclassconst cl, i)
					  end
					| _  => raise InvalidOpcode "pCode 18d"
				  end
			  | 19 => let val (idx, i) = (parseU2_idx v i)  (* ldc_w *)
				      val cp_item = Constpool.lookup pool idx
				  in  case cp_item of
					  Constpool.CPint integer    => (Jiconst integer, i)
					| Constpool.CPfloat float    =>
					  let val r_opt = (Real32.fromBytes float)
					  in  case r_opt of
						  SOME r => (Jfconst r, i)
						| NONE   => raise InvalidOpcode "pCode 19a"
					  end
					| Constpool.CPstring str_ref =>
					  let val cp_item = (Constpool.lookup pool str_ref)
					  in  case cp_item of
						  Constpool.CPutf8 s => (Jsconst s, i)
						| _                  => raise InvalidOpcode "pCode 19b"
					  end
					| Constpool.CPclass c_ref =>  (* New in Java 5 *)
                                          let
					      val n = extract_name pool c_ref
					      val cl = crefToClass n
					  in
					      (Jclassconst cl, i)
					  end
					| _                          => raise InvalidOpcode "pCode 19d"
				  end
			  | 20 => let val (idx, i) = (parseU2_idx v i)
				      val cp_item = Constpool.lookup pool idx
				  in  case cp_item of
					  Constpool.CPdouble double =>
					  let val d_opt = (Real64.fromBytes double)
					  in  case d_opt of
						  SOME d => (Jdconst d, i)
						| NONE   => raise InvalidOpcode "pCode 20a"
					  end
					| Constpool.CPlong   long   => (Jlconst long, i)
					| _                         => raise InvalidOpcode "pCode 20c"
				  end
			  | 21 => ( fn (byte,j) => ( Jiload (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 22 => ( fn (byte,j) => ( Jlload (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 23 => ( fn (byte,j) => ( Jfload (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 24 => ( fn (byte,j) => ( Jdload (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 25 => ( fn (byte,j) => ( Jaload (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 26 => ( Jiload (fromInt 0) , i )
			  | 27 => ( Jiload (fromInt 1) , i )
			  | 28 => ( Jiload (fromInt 2) , i )
			  | 29 => ( Jiload (fromInt 3) , i )
			  | 30 => ( Jlload (fromInt 0) , i )
			  | 31 => ( Jlload (fromInt 1) , i )
			  | 32 => ( Jlload (fromInt 2) , i )
			  | 33 => ( Jlload (fromInt 3) , i )
			  | 34 => ( Jfload (fromInt 0) , i )
			  | 35 => ( Jfload (fromInt 1) , i )
			  | 36 => ( Jfload (fromInt 2) , i )
			  | 37 => ( Jfload (fromInt 3) , i )
			  | 38 => ( Jdload (fromInt 0) , i )
			  | 39 => ( Jdload (fromInt 1) , i )
			  | 40 => ( Jdload (fromInt 2) , i )
			  | 41 => ( Jdload (fromInt 3) , i )
			  | 42 => ( Jaload (fromInt 0) , i )
			  | 43 => ( Jaload (fromInt 1) , i )
			  | 44 => ( Jaload (fromInt 2) , i )
			  | 45 => ( Jaload (fromInt 3) , i )
			  | 46 => ( Jiaload  , i )
			  | 47 => ( Jlaload  , i )
			  | 48 => ( Jfaload  , i )
			  | 49 => ( Jdaload  , i )
			  | 50 => ( Jaaload  , i )
			  | 51 => ( Jbaload  , i )
			  | 52 => ( Jcaload  , i )
			  | 53 => ( Jsaload  , i )
			  | 54 => ( fn (byte,j) => ( Jistore (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 55 => ( fn (byte,j) => ( Jlstore (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 56 => ( fn (byte,j) => ( Jfstore (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 57 => ( fn (byte,j) => ( Jdstore (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 58 => ( fn (byte,j) => ( Jastore (fromInt byte), j )
				  ) (parseU1_int v i)
			  | 59 => ( Jistore (fromInt 0), i )
			  | 60 => ( Jistore (fromInt 1), i )
			  | 61 => ( Jistore (fromInt 2), i )
			  | 62 => ( Jistore (fromInt 3), i )
			  | 63 => ( Jlstore (fromInt 0), i )
			  | 64 => ( Jlstore (fromInt 1), i )
			  | 65 => ( Jlstore (fromInt 2), i )
			  | 66 => ( Jlstore (fromInt 3), i )
			  | 67 => ( Jfstore (fromInt 0), i )
			  | 68 => ( Jfstore (fromInt 1), i )
			  | 69 => ( Jfstore (fromInt 2), i )
			  | 70 => ( Jfstore (fromInt 3), i )
			  | 71 => ( Jdstore (fromInt 0), i )
			  | 72 => ( Jdstore (fromInt 1), i )
			  | 73 => ( Jdstore (fromInt 2), i )
			  | 74 => ( Jdstore (fromInt 3), i )
			  | 75 => ( Jastore (fromInt 0), i )
			  | 76 => ( Jastore (fromInt 1), i )
			  | 77 => ( Jastore (fromInt 2), i )
			  | 78 => ( Jastore (fromInt 3), i )
			  | 79 => ( Jiastore , i )
			  | 80 => ( Jlastore , i )
			  | 81 => ( Jfastore , i )
			  | 82 => ( Jdastore , i )
			  | 83 => ( Jaastore , i )
 			  | 84 => ( Jbastore , i )
			  | 85 => ( Jcastore , i )
			  | 86 => ( Jsastore , i )
			  | 87 => ( Jpop     , i )
			  | 88 => ( Jpop2    , i )
			  | 89 => ( Jdup     , i )
			  | 90 => ( Jdup_x1  , i )
			  | 91 => ( Jdup_x2  , i )
			  | 92 => ( Jdup2    , i )
			  | 93 => ( Jdup2_x1 , i )
			  | 94 => ( Jdup2_x2 , i )
			  | 95 => ( Jswap    , i )
			  | 96 => ( Jiadd    , i )
			  | 97 => ( Jladd    , i )
			  | 98 => ( Jfadd    , i )
			  | 99 => ( Jdadd    , i )
			  | 100 => ( Jisub    , i )
			  | 101 => ( Jlsub    , i )
			  | 102 => ( Jfsub    , i )
			  | 103 => ( Jdsub    , i )
			  | 104 => ( Jimul    , i )
			  | 105 => ( Jlmul    , i )
			  | 106 => ( Jfmul    , i )
			  | 107 => ( Jdmul    , i )
			  | 108 => ( Jidiv    , i )
			  | 109 => ( Jldiv    , i )
			  | 110 => ( Jfdiv    , i )
			  | 111 => ( Jddiv    , i )
			  | 112 => ( Jirem    , i )
			  | 113 => ( Jlrem    , i )
			  | 114 => ( Jfrem    , i )
			  | 115 => ( Jdrem    , i )
			  | 116 => ( Jineg    , i )
			  | 117 => ( Jlneg    , i )
			  | 118 => ( Jfneg    , i )
			  | 119 => ( Jdneg    , i )
			  | 120 => ( Jishl    , i )
			  | 121 => ( Jlshl    , i )
			  | 122 => ( Jishr    , i )
			  | 123 => ( Jlshr    , i )
			  | 124 => ( Jiushr   , i )
			  | 125 => ( Jlushr   , i )
			  | 126 => ( Jiand    , i )
			  | 127 => ( Jland    , i )
			  | 128 => ( Jior     , i )
			  | 129 => ( Jlor     , i )
			  | 130 => ( Jixor    , i )
			  | 131 => ( Jlxor    , i )
			  | 132 => let
				val (b1,i) = parseU1_int v i
				val (b2,i) = parseU1_intX v i
			    in
				( Jiinc { var   = fromInt b1, const = b2 }, i )
			    end
			  | 133 => ( Ji2l     , i )
			  | 134 => ( Ji2f     , i )
			  | 135 => ( Ji2d     , i )
			  | 136 => ( Jl2i     , i )
			  | 137 => ( Jl2f     , i )
			  | 138 => ( Jl2d     , i )
			  | 139 => ( Jf2i     , i )
			  | 140 => ( Jf2l     , i )
			  | 141 => ( Jf2d     , i )
			  | 142 => ( Jd2i     , i )
			  | 143 => ( Jd2l     , i )
			  | 144 => ( Jd2f     , i )
			  | 145 => ( Ji2b     , i )
			  | 146 => ( Ji2c     , i )
			  | 147 => ( Ji2s     , i )
			  | 148 => ( Jlcmp    , i )
			  | 149 => ( Jfcmpl   , i )
			  | 150 => ( Jfcmpg   , i )
			  | 151 => ( Jdcmpl   , i )
			  | 152 => ( Jdcmpg   , i )
			  | 153 => ( fn (bytes2,j) => ( Jifeq      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 154 => ( fn (bytes2,j) => ( Jifne      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 155 => ( fn (bytes2,j) => ( Jiflt      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 156 => ( fn (bytes2,j) => ( Jifge      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 157 => ( fn (bytes2,j) => ( Jifgt      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 158 => ( fn (bytes2,j) => ( Jifle      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 159 => ( fn (bytes2,j) => ( Jif_icmpeq (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 160 => ( fn (bytes2,j) => ( Jif_icmpne (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 161 => ( fn (bytes2,j) => ( Jif_icmplt (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 162 => ( fn (bytes2,j) => ( Jif_icmpge (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 163 => ( fn (bytes2,j) => ( Jif_icmpgt (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 164 => ( fn (bytes2,j) => ( Jif_icmple (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 165 => ( fn (bytes2,j) => ( Jif_acmpeq (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 166 => ( fn (bytes2,j) => ( Jif_acmpne (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 167 => ( fn (bytes2,j) => ( Jgoto      (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 168 => ( fn (bytes2,j) => ( Jjsr       (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 169 => ( fn (byte  ,j) => ( Jret       (fromInt byte)        , j )
				   ) (parseU1_int v i)
			  | 170 => (let val offset_size   = ~i mod 4
				       val (offset,j)    = iterate parseU1_int ([],i,offset_size) v
				       val (default,j)   = parseU4_i32 v j
				       val (low,j)       = parseU4_i32 v j
				       val (high,j)      = parseU4_i32 v j
				       val ntargets      = Int32.toInt (Int32.- (high, low))+1
				       val (targets,j)   = iterate parseU4_i32 ([],j,ntargets) v
				       fun mkLbl n       = Label.fromInt (
							   Int32.toInt (Bytecode.make_key (n,i)))
				       val t_vec         = Vector.fromList (map mkLbl targets)
				   in  ( Jtableswitch { default = mkLbl default,
							offset  = low,
							targets = t_vec },
					 j )
				   end
				   handle Int32.Int32Overflow _ =>
					  error "Int32Overflow in tableswitch")
	 		  | 171 => (let val offset_size = ~i mod 4
				       val (offset,j)  = iterate parseU1_int ([],i,offset_size) v
				       val (default,j) = parseU4_i32 v j
				       val (npairs,j)  = parseU4_i32 v j
				       fun mkLbl n = Label.fromInt (i-1+Int32.toInt n)
				       fun pPair v n   =
					   let
					       val (match,  n1) = parseU4_i32 v n
					       val (offset, n2) = parseU4_i32 v n1
					   in
					       ((match, mkLbl offset), n2 )
					   end
				       val (targets,j) = iterate pPair ([],j,Int32.toInt npairs) v
				    in
					( Jlookupswitch { default = mkLbl default, cases = targets  },
					  j )
				   end
				    handle Int32.Int32Overflow _ =>
					   error "Int32Overflow in lookupswitch")
			  | 172 => ( Jreturn , i )
			  | 173 => ( Jreturn , i )
			  | 174 => ( Jreturn , i )
			  | 175 => ( Jreturn , i )
			  | 176 => ( Jreturn , i )
			  | 177 => ( Jreturn  , i )
			  | 178 => let val (idx,i) = parseU2_idx v i
				       val fld_ref = getField_ref pool idx
				   in  ( Jgetstatic     fld_ref , i )
				   end
			  | 179 => let val (idx,i) = parseU2_idx v i
				       val fld_ref = getField_ref pool idx
				   in  ( Jputstatic     fld_ref , i )
				   end
			  | 180 => let val (idx,i) = parseU2_idx v i
				       val fld_ref = getField_ref pool idx
				   in  ( Jgetfield      fld_ref , i )
				   end
			  | 181 => let val (idx,i) = parseU2_idx v i
				       val fld_ref = getField_ref pool idx
				   in  ( Jputfield      fld_ref , i )
				   end
			  | 182 => let val (idx,i) = parseU2_idx v i
				       val mthd_ref = getMethod_ref pool idx
				   in  ( Jinvokevirtual mthd_ref, i )
				   end
			  | 183 => let val (idx,i) = parseU2_idx v i
				       val mthd_ref = getMethod_ref pool idx
				   in  ( Jinvokespecial mthd_ref, i )
				   end
			  | 184 => let val (idx,i) = parseU2_idx v i
				       val mthd_ref = getMethod_ref pool idx
				   in  ( Jinvokestatic  mthd_ref, i )
				   end
			  | 185 => let val (idx,i) =   parseU2_idx v i
				       val mthd_ref =  getIMethod_ref pool idx
				       val (args, i) = parseU1_int v i  (* kwxm: don't need to save args*)
				       val (zero, i) = parseU1_int v i  (* kwxm: zero should be 0 *)
				   in  ( Jinvokeinterface mthd_ref, i )
				   end
				       (* | 186 =>  not in the JVM-set *)
			  | 187 => let val (idx,i) = parseU2_idx v i
				       val cl_name = extract_class pool idx
				   in  ( Jnew (Jvmtype.class { pkgs = []
							     , name = cl_name }
					      )
				       , i )
				   end
			  | 188 => let val (kind_int,i) = parseU1_int v i (* newarray *)
				       val kind = case kind_int of
						      4 => Jvmtype.Tboolean
						    | 5 => Jvmtype.Tchar
						    | 6 => Jvmtype.Tfloat
						    | 7 => Jvmtype.Tdouble
						    | 8 => Jvmtype.Tbyte
						    | 9 => Jvmtype.Tshort
						    | 10 => Jvmtype.Tint
						    | 11 => Jvmtype.Tlong
						    | n => raise InvalidOpcode "kind IN jvm_instr 188"
				   in
				       (Jnewarray {elem=kind, dim=1}, i )
					   (* (Jnop,i) (*mprowse*) --- WHY?? kwxm *)
				   end
			  | 189 => let val (idx,i)   = parseU2_idx v i  (* anewarray *)
				       val cl_name   = extract_class pool idx
				       val char_list = explode cl_name
				       val fst_char  = hd char_list
				       val rest      = tl char_list
				   in  case fst_char of
					   #"[" => let
					       val t = #1 (parseType_aux rest)
					   in
					       (Jnewarray {elem=t, dim=1}, i )
					   end
					 | _ => let
					       val t = Jvmtype.Tclass (Jvmtype.class
									   { pkgs = [], name = cl_name })
					   in
					       (Jnewarray {elem=t, dim=1}, i )
					   end
				   end
			  | 190 => ( Jarraylength, i )
			  | 191 => ( Jathrow     , i )
			  | 192 => let val (idx,i) = parseU2_idx v i
				       val cl_ref  = parseClass_ref pool idx
				   in  ( Jcheckcast cl_ref , i )
				   end
			  | 193 => let val (idx,i) = parseU2_idx v i
				       val cl_ref  = parseClass_ref pool idx
				   in  ( Jinstanceof cl_ref, i )
				   end
			  | 194 => ( Jmonitorenter, i )
			  | 195 => ( Jmonitorexit , i )
			  | 196 => decompileWideOpcode v i  (* wide *)
			  | 197 => let val (idx,i)   = parseU2_idx v i
				       val cl_name = extract_class pool idx
				       val (dims, i) = parseU1_int v i
				   in  ( Jnewarray { elem = parseType cl_name
  						   , dim=dims }
				       , i )
				   end
			  | 198 => ( fn (bytes2,j) => ( Jifnull (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 199 => ( fn (bytes2,j) => ( Jifnonnull (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | 200 => (* goto_w *)
				   ( ( fn (bytes4,j) =>
					( Jgoto (Label.fromInt (i0+bytes4)), j )
				       ) (parseU4_intX v i)
				     handle Word32.Word32Overflow _ =>
					    error "Enormous offset in goto_w")
				   (* Shouldn't happen (64K limit on method sizes) *)
			  | 201 => (* jsr_w *)
				   ( fn (bytes2,j) => ( Jjsr(*_w*) (Label.fromInt (i0+bytes2)), j )
				   ) (parseU2_intX v i)
			  | n   => raise InvalidOpcode (Int.toString n)
		    end

		and decompileWideOpcode v i0 =
		    let
 			val (instr,i) = parseU1_int v i0
		    in
			case instr of
			    21 => ( fn (bytes2,j) => ( Jiload (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 22 => ( fn (bytes2,j) => ( Jlload (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 23 => ( fn (bytes2,j) => ( Jfload (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 24 => ( fn (bytes2,j) => ( Jdload (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 25 => ( fn (bytes2,j) => ( Jaload (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 54 => ( fn (bytes2,j) => ( Jistore (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 55 => ( fn (bytes2,j) => ( Jlstore (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 56 => ( fn (bytes2,j) => ( Jfstore (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 57 => ( fn (bytes2,j) => ( Jdstore (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 58 => ( fn (bytes2,j) => ( Jastore (fromInt bytes2), j )
				  ) (parseU2_int v i)
			  | 132 =>
			    let
				val (w1,i) = parseU2_int v i
				val (w2,i) = parseU2_intX v i
			    in
				( Jiinc { var = fromInt w1, const = w2 }, i )
			    end
			  | 169 => ( fn (bytes2,j) => ( Jret (fromInt bytes2), j )
				   ) (parseU2_int v i)
			  | _ => raise InvalidOpcode ("wide " ^ Int.toString instr)
		    end

		fun decompileCodeArray v index acc =
		    if index < code_size
		    then
			let
			    val (s, index') = decompileOpcode v index
			    (* val () = printJvmInstr.prInstr s *)
			in
			    decompileCodeArray v index' (s::acc)
			end
		    else
			(rev acc, index)
		val (bytecode_list, end_of_code) = decompileCodeArray code 0 []

	    in
		(bytecode_list, end_of_code, !offset_map)
	    end (* tranlsateCode *)



	local
	    val LABELS = ref Intset.empty  (* Positions of instructions which will require labels *)
	in
	    fun SAVE_LABEL n = LABELS := Intset.add (!LABELS, n)
	    fun insertLabels code =
		let
		    fun ins [] pos acc =
			rev (Jlabel (Label.fromInt pos)::acc) (* marker at end of code *)
		      | ins (h::t) pos acc =
			if Intset.member (!LABELS, pos) then
 			     ins t (pos+1) (h::(Jlabel (Label.fromInt pos))::acc)
			else ins t (pos+1) (h::acc)
		in
		    ins code 0 []
		end
	end

	fun findPos offset_map n =
	    case Intmap.peek(offset_map,n) of
		NONE => error ("Can't find index at offset " ^ Int.toString n)
	      | SOME q => q


	fun findLabel offset_map n = (* side-effects on LABELS *)
	    let (* find appropriate label, remembering that we'll need to emit it later *)
		val p = findPos offset_map n
	        val () = SAVE_LABEL p
	    in
		Label.fromInt p
	    end


	fun transformTarget offset_map instr = (* side-effects on LABELS *)
	    let
		fun transformTgt tgt = (* translate jump target,  remembering to label target later *)
		let
		    val tgt' = findPos offset_map (Label.toInt tgt)
		    val () = SAVE_LABEL tgt'
		in
		    Label.fromInt tgt'
		end
	    in
		case instr of
		    Jifeq tgt      => Jifeq (transformTgt tgt)
		  | Jifne tgt      => Jifne (transformTgt tgt)
		  | Jiflt tgt      => Jiflt (transformTgt tgt)
		  | Jifge tgt      => Jifge (transformTgt tgt)
		  | Jifgt tgt      => Jifgt (transformTgt tgt)
		  | Jifle tgt      => Jifle (transformTgt tgt)
		  | Jif_icmpeq tgt => Jif_icmpeq (transformTgt tgt)
		  | Jif_icmpne tgt => Jif_icmpne (transformTgt tgt)
		  | Jif_icmplt tgt => Jif_icmplt (transformTgt tgt)
		  | Jif_icmpge tgt => Jif_icmpge (transformTgt tgt)
		  | Jif_icmpgt tgt => Jif_icmpgt (transformTgt tgt)
		  | Jif_icmple tgt => Jif_icmple (transformTgt tgt)
		  | Jif_acmpeq tgt => Jif_acmpeq (transformTgt tgt)
		  | Jif_acmpne tgt => Jif_acmpne (transformTgt tgt)
		  | Jifnull tgt    => Jifnull (transformTgt tgt)
		  | Jifnonnull tgt => Jifnonnull (transformTgt tgt)
		  | Jgoto tgt      => Jgoto (transformTgt tgt)
		  | Jtableswitch {default, offset, targets}
		    => Jtableswitch {default = transformTgt default,
				     offset = offset,
				     targets = Vector.map transformTgt targets}
		  | Jlookupswitch {default, cases} =>
		    Jlookupswitch {default = transformTgt default,
				   cases = map (fn (a,b) => (a,transformTgt b)) cases}
		  | Jjsr tgt       => Jjsr (transformTgt tgt)
		  | _              => instr
	    end


	fun getUtf8 idx =
	    case lookup pool idx of
		CPutf8 str => str
	      | _ => error "getUtf8"


	fun getClass class_idx =
            case lookup pool class_idx of
                CPclass idx => Jvmtype.qualNameToClass (getUtf8 idx)
              | _ => error "getClass"

	fun fixLineNumberOffset offset_map {start, line} =
	    let val start' = findLabel offset_map (i16 start)
	    in
                 {start = start', line = i16 line}
	    end
	    handle DecompileError _ =>
		   (print "WARNING: discarding bad entry in line number table\n";
		    {start = Label.fromInt 0, line=0})
            (* Maybe we should let the exception through.  I've seen corrupt
               line number info in real classfiles,  and it causes most decompilers
               to crash. *)

	fun fixLocalVarRange offset_map {start, length, name, desc, index} =
	    let
		val s = i16 start
		val e = s + i16 length
		val from = findLabel offset_map s
		val thru = findLabel offset_map e
		val ty = case Jvmtype.scanTypeDesc (getUtf8 desc) of
			     SOME s => s
			   | _ => error "fix_local_var"
		val name = getUtf8 name
	    in
		{from = from,
		 thru = thru,
		 name = name,
		 ty = ty,
		 index = Localvar.fromInt (i16 index) }
	    end

	fun fixLocalVarTypes offset_map {start, length, name, sign, index} =
	    let
		val s = i16 start
		val e = s + i16 length
		val from = findLabel offset_map s
		val thru = findLabel offset_map e
		val sign = getUtf8 sign
		val name = getUtf8 name
	    in
		{from = from,
		 thru = thru,
		 name = name,
		 sign = sign,
		 index = Localvar.fromInt (i16 index) }
	    end

	fun fixExnHdl offset_map {start, stop, entry, catch}
	  = {start = findLabel offset_map (i16 start),
	     stop =  findLabel offset_map (i16 stop),
	     entry = findLabel offset_map (i16 entry),
	     catch = Option.map getClass catch}



	fun fixCodeAttr offset_map attr =
	    let
		fun err attr = error ("Code object has unexpected "
				      ^ getUtf8 attr
				      ^ " attribute")
	    in
		case attr of
		    LINENUM {attr, lines}  => Classdecl.LINENUM (map (fixLineNumberOffset offset_map) lines)
		  | LOCALVAR {attr, vars}  => Classdecl.LOCALVAR (map (fixLocalVarRange offset_map) vars)
		  | LOCALVARTYPES {vartypes, ...}
		       => Classdecl.LOCALVARTYPES (print "+++++++++++\n";
						   map (fixLocalVarTypes offset_map) vartypes)
		  | CODE {attr, ...}       => err attr
		  | SRCFILE {attr, ...}    => err attr
		  | CONSTVAL {attr, ...}   => err attr
		  | EXNS {attr, ...}       => err attr
		  | INNER {attr, ...}      => err attr
		  | SYNTHETIC {attr}       => err attr
		  | DEPRECATED {attr}      => err attr
		  | ENCLOSING {attr, ...}  => err attr
		  | RTV_ANN {attr, ...}    => err attr
		  | RTI_ANN {attr, ...}    => err attr
		  | RTV_PARAM_ANN {attr, ...} => err attr
		  | RTI_PARAM_ANN {attr, ...} => err attr
		  | ANN_DEFAULT {attr, ...}   => err attr
		  | SIGNATURE {attr, ...}    => err attr
		  | ATTR {attr, info}      => Classdecl.ATTR {attr=getUtf8 attr, info=info}
	    end

	fun fixCodeLabels attr =
	    case attr of
		CODE {attr, stack, locals, code, hdls, attrs} =>
		let
		    val (bytecode_list, end_of_code, offset_map0) = translateCode code

		    val offset_map = Intmap.insert(offset_map0, end_of_code, Intmap.numItems(offset_map0))
		    (* Position just after end of code - needed by local var info etc. *)

		    val hdls'  = map (fixExnHdl offset_map) hdls     (* side-effects on LABELS *)
		    val attrs' = map (fixCodeAttr offset_map) attrs  (* side-effects on LABELS *)

		    val bytecode' = map (transformTarget offset_map) bytecode_list

		    val labelled_bytecode = insertLabels bytecode'
                    (* val () = app printJvmInstr.prInstr labelled_bytecode*)
		in
		    Classdecl.CODE {stack = i16 stack,
				    locals = i16 locals,
				    code = labelled_bytecode,
				    hdls = hdls',
				    attrs = attrs'}
		end
	      | _ => error "CODE attribute expected in fixCodeLabels"
    in
	fixCodeLabels codeAttr
    end

(* The labels which we emit are non-sequential,  referring to instruction positions.
   It would be nice if they were sequential (1,2,3,4,5,...),  but this would require
   a lot more work.  We'd have to scan the code and metadata to identify all instructions
   which have to be labelled and then go back and use this information to modify the jump
   targets and meatadata. *)


in


fun toClassDecl ({magic, minor, major, pool, flags, this,
		  super, ifcs, fields, methods, attrs}:Classfile.class_file) =
    let
        fun cperror s = raise InvalidClass (s^": Constant Pool index refers to the wrong type of entry")
	fun error s = raise InvalidClass s
        val lookup = lookup pool

        fun getUtf8 idx = case lookup idx of CPutf8 str => str | _ => cperror "getUtf8"
        fun getClass class_idx =
            case lookup class_idx of
                CPclass idx => Jvmtype.qualNameToClass (getUtf8 idx)
              | _ => cperror "getClass"

	fun getNameType idx =
            case lookup idx of
                CPnametype n => n
              | _ => error "getNameType"

	fun fixMeth idx =
	    let val {name, desc} = getNameType idx
	    in
		{mname = getUtf8 name, msig = valOf (Jvmtype.scanMethodDesc (getUtf8 desc))}
	    end

        fun getConst idx =
            case lookup idx of
                CPint i => Cint i
              | CPfloat f => Cfloat (valOf (Real32.fromBytes f))
              | CPlong l => Clong l
              | CPdouble d => Cdouble (valOf (Real64.fromBytes d))
              | CPstring idx => Cstring (getUtf8 idx)
              | _ => cperror "getConst"

	fun fix_inner {inner_info, outer_info, inner_name, inner_flags}
	  = {inner_info = Option.map getClass inner_info,
	     outer_info = Option.map getClass outer_info,
	     inner_name = Option.map getUtf8 inner_name,
	     inner_flags = classFlagsFromWord inner_flags }: Classdecl.inner_class_info

	fun getInt idx =
	    case lookup idx of
		CPint i => i
	      | _ => cperror "getInt"

	fun getint idx = Int32.toInt (getInt idx)

	fun getFloat idx =
	    case lookup idx of
		CPfloat f => valOf (Real32.fromBytes f)
	      | _ => cperror "getFloat"

	fun getLong idx =
	    case lookup idx of
		CPlong l => l
	      | _ => cperror "getLong"

	fun getDouble idx =
	    case lookup idx of
		CPdouble d => valOf (Real64.fromBytes d)
	      | _ => cperror "getDouble"

	fun getString idx =
	    case lookup idx of
		CPstring i => getUtf8 i
	      | _ => cperror "getString"

	fun fix_constant {type_tag, idx} =
	    case type_tag of
		Byte   => Classdecl.Byte   (getint idx)
	      | Char   => Classdecl.Char   (getInt idx)
	      | Double => Classdecl.Double (getDouble idx)
	      | Float  => Classdecl.Float  (getFloat idx)
	      | Int    => Classdecl.Int    (getInt idx)
	      | Long   => Classdecl.Long   (getLong idx)
	      | Short  => Classdecl.Short  (getint idx)
	      | Bool   => Classdecl.Bool   (getint idx)
	      | String => Classdecl.String (getString idx)

	fun fix_element e =
	    case e of
		Const_Value c =>
		   Classdecl.Const_Value (fix_constant c)
	      | Enum_Const_Value {type_name, const_name} =>
		   Classdecl.Enum_Const_Value {type_name = getUtf8 type_name,
					       const_name = getUtf8 const_name}
	      | Class_Info i => Classdecl.Class_Info (getUtf8 i)
	      | Annot_Value a => Classdecl.Annot_Value (fix_annotation a)
	      | Array_Value l => Classdecl.Array_Value (map fix_element l)

	and fix_pair {name_index, value} =
	    {name = getUtf8 name_index, value = fix_element value}

	and fix_annotation {type_index, values} =
	    {atype = getUtf8 type_index, values = map fix_pair values}

	fun fixEncl meth =
	    case meth of NONE => NONE
		       | SOME j =>
			 let in
			     case lookup j
			      of CPnametype {name, desc} =>
				 SOME {mname = getUtf8 name, msig = getUtf8 desc}
			       | _ => cperror "fixEncl"
			 end

	fun attrName (a: attribute) =
	    let
		val idx =
		    case a of
			CONSTVAL      x => #attr x
		      | CODE          x => #attr x
		      | EXNS          x => #attr x
		      | INNER         x => #attr x
		      | ENCLOSING     x => #attr x
		      | SYNTHETIC     x => #attr x
		      | SIGNATURE     x => #attr x
		      | SRCFILE       x => #attr x
		      | LINENUM       x => #attr x
		      | LOCALVAR      x => #attr x
		      | LOCALVARTYPES x => #attr x
		      | DEPRECATED    x => #attr x
		      | RTV_ANN       x => #attr x
		      | RTI_ANN       x => #attr x
		      | RTV_PARAM_ANN x => #attr x
		      | RTI_PARAM_ANN x => #attr x
		      | ANN_DEFAULT   x => #attr x
		      | ATTR          x => #attr x
	    in
		case lookup idx of
		    CPutf8 str => str ^ "\n"
		  | _ => classError "attrName: wrong type in constant pool"
	    end


	fun getAttr attr =
	    let
		fun err attr = error (getUtf8 attr ^ " attribute found outside code")
	    in
		case attr of
		    CONSTVAL {value, ...} => Classdecl.CONSTVAL (getConst value)
		  | CODE _                => decompileBytecode attr pool
		  | EXNS {exns, ...}      => Classdecl.EXNS (List.map getClass exns)
		  | INNER{classes, ...}   => Classdecl.INNER (map fix_inner classes)
		  | ENCLOSING {class, method, ...} =>
		       Classdecl.ENCLOSING {class=getClass class,
					    method = Option.map fixMeth method}
		  | SYNTHETIC _           => Classdecl.SYNTHETIC
		  | SIGNATURE {sign, ...} =>
		      Classdecl.SIGNATURE (getUtf8 sign)
		  | SRCFILE {file, ...}   => Classdecl.SRCFILE (getUtf8 file)
		  | LINENUM  {attr,...}      => err attr
		  | LOCALVAR {attr,...}      => err attr
		  | LOCALVARTYPES {attr,...} => err attr
		  | DEPRECATED _          => Classdecl.DEPRECATED
		  | RTV_ANN {annotations, ...} =>
		       Classdecl.RTV_ANN (map fix_annotation annotations)
		  | RTI_ANN {annotations, ...} =>
		       Classdecl.RTI_ANN (map fix_annotation annotations)
		  | RTV_PARAM_ANN {annotations, ...} =>
		       Classdecl.RTV_PARAM_ANN (map (map fix_annotation) annotations)
		  | RTI_PARAM_ANN {annotations, ...} =>
		       Classdecl.RTI_PARAM_ANN (map (map fix_annotation) annotations)
		  | ANN_DEFAULT {default_value, ...} =>
		       Classdecl.ANN_DEFAULT (fix_element default_value)
		  | ATTR {attr, info} =>
		       Classdecl.ATTR {attr=getUtf8 attr, info=info}
    end

fun getField {flags, name, desc, attrs} =
    let val ty = case Jvmtype.scanTypeDesc (getUtf8 desc) of
                     SOME s => s
                   | _ => error "can't find type in getField"
    in
        {flags = fieldFlagsFromWord flags,
         name = getUtf8 name,
         ty = ty,
         attrs = map getAttr attrs}
    end

fun getMethod {flags, name, desc, attrs} =
    let
	val msig = case Jvmtype.scanMethodDesc (getUtf8 desc) of
                       SOME s => s
                     | _ => error "can't find type in getMethod"
	val flags' = methodFlagsFromWord flags
	val attrs' = map getAttr attrs
	    handle e => (print ("exception in getAttr in getMethod ["
				^ getUtf8 name ^ " " ^ getUtf8 desc ^ "]\n");
			 raise e)
    in
        {flags = flags',
         name = getUtf8 name,
         msig = msig,
         attrs = attrs'}
    end
    in
        {major = Word16.toInt major,
	 minor = Word16.toInt minor,
	 flags = classFlagsFromWord flags,
         this = getClass this,
         super = Option.map getClass super,
         ifcs = List.map getClass ifcs,
         fdecls = List.map getField fields,
         mdecls = List.map getMethod methods,
         attrs = List.map getAttr attrs} : class_decl
end

end (* local *)

