(* Constpool.sml
 *
 * Peter Bertelsen
 * December 1997
 *)

exception InvalidEntry of string
exception UnsupportedCPitem of string;

fun entryError s = raise InvalidEntry("" ^ s)

datatype index = IDX of int

val maxPool = 0xffff

fun makeIndex k = if 0 <= k andalso k < maxPool then IDX k
		  else raise Overflow

fun indexValue (IDX i) = i

fun add_idx_option 0 = NONE
  | add_idx_option n = SOME (makeIndex n)

(* NOTE: index values produced by the insert functions will never
 * exceed maxPool-1, due to the use of Array.sub in insert.
 *)

datatype entry =
    CPutf8 of string
  | CPint of Int32.int
  | CPfloat of Word8Vector.vector
  | CPlong of Int64.int
  | CPdouble of Word8Vector.vector
  | CPclass of index
  | CPstring of index
  | CPfieldref of {class: index, nameType: index}
  | CPmethodref of {class: index, nameType: index}
  | CPimethodref of {class: index, nameType: index}
  | CPnametype of {name: index, desc: index}
  | CPunused  (* placeholder *)

(* NOTE: float and double entries are stored as byte vectors, rather
 * than Real32.real/Real64.real values, since polymorphic equality is
 * used in the constant pool implementation.
 *)

datatype pool =
    CP of {pool  : entry Array.array ref,
	   cache : (entry, index) Polyhash.hash_table,
	   count : int ref}

(* NOTE: the first valid entry in the constant pool is pool[1]. *)

val exnFind = Bytecode.InternalError
              "insert: unexpected use of Polyhash.find"

fun create () =
    CP {pool  = ref (Array.array(maxPool, CPunused)),
	cache = Polyhash.mkPolyTable(1021, exnFind),
	count = ref 1}

fun insert size (CP {pool, cache, count}) entry =
    let val count' = !count
	val index  = IDX count'
    in
	case Polyhash.peekInsert cache (entry, index) of
	    NONE => (Array.update(!pool, count', entry);
		     count:= count' + size;
		     index)
	  | SOME index' => index'
    end

val insert1 = insert 1   (* insert constant occupying one entry *)
val insert2 = insert 2   (* insert constant occupying two entries *)

fun insUtf8 cp str = insert1 cp (CPutf8 str)

fun insInt cp k = insert1 cp (CPint k)

fun insFloat cp r = insert1 cp (CPfloat (Real32.toBytes r))

fun insLong cp k = insert2 cp (CPlong k)

fun insDouble cp r = insert2 cp (CPdouble (Real64.toBytes r))

fun insString cp str =
    let val ins = insert1 cp
    in
	ins(CPstring (ins(CPutf8 str)))
    end

fun insClass' ins class =
    let val name = Jvmtype.qualName class
    in
	ins(CPclass (ins(CPutf8 name)))
    end

fun insClass cp = insClass'(insert1 cp)

fun insArrayClass cp t =
    let open Jvmtype
	val ins = insert1 cp
	val desc = (case t of
			Tarray t'    => typeDesc t
		      | Tclass class => qualName class
		      | _ => entryError "insArrayClass: invalid array class")
    in
	ins(CPclass (ins(CPutf8 desc)))
    end

fun insNameType' ins name desc = ins(CPnametype {name = ins(CPutf8 name),
						 desc = ins(CPutf8 desc)})

fun insNameType cp {name, desc} = insNameType' (insert1 cp) name desc

fun insFieldref cp {class, name, ty} =
    let val ins  = insert1 cp
	val desc = Jvmtype.typeDesc ty
    in
	ins(CPfieldref {class    = insClass' ins class,
			nameType = insNameType' ins name desc})
    end

fun insMethodref cp {class, name, msig} =
    let val ins  = insert1 cp
	val desc = Jvmtype.methodDesc msig
    in
	ins(CPmethodref {class    = insClass' ins class,
			 nameType = insNameType' ins name desc})
    end

fun insIMethodref cp {class, name, msig} =
    let val ins = insert1 cp
	val desc = Jvmtype.methodDesc msig
    in
	ins(CPimethodref {class    = insClass' ins class,
			  nameType = insNameType' ins name desc})
    end

local
    open Bytecode
in
    fun insConst cp value =
	(case value of
	     Cint i    => insInt cp i
	   | Cfloat f  => insFloat cp f
	   | Clong l   => insLong cp l
	   | Cdouble d => insDouble cp d
	   | Cstring s => insString cp s
	   | _ => entryError "insConst: invalid constant")
end

fun lookup (CP {pool, count, ...}) (IDX i) =
    if 1 <= i andalso i < !count then
	Array.sub(!pool, i)
    else
	raise Subscript

fun charToUtf8 (c, res) =  (* convert character to UTF-8 representation *)
    let val k = ord c
    in
	if 1 <= k andalso k <= 127 then
	    (Word8.fromInt k) :: res
	else
	 (* if k <= 2047 then *)
		let
		    val x = 0xc0 + (k div 0x40)
		    val y = 0x80 + (k mod 0x40)
		in
		    (Word8.fromInt x) :: (Word8.fromInt y) :: res
		end
	 (* else
		let
		    val k' = k div 0x40
		    val x  = 0xe0 + (k' div 0x40)
		    val y  = 0x80 + (k' mod 0x40)
		    val z  = 0x80 + (k  mod 0x40)
		in
		    (Word8.fromInt x) :: (Word8.fromInt y) ::
		    (Word8.fromInt z) :: res
		end *)
         (* NOTE: the latter section will be necessary if WideChar.char is
	  * used for characters
	  *)
    end

fun emit out (CP {pool, count, ...}) =
    let val emitInt  = Int32.emit out
	val emitLong = Int64.emit out
	val emitU2   = Word16.emit out
	val emitU2i  = emitU2 o Word16.fromInt

	fun emitIndex (IDX i) = emitU2i i

	fun emit' (CPutf8 s) =
	        let val cs = CharVector.foldr charToUtf8 [] s
		in
		    out 0w1;   (* CONSTANT_Utf8 *)
		    emitU2i(length cs);
		    List.app out cs
		end
	  | emit' (CPint i) =
		(out 0w3;   (* CONSTANT_Integer *)
		 emitInt i)
	  | emit' (CPfloat bs) =
		(out 0w4;   (* CONSTANT_Float *)
		 Word8Vector.app out bs)
	  | emit' (CPlong l) =
		(out 0w5;   (* CONSTANT_Long *)
		 emitLong l)
	  | emit' (CPdouble bs) =
		(out 0w6;   (* CONSTANT_Double *)
		 Word8Vector.app out bs)
	  | emit' (CPclass i) =
		(out 0w7;   (* CONSTANT_Class *)
		 emitIndex i)
	  | emit' (CPstring i) =
		(out 0w8;   (* CONSTANT_String *)
		 emitIndex i)
	  | emit' (CPfieldref {class = c, nameType = nt}) =
		(out 0w9;   (* CONSTANT_Fieldref *)
		 emitIndex c;
		 emitIndex nt)
	  | emit' (CPmethodref {class = c, nameType = nt}) =
		(out 0w10;   (* CONSTANT_Methodref *)
		 emitIndex c;
		 emitIndex nt)
	  | emit' (CPimethodref {class = c, nameType = nt}) =
		(out 0w11;   (* CONSTANT_IMethodref *)
		 emitIndex c;
		 emitIndex nt)
	  | emit' (CPnametype {name = n, desc = d}) =
		(out 0w12;   (* CONSTANT_NameAndType *)
		 emitIndex n;
		 emitIndex d)
          | emit' CPunused = ()
    in
	emitU2i((!count));
	Array.appi (fn (_, e) => emit' e) (!pool, 1, SOME(!count))
    end

(* LK 2002 .... added some *)
fun createPool (pool,cache,count) =
    CP { pool=pool,
	 cache=cache,
	 count=count }

(* ============ Extracting stuff from the constant pool ============ *)

(* ======== Conversion of byte-array slices to Word*/Int* values ======== *)

fun parseU2_w16 v n =
    let val u2_vector = Word8Vector.extract (v, n, SOME 2)
    in  ( valOf (Word16.fromBytes u2_vector),
	  n+2 )
    end

fun parseU4_w32 v n =
    let val u4_vector = Word8Vector.extract (v, n, SOME 4)
    in  ( valOf (Word32.fromBytes u4_vector),
	  n+4 )
    end

fun parseU4_i32 v n =
    let val u4_vector = Word8Vector.extract (v, n, SOME 4)
    in  ( valOf (Int32.fromBytes u4_vector),
	  n+4 )
    end

fun parseU8_i64 v n =
    let val u8_vector = Word8Vector.extract (v, n, SOME 8)
    in  ( valOf (Int64.fromBytes u8_vector),
	  n+8 )
    end


(* ========= Convserion of byte-array slices to int values ======== *)
(* Only used in Decompile.sml *)

fun parseU1_int v n = (* unsigned byte -> int *)
    let val new_index = n+1
	val b = Word8Vector.sub (v,n)
    in  (Word8.toInt b, new_index)
    end

fun parseU1_intX v n = (* signed byte -> int *)
    let val new_index = n+1
	val b = Word8Vector.sub (v,n)
    in  (Word8.toIntX b, new_index)
    end

fun parseU2_int v n = (* unsigned 2-byte word -> int *)
    let val (w16, new_index) = parseU2_w16 v n
    in
	( Word16.toInt w16, new_index )
    end

fun parseU2_intX v n = (* signed 2-byte word -> int *)
    let val (w16, new_index) = parseU2_w16 v n
    in
	( Word16.toIntX w16, new_index )
    end

fun parseU4_int v n = (* 4-byte word -> unsigned int *)
    let val (w32, new_index) = parseU4_w32 v n
    in
	( Word32.toInt w32, new_index )
    end

fun parseU4_intX v n = (* 4-byte word -> signed int *)
    let val (w32, new_index) = parseU4_w32 v n
    in
	( Word32.toIntX w32, new_index )
    end


(* ======== Conversion of byte-array slices to Index values ======== *)

fun parseU1_idx v n =
    let val (n, new_index) = parseU1_int v n
    in  ( makeIndex n,
	  new_index )
    end

fun parseU2_idx v n =
    let val (n, new_index) = parseU2_int v n
    in  ( makeIndex n,
	  new_index )
    end


(* ======== Extraction of array slices ======== *)

fun parseU4_w v n =
    let val u4_vector = Word8Vector.extract (v, n, SOME 4)
    in  ( u4_vector,
	  n+4 )
    end

fun parseU8_w v n =
    let val u8_vector = Word8Vector.extract (v, n, SOME 8)
    in  ( u8_vector,
	  n+8 )
    end


(* ======== Extracting various objects from the constant pool ======== *)

(* apply <parser> to <vector> <number> times *)
(* NB. only for parsers which yield strings  *)
fun <*> parser (result,vector_index,number) vector =
    if number > 0
    then let val (s,new_vi) = parser vector vector_index
	 in  <*> parser (result ^ s, new_vi, number-1) vector
	 end
    else (result,vector_index)

fun parseUtf8 v n =
    let (* val tag     = "Utf8" *)
        val (len,i) = parseU2_int v n
        fun intWord8toS (tokens,n) = (String.str o Char.chr o Word8.toInt o Word8Vector.sub) (tokens, n)
        fun parseChar v m  = (intWord8toS (v,m),m+1)
        val (s,i)   = <*> parseChar ("",i,len) v
    in  ( CPutf8 s,
	  i,
	  1 )
    end

fun parseMethodref v n =
    let (* val tag = "MethodRef" *)
	val (class_index        ,i) = parseU2_idx v n
	val (name_and_type_index,i) = parseU2_idx v i
    in  ( CPmethodref { class    = class_index,
			nameType = name_and_type_index },
	  i,
	  1 )
    end

fun parseFieldref v n =
    let (* val tag = "FieldRef" *)
	val (class_index        ,i) = parseU2_idx v n
	val (name_and_type_index,i) = parseU2_idx v i
    in  ( CPfieldref { class    = class_index,
                       nameType = name_and_type_index },
	  i,
	  1 )
    end

fun parseInterfaceMethodref v n =
    let (* val tag = "InterfaceMethodRef" *)
	val (class_index        ,i) = parseU2_idx v n
	val (name_and_type_index,i) = parseU2_idx v i
    in  ( CPimethodref { class    = class_index,
			 nameType = name_and_type_index },
	  i,
	  1 )
    end

fun parseNameAndType v n =
    let (* val tag = "NameAndType" *)
	val (name_index      ,i) = parseU2_idx v n
	val (descriptor_index,i) = parseU2_idx v i
    in  ( CPnametype { name = name_index,
		       desc = descriptor_index },
	  i,
	  1 )
    end

fun parseClass v n =
    let (* val tag = "Class" *)
	val (name_index,i) = parseU2_idx v n
    in  ( CPclass name_index,
	  i,
	  1 )
    end


fun parseString v n =
    let (* val tag = "String" *)
	val (name_index,i) = parseU2_idx v n
    in  ( CPstring name_index,
	  i,
	  1 )
    end


fun parseInteger v n =
    let (* val tag = "Integer" *)
	val (int_vector,i) = parseU4_i32 v n
    in  ( CPint int_vector,
	  i,
	  1 )
    end

fun parseFloat v n =
    let (* val tag = "Float" *)
	val (float_vector,i) = parseU4_w v n
    in  ( CPfloat float_vector,
	  i,
	  1 )
    end

fun parseLong v n =
    let (* val tag = "Long" *)
	val (long_vector,i) = parseU8_i64 v n
    in  ( CPlong long_vector,
	  i,
	  2 )
    end

fun parseDouble v n =
    let (* val tag = "Double" *)
	val (double_vector,i) = parseU8_w v n
    in  ( CPdouble double_vector,
	  i,
	  2 )
    end

