(* Constpool.sig
 *
 * Peter Bertelsen
 * December 1997
 *)

exception InvalidEntry of string

type pool

eqtype index

datatype entry =
    CPutf8 of string
  | CPint of Int32.int
  | CPfloat of Word8Vector.vector
  | CPlong of Int64.int
  | CPdouble of Word8Vector.vector
  | CPclass of index
  | CPstring of index
  | CPfieldref of {class: index, nameType: index}
  | CPmethodref of {class: index, nameType: index}
  | CPimethodref of {class: index, nameType: index}
  | CPnametype of {name: index, desc: index}
  | CPunused  (* placeholder *)

val createPool :  (entry Array.array ref * (entry, index) Polyhash.hash_table * int ref) -> pool
val create : unit -> pool

val makeIndex  : int -> index
val indexValue : index -> int

val insUtf8       : pool -> string              -> index
val insInt        : pool -> Int32.int           -> index
val insFloat      : pool -> Real32.real         -> index
val insLong       : pool -> Int64.int           -> index
val insDouble     : pool -> Real64.real         -> index
val insClass      : pool -> Jvmtype.jclass      -> index
val insArrayClass : pool -> Jvmtype.jtype       -> index
val insString     : pool -> string              -> index
val insNameType   : pool -> {name : string,
			     desc : string}     -> index
val insFieldref   : pool -> Bytecode.field_ref  -> index
val insMethodref  : pool -> Bytecode.method_ref -> index
val insIMethodref : pool -> Bytecode.method_ref -> index
val insConst      : pool -> Bytecode.jvm_const  -> index

val lookup : pool -> index -> entry
val emit   : (Word8.word -> unit) -> pool -> unit

val add_idx_option : int -> index option

val parseU2_w16  : Word8Vector.vector -> int -> Word16.word * int
val parseU4_w32  : Word8Vector.vector -> int -> Word32.word * int
val parseU4_i32  : Word8Vector.vector -> int -> Int32.int * int
val parseU8_i64  : Word8Vector.vector -> int -> Int64.int * int
val parseU1_int  : Word8Vector.vector -> int -> int * int
val parseU1_intX : Word8Vector.vector -> int -> int * int
val parseU2_int  : Word8Vector.vector -> int -> int * int
val parseU2_intX : Word8Vector.vector -> int -> int * int
val parseU4_int  : Word8Vector.vector -> int -> int * int
val parseU4_intX : Word8Vector.vector -> int -> int * int
val parseU1_idx  : Word8Vector.vector -> int -> index * int
val parseU2_idx  : Word8Vector.vector -> int -> index * int
val parseU4_w    : Word8Vector.vector -> int -> Word8Vector.vector * int
val parseU8_w    : Word8Vector.vector -> int -> Word8Vector.vector * int

val parseUtf8               : Word8Vector.vector -> int -> entry * int * int
val parseMethodref          : Word8Vector.vector -> int -> entry * int * int
val parseFieldref           : Word8Vector.vector -> int -> entry * int * int
val parseInterfaceMethodref : Word8Vector.vector -> int -> entry * int * int
val parseNameAndType        : Word8Vector.vector -> int -> entry * int * int
val parseClass              : Word8Vector.vector -> int -> entry * int * int
val parseString             : Word8Vector.vector -> int -> entry * int * int
val parseInteger            : Word8Vector.vector -> int -> entry * int * int
val parseFloat              : Word8Vector.vector -> int -> entry * int * int
val parseLong               : Word8Vector.vector -> int -> entry * int * int
val parseDouble             : Word8Vector.vector -> int -> entry * int * int
