(* Classfile.sml
 *
 * Peter Bertelsen
 * December 1997
 *)

open Classdecl Constpool Bytecode

(* NOTE: it is assumed that Constpool.indexValue and the labelMap
 * returned by Emitcode.emit return integers in the range
 * [0, 0xffff].
 *)

exception InvalidClass of string

fun classError s = raise InvalidClass("Classfile." ^ s)

fun bug s = raise Bytecode.InternalError("Classfile." ^ s)

type exn_hdl =
    {start : Word16.word,
     stop  : Word16.word,
     entry : Word16.word,
     catch : index option}

type inner_class_info =
    {inner_info  : index option,
     outer_info  : index option,
     inner_name  : index option,
     inner_flags : Word16.word}

type line_number_info =
    {start : Word16.word,
     line  : Word16.word}

type local_var_info =
    {start  : Word16.word,
     length : Word16.word,
     name   : index,
     desc   : index,
     index  : Word16.word}

type local_var_type_info =
     {start  : Word16.word,
      length : Word16.word,
      name   : index,
      sign   : index,
      index  : Word16.word}

datatype constant =
	 Byte
       | Char
       | Double
       | Float
       | Int
       | Long
       | Short
       | Bool
       | String

datatype element =
     Const_Value of {type_tag: constant, idx: index}
   | Enum_Const_Value of
        {type_name: index, const_name: index}
   | Class_Info  of index
   | Annot_Value of annotation
   | Array_Value of element list

withtype annotation =
	  {type_index : index,
	   values     : {name_index: index,
			 value: element} list
	  }


datatype attribute =
    CONSTVAL of { attr  : index,
		  value : index}
  | CODE of
    {attr   : index,
     stack  : Word16.word,
     locals : Word16.word,
     code   : Word8Vector.vector,
     hdls   : exn_hdl list,
     attrs  : attribute list}
  | EXNS of
    {attr : index,
     exns : index list}
  | INNER of
    {attr    : index,
     classes : inner_class_info list}
  | ENCLOSING of
    {attr: index,   (* Java 5 *)
     class: index,
     method: index option}
  | SYNTHETIC of
    {attr : index}
  | SIGNATURE of
    {attr: index,   (* Java 5 *)
     sign: index}
  | SRCFILE of
    {attr : index,
     file : index}
  | LINENUM of
    {attr  : index,
     lines : line_number_info list}
  | LOCALVAR of
    {attr : index,
     vars : local_var_info list}
  | LOCALVARTYPES of
    {attr: index,   (* Java 5 *)
     vartypes: local_var_type_info list}
  | DEPRECATED of
    {attr : index}  (* Java 5 *)
  | RTV_ANN of
    {attr: index,   (* Java 5 *)
     annotations: annotation list}
  | RTI_ANN of
    {attr: index,   (* Java 5 *)
     annotations: annotation list}
  | RTV_PARAM_ANN of
    {attr: index,   (* Java 5 *)
     annotations: annotation list list}
  | RTI_PARAM_ANN of
    {attr: index,   (* Java 5 *)
     annotations: annotation list list}
  | ANN_DEFAULT of   (* Java 5 *)
    {attr: index,
     default_value: element}
  | ATTR of
    {attr : index,
     info : Word8Vector.vector}

type member =
    {flags : Word16.word,
     name  : index,
     desc  : index,
     attrs : attribute list}

type class_file =
    {magic   : Word32.word,
     minor   : Word16.word,
     major   : Word16.word,
     pool    : pool,
     flags   : Word16.word,
     this    : index,
     super   : index option,
     ifcs    : index list,
     fields  : member list,
     methods : member list,
     attrs   : attribute list}

val magic' = Word8Vector.fromList[0wxca, 0wxfe, 0wxba, 0wxbe]
val magic  = valOf(Word32.fromBytes magic')
(* val minor  = Word16.fromWord 0w3 *)
(* val major  = Word16.fromWord 0w45 *)





(* ---------------- Conversion from Classdecl to Classfile ---------------- *)

local
fun toWordC' flag =
	case flag of
	    C_ACCpublic       => 0wx0001
	  | C_ACCprivate      => 0wx0002
	  | C_ACCprotected    => 0wx0004
	  | C_ACCstatic       => 0wx0008
	  | C_ACCfinal        => 0wx0010
	  | C_ACCsuper        => 0wx0020
	  | C_ACCabstract     => 0wx0400
	  | C_ACCinterface    => 0wx0200
	  | C_ACCsynthetic    => 0wx1000
	  | C_ACCannotation   => 0wx2000
	  | C_ACCenum         => 0wx4000

fun toWordC (flag, w) = Word.orb(toWordC' flag, w)
in
val classFlagsToWord = Word16.fromWord o (List.foldl toWordC 0w0)
end


local fun toWordF' flag =
	  case flag of
	      F_ACCpublic       => 0wx0001
	    | F_ACCprivate      => 0wx0002
	    | F_ACCprotected    => 0wx0004
	    | F_ACCstatic       => 0wx0008
	    | F_ACCfinal        => 0wx0010
	    | F_ACCvolatile     => 0wx0040
	    | F_ACCtransient    => 0wx0080
	    | F_ACCsynthetic    => 0wx1000
	    | F_ACCenum         => 0wx4000

    fun toWordF (flag, w) = Word.orb(toWordF' flag, w)
in
val fieldFlagsToWord = Word16.fromWord o (List.foldl toWordF 0w0)
end



local fun toWordM' flag =
	  case flag of
	      M_ACCpublic       => 0wx0001
	    | M_ACCprivate      => 0wx0002
	    | M_ACCprotected    => 0wx0004
	    | M_ACCstatic       => 0wx0008
	    | M_ACCfinal        => 0wx0010
	    | M_ACCsynchronized => 0wx0020
	    | M_ACCbridge       => 0wx0040
	    | M_ACCvarargs      => 0wx0080
	    | M_ACCnative       => 0wx0100
	    | M_ACCabstract     => 0wx0400
	    | M_ACCstrictfp     => 0wx0800
	    | M_ACCsynthetic    => 0wx1000

    fun toWordM (flag, w) = Word.orb(toWordM' flag, w)
in
val methodFlagsToWord = Word16.fromWord o (List.foldl toWordM 0w0)
end




(*NW 20030826*)
infixr 5 ::?
fun NONE ::? l = l
  | (SOME x) ::? l = x::l

fun flagSet word flag result = if Word.toInt (Word.andb(word,flag)) <> 0 then SOME result else NONE

fun classFlagsFromWord word =
    let val f = flagSet (Word16.toWord word) in
            f 0wx0001 C_ACCpublic
        ::? f 0wx0002 C_ACCprivate
	::? f 0wx0004 C_ACCprotected
        ::? f 0wx0008 C_ACCstatic
	::? f 0wx0010 C_ACCfinal
	::? f 0wx0200 C_ACCinterface
	::? f 0wx0400 C_ACCabstract
	::? f 0wx1000 C_ACCsynthetic
	::? f 0wx2000 C_ACCannotation
	::? f 0wx4000 C_ACCenum
        ::? []
    end

fun fieldFlagsFromWord word =
    let val f = flagSet (Word16.toWord word) in
            f 0wx0001 F_ACCpublic
        ::? f 0wx0002 F_ACCprivate
	::? f 0wx0004 F_ACCprotected
        ::? f 0wx0008 F_ACCstatic
	::? f 0wx0010 F_ACCfinal
	::? f 0wx0040 F_ACCvolatile
        ::? f 0wx0080 F_ACCtransient
	::? f 0wx1000 F_ACCsynthetic
	::? f 0wx4000 F_ACCenum
        ::? []
    end

fun methodFlagsFromWord word =
    let val f = flagSet (Word16.toWord word) in
            f 0wx0001 M_ACCpublic
        ::? f 0wx0002 M_ACCprivate
	::? f 0wx0004 M_ACCprotected
        ::? f 0wx0008 M_ACCstatic
	::? f 0wx0010 M_ACCfinal
	::? f 0wx0020 M_ACCsynchronized
	::? f 0wx0040 M_ACCbridge
        ::? f 0wx0080 M_ACCvarargs
	::? f 0wx0100 M_ACCnative
	::? f 0wx0400 M_ACCabstract
	::? f 0wx0800 M_ACCstrictfp
	::? f 0wx1000 M_ACCsynthetic
        ::? []
    end

val isU2 = Emitcode.isU2

fun fromClassDecl cp {major, minor, flags, this, super, ifcs, fdecls, mdecls, attrs} =
    let
	val insUtf8'  = insUtf8  cp
	val insConst' = insConst cp
	val insClass' = insClass cp

	fun insExnHandler labelMap {start, stop, entry, catch} =
	    let
		val labelMap' = Word16.fromInt o labelMap
	    in
		{start = labelMap' start,
		 stop  = labelMap' stop,
		 entry = labelMap' entry,
		 catch = Option.map insClass' catch}
	    end

	fun insLnumInfo labelMap {start, line} =
	    if isU2 line then
		{start = Word16.fromInt(labelMap start),
		 line  = Word16.fromInt line}
	    else classError
		 "fromClassDecl.insLnumInfo: line number out of range"

	fun insLvarInfo labelMap {from, thru, name, ty, index} =
	    let val start' = labelMap from
		val length = labelMap thru - start'
		val index' = Localvar.toInt index
	    in
		if isU2 length then
		    if isU2 index' then
			{start  = Word16.fromInt start',
			 length = Word16.fromInt length,
			 name   = insUtf8' name,
			 desc   = insUtf8'(Jvmtype.typeDesc ty),
			 index  = Word16.fromInt index'}
		    else classError
			"fromClassDecl.insLvarInfo: index out of range"
		else classError
		     "fromClassDecl.insLvarInfo: length out of range"
	    end


	fun insLvarTypeInfo labelMap {from, thru, name, sign, index} =
	    let val start' = labelMap from
		val length = labelMap thru - start'
		val index' = Localvar.toInt index
	    in
		if isU2 length then
		    if isU2 index' then
			{start  = Word16.fromInt start',
			 length = Word16.fromInt length,
			 name   = insUtf8' name,
			 sign   = insUtf8' sign,
			 index  = Word16.fromInt index'}
		    else classError
			"fromClassDecl.insLvarTypeInfo: index out of range"
		else classError
		     "fromClassDecl.insLvarTypeInfo: length out of range"
	    end

	fun insInnerClassInfo {inner_info,
			       outer_info,
			       inner_name,
			       inner_flags}
	  = {inner_info =  Option.map insClass' inner_info,
	     outer_info =  Option.map insClass' outer_info,
	     inner_name =  Option.map insUtf8' inner_name,
	     inner_flags = classFlagsToWord inner_flags}


	fun insMdesc {mname, msig} =
 	    insNameType cp {name = mname, desc = Jvmtype.methodDesc msig}

	fun insConstval c =
	    let
		fun insint i = insInt cp (Int32.fromInt i)
	    in case c of
		   Classdecl.Byte b   => {type_tag = Byte,   idx = insint b}
		 | Classdecl.Char i   => {type_tag = Char,   idx = insInt cp i}
		 | Classdecl.Double d => {type_tag = Double, idx = insDouble cp d}
		 | Classdecl.Float f  => {type_tag = Float,  idx = insFloat cp f}
		 | Classdecl.Int i    => {type_tag = Int,    idx = insInt cp i}
		 | Classdecl.Long l   => {type_tag = Long,   idx = insLong cp l}
		 | Classdecl.Short s  => {type_tag = Short,  idx = insint s}
		 | Classdecl.Bool b   => {type_tag = Bool,   idx = insint b }
		 | Classdecl.String s => {type_tag = String, idx = insString cp s}
	    end


	fun insElement e =
	    case e of
		Classdecl.Const_Value c => Const_Value (insConstval c)
	      | Classdecl.Enum_Const_Value {type_name, const_name} =>
		   Enum_Const_Value {type_name = insUtf8' type_name,
				     const_name = insUtf8' const_name}
	      | Classdecl.Class_Info s => Class_Info (insUtf8' s)
	      | Classdecl.Annot_Value a => Annot_Value (insAnnot a)
	      | Classdecl.Array_Value l => Array_Value (map insElement l)

	and insPair {name, value} = {name_index = insUtf8' name, value = insElement value}

	and insAnnot {atype, values} = {type_index = insUtf8' atype, values = map insPair values}

	fun insAttr' labelMap ty attr =
	    case attr of
		Classdecl.CONSTVAL value =>
		    CONSTVAL {attr  = insUtf8' "ConstantValue",
			  value = insConst' value}
	      | Classdecl.CODE {stack, locals, code, hdls, attrs} =>
		let val (code', labelMap) = Emitcode.emit cp ty code
		    val insExnHandler'    = insExnHandler labelMap
		    val insAttr''         = insAttr' labelMap ty
		in
		    if isU2 stack then
			if isU2 locals then
			    CODE {attr   = insUtf8' "Code",
				  stack  = Word16.fromInt stack,
				  locals = Word16.fromInt locals,
				  code   = code',
				  hdls   = List.map insExnHandler' hdls,
				  attrs  = List.map insAttr'' attrs}
			else classError
				 "insAttr [CODE]: max_locals out of range"
		    else classError "insAttr [CODE]: max_stack out of range"
		end

              | Classdecl.EXNS exns =>
		   EXNS {attr = insUtf8' "Exceptions",
		      exns = List.map insClass' exns}
	      | Classdecl.ENCLOSING {class, method} =>
                   ENCLOSING {attr = insUtf8' "EnclosingMethod",
			      class = insClass' class,
			      method = Option.map insMdesc method}
              | Classdecl.INNER classes =>
                   INNER {attr = insUtf8' "InnerClasses",  (* INNER *)
                       classes = List.map insInnerClassInfo classes}
              | Classdecl.SYNTHETIC =>
                   SYNTHETIC {attr = insUtf8' "Synthetic"}
	      | Classdecl.SIGNATURE s =>
		   SIGNATURE {attr = insUtf8' "Signature", sign = insUtf8' s}
	      | Classdecl.SRCFILE file =>
	           SRCFILE {attr = insUtf8' "SourceFile",
			 file = insUtf8' file}
	      | Classdecl.LINENUM lines =>
		   LINENUM {attr  = insUtf8' "LineNumberTable",
			 lines = List.map (insLnumInfo labelMap) lines}
	      | Classdecl.LOCALVAR vs =>
		   LOCALVAR {attr = insUtf8' "LocalVariableTable",
			  vars = List.map (insLvarInfo labelMap) vs}
	      | Classdecl.LOCALVARTYPES vs =>
		   LOCALVARTYPES {attr = insUtf8' "LocalVariableTypeTable",
				  vartypes = List.map (insLvarTypeInfo labelMap) vs}
              | Classdecl.DEPRECATED =>
                   DEPRECATED {attr = insUtf8' "Deprecated"}
	      | Classdecl.RTV_ANN l =>
		   RTV_ANN {attr = insUtf8' "RuntimeVisibleAnnotations",
			    annotations = map insAnnot l }
	      | Classdecl.RTI_ANN l =>
		   RTI_ANN {attr = insUtf8' "RuntimeInvisibleAnnotations",
			    annotations = map insAnnot l }
	      | Classdecl.RTV_PARAM_ANN l =>
		   RTV_PARAM_ANN {attr = insUtf8' "RuntimeVisibleParameterAnnotations",
			    annotations = map (map insAnnot) l }
	      | Classdecl.RTI_PARAM_ANN l =>
		   RTI_PARAM_ANN {attr = insUtf8' "RuntimeInvisibleParameterAnnotations",
			    annotations = map (map insAnnot) l }
	      | Classdecl.ANN_DEFAULT v => ANN_DEFAULT {attr = insUtf8' "AnnotationDefault",
							default_value = insElement v}
	      | Classdecl.ATTR {attr, info} =>
		   ATTR {attr = insUtf8' attr,
		      info = info}


	fun emptyLblMap _ = classError
	                    "fromClassDecl.insAttr: invalid nested attributes"

	val insAttr = insAttr' emptyLblMap

	    (* NOTE: when insAttr is used for inserting a method
	     * attribute, the second argument specifies the (optional)
	     * return type for the method; when insAttr is used for
	     * inserting a field or class attribute, the faked `return
	     * type' is NONE.
	     *)

	fun insField {flags, name, ty, attrs} =
	    {flags = fieldFlagsToWord flags,
	     name  = insUtf8' name,
	     desc  = insUtf8'(Jvmtype.typeDesc ty),
	     attrs = List.map (insAttr NONE) attrs}

	fun insMethod {flags, name, msig, attrs} =
	    let val (_, returnTy) = msig
	    in
		{flags = methodFlagsToWord flags,
		 name  = insUtf8' name,
		 desc  = insUtf8'(Jvmtype.methodDesc msig),
		 attrs = List.map (insAttr returnTy) attrs}
	    end
    in
	(* NOTE: cp need not be completed before we start building the
	 * resulting class_file since Constpool.pool is imperative. *)
	{magic   = magic,
	 minor   = Word16.fromInt minor,
	 major   = Word16.fromInt minor,
	 pool    = cp,
	 flags   = classFlagsToWord flags,
	 this    = insClass' this,
	 super   = (case super of
			NONE => NONE
		      | SOME class => SOME (insClass' class)),
	 ifcs    = List.map insClass' ifcs,
	 fields  = List.map insField fdecls,
	 methods = List.map insMethod mdecls,
	 attrs   = List.map (insAttr NONE) attrs}
    end

(* Former location of toClassDecl *)



(* ---------------------------- Classfile output ----------------------------*)

val word16_0 = Word16.fromWord 0w0
val word32_2 = Word32.fromWord 0w2
val word32_4 = Word32.fromWord 0w4

fun emitClassFile out {magic, minor, major, pool, flags, this,
		       super, ifcs, fields, methods, attrs} =
    let
	val emitU2      = Word16.emit out
	val emitU2i     = emitU2 o Word16.fromInt
	val emitU4      = Word32.emit out
	val emitU4i     = emitU4 o Word32.fromInt
	val emitCpIndex = emitU2i o Constpool.indexValue
	val emitU1i     = out o Word8.fromInt

fun attrName (a: attribute) =
    let
	val idx =
	    case a of
		CONSTVAL      x => #attr x
	      | CODE          x => #attr x
	      | EXNS          x => #attr x
	      | INNER         x => #attr x
	      | ENCLOSING     x => #attr x
	      | SYNTHETIC     x => #attr x
	      | SIGNATURE     x => #attr x
	      | SRCFILE       x => #attr x
	      | LINENUM       x => #attr x
	      | LOCALVAR      x => #attr x
	      | LOCALVARTYPES x => #attr x
	      | DEPRECATED    x => #attr x
	      | RTV_ANN       x => #attr x
	      | RTI_ANN       x => #attr x
	      | RTV_PARAM_ANN x => #attr x
	      | RTI_PARAM_ANN x => #attr x
	      | ANN_DEFAULT   x => #attr x
	      | ATTR          x => #attr x
    in
	case lookup pool idx of
	    CPutf8 str => str ^ "\n"
	  | _ => classError "attrName: wrong type in constant pool"
    end

fun emitCpIndexOpt  NONE    = emitU2 word16_0
  | emitCpIndexOpt (SOME i) = emitCpIndex i

fun emitExnHdl {start, stop, entry, catch} =
    (emitU2 start;
     emitU2 stop;
     emitU2 entry;
     emitCpIndexOpt catch)

fun emitLineNumInfo {start, line} =
    (emitU2 start;
     emitU2 line)

fun emitLocalVarInfo {start, length, name, desc, index} =
    (emitU2 start;
     emitU2 length;
     emitCpIndex name;
     emitCpIndex desc;
     emitU2 index)

fun emitLocalVarTypeInfo {start, length, name, sign, index} =
    (emitU2 start;
     emitU2 length;
     emitCpIndex name;
     emitCpIndex sign;
     emitU2 index)

fun emitInnerClassInfo {inner_info, outer_info, inner_name, inner_flags} =
    (emitCpIndexOpt inner_info;
     emitCpIndexOpt outer_info;
     emitCpIndexOpt inner_name;
     emitU2 inner_flags)


fun listSize f l = foldl (fn (h,s) => f h + s) 0 l

fun elt_size e =
    let val s1 =
	    case e of
		Const_Value _      => 2
	      | Enum_Const_Value _ => 4
	      | Class_Info _       => 2
	      | Annot_Value a      => ann_size a
	      | Array_Value l      => 2 + listSize elt_size l
    in
	1 + s1  (* 1 for tag *)
    end

and pair_size {name_index, value} = 2 + elt_size value

and ann_size {type_index, values} =
    4 + listSize pair_size values (* 2 for index, 2 for len values *)

fun attrSize (LINENUM {lines, ...}) = 8 + List.length lines * 4
  | attrSize (LOCALVAR {vars, ...}) = 8 + List.length vars * 10
  | attrSize (ATTR {info, ...}) = 2 + 4 + Word8Vector.length info
							     (* changed - kwxm*)
  | attrSize _ = classError "emit.attrSize: invalid nested attributes"

fun attrsSize attrs =
    List.foldl (fn (a, sz) => attrSize a + sz) 0 attrs

fun emitChar c = emitU1i (Char.ord c)

fun emitTag ty =
    let val c =
	    case ty of
		Const_Value {type_tag, ...} =>
		let in case type_tag of
			   Byte    => #"B"
			 | Char    => #"C"
			 | Double  => #"D"
			 | Float   => #"F"
			 | Int     => #"I"
			 | Long    => #"J"
			 | Short   => #"S"
			 | Bool    => #"Z"
			 | String  => #"s"
		end
	      | Enum_Const_Value _ => #"e"
	      | Class_Info _       => #"c"
	      | Annot_Value _      => #"@"
	      | Array_Value _      => #"["
    in
	emitChar c
    end


fun emitElement i =
    (emitTag i;
     case i of
	 Const_Value {type_tag,idx} => emitCpIndex idx
       | Enum_Const_Value {type_name, const_name} =>
	    (emitCpIndex type_name; emitCpIndex const_name)
       | Class_Info j => emitCpIndex j
       | Annot_Value a => emitAnnot a
       | Array_Value l => (emitU2i (List.length l); app emitElement l)
    )

and emitPair {name_index, value} =
    (emitCpIndex name_index; emitElement value)   (* Utf8' ???? *)

and emitAnnot {type_index, values} =
    (emitCpIndex type_index;
     emitU2i (List.length values);
     app emitPair values)

fun emitAnnotSize a b size_fn l = emitU4i (a + (b * length l) + listSize size_fn l)
(* For RT?_ANN and RT?_PARAM_ANN: need a extra bytes for list len, b extra bytes per entry *)

fun emitAnnotList l =
    let val atCount = List.length l
    in
	if isU2 atCount then
	     (emitU2i atCount;
	      List.app emitAnnot l)
	else classError "emitAnnotList: too many annotations"
    end


fun emitAttr a =
    case a of
	CONSTVAL {attr, value} =>
	(emitCpIndex attr;
	 emitU4 word32_2;
	 emitCpIndex value)
      | CODE {attr, stack, locals, code, hdls, attrs} =>
	let val codeLength = Word8Vector.length code
	    val hdlsCount  = List.length hdls
	in
	    if isU2 hdlsCount then
		(emitCpIndex attr;
		 emitU4i(2 + 2 + 4 + codeLength +
			 2 + hdlsCount * 8 +
			 2 + attrsSize attrs);
		 emitU2 stack;
		 emitU2 locals;
		 emitU4i codeLength;
		 Word8Vector.app out code;
		 emitU2i hdlsCount;
		 List.app emitExnHdl hdls;
		 emitAttrs attrs)
	    else classError
		     "emit.emitAttr [CODE]: too many exception handlers"
	end
      | EXNS {attr, exns} =>
	let val exnCount = List.length exns
	in
	    if isU2 exnCount then
		(emitCpIndex attr;
		 emitU4i(2 + exnCount * 2);
		 emitU2i exnCount;
		 List.app emitCpIndex exns)
	    else classError "emit.emitAttr [EXNS]: too many exceptions"
	end
      | INNER {attr, classes} =>
	let
            val count = List.length classes
	in
            if isU2 count then
		(emitCpIndex attr;
		 emitU4i (2 + count * 8);
		 emitU2i count;
		 List.app emitInnerClassInfo classes)
            else classError "emit.emitAttr [INNER]: too many inner classes"
	end
      | ENCLOSING {attr, class, method} =>
	(emitCpIndex attr;
	 emitU4 word32_4;
	 emitCpIndex class;
	 emitCpIndexOpt method)
      | SYNTHETIC{attr} =>
	(emitCpIndex attr;
	 emitU4i 0)
      | SIGNATURE {attr, sign} =>
	(emitCpIndex attr;
	 emitU4 word32_2;
	 emitCpIndex sign)
      | SRCFILE {attr, file} =>
	(emitCpIndex attr;
	 emitU4 word32_2;
	 emitCpIndex file)
      | LINENUM {attr, lines} =>
	let val lineCount = List.length lines
	in
	    if isU2 lineCount then
		(emitCpIndex attr;
		 emitU4i(2 + lineCount * 4);
		 emitU2i lineCount;
		 List.app emitLineNumInfo lines)
	    else classError "emit.emitAttr [LINENUM]: too many lines"
	end
      | LOCALVAR {attr, vars} =>
	let val varCount = List.length vars
	in
	    if isU2 varCount then
		(emitCpIndex attr;
		 emitU4i(2 + varCount * 10);
		 emitU2i varCount;
		 List.app emitLocalVarInfo vars)
	    else classError
		     "emit.emitAttr [LOCALVAR]: too many variables"
	end
      | LOCALVARTYPES {attr, vartypes} =>
	let val varCount = List.length vartypes
	in
	    if isU2 varCount then
		(emitCpIndex attr;
		 emitU4i(2 + varCount * 10);
		 emitU2i varCount;
		 List.app emitLocalVarTypeInfo vartypes)
	    else classError
		     "emit.emitAttr [LOCALVARTYPES]: too many variables"
	end

      | DEPRECATED{attr} =>
	(emitCpIndex attr;
	 emitU4i 0)
      | RTV_ANN {attr, annotations} =>
	(emitCpIndex attr;
	 emitAnnotSize 2 0 ann_size annotations;  (* 2 bytes for #annotations *)
	 emitAnnotList annotations)
      | RTI_ANN {attr, annotations} =>
	(emitCpIndex attr;
	 emitAnnotSize 2 0 ann_size annotations;
	 emitAnnotList annotations)
      | RTV_PARAM_ANN {attr, annotations} =>
	(emitCpIndex attr;
	 emitAnnotSize 1 2 (listSize ann_size) annotations;
	   (* 1 byte for #params, 2 bytes per param for #annotantions *)
	 emitU1i (List.length annotations);
	 app emitAnnotList annotations)
      | RTI_PARAM_ANN {attr, annotations} =>
	(emitCpIndex attr;
	 emitAnnotSize 1 2 (listSize ann_size) annotations;
	 emitU1i (List.length annotations);
	 app emitAnnotList annotations)
      | ANN_DEFAULT {attr, default_value} =>
	(emitCpIndex attr;
	 emitU4i (elt_size default_value);
	 emitElement default_value)
      | ATTR {attr, info} =>
	let val len = Word8Vector.length info
	in
	    emitCpIndex attr;
	    emitU4i len;
	    Word8Vector.app out info
	end
and emitAttr2 attr = (print (attrName attr); emitAttr attr)  (* debuffing *)
and emitAttrs attrs =
    let val len = List.length attrs
    in
	if isU2 len then (emitU2i len;
			  List.app emitAttr attrs)
	else classError "emit.emitAttrs: too many attributes"
    end

fun emitMember {flags, name, desc, attrs} =
    (emitU2 flags;
     emitCpIndex name;
     emitCpIndex desc;
     emitAttrs attrs)

val ifcsCount    = List.length ifcs
val fieldsCount  = List.length fields
val methodsCount = List.length methods
    in
	if isU2 ifcsCount then
	    if isU2 fieldsCount then
		if isU2 methodsCount then
		    (emitU4 magic;
		     emitU2 minor;
		     emitU2 major;
		     Constpool.emit out pool;
		     emitU2 flags;
		     emitCpIndex this;
		     emitCpIndexOpt super;
		     emitU2i ifcsCount;
		     List.app emitCpIndex ifcs;
		     emitU2i fieldsCount;
		     List.app emitMember fields;
		     emitU2i methodsCount;
		     List.app emitMember methods;
		     emitAttrs attrs)
		else classError "emit: too many methods"
	    else classError "emit: too many fields"
	else classError "emit: too many direct superinterfaces"
    end

fun emit out cp decl =
    emitClassFile out (fromClassDecl cp decl)

fun writeClassDecl c file =
let
    val os = BinIO.openOut file
    fun emitword w = BinIO.output1 (os, w)
    val cp = Constpool.create ()
    val () = emit emitword cp c
    val () = BinIO.closeOut os
in
    ()
end



(* ---------------------------- Classfile input ----------------------------*)

exception ClassfileInputError of string
exception KeyNotFoundError


(* -------------------- auxilliary functions -------------------- *)

(* apply <parser> to <vector> <number> times *)
fun <**> parser (result,vector_index,number) vector =
    let fun <++> parser (result,vector_index,number) vector =
            if number > 0
            then let val (s,new_vi) = parser vector vector_index
		 in  <++> parser (s :: result, new_vi, number-1) vector
		 end
	    else (result,vector_index)
	val (result, vector_index) = <++> parser (result,vector_index,number) vector
    in  (rev result, vector_index)
    end

val iterate = <**>

fun parseCP v n size =
    let (* <cp_array> use "size", because 0 is not a valid index *)
      val cp_array = Array.array(size, CPunused)

      fun parseItem (integer,m) =
	  case integer of 1  => parseUtf8 v m
			| 3  => parseInteger v m
			| 4  => parseFloat v m
			| 5  => parseLong v m
			| 6  => parseDouble v m
			| 7  => parseClass v m
			| 8  => parseString v m
			| 9  => parseFieldref v m
			| 10 => parseMethodref v m
			| 11 => parseInterfaceMethodref v m
			| 12 => parseNameAndType v m
			| _  => raise ClassfileInputError ("Unknown constant pool tag ["
						      ^ Int.toString integer ^ "]")

      fun parseTag m = parseU1_int v m

      fun parseItems (u,m,z) = if z < size
                               then let val (j,y,delta_index) = parseItem (parseTag m)
				    in parseItems ( Array.update(cp_array, z, j),
						    y,
						    z+delta_index )
				    end
			       else ((),m)
      val (_, i) = parseItems ((),n,1)
    in
	( cp_array, i )
    end

fun parseInterfaces v n size = iterate parseU2_idx ([],n,size) v

fun parseExceptionTable v n size =
    let
	fun parseItem v m =
	    let val (start_pc  ,i) = parseU2_w16 v m
		val (end_pc    ,i) = parseU2_w16 v i
	  	val (handler_pc,i) = parseU2_w16 v i
		val (catch_type,i) = parseU2_int v i
		val catch = add_idx_option catch_type
	    in  ( { start = start_pc,
		    stop  = end_pc,
		    entry = handler_pc,
		    catch = catch } : exn_hdl,
		  i )
	    end
    in  iterate parseItem ([],n,size) v
    end


fun parseElement v i =
    let
	val (tag, i) = parseU1_int v i

	fun mk_const type_tag =
	    let
		val (b,i) = parseU2_idx v i
	    in
		(Const_Value {type_tag = type_tag, idx = b}, i)
	    end
    in
	case chr tag of
	    #"B" => mk_const Byte
	  | #"C" => mk_const Char
	  | #"D" => mk_const Double
	  | #"F" => mk_const Float
	  | #"I" => mk_const Int
	  | #"J" => mk_const Long
	  | #"S" => mk_const Short
	  | #"Z" => mk_const Bool
	  | #"s" => mk_const String
	  | #"e" =>
	    let
		val (tn_idx, i) = parseU2_idx v i
		val (cn_idx, i) = parseU2_idx v i
	    in
		(Enum_Const_Value{type_name = tn_idx, const_name = cn_idx}, i)
	    end
	  | #"c" =>
	    let
		val (p,i) = parseU2_idx v i
	    in
		(Class_Info p, i)
	    end
	  | #"@" =>
	    let
		val (a,i) = parseAnnotation v i
	    in
		(Annot_Value a, i)
	    end
	  | #"[" =>
	    let
		val (num_values, i) = parseU2_int v i
		val (l,i) = iterate parseElement ([], i, num_values) v
	    in
		(Array_Value l, i)
	    end
	  | c => raise ClassfileInputError ("Invalid tag '" ^ Char.toString c
					    ^ "' in Classfile.parseElement")
    end

and parsePair v i =
    let
	val (name_index,i) = parseU2_idx v i
	val (value, i) = parseElement v i
    in
	({name_index = name_index, value = value}, i)
    end

and parseAnnotation v i =
    let
	val (type_index, i) = parseU2_idx v i
	val (num_ev_pairs, i) = parseU2_int v i
	val (values, i) = iterate parsePair ([], i, num_ev_pairs) v
    in
	({type_index = type_index,
	 values = values}, i)
    end

fun parseList parser v i =
    let
	val (numEls, i) = parseU2_int v i
    in
	iterate parser ([], i, numEls) v
    end

fun parseList1 parser v i =
    let
	val (numEls, i) = parseU1_int v i  (* for emedded lists in RTx_PARAM_ANN *)
    in
	iterate parser ([], i, numEls) v
    end

fun parseAttributes v n size pool =
    let fun parseItem w m =
	    let val (attribute_name_index,j) = parseU2_int w m
		val cp_item = Array.sub (pool,attribute_name_index)
		val (attribute_length,j) = parseU4_int w j
		val (attribute,j) =
		    parseInfo w
			      j
			      attribute_length
			      pool
			      (cp_item, attribute_name_index)
            in  (attribute, j)
	    end
    in  iterate parseItem ([],n,size) v
    end

and parseInfo v n size pool (cp_item,attribute_name_index) =
    let val empty_vector = Word8Vector.fromList []
	fun parseLineNr v j =
	    let val (start_pc,i) = parseU2_w16 v j
                val (line_number,i) = parseU2_w16 v i
	    in  ( { start = start_pc,
		    line  = line_number } : line_number_info,
		  i )
	    end
	fun parseLineNumberTable m table_length = iterate parseLineNr ([],m,table_length) v

    in case cp_item of
	   CPutf8 "ConstantValue" =>
	       let val (constant_value_index,i) = parseU2_int v n
	       in  ( CONSTVAL { attr  = Constpool.makeIndex attribute_name_index,
				value = Constpool.makeIndex constant_value_index },
		     i )
	       end
         | CPutf8 "Code" =>
	   let val (max_stack,i) = parseU2_w16 v n
       	       val (max_locals,i) = parseU2_w16 v i
	       val (code_length,i) = parseU4_int v i
	       val code_length_s = Int.toString code_length
	       val code_vector = Word8Vector.extract (v, i, SOME code_length)
	       val i = i+code_length
	       val (exception_table_length,i) = parseU2_int v i
	       val exception_table_length_s = Int.toString exception_table_length
	       val (exception_table,i) = parseExceptionTable v i exception_table_length
	       val (attributes_count,i) = parseU2_int v i
	       val attributes_count_s = Int.toString attributes_count
	       val (attributes,i) = parseAttributes v i attributes_count pool
	   in  ( CODE { attr   = Constpool.makeIndex attribute_name_index,
			stack  = max_stack,
			locals = max_locals,
			code   = code_vector,
			hdls   = exception_table,
			attrs  = attributes },
		 i )
	   end
	  | CPutf8 "Exceptions" =>
	       let val (number_of_exceptions,i) = parseU2_int v n
       		   val (exception_index_table,i) = iterate parseU2_idx ([],i,number_of_exceptions) v
	       in  ( EXNS { attr = Constpool.makeIndex attribute_name_index,
			    exns = exception_index_table },
		     i )
	       end
	  | CPutf8 "InnerClasses" =>
	    let val (number_of_classes,i) = parseU2_int v n
		fun parseItem w m =
		    let val (inner_info_index,i) = parseU2_int w m
			val (outer_info_index,i) = parseU2_int w i
			val (inner_name_index,i) = parseU2_int w i
			val (inner_flags     ,i) = parseU2_w16 w i
		    in  ( { inner_info  = add_idx_option inner_info_index,
			    outer_info  = add_idx_option outer_info_index,
			    inner_name  = add_idx_option inner_name_index,
			    inner_flags = inner_flags } : inner_class_info,
			  i )
		    end
 		val (classes, i) = iterate parseItem ([],i,number_of_classes) v
	    in  ( INNER { attr = Constpool.makeIndex attribute_name_index,
			  classes = classes },
		  i )
	    end
	  | CPutf8 "EnclosingMethod" =>
	    let
		val (class_index, i) = parseU2_int v n
		val (method_index, i) = parseU2_int v i
	    in
		(ENCLOSING {attr = Constpool.makeIndex attribute_name_index,
			    class = Constpool.makeIndex class_index,
			    method = add_idx_option method_index}, i)
	    end
	  | CPutf8 "Synthetic" =>
	    if size = 0
	    then ( SYNTHETIC { attr = Constpool.makeIndex attribute_name_index},
		   n )
	    else raise ClassfileInputError "Invalid 'Synthetic' attribute"
	  | CPutf8 "Signature" =>
	    let
		val (sig_index, i) = parseU2_int v n
	    in
		(SIGNATURE {attr = Constpool.makeIndex attribute_name_index,
			    sign = Constpool.makeIndex sig_index}, i)
	    end

	  | CPutf8 "SourceFile" =>
	    let val (index,i) = parseU2_int v n
	    in  ( SRCFILE { attr = Constpool.makeIndex attribute_name_index,
	 	 	    file = Constpool.makeIndex index },
		  i )
	    end
	  | CPutf8 "LineNumberTable" =>
	    let val (line_number_table_length,i) = parseU2_int v n
		val (line_number_table,i) = parseLineNumberTable i line_number_table_length
	    in  ( LINENUM { attr = Constpool.makeIndex attribute_name_index,
			    lines = line_number_table },
		  i )
	    end
	  | CPutf8 "LocalVariableTable" =>
	    let
		   val (table_length,i) = parseU2_int v n  (*untested*)
	           fun parseItem v m =
		       let val (start_pc  ,j) = parseU2_w16 v m
			   val (length    ,j) = parseU2_w16 v j
			   val (name_index,j) = parseU2_idx v j
			   val (desc_index,j) = parseU2_idx v j
			   val (index,j) = parseU2_w16 v j
		       in  ( { start  = start_pc,
			       length = length,
			       name   = name_index,
			       desc   = desc_index,
			       index  = index } : local_var_info,
			     j )
		       end
		   val (vars,i) = iterate parseItem ([],i,table_length) v
	       in  ( LOCALVAR { attr = Constpool.makeIndex attribute_name_index,
				vars = vars },
		     i )
	       end
	  | CPutf8 "LocalVariableTypeTable" =>
	    let
		   val (table_length,i) = parseU2_int v n  (*untested*)
	           fun parseItem v m =
		       let val (start_pc  ,j) = parseU2_w16 v m
			   val (length    ,j) = parseU2_w16 v j
			   val (name_index,j) = parseU2_idx v j
			   val (sign_index,j) = parseU2_idx v j
			   val (index,j) = parseU2_w16 v j
		       in  ( { start  = start_pc,
			       length = length,
			       name   = name_index,
			       sign   = sign_index,
			       index  = index } : local_var_type_info,
			     j )
		       end
		   val (vartypes,i) = iterate parseItem ([],i,table_length) v
	       in  ( LOCALVARTYPES { attr = Constpool.makeIndex attribute_name_index,
				vartypes = vartypes },
		     i )
	       end

	  | CPutf8 "Deprecated" =>
	       if size = 0
	       then ( DEPRECATED { attr = Constpool.makeIndex attribute_name_index}
		    , n )
	       else raise ClassfileInputError "Invalid 'Deprecated' attribute"
	  | CPutf8 "RuntimeVisibleAnnotations" =>
	    let
		val (l, i) = parseList parseAnnotation v n
	    in
		(RTV_ANN {attr = Constpool.makeIndex attribute_name_index,
			  annotations = l}, i)
	    end
	  | CPutf8 "RuntimeInvisibleAnnotations" =>
	    let
		val (l, i) = parseList parseAnnotation v n
	    in
		(RTI_ANN {attr = Constpool.makeIndex attribute_name_index,
			  annotations = l}, i)
	    end
	  | CPutf8 "RuntimeVisibleParameterAnnotations" =>
	    let
		val (l, i) = parseList1 (parseList parseAnnotation) v n
	    in
		(RTV_PARAM_ANN {attr = Constpool.makeIndex attribute_name_index,
			  annotations = l}, i)
	    end
	  | CPutf8 "RuntimeInvisibleParameterAnnotations" =>
	    let
		val (l, i) = parseList1 (parseList parseAnnotation) v n
	    in
		(RTI_PARAM_ANN {attr = Constpool.makeIndex attribute_name_index,
			  annotations = l}, i)
	    end
	  | CPutf8 "AnnotationDefault" =>
	    let
		val (elt, i) = parseElement v n
	    in
		(ANN_DEFAULT {attr = Constpool.makeIndex attribute_name_index,
			      default_value = elt}, i)
	    end
	  | CPutf8 _ =>
	       ( ATTR { attr = Constpool.makeIndex attribute_name_index,
			info = Word8Vector.extract (v, n, SOME size)},
		 n + size )
	  | _ => raise ClassfileInputError "utf8 expected in <parseInfo>"
    end

fun parseFields v n size pool =
    let fun parseItem v m =
	    let val (access_flags,j) = parseU2_w16 v m
		val (name_index,j) = parseU2_idx v j
 		val (descriptor_index,j) = parseU2_idx v j
 		val (attributes_count,j) = parseU2_int v j
 		val attributes_count_s = Int.toString attributes_count
 		val (attributes,j) = parseAttributes v j attributes_count pool
 		val member = { flags = access_flags,
			     name  = name_index,
			     desc  = descriptor_index,
			     attrs = attributes } : member
	    in  ( member, j )
	    end
    in  iterate parseItem ([],n,size) v
    end

(* -------------------- Main function -------------------- *)

fun vectorToClass tokens =
let
    val len = Word8Vector.length tokens
    val () = if len = 0 then raise ClassfileInputError "No data in vectorToClass" else ()
    val empty_hash_table = Polyhash.mkPolyTable (0,KeyNotFoundError)

    val (magic,i) = parseU4_w32 tokens 0
    val () = if (Word32.toList magic = [0wxCA, 0wxFE, 0wxBA, 0wxBE]) then ()
	     else raise InvalidClass "Bad magic number"
    val (minor,i) = parseU2_w16 tokens i
    val (major,i) = parseU2_w16 tokens i

    val (cp_count,i) = parseU2_int tokens i
    val cp_count_s = Int.toString cp_count
    val (pool_array,i) = parseCP tokens i cp_count

    val (access_flags,i) = parseU2_w16 tokens i
    val (this_class,i) = parseU2_int tokens i
    val this_class_s = Int.toString this_class
    val (super_class,i) = parseU2_int tokens i     (* = 0 or index *)
    val super_class_s = Int.toString super_class
    val super_class_option = add_idx_option super_class

    val (interfaces_count,i) = parseU2_int tokens i
    val interfaces_count_s = Int.toString interfaces_count
    val (interfaces,i) = parseInterfaces tokens i interfaces_count

    val (fields_count,i) = parseU2_int tokens i
    val fields_count_s = Int.toString fields_count
    val (fields,i) = parseFields tokens i fields_count pool_array

    val (methods_count,i) = parseU2_int tokens i
    val methods_count_s = Int.toString methods_count
    val (methods,i) = parseFields tokens i methods_count pool_array

    val (attributes_count,i) = parseU2_int tokens i
    val attributes_count_s = Int.toString attributes_count

    val (attributes,i) = parseAttributes tokens i attributes_count pool_array

    val class_file =
	{ magic   = magic,
	  minor   = minor,
	  major   = major,
	  pool    = Constpool.createPool ( ref pool_array
			                 , empty_hash_table
					 , ref cp_count
				         (*ref (Array.length pool_array))*)
					 ),
	  flags   = access_flags,
	  this    = Constpool.makeIndex this_class,
	  super   = super_class_option,
	  ifcs    = interfaces,
	  fields  = fields,
	  methods = methods,
	  attrs   = attributes } : class_file
in  if i = len
    then class_file
    else raise ClassfileInputError "File too short"
end


fun inputClassFile infile =
let
    val is = BinIO.openIn infile
    val vec = BinIO.inputAll is
    val () = BinIO.closeIn is
in
    vectorToClass vec
end

