(* Classdecl.sml
 *
 * Peter Bertelsen
 * September 1997
 *)

local
    open Label Bytecode Jvmtype
in

datatype class_access_flag =
	 C_ACCpublic
       | C_ACCprivate       (* Inner classes only *)
       | C_ACCprotected     (* Inner classes only *)
       | C_ACCstatic        (* Inner classes only *)
       | C_ACCfinal
       | C_ACCsuper
       | C_ACCinterface
       | C_ACCabstract
       | C_ACCsynthetic
       | C_ACCannotation
       | C_ACCenum

datatype field_access_flag =
	 F_ACCpublic
       | F_ACCprivate
       | F_ACCprotected
       | F_ACCstatic
       | F_ACCfinal
       | F_ACCvolatile
       | F_ACCtransient
       | F_ACCsynthetic
       | F_ACCenum

datatype method_access_flag =
	 M_ACCpublic
       | M_ACCprivate
       | M_ACCprotected
       | M_ACCstatic
       | M_ACCfinal
       | M_ACCsynchronized
       | M_ACCbridge
       | M_ACCvarargs
       | M_ACCnative
       | M_ACCabstract
       | M_ACCstrictfp
       | M_ACCsynthetic


(*--- Java 5 annotations ---*)

datatype constant =
	 Byte of int              (* B *)
       | Char of Int32.int        (* C *)
       | Double of Real64.real    (* D *)
       | Float of Real32.real     (* F *)
       | Int of Int32.int         (* I *)
       | Long of Int64.int        (* J *)
       | Short of int             (* S *)
       | Bool of int              (* Z *)
       | String of string         (* s *)

datatype element =
	 Const_Value of constant              (* BCDFIJSZs *)
       | Enum_Const_Value of
	    {type_name: string,
	     const_name: string}              (* e *)
       | Class_Info  of string                (* c *)
       | Annot_Value of annotation            (* @ *)
       | Array_Value of element list          (* [ *)

withtype annotation =
       {atype : string,
	values: {name: string, value: element} list}

(*--- end ---*)

type exn_hdl =                  (* exception handler declaration *)
    {start: label,		(* start of handler scope *)
     stop:  label,		(* end of handler scope *)
     entry: label,		(* handler entry point *)
     catch: jclass option}	(* class of handled exception; NONE = any *)

datatype attribute =
	 CONSTVAL      of jvm_const		 (* constant value *)
       | CODE          of code_info              (* code attribute, including bytecode *)
       | EXNS          of jclass list		 (* exception classes *)
       | INNER         of inner_class_info list
       | ENCLOSING     of enclosing_class_info   (* Java 5 *)
       | SYNTHETIC
       | SIGNATURE     of string                 (* Java 5 *)
       | SRCFILE       of string		 (* source file name *)
       | LINENUM       of line_number_info list
       | LOCALVAR      of localvar_info list
       | LOCALVARTYPES of localvartype_info list (* Java 5 *)
       | DEPRECATED
       | RTV_ANN       of annotation list        (* Java 5 - RuntimeVisibleAnnotation *)
       | RTI_ANN       of annotation list        (* Java 5 - RuntimeInvisibleAnnotation *)
       | RTV_PARAM_ANN of annotation list list   (* Java 5 - RuntimeVisibleParameterAnnotation *)
       | RTI_PARAM_ANN of annotation list list   (* Java 5 - RuntimeInvisibleParameterAnnotation *)
       | ANN_DEFAULT   of element                (* Java 5 *)
       | ATTR          of generic_attribute

withtype code_info =
    {stack:  int,		(* max operand stack depth *)
     locals: int,		(* max size of local vars *)
     code:   jvm_instr list,	(* instruction sequence *)
     hdls:   exn_hdl list,	(* exception handlers *)
     attrs:  attribute list}

and inner_class_info =
    {inner_info: jclass option,
     outer_info: jclass option,
     inner_name: string option,
     inner_flags: class_access_flag list }

and line_number_info =
    {start: label,
     line:  int}

and localvar_info =
    {from:  label,
     thru:  label,
     name:  string,
     ty:    jtype,
     index: Localvar.index}

and localvartype_info =
    {from: label,
     thru: label,
     name: string,
     sign: string,  (* signature *)
     index: Localvar.index}

and enclosing_class_info =
    {class: jclass,
     method: {mname: string,
	      msig: method_sig} option}

and generic_attribute =
    {attr: string,
     info: Word8Vector.vector}


(* Fields, methods and classes *)


type field_decl =	                (* field declaration *)
   {flags:  field_access_flag list,	(* access flags *)
    name:   string,		        (* unqualified name *)
    ty:     jtype,			(* field type *)
    attrs:  attribute list}	        (* field attributes *)

type method_decl =	                (* method declaration *)
   {flags:  method_access_flag list,	(* access flags *)
    name:   string,		        (* unqualified name *)
    msig:   method_sig,		        (* method signature *)
    attrs:  attribute list}	        (* method attributes *)

val major = 3                           (* default major version number *)
val minor = 45                          (* default minor version number *)
(* UPDATE THESE FOR JAVA 5 VM *)

type class_decl =	                (* class/interface declaration *)
    {major: int,                        (* major version number *)
     minor: int,                        (* minor version number *)
     flags:  class_access_flag list,	(* access flags *)
     this:   jclass,	                (* this class/interface *)
     super:  jclass option,		(* direct superclass *)
     ifcs:   jclass list,		(* direct superinterfaces *)
     fdecls: field_decl list,	        (* field declarations *)
     mdecls: method_decl list,	        (* method declarations *)
     attrs:  attribute list}

end (* local *)



(*
(* Debugging annotations *)

fun listToString toString separator l =
case l of [] => ""
        | [h] => toString h
        | h::t => (toString h) ^ separator ^ (listToString toString separator t)

fun constantToString c =
    case c of
	Byte  _ => "Byte"
      | Char  _ => "Char"
      | Double _ => "Double"
      | Float  _ => "Float"
      | Int    _ => "Int"
      | Long   _ => "Long"
      | Short  _ => "Short"
      | Bool   _ => "Bool"
      | String _ => "String"

fun elementToString e =
    case e of
	Const_Value c =>
	"Const_Value(" ^ constantToString c ^ ")"
      | Enum_Const_Value {type_name, const_name} =>
	   "Enum_Const_Value (" ^ type_name ^ ", "
	   ^ const_name ^")"
      | Class_Info s => "Class_Info (" ^ s ^ ")"
      | Annot_Value a => "Annot_Value " ^ annotationToString a
      | Array_Value l => "Array_Value [\n    "
			 ^ listToString elementToString ",\n    " l
			 ^ "\n   ]"

and pairToString {name, value} = name ^ " -> " ^ elementToString value

and annotationToString {atype, values} =
    "< atype = " ^ atype
    ^ "\n  annotations = [" ^ listToString pairToString "\n " values
    ^ "\n  ] >\n"

*)
