(* Bytecode.sml
 *
 * Peter Bertelsen
 * December 1997
 *)

exception InternalError of string

datatype jvm_const =
    Cnull
  | Cint    of Int32.int
  | Cfloat  of Real32.real
  | Clong   of Int64.int
  | Cdouble of Real64.real
  | Cstring of string

local
    open Label Localvar Jvmtype
in
    datatype class_ref =
	CLASS of jclass   (* a class or interface *)
      | ARRAY of jtype    (* an `array class' of the given element type *)

    type field_ref = {class : jclass,   (* not an array class *)
		      name  : string,
		      ty    : jtype}

    type method_ref = {class : jclass,  (* not an array class *)
		       name  : string,
		       msig  : method_sig}

    (* The type of JVM instructions, extended with labels *)
    datatype jvm_instr =
	Jlabel of label
      | Jsconst of string
      | Jaaload
      | Jaastore
      | Jaconst_null
      | Jaload of index
   (* | Jaload_0 | Jaload_1 | Jaload_2 | Jaload_3 *)
   (* | Janewarray of class_ref *)
   (* | Jareturn *)
      | Jarraylength
      | Jastore of index
   (* | Jastore_0 | Jastore_1 | Jastore_2 | Jastore_3 *)
      | Jathrow
      | Jbaload
      | Jbastore
   (* | Jbipush of int *)
      | Jcaload
      | Jcastore
      | Jcheckcast  of class_ref
      | Jclassconst of class_ref    (* for Java 5 ldc modifications *)
      | Jd2f
      | Jd2i
      | Jd2l
      | Jdadd
      | Jdaload
      | Jdastore
      | Jdcmpg | Jdcmpl
      | Jdconst of Real64.real
   (* | Jdconst_0 | Jdconst_1 *)
      | Jddiv
      | Jdload of index
   (* | Jdload_0 | Jdload_1 | Jdload_2 | Jdload_3 *)
      | Jdmul
      | Jdneg
      | Jdrem
   (* | Jdreturn *)
      | Jdstore of index
   (* | Jdstore_0 | Jdstore_1 | Jdstore_2 | Jdstore_3 *)
      | Jdsub
      | Jdup
      | Jdup_x1
      | Jdup_x2
      | Jdup2
      | Jdup2_x1
      | Jdup2_x2
      | Jf2d
      | Jf2i
      | Jf2l
      | Jfadd
      | Jfaload
      | Jfastore
      | Jfcmpg | Jfcmpl
      | Jfconst of Real32.real
   (* | Jfconst_0 | Jfconst_1 | Jfconst_2 *)
      | Jfdiv
      | Jfload of index
   (* | Jfload_0 | Jfload_1 | Jfload_2 | Jfload_3 *)
      | Jfmul
      | Jfneg
      | Jfrem
   (* | Jfreturn *)
      | Jfstore of index
   (* | Jfstore_0 | Jfstore_1 | Jfstore_2 | Jfstore_3 *)
      | Jfsub
      | Jgetfield of field_ref
      | Jgetstatic of field_ref
      | Jgoto of label
   (* | Jgoto_w of label *)
      | Ji2b
      | Ji2c
      | Ji2d
      | Ji2f
      | Ji2l
      | Ji2s
      | Jiadd
      | Jiaload
      | Jiand
      | Jiastore
      | Jiconst of Int32.int
   (* | Jiconst_m1 | Jiconst_0 | Jiconst_1 | Jiconst_2 *)
   (* | Jiconst_3 | Jiconst_4 | Jiconst_5 *)
      | Jidiv
      | Jif_acmpeq of label | Jif_acmpne of label
      | Jif_icmpeq of label | Jif_icmpne of label | Jif_icmplt of label
      | Jif_icmpge of label | Jif_icmpgt of label | Jif_icmple of label
      | Jifeq of label | Jifne of label | Jiflt of label
      | Jifge of label | Jifgt of label | Jifle of label
      | Jifnonnull of label | Jifnull of label
      | Jiinc of {var   : index,
		  const : int}
      | Jiload of index
   (* | Jiload_0 | Jiload_1 | Jiload_2 | Jiload_3 *)
      | Jimul
      | Jineg
      | Jinstanceof of class_ref
      | Jinvokeinterface of method_ref
      | Jinvokespecial of method_ref
      | Jinvokestatic of method_ref
      | Jinvokevirtual of method_ref
      | Jior
      | Jirem
   (* | Jireturn *)
      | Jishl
      | Jishr
      | Jistore of index
   (* | Jistore_0 | Jistore_1 | Jistore_2 | Jistore_3 *)
      | Jisub
      | Jiushr
      | Jixor
      | Jjsr of label
   (* | Jjsr_w of label *)
      | Jl2d
      | Jl2f
      | Jl2i
      | Jladd
      | Jlaload
      | Jland
      | Jlastore
      | Jlcmp
      | Jlconst of Int64.int
   (* | Jlconst_0 | Jlconst_1 *)
   (* | Jldc of    jvm_const *)
   (* | Jldc_w of  jvm_const *)
   (* | Jldc2_w of jvm_const *)
      | Jldiv
      | Jlload of index
   (* | Jlload_0 | Jlload_1 | Jlload_2 | Jlload_3 *)
      | Jlmul
      | Jlneg
      | Jlookupswitch of {default : label,
			  cases   : (Int32.int * label) list}
      | Jlor
      | Jlrem
   (* | Jlreturn *)
      | Jlshl
      | Jlshr
      | Jlstore of index
   (* | Jlstore_0 | Jlstore_1 | Jlstore_2 | Jlstore_3 *)
      | Jlsub
      | Jlushr
      | Jlxor
      | Jmonitorenter
      | Jmonitorexit
   (* | Jmultianewarray of {elem : jtype,
                            dim  : int} *)
      | Jnew of jclass
   (* | Jnewarray of jtype *)
      | Jnewarray of {elem : jtype,
		      dim  : int}
      | Jnop
      | Jpop
      | Jpop2
      | Jputfield of field_ref
      | Jputstatic of field_ref
      | Jret of index
      | Jreturn
      | Jsaload
      | Jsastore
   (* | Jsipush of int *)
      | Jswap
      | Jtableswitch of {default : label,
			 offset  : Int32.int,   (* key of first target *)
			 targets : label Vector.vector}
   (* | Jwide of jvm_instr *)

end (* local *)

(* NOTE: Some JVM instructions have been omitted in the jvm_instr
 * datatype.  For example, there is no need to distinguish aload 0
 * from aload_0 until the binary byte code is emitted; the emitter can
 * choose the most compact aload variant.  These are the translations
 * made by the bytecode emitter (Emitcode.emit):
 *
 *   Jsconst   ==> ldc or ldc_w
 *   Jaload    ==> aload_<n>, aload, or wide aload
 *   Jastore   ==> astore_<n>, astore, or wide astore
 *   Jdconst   ==> dconst_<d> or ldc2_w
 *   Jdload    ==> dload_<n>, dload, or wide dload
 *   Jdstore   ==> dstore_<n>, dstore, or wide dstore
 *   Jfconst   ==> fconst_<f>, ldc, or ldc_w
 *   Jfload    ==> fload_<n>, fload, or wide fload
 *   Jfstore   ==> fstore_<n>, fstore, or wide fstore
 *   Jgoto     ==> goto or goto_w
 *   Jiconst   ==> iconst_<i>, bipush, sipush, ldc, or ldc_w
 *   Jiinc     ==> iinc or wide iinc
 *   Jiload    ==> iload_<n>, iload, or wide iload
 *   Jistore   ==> istore_<n>, istore, or wide istore
 *   Jjsr      ==> jsr or jsr_w
 *   Jlconst   ==> lconst_<l> or ldc2_w
 *   Jlload    ==> lload_<n>, lload, or wide lload
 *   Jlstore   ==> lstore_<n>, lstore, or wide lstore
 *   Jnewarray ==> newarray, anewarray, or multianewarray
 *   Jret      ==> ret or wide ret
 *   Jreturn   ==> areturn, dreturn, freturn, ireturn, lreturn, or return
 *)


(* Utility functions *)
(* Unused? *)
fun intConst   i = Jiconst (Int32.fromInt i)
fun wordConst  w = Jiconst (Int32.fromInt(Word.toIntX w))
fun charConst  c = Jiconst (Int32.fromInt(Char.ord c))
fun realConst  r = Jdconst (Real64.fromReal r)
fun checkCast  c = Jcheckcast (CLASS c)
fun checkArray t = Jcheckcast (ARRAY t)


(* kwxm: Print a Jasmin-like representation of a JVM opcode.
  [Mostly for debugging purposes:  feel free to make improvements.] *)

local

open Jvmtype
val strOfIndex = Int.toString o Localvar.toInt
fun strOfLabel l = Label.toString l

fun slashToDot c =
    case c of #"/" => #"."
	    | _ => c

fun dottedQualName c = String.implode (map slashToDot (String.explode (Jvmtype.qualName c)))

fun strOfClass jc = Jvmtype.qualName jc

fun strOfCref c =
    case c of CLASS jc => strOfClass jc
	    | ARRAY jt => "<array>"

and strOfType t = Jvmtype.typeDesc t

fun strOfFref {class, name, ty} =
    dottedQualName class ^ "." ^ name ^ " " ^ Jvmtype.typeDesc ty

fun strOfMref {class, name, msig as (argtys, rty)} =
    strOfClass class ^ "/" ^ name ^ Jvmtype.methodDesc (argtys, rty)


in

fun make_key (low, index) = Int32.- (Int32.+ (low, Int32.fromInt index), Int32.fromInt 1)
(* low + len - 1, for tableswitch *)

fun toString i =
    case i of
	Jlabel l => "Jlabel " ^ strOfLabel l
      | Jsconst s => "Jsconst " ^ "\"" ^ s ^ "\""
      | Jaaload => "Jaaload"
      | Jaastore => "Jaastore"
      | Jaconst_null => "Jaconst_null"
      | Jaload i => "Jaload " ^ strOfIndex i
      | Jarraylength => "Jarraylength"
      | Jastore i => "Jastore " ^ strOfIndex i
      | Jathrow => "Jathrow"
      | Jbaload => "Jbaload"
      | Jbastore => "Jbastore"
      | Jcaload => "Jcaload"
      | Jcastore => "Jcastore"
      | Jcheckcast c =>  "Jcheckcast " ^ strOfCref c
      | Jclassconst c => "Jclassconst " ^ strOfCref c
      | Jd2f => "Jd2f"
      | Jd2i => "Jd2i"
      | Jd2l => "Jd2l"
      | Jdadd => "Jdadd"
      | Jdaload => "Jdaload"
      | Jdastore => "Jdastore"
      | Jdcmpg  => "Jdcmpg"
      | Jdcmpl => "Jdcmpl"
      | Jdconst x => "Jdconst " ^ Real64.toString x ^ " // APPROXIMATELY"
      | Jddiv => "Jddiv"
      | Jdload i => "Jdload " ^ strOfIndex i
      | Jdmul => "Jdmul"
      | Jdneg => "Jdneg"
      | Jdrem => "Jdrem"
      | Jdstore i => "Jdstore " ^ strOfIndex i
      | Jdsub => "Jdsub"
      | Jdup => "Jdup"
      | Jdup_x1 => "Jdup_x1"
      | Jdup_x2 => "Jdup_x2"
      | Jdup2 => "Jdup2"
      | Jdup2_x1 => "Jdup2_x1"
      | Jdup2_x2 => "Jdup2_x2"
      | Jf2d => "Jf2d"
      | Jf2i => "Jf2i"
      | Jf2l => "Jf2l"
      | Jfadd => "Jfadd"
      | Jfaload => "Jfaload"
      | Jfastore => "Jfastore"
      | Jfcmpg  => "Jfcmpg"
      | Jfcmpl => "Jfcmpl"
      | Jfconst x => "Jfconst " ^ Real32.toString x ^ " // APPROXIMATELY"
      | Jfdiv => "Jfdiv"
      | Jfload i => "Jfload " ^ strOfIndex i
      | Jfmul => "Jfmul"
      | Jfneg => "Jfneg"
      | Jfrem => "Jfrem"
      | Jfstore i => "Jfstore " ^ strOfIndex i
      | Jfsub => "Jfsub"
      | Jgetfield f => "Jgetfield " ^ strOfFref f
      | Jgetstatic f => "Jgetstatic " ^ strOfFref f
      | Jgoto l => "Jgoto " ^ strOfLabel l
      | Ji2b => "Ji2b"
      | Ji2c => "Ji2c"
      | Ji2d => "Ji2d"
      | Ji2f => "Ji2f"
      | Ji2l => "Ji2l"
      | Ji2s => "Ji2s"
      | Jiadd => "Jiadd"
      | Jiaload => "Jiaload"
      | Jiand => "Jiand"
      | Jiastore => "Jiastore"
      | Jiconst c => "Jiconst " ^ Int32.toString c
      | Jidiv => "Jidiv"
      | Jif_acmpeq l  => "Jif_acmpeq " ^ strOfLabel l
      | Jif_acmpne l => "Jif_acmpne " ^ strOfLabel l
      | Jif_icmpeq l  => "Jif_icmpeq " ^ strOfLabel l
      | Jif_icmpne l  => "Jif_icmpne " ^ strOfLabel l
      | Jif_icmplt l => "Jif_icmplt " ^ strOfLabel l
      | Jif_icmpge l  => "Jif_icmpge " ^ strOfLabel l
      | Jif_icmpgt l  => "Jif_icmpgt " ^ strOfLabel l
      | Jif_icmple l => "Jif_icmple " ^ strOfLabel l
      | Jifeq l  => "Jifeq " ^ strOfLabel l
      | Jifne l  => "Jifne " ^ strOfLabel l
      | Jiflt l => "Jiflt " ^ strOfLabel l
      | Jifge l  => "Jifge " ^ strOfLabel l
      | Jifgt l  => "Jifgt " ^ strOfLabel l
      | Jifle l => "Jifle " ^ strOfLabel l
      | Jifnonnull l  => "Jifnonnull " ^ strOfLabel l
      | Jifnull l => "Jifnull " ^ strOfLabel l
      | Jiinc {var, const} => "Jiinc " ^ strOfIndex var ^ " " ^ Int.toString const
      | Jiload i => "Jiload " ^ strOfIndex i
      | Jimul => "Jimul"
      | Jineg => "Jineg"
      | Jinstanceof c => "Jinstanceof " ^ strOfCref c
      | Jinvokeinterface m => "Jinvokeinterface " ^ strOfMref m
      | Jinvokespecial m => "Jinvokespecial " ^ strOfMref m
      | Jinvokestatic m => "Jinvokestatic " ^ strOfMref m
      | Jinvokevirtual m => "Jinvokevirtual " ^ strOfMref m
      | Jior => "Jior"
      | Jirem => "Jirem"
      | Jishl => "Jishl"
      | Jishr => "Jishr"
      | Jistore i => "Jistore " ^ strOfIndex i
      | Jisub => "Jisub"
      | Jiushr => "Jiushr"
      | Jixor => "Jixor"
      | Jjsr l => "Jjsr " ^ strOfLabel l
      | Jl2d => "Jl2d"
      | Jl2f => "Jl2f"
      | Jl2i => "Jl2i"
      | Jladd => "Jladd"
      | Jlaload => "Jlaload"
      | Jland => "Jland"
      | Jlastore => "Jlastore"
      | Jlcmp => "Jlcmp"
      | Jlconst c => "Jlconst " ^ Int64.toString c
      | Jldiv => "Jldiv"
      | Jlload i => "Jlload " ^ strOfIndex i
      | Jlmul => "Jlmul"
      | Jlneg => "Jlneg"
      | Jlookupswitch {default, cases} =>
	"Jlookupswitch\n"
	^ (foldl (fn ((n,l),s) => s ^ "\t" ^ Int32.toString n ^ ": " ^ strOfLabel l ^ "\n") ""  cases)
	^ "\tdefault: " ^ strOfLabel default
      | Jlor => "Jlor"
      | Jlrem => "Jlrem"
      | Jlshl => "Jlshl"
      | Jlshr => "Jlshr"
      | Jlstore i => "Jlstore " ^ strOfIndex i
      | Jlsub => "Jlsub"
      | Jlushr => "Jlushr"
      | Jlxor => "Jlxor"
      | Jmonitorenter => "Jmonitorenter"
      | Jmonitorexit => "Jmonitorexit"
      | Jnew c => "Jnew " ^ strOfClass c
      | Jnewarray {elem, dim} => "Jnewarray " ^ strOfType elem ^ " " ^ Int.toString dim
      | Jnop => "Jnop"
      | Jpop => "Jpop"
      | Jpop2 => "Jpop2"
      | Jputfield f => "Jputfield " ^ strOfFref f
      | Jputstatic f => "Jputstatic " ^ strOfFref f
      | Jret i => "Jret " ^ strOfIndex i
      | Jreturn => "Jreturn"
      | Jsaload => "Jsaload"
      | Jsastore => "Jsastore"
      | Jswap => "Jswap"
      | Jtableswitch {default, offset, targets} =>
	"Jtableswitch "
	^ Int32.toString offset
	^ " "
	^ Int.toString (Int32.toInt (make_key (offset, Vector.length targets)))
	^ "\n"
	^ (Vector.foldl (fn (l,s) => s ^ "\t" ^ strOfLabel l ^ "\n") ""  targets)
	^ "\tdefault: "
	^ strOfLabel default

fun printInstr k = TextIO.print (toString k ^ "\n")

end (* local *)
