(*
 * jGrailState.thy
 * Matthew Prowse
 * 17th November 2002
 *)

theory jGrailState = jGrailInstructions:

(*
 * Our state consists of a quadruple (F,H,M,C) where
 *
 *     F \<in> FrameStack = Frame*
 *     H \<in> Heap       = Addr \<rightharpoonup> Object
 *     M \<in> MethodArea = ClassName \<rightharpoonup> (ClassFile \<times> ClassVars)
 * and C is a structural cost quadruple
 *
 *)


(* Here follows code from MicroJava/Type.thy without bits about exceptions *)
(* Included just so the jGrail formalisation is not dependent on MicroJava *)
typedecl cnam 

datatype cname  
  = Object 
  | Cname cnam 

typedecl vnam
typedecl mname

datatype vname 
  = This
  | VName vnam

datatype prim_ty 
  = Void
  | Boolean
  | Integer

datatype ref_ty   
  = NullT
  | ClassT cname

datatype ty 
  = PrimT prim_ty
  | RefT  ref_ty



(* Here follows code from MicroJava/Value.thy without bits about exceptions *)
(* Included just so the jGrail formalisation is not dependent on MicroJava  *)

typedecl loc_

datatype loc = Loc loc_

datatype val
  = Unit
  | Null
  | Bool bool
  | Intg int
  | Addr loc

consts
  the_Bool :: "val => bool"
  the_Intg :: "val => int"
  the_Addr :: "val => loc"

primrec
  "the_Bool (Unit)   = arbitrary"
  "the_Bool (Null)   = arbitrary"
  "the_Bool (Bool b) = b"
  "the_Bool (Intg i) = arbitrary"
  "the_Bool (Addr a) = arbitrary"

primrec
  "the_Intg (Unit)   = arbitrary"
  "the_Intg (Null)   = arbitrary"
  "the_Intg (Bool b) = arbitrary"
  "the_Intg (Intg i) = i"
  "the_Intg (Addr a) = arbitrary"

primrec
  "the_Addr (Unit)   = arbitrary"
  "the_Addr (Null)   = arbitrary"
  "the_Addr (Bool b) = arbitrary"
  "the_Addr (Intg i) = arbitrary"
  "the_Addr (Addr a) = a"


consts
  defpval :: "prim_ty => val"
  default_val :: "ty => val"

primrec
  "defpval Void    = Unit"
  "defpval Boolean = Bool False"
  "defpval Integer = Intg 0"

primrec
  "default_val (PrimT pt) = defpval pt"
  "default_val (RefT  r ) = Null"


(* Here follows my code specifically for the jGrail formalisation *)
(* A great deal of it is, however, based on parts of MicroJava *)

types
  fldref  = "cname \<times> vname \<times> ty"
  sig     = "mname \<times> ty list"
  methref = "cname \<times> sig \<times> ty"

datatype poolEntry = Value val | ClassName cname | FieldRef fldref | MethodRef methref
datatype varValue = String string | Value' val


types

  (* Frame *)
  varenv     = "val list"
  stack      = "val list"
  pc         = "nat"
  frame      = "varenv \<times> stack \<times> cname \<times> sig \<times> pc"


  (* Heap *)
  fields     = "(vname \<times> cname) \<leadsto> val"
  object     = "cname \<times> fields"
  heap       = "loc \<leadsto> object"

  (* Method Area *)
  fdecl      = "vname \<times> ty"
  mdecl      = "sig \<times> ty \<times> nat \<times> nat \<times> (jGrailInstr list)"
  constpool  = "nat \<leadsto> poolEntry"
  cvars      = "vname \<leadsto>  varValue"
  cfile      = "cname option \<times> fdecl list \<times> mdecl list \<times> constpool"
  
  methodarea = "cname \<leadsto> (cfile \<times> cvars)"


  (* Structural cost quadruple *)
  framecost  = "(sig \<times> cname) \<Rightarrow> nat"
  heapcost   = "cname \<Rightarrow> nat"
  methodcost = "cname list"
  timecost   = "jGrailCode \<Rightarrow> nat"

  structcost = "framecost \<times> heapcost \<times> methodcost \<times> timecost"


  (* jGrailState *)
  jGrailState = "frame list \<times> heap \<times> methodarea \<times> structcost"



(* Semantic Functions *)
constdefs

  nextPC :: "pc \<Rightarrow> pc"
  "nextPC pc \<equiv> pc + 1"

  newAddr   :: "heap \<Rightarrow> loc"
  "newAddr h \<equiv> SOME a. h a = None"

  method :: "mdecl list \<Rightarrow> sig \<leadsto> ty \<times> nat \<times> nat \<times> jGrailInstr list"
  "method \<equiv> map_of"

  init_vars :: "((vname \<times> cname) \<times> ty) list \<Rightarrow> fields"
  "init_vars \<equiv> map_of \<circ> map (\<lambda>(n,T). ((n,default_val(T))))"


(* Subclass relation and recursive function *)

consts subclass :: "methodarea \<Rightarrow> (cname \<times> cname) set"
inductive "subclass M" intros
  subclassI: "\<lbrakk>M cn = Some ((sc,rest),cvars); cn \<noteq> Object\<rbrakk> \<Longrightarrow> (cn,the sc)\<in>subclass M"


consts
  class_rec ::"methodarea \<times> cname \<Rightarrow> 
        'a \<Rightarrow> (cname \<Rightarrow> cname option \<Rightarrow> fdecl list \<Rightarrow> mdecl list \<Rightarrow> constpool \<Rightarrow> cvars \<Rightarrow> 'a \<Rightarrow> 'a) \<Rightarrow> 'a"

recdef class_rec "same_fst (\<lambda>M. wf ((subclass M)^-1)) (\<lambda>M. (subclass M)^-1)"
  "class_rec (M,cn) = (\<lambda>t f. case (M cn) of None \<Rightarrow> arbitrary
                      | Some(cfile,cvars) \<Rightarrow> (let (sc,fdecls,mdecls,cp)=cfile in
                          (if wf ((subclass M)^-1) then 
                              f cn sc fdecls mdecls cp cvars (
                                 if cn=Object then t else
                                     (if sc=None then arbitrary else
                                         class_rec (M,the sc) t f)) else arbitrary)))"
(hints intro: subclassI)



(* Method and field retreival *)
constdefs
  lookup :: "methodarea \<times> cname \<Rightarrow> (sig \<leadsto> cname \<times> ty \<times> nat \<times> nat \<times> jGrailInstr list)"
  "lookup \<equiv> \<lambda>(M,cn). class_rec (M,cn) empty
                (\<lambda>cn sc fdecls mdecls cp cvars ts.
                    ts ++ map_of (map (\<lambda>(s,m). (s,(cn,m))) mdecls))"

  fields :: "methodarea \<times> cname \<Rightarrow> ((vname \<times> cname) \<times> ty) list"
  "fields \<equiv> \<lambda>(M,cn). class_rec (M,cn) []
                (\<lambda>cn sc fdecls mdecls cp cvars ts.
                    map (\<lambda>(fn,ft). ((fn,cn),ft)) fdecls @ ts)"

  instanceOf :: "cname \<Rightarrow> loc \<Rightarrow> heap \<Rightarrow> methodarea \<Rightarrow> bool"
  "instanceOf cn my_ref H M \<equiv> (fst(the (H my_ref)),cn) \<in> subclass M"




(* poolEntry functions *)
consts
  the_val  :: "poolEntry \<Rightarrow> val"
  the_cn   :: "poolEntry \<Rightarrow> cname"
  the_fref :: "poolEntry \<Rightarrow> fldref"
  the_mref :: "poolEntry \<Rightarrow> methref"
primrec
  "the_val  (Value v)     = v"
  "the_val  (ClassName c) = arbitrary"
  "the_val  (FieldRef f)  = arbitrary"
  "the_val  (MethodRef m) = arbitrary"
primrec
  "the_cn   (Value v)     = arbitrary"
  "the_cn   (ClassName c) = c"
  "the_cn   (FieldRef f)  = arbitrary"
  "the_cn   (MethodRef m) = arbitrary"

primrec
  "the_fref (Value v)     = arbitrary"
  "the_fref (ClassName c) = arbitrary"
  "the_fref (FieldRef f)  = f"
  "the_fref (MethodRef m) = arbitrary"

primrec
  "the_mref (Value v)     = arbitrary"
  "the_mref (ClassName c) = arbitrary"
  "the_mref (FieldRef f)  = arbitrary"
  "the_mref (MethodRef m) = m"




(* varValue functions - awkward naming of "the_val'" *)
consts
  the_val'   :: "varValue \<Rightarrow> val"
  the_string :: "varValue \<Rightarrow> string"
primrec
  "the_val' (Value' v) = v"
  "the_val' (String s) = arbitrary"
primrec
  "the_string (Value' v) = arbitrary"
  "the_string (String s) = s"



(* cost model functions *)
constdefs
  instrInc :: "[timecost, jGrailCode] \<Rightarrow> timecost"
  "instrInc t c \<equiv> t(c:=t(c)+1)"

  frameInc :: "[framecost, (sig \<times> cname)] \<Rightarrow> framecost"
  "frameInc f m \<equiv> f(m:=f(m)+1)"

  frameDec :: "[framecost, (sig \<times> cname)] \<Rightarrow> framecost"
  "frameDec f m \<equiv> f(m:=f(m)+-1)"

end
