header{*Grail to jGrail Translation*}
theory GrailTrans = GrailAbsyn_annotated + jGrailInstructions:

text{*\subsection*{Type Definitions}*}
types
  locals   = "Vname list"
  funs     = "Fname list"
  funtable = "(Fname \<times> nat) list"
  bytecode = "jGrailInstr list"


text{*\subsection*{Auxiliary Functions}*}

text {*The function \texttt{find} searches recursively through a finite list and returns the index of the specified element if it is found.*}
consts
  find :: "'a list \<Rightarrow> 'a \<Rightarrow> nat \<Rightarrow> nat option"

primrec
  "find [] x index = None"

  "find (head#tail) x index =
    (if head=x then
      Some(index)
    else
      find tail x (index+1))"


text{*\vspace{0.5cm}The functions \texttt{localIndex} and \texttt{funIndex} are lookup functions for variable and function names in their respective lists. They make direct use of the \texttt{find} function defined above.*}
consts
  localIndex :: "locals \<Rightarrow> Vname \<Rightarrow> nat option"
  funIndex   :: "funs \<Rightarrow> Fname \<Rightarrow> nat option"

defs
  localIndex : "localIndex ls l == find ls l 0"
  funIndex   : "funIndex fs f == find fs f 0"


text{*\vspace{0.5cm}The function \texttt{name\_of\_fun} and \texttt{name\_of\_arg} simple extract the function name and variable name of a function declaration or an argument declaration, respectively.*}
consts
  name_of_fun :: "FunDec \<Rightarrow> Fname"
  name_of_arg :: "Arg \<Rightarrow> Vname"

primrec
  "name_of_fun (FDEC f x l) = f"
primrec
  "name_of_arg (ARG t v) = v"


text{*\vspace{0.5cm}The function \texttt{make\_funs} represents the first pass across the finite list of function declarations in a method body. A finite list of the names of the functions defined is returned.*}

consts make_funs :: "FunDec list \<Rightarrow> funs"

primrec
  "make_funs [] = []"
  "make_funs (h#t) = [(name_of_fun h)]@(make_funs t)"


text{*\vspace{0.5cm}The functions \texttt{merge\_locals} and \texttt{merge\_locals\_args} merge two finite lists of variable names, and a finite list of variable names with the list of variable names extracted from a finite list of argument declarations, respectively.*}
consts
  merge_locals      :: "locals \<Rightarrow> locals \<Rightarrow> locals"
  merge_locals_args :: "locals \<Rightarrow> Arg list \<Rightarrow> locals"

primrec 
 "merge_locals a [] = a"
 "merge_locals a (h#t) = (if (localIndex a h) = None then
                            merge_locals (a@[h]) t
                        else
                            merge_locals a t)"

defs
  merge_locals_args : "merge_locals_args a b ==
                         merge_locals a (map name_of_arg b)"


text{*\subsection*{Translation Functions}*}

text{*There are 9 translation functions, each of which translates a different class of Grail abstract syntax node to jGrail bytecode. The functions also return an annotation of the abstract syntax tree, where each node is carries the position of its translation in the resulting bytecode.*}

consts
  transValue  ::"[nat, locals,       Value  ] \<Rightarrow> bytecode \<times> Value_a"
  transBinOp  ::"[nat,               BinOp  ] \<Rightarrow> bytecode \<times> BinOp_a"
  transPrimOp ::"[nat, locals,       PrimOp ] \<Rightarrow> bytecode \<times> PrimOp_a"
  transPrimRes::"[nat, locals, funs, PrimRes] \<Rightarrow> bytecode \<times> PrimRes_a"
  transTest   ::"[nat,      int,     Test   ] \<Rightarrow> bytecode \<times> Test_a"
  transLetDecs::"[nat, locals, funs, LetDecs] \<Rightarrow> bytecode \<times> locals \<times> LetDecs_a"

  transFunDec   :: "[nat, locals, funs, FunDec ] \<Rightarrow>
                      bytecode \<times> locals \<times> FunDec_a"
  transFunDecs  :: "[nat, locals, funs, FunDec_a list, FunDec list] \<Rightarrow>
                      bytecode \<times> locals \<times> funtable \<times> FunDec_a list"
  transMethBody :: "[locals, MethBody] \<Rightarrow> bytecode \<times> MethBody_a"

primrec
  "transValue pos locals (VARval v) =
      ([iload (the (localIndex locals v))], VARval_a pos v)"

  "transValue pos locals (INTval i) =
      ([sipush i], INTval_a pos i)"
(*
  "transValue pos locals (NULLval s) = ([], NULLval_a pos s)"
  text{*Note: Null values are not supported, and we translate a null value as an empty list of instructions. As a result, any program containing null values, when translated, will not execute properly.*}
 *)


primrec
  "transBinOp pos ADDop = ([iadd], ADDop_a pos)"
  "transBinOp pos SUBop = ([isub], SUBop_a pos)"
  "transBinOp pos MULop = ([imul], MULop_a pos)"

primrec
  "transPrimOp pos locals (VALop v) =
      (let
          (code,t) = transValue pos locals v
      in
          (code, VALop_a pos t))"

  "transPrimOp pos locals (BINop b v1 v2) =
      (let
          (code,t)      = transValue pos locals v1;
          pos'          = pos + (length code);
          (code',t')    = transValue pos' locals v2;
          pos''         = pos' + (length code');
          (code'',t'')  = transBinOp pos'' b
      in
          (code @ code' @ code'', BINop_a pos t'' t t'))"

text{*Note: the translation of the other six primitive operations is, as yet, undefined.*}
(*  "transPrimOp pos locals (INVOKESTATICop m xs) =
    ([], INVOKESTATICop_a pos m xs)"
  "transPrimOp pos locals (PUTSTATICop x v)     =
    (let (code,t) = transValue pos locals v in
      ([], PUTSTATICop_a pos x t))"
  "transPrimOp pos locals (GETSTATICop x)       =
    ([], GETSTATICop_a pos x)"
  "transPrimOp pos locals (PUTFIELDop x f v)    =
    (let (code,t) = transValue pos locals v in
      ([], PUTFIELDop_a pos x f t))"
  "transPrimOp pos locals (GETFIELDop x f)      =
    ([], GETFIELDop_a pos x f)"
  "transPrimOp pos locals (NEWop c)             =
    ([], NEWop_a pos c)"
*)

primrec
  "transPrimRes pos locals funs (OPres p) =
      (let
          (code,t) = transPrimOp pos locals p
      in
          (code @ [ireturn], OPres_a pos t))"
 
  "transPrimRes pos locals funs (VOIDres) =
      ([return], VOIDres_a pos)"

  "transPrimRes pos locals funs (FUNres f xs) =
      ([goto (int (the (funIndex funs f)))], FUNres_a pos f xs)"


primrec
  "transTest pos n EQUALStest = ([if_icmpeq n], EQUALStest_a pos)"

  "transTest pos n LESStest = ([if_icmplt n], LESStest_a pos)"

  (*not supported
  "transTest pos n IStest = ([], IStest_a pos)"
*)

primrec
  "transLetDecs pos locals funs (VALdec v p ls) =
     (let
         locals' = merge_locals locals [v];
         index = the(localIndex locals' v)
     in
         (let
             (code, t)  = transPrimOp pos locals' p;
             pos' = (pos + 1) + (length code);
             (code', locals'', t') = transLetDecs pos' locals' funs ls
         in
             (code @ [istore index] @ code', locals', VALdec_a pos v t t')))"

(*
  "transLetDecs pos locals funs (VOIDdec p ls) =
     (let
         (code,t) = transPrimOp pos locals p;
         (code',locals',t') = transLetDecs pos locals funs ls
     in
         (code, locals', VOIDdec_a pos t t'))"
*)
  "transLetDecs pos locals funs (PRIMres p) =
     (let
         (code,t) = transPrimRes pos locals funs p
     in
         (code, locals, PRIMres_a pos t))"

  "transLetDecs pos locals funs (CHOICEres v1 tst v2 p1 p2) =
     (let
         (code,t)          = transValue   pos locals v1;
         pos'              = pos + (length code);
         (code',t')        = transValue   pos' locals v2;
         pos''             = pos' + (length code');
         (code''',t''')    = transPrimRes (pos''+1) locals funs p2;
         (code'',t'')      = transTest    pos'' (int (length code''')) tst;
         pos'''            = (pos'' + 1) + (length code''');
         (code'''', t'''') = transPrimRes pos''' locals funs p1
     in
         (code @ code' @ code'' @ code''' @ code'''', locals, CHOICEres_a pos t t'' t' t'''' t'''))"




text{*The following are the definitions of the functions for translation function declarations, and finally the method body and declaration itself.*}

primrec
  "transFunDec pos locals funs (FDEC f xs ls) =
      (let
          locals' = merge_locals_args locals xs;
          (code, locals'', t) = transLetDecs pos locals' funs ls
      in
          (code, locals'', FDEC_a pos f xs t))"

primrec
  "transFunDecs pos locals funs ts []     = ([], locals, [], ts)"

  "transFunDecs pos locals funs ts (h#tail) =
      (let
          (code, locals', t) = transFunDec pos locals funs h;
          new_pos = pos + (length code);
          (code', locals'', ft, ts')  = transFunDecs new_pos locals' funs ts tail;
          ft' = [(name_of_fun h, pos)]@ft
      in
          (code @ code', locals'', ft', t#ts'))"

primrec
  "transMethBody args (MBODY fs ls) =
     (let
         funs = make_funs fs;
         (code, locals, t) = transLetDecs 0 args funs ls;
         pos = length code;
         (code', locals', funtable, t') = transFunDecs pos locals funs [] fs
     in
         (code @ code', MBODY_a t' t))"



end