structure Translate =
struct
structure A = Absyn
structure C = CamelotAbsyn
(* Each definition or declaration, whether of a type
 * or function or whaterver else, in Camelot each definition
 * cannot clash with any other, but in Galahad we can make
 * declarations within modules, hence each definition must be
 * given a unique name. Each name that is visible (in scope) 
 * will be held in the label environment, when we come across
 * the use of such a name we look it up in the label
 * environment and change the name*)
val labels = ref 0
(* Here we return a new unique label, the additional string is
 * appended as a suffix, this is purely to increase the readability
 * of the produced Camelot code. So for example a variable declaration
 * val no_of_students : int
 * can turn translated as
 * val L_22_no_of_students
 * instead of the less readable
 * val L_22*)
fun newlabel s = (labels := !labels+1;
		  Symbol.symbol ("L" ^ (Int.toString (!labels)) ^ "_" ^ s))


(* When type constructors are applied in Camelot they are just the same
 * as an ordinary function application. However we also want to allow
 * output to SML, so we give a slightly different function to the above
 * to allow type constructors to have a different prefix than real function
 * names, this means that when we come to output SML code we can tell
 * type constructor use from a proper function application and thus output
 * them differently (since type constructors in ML require their arguments
 * to be surrounded by brackets and separated by commas)*)
fun new_type_label s = (labels := !labels+1;
			Symbol.symbol ("T" ^ (Int.toString (!labels)) ^ "_" ^ s))


(* We use the same trick for arrays, currently (13/11/2002) this is
 * mostly for output to SML, but may be needed for output to Camelot
 * at some later stage
 * UPDATE: The trick isn't actually used at the moment but it doesn't do
 * any harm*)
fun new_array_label s = (labels := !labels+1;
			Symbol.symbol ("A1" ^ (Int.toString (!labels)) ^ "_" ^ s))

type label = Symbol.symbol
type label_env = label Symbol.table

(* When we use a type name, we have to use the label environment to
 * convert the type name (which is currently in scope) to it's unique
 * name, however the built in types, just map to themeselves. You can
 * however override a built-in type's meaning, but in that case the
 * environment will take care of that. To sum up here we are adding a mapping
 * from the built-in types to themeselves so that they can be found in
 * environment when searched for*)
val base_typeEnv = foldl (fn (s, table) => Symbol.enter(table,
							Symbol.symbol s,
							Symbol.symbol s))
			 Symbol.empty
			 ["int", "char", "bool", "float", "string",
			  "unit", "array"(*??*), "list"]

(* Now we require a table for variables, however we also require one for
 * modules and functors, so we make these three tables into one.
 * The variable entries will only require that we hold the label associated
 * with that variable, that is the munged name of that variable, so that when
 * we encounter a use of that variable we can correctly use the munged name.
 * Now we also need a module entry, a module entry maps
 * module names to environments label environments. This is so that,
 * when we encounter a use of a construct defined within a module but
 * used outwith that module we can find the correct environment within
 * which to find the correct munched name. So for example if we define
 * a module A and within that module A we define the function f.
 * Then within say the main program we might have a reference to the
 * the function f which will look like:
 * A.f <args>
 * So we look up the name A in the module environment, and we are
 * returned a label environment. In this label environment we look up
 * the name f and we find that we get L21_f (or whatever f was mapped
 * to when we munged the definition) so now we can change the function
 * called to L21_f instead of A.f. We need to label environments, one for
 * types and one for val declarations.
 * If we allow nested modules, then we have to
 * also munch the module name, and so module_entry will be three tables, one
 * for types, one for values and one for modules.
 * Also looking up the correct label environment would have to be recursive*)

(* The functor entry is just like the module entry, except
 * that it maps names to functor implementations, when we come across
 * a module definition that uses a functor, for example
 * module A = B(increment), then we translate the functor initialising
 * arguments to their given implementations.
 * Unfortunately we must also store the val and type environments at the
 * time of the functor definition. This is because we might overide a function
 * used in the functor definition before the functor is applied, in which case
 * we will use the wrong function when we translate the module made up of 
 * the funtor's definitions and the arguements to the functor*)

(* Instead of having three tables for the modules, functors and values,
 * if we force their names to be disjoint (ie A program cannot see both
 * a module and a value of the same name at the same time) then we only need
 * one table, with each entry being a valEntry which is a datatype that
 * can either be a value entry, a module entry or a functor entry.
 * Now added a signature entry to allow signatures to be defined and referenced
 * by name*)
datatype valEntry = ValEntry of label | ModEntry of (label_env * valEntry Symbol.table)
		  | FunctEntry of (label_env * valEntry Symbol.table * A.functordef)
		  | ArrayEntry of label
		  | SignatureEntry of A.signatureExp



(* The array get and set operations are added to the environment, the
 * rest of the library should also be added, we add exit, to allow a
 * graceful exit for example if an assert fails*)
val base_valEnv = foldl (fn (s, table) => Symbol.enter (table,
							Symbol.symbol s,
							ValEntry(Symbol.symbol s)))
			Symbol.empty
			["_", "set", "get_array_element", "exit"]




			 
exception Unimplemented
exception Impossible (*Compiler error*)
exception Semantic (*The program is incorrect*)

(*This just makes error reporting a little neater*)
fun Error s = TextIO.output(TextIO.stdOut, s)

fun translate (A.GPROG(declarations, m)) = 
    let
	(* Arrange essentially turns all the definitions of a Galahad
	 * program into valid definitions of a Camelot program.
	 * To do this it must arrange them in the order required by
	 * Camelot, translate non-Camelot definitions into Camelot definitions,
	 * and finally translate expressions into Camelot expressions (which
	 * is mostly a task in simplification).
	 * In order to munge names, that is make sure every declaration has
	 * a unique name, we must keep track of label environments. Label
	 * environments map from the names that are in scope in the Galahad
	 * program to unique names in the flat name space of Camelot.
	 * To do this arrange takes in a type label environment, and a val label
	 * environment. We need two different enivronments since we want to allow
	 * variables and functions to have the same name as types, and hence must
	 * have a separate mapping. We then have a third environment, this is
	 * the module environment and maps modules to their two (type and val)
	 * environments, this is because an expression which "opens" up a module
	 * (ie A.<fname> or something)
	 * must be allowed to see the definitions inside that module.
	 * The module environment however is much easier to maintain,
	 * if we do not allow nested modules (and also we don't allow module
	 * definitions within ANY kind of other defintition, that is we don't
	 * allow for module definitions to be within, for example a function definition)
	 * for definitions that are not modules we simply return the given module
	 * environment, when we are translating a module we will be returned by the
	 * recursive calls to arrange, the two label environments which must be newly
	 * stored in the module environment.
	 * To translate expressions arrange calls upon translateExp, which
	 * takes a type and val label  environment and returns 
	 * the translated expression
	 * The auxiliary functions transTypeDec, transValDec, transFunDef etc, ie the
	 * top level translate functions don't strictly need to be defined separate,
	 * the code contained within each could just be contained within the code for
	 * arrange, but I think this way is a little neater*)
	fun transTypeDec {typeEnv=typeEnv, valEnv=valEnv} 
			 (pos, A.TYPEdec (tvarl, var, tyconl)) =
			 let
			     (* Currently there are no type declarations in Galahad that are
			      * not also in Camelot, therefore a type definition need only be
			      * changed in name, we must change the type name, and the names
			      * of the constructors (since otherwise we might not know which constructor
			      * we were meaning to use). Each of the type variables (of the form 'a)
			      * must be added to the environment that is used to translate the types
			      * of the constructors, there is no need to munge them since they are
			      * only in scope for the length of the type declaration.*)
			     (* Here is the new name for the type*)
			     val new_typename = newlabel (Symbol.name var)
						
			     (* We have to add in the type name since it might be a recursive
			      * type and hence will be needed when translating the rest of
			      * type definition*)
			     val fst_newtypeEnv = Symbol.enter (typeEnv,
								var,
								new_typename)
						  
						  
			     val Snd_newTypeEnv = foldl (fn ((_,x),y) => Symbol.enter(y,x,x)) fst_newtypeEnv tvarl
						  
			     (* For each constructor we have to munch it's name, that is we must give it
			      * a unique name. The reason is we want to be able to have different constructors
			      * with the same name (because they are not in scope at the same time, for example
			      * they may be defined in separate modules).
			      * We take in a typecon, a list of translated typecons and a label environment,
			      * and we return the list of translated typecons appended with the translation
			      * of the one given, and the augmented label environment.
			      * We call new type label here so that type constructor labels are given
			      * different prefixes, this is simply to allow us to distinquish them from
			      * ordinary functions applications if/when we output to SML*)
			     fun translateTypeConL ((pos,A.TYPEcon(var,ty_list)), (tycons, vEnv)) =
				 let val newvar = new_type_label (Symbol.name var)
				 in
				     ((pos,C.TYPEcon(Symbol.name newvar,
						     (* Note that we translateTyL with the second new typeEnv
						      * and not the one given to translateTypeConL*)
						     (map (translateTyL (Snd_newTypeEnv,vEnv)) ty_list)))::tycons,
				      Symbol.enter (vEnv,
						    var,
						    ValEntry(newvar)))
				 end
			
			     (* This is just because the Camelot absyn represents a Var as a string
			      * while the Galahad absyn represents a Var as a symbol*)
			     val new_tvarl = map (fn (x,y) => (x,Symbol.name y)) tvarl
				
			     (* The constructors are added to the *valEnv* so that they can be used to create
			      * instances of that type*)
			     val (new_tyconl,newValEnv) = foldr translateTypeConL ([], valEnv) tyconl
							  
			     val newt = (pos, C.TYPEdec(new_tvarl, Symbol.name new_typename, new_tyconl))
			 (* Note when we return we return the first (fs_newtypeEnv) and not the second
			  * (Snd_newtypeEnv) this is because the ones added to create the second are only
			  * in scope for the length of the type declaration*)
			 in {new_t=newt,
			     typeEnv=fst_newtypeEnv,
			     valEnv=newValEnv}
			 end

	and transValDec {valEnv=valEnv,
			 typeEnv=typeEnv} (pos,A.VALdec(vname,ty,ext)) =
	    (* Val declarations are pretty straightforward, we just munge the names,
	     * we also need to watch our for the types, these need to be munged as well*)
	    let val new_valdecname = if ext = false
				     then case ty
					   of (_,A.ARRAYty(_)) => new_array_label (Symbol.name vname)
					    | _ => newlabel (Symbol.name vname)
				     else vname
		(* If it is an array then we have to add it as an array so that get and set operations
		 * on the array will be translated correctly*)
		val new_valEnv = case ty 
				  of (_,A.ARRAYty(_)) => Symbol.enter (valEnv,
								       vname,
								       ArrayEntry(new_valdecname))
				   | _ => Symbol.enter(valEnv,
						       vname,
						       ValEntry(new_valdecname))
		val newValdec = (pos, C.VALdec(Symbol.name new_valdecname,
					       (* Whether it is the new valEnv or the old one
						* passed into translateTyL makes little difference
						* since we haven't added a module name to the old one*)
					       translateTyL (typeEnv,new_valEnv) ty))
	    in 
		{newValdec=newValdec,
		 newvalEnv=new_valEnv}
	    end 
	and transFunDef {valEnv=valEnv,
			 typeEnv=typeEnv} fdlist =
	    let (* We first enter all the names of the functions in the list
		 * into the val environment, this is because the functions are
		 * potentially mutually recursive, a function may call a function
		 * which occurs after it in the list, therefore we munge all the names
		 * before translating all the expressions since an expression may need
		 * the munged name of a function occuring after it in the list,
		 * Note that currently the boolean (first) component of the FUNdef is
		 * ignored, see Absyn.sml*)
		fun enter_name ((pos, A.FUNdef(_,var,_,_)), valEnv) =
		    Symbol.enter (valEnv,
				  var, ValEntry(newlabel (Symbol.name var)))

		fun auxTransFunDef ((pos, A.FUNdef(_, var, args, exp)),
				 {new_fdlist=old_fdlist,
				  typeEnv=typeEnv,
				  valEnv=valEnv}) =
		    let 
			val new_fname = case Symbol.look(valEnv, var)
					 of SOME(ValEntry(v)) => v
					  | SOME(_) => (Error ("Looking up function " ^ (Symbol.name var) ^
							       " not a value\n");
							raise Semantic)
					  | NONE => raise Semantic
			(* To translate the expression the arguments must
			 * be added to the val environment so that when one
			 * is used we don't think that it is a semantic error,
			 * also the same name may have already been added before
			 * say as a function name by an earlier definition.
			 * The args don't need to be munged if we don't allow nested
			 * functions. However it doesn't do any harm to munch the arg
			 * names so we do so anyway
			 * This temporary environment is temporary since we only use
			 * it to translate the function's expression, after that we
			 * don't need (or want) it anymore and we return to the original
			 * environment.*)
			(* Builds up a list of oldname,newname pairs*)
			(* While we aren't allowing nested functions the arguments don't need to be munged
			 * so for just now to increase the readability of the generated code I'm not munging them*)
			val munged_args = map (fn x => (x, x(*newlabel (Symbol.name x)*))) args
			(* Takes the above list and adds each mapping to the val environment*)
			val temp_valEnv = foldl (fn ((oldname,newname),env) => Symbol.enter(env,
											    oldname,
											    ValEntry(newname)))
						valEnv
						munged_args
			val new_arg_names = map (fn (_,x) => Symbol.name x) munged_args
			val {exp=newExp, funs=new_funs}  = translateExp {typeEnv=typeEnv,
									 valEnv=temp_valEnv} exp
			val new_fd = (pos, C.FUNdef(Symbol.name new_fname,
						    new_arg_names, newExp))
				     
		    in (* When we return we throw away the environments used to compute
			* expression, since those defines that occured within the function
			* as arguments are now out of scope. The function name itself should
			* have already been added to the val environment when we entered all
			* the names in the list (see enter_name)*)
			{new_fdlist = old_fdlist@new_funs@[new_fd],
			 typeEnv=typeEnv,
			 valEnv=valEnv}
		    end

		(* As stated above we need a new environment within which
		 * to translate all the functions, which contains all the function
		 * names in order to facilitate recursive calls, so here it is*)
		val fnames_valEnv = foldl enter_name valEnv fdlist
	    in foldl auxTransFunDef {new_fdlist=[],
				     typeEnv=typeEnv,
				     valEnv=fnames_valEnv} fdlist
	    end
		


	and transModuleDef {typeEnv=typeEnv,
			    valEnv=valEnv} (v, msignature, mExp) =
	    (* If we have a proper signature, we must translate the body of the module
	     * with its own environment. We iterate through the val declarations
	     * in the signature, for each one we check that such a declaration has been added to
	     * the module's body environment, if not we raise and error, if so then (by looking
	     * in the environment) we have the new-name for the val which we can add to the
	     * current environment, once all of those are added then only those that are
	     * in the signature will be added to the module environment and hence visible
	     * outside the module.
	     * If we do not have a signature then we basically want to add all the definitions
	     * to the environment stored under the module name, since we basically want all the
	     * definitions inside the module to be implicitly visible. In this case for just now
	     * we just store the current environment, which is incorrect (see below).
	     * Note that the types are NOT checked, this would involve full semantic analysis
	     * which should be added but for now we are letting the Camelot compiler perform
	     * the semantic analysis, although this will not work here it is a temporary solution.
	     * The problem essentially boils down to the fact that we do not want to do full
	     * semantic analysis here only to output the code to the Camelot compiler which will
	     * perform the semantic analysis for us, however not doing full semantic analysis here
	     * does restrict us slightly as is shown in the above example. The ideal solution would
	     * be to do the full semantic analysis but instead of outputting the Camelot code we
	     * could just use the Camelot compiler's back end*)
	    let val {defs=(mts,mvs,mfs),
		     typeEnv=module_body_typeEnv,
		     valEnv=module_body_valEnv,
		     ...} = case mExp
			     of A.MODEXP(dfs) =>
				foldl arrange {defs=([],[],[]),
					   (* There is a bug here, since we pass to arrange typeEnv,
					    * valEnv and moduleEnv, 
					    * rather than 3 empty environments, 
					    * this is so that the definitions inside the module
					    * can see definitions made outside the module as it should be able to.
					    * But it means that the environments made from folding left 
					    * contain the mappings for definitions made outside the module,
					    * we then place these in the moduleEnv
					    * as being associated with this module.
					    * See the modEnv_bugTest.gal test program
					    * for an example of how this can be exposed,
					    * note that it isn't disastrous since
					    * the function does exist and we do call the correct one, 
					    * it's just that we shouldn't
					    * be using a module opener (module_name.fun_name)
					    * to access that function since
					    * it is defined outside the module.
					    * Also wherever the module itself is visible the definitions erroraneously
					    * visible inside the module would be visible outside anyway.
					    * This bug does not occur when the module has a signature and we should be able
					    * to fix this in a similar way (once we allow the signatures to have any kind
					    * of definition in them)*)
					   typeEnv=typeEnv,
					   valEnv=valEnv} dfs
			      (* If it is just a name expressions (so the whole thing is something like module A = B)
			       * then we just look up the environments, we may constrain the module further, but
			       * the code below will take care of that*)
			      | A.MODNAME(alias) => (case Symbol.look (valEnv, alias)
						      of SOME(ModEntry(t,m)) => {defs=([],[],[]),
										 typeEnv=t,
										 valEnv=m}
						       | _ => (Error "Module expression name does not exist";
							       raise Semantic))


		
		fun process_signatures a [] = a
		  | process_signatures {valEnv=vE,
					typeEnv=tE} (h::t) =
		    let
			(* We ignore the external argument (indicating it is a declaration
			 * of an external function), this is because it should not be
			 * an external declaration within a signature, this should maybe
			 * raise an error but for now we ignore it*)
			fun add_vsig_variables ((pos, A.VALdec(vname, ty, _)),
						labEnv) =
			    (case Symbol.look(module_body_valEnv,
					      vname)
			      of SOME(v as ValEntry(_)) => Symbol.enter(labEnv,vname, v)
			       | SOME(_) => (Error "Module does not match signature values";
					     raise Semantic)
			       | NONE => (Error "Module does not match signature";
					  raise Semantic))
			    

			fun add_tsig_variables ((pos, A.TYPEmarker(tname)),
						labEnv) =
			    (case Symbol.look (module_body_typeEnv,
					       tname)
			      of SOME(v) => Symbol.enter(labEnv, tname, v)
			       | NONE => (Error ("Module does not match signature\n missing type"
						 ^ (Symbol.name tname) ^ "\n");
					  raise Semantic))
		    in
		    (case h
		      of A.SIGEXP(tsiglist,vsiglist) => 
			 process_signatures {valEnv=foldl add_vsig_variables vE vsiglist,
					     typeEnv=foldl add_tsig_variables tE tsiglist} t
		       | A.SIGNAME(v) => (case Symbol.look (valEnv,v)
					   of SOME(SignatureEntry(A.SIGEXP(tsiglist,vsiglist))) =>
					      process_signatures {valEnv=foldl add_vsig_variables vE vsiglist,
								  typeEnv=foldl add_tsig_variables tE tsiglist} t
					   | SOME(SignatureEntry(A.SIGNAME(_))) => raise Impossible
					   | _ => raise Semantic))
		    end


									   

		val {valEnv=new_module_body_valEnv,
		     typeEnv=new_module_body_typeEnv} = case msignature
							 of [] => {valEnv=module_body_valEnv,
								  typeEnv=module_body_typeEnv}
							  | l => process_signatures {valEnv=Symbol.empty,
										     typeEnv=Symbol.empty} l

(*
		val new_module_body_valEnv = 
		    case msignature
		     of NONE => module_body_valEnv
		      | SOME(A.SIGEXP(_,vsiglist)) => foldl add_vsig_variables Symbol.empty vsiglist
		      | SOME(A.SIGNAME(v)) => (case Symbol.look(valEnv, v)
						of SOME(SignatureEntry(A.SIGEXP(_,vsiglist))) =>
						   foldl add_vsig_variables Symbol.empty vsiglist
						 | SOME(SignatureEntry(A.SIGNAME(_))) => raise Impossible
						 | _ => raise Semantic)
							      
		val new_module_body_typeEnv = 
		    case msignature
		     of NONE => module_body_typeEnv
		      | SOME(A.SIGEXP(tsiglist,_)) => foldl add_tsig_variables Symbol.empty tsiglist
		      | SOME(A.SIGNAME(v)) => (case Symbol.look(valEnv, v)
						of SOME(SignatureEntry(A.SIGEXP(tsiglist,_))) =>
						   foldl add_tsig_variables Symbol.empty tsiglist
						 | SOME(SignatureEntry(A.SIGNAME(_))) => raise Impossible
						 | _ => raise Semantic)*)
						    
		val new_valEnv = Symbol.enter (valEnv, v,
					       ModEntry(new_module_body_typeEnv,
							new_module_body_valEnv))
	    (* Here we don't return the label environments computed
	     * when all the above definitions are translated, since outside
	     * the module they will only be accessable with dot notation, and
	     * hence not visible on their own, they can only be accessed with
	     * a module expression such as A.<name> in which case they will be
	     * looked up in the module environment to find the correct type and val
	     * environments to look up the type or val*)
	    in {defs=(mts,mvs,mfs),
		valEnv=new_valEnv}
	    end
	and arrange (A.TYPEDEC(t), {defs=(ts,vs,fs), 
				    typeEnv=typeEnv,
				    valEnv=valEnv}) =
	    let val {new_t=newt,
		     typeEnv=newTypeEnv,
		     valEnv=newValEnv} = transTypeDec {typeEnv=typeEnv,
						       valEnv=valEnv} t
		    
	    in
		{defs=(ts@[newt],vs,fs),
		 typeEnv=newTypeEnv,
		 valEnv=newValEnv}
	    end
	  (* Type aliases are the easiest we just add to the current type environment *)
	  | arrange (A.TYPEALIAS((pos, A.TYPEalias(alias, typ))), {defs=(ts,vs,fs),
								   typeEnv=typeEnv,
								   valEnv=valEnv}) =
	    (case Symbol.look(typeEnv, typ)
	      of SOME(l) => {defs=(ts,vs,fs),
			     typeEnv=Symbol.enter(typeEnv, alias, l),
			     valEnv=valEnv}
	       | NONE => (Error ("type " ^
				 (Symbol.name typ) ^
				 " does not exist\n");
			  raise Semantic))

	  | arrange (A.VALDEC(v), {defs=(ts,vs,fs),
				    typeEnv=typeEnv,
				    valEnv=valEnv}) =
	    let val {newValdec=newValdec,
		     newvalEnv=new_valEnv} = transValDec {valEnv=valEnv,
							  typeEnv=typeEnv} v
	    in
		{defs=(ts, vs@[newValdec], fs),
		 typeEnv=typeEnv,
		 valEnv=new_valEnv}
	    end
		
	  (* For the time being this does not handle fundef lists of more than one*)
	  | arrange (A.FUNDEF(fdlist),  {defs=(ts,vs,fs), 
					 typeEnv=typeEnv,
					 valEnv=valEnv}) =
	    let val {new_fdlist=trans_fdList,
		     typeEnv=newTypeEnv,
		     valEnv=newValEnv} = transFunDef {valEnv=valEnv,
						      typeEnv=typeEnv} fdlist
		    
	    in
		{defs=(ts,vs,fs@[trans_fdList]),
		 typeEnv=newTypeEnv,
		 valEnv=newValEnv}
	    end

	  | arrange (A.MODULEDEF(mod_name, sigExps, mod_exp), {defs=(ts,vs,fs),
				      typeEnv=typeEnv,
				      valEnv=valEnv}) =
 	    let val {defs=(mts,mvs,mfs),
		     valEnv=new_valEnv} = transModuleDef {typeEnv=typeEnv,
							  valEnv=valEnv} (mod_name, sigExps, mod_exp)
	    in {defs=(ts@mts,vs@mvs,fs@mfs),
		typeEnv=typeEnv,
		valEnv=new_valEnv}
	    end
	  | arrange (A.FUNCTORDEF(a as A.FUNCTOR(ftname, args, defines)),{defs=(ts,vs,fs),
										  typeEnv=typeEnv,
										  valEnv=valEnv}) =
	    (* A functor def is easy, since we don't want to translate the body until the functor
	     * is actually used and hence the arguments have been filled in.
	     * We store the current type and val Env which will be used to translate the body
	     * of the functor when the functor is applied*)
	    let val new_valEnv = Symbol.enter(valEnv, ftname, FunctEntry(typeEnv, valEnv, a))
	    in {defs=(ts,vs,fs),
		typeEnv=typeEnv,
		valEnv=new_valEnv}
	    end
	  (* See the old legacy code at the bottom of this file, I think you'll agree that
	   * this is far neater*)
	  | arrange (A.FUNCTORAPP (module_name, functor_name, fsignature, modExps),{defs=(ts,vs,fs),
										    typeEnv=typeEnv,
										    valEnv=valEnv}) =
	    let val (functorTypeEnv,
		     functorValEnv,
		     A.FUNCTOR(ftname, fargs, functor_defines)) = 
		    case Symbol.look(valEnv, functor_name)
		     of SOME(FunctEntry(a)) => a
		      | SOME(_) => (Error "Name not defined as a functor\n";
				    raise Semantic)
		      | NONE => (Error ("Error while defining " ^
					(Symbol.name module_name) ^
					". No such functor " ^
					(Symbol.name functor_name) ^ "\n");
				 raise Semantic)

		fun addArgs v ([],[]) = v
		  | addArgs _ (_,[]) = (Error "not enough arguments given to functor";
					raise Semantic)
		  | addArgs _ ([],_) = (Error "too many arguments given to functor";
					raise Semantic)
		  | addArgs {defs=(ts, vs, fs),
			    valEnv=venv} ((farg,sigs)::restfargs, mExp::restmExps) = 
		    let val {defs=(nts, nvs,nfs),
			     valEnv=vE} = transModuleDef {typeEnv=typeEnv,
							  valEnv=venv} (farg, sigs, mExp)
		    in addArgs {defs=(ts@nts, vs@nvs, fs@nfs), valEnv=vE} (restfargs, restmExps)
		    end


		(* First of all we translate the module expressions
		 * given as arguments in the current environment *)
		val {defs=(mts, mvs, mfs),
		     valEnv=Venv} = addArgs {defs=([],[],[]),
					     valEnv=valEnv} (fargs, modExps)

		(* Then for each argument name we look up the definition in the new Venv
		 * just calculated, and add it's entry to the functorValEnv, this gives
		 * us the environment which will be used to translate the defines within
		 * the functor*)
		val functorVenv = foldl (fn ((x,_),y) => 
					    case Symbol.look (Venv, x)
					     of SOME(v as ModEntry(_)) => Symbol.enter(y,x,v)
					      | _ => (Error "Module name not entered\n";
						      raise Semantic)) functorValEnv fargs
				  
		(* We then translate the functor, expressions within the functor will access
		 * it's arguments which have been added to this environment*)
		val {defs=(fts,fvs,ffs),
		     valEnv=fvenv} = transModuleDef {typeEnv=typeEnv,
						     valEnv=functorVenv} 
						    (module_name, fsignature, A.MODEXP(functor_defines))
			
		(* Finally the environment that we actually return should be the same
		 * as the one given except that it should have the new module defined
		 * by the functor application added to it, to do this we look it
		 * up in the environment returned from the above*)
		val EndValVenv = case Symbol.look(fvenv, module_name)
				  of SOME(v as ModEntry(_)) => Symbol.enter(valEnv, module_name, v)
				   | _ => raise Impossible

	    in{defs=(ts@mts@fts, vs@mvs@fvs, fs@mfs@ffs),
		typeEnv=typeEnv,
		valEnv=EndValVenv}
	    end

	    

	  | arrange (A.SIGDEF(sig_name, sigExp), {defs=(ts,vs,fs),
						  typeEnv=typeEnv,
						  valEnv=valEnv}) =
	    (* We have two choices here we can simply store the sigExp with
	     * the sig_name, or if the sigExp turns out to be just a name
	     * then we could look it up until we find an actual sig expression
	     * and store that, I will go for the later here so that any function
	     * which has to look in the environment for a signature expression can
	     * handle the case where it is given back a name by raising an Impossible
	     * exception*)
	    let val newValEnv =
		    case sigExp
		     of A.SIGNAME(v) => (case Symbol.look(valEnv, v)
					  (* it appears here as if we should recursively look up the name,
					   * but only SIGEXP's should be added to the environment, therefore
					   * the first lookup should return a SIGEXP and NOT a SIGNAME*)
					  of SOME(SignatureEntry (s as A.SIGEXP(_))) =>
					     Symbol.enter(valEnv, sig_name, SignatureEntry(s))
					   | SOME(SignatureEntry (A.SIGNAME(_))) => raise Impossible
					   | _ => (Error "Signature name unrecognised\n"; raise Semantic))
		      | s as A.SIGEXP(_) => Symbol.enter(valEnv, sig_name, SignatureEntry(s))
	    in {defs=(ts,vs,fs),
		typeEnv=typeEnv,
		valEnv=newValEnv}
	    end


		     





	(* translateTyL must take a TyL and return the equivalent TyL translated,
	 * for the simple built-in types this is straightforward, we just return
	 * the TyL given to us, but for the composite types (Arrow, Con List and Array,
	 * basically those that may contain other TyL's) then we have to recursively
	 * translate them, since they may down the line contain a TVARty which has to
	 * have it's name translated according to the labelEnv.
	 * We must now also pass the val environment into translateTyL since we may be
	 * accessing a type declared within a module, even though in most case it can
	 * be safely ignored*)
	and translateTyL _ ((p,A.INTty)) = (p,C.INTty)
	  | translateTyL _ ((p,A.CHARty)) = (p,C.CHARty)
	  | translateTyL _ ((p,A.BOOLty)) = (p,C.BOOLty)
	  | translateTyL _ ((p,A.FLOATty)) = (p,C.FLOATty)
	  | translateTyL _ ((p,A.STRINGty)) = (p,C.STRINGty)
	  | translateTyL _ ((p,A.UNITty)) = (p,C.UNITty)
	  | translateTyL _ (pos,A.TVARty(var)) =
	    (* We don't add 'a type variables into the environment,
	     * so we don't look them up here as might be expected, instead
	     * we just translate them as is*)
	    (pos, C.TVARty(Symbol.name var))
	  (*	    (case Symbol.look(labelEnv,
				      var)
	      of SOME(v) => (pos,C.TVARty(Symbol.name v))
	       | NONE => raise Semantic)*)
	  | translateTyL (labelEnv,vEnv) (pos,A.ARRAYty(ty)) = 
	    (pos,C.ARRAYty(translateTyL (labelEnv,vEnv) ty))
	  | translateTyL (labelEnv,vEnv) (pos,A.ARROWty(ty1,ty2)) =
	    (pos, C.ARROWty(translateTyL (labelEnv,vEnv) ty1,
			      translateTyL (labelEnv,vEnv) ty2))
	  | translateTyL (labelEnv,vEnv) (pos,A.LISTty(ty)) =
	    (pos,C.LISTty(translateTyL (labelEnv,vEnv) ty))
	    
	  (* If it is a simple CONty, that is the type name is accessing a module
	   * then we do the simple thing and look up the type name in the type
	   * environment, see below for otherwise*)
	  | translateTyL (labelEnv,vEnv) (pos, A.CONty(tylist, A.VARval(var))) =
	    (case Symbol.look(labelEnv,var)
	      of SOME(v) => (pos,C.CONty((map (translateTyL (labelEnv,vEnv)) tylist), Symbol.name v))
	       | NONE => raise Semantic)
	  | translateTyL (labelEnv,valEnv) (pos,A.CONty(tylist, lvar)) =
	    let
		(* Now the lvar may be a name, or a name inside a module
		 * eg it might be tree or A.B.tree, so we have to recursively
		 * translate the module names to get the environments in which
		 * we should translate the rest of the type name, this is just the
		 * same as it is for translating values inside translateExp. The main
		 * difference is that the final lookup of the typename will be done
		 * in the type environment and not the value environment, that is also
		 * the reason for the extra case above, to catch when we only have an
		 * identifier, and also why here our base case is "higher up" than might
		 * otherwise be expected*)
		fun transLval (A.LVALval(l, A.VARval(r)), vE) =
		    let val (mTE,_) = case Symbol.look(vE,l)
				       of SOME(ModEntry(v)) => v
					| SOME(_) => (Error ("Symbol " ^ (Symbol.name l) ^
							     "not a module name but used as such\n");
						      raise Semantic)
					| NONE => raise Semantic
		    in case Symbol.look(mTE, r)
			of SOME (v) => v
			 | NONE => raise Semantic
		    end
		  | transLval (A.LVALval(l,r), vE) =
		    (case Symbol.look(vE, l)
		      of SOME(ModEntry((_,mVE))) => transLval (r, mVE)
		       | SOME(_) => (Error ("Symbol " ^ (Symbol.name l) ^
					    "not a module name but used as such\n");
				     raise Semantic)
		       | NONE => raise Semantic)
		  | transLval (_,_) = (* If it is not a lvalue then it is a compiler
				       * error*) raise Impossible
		val v1 = transLval (lvar, valEnv)
	    in
		(pos,C.CONty((map (translateTyL (labelEnv,valEnv)) tylist), Symbol.name v1))
	    end


	(* translateExp must take in label environments which it uses to translate
	 * the given expression, and then return the translated expression, for example
	 * any call to a function must have the name of that changed to reflect the fact
	 * that we munged the declaration name of that function inside arrange FUNdef.
	 * Also any expressions which are not legal in Camelot (but are legal in Galahad)
	 * must be translated into legal Camelot expressions*)
	  (* translateExp must now also return a list of new function declarations,
	   * this is necessary to implement unconstant loops, and also if we decide
	   * allow anonymous function declaration*)
	and translateExp {typeEnv=typeEnv,valEnv=valEnv} e = 
	let fun transExp (pos, A.VALexp(v)) = {exp=(pos, C.VALexp(transValue(v))),
					       funs=[]}
	      | transExp (pos, A.UNARYexp(u,e)) = 
		let val {exp=te, funs=funs} = transExp e
		in {exp=(pos, C.UNARYexp(transUnary u, te)),
		    funs=funs}
		end
	      | transExp (pos, A.BINexp(b,e1,e2)) = 
		let val {exp=te1, funs=funs1} = transExp e1
		    val {exp=te2, funs=funs2} = transExp e2
		in {exp=(pos, C.BINexp(transBinary b, te1, te2)),
		    funs=funs1@funs2}
		end
	      | transExp (pos, A.IFexp (e1,e2,e3)) = 
		let val {exp=te1, funs=funs1} = transExp e1
		    val {exp=te2, funs=funs2} = transExp e2
		    val {exp=te3, funs=funs3} = transExp e3
		in {exp=(pos, C.IFexp(te1,te2,te3)),
		    funs=funs1@funs2@funs3}
		end
	      | transExp (pos, A.MATCHexp(e, mrl)) = 
		(* Match expressions are a little awkward since the expressions may contain
		 * expressions that defines a new function so the foldl construction there is
		 * just doing exactly that*)
		let val {exp=te, funs=funs1} = transExp e
		    val {expList=teL, funs=funs2} = foldl (fn (matchExp, {expList=eL, funs=fl}) =>
							      let val {exp=tme, funs=mfl} = transMatchRuleL matchExp
							      in {expList=eL@[tme], funs=fl@mfl}
							      end) {expList=[],funs=[]} mrl
		in {exp=(pos, C.MATCHexp(te, teL)),
		    funs=funs1@funs2}
		end
	      (* Strictly speaking there is no need to give the declarations
	       * within a let expression unique names, UNLESS
	       * we allow nested functions, if not then the scoping
	       * rules of Camelot are sufficient.
	       * However if we don't add the name to the valEnv then when
	       * translating the in expression of the let expression if we encounter
	       * the name for example inside a VALexp then we will think that there
	       * has been a semantic error, so we need to add it to the valEnv, we could
	       * just add a mapping to itself but we may as well create a unique name
	       * while we are there.
	       *
	       * Now we are also allowing an arbitrary amount of such declarations,
	       * so we have to translate these let expressions into equivalent ones
	       * with only one declaration per let, this is simple since we just chain
	       * together let expressions.*)
	      | transExp (pos, A.LETexp (veList, e2)) = 
		let fun transDecs vE [] = translateExp {typeEnv=typeEnv,
							valEnv=vE} e2
		      | transDecs vE ((v,e as (i,e'))::rest) =
			let val new_v = if v = A.uscore_sym
					then v
					else newlabel (Symbol.name v)
			    val nextvE = Symbol.enter(vE, v, ValEntry(new_v))
						      
			    (* We have to call translateExp recursively
			     * since we need to provide it with the given vE since
			     * the initialising expression may contain a reference to
			     * a variable defined previously in the let declaration list*)
			    val {exp=te, funs=funs1} = translateExp {typeEnv=typeEnv,
								     valEnv=vE} e
			    val {exp=restExp, funs=funs2} = transDecs nextvE rest
			in {exp=(pos, C.LETexp(Symbol.name new_v,
					       te, restExp)),
			    funs=funs1@funs2}
			end

		in (transDecs valEnv veList)
		end
	      (* When translating an application, we first check if the applied function is really
	       * an array, if it is then we check its arguments, if it has one then we treat it as
	       * a get operation on the array, if it has two then we treat it as a set operation on
	       * the array, more than two (or none) and we treat it as an error, note that in either
	       * case the first expression must be a list expression*)
	      | transExp (pos, A.APPexp(array_exp as (_,A.VALexp(A.VARval(v))), el)) = 
		(case Symbol.look(valEnv,v)
		  (* Here we check if the function applied is actually an array*)
		  of SOME(ArrayEntry(a)) => 
		     (case el
		       (* Once that is checked we check it is of the correct construction
			* to be an array operation and do the straightforward conversion*)
		       of [(_,A.LISTexp([e]))] =>
			  let val {exp=te, funs=funs1} = transExp e
			  in {exp=(pos, C.APPexp ((pos,C.VALexp(C.VARval("get_array_element"))),
						  [(pos,C.VALexp(C.VARval(Symbol.name a))), te])),
			      funs=funs1}
			  end      
			| [(_,A.LISTexp([e])), value_exp] =>
			  let val {exp=te1, funs=funs1} = transExp e
			      val {exp=te2, funs=funs2} = transExp value_exp
			  in {exp=(pos, C.APPexp ((pos, C.VALexp(C.VARval("set"))),
						  [(pos,C.VALexp(C.VARval(Symbol.name a))), te1, te2])),
			      funs=funs1@funs2}
			  end
			| _ => (Error("Array applied to invalid argument\n");
						       raise Semantic))
		   (* Here we see that although the function being applied is only a 
		    * variable and therefore could be an array it is not an array, so
		    * we just go ahead and translate it as normal*)
		   | SOME(_) => 
		     let val {exp=tr_arrayexp, funs=funs1} = transExp array_exp
			 val {expList=tel, funs=funs2} =
			     foldl (fn (e, {expList=tel, funs=funs}) =>
						  let val {exp=te, funs=funs3} = transExp e
						  in {expList=tel@[te], funs=funs@funs3}
						  end) {expList=[], funs=[]} el
				    
		     in {exp=(pos, C.APPexp(tr_arrayexp, tel)),
			 funs=funs1@funs2}
		     end
		   | NONE => (Error ("Unidentified identifier " ^ Symbol.name v);
			      raise Semantic))
			     

	      | transExp (pos, A.APPexp(e, el)) = 
		let val {exp=appExp, funs=funs1} = transExp e
		    val {expList=trel, funs=funs2} = foldl (fn (e, {expList=tel, funs=funs}) =>
							     let val {exp=te, funs=funs3} = transExp e
							     in {expList=tel@[te], funs=funs@funs3}
							     end) {expList=[], funs=[]} el
		in {exp=(pos, C.APPexp(appExp, trel)),
		    funs=funs1@funs2}
		end

	      | transExp (pos, A.LISTexp (el)) = 
		let val {expList=trel, funs=funs1} = foldl (fn (e, {expList=tel, funs=funs}) =>
							     let val {exp=te, funs=funs3} = transExp e
							     in {expList=tel@[te], funs=funs@funs3}
							     end) {expList=[], funs=[]} el
		in {exp=(pos, C.LISTexp(trel)),
		    funs=funs1}
		end

	      (* Galahad extensions *)
	      (* The array operations get and set are just mapped to equivalent functions*)
	      (* We don't need these anymore since we are now using the above, see also Absyn.sml*)
	      (*| transExp (pos, A.ARRAYSETexp(e1,e2,e3)) = (pos, C.APPexp((pos,C.VALexp(C.VARval("set"))),
									 (map transExp [e1,e2,e3])))
	      | transExp (pos, A.ARRAYGETexp(e1,e2)) = (pos, C.APPexp((pos,C.VALexp(C.VARval("get_array_element"))),
								      (map transExp [e1,e2])))*)
	      (*Galahad for loops*)
	      | transExp (pos, A.FORexp(v, lo, hi, e)) =
		(* These types of loops are static so, for each iteration
		 * we give a let expression, that first sets the loop variant
		 * to its new value, and then evaluates one iteration,
		 * oh and I'll admit right now the positions here are probably
		 * not very helpful if we just pipe this right through to the
		 * Camelot compiler rather than unparsing it*)
		let val {exp=loop_exp, funs=funs1} = translateExp {typeEnv=typeEnv,
								   valEnv=Symbol.enter(valEnv, v, ValEntry(v))} e
		    fun iterate (i) = 
			if (i < hi)
			then (pos, C.LETexp (Symbol.name v,
					     (pos,C.VALexp(C.INTval(i))),
					     (pos, C.LETexp(Symbol.name A.uscore_sym,
							    loop_exp,
							    iterate(i+1)))))
			else (*Assume i = hi*) (pos, C.LETexp(Symbol.name v,
							      (pos, C.VALexp(C.INTval(i))),
							      loop_exp))
			    
		in if lo > hi
		   then (Error "lo value in for loop lower than hi value";
			 raise Semantic)
		   else {exp=iterate(lo), funs=funs1}
		    
		end
	      | transExp (pos, A.ANONYFNexp(vlist, e)) =
		let val new_fname = Symbol.symbol "fn"
		    val newFunDef = A.FUNdef(false, new_fname, vlist, e)
		    val {new_fdlist=trans_fdList,
			 typeEnv=_,
			 valEnv=newValEnv} = transFunDef {valEnv=valEnv, typeEnv=typeEnv} [(pos, newFunDef)]
		    val munged_fname = case Symbol.look (newValEnv, new_fname)
					of SOME(ValEntry (v)) => v
					 | SOME(_) => raise Semantic
					 | NONE => raise Semantic
		in
		    (* We throw away the new value environment since we don't
		     * want the new function to be visible else where anyway*)
		    {exp=(pos, C.VALexp(C.VARval(Symbol.name munged_fname))),
		     funs=trans_fdList}
				  
		end
	      | transExp (pos, A.HOAREexp(e1,e2,e3)) =
		(* Assert statements are translated quite literally, we simply
		 * translate given e1 do e2 assert e3 end
		 * into
		 * if e1 
		 * then let newvar = e2 in if e3 then newvar else exit
		 * else exit
		 * TODO: We should also allow the compiler to accept a flag
		 * to turn assertions of, in which case this just equals
		 * te2.
		 *)
		let val {exp=te1, funs=funs1} = transExp e1
		    val {exp=te2, funs=funs2} = transExp e2
		    val {exp=te3, funs=funs3} = transExp e3
		    val temp = Symbol.name (newlabel "")
		in {exp=(pos, C.IFexp(te1, 
				      (pos, C.LETexp(temp,
						     te2,
						     (pos, C.IFexp(te3, 
								   (pos, C.VALexp(C.VARval(temp))),
								   (pos, C.APPexp ((pos, C.VALexp(C.VARval("exit"))), 
										   [])))))),
				       (pos, C.APPexp ((pos, C.VALexp(C.VARval("exit"))), [])))),
		    funs=funs1@funs2@funs3}
		end
		
		    (* Not yet sure how until (/while) loops are going to be
		     * implemented (comment closed at the end of this clause
	      | transExp (pos, A.UNTILexp (e1, e2, e3)) =
		(* until loops have turned out to be a little trickier than I
		 * thought, since in Camelot we cannot declare anonymous functions,
		 * transExp would need to return not only an expression but also
		 * a list of newly declared functions, this causes a little bother,
		 * but I think it will be necessary anyway (for one thing I would like
		 * to allow anonymous functions in Galahad)*)
		let val new_fun_name = newlabel ""
		    val newFunDef = let val arg1 = newlabel ""
					val arg2 = newlabel ""
					val test = (pos, A.BINexp(A.EQUALSop,
								  A.VALexp(A.VARval(arg1)),
								  e1))
				    in
					(pos, A.FunDef(true, newlabel "", 
						   [newlabel"", newlabel ""],
						   A.IFexp(test, e3, e2)))
				    end

		    val {new_fdlist=trans_fdList,
			 typeEnv=_,
			 valEnv=_} = transFunDef {valEnv=valEnv, typeEnv=typeEnv} [newFunDef]						   
		in
		    (* We throw away the new value environment since we do not
		     * want this function to visible as it is only used by us*)
		    {exp=(pos, C.APP(
		end*)


		
	    and transValue (A.VARval(v)) = 		
		(case Symbol.look (valEnv,
				   v)
		  of SOME(ValEntry(nv)) => (C.VARval(Symbol.name nv))
		   (* Arrays can be values too *)
		   | SOME(ArrayEntry(a)) => (C.VARval(Symbol.name a))
		   | SOME(_) => (Error ("In looking up " ^ Symbol.name v ^ " not a value name\n");
				 raise Semantic)
		   | NONE => (Error ("Failed in lookup of " ^ Symbol.name v ^ "\n");
			      raise Semantic))
	      | transValue (A.INTval(i)) = C.INTval(i)
	      | transValue (A.FLOATval(f)) = C.FLOATval(f)
	      | transValue (A.STRINGval(s)) = C.STRINGval (s)
	      | transValue (A.BOOLval(b)) = C.BOOLval(transBool b)
	    (* Modules can now be nested,
	     * so what we do is recurse until we are left with an lvalue of the
             * the form id.id, we then simply look up the left identifier to find
             * the module environment in which to look up the right identifier and
             * we are done. To recurse we look up the left identifier which should give
             * us an environment in which to translate the right hand side lvalue (which
	     * may be the base case or may have to be recursed further)*)
	      | transValue (lvalue as A.LVALval(_)) =
		let fun transLval (A.VARval(r), vE) =
			(case Symbol.look (vE, r)
			  of SOME (ValEntry(v)) => v
			   | SOME (_) => (Error ("looking up value " ^ Symbol.name r ^ "not a value\n");
					  raise Semantic)
			   | NONE => raise Semantic)

		      | transLval (A.LVALval(var, l), vE) =
			let val (_,mVE) = case Symbol.look(vE, var)
					   of SOME(ModEntry(v)) => v
					    | SOME(_) => (Error (Symbol.name var ^ " is not a module\n");
							  raise Semantic)
					    | NONE => (Error ("Cannot find module " ^ Symbol.name var);
						       raise Semantic)
			in
			    transLval(l, mVE) 
			end
		      | transLval (_,_) = raise Impossible
		    val v1 = transLval (lvalue, valEnv)
		in
		    C.VARval(Symbol.name v1)
		end
			
	    and transBool A.TRUEval = C.TRUEval
	      | transBool A.FALSEval = C.FALSEval
	    and transUnary A.NOTop = C.NOTop
	      | transUnary A.FTOIop = C.FTOIop
	      | transUnary A.ITOFop = C.ITOFop
	    and transBinary A.TIMESop = C.TIMESop
	      | transBinary A.DIVop = C.DIVop
	      | transBinary A.PLUSop = C.PLUSop
	      | transBinary A.MINUSop = C.MINUSop
	      | transBinary A.LESSop = C.LESSop
	      | transBinary A.GREATERop = C.GREATERop
	      | transBinary A.EQUALSop = C.EQUALSop
	      | transBinary A.CONSop = C.CONSop
	    and transMatchRuleL (pos, A.MATCHrule ((p,v), vl, e)) =
		let (* Here we do not need to munch the patterns since
		     * they are only in scope for the duration of the 
		     * associated exp, we assume that the first variable
		     * is a constructor and the rest are free variables
		     * to be used in the expression following the =>,
		     * Note that _ map appear in the pattern list but
		     * can be added to the temporary environment as itself
		     * with no adverse effects, so we don't need to take
		     * special care of it*)
		    val new_v = case Symbol.look (valEnv, v)
				 of SOME (ValEntry(n)) => n
				  | SOME (_) => (Error ((Symbol.name v) ^ "Not a constructor but used as one\n");
						 raise Semantic)
				  | NONE => (Error "Unrecognised constructor used in pattern\n";
					     raise Semantic)
		    val tmp_valEnv = foldl (fn ((p,x),y) => Symbol.enter (y,x,ValEntry(x))) valEnv vl
		    val {exp=new_exp, funs=funs1} = translateExp {typeEnv=typeEnv,
						valEnv=tmp_valEnv} e
		in {exp=(pos, C.MATCHrule((p, Symbol.name new_v),
					  map (fn (i,x) => (i,Symbol.name x)) vl,
					  new_exp)),
		    funs=funs1}
		end
	in transExp e
	end
	      

	val {defs=(types,vals,fdefs),
	     typeEnv=mainTypeEnv,
	     valEnv=mainValEnv, ...} =
	    (* Here we just throw away the type and val environments since we don't
	     * need them anymore, however to allow separate compilation of files, we
	     * could instead marhall (or flatten export whatever the terminallogy is)
	     * and write them out to a separate file, and then they can be used
	     * to add the mappings exported by this file. Separate compilation would require
	     * more, but that would at least be needed*)
	    foldl arrange {defs=([],[],[]),
			   typeEnv=base_typeEnv,
			   valEnv=base_valEnv} declarations
	val {exp=mainExp, funs=extra_funs} = translateExp {typeEnv=mainTypeEnv,
							   valEnv=mainValEnv} m
    in (* In the arrange function we use the @ operator to tag on the definitions
	* in the correct order, if we used foldr we could do use the :: operator
	* which would be faster. However that would mean that the last definitions
	* would be translated before the earlier ones, and the last definitions may
	* rely on the earlier ones so we can't use foldr. However it is probably faster
	* to use :: operator and then just reverse the lists here before returning*)
	C.PROG (types,vals,fdefs@[extra_funs], mainExp)
    end

end

(* LEGACY CODE
 * This is the old code for functor application, this was when the arguments
 * were type and val declarations as opposed to module expressions

	  (* For functor application so far we have gone for the simple simple approach, so we
	   * don't check the types of the arguments or even whether the correct list of arguments
	   * is this is all left up to the Camelot type checker. This does mean that a few incorrect
	   * programs will get through and compile (not just to Camelot but be accepted by the Camelot
	   * compiler). For example if the programmer gives all the correct arguments to the functor
	   * plus some extra ones (which don't overide the other function definitions in scope at the
	   * time of the functor definition) then it will compile, however the programmer must have not
	   * made use of the extra arguement so in general such programs will do what the programmer
	   * expected it might just produce some superflous functions (that aren't used)*)


	  (* For functor application so far we have gone for the simple simple approach, so we
	   * don't check the types of the arguments or even whether the correct list of arguments
	   * is this is all left up to the Camelot type checker. This does mean that a few incorrect
	   * programs will get through and compile (not just to Camelot but be accepted by the Camelot
	   * compiler). For example if the programmer gives all the correct arguments to the functor
	   * plus some extra ones (which don't overide the other function definitions in scope at the
	   * time of the functor definition) then it will compile, however the programmer must have not
	   * made use of the extra arguement so in general such programs will do what the programmer
	   * expected it might just produce some superflous functions (that aren't used)*)
	  | arrange (A.FUNCTORAPP (module_name, functor_name, fsignature, defines),{defs=(ts,vs,fs),
										    typeEnv=typeEnv,
										    valEnv=valEnv}) =
	    let
		val (functorTypeEnv,
		     functorValEnv,
		     A.FUNCTOR(ftname, typeargs, valargs, functor_defines)) = 
		    case Symbol.look(valEnv, functor_name)
		     of SOME(FunctEntry(a)) => a
		      | SOME(_) => (Error "Name not defined as a functor\n";
				    raise Semantic)
		      | NONE => (Error ("Error while defining " ^
					(Symbol.name module_name) ^
					". No such functor " ^
					(Symbol.name functor_name) ^ "\n");
				 raise Semantic)
		(* Make a quick check that at least the number of arguments is indeed correct*)
		val _ = if (List.length(typeargs) + List.length (valargs)) <> List.length (defines)
			then (Error "Wrong number of arguments to a functor\n";
			      raise Semantic)
			else ()
				     
		(* We need to use the CURRENT environments to translate the
		 * defines given as arguments*)
		val {defs=(mts,mvs,mfs),
		     typeEnv=module_body_typeEnv,
		     valEnv=module_body_valEnv} = foldl arrange {defs=([],[],[]),
								      typeEnv=typeEnv,
								      valEnv=valEnv} (defines)
		    

		    
		(* Then we use the argument names to look up the definitions in the
		 * returned environments. This gives us mappings which we add to the
		 * environments stored with the functor definition*)
		fun add_type_names typEnv [] = typEnv
		  | add_type_names typEnv (h::t) = (case Symbol.look (module_body_typeEnv, h)
						     of SOME(v) => add_type_names (Symbol.enter (typEnv, h, v)) t
						      | NONE => (Error "Functor arguments do not match those required\n";
								 raise Semantic))
						   
		fun add_val_names vEnv [] = vEnv
		  | add_val_names vEnv (h::t) = (case Symbol.look (module_body_valEnv, h)
						  of SOME(v) => add_val_names (Symbol.enter (vEnv, h, v)) t
						   | NONE => (Error "Functor arguments do not match those required\n";
							      raise Semantic))
						
		val type_arg_names = map (fn (_, A.TYPEmarker(v))=>v) typeargs
		val val_arg_names = map (fn (_,A.VALdec(v,_,_))=>v) valargs
		val Tenv = add_type_names functorTypeEnv type_arg_names
		val Venv = add_val_names functorValEnv val_arg_names
			   
		(* The Main bug is here, the fact is we now translate the module,
		 * but transModuleDef will add to the value environment a ModEntry
		 * containing that modules enviroments for use in expressions such
		 * as A.b, but it won't contain any of the constructors for the
		 * types given as arguments, since we have only added those to the
		 * current environment which we used to translate the arguments 
		 * (but which we will throw away)*)
		(* As a temporary fix, here is a function which will traverse the
		 * definitions given as parameters and add any type constructors*)
		(* TODO: I should add to this the functionality of the above functions
		 * add_[type/val]_name, that is check the arguments match the parameters
		 * to the functor and add their names to the environments that will be
		 * passed to transModuleDef*)
		fun add_constructors ((A.TYPEDEC(_,A.TYPEdec(_,_,l))), venv) = 
		    let fun aux_add_constructors a [] = a
			  | aux_add_constructors vE ((_,A.TYPEcon(h,_))::t) =
			    case Symbol.look(module_body_valEnv, h)
			     of SOME(v as ValEntry(_)) => aux_add_constructors (Symbol.enter(vE,h,v)) t
			      | _ => raise Impossible
		    in aux_add_constructors venv l
		    end
		(* If it's not a type declaration in the arguments it must be a valdec and
		 * we will have already added it*)
		  | add_constructors (_, venv) = venv

		val constructorVenv = foldl add_constructors Venv defines
		
  
		(* So we now translate the functor as if it were a module, but the environments
		 * used to translate the module is the environments at the time that the functor
		 * was defined, with the arguements added to it, thats why we just added the
		 * arguements to the environments stored with the functor definition in the
		 * val environment*)
		val {defs=(fts,fvs,ffs),
		     valEnv=fvenv} = transModuleDef {typeEnv=Tenv,
							  valEnv=constructorVenv} 
						    (module_name, fsignature, 
						     A.MODEXP (functor_defines))

		(* To create the final val environment we return we simply take the current
		 * one and add the newly defined module, which is found by looking in the
		 * val environment returned by transModuleDef*)
		val EndValEnv = (case Symbol.look (fvenv, module_name)
				  of SOME(m as ModEntry(_)) => Symbol.enter (valEnv, module_name, m)
				   | _ => raise Impossible)
				
	    (* We then translate the functor defines with the environments just worked
	     * out*)
	    (*val {defs=(fts,fvs, ffs),
		   typeEnv=ftenv,
		   valEnv=fvenv} = foldl arrange {defs=([],[],[]),
						  typeEnv=Tenv,
						  valEnv=Venv} functor_defines*)
				
	    (* Finally the type environment we return is just the same as the one we were given,
	     * and the val environment we return is the one we were given with the new module
	     * defined by the functor application added*)
	    (* BUG BUG: the environments stored in the ModEntry should have been altered due to the
	     * signature constraints on the functor application*)
	    (*val EndValEnv = Symbol.enter(valEnv, module_name,
					   ModEntry(ftenv,
						    fvenv))*)
					     
				
				
	    in
		(* What do we return? as mentioned above the environments are just the
		 * same as those given except the new module is added to the valEnv,
		 * (this is done by transModuleDef)
		 * the defines are those of the arguements and those of the functor*)
		{defs=((ts@mts@fts), (vs@mvs@fvs), (fs@mfs@ffs)),
		 typeEnv=typeEnv,
		 valEnv=EndValEnv}
	    end

****)