(* Copyright (C) 2002 Allan Clark
 * 	     
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *)
(* DOC_ENTRY
 * \start{text}
 * The symbol module implements a mapping from strings to integers.
 * As each string in the program is encountered it is searched for
 * in a hash table containing all strings that have thus far be encountered
 * in the program and given an associated integer. If it is found the new
 * string is simply given the existing integer, if not a new integer is created for
 * it. This means that all strings (apart from keywords which are just turned into tokens
 * by the lexical analyser) have an associated integer, and all equal strings have the same
 * associated integer. The string and associated integer is known as the symbol, checking if
 * two symbols are equal (for example when checking if a used variable has been declared) is
 * the simple and efficient task of checking the two symbols' integers.
 * The symbol module exports a table type, this is a use of the table functor, with the
 * polymorphic type key instantiated to Symbols (the getInt function merely returns the integer
 * associated with the symbol). 
 * \end{text}
 * \start {last_modified} Saturday 27 July \end {last_modified}
 * \YRTNE_COD
 *)


structure Symbol :> Symbol =
struct

  type symbol = string * int

  structure H = Polyhash

  exception Symbol
  val nextsym = ref 0
  val sizeHint = 128
  val hashtable : (string,int) H.hash_table = 
		H.mkTable(H.hash, op = ) (sizeHint,Symbol)
  
  fun symbol name =
      case H.peek hashtable name
       of SOME i => (name,i)
        | NONE => let val i = !nextsym
	           in nextsym := i+1;
		      H.insert hashtable (name,i);
		      (name,i)
		  end

  fun name(s,n) = s

  (*To check if two symbols are equal we just need to check their symbol number,
   *the string comparison was already done when the symbol was made and the string
   *inserted (or retrieved) from the hash table
   *)
  fun equal ((_,n1), (_,n2)) = n1 = n2


  structure T = Table.IntMapTable(struct type key = symbol
					 fun getInt(s,n) = n
				  end)
  type 'a table= 'a T.table
  val empty = T.empty
  val enter = T.enter
  val look = T.look
end
