structure Absyn =
struct

type pos = Location.Location

val uscore_sym = Symbol.symbol "_"

datatype Ty =
         INTty
       | CHARty
       | BOOLty
       | FLOATty
       | STRINGty
       | UNITty
       | ARRAYty of TyL
       | ARROWty of TyL * TyL
       | LISTty of TyL
       | TVARty of TVar
       (* Note that the value can only be a VARval or a LVALval*)
       | CONty of (TyL list) * Value
and Value =
    VARval of Var
  | INTval of int
  | FLOATval of real
  | STRINGval of string
  | BOOLval of Bool
  (* We need module openers eg Mod.local so we need
   * away of representing them, this allows lvalues
   * to be lvalue.Var*)
  | LVALval of Var * Value
and Bool =
    TRUEval
  | FALSEval
and Exp =
    VALexp of Value
  | UNARYexp of UnaryOperator * ExpL
  | BINexp of BinaryOperator * ExpL * ExpL
  | IFexp of ExpL * ExpL * ExpL
  | MATCHexp of ExpL * (MatchRuleL list)
  (* In Camelot let expressions may only have the one
   * var = exp but in Galahad we allow an arbitrary lengthed
   * list of such declarations*)
  | LETexp of (Var * ExpL) list * ExpL
  | APPexp of ExpL * (ExpL list)
  | LISTexp of (ExpL list)
(* We don't need these anymore as we just parse array operations
 * as a sequence of expressions (namely an identifier, followed by
 * a list expression), possibly followed by another expression
 * (for an array set operation)
 * we then work out that it must have been an array operation, if
 * the identifier is declared as having the array type*)
(*  | ARRAYSETexp of (ExpL * ExpL * ExpL)
  | ARRAYGETexp of (ExpL * ExpL)*)
  | FORexp of (Var * int * int * ExpL)
(*  | UNTILexp of (ExpL * ExpL * ExpL)*)
(* The same as a FUNdef, except we don't need a boolean
 * since it can't be recursive, and we don't need a name
 * since it is anonymous*)
  | ANONYFNexp of (Var list) * ExpL
  | HOAREexp of ExpL * ExpL * ExpL
and ValDec =
    VALdec of Var * TyL * bool
(* TypeMarkers are just the same for types that ValDecs are for vals,
 * it is so that they can be used in signatures to modules and also
 * arguments to functors*)
and TypeMarker =
    TYPEmarker of Var
and TypeDec =
    TYPEdec of (TVarL list) * Var * (TypeConL list)
and TypeCon =
    TYPEcon of Var * (TyL list)
and TypeAlias =
    TYPEalias of Var * Var
and FunDef =
(* Added a boolean to allow printing out the rec keyword (or not), however
 * at the moment the Camelot compiler does not seem to take notice of this
 * so in order to keep the CamelotAbsyn the same as the Camelot compiler
 * when translating this we just ignore the bool component and always print
 * out the rec keyword*)
    FUNdef of bool * Var * (Var list) * ExpL
and MatchRule =
    MATCHrule of VarL * VarL list * ExpL
and Program =
    PROG of (TypeDecL list) * (ValDecL list) * (FunDefL list list) * ExpL
and UnaryOperator =
    NOTop
  | FTOIop
  | ITOFop
and BinaryOperator =
    TIMESop
  | DIVop
  | PLUSop
  | MINUSop
  | LESSop
  | GREATERop
  | EQUALSop
  | CONSop
withtype Var = Symbol.symbol (*Var is string in CamelotAbsyn*)
and VarL = (pos * Symbol.symbol)
and TVar = Symbol.symbol
and TVarL = (pos * Symbol.symbol)
and TyL = (pos * Ty)
and ValDecL = (pos * ValDec)
and TypeMarkerL = (pos * TypeMarker)
and TypeConL = (pos * TypeCon)
and TypeDecL = (pos * TypeDec)
and TypeAliasL = (pos * TypeAlias)
and FunDefL = (pos * FunDef)
and MatchRuleL = (pos * MatchRule)
and ExpL = (pos * Exp)


(*Now we define a galahad program*)
datatype GProgram =
	 GPROG of define list * ExpL

and define = 
    TYPEDEC of (TypeDecL)
  | TYPEALIAS of (TypeAliasL)
  | VALDEC of (ValDecL)
  | FUNDEF of (FunDefL list)
  | MODULEDEF of Var * signatureExp list * moduleExp
  | FUNCTORDEF of functordef
  | FUNCTORAPP of Var * Var * signatureExp list * moduleExp list
  | SIGDEF of Var * signatureExp

(* Currently I don't think I'll allow nested modules (though I might)
 * this means that the define list that is part of a module cannot
 * contain a module definition, however this does not need to be
 * forced in abstract syntax but can be part of the parser*)
and moduleExp = MODEXP of define list | MODNAME of Var
and functordef = FUNCTOR of Var * (Var * signatureExp list) list * define list
and signatureExp = SIGEXP of TypeMarkerL list * ValDecL list | SIGNAME of Var


end

(************************************************************************************************
structure PrintAbsyn : sig val printGalahad : TextIO.outstream * Absyn.GProgram -> unit
			   val stdPrintGalahad : Absyn.GProgram -> unit
			   val printCamelot : TextIO.outstream * CamelotAbsyn.Program -> unit
			   val stdPrintCamelot : CamelotAbsyn.Program -> unit
		       end
=
struct
structure A = Absyn
	
fun printGalahad (f,_) = TextIO.output(f, "Sorry not implemented printGalahad yet\n")
fun stdPrintGalahad (a) = printGalahad(TextIO.stdOut, a)
fun printCamelot (f,_) = TextIO.output(f, "Sorry not implemented printCamelot yet\n")
fun stdPrintCamelot (a) = printCamelot(TextIO.stdOut, a)

end
**************************************************************************************************)