/*
 * Created on Apr 5, 2004
 *
 * To change the template for this generated file go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
package mrg.javaagent;

import java.lang.instrument.Instrumentation;
import mrg.security.MRGSecurityManager;
import mrg.proofchecker.*;
import mrg.utils.MRGUtils;
import java.lang.instrument.ClassFileTransformer;
import java.lang.instrument.IllegalClassFormatException;
import java.security.ProtectionDomain;
import org.apache.bcel.classfile.*;
import org.apache.bcel.generic.*;
import java.io.*;
import java.util.*;


/**
 * A Java agent to instrument method bodies before running them on the JVM.
 * <p>
 * The <code>-javaagent:</code><i>agentname</i><code>=</code><i>options</i> command line switch is used to specify a Java agent and its options.
 * The <code>premain</code> method is invoked before any classes are loaded into memory.
 * <p>
 * The <code>addTransformer</code> method of the <code>Instrumentation</code> interface enables us to register a <code>ClassFileTransformer</code> object which implements the <code>transform</code> method, to perform bytecode instrumentation at load-time
 * In this MRG PCC framework, we use this facility to maintain a counter of the number of object instantiations performed at runtime.
 * We can the use this to obtain empirical data supporting any claims of resource usage.
 * 
 * @author mprowse
 * 
 * @see java.lang.instrument.Instrumentation
 * @see mrg.javaagent.MRGTransformer
 */
public class MRGAgent {
	
    /** Never Called */
    public MRGAgent() {}
	
    /**
     * The number of objects instantiated during execution.
     * <br>
     * The bytecode of a method body is instrumented with code to increment this counter each time an object is instantiated on the heap.
     * It is read by the jvmtiagent in order to produce the execution report.
     */
    public static int num_objects = 0;


    /**
     * The method called when the Java agent is loaded, but before any classes are loaded into memory.
     * 
     * @param arg comma separated arguments passed to the agent from the command line.
     * @param i the interface through which to perform bytecode instrumentation.
     */
    public static void premain(String arg, Instrumentation i) {		
	if (arg != null) {
	    String[] args = arg.split(",");
						
	    for (int j=0; j<args.length; j++) {
		if (args[j].equals("debug")) {
		    MRGUtils.setDebug(true);
		    MRGUtils.debug("MRGAgent\t: Enabling debugging");
		}
	    }
	}

	i.addTransformer(new MRGTransformer());

	return;
    }

    /**
     * An implementation of the ClassFileTransformer interface to instrument bytecode at load-time.
     * <p>
     * In this framework, we wish to keep a record of how many objects are instantiated at runtime. The <a href='http://java.sun.com/j2se/1.5.0/docs/guide/jvmti/jvmti.html#EventSection'>JVMTI events</a> are insufficient for this purpose and so we instrument the bytecode directly.
     * For simplicity (although heavy on resources) we use the <a href='http://jakarta.apache.org/bcel/'>Byte Code Engineering Library</a> to perform the necessary changes to the class file.
     * 
     * @author mprowse
     */
    private static class MRGTransformer implements ClassFileTransformer {

	private static String main_method = "main";
	private static String main_signature = "([Ljava/lang/String;)V";

	//private static String start_method = "start";
	//private static String start_signature = "()V";
	//private static String start_signature = "(LInsSortM$dia_0;)V";
	
	/**
	 * This method can choose to return a modified classfile (or not).
	 * <p>
	 * It is called before a class is loaded into memory, providing the opportunity to perform any required bytecode instrumentation, thus altering the runtime behaviour of the class and/or its methods.<p>
	 * When a class (whose name has been specified to the MRGAgent as a command line argument) is loaded, a call to the <code>checkPermission()</code> method of the system security manager is prepended to the <code>main()</code> method of the class - BCEL is used here. Later, when this newly inserted code is executed, it will invoke the PCC verification, which will require access to the classfile. Since this may be contained within a jar file, the byte array itself (<code>classfileBuffer</code>) is passed by this method to the MRGProofChecker (using the static <code>setBuffer()</code> method).
	 * 
	 * @see java.lang.instrument.ClassFileTransformer#transform(java.lang.ClassLoader, java.lang.String, java.lang.Class, java.security.ProtectionDomain, byte[])
	 */

	public byte[] transform(
				ClassLoader loader,
				String className,
				Class<?> classBeingRedefined,
				ProtectionDomain protectionDomain,
				byte[] classfileBuffer)
	    throws IllegalClassFormatException {

	    String cname = System.getProperty("mrg.classname");
	    if (className.equals(cname)) {
		MRGUtils.debug("MRGTransformer\t: Loading "+cname+"...");
		ClassLoader cl = ClassLoader.getSystemClassLoader();
		
		// READ IN CERTIFICATE FILE
		String certname = cname+"Certificate";//System.getProperty("mrg.certificatename");
		MRGUtils.debug("MRGTransformer\t: Reading "+certname+"...");
		InputStream certis = cl.getResourceAsStream(certname+".thy");

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
		    while (certis.available() != 0) {
			baos.write(certis.read());
		    }
		} catch (IOException ioe) {
		    System.out.println("ERROR READING CERTIFICATE!!!");
		    System.exit(1);
		}

		byte[] certificateBuffer = baos.toByteArray();
		

		MRGUtils.debug("MRGTransformer\t: Setting \""+className+"\" and its files in MRGProofChecker");
		HashMap<String,MRGFile> hm = new HashMap<String,MRGFile>();
		hm.put("class", new MRGFile(className,classfileBuffer));
		hm.put("certificate", new MRGFile(certname,certificateBuffer));
		MRGProofChecker.set(className, hm);
	       
		MRGUtils.debug("MRGTransformer\t: Parsing \""+className+"\" with BCEL 5.1");
		InputStream is = new ByteArrayInputStream(classfileBuffer);
		ClassParser parser = new ClassParser(is,className);
		try {
		    JavaClass       klass   = parser.parse();
		    Method[]        methods = klass.getMethods();
		    ConstantPoolGen cp      = new ConstantPoolGen(klass.getConstantPool());

		    // Add elements to the ConstantPool
		    int cp_getSecurityManager     = cp.addMethodref("java/lang/System","getSecurityManager","()Ljava/lang/SecurityManager;");
		    int cp_checkPermission        = cp.addMethodref("java/lang/SecurityManager","checkPermission","(Ljava/security/Permission;)V");
		    int cp_SecurityException      = cp.addClass("java/lang/SecurityException");
		    int cp_Message                = cp.addString("Must have Security Manager");
		    int cp_SecurityException_init = cp.addMethodref("java/lang/SecurityException","<init>","(Ljava/lang/String;)V");
		    int cp_MRGPermission          = cp.addClass("mrg/security/MRGPermission");
		    int cp_MRGPermission_init     = cp.addMethodref("mrg/security/MRGPermission","<init>","(Ljava/lang/String;)V");
		    int cp_MRGPermission_init_byte= cp.addMethodref("mrg/security/MRGPermission","<init>","(Ljava/lang/String;[B)V");
		    int cp_ClassName              = cp.addString(className);

		    int cp_Array_NameAndType      = cp.addNameAndType("_MRG_classfileBuffer","[B");
		       
		    // Locate and modify "main" method
		    boolean found_main = false;

		    for(int i=0; i < methods.length; i++) {
			if (methods[i].getName().equals(main_method) && methods[i].getSignature().equals(main_signature)) {
			    // ADD SECURITY CHECK TO BEGINNING OR MAIN METHOD
			    MRGUtils.debug("MRGTransformer\t: Found \"main\" method");
			    found_main = true;

			    MethodGen mg = new MethodGen(methods[i], klass.getClassName(), cp);
			    int lv_SecurityManager = (mg.addLocalVariable("sm",new ObjectType("java/lang/SecurityManager"),null,null)).getIndex();

			    // Create and insert InstructionList corresponding to the security check
			    MRGUtils.debug("MRGTransformer\t: Inserting security check into \"main\" method");
			    InstructionList il = mg.getInstructionList(); // main method body
			    InstructionHandle il_start = il.getStart(); // start of main method body

			    // The check we insert is equivalent to:
			    //
			    //     SecurityManager sm = System.getSecurityManager();
			    //     if (sm == null)
			    //         throw new SecurityException("Must have Security Manager");
			    //     sm.checkPermission(new MRGPermission(className));
			    //
			    // We build the last line first so we have a target for the 'if-false' case
			    // We then prepend the two blocks, second one first

			    InstructionList il1 = new InstructionList();
			    InstructionList il2 = new InstructionList();
			    
			    il2.append(new ALOAD         (lv_SecurityManager));
			    il2.append(new NEW           (cp_MRGPermission));
			    il2.append(new DUP           ());
			    il2.append(new LDC           (cp_ClassName));
			    il2.append(new INVOKESPECIAL (cp_MRGPermission_init));
			    il2.append(new INVOKEVIRTUAL (cp_checkPermission));
			    
			    il1.append(new INVOKESTATIC   (cp_getSecurityManager));
			    il1.append(new ASTORE         (lv_SecurityManager));
			    il1.append(new ALOAD          (lv_SecurityManager));
			    il1.append(new IFNONNULL      (il2.getStart()));
			    il1.append(new NEW            (cp_SecurityException));
			    il1.append(new DUP            ());
			    il1.append(new LDC            (cp_Message));
			    il1.append(new INVOKESPECIAL  (cp_SecurityException_init));
			    il1.append(new ATHROW         ());

			    il.insert(il2); // add security check to start of main method body.
			    il.insert(il1);  // add security check to start of main method body.
				    
			    // Update this method and dispose of InstructionLists.
			    mg.setMaxStack();
			    methods[i] = mg.getMethod();
			    il.dispose();
			    il1.dispose();
			    il2.dispose();

			}
			/*
			else if (methods[i].getName().equals(start_method) && methods[i].getSignature().equals(start_signature)) {
			    // INSTRUMENT BYTECODE OF START METHOD ...?
			    MRGUtils.debug("MRGTransformer\t: Found \"start\" method");
			    found_start = true;

			    MethodGen mg = new MethodGen(methods[i], klass.getClassName(), cp);
			    InstructionList il = mg.getInstructionList();
			    methods[i] = mg.getMethod();
			    il.dispose();
			}
			*/
		    }

		    if (!found_main) {
			System.out.println("MRGTransformer\t: ERROR! Failed to find \"main\" methods!");
			return null;
		    }

		    MRGUtils.debug("MRGTransformer\t: modified \""+className+"\" ("+klass.getBytes().length+" bytes)");
		    klass.setConstantPool(cp.getFinalConstantPool());

		    return klass.getBytes();

		} catch(Exception e) { e.printStackTrace(); }
	    }
	    return null;
	}
    }

}
