module Main where

import Camelot_absyn
import Camelot_absyn_ASDL
import Camelot_annot
import Camelot_annot_UA (deannot_UA)
import Camelot_annot_html
import Camelot_infer_annot
import Camelot_infer_destr
import Camelot_infer_annot_UA (infer_annot_UA)
import Camelot_infer_destr_UA (infer_destr_UA)
import Misc

import GetOpt
import System
import IO
import List

data AnalysisKind = DEEL | UA

main =
    do
    (infile, outfile, analysis, optimize, maybe_HTMLfile) <- doArgs
    -- read input
    h_in <- openIn infile
    prg0 <- read_camelot h_in
    closeIn h_in infile
    -- do the inference
    prg2 <-
	case analysis
	     of DEEL ->
		    do
		    (aprg1, dataFM) <- return $ infer_annot $ min_annot prg0
		    -- optimize
		    let aprg2 = if optimize
				then infer_destr aprg1
				else aprg1
	            -- output HTML
		    case maybe_HTMLfile
			 of
			 Nothing -> return ()
			 Just prg_htmlname -> 
			     do
			     output_html dataFM (prg_htmlname ++ ".html") aprg1 prg_htmlname
			     if optimize 
				then output_html dataFM (prg_htmlname ++ "_OPT.html")
				     aprg2 ("optimized " ++ prg_htmlname)
				else return ()
		    ok <- report_status aprg1
		    if not ok
		       then exitWith $ ExitFailure 1
		       else return ()
		    return $ deannot $ complete_guars dataFM aprg2 -- some annotations turn into assertions
		UA ->
		    error "Usage aspects not implemeted yet"
    -- output the annotated and optimized program
    h_out <- openOut outfile
    write_camelot prg2 h_out
    closeOut h_out outfile


output_html dataFM htmlfile aprg prgname =
    do
    h_out <- openOut (OutputFile htmlfile)
    str <- camelot_annot2html dataFM aprg prgname
    hPutStrLn h_out str
    closeOut h_out (OutputFile htmlfile)

{------- arguments --------}

type ProgParams = (InputFile, -- input camelot pickle
		   OutputFile, -- annotated and optimised camelot pickle
		   AnalysisKind,
		   Bool, -- optimize or not?
		   Maybe FilePath) -- HTML output

usage :: IO ()
usage =
    do
    putStrLn $ usageInfo helpMessage options
    exitWith $ ExitFailure 1
	     
helpMessage = "\nUsage: check_sharing <options> [<input file name> [<output file name>]]"

options :: [OptDescr Flag]
options = [
	   Option ['h'] ["help"] (NoArg FlagHelp)
	     "Display usage information",
	   Option ['l'] ["layered"] (NoArg FlagDeel)
	     "Use layered sharing and destruction analysis (default)",
	   Option ['u'] ["usage-aspects"] (NoArg FlagUA)
	     "Use usage aspect analysis (NOT WORKING YET)",
	   Option ['o'] ["optimize"] (NoArg FlagOpt)
	     "Make matching destructive where proven safe",
	   Option ['x'] ["html"] (ReqArg FlagHTML "f.html")
	     "Write annotations to f.html (with -o also to f_OPT.html)." 
	  ]

data Flag = FlagHelp | FlagDeel | FlagUA | FlagOpt | FlagHTML FilePath
	  deriving Eq

doArgs :: IO ProgParams
doArgs = do
	 args <- getArgs
	 parseArgs args

parseArgs :: [String] -> IO ProgParams
parseArgs args =
    do
    (options, extra, errors) <- return (getOpt Permute options args)
    checkArgs options extra errors
    let optimize = elem FlagOpt options
	analysis = if elem FlagUA options then UA else DEEL
    (infile, outfile) <- extractFiles extra
    maybe_HTMLfile <- extractHTMLFile options
    return (infile, outfile, analysis, optimize, maybe_HTMLfile)

checkArgs options extra errors =
    if elem FlagHelp options then usage
    else case errors
	 of [] -> return ()
	    _ ->
	      do
	      putStr (concat errors)
	      usage

extractFiles :: [String] -> IO (InputFile, OutputFile)
extractFiles []             = return (Stdin, Stdout)
extractFiles [inNm]         = return (InputFile inNm, Stdout)
extractFiles [inNm,outNm]   = return (InputFile inNm, OutputFile outNm)
extractFiles _ =
    do { usage; error "" }

extractHTMLFile :: [Flag] -> IO (Maybe FilePath)
extractHTMLFile [] = return Nothing
extractHTMLFile ((FlagHTML fileNm):_) = return $ Just $ strip_html_ext fileNm
extractHTMLFile (_:flags) = extractHTMLFile flags

strip_html_ext s =
    if ".html" `isSuffixOf` s
       then fst $ splitAt (length s - 5) s
       else s

{------- files --------}

data InputFile = Stdin | InputFile String
data OutputFile = Stdout | OutputFile String

openIn :: InputFile -> IO Handle
openIn Stdin = return stdin
openIn (InputFile s) = openFile s ReadMode

closeIn :: Handle -> InputFile -> IO ()
closeIn _ Stdin = return ()
closeIn h _ = hClose h

openOut :: OutputFile -> IO Handle
openOut Stdout = return stdout
openOut (OutputFile s) = openFile s WriteMode

closeOut :: Handle -> OutputFile -> IO ()
closeOut _ Stdout = return ()
closeOut h _ = hClose h

