(*

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: syntax.ml,v $
   Id:       $Id: syntax.ml,v 1.10 2003/12/11 18:45:27 sjost Exp $ 

	
   What this File is all about:
   ----------------------------
   Anything concerned with abstract syntax as well as the abstract syntax itself.


   ToDos: 

    - All these "compile_xyz"-functions are pretty similar (as they just specialise List.fold_left) 
      but I feel it is sensible to have them instead of inlining List.fold 
      or writing "let compile = List.fold_left". Think again about this...

*)


open Support.Error


(* THE ABSTRACT SYNTAX ITSELF: *)

(* Identifiertypes *) 
type variable       = string 
type typvariable    = string 
type constructor    = string
type funcidentifier = string   
type typeidentifier = string   

type constructor_size = 
  | Unspecified
  | Specified of int (* Should be rational! *)

(* TYPES *) (* wuerde gerne first und higher order unterscheiden, macht aber Probleme *)
type plain_typ =
    UnitTyp 
  | DiamondTyp (* Just for compatibility with Camelot -> we treat it like unit, which does not cost anything *)
  | BoolTyp 
  | IntTyp 
  | FloatTyp
  | ArrayTyp of typ
  | CharTyp 
  | StringTyp 
  | TvarTyp of typvariable 
  | ConTyp of ((typ list) * typeidentifier)  (* argumente-list, typ-identifier. Bsp: int ilist, int bool tree, etc *)
	(* ConTyp sollte besser noch ein int enthalten = Anzahl Konstruktoren dieses Typs! *)
  | ArrowTyp of (typ * typ) 
(* Listen & Produkte muss man mit den Konstruktor-Typen selber bauen! *)
and typ = plain_typ Support.Error.withinfo


(* Eigentlich finde ich das nicht so gut, die Operation nur als Blatt anzuhangen
   anstatt ihn mit eigenen Expression-Childs einzuhngen... *)

(* Operators, unary *)
type plain_unaryoperator = (* There MUST NOT be an unary operator on constructor types! -> sharing in constraint.ml *)
    NotOp 
  | UMinusOp
  | UFminusOp
and unaryoperator = plain_unaryoperator Support.Error.withinfo

(* Operators, binary *)
type plain_binaryoperator = (* There MUST NOT be a binary operator on constructor types! -> sharing in constraint.ml *)
    TimesOp 
  | DivOp 
  | PlusOp
  | MinusOp
  | FtimesOp
  | FdivOp
  | FplusOp
  | FminusOp
  | LessOp 
  | LteqOp 
  | GreaterOp
  | GteqOp
  | EqualOp
  | ConsOp 
  | AppendOp
  | AndOp
  | OrOp
  | AndalsoOp       (* Sollte es gar nicht geben, da sie gleich als "if operand1 then operand2 else false" geparst werden knnten, wenn IF Ausdrcke im guard akzeptieren wrde. *)
  | OrelseOp        (* Sollte es gar nicht geben, da sie gleich als "if operand1 then true else operand2 " geparst werden knnten, wenn IF Ausdrcke im guard akzeptieren wrde. *)
  | ModOp           (* Recently added to Camelot *)
and binaryoperator = plain_binaryoperator Support.Error.withinfo


(* Values *)
type plain_value =
    VarVal of variable  (* Strictly, this should not be allowed to be a FunctionCall for 0-arity function, but... *)
  | IntVal of int 
  | FloatVal of float 
  | CharVal of char 
  | StringVal of string 
  | BoolVal of bool     (* in Camelot wird BOOL definiert, wozu? *)
  | UnitVal 
  | UnaryOpVal   of unaryoperator * value
  | BinaryOpVal  of binaryoperator * value * value
and value = plain_value Support.Error.withinfo

type diamond =
    New                 (* this tells us that it is a @_   *)
  | Reuse of variable   (* this holds the variable of @var *)
  | Void                (* in read-only matches, the diamond must be ignored *)
      
(* Expressions *)
type plain_expression =
    ValueExp   of value (* Include Variables, etc., but NOT Constructors! *) 
  | ConstrExp  of (constructor * (value list) * diamond) 
  | AppExp     of (funcidentifier * (value list))   
  | LetExp     of (variable * expression * expression) 
  | SeqExp     of (expression * expression)            (* Hintereinanderausfhrung, fr die Freunde imperativer Sprachen... *)  
  | IfExp      of (value * expression * expression) (* Variable could be replaced by value here *)
  | MatchExp   of (variable * ( matchrule list )) 
(*| MatchPrExp of (variable * ( matchrule list ))  DEPRECATED, each single matchrule can be destructive or not (due to compatibility with CAMELOT) *)
and expression 
      = plain_expression Support.Error.withinfo
and matchrule 
      = Matchrule of Support.Error.info * constructor * (variable list) * bool * expression * diamond
                                     (* Fileinfo, constructor, constructor_args, is_it_read_only? {true=read_only, false=destructive}, action, a possible diamond-variable *)
(* Funktionen *) 
type fundef = FunctionDef of  Support.Error.info * funcidentifier * (variable list) * expression
type funblockdef = fundef list


(* DECLARATIONS and PROGRAM follow after enriched types*)

(* TABLES: *) 

(* Function Tables *)
module FunTab= Map.Make(struct type t = funcidentifier let compare = String.compare end)
type funtab = fundef FunTab.t

(* User-Defined-Constructor-LookUp-Tables *)
module ConTab= Map.Make(struct type t = constructor let compare = String.compare end)
type coninfo = {order: int; size: constructor_size; arg_types: (typ list); own_typ: typ; typid: typeidentifier }
type contab= coninfo ConTab.t		   


(* Extensions to abstract syntax: enriched types *)
type cvar = string                                 (* Constraint Variables *)

let diamant = "<>" (* Einfach nur das Diamant-Symbol --- key *)

type rich_typ =
  | RUnitTyp 
  | RDiamantTyp (* For compatibilty with camelot *) 
  | RBoolTyp 
  | RIntTyp 
  | RFloatTyp
  | RCharTyp 
  | RStringTyp 
  | RTvarTyp of typvariable 
  | RConTyp of ((rich_typ list) * typeidentifier * rt_contab)  (* argumente-list, typ-identifier und resource variable. Bsp: int ilist[3], int bool tree[4,5], etc *)
  | RArrowTyp of rich_typ * rich_typ                 (* higher-order types kennen wir nicht: muss also rechtsgeklammert sein und somit annotationsfrei, da diese in der ValDec drin sind *)
  | RSelfTyp of typeidentifier  (* Wird bentigt um Endlossschleifen bei rekursiven user-typen zu vermeiden! *)
and rich_coninfo = {rcvar: cvar; rorder: int; rarg_types: rich_typ list; rsize: constructor_size} (* rorder = 0 denotes unknown order! - Wir brauchen mehr Information: JEDER Konstruktor muss seine eigene Constraint-Variable haben und seine angereicherten Argument-Typen (geordnet) kennen! *)
and rt_contab = rich_coninfo ConTab.t

let rSelfTypUnknown = RSelfTyp("Unknown Self")  (* Sometimes we must infer self later - use this abbreviation in these cases *)

let selfSymbol = "#" (* Just in case that we need to alter this symbol - alter in parser.mly as well. Current token: SHARP *)

type rich_valdec = RValDec of funcidentifier * cvar * rich_typ * cvar
type rvaltab = rich_valdec FunTab.t
      
(* DECLARATIONS AND PROGRAMS CONTINUED HERE: *)

(* Declarations *)
type valdec = 
    ValDec of Support.Error.info * funcidentifier * typ
  | AnnValDec of Support.Error.info * funcidentifier * (float option) * rich_typ * (float option)

type typcon = TypCon of Support.Error.info * constructor * constructor_size * (typ list)
type typdec = TypDec of Support.Error.info * (typvariable list) * typeidentifier * (typcon list)

(* Programme *)
type program = Program of Support.Error.info * (typdec list) * (valdec list) * (funblockdef list) 



(* Tools for the abstract syntax: *)
let rec curry: (typ list) -> typ = (* Fold typlist into higher-order type for recursion, not really currying, I know *)
  function
    | []    -> raise (Invalid_argument "curry")
    | [t]   -> t
    | h::ts -> Support.Error.addinfo (ArrowTyp(h,(curry ts)))

(*
let rec curry_rev: typ -> (typ list) -> typ = (* more like curry with last explicit , but 1st arg is reversed *)
  fun last l ->
    let rec cr_aux: typ -> (typ list) -> typ = 
      fun acc l ->
	match l with ->
	| []  ->
	| [t] -> Support.Error.addinfo (ArrowTyp(t,acc))
	| h::ts -> cr_aux (Support.Error.addinfo (ArrowTyp(h,acc))) ts
    in cr_aux last l
*)

let rec uncurry: typ -> (typ list) = (* Unfold ArrowTyp into a list of types, not really uncurrying, I know... *) 
  fun t ->
    match t.v with
    | ArrowTyp(r,d) -> r::(uncurry d)
    | _             -> [t]


let get_val_for: funcidentifier -> (valdec list) -> valdec =
  fun fid vds ->
    let p = 
      function 
	| ValDec(_,id,_)
	| AnnValDec(_,id,_,_,_) when (fid = id) -> true
	| _ -> false
    in List.find p vds 

let rec get_constrs: (typdec list) -> typeidentifier -> (typcon list) =
  function
    | [] -> raise (Invalid_argument "Typeidentifier not declared.")
    | TypDec(info,param,typid,cs) :: tds -> 
	( 
	  function
	    | id when id = typid -> cs
	    | id                 -> get_constrs tds id
        )

let rec select_constr: (typ list) -> (typcon list) -> typcon = (* given an unordered typ list, a constructor with fitting arguments is searched for *)
  fun typl constrl ->
    List.find
      (function 
	| TypCon(_,constr,_,arg_typs) 
	    when (Common.list_unordered_equal typl arg_typs)
	    -> true
	| _ -> false
      )
      constrl

let tcon_arg_compare: typcon -> typcon -> int = (* Compares constructors by number of arguments, the more arguments means greater *)
 fun tc1 tc2 ->
   let n1 = 
     let TypCon(_,_,_,arg1) = tc1 in
     List.length arg1
   in
   let n2 = 
     let TypCon(_,_,_,arg2) = tc2 in
     List.length arg2
   in 
   (n1 - n2)


let rec strip_rich_typ: rich_typ -> typ = (* Removes annotations *)
  function 
    | RUnitTyp       -> Support.Error.addinfo UnitTyp
    | RDiamantTyp    -> Support.Error.addinfo UnitTyp
    | RBoolTyp       -> Support.Error.addinfo BoolTyp
    | RIntTyp        -> Support.Error.addinfo IntTyp
    | RFloatTyp      -> Support.Error.addinfo FloatTyp
    | RCharTyp       -> Support.Error.addinfo CharTyp
    | RStringTyp     -> Support.Error.addinfo StringTyp
    | RTvarTyp(tvar) -> Support.Error.addinfo (TvarTyp(tvar))
    | RConTyp(pars,tid,rt_contab) -> Support.Error.addinfo (ConTyp((List.map strip_rich_typ pars),tid))
    | RArrowTyp(rta,rtb)          -> Support.Error.addinfo (ArrowTyp((strip_rich_typ rta), (strip_rich_typ rtb)))
    | RSelfTyp(tid)               -> bug ("Cannot strip RSelfTyp("^tid^"). Unfold first.")



(** LOOK-UP-TABLES (MAPS) for different purposes: **)


(* Tools for FunTabs *)
let compile_fun_aux: funtab -> (fundef list) -> funtab =
  List.fold_left 
    (fun tbl f -> 
      let FunctionDef(info,id,_,_) = f in 
      if FunTab.mem id tbl 
      then 
	errAt info ("Multiple definition of function-body for function '"^id^"'.")  
          (* This ERROR returns the previous duplicated definition. -using fold_right would give the last! *)
      else FunTab.add id f tbl
    )

let compile_fun: (fundef list) -> funtab = (compile_fun_aux FunTab.empty)


(* Tools for ConTabs: *)
let rec compile_contab: (typdec list) -> contab = 
  function 
    | [] -> ConTab.empty
    | TypDec(info,param,typid,cs) :: tds -> 
	      let paramtyp = List.map (fun x -> addinfo (TvarTyp(x))) param in
	      let owntyp   = {i=info; v=ConTyp(paramtyp, typid)} in
	      let ct = compile_contab tds (* Could be transformed into tail-recursion if necessary *)
	      in compile_contab_aux ct owntyp typid cs 0

and compile_contab_aux: contab -> typ -> typeidentifier -> (typcon list)-> int -> contab =
  fun ct otyp typid tcs ord -> match tcs with
  | [] -> ct
  | TypCon(info, constr, csize, args) :: tcs -> 
      let cinf = {order = ord; size = csize; arg_types = args; own_typ = otyp; typid = typid;} in
      let ct'  = if ConTab.mem constr ct 
                 then errAt info ("Duplicate constructor '"^constr^"'. Each constructor must be unique!")
	         else ConTab.add constr cinf ct 
      in compile_contab_aux ct' otyp typid tcs (ord+1)


let rec conTab_fold2: (constructor -> 'a -> 'b -> 'c -> 'c) -> 'a ConTab.t -> 'b ConTab.t -> 'c -> 'c =
  fun f tba tbb acc ->
    let appf = fun key reca acc -> f key reca (ConTab.find key tbb) acc in
    ConTab.fold appf tba acc



(* Sizes *)

(* WARNING: We cannot check the uniform option '-uni' here, this has to be done wherever one of the following three functions is called! *)

let theSIZE: typ -> int = (* This may be the SIZE function as in our paper for all user-unspecified types *)
  fun t -> (* (print_string "Call to theSIZE\n");   (*DEBUG*) *)
    match t.v with 
    | UnitTyp             -> 1
    | DiamondTyp          -> 0 (* What value shall this have? *)
    | BoolTyp             -> 1
    | IntTyp              -> 1
    | FloatTyp            -> 1
    | ArrayTyp(styp)      -> Support.Error.warning "Resource inference for array-types not possible. All arrays are treated as occupying no heap space!\n";
        0 (* ? *)
    | CharTyp             -> 1
    | StringTyp           -> Support.Error.warning "All strings are assumed to occupy ONE heap-cell only. See documentation.\n";
        1 (* ? *)
    | TvarTyp(tvar)       -> Support.Error.notImplemented "Resource inference for type-variables. See documentation.\n"
    | ConTyp(pars,id)     -> 1 
    | ArrowTyp(dot,rat)   -> Support.Error.warning "Sizes of function closures cannot be determined. See documentation.\n";
        0 (* ? *)  
	  
let wgt_size: coninfo -> int = (* Returns the sum of 'theSIZE' for each constructor argument *)
  fun ci -> (* (print_string "Call to wgt_size\n");   (*DEBUG*) *)
    List.fold_left (fun a x -> a + (theSIZE x)) 0 ci.arg_types
      (* i.e. = sum (List.map (fun x -> theSIZE x) ci.arg_types) *)
      
let usr_size: coninfo -> int = (* Returns user-specified size *)
  fun ci -> (* (print_string "Call to usr_size\n");   (*DEBUG*) *)
    match ci.size with
    | Specified(i) -> i
    | Unspecified  -> wgt_size ci
	  
let ifuni_size: bool -> coninfo -> int = (* Just to save some code redundancy. The first argument shall indicate if option '-uni' is set. *)
  function
(*    | true  -> (fun ci -> 1)                                  (* ALL are set to one, even those previously set to zero *) *)
    | true  -> (fun ci -> if ((usr_size ci) = 0) then 0 else 1) (* ONLY those above zero are set to one.                 *)
    | false -> usr_size
	  

(** EXTENSIONS to abstract syntaxt **)


(* Extension to abstract syntax: Operational Values and Locations: *)
type location = int 
let null = 0
let loccomp x y = x - y

type rvalue = 
  | IntRVal of int 
  | FloatRVal of float 
  | CharRVal of char 
  | StringRVal of string 
  | BoolRVal of bool 
  | UnitRVal 
  | PointerRVal of location

(* Further operations on locations, stack, heap, etc. are defined in exec.ml *)


(* BUILT-IN Functions (built-in functions) *)

let rec is_function_call: string -> funtab -> bool =
  fun id ftab -> (FunTab.mem id ftab) || (is_built_in_function id)
and is_built_in_function: string -> bool =
  function
    | "int_of_float"    
    | "float_of_int"    
    | "char_of_int"     
    | "int_of_char"     
    | "float_of_string" 
    | "string_of_float" 
    | "int_of_string"   
    | "string_of_int"   
	(* The conversion functions behave exactly as their OCaml counterparts. *)
    | "print_int"
    | "print_float"
    | "print_char"
    | "print_string"    
    | "print_int_newline"
    | "print_float_newline"
    | "print_char_newline"
    | "print_string_newline"    
    | "print_newline"   
	(* The print functions just do the expected. *)
    | "print"           
    | "print_debug"     
    | "print'"    
    | "print_debug'"    
	(* The polymorphic print functions try their best, but usually
	it is more sensible to program specific print-routines (e.g. for lists).
	The primed variants just return their arguments, but the unprimed
 	functions wont destroy their arguments either.
	The _debug variants print the heap-locations in front of values
	which are stored in the heap, seperated by a '*' symbol.
	*) 	
    | "print_heap_u"
    | "print_heap_v"
    | "print_heap_s"
	(* Prints current heap size during sandboxing, accoring to various counting methods:
	  u - uniformly; v - variably; s - user defined
	*)
    | "free"            -> true
	(* Does nothing, except for sandboxing: adds diamond to freelist *)
    | _                 -> false


let rt_of_built_in_function: string -> rich_typ = 
  fun s ->
(*    assert (is_built_in_function s); Bad Idea. Without we use a 'try' to catch the exception! *)
    match s with
    | "int_of_float"    -> RIntTyp
    | "float_of_int"    -> RFloatTyp
    | "char_of_int"     -> RCharTyp
    | "int_of_char"     -> RIntTyp
    | "float_of_string" -> RFloatTyp
    | "string_of_float" -> RStringTyp
    | "int_of_string"   -> RIntTyp
    | "string_of_int"   -> RStringTyp
    | "print_int"
    | "print_float"
    | "print_char"
    | "print_string"    
    | "print_int_newline"
    | "print_float_newline"
    | "print_char_newline"
    | "print_string_newline"    
    | "print"           
    | "print_debug"     -> RUnitTyp
    | "print'"    
    | "print_debug'"    -> RSelfTyp "built-in-function" (* Must be inferred from last argument! *)
    | "print_heap_u"
    | "print_heap_v"
    | "print_heap_s"
    | "print_newline"   
    | "free"            -> RUnitTyp
    | _                 -> raise (Invalid_argument "Not a built-in function")



(* SOME DEBUG-UTILITIES: *)

let print_expr: expression -> unit =
  fun expr ->
    let _ = printInfo expr.i in
    let msg = 
      match expr.v with
      | ValueExp _   -> "VAL"
      | ConstrExp _  -> "CONSTR"
      | AppExp _     -> "APP"
      | LetExp _     -> "LET"
      | SeqExp _     -> "SEQ"
      | IfExp _      -> "IF"
      | MatchExp _   -> "MATCH"
    in print_string msg

let rec print_typ: typ -> string =
  fun t ->
    match t.v with
    | UnitTyp     -> "()"
    | DiamondTyp  -> "<>"
    | BoolTyp     -> "bool"
    | IntTyp      -> "int"
    | FloatTyp    -> "float"
    | ArrayTyp(s) -> ("array["^(print_typ s)^"]")
    | CharTyp     -> "char"
    | StringTyp   -> "string"
    | TvarTyp(v)  -> ("'"^v) 
    | ConTyp(p,id) -> ((String.concat " " (List.map print_typ p))^" "^id)
    | ArrowTyp(d,r) -> ((print_typ d)^" -> "^(print_typ r))







