(* This code is from Benjamin Pierce, Indiana University *)

(* 
   Slightly modified by Steffen Jost <jost@informatik.uni-muenchen.de>. 
   
   ToDo for Steffen: 
   
   - Understand how the module format is used and decide whether we should
     use it throughout or not at all.

   - I think that the different bug,err,...-functions should be rewritten 
     in a more generalised/parameterised style, allowing to easily introduce 
     new functions like cerr with ONE definition only, reducing redundancies 
     in the code! 
*) 

open Format

module Error = struct

exception Exit of int

type info = FI of string * int * int | UNKNOWN
type 'a withinfo = {i: info; v: 'a}

let unknown = UNKNOWN
let progName = FI(Array.get Sys.argv 0, 0, 0)
let create f l c = FI(f, l, c)

let addinfo valu = {i=UNKNOWN; v=valu}
let stripinfo x  = x.v

let errf f =
  set_formatter_out_channel stderr;
  print_flush(); (* Strangely, this is not enough *)
  flush stdout; flush stderr; (* This helps - i should understand what's going on here! *)
  print_newline();
  open_vbox 0; 
  open_hvbox 0; f(); print_cut(); close_box(); print_newline();
  raise (Exit 1)

(* Special undocumented feature, so that error messages appearing in
   the documentation can be printed without hideous long automounter 
   pathnames *)
let fakeFileName = ref("")
let printErrorsAsIfInFile s = fakeFileName := s

let printInfo = function
  FI(f,l,c) ->
    print_string (if !fakeFileName = "" then f else !fakeFileName); 
    print_string ": line "; 
    print_int l; 
    if c > 0 then begin print_string ", column "; print_int c end;
    print_string ":"
| UNKNOWN ->
    print_string "<Unknown file and line>: "

let printInfoNF': info -> string =  (* Druckt nur Zeile und Spalte, als String *)
  function
    | FI(f,l,c) -> 
	let line   = string_of_int l in
	let column = string_of_int c in
	let out = (line^"."^column^":") in 
	let len = (String.length out) in
	if len < 9 
	then ((String.make (9 - len) ' ') ^ out)
	else out
    | UNKNOWN   ->
	("   ??.??:")


(* argument funktion unit -> unit zur Auswertung *)
let errfAt fi f = errf(fun()-> printInfo fi; print_space(); f())

(* argument nur string *)
let err s = errf (fun()-> print_string s; print_newline())

let errAt fi s = errfAt fi (fun()-> print_string s; print_newline())

let bug s = err ("Compiler bug: " ^ s)

let cerr s = err ("ERROR while constructing constraints: \n " ^ s)

let notImplemented s = err ("Not implemented: " ^ s)

let bugf f = errf (fun()->
  print_string "Compiler bug: "; print_space(); f()
)

let bugfAt fi f = errfAt fi (fun()->
  print_string "Compiler bug: "; print_space(); f()
)

let bugAt fi s = errAt fi ("Compiler bug: " ^ s)

let cerrAt fi s = errAt fi ("ERROR while constructing constraints: " ^ s)

(* WARNINGS *)

  let last_warning = ref ""          (* Remebers last warning message *)
  let warn_qflag = ref false (* Shall we queue warning messages? *)
  let warn_queue = ref []         (* The warning message queue *)

  let forcewarningAt fi s =
    if (s <> !last_warning) 
    then
      begin
	last_warning := s;
	(if fi <> UNKNOWN then printInfo fi); 
	print_string "\n WARNING: ";
	print_string s; 
	print_newline()
      end
	
  let warningAt = 
    fun fi s -> 
      if !warn_qflag
      then 
	begin
	  warn_queue := ((fi,s)::(!warn_queue))
	end
      else forcewarningAt fi s

  let warning = warningAt UNKNOWN
      
  let flush_warnings = fun () ->
    let rec flush_warnings_aux =
      function
	| [] -> ()
	| (fi,s)::t ->
	    begin
	      forcewarningAt fi s;
	      flush_warnings_aux t
	    end
    in
    begin
      (if !warn_queue <> [] then print_string "\nSOME WARNINGS HAVE BEEN DELAYED:\n");
      flush_warnings_aux (List.rev !warn_queue);
      warn_queue := []
    end
	
  let queue_warnings = fun () -> warn_qflag := true
  let print_warnings = fun () -> begin (warn_qflag := false); flush_warnings () end


end (* of module *)
