# 4 "lexer.mll"
 
 open Parser
 open Support
 open Error


type buildfun = Error.info -> Parser.token
let (symbolTable : (string,buildfun) Hashtbl.t) = Hashtbl.create 42

let keywords = (* All keywords are lower-case only *)
     [	("unit",	fun fi -> Parser.UNIT fi);	
	("diamond",	fun fi -> Parser.DIAMOND fi);	
	("bool",	fun fi -> Parser.BOOL fi);
	("int",		fun fi -> Parser.INTEGER fi);
	("float",	fun fi -> Parser.FLOAT fi);	(* new *)
(*	("list",        fun fi -> Parser.LIST fi);      (* NOT BUILT-IN in CAMELOT! *) *)
	("array",	fun fi -> Parser.ARRAY fi);	(* new *)
	("char",	fun fi -> Parser.CHAR fi);	(* new *)
	("string",	fun fi -> Parser.STRING fi);	(* new *)

	("type",	fun fi -> Parser.TYPE fi);	(* new *)
	("val",       	fun fi -> Parser.VAL fi);	(* new *)
	("richval",    	fun fi -> Parser.ANNVAL fi);	(* new *)
	("of",       	fun fi -> Parser.OF fi);	(* new *)

	("true",	fun fi -> Parser.TRUEVAL fi);	
	("false",	fun fi -> Parser.FALSEVAL fi);

	("not",		fun fi -> Parser.NOT fi);	(* new *)
	("and",		fun fi -> Parser.LAND fi);	(* Not in CAMELOT. Also as Symbol "&" defined *)
	("or",		fun fi -> Parser.LOR fi);	(* new *)
	("andalso",	fun fi -> Parser.ANDALSO fi);	(* new *)
	("orelse",	fun fi -> Parser.ORELSE fi);	(* new *)
        ("mod",	        fun fi -> Parser.MOD fi);	(* Newly introduced to CAMELOT *)

	("begin",	fun fi -> Parser.BEGIN fi);     (* kwxm: new Camelot syntax *)
        ("end",		fun fi -> Parser.END fi);       (* kwxm: new Camelot syntax *)
	("if",		fun fi -> Parser.IF fi);
	("then", 	fun fi -> Parser.THEN fi);
	("else",     	fun fi -> Parser.ELSE fi);
	("match",    	fun fi -> Parser.MATCH fi);
	("match\'",    	fun fi -> Parser.MATCHPR fi);
      	("with",   	fun fi -> Parser.WITH fi);
       	("let",		fun fi -> Parser.LET fi);
       	("rec",		fun fi -> Parser.REC fi);	(* new *)
	("in",		fun fi -> Parser.IN fi);
	("where",       fun fi -> Parser.WHERE fi);     (* new *)
       	("fun",		fun fi -> Parser.FUN fi);	(* newer *)
]

(* These are all KEYWORDS in Camelot, but I think this is unnatural. We parse them like any other identifier
   and assume they are automatically included in the signature...	
	("int_of_float",	fun fi -> Parser.INT_OF_FLOAT fi);	
	("float_of_int",	fun fi -> Parser.FLOAT_OF_INT fi);	
	("char_of_int",		fun fi -> Parser.CHAR_OF_INT fi);	
	("int_of_char",		fun fi -> Parser.INT_OF_CHAR fi);	
	("float_of_string",	fun fi -> Parser.FLOAT_OF_STRING fi);	
	("string_of_float",	fun fi -> Parser.STRING_OF_FLOAT fi);	
	("int_of_string",	fun fi -> Parser.INT_OF_STRING fi);	
	("string_of_int",	fun fi -> Parser.STRING_OF_INT fi);
	("print_int",		fun fi -> Parser.PRINT_INT fi);		
	("print_float",		fun fi -> Parser.PRINT_FLOAT fi);	
	("print_char",		fun fi -> Parser.PRINT_CHAR fi);	
	("print_string",	fun fi -> Parser.PRINT_STRING fi);	
	("print_newline",	fun fi -> Parser.PRINT_NEWLINE fi);	
	("array_head",		fun fi -> Parser.ARRAY_HEAD fi);	
*)


let _ = 
   List.iter (fun (kwd,(tokfun:buildfun)) -> Hashtbl.add symbolTable kwd tokfun) keywords

let keyword_or_identifier fi str =
  try (Hashtbl.find symbolTable (String.lowercase str)) fi  
	with Not_found -> IDENTIFIER {i=fi; v=str}

let keyword_or_constructor fi str =
  try (Hashtbl.find symbolTable (String.lowercase str)) fi  
	with Not_found -> CONSTRUCTOR {i=fi; v=str} 

let lineno = ref 1
and depth = ref 0
and start = ref 0 
and filename = ref ""
and startLex = ref Error.unknown

let setFilename s = filename := s

let newline lexbuf = incr lineno; start := (Lexing.lexeme_start lexbuf)

let info lexbuf =
  Error.create (!filename) (!lineno) ((Lexing.lexeme_start lexbuf - !start)-1) (* -1 Fr emacs-kompatibilitt: der zhlt von 0 an *)

let text = Lexing.lexeme
	
let extractLineno yytext offset =
  int_of_string (String.sub yytext offset (String.length yytext - offset))

# 101 "lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\255\255\202\255\005\000\118\000\118\000\245\255\244\255\
    \070\000\031\000\034\000\124\000\228\255\227\255\226\255\225\255\
    \224\255\223\255\222\255\221\255\091\000\010\000\216\255\089\000\
    \091\000\097\000\093\000\192\000\096\001\199\001\046\002\149\002\
    \252\002\099\003\202\003\122\000\155\000\015\001\025\001\035\001\
    \056\001\102\000\049\004\152\004\255\004\247\255\248\255\208\255\
    \251\255\209\255\233\255\210\255\211\255\218\255\219\255\175\001\
    \231\255\148\000\230\255\080\005\101\000\250\255\232\255\234\255\
    \236\255\235\255\188\000\246\255\187\000\034\000\071\000\077\000\
    \046\000\254\255\047\000\190\000\252\255\107\000\109\000\253\255\
    \219\000";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\002\000\003\000\053\000\255\255\255\255\
    \017\000\018\000\016\000\026\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\035\000\038\000\255\255\040\000\
    \041\000\042\000\043\000\048\000\049\000\050\000\049\000\012\000\
    \013\000\014\000\015\000\051\000\255\255\255\255\255\255\052\000\
    \052\000\255\255\048\000\048\000\008\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\006\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\004\000\004\000\255\255\
    \255\255";
  Lexing.lex_default = 
   "\002\000\000\000\000\000\255\255\255\255\066\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\041\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\255\255\000\000\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\066\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\048\000\000\000\255\255\255\255\000\000\
    \073\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\004\000\000\000\003\000\000\000\003\000\000\000\
    \000\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\007\000\005\000\032\000\031\000\003\000\020\000\027\000\
    \011\000\012\000\025\000\023\000\017\000\024\000\030\000\026\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\018\000\019\000\008\000\010\000\009\000\030\000\
    \006\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\015\000\063\000\016\000\022\000\030\000\
    \062\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\013\000\021\000\014\000\030\000\068\000\
    \255\255\054\000\068\000\065\000\064\000\057\000\053\000\052\000\
    \057\000\051\000\048\000\047\000\046\000\045\000\061\000\049\000\
    \074\000\073\000\073\000\073\000\001\000\071\000\068\000\073\000\
    \067\000\050\000\000\000\000\000\057\000\057\000\000\000\000\000\
    \057\000\000\000\000\000\033\000\030\000\056\000\055\000\034\000\
    \036\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\057\000\072\000\000\000\070\000\
    \000\000\000\000\000\000\000\000\000\000\058\000\000\000\037\000\
    \000\000\000\000\000\000\000\000\068\000\030\000\255\255\068\000\
    \079\000\000\000\255\255\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\000\000\000\000\069\000\
    \000\000\000\000\000\000\068\000\000\000\000\000\067\000\037\000\
    \000\000\000\000\000\000\042\000\042\000\001\000\078\000\043\000\
    \077\000\000\000\000\000\000\000\000\000\000\000\042\000\000\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\000\000\000\000\070\000\000\000\000\000\042\000\
    \001\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\000\000\069\000\000\000\000\000\042\000\
    \000\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\000\000\038\000\000\000\042\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\042\000\042\000\000\000\000\000\042\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\037\000\000\000\000\000\
    \000\000\000\000\000\000\028\000\028\000\042\000\000\000\028\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\028\000\000\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\000\000\000\000\000\000\037\000\000\000\028\000\
    \000\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\000\000\255\255\000\000\076\000\028\000\
    \255\255\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\255\255\000\000\000\000\028\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\000\000\029\000\029\000\000\000\000\000\029\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\029\000\000\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\000\000\000\000\028\000\028\000\000\000\029\000\028\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\000\000\000\000\000\000\028\000\029\000\000\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\000\000\000\000\000\000\029\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\030\000\030\000\000\000\000\000\030\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\029\000\029\000\000\000\030\000\029\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\000\000\000\000\029\000\030\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\000\000\000\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \030\000\030\000\000\000\000\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\030\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\030\000\030\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\030\000\
    \030\000\000\000\000\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\030\000\030\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\030\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\030\000\030\000\
    \000\000\000\000\030\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\030\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \000\000\030\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \000\000\030\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\030\000\030\000\000\000\
    \000\000\030\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \030\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\030\000\030\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \030\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\043\000\043\000\000\000\000\000\
    \044\000\000\000\000\000\000\000\000\000\000\000\000\000\043\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\030\000\030\000\000\000\
    \043\000\030\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\000\000\000\000\000\000\030\000\
    \043\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\000\000\000\000\000\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\043\000\043\000\000\000\000\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\043\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\000\000\000\000\043\000\043\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\000\000\043\000\043\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\000\000\043\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\043\000\000\000\000\000\043\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\043\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\000\000\000\000\043\000\043\000\000\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\000\000\000\000\000\000\043\000\043\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\060\000\000\000\000\000\043\000\000\000\000\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\043\000\000\000\000\000\043\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\043\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\000\000\255\255\003\000\255\255\
    \255\255\003\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\009\000\000\000\000\000\000\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \005\000\020\000\004\000\008\000\008\000\011\000\021\000\023\000\
    \011\000\024\000\025\000\026\000\026\000\041\000\060\000\025\000\
    \069\000\071\000\072\000\074\000\077\000\070\000\004\000\078\000\
    \005\000\024\000\255\255\255\255\011\000\057\000\255\255\255\255\
    \057\000\255\255\255\255\000\000\000\000\011\000\011\000\000\000\
    \035\000\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\057\000\070\000\255\255\004\000\
    \255\255\255\255\255\255\255\255\255\255\057\000\255\255\035\000\
    \255\255\255\255\255\255\255\255\068\000\000\000\066\000\068\000\
    \075\000\255\255\027\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\255\255\255\255\004\000\
    \255\255\255\255\255\255\068\000\255\255\255\255\066\000\035\000\
    \255\255\255\255\255\255\027\000\027\000\080\000\075\000\027\000\
    \075\000\255\255\255\255\255\255\255\255\255\255\027\000\255\255\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\255\255\255\255\068\000\255\255\255\255\027\000\
    \000\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\255\255\068\000\255\255\255\255\027\000\
    \255\255\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\255\255\037\000\255\255\027\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\027\000\027\000\255\255\255\255\027\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\255\255\255\255\255\255\255\255\005\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\040\000\255\255\255\255\
    \255\255\255\255\255\255\028\000\028\000\027\000\255\255\028\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\028\000\255\255\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\255\255\255\255\255\255\040\000\255\255\028\000\
    \255\255\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\255\255\066\000\255\255\075\000\028\000\
    \027\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\080\000\255\255\255\255\028\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\255\255\029\000\029\000\255\255\255\255\029\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\029\000\255\255\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\255\255\255\255\028\000\028\000\255\255\029\000\028\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\255\255\255\255\255\255\028\000\029\000\255\255\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\255\255\255\255\255\255\029\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\030\000\030\000\255\255\255\255\030\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\030\000\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \255\255\255\255\029\000\029\000\255\255\030\000\029\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\029\000\030\000\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\030\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\031\000\255\255\255\255\031\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \255\255\030\000\030\000\255\255\031\000\030\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\255\255\255\255\030\000\031\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\255\255\255\255\031\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\032\000\
    \032\000\255\255\255\255\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \031\000\031\000\255\255\032\000\031\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \255\255\255\255\031\000\032\000\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \255\255\255\255\032\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\033\000\033\000\
    \255\255\255\255\033\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\033\000\255\255\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\255\255\255\255\032\000\
    \032\000\255\255\033\000\032\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\255\255\
    \255\255\032\000\033\000\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\255\255\
    \255\255\033\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\034\000\034\000\255\255\
    \255\255\034\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\255\255\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\255\255\255\255\033\000\033\000\
    \255\255\034\000\033\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\255\255\255\255\255\255\
    \033\000\034\000\255\255\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\255\255\255\255\255\255\
    \034\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\042\000\042\000\255\255\255\255\
    \042\000\255\255\255\255\255\255\255\255\255\255\255\255\042\000\
    \255\255\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\255\255\255\255\034\000\034\000\255\255\
    \042\000\034\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\034\000\
    \042\000\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\042\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\043\000\043\000\255\255\255\255\043\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\043\000\255\255\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\255\255\255\255\042\000\042\000\255\255\043\000\
    \042\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\042\000\043\000\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\043\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\044\000\255\255\255\255\044\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\044\000\255\255\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\255\255\255\255\043\000\043\000\255\255\044\000\043\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\255\255\255\255\255\255\043\000\044\000\255\255\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\059\000\255\255\255\255\044\000\255\255\255\255\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\044\000\255\255\255\255\044\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255";
  Lexing.lex_base_code = 
   "";
  Lexing.lex_backtrk_code = 
   "";
  Lexing.lex_default_code = 
   "";
  Lexing.lex_trans_code = 
   "";
  Lexing.lex_check_code = 
   "";
  Lexing.lex_code = 
   "";
}

let rec token lexbuf =
    __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 121 "lexer.mll"
         ( EOF (info lexbuf) )
# 568 "lexer.ml"

  | 1 ->
# 122 "lexer.mll"
                                           ( newline lexbuf; AND (info lexbuf))
# 573 "lexer.ml"

  | 2 ->
# 123 "lexer.mll"
                ( token lexbuf )
# 578 "lexer.ml"

  | 3 ->
# 124 "lexer.mll"
                                ( newline lexbuf; token lexbuf )
# 583 "lexer.ml"

  | 4 ->
# 125 "lexer.mll"
                ( Error.errAt (info lexbuf) "Unmatched end of comment" )
# 588 "lexer.ml"

  | 5 ->
# 127 "lexer.mll"
                                ( let inf = info lexbuf in
				  let fstr = text lexbuf in
				 (* let _ = print_string ("\n###"^fstr^"@") in *)
				  let len = String.length fstr in
                                  let str = (String.sub fstr 2 (len-4)) in
				 (* let _ = print_string ("@"^str^"@") in *)
				  let num = try int_of_string str 
				  with Failure s -> errAt inf "Diamond-Comment unrecognizeable."
				  in
				 (* let _ = print_string ("@"^(string_of_int num)^"###") in *)
				  DIASIZE {i=inf; v=num} )
# 603 "lexer.ml"

  | 6 ->
# 138 "lexer.mll"
                                    ( depth:=1; startLex := info lexbuf; 
				  comment lexbuf; token lexbuf )
# 609 "lexer.ml"

  | 7 ->
# 140 "lexer.mll"
                               ( commentline lexbuf; token lexbuf )
# 614 "lexer.ml"

  | 8 ->
# 141 "lexer.mll"
                            ( CHARVAL {i=info lexbuf; v=(String.get (text lexbuf) 1)} )
# 619 "lexer.ml"

  | 9 ->
# 142 "lexer.mll"
                                ( let fstr = (text lexbuf) in (* Is always escaped - unescaping is not really possible here! *)
                                  let len  = String.length fstr in
                                  let str  = (String.sub fstr 1 (len-2)) in
				  STRINGVAL {i=info lexbuf; v=str} )
# 627 "lexer.ml"

  | 10 ->
# 146 "lexer.mll"
                                ( AT (info lexbuf) )
# 632 "lexer.ml"

  | 11 ->
# 147 "lexer.mll"
                                ( BANG (info lexbuf) )
# 637 "lexer.ml"

  | 12 ->
# 148 "lexer.mll"
                                ( DOLLAR (info lexbuf) )
# 642 "lexer.ml"

  | 13 ->
# 149 "lexer.mll"
                                ( SHARP (info lexbuf) )
# 647 "lexer.ml"

  | 14 ->
# 150 "lexer.mll"
                                ( POUND (info lexbuf) )
# 652 "lexer.ml"

  | 15 ->
# 151 "lexer.mll"
                                ( PARAGRAPH (info lexbuf) )
# 657 "lexer.ml"

  | 16 ->
# 154 "lexer.mll"
         ( EQUALS (info lexbuf) )
# 662 "lexer.ml"

  | 17 ->
# 155 "lexer.mll"
         ( LESS (info lexbuf) )
# 667 "lexer.ml"

  | 18 ->
# 156 "lexer.mll"
         ( GREATER (info lexbuf) )
# 672 "lexer.ml"

  | 19 ->
# 157 "lexer.mll"
                                ( DIAMOND (info lexbuf) )
# 677 "lexer.ml"

  | 20 ->
# 158 "lexer.mll"
          ( LTEQ (info lexbuf) )
# 682 "lexer.ml"

  | 21 ->
# 159 "lexer.mll"
          ( GTEQ (info lexbuf))
# 687 "lexer.ml"

  | 22 ->
# 160 "lexer.mll"
          ( ARROW (info lexbuf))
# 692 "lexer.ml"

  | 23 ->
# 161 "lexer.mll"
          ( WARROW (info lexbuf))
# 697 "lexer.ml"

  | 24 ->
# 162 "lexer.mll"
                          ( UNITVAL    (info lexbuf))
# 702 "lexer.ml"

  | 25 ->
# 163 "lexer.mll"
                       ( EMPTYPAREN (info lexbuf))
# 707 "lexer.ml"

  | 26 ->
# 164 "lexer.mll"
         ( LPAREN (info lexbuf))
# 712 "lexer.ml"

  | 27 ->
# 165 "lexer.mll"
         ( RPAREN (info lexbuf))
# 717 "lexer.ml"

  | 28 ->
# 166 "lexer.mll"
         ( LBRACE (info lexbuf))
# 722 "lexer.ml"

  | 29 ->
# 167 "lexer.mll"
         ( RBRACE (info lexbuf))
# 727 "lexer.ml"

  | 30 ->
# 168 "lexer.mll"
         ( LBRAK (info lexbuf))
# 732 "lexer.ml"

  | 31 ->
# 169 "lexer.mll"
         ( RBRAK (info lexbuf))
# 737 "lexer.ml"

  | 32 ->
# 170 "lexer.mll"
         ( COMMA (info lexbuf))
# 742 "lexer.ml"

  | 33 ->
# 171 "lexer.mll"
         ( COLON (info lexbuf))
# 747 "lexer.ml"

  | 34 ->
# 172 "lexer.mll"
         ( SEMICOLON (info lexbuf))
# 752 "lexer.ml"

  | 35 ->
# 173 "lexer.mll"
         ( LAND (info lexbuf))
# 757 "lexer.ml"

  | 36 ->
# 174 "lexer.mll"
          ( LAND (info lexbuf))
# 762 "lexer.ml"

  | 37 ->
# 175 "lexer.mll"
          ( LOR  (info lexbuf))
# 767 "lexer.ml"

  | 38 ->
# 176 "lexer.mll"
         ( BAR (info lexbuf))
# 772 "lexer.ml"

  | 39 ->
# 177 "lexer.mll"
         ( CARET (info lexbuf))
# 777 "lexer.ml"

  | 40 ->
# 178 "lexer.mll"
         ( PLUS (info lexbuf))
# 782 "lexer.ml"

  | 41 ->
# 179 "lexer.mll"
         ( MINUS (info lexbuf))
# 787 "lexer.ml"

  | 42 ->
# 180 "lexer.mll"
         ( TIMES (info lexbuf))
# 792 "lexer.ml"

  | 43 ->
# 181 "lexer.mll"
         ( DIVIDE (info lexbuf))
# 797 "lexer.ml"

  | 44 ->
# 182 "lexer.mll"
          ( FPLUS (info lexbuf))
# 802 "lexer.ml"

  | 45 ->
# 183 "lexer.mll"
          ( FMINUS (info lexbuf))
# 807 "lexer.ml"

  | 46 ->
# 184 "lexer.mll"
          ( FTIMES (info lexbuf))
# 812 "lexer.ml"

  | 47 ->
# 185 "lexer.mll"
          ( FDIVIDE (info lexbuf))
# 817 "lexer.ml"

  | 48 ->
# 186 "lexer.mll"
                                ( TYPEVAR {i=info lexbuf; v= text lexbuf} )
# 822 "lexer.ml"

  | 49 ->
# 187 "lexer.mll"
                           ( keyword_or_identifier (info lexbuf) (text lexbuf) )
# 827 "lexer.ml"

  | 50 ->
# 188 "lexer.mll"
                           ( keyword_or_constructor (info lexbuf) (text lexbuf) )
# 832 "lexer.ml"

  | 51 ->
# 189 "lexer.mll"
          ( INTVAL {i=info lexbuf; v=int_of_string(text lexbuf)} )
# 837 "lexer.ml"

  | 52 ->
# 191 "lexer.mll"
                                ( FLOATVAL {i=info lexbuf; 
					    v=try 
					        float_of_string(text lexbuf)
                                              with 
				                Failure("float_of_string") -> (Error.errAt (info lexbuf) ("Unidentifiable float value:"^(text lexbuf)))
				} )
# 847 "lexer.ml"

  | 53 ->
# 197 "lexer.mll"
                                ( Error.errAt (info lexbuf) "Illegal character" )
# 852 "lexer.ml"

  | n -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf n

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 75
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 201 "lexer.mll"
              ( depth:=!depth-1;
                  if !depth>0 then comment lexbuf )
# 864 "lexer.ml"

  | 1 ->
# 203 "lexer.mll"
                ( depth:=!depth+1; comment lexbuf )
# 869 "lexer.ml"

  | 2 ->
# 204 "lexer.mll"
        ( newline lexbuf; comment lexbuf )
# 874 "lexer.ml"

  | 3 ->
# 205 "lexer.mll"
                ( errAt (info lexbuf) "Lexing Error: File ends with a runaway comment. Probably a '*)' missing." )
# 879 "lexer.ml"

  | 4 ->
# 206 "lexer.mll"
                ( comment lexbuf )
# 884 "lexer.ml"

  | n -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf n

and commentline lexbuf =
    __ocaml_lex_commentline_rec lexbuf 80
and __ocaml_lex_commentline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 209 "lexer.mll"
        ( newline lexbuf )
# 895 "lexer.ml"

  | 1 ->
# 210 "lexer.mll"
     ( commentline lexbuf)
# 900 "lexer.ml"

  | n -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_commentline_rec lexbuf n

;;

