(*

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: exec.ml,v $
   Id:       $Id: exec.ml,v 1.12 2004/02/20 12:57:40 sjost Exp $ 

	

   What this File is all about:
   ----------------------------
   Doing a 'sandboxed' execution of lfd-programs, that is,
   interpret the program and count all the heap-cell operations.

 
   ToDos: 

   - Parse the command-line options into the appropriate datatype! (Char List remain to be done only)

*)

open Support.Error
open Common
open Syntax
open Argument

(* Extension to abstract syntax: Operational Values -> see file syntax.ml*)

(* Stack *)

module Stack= Map.Make(struct type t = variable let compare = String.compare end)
type stack = rvalue Stack.t

let stack_bind: variable -> rvalue -> stack -> stack =
  fun v r s ->
    if v <> "_" 
    then
      let _ = 
	if Stack.mem v s 
	then warning ("Variable '"^v^"' was overwritten on stack.")
      in Stack.add v r s
    else s 
      
let compile_stack_aux: stack -> (variable list) -> (rvalue list) -> stack =
  List.fold_left2 (fun s v r -> stack_bind v r s) 
let compile_stack: (variable list) -> (rvalue list) -> stack =
  (compile_stack_aux Stack.empty)

  
(* Heap *)

module Heap = Map.Make(struct type t = location let compare = loccomp end)
type heapcell = constructor * (rvalue list)
type heap = heapcell Heap.t
type result = (heap * rvalue)
type heap_op =
    {
     mutable curr_loc: int;  
     mutable freelist: (location list);
   }

let the_heap_op: heap_op = (* GLOBAL: heap-operating information *)
  {
   curr_loc =  0;  (* Counter, which holds the address of the next artificial heap-cell to be 'allocated'. This counter shall never decrease. *)
   freelist =  []  (* Freelist, containing previousliy withdrawn cells *)
 }



let new_loc_proposed_at: (location option) -> location =
  fun ol ->
    (
     match (the_heap_op.freelist, !the_options.inplace, ol) with
     | (ls, true, Some(d)) -> d
     | ((l::ls), true, None) -> 
	 let _ = the_heap_op.freelist <- ls in (* remove l from freelist *)
	 l                                     (* return l *)
     | _ ->
	 let nl = 
	   try save_increment the_heap_op.curr_loc
	   with _ ->  bug  ("Not enough heapcell addresses available.")
	 in
	 let _ = the_heap_op.freelist <- [] in (* Either the freelist is empty anyway, or we dont use a freelist at all *)
	 let _ = the_heap_op.curr_loc <- nl in (* Store the increased reference counter *)
	 nl
    ) 
      

let new_loc: unit -> location =                (* Deprecated. Name-alias only. *)
  fun () -> new_loc_proposed_at None


(* Heap Statistics *)
type heap_stats =
  {
   mutable num_heap_store:   int;
   mutable num_heap_read:    int;
   mutable num_heap_remove:  int;
   mutable uniform_max:  int;  (* Uniform heap-cell counting *) (* Should be equal to !the_heap_op.curr_loc now...*)
(* mutable uniform_cur:  int;  = (heapsize h) -each map-key denotes one cell *)
   mutable uniform_ini:  int;  (* Initially occupied memory *)
   mutable weighted_cur: int; (* Counting sizes by number of constructor arguments/theSIZE *)
   mutable weighted_max: int;
   mutable weighted_ini: int;
   mutable usrsized_cur: int; (* Counting according to user (declare size via '(*#*)' in typedeclaration. Set to one if unspecified. *)
   mutable usrsized_max: int;
   mutable usrsized_ini: int;
 }

let the_heap_stats = (* GLOBAL: heap-statistics *)
  {
   num_heap_store   = 0;
   num_heap_read    = 0;
   num_heap_remove  = 0;
   uniform_max  = 0; 
(* uniform_cur  = 0;  = (heapsize h) -each map-key denotes one cell *)
   uniform_ini  = -1; 
   weighted_cur = 0; 
   weighted_max = 0;
   weighted_ini = -1;
   usrsized_cur = 0; 
   usrsized_max = 0;
   usrsized_ini = -1;
 }

(* Cheating again: this uses a mutable, global variable "the_contab", weil nachtrglich eingeflickt: *)
let the_contab = ref ConTab.empty

(* Heap Access --- You shall only use these functions for manipulating the heap. *)
(*  (Maybe it would be more sensible to use a globally referenced heap instead of passing it around - although against the paradigm of functional programming...) *)

let heap_size: heap -> int = fun h -> Heap.fold (fun k a n -> n+1) h 0 

let heap_store_at: heap -> heapcell -> (location option) -> (heap * location) = (* Storing in heap at given address or new address *)
  fun h c ol ->
    let l = new_loc_proposed_at ol in
    let h' = 
      if (Heap.mem l h) 
      then err "Reallocation of undisposed heap-cell detected. Check for duplicated '@' annotations!" 
      else Heap.add l c h 
    in
    (* Counting *)
    let (constr, cargs) = c in
    let cinf = try ConTab.find constr !the_contab with Not_found -> err "Undeclared constructor encountered." in
    let _ = 
      the_heap_stats.num_heap_store <-  the_heap_stats.num_heap_store + 1;
      the_heap_stats.uniform_max    <-  max (heap_size h') the_heap_stats.uniform_max;
      the_heap_stats.weighted_cur   <-  the_heap_stats.weighted_cur + (wgt_size cinf);
      the_heap_stats.weighted_max   <-  max the_heap_stats.weighted_cur the_heap_stats.weighted_max;
      the_heap_stats.usrsized_cur   <-  the_heap_stats.usrsized_cur + (ifuni_size (!the_options.uniform) cinf);
      the_heap_stats.usrsized_max   <-  max the_heap_stats.usrsized_cur the_heap_stats.usrsized_max
    in
    let _ = 
      if !the_options.debug then 
	( print_string ("\n [Allocating   address "^(print_aligned_int' l)
	  ^"   ---   Usr: "^(print_pretty_int' (ifuni_size (!the_options.uniform) cinf))
	  ^"   ---   '"^constr
	  ^"'] ")
	) 
    in
    (h', l)

let heap_store: heap -> heapcell -> (heap * location) = (* Storing at arbitrary address *)
  fun h c -> heap_store_at h c None
      
let heap_read:    heap -> location -> heapcell =        (* READ-ONLY access *)
  fun h l -> 
    let _ = the_heap_stats.num_heap_read <-  the_heap_stats.num_heap_read + 1 in
    Heap.find l h 


let heap_withdraw: heap -> location -> (heap * heapcell) =    (* DESTRUCTIVE READ access -> cell goes to freelist *)
  fun h l -> 
    let c  = Heap.find l h in (* So it does not count! *)
    let h' = Heap.remove l h in 
    (* Add to freelist *)
    let _ = the_heap_op.freelist <- (l::the_heap_op.freelist) in

    (* Counting: *)
    let (constr, cargs) = c in
    let cinf = try ConTab.find constr !the_contab with Not_found -> err "Undeclared Constructor encountered." in
    let _ = 
      the_heap_stats.num_heap_remove <-  the_heap_stats.num_heap_remove + 1;
      the_heap_stats.num_heap_read   <-  the_heap_stats.num_heap_read + 1;
      the_heap_stats.weighted_cur    <-  the_heap_stats.weighted_cur - (wgt_size cinf);
      the_heap_stats.usrsized_cur    <-  the_heap_stats.usrsized_cur - (ifuni_size (!the_options.uniform) cinf)
    in
    let _ = 
      if !the_options.debug then 
	( print_string ("\n [Deallocating address "^(print_aligned_int' l)
	  ^"   ---   Usr: "^(print_pretty_int' (-(ifuni_size (!the_options.uniform) cinf)))^"] ")
	) 
    in
    (h', c)

let heap_remove: heap -> location -> (heap) =              (* Just destructs a heapcell, no reading *)
    fun h l ->
      let _ = the_heap_stats.num_heap_read <-  the_heap_stats.num_heap_read - 1 in 
      fst (heap_withdraw h l)

let heap_pop_freelist: location -> unit =                  (* Removes cell from freelist. *)
  fun l -> 
    if (List.mem l the_heap_op.freelist)
    then
      let _ = the_heap_op.freelist <- (list_remove l the_heap_op.freelist) 
      in ()
    else 
      bug ("Location '"^(string_of_int l)^"' to be removed from freelist is not contained in freelist.")

let heap_push_freelist: location -> unit =                 (* Adds cell to freelist. Use with CARE! *)
  fun l ->
    if not (List.mem l the_heap_op.freelist)
    then
      let _ = the_heap_op.freelist <- (l::the_heap_op.freelist) 
      in ()
    else 
      bug ("Location '"^(string_of_int l)^"' to be added to freelist is already contained in freelist.")


(* Printing... *)

let string_of_loc: location -> string =
  fun l -> "*("^(string_of_int l)^") "

let print_loc: location -> unit =
  fun l -> print_string (string_of_loc l)

let rec print_rval: result -> unit =
  function 
    | (h, IntRVal(x)) -> print_int x
    | (h, FloatRVal(x)) -> print_float x
    | (h, CharRVal(x)) -> print_char x
    | (h, StringRVal(x)) -> 
	let out =
	  if !the_options.debug 
	  then ("\""^x^"\"") (* Explicit strings *)
	  else string_captured x
	in print_string out
    | (h, BoolRVal(true)) -> print_string "True"
    | (h, BoolRVal(false)) -> print_string "False"
    | (h, UnitRVal) -> print_string "()"
    | (h, PointerRVal(loc)) -> 
	let (cname, rvals) = 
	  try heap_read h loc with
	    Not_found -> err ("Dangling pointer encountered."^(string_of_loc loc))
	in
	let _ = print_string cname in
	let _ = print_char '('  in
	let _ = print_rvals h rvals in 
	let _ = print_char ')'  in () 

and print_rvals: heap -> rvalue list -> unit =
  function h ->
    function [] -> ()
      | x :: [] -> print_rval (h, x) 
      | x :: xs -> print_rval (h, x); print_string ","; print_rvals h xs
	    
let rec print_rval_debug: result -> unit =
  function 
      (h, PointerRVal(loc)) ->
	let (cname, rvals) = try heap_read h loc with Not_found -> err ("Dangling pointer encountered."^(string_of_loc loc)) in
	let _ = (* print_char '*'; *)  print_int loc; print_char '*' in
	let _ = print_string cname in
	let _ = print_char '(' in
	let _ = print_rvals_debug h rvals in 
	let _ = print_char ')' in () 
    | other_rval -> print_rval other_rval

and print_rvals_debug: heap -> rvalue list -> unit =
  function h ->
    function
	     [] -> ()
      | x :: [] -> print_rval_debug (h, x) 
      | x :: xs -> print_rval_debug (h, x); print_string ", "; print_rvals_debug h xs


(* Command-Line-Argument parsing *)
exception Parsing_failed of string (* Signals that a call to parse_rval_aux failed *)

let default_rvalue: (typdec list) -> heap -> typ -> result =
  fun typdecs h t ->
    match t.v with
    | UnitTyp   -> (h, UnitRVal)
    | BoolTyp   -> (h, BoolRVal(false))
    | IntTyp    -> (h, IntRVal(0) )
    | FloatTyp  -> (h, FloatRVal(0.0)) 
    | CharTyp   -> (h, CharRVal(' '))
    | StringTyp -> (h, StringRVal("")) 
    | ConTyp([],tid) -> 
	let constrs = get_constrs typdecs tid in    (* get all constructor for this type *)
	let TypCon(info, constr, csize, arg_typs) = (* select a Nil-Constructor *)
	  try
	    select_constr [] constrs 
	  with 
	    Not_found -> err ("Unable to generate default value for type '"^tid^"': Nil-like constructor is missing.")
	in 
	let (h',l) = heap_store h (constr, []) in
	(h', PointerRVal(l))
    | _ -> err ("Unable to generate default value for this type.")


let parse_rval: (typdec list) -> heap -> (typ list)-> (string list) -> (heap * (rvalue list)) =
  fun typdecs h_ t_ input_ ->
    if input_ = []                 
    then (* generate a default-values only *)
      let f: typ -> (heap * (rvalue list)) -> (heap * (rvalue list)) =
	fun t (h,acc) -> 
	  let (h', rv) = (default_rvalue typdecs h t) 
	  in (h', rv::acc)
      in List.fold_right f t_ (h_,[])
    else (* parse the commandline *)
    let _ = print_string (" Parsing command line execution arguments...\n") in
    let max_rec_depth = (List.length input_) + 3 in
    let rec parse_aux: int -> (bool * bool) -> heap -> (typ list) -> (string list) -> (heap * (rvalue list) * (string list)) =
      fun lc csort_ord h tl input -> (* lc is a loop-counter to ensure termination *)
	begin
	  let _ = (* DEBUG *)
	    if !the_options.debug 
	    then
	      begin
		print_string (String.make (2*lc) ' ');
		print_string "lc: ";
		print_int lc;
		print_string " - "; 
		print_string "[";
		print_string (String.concat "; " (List.map print_typ tl));
		print_string "]";
		print_string " - "; 
		print_string "['";
		print_string (String.concat "'; '" input);
		print_string "']";
		print_string "\n"
	      end
	  in
	  match tl with
	  | []    -> (h,[],input)
	  | t::ts -> 
	      let (h1, rvs1, input1) = (* Parsing the first type in list... *)
		begin
		  match t.v with
		  | DiamondTyp      (* treated as UnitTyp *)
		  | UnitTyp      -> (try debug_string "  Unit   parsed. "; (if (List.hd input = "()")||(List.hd input = "*") then (h, [UnitRVal], (List.tl input)) else raise (Parsing_failed "'()' expected")) with _ -> raise (Parsing_failed "Unit argument expected"))
		  | BoolTyp      -> (try debug_string "  Bool   parsed. "; (h, [  BoolRVal( ext_bool_of_string(List.hd input))], (List.tl input)) with _ -> raise (Parsing_failed "Bool expected"))
		  | IntTyp       -> (try debug_string "  Int    parsed. "; (h, [   IntRVal(  int_of_string(List.hd input))], (List.tl input)) with _ -> raise (Parsing_failed "Int expected"))
		  | FloatTyp     -> (try debug_string "  Float  parsed. ";  (h, [ FloatRVal(float_of_string(List.hd input))], (List.tl input)) with _ -> raise (Parsing_failed "Float expected"))
		  | CharTyp      -> (try debug_string "  Char   parsed. ";(if (String.length (List.hd input) = 1) then (h, [CharRVal(String.get (List.hd input) 0)], (List.tl input)) else raise (Parsing_failed "Char expected, argument is too long")) with _ -> raise (Parsing_failed "Char expected"))
		  | StringTyp    -> (try debug_string "  String parsed. "; (h, [StringRVal(List.hd input)],                  (List.tl input)) with _ -> raise (Parsing_failed "String expected"))
		  | ConTyp([],tid)  -> 
		      begin 
			let sorted_cs =   (* We distinguih the constructors in order to try some certain constructors first... *)
			  let constrs = get_constrs typdecs tid in   (* first get all constructors for this type *)
			  let (csort, cord) = csort_ord in           (* (true->sort constrs; true->keep_order/big_first *)
			  let p_nil     : typcon -> bool = function TypCon(_,_,_,arg_types) -> ((List.length arg_types) = 0) in
			  let (nil_cs, nonnil_cs) = List.partition p_nil constrs in
			  match (input, csort, cord) with (* THIS SORTS THE CONSTRUCTORS *)
			  | ([],  _  , true ) -> nil_cs            (* When no more input is available, we directly try nil constructors! *)
			  | ([],  _  , false) -> List.rev nil_cs   (* When no more input is available, we directly try nil constructors! *)
			  | ( _, true, true)  -> List.rev_append (List.stable_sort tcon_arg_compare nonnil_cs)           nil_cs   (* Sorting is in increasing order, but we want decreasing, hence reverse! *)
			  | ( _, true, false) ->     List.append (List.stable_sort tcon_arg_compare nonnil_cs) (List.rev nil_cs)
			  | ( _, false, true) ->          constrs (*     List.append                                    nonnil_cs            nil_cs   *)
			  | ( _, false, false)-> List.rev constrs (* List.rev_append                                    nonnil_cs  (List.rev nil_cs)  *)
			in
			let rec loop_constrs: (typcon list) -> (heap * (rvalue list) * (string list)) =  (* try constructors satifying p subsequently *)
			  fun tcons ->
			    match tcons with
			    | []        -> raise (Parsing_failed "No suitable constructor found")
			    | TypCon(info,constr,csize,arg_types)::rest ->
				begin 
				  try
				    begin
				      let _ = debug_string ((String.make (2*lc) ' ')^"Trying constructor "^constr^"\n") in
				      let (h1, rvs1, input1) = (* Constructor-arguments *)
					if (List.length arg_types = 0)
					then (* Nil-constructor, nothing to do *)
					  (h, [], input) 
					else (* Non-nil-constructor *)
					  (* We must try the different search depths manually! *)
					  let rec try_loop: int -> (heap * (rvalue list) * (string list)) =  (* try different depths subsequently *)
					    fun i ->
					      if (i > max_rec_depth) 
					      then 
						let _ = debug_string ((String.make (2*lc) ' ')^"Max Loop Depth exceeded\n") in
						raise (Parsing_failed "Constructor failed: Max loop depth exceeded")						
					      else
						try
						  if (* Parse a string as a char lists *)
						    ((List.map stripinfo arg_types) = [CharTyp; ConTyp([],tid)])
						      &&
						    (try (String.length (List.hd input) > 1) with _ -> false)
						  then (* Chop the next input element into pieces *)
						    let (inp_chop) = (* Using string_chop did not work for some unkown reason: recursive calls got confused by counter! Resetting counter did not work! *)
						      match input with
						      | (clh :: clt) ->
							  let _ = debug_string ((String.make (2*lc) ' ')^"Parsing input string '"^clh^"' as a character list.\n") in
							  let (clhc,clhr) = string_split clh 1 in
							  clhc :: clhr :: clt
						      | _ -> bug "Chopping string to char list failed."
						    in parse_aux (i-1) csort_ord h arg_types inp_chop  (* Replacing (i-1) with (i-"length clh") did not work! The recursive call was immediately terminated! WHY?!? Bug due to optimizing? *)
						  else (* business as usual *)
						    parse_aux i csort_ord h arg_types input 
						with _ -> 
						  let _ = debug_string ((String.make (2*lc) ' ')^"Trying depth "^(string_of_int(i+1))^" for "^constr^"\n") in
						  try_loop (i+1)
					  in try_loop (lc+1)
				      in
				      let _ = debug_string ((String.make (2*lc) ' ')^"Constructor "^constr^" parsed\n") in
				      let (h2, l) = heap_store h1 (constr, rvs1) in
				      (h2, [PointerRVal(l)], input1)
				    end
				  with _ -> 
				    let _ = debug_string ((String.make (2*lc) ' ')^"Backtracking\n") in
				    loop_constrs rest (* Input does not fit the requires types for this constructor *) 
				end
			in loop_constrs sorted_cs 		      (* try them now subsequently *)
		      end
		  | ArrowTyp(dom,rng) -> (notImplemented "Argument parsing: First-oder types")
		  | TvarTyp(tv)       -> (notImplemented "Argument parsing: Typ variables.")
		  | ArrayTyp(st)      -> (notImplemented "Argument parsing: Arrays types.")
		  | _                 -> (notImplemented "Argument parsing: Unkown type (maybe parameterized constructor)")
		end
	      in 
	      let (h2, rvs2, input2) = parse_aux lc csort_ord h1 ts input1 in (* parsing remaining types, after first type is parsed *)
	      (h2, (List.append rvs1 rvs2), input2)
	end
    in
    let check_result: (heap * (rvalue list) * (string list)) -> (heap * (rvalue list)) =
      fun (h, rvs, input) ->
	match input with
	| []   -> (h, rvs)
	| h::t -> raise (Parsing_failed ("Some unparsed arguments are leftover: "^h) )
    in
    begin
      try                                        (*sort*)  (*keep order/big first*)
	check_result (parse_aux 0 (!the_options.clap_sort, !the_options.clap_ord)   h_ t_ input_)
      with _ -> (* Either input remained or parsing failed *)
	let _ = debug_string "\n*** Parsing failed. Trying again with reversed constructor order! ***\n" in
	check_result (parse_aux 0 (!the_options.clap_sort, not !the_options.clap_ord)  h_ t_ input_)
    end


(* EVALUATING FUNCTIONS: *)
(* eterm shall mean "evaluate term" *)

let runtime = ref 0.  (* for runtime measurement *)
(* MAIN *)
let rec eprogram: options -> program -> unit = 
  fun opt pr ->
    match pr with 
      Program(info, typdecs, valdecs, funblocks) -> 
	let _ = the_contab := compile_contab typdecs  in  (* Create Constructor-User-Size Lookup-Table --- !!! for statistical purposes only !!! *)
	let ft = compile_fun (List.flatten funblocks) in  (* Create Function-table *)
	let fct_main:funcidentifier =                     (* Get name of function main *)
	  if (FunTab.mem opt.main ft)
	  then 
	    opt.main
	  else
	    if (FunTab.mem Argument.fct_main_alt_default ft)
	    then
	      let _ = warning ("Function '"^opt.main^"' does not exist. Trying alternate default instead.") in
  	      Argument.fct_main_alt_default
            else
              err ("Function '"^opt.main^"' does not exist. Please specify the main function to be executed.")
        in
	(* OCAML IS EVIL: It tempts one to quickly hack some code which gets quickly messy and unitelligible!!! *)
	let (h, main_args) = (* Create heap with identified arguments for fct_main on it! *)
  	  let mainval:valdec = get_val_for fct_main valdecs in      (* Get signature for function main *)	  
	  let (decinfo, fct_main_typ) =                             (* Get type of function main *)
	    match mainval with
	    | ValDec(decinfo,_,typ)         -> (decinfo,  typ)
	    | AnnValDec(decinfo,_,_,rtyp,_) -> (decinfo, (strip_rich_typ rtyp))
	  in	
	  let fct_main_args_typs = 
	    try list_remove_last (uncurry fct_main_typ) (* Turn into list and remove the result type *)
	    with _ -> errAt decinfo ("Wrong type declaration for function '"^fct_main^"': must take at least argument.")
	  in
	  let pre_h = Heap.empty in (* Create empty heap - There can be only one! *)
	  try
	    parse_rval typdecs pre_h fct_main_args_typs opt.argmain
	  with (Parsing_failed s)-> errAt decinfo ("Commandline argument mismatches type declaration for function '"^fct_main^"'") (* s does not help here *)
	in
	let _ = 
	  print_string ("\n Executing '"^(fct_main)^"' with argument: \n   ");
	  begin
	    if not opt.debug 
	    then print_rvals       h main_args
	    else print_rvals_debug h main_args
	  end; 
	  print_string "   \n   " 
	in 
	let _ = (* Save current heapsize to ini-variables *)
	  the_heap_stats.uniform_ini  <- the_heap_stats.uniform_max;
	  the_heap_stats.weighted_ini <- the_heap_stats.weighted_cur;
	  the_heap_stats.usrsized_ini <- the_heap_stats.usrsized_cur;
	in
	let sep_bar = ("\n"^(String.make opt.screen_width '_')^"\n") in
	let _ = if not opt.debug then queue_warnings () in
	let _ = print_string "\n Program output follows: " in
	let _ = print_string sep_bar in
(*	let _ = print_string "\n--------------------------------------------------------------------------------\n" in *)
	let _ = runtime := Sys.time () in
 (*->*) let result = efunction ft h info (fct_main) main_args in  (* Evaluate function "fct_main" *) 
	let _ = runtime := (Sys.time ()) -. !runtime in
	let _ = print_string sep_bar in
	let _ = print_warnings () in
	let _ = print_string "\n Result of program evaluation: \n   ";
	        begin
	          if not opt.debug 
	          then print_rval       result
	          else print_rval_debug result
	        end; 
	        print_string "   \n   ";
	        begin
		 if (!runtime >= 0.03 ) 
		 then
	           (
		    print_string "\n Evaluation runtime: ";
                    print_float !runtime
                   )
		 else ()
                end;
	        print_string "\n Heap-consumption in terms of";
	        print_string "\n  * Uniform sized heap cells (one cell for each constructor):";
	        print_string "\n     - Number of heap store operations    : ";
	        print_aligned_int_ext the_heap_stats.num_heap_store;
	        print_string "\n     - Number of heap read operations     : ";
	        print_aligned_int_ext the_heap_stats.num_heap_read;
	        print_string "\n     - Number of heap remove operations   : ";
	        print_aligned_int_ext the_heap_stats.num_heap_remove;
	        (
		 if opt.debug 
		 then 
		   (
	            print_string "\n     - Current freelist after evaluation  : ";
	            print_string (print_pretty_int_list the_heap_op.freelist);
		    (*
		    print_rval ((fst result),PointerRVal(1));print_newline ();
		    print_rval ((fst result),PointerRVal(9));print_newline ();
		    print_rval ((fst result),PointerRVal(14));print_newline ();
		    print_rval ((fst result),PointerRVal(31));print_newline ();
		    print_rval ((fst result),PointerRVal(45));print_newline ();
		    print_rval ((fst result),PointerRVal(26));print_newline ();
		    print_rval ((fst result),PointerRVal(17));print_newline ();
		    print_rval ((fst result),PointerRVal(18));print_newline ();
		    *)
		   )
		);
	        print_string "\n     - Initial heap size                  : ";
	        print_aligned_int_ext the_heap_stats.uniform_ini;
	        print_string "\n     - Maximum heap size during evaluation: ";
	        print_aligned_int_ext the_heap_stats.uniform_max;
	        print_string "\n     - Current heap size after evaluation : ";
	        print_aligned_int_ext (heap_size (fst result));
	        begin
		  if (the_heap_stats.weighted_max = the_heap_stats.usrsized_max) 
		      && 
		    (the_heap_stats.weighted_cur = the_heap_stats.usrsized_cur) 
		      &&
		    (the_heap_stats.weighted_ini = the_heap_stats.usrsized_ini) 
		  then begin
		    print_string "\n\n  * User/Variable-sized heap cells (one cell per constructor argument):";
	            print_string "\n     - Initial heap size                  : ";
	            print_aligned_int_ext the_heap_stats.weighted_ini;
	            print_string "\n     - Maximum heap size during evaluation: ";
	            print_aligned_int_ext the_heap_stats.weighted_max;
	            print_string "\n     - Current heap size after evaluation : ";
	            print_aligned_int_ext the_heap_stats.weighted_cur
		  end else begin
	            print_string "\n\n  * Variable-sized heap cells (one cell per constructor argument):";
	            print_string "\n     - Maximum heap size during evaluation: ";
	            print_aligned_int_ext the_heap_stats.weighted_max;
	            print_string "\n     - Current heap size after evaluation : ";
	            print_aligned_int_ext the_heap_stats.weighted_cur;
		    begin
  	              if (the_heap_stats.usrsized_max > 0)
		      then begin
			print_string "\n\n  * Heap cells according to user defined constructor sizes:";
			print_string "\n     - Initial heap size                  : ";
			print_aligned_int_ext the_heap_stats.usrsized_ini;
			print_string "\n     - Maximum heap size during evaluation: ";
			print_aligned_int_ext the_heap_stats.usrsized_max;
			print_string "\n     - Current heap size after evaluation : ";
			print_aligned_int_ext the_heap_stats.usrsized_cur;
		      end
		    end
		  end
		end;
	  print_string "\n\n"
	in  ()

and efunction: funtab -> heap -> info -> funcidentifier -> (rvalue list) -> result =
  fun ft h info fid args -> 
    if (FunTab.mem fid ft) 
    then
      let FunctionDef(_, _, vars, defbdy) = FunTab.find fid ft in 
      let s = try compile_stack vars args with
	Invalid_argument(s) -> errAt info ("Wrong number of arguments for function call to '"^fid^"'.")
      in	                     
      eexpression ft h s defbdy
    else match (fid, args) with 
(* ----  THE BUILT-IN FUNCTIONS: ---- *)                                            (* "with _" means that the conversion cannot normally fail *)                        
      ("int_of_float", [FloatRVal(x)]) -> (try (h, IntRVal(int_of_float(x)))           with _                   -> errAt info ("Typecast int_of_float failed.") )
    | ("float_of_int", [IntRVal(x)])   -> (try (h, FloatRVal(float_of_int(x)))         with _                   -> errAt info ("Typecast float_of_int failed.") )
    | ("char_of_int",  [IntRVal(x)])   -> (try (h, CharRVal(char_of_int(x)))           with Invalid_argument(_) -> errAt info ("Typecast char_of_int failed.") )
    | ("int_of_char", [CharRVal(x)])   -> (try (h, IntRVal(int_of_char(x)))            with _                   -> errAt info ("Typecast int_of_char failed.") )
    | ("float_of_string", [StringRVal(x)]) -> (try (h, FloatRVal(float_of_string(x)))  with Failure(_)          -> errAt info ("Typecast float_of_string failed.") )
    | ("string_of_float", [FloatRVal(x)])  -> (try (h, StringRVal(string_of_float(x))) with _                   -> errAt info ("Typecast string_of_float failed.") )
    | ("int_of_string", [StringRVal(x)])   -> (try (h, IntRVal(int_of_string(x)))      with Failure(_)          -> errAt info ("Typecast int_of_string failed.") )
    | ("string_of_int", [IntRVal(x)])      -> (try (h, StringRVal(string_of_int(x)))   with _                   -> errAt info ("Typecast string_of_int failed.") )
    | ("print_int", [x])
    | ("print_float", [x])
    | ("print_char", [x])
    | ("print_string", [x])  -> (print_rval (h,x)); (h, UnitRVal)
    | ("print_int_newline", [x])
    | ("print_float_newline", [x])
    | ("print_char_newline", [x])
    | ("print_string_newline", [x])  -> (print_rval (h,x)); (print_newline ()); (flush stdout); (h, UnitRVal)
	  (* For convenience... *)
    | ("print", xs)          -> (print_rvals h xs); flush stdout; (h, UnitRVal) 
	  (* For even more convenience (returns last argument)... *)
    | ("print'", xs)         -> (print_rvals h xs); flush stdout; (h, (get_last UnitRVal xs)) 
	  (* print_debug and print_debug' print each constructor with its location *)
    | ("print_debug", xs)    -> print_rvals_debug h xs; (h, UnitRVal)        
    | ("print_debug'", xs)   -> print_rvals_debug h xs; (h, (get_last UnitRVal xs)) 
    | ("print_heap_u", _)   -> print_string "  Current uniform-sized heapsize: ";
      	                        print_int (heap_size h); 
	                        print_string "\n";
	                        (h, UnitRVal)
    | ("print_heap_v", _)   -> print_string " Current variable-sized heapsize: ";
      	                        print_int the_heap_stats.weighted_cur; 
	                        print_string "\n";
	                        (h, UnitRVal)
    | ("print_heap_s", _)   -> print_string "     Current user-sized heapsize: ";
      	                        print_int the_heap_stats.usrsized_cur; 
	                        print_string "\n";
	                        (h, UnitRVal)
    | ("print_newline", _)  -> (print_newline ()); flush stdout;   (h, UnitRVal)
    | ("array_head", _)      -> bugAt info "Array operations not implemented"
    | ("free", [PointerRVal(l)]) -> (* For compatibility with camelot *)
	if !the_options.inplace	then heap_push_freelist l;
	(h, UnitRVal)
    | _ -> errAt info ("Unknown identifier '"^fid^"' OR type mismatch by call to built-in function.")

and eexpression: funtab -> heap -> stack -> expression -> result =
  fun ft h s expr -> match expr.v with
  | ValueExp(value) -> 
      ( match value.v with (* Catch misparsed 0-arity function calls here: *)
        | VarVal(id) when not (Stack.mem id s) -> efunction ft h value.i id [] (* Not necessarily true that (FunTab.mem id ft) holds, as it could be a built-in function (currently not inside ft) *)
        | _ ->  (h, evalue s value) (* Heap not modified by evalue! *)
      )
  | ConstrExp(constr,vals,dia)  -> 
      let loc = 
	match dia with
	| New -> None
	| Reuse(dvar) -> 
	    (
	     match 
	       (
		try (Stack.find dvar s) 
		with Not_found -> errAt expr.i ("Unknown resource-variable '"^dvar^"'.")
	       )
	     with
	     | PointerRVal(l) -> Some(l)
	     | _ -> bugAt expr.i "PointerRVal expected as value of diamond type."
	    )
	| Void -> errAt expr.i "Constructor with invalid '@' location encountered. (Void)"
      in
      let rvals = List.map (fun v -> evalue s v) vals in
      let (h', l) = heap_store_at h (constr, rvals) loc
      in  (h', PointerRVal(l))
  | AppExp(fid,vals) -> 
      let rvals = List.map (fun v -> evalue s v) vals
      in  efunction ft h expr.i fid rvals
      (* HISTORICAL version from a time where values had access to the heap:
          let (h', rvals) = mapfold (fun (h1,v1) -> (evalue ft h1 s v1)) (h,vals) 
	  in   efunction ft h' expr.i fid rvals
       *)
  | LetExp(var, expr1, expr2) ->
     let (h', rval1) = eexpression ft h s expr1 in
     let s' = stack_bind var rval1 s 
     in eexpression ft h' s' expr2
  | SeqExp(expr1, expr2) ->
     let (h', rval1) = eexpression ft h s expr1 
     in eexpression ft h' s  expr2
  | IfExp(ifvalu , expr1, expr2) -> 
      let istrue = evalue s ifvalu in
      (
       match istrue with
	 BoolRVal(true)  -> eexpression ft h s expr1
       | BoolRVal(false) -> eexpression ft h s expr2
       | _ -> errAt expr.i "If-construct expects boolean value."
      )
(* 
   DEPRECATED: match/match' is decided in each matchrule!
  | MatchExp(var,  mrules) -> 
     let rv = Stack.find var s in (* By assumption we know that the program is well typed *)
     (
      match rv with 
	PointerRVal(loc) -> 
	  let (h', cell) = 
	    try heap_withdraw h loc with
	      Not_found -> errAt expr.i ("Dangling pointer encountered."^(string_of_loc loc))
	  in 
	  ematchrule ft h' s expr.i cell mrules
      | _ -> print_string "\n DEBUG:";
	     print_rval (h,rv);  (* DEBUG *)
	     print_string " \n  ";
	     errAt expr.i "Matching non-constructor types is not allowed."
     )
*)
  | MatchExp(var,  mrules) -> 
     let rv = 
       try Stack.find var s (* By assumption we know that the program is well typed *)
       with Not_found -> errAt expr.i ("Unknown variable '"^var^"'.")
     in 
     (
      match 
	rv 
      with
      |	PointerRVal(loc) -> 
	  let cell = 
	    try heap_read h loc with
	      Not_found -> errAt expr.i ("Dangling pointer encountered."^(string_of_loc loc))
	  in 
	  ematchrule ft h s expr.i loc cell mrules
      | _ -> print_string "\n DEBUG:";
	     print_rval_debug (h,rv);  (* DEBUG *)
	     print_string "! \n  ";
	     errAt expr.i "Matching non-constructor types is not allowed."
     )

and ematchrule: funtab -> heap -> stack -> info -> location -> heapcell -> matchrule list -> result =
  fun ft h s info loc (constr, rvals) mrules ->
    match mrules with
      [] -> errAt info ("Non-exhaustive match: constructor '"^constr^"' not covered.")
    | Matchrule(info', constr', vars, readonly, expr, dia)::_ when (constr = constr') ->
	let (h',s') = 
	  if readonly
	  then
	    (h,s)
	  else
	    let hact = heap_remove h loc (* Does not count as heap_read *)
	    in
	    let sact =
	      match dia with 
	      | Void -> bugAt expr.i "Destructive matching with read-only tag encountered."
	      | New -> (* Just use freelist *)
		  s
	      | Reuse(dvar) -> 
		  (heap_pop_freelist loc);
		  (stack_bind dvar (PointerRVal(loc)) s)
	    in
	    (hact,sact)
	in
	let s'' = 
	  try compile_stack_aux s' vars rvals 
	  with _ ->
	    (print_rvals h rvals);  (* DEBUG *)
	    bugAt info ("Numbers of variables ("^(string_of_int(List.length vars))^") and values ("^(string_of_int(List.length rvals))^") mismatch in matchingrule.")
	in
	eexpression ft h' s'' expr
    | _::rs -> ematchrule ft h s info loc (constr, rvals) rs


and evalue: stack -> value -> rvalue =
  fun s v -> match v.v with
  | VarVal(vari) -> (try Stack.find vari s with Not_found -> errAt v.i ("Unknown identifier '"^vari^"' encountered."))   (* PROBLEM: es knnte auch eine built-in-function ohne argument sein!!! *)
  | IntVal(x) -> IntRVal(x)
  | FloatVal(x) -> FloatRVal(x)
  | CharVal(x) -> CharRVal(x)
  | StringVal(x) -> StringRVal(x)
  | BoolVal(x) -> BoolRVal(x)
  | UnitVal -> UnitRVal
  | UnaryOpVal(unop, val1) -> 
      let rval1 = evalue s val1 
      in eunop unop rval1
  | BinaryOpVal(binop, val1, val2) -> 
      let rval1 = evalue s val1 in
      (* Da OCaml leider nicht LAZY ist, knnen wir hier nicht "let rval2 = evalue s val2" schreiben, sondern muessen es dreimal wiederholen...  *)
      (
        match (binop.v, rval1) with    (* Catch all partial-binary operators here... admittingly not very elegant! AND NOT EVEN NECESSARY -> No Side-Effects anymore! *)
	                               (* Note that andalso/orelse should be parsed as if-expressions, but this is not possible here. See comment in syntax.ml *)     
        | (AndalsoOp, BoolRVal(true))  -> (evalue s val2)  (* By assumption we know that the program type-checks... *)
        | (AndalsoOp, BoolRVal(false)) -> BoolRVal(false)
        | (OrelseOp,  BoolRVal(true))  -> BoolRVal(true)
        | (OrelseOp,  BoolRVal(false)) -> (evalue s val2)  (* By assumption we know that the program type-checks... *)
        | _ -> let rval2 = evalue s val2 
	       in ebinop binop rval1 rval2 
      )

and eunop: unaryoperator -> rvalue -> rvalue =
  fun op rval -> match (op.v,rval) with
    (NotOp,     BoolRVal(b)) -> BoolRVal(not b)
  | (UMinusOp,  IntRVal(w))  -> IntRVal(-w)
  | (UFminusOp, FloatRVal(w)) -> FloatRVal(-.w)
  | _                        -> errAt op.i "Type mismatch between unary operator and operand."
   
and ebinop: binaryoperator -> rvalue -> rvalue -> rvalue =
  fun op rv1 rv2 -> match (op.v,rv1,rv2) with
  | (TimesOp, IntRVal(i1), IntRVal(i2)) -> IntRVal(i1 * i2)
  | (DivOp, IntRVal(i1), IntRVal(i2))   -> IntRVal(i1 / i2)
  | (PlusOp, IntRVal(i1), IntRVal(i2))  -> IntRVal(i1 + i2)
  | (MinusOp, IntRVal(i1), IntRVal(i2)) -> IntRVal(i1 - i2)
  | (FtimesOp, FloatRVal(v1), FloatRVal(v2)) -> FloatRVal(v1 *. v2)
  | (FdivOp, FloatRVal(v1), FloatRVal(v2))   -> FloatRVal(v1 /. v2)
  | (FplusOp, FloatRVal(v1), FloatRVal(v2))  -> FloatRVal(v1 +. v2)
  | (FminusOp, FloatRVal(v1), FloatRVal(v2)) -> FloatRVal(v1 -. v2)
  | (LessOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 < v2)
  | (LessOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 < v2)
  | (LessOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 < v2)
  | (LessOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 < v2)
  | (LteqOp, IntRVal(v1),IntRVal(v2))        -> BoolRVal(v1 <= v2)
  | (LteqOp, FloatRVal(v1),FloatRVal(v2))    -> BoolRVal(v1 <= v2)
  | (LteqOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 <= v2)
  | (LteqOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 <= v2)
  | (GreaterOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 > v2)
  | (GreaterOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 > v2)
  | (GreaterOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 > v2)
  | (GreaterOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 > v2)
  | (GteqOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 >= v2)
  | (GteqOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 >= v2)
  | (GteqOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 >= v2)
  | (GteqOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 >= v2)
  | (EqualOp, IntRVal(v1), IntRVal(v2))         -> BoolRVal(v1 = v2)
  | (EqualOp, FloatRVal(v1), FloatRVal(v2))     -> BoolRVal(v1 = v2)
  | (EqualOp, CharRVal(v1), CharRVal(v2))       -> BoolRVal(v1 = v2)
  | (EqualOp, StringRVal(v1), StringRVal(v2))   -> BoolRVal(v1 = v2)
  | (EqualOp, BoolRVal(v1), BoolRVal(v2))       -> BoolRVal(v1 = v2)
  | (EqualOp, UnitRVal, UnitRVal)               -> BoolRVal(true)
  | (EqualOp, PointerRVal(v1), PointerRVal(v2)) -> BoolRVal(v1 = v2)
  | (ConsOp, _, _) ->  errAt op.i "Not implemented: Cons-operator."
  | (AppendOp, StringRVal(v1), StringRVal(v2)) -> StringRVal(v1^v2)
  | (AppendOp, StringRVal(v1), CharRVal(v2))   -> StringRVal(v1^(Char.escaped v2))
  | (AppendOp, CharRVal(v1), StringRVal(v2))   -> StringRVal((Char.escaped v1)^v2)
  | (AppendOp, CharRVal(v1), CharRVal(v2))     -> StringRVal((Char.escaped v1)^(Char.escaped v2))
  | (AndOp, BoolRVal(b1), BoolRVal(b2)) -> BoolRVal(b1 && b2)
  | (OrOp, BoolRVal(b1), BoolRVal(b2))  -> BoolRVal(b1 || b2)
  | (AndalsoOp, v1, v2) ->  errAt op.i "Not implemented: ANDALSO-operator." (* Now catched in eexpression... *)
  | (OrelseOp,  v1, v2)  ->  errAt op.i "Not implemented: ORELSE-operator." (* Now catched in eexpression... *)
  | (ModOp, IntRVal(i1), IntRVal(i2))   -> IntRVal(i1 mod i2)
  | _ -> errAt op.i "Type mismatch between binary operator and operands."   (* Should not occur by our assumption on input programs *)




