(**************************************************************************************

   Author:   Steffen Jost  <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: constraint.ml,v $
   Id:       $Id: constraint.ml,v 1.21 2005/04/01 02:01:49 a1hloidl Exp $ 

   This module constructs the LP, 
   feeds it to an LP-solver,
   and prints the resulting annotated type for each function.


   ToDos: 
   ------

   - Solution from lp_solve is not verified. (Since a simple recalculation may suffer the same numeric mistakes.)

   - Mutual recursive datatypes lead to non-termination. (Due to the stupid enrichment algorithm - may be fixed if necessary.)

   - Change the way the size of a constructor is obtained: 
     Use rcinfo.size instead of using csize_to_num on global.ct/constr (enhances readability and may remove old left-over code.)
     Remember to implement checking of option '-uni' and determining unspecified sizes, as done in csize_to_num.

   - Support parametric/polymorphic variant types. 
     (camelot -a2 eliminates those, so this is of low priority)

   - Catch wrong types in annotated val declarations. See comment "BUGGY" in function complete_annval_rt


   NotToDos (in order to remember what has already been considered):
   --------

   - Die ConTab (global.ct) ist NICHT berflssig geworden:
     Alle Informationen der ConTab sind mittlerweile auch im angereicherten Konstruktor-Typ geschpeichert;
     jedoch wird global.ct noch in der Funktion infer_type benoetigt.


   Notes:
   ------

   - Performance may be considerably increased by:
      - Eliminate quadratic runtime by remembering the inferred type
        of earlier runs of our mini-(incomplete)-type inference.
        (occurs only in nested let constructs like: "let x = let y=0 in y in x")
      - Eliminate quadratic runtime in printing routines 
        (do not remove duplicats, do not sort), 
        and call this routine only once instead of twice (once for the file, once for piping to the lp_solver),
        i.e. feed the file to the lp_solver instead of piping directly.
       
      - Removing all queries of the variable global.options.debug 


**************************************************************************************)


open Common
open Support.Error
open Syntax
open Argument

(*For Debugging*)
let act_fkt = ref "None"

(* Anything related to constraint-variables themselves... *)
type varuse = Lhs | Rhs | Datin | Datout | HDat | Aux | Misc | Konst (* Indicates the usage of a constraint variable and shall influence its name... *)



(* Unique counters for naming conventions *)
let num_lhs    = ref 0
let num_rhs    = ref 0
let num_datin  = ref 0
let num_datout = ref 0
let num_hdat   = ref 0
let num_aux    = ref 0
let num_misc   = ref 0

let varname_convention: varuse -> string =
  function 
    | Lhs    -> "x" (* Toplevel, lefthandside *)
    | Rhs    -> "y" (* Toplevel, righthandside *)
    | Datin  -> "u" (* Datatype annotation, occurring in signature (occurs left of an arrow) *)
    | Datout -> "v" (* Datatype annotation, occurring in signature (occurs only on the righthandside of an arrow) *)
    | HDat   -> "c" (* Datatype annotation, not occurring in signature *)
    | Aux    -> "a" (* A variable only found inside the type derivation tree *)
    | Misc   -> "b" (* A konstant which isnt one, ie AnnValDec containing a '*' instead of a value *)
    | Konst  -> "K" (* Prefix for constants. Necessary as lp_solve does not allow arithmetic expressions as factors. Must be a string of length 1 *)

let new_varname: varuse -> cvar =
  fun vu ->
    let x = 
      match vu with
      | Lhs    -> num_lhs
      | Rhs    -> num_rhs
      | Datin  -> num_datin
      | Datout -> num_datout
      | HDat   -> num_hdat
      | Aux    -> num_aux
      | Misc   -> num_misc
      | Konst  -> bug "Constant constraint variables cannot be generated arbitrarily."
    in
    let nums = string_of_int (save_ref_increment x) in
    let fill = if String.length(nums) = 1 then "0" else "" in
    ((varname_convention vu)^fill^nums)

let new_constant: float option -> cvar = (* We encode a float into a variable name *)
  fun f ->
    match f with 
    | Some c ->
	let o1 = print_unsigned_pretty_float c in
	let o2 = string_replace o1 '-' 'M' in (* there may be operands in the exponent *)
	let o3 = string_replace o2 '+' 'P' in
	if (string_contains_operator o3)
	then bug ("Cannot convert '"^o3^"' into a syntactic correct constant-variable.\n Fix: Manually edit constraint-file!")
	else ((varname_convention Konst)^o3)
    | None -> 
	new_varname Misc (* We cannot know where the constant is used *)

let is_cvar_constant: cvar -> bool =
  let k = String.get (varname_convention Konst) 0 in
  fun v -> ((String.get v 0) = k) 

let decode_constant: cvar -> string = (* Decode a variable name into a float *)
  fun v ->
    if (is_cvar_constant v) 
    then (* A coded constant *)
      let v1 = String.sub v 1 (String.length v - 1) in (* Remove leading 'K' *)
      let v2 = string_replace v1 'P' '+' in
      let v3 = string_replace v2 'M' '-' in
      let _ = float_of_string v3 in (* Test if convertible *)
      v3
    else raise (Failure "float_of_string")

let cv_one: cvar = (new_constant (Some 1.0))  (* Die Konstante 1 *)


(* Not sure if sensible:
   module SCVar = Set.Make(struct type t = cvar let compare = String.compare end)
   type scvar = SCVar.t
*)


(* The constraints themselves and related stuff: *)
module Constr= Map.Make(struct type t = cvar let compare = String.compare end)
type row = int Constr.t 
type constr = 
  | Eq0   of row * string   (* corresponds to 0  = (find var1)*var1 + (find var2)*var2 + ... *) (* the string argument may contain a label *)
  | Geq0  of row * string   (* corresponds to 0 >= (find var1)*var1 + (find var2)*var2 + ... *)
  | Trivial of string                  (* The empty constraint, merely for technical convenience (had been added later) *)
(* type constr_set = constr list (* Using module Set is akward, as it would require a total ordering... *) *)

type solution = float Constr.t (* A (partial) solution to an lp *)
let nonempty_solution: solution = (Constr.add "0" 0.0 (Constr.empty)) (* An empty solution to enforce some printout routines. *)

let linetag: info -> string = (* Print lines numbers within constraints *)
  fun i ->
    if !the_options.performance 
    then "_"
    else 
      match i with
      | Support.Error.UNKNOWN    -> ("_____")
      | Support.Error.FI(_,x,_) -> ("_"^(lined_aligned_int x)^"_")
	    

(* Type Global shall contain all global CONSTANTS, ie. things that are needed anywhere but are not changed during constraint building. The idea is to be able to add further types to this record later on. *)
type global = {ft: funtab; ct: contab; vt: rvaltab; td: typdec list; opt: Argument.options }
(* Use this template to generate new values of type global using the where construct. This makes sure that later change to type global wont produce compiler error *)
let global_template: global =
  {ft = FunTab.empty;    ct = ConTab.empty;  vt = FunTab.empty; td = []; opt = Argument.default } 
  (*  = defining bodies;    = constructor table; = enriched val decs; *)  

let the_global: global ref = ref global_template (* Nicht die feine Art, aber ich bins echt leid staendig global herumzureichen. Was sind eigentlich Monaden? *)


(* NOTE: The constr-type definition above is not really useful. Its pretty inconvenient to use! *)


let constr_add (*: cvar -> int -> row -> row*) =
  fun v i c ->
    if (Constr.mem v c)
    then 
      let oldi = Constr.find v c in
      let newi = oldi + i in
      if newi = 0
      then Constr.remove v c
      else Constr.add v newi c
    else 
      Constr.add v i c

(* What about polymorphism ?*)
let constr_add_float: cvar -> float -> (float Constr.t) -> (float Constr.t) =
  fun v i c ->
    if (Constr.mem v c)
    then 
      let oldi = Constr.find v c in
      let newi = oldi +. i in
      if newi = 0.
      then Constr.remove v c
      else Constr.add v newi c
    else 
      Constr.add v i c

let equal_l: string -> cvar -> cvar -> constr =
  fun l va vb -> (* va = vb   ===   0 = -va +vb *)
    if va = vb 
    then Trivial(l)
    else Eq0((constr_add va (-1) (constr_add vb 1 (Constr.empty))),l)
let equal: cvar -> cvar -> constr = equal_l ""

let equal0_l: string -> ((int * cvar) list) -> constr =
  fun l vs -> (* 0 = a1*v1 + a2*v2 + ... *)
  let f = fun cstr (ai,vi) -> constr_add vi ai cstr in
  let addrow = (List.fold_left f (Constr.empty) vs) in
  if addrow = Constr.empty
  then Trivial(l)
  else Eq0(addrow, l)

let equal0: ((int * cvar) list) -> constr = equal0_l ""

let geq_l: string -> (int * cvar) -> (int * cvar) list -> constr =
  fun l (a,v) vs -> (* a*v >= a1*v1 + a2*v2 + ...    ===   0 >= -a*v +a1*v1 +a2*v2 +...*)
    let f = fun cstr (ai,vi) -> constr_add vi ai cstr in
    let addrow = (List.fold_left f (constr_add v (-a) Constr.empty) vs) in
    if addrow = Constr.empty
    then Trivial(l)
    else Geq0(addrow ,l)

let geq: (int * cvar) -> (int * cvar) list -> constr = geq_l ""

let eq_l: string -> (int * cvar) -> (int * cvar) list -> constr =
  fun l (a,v) vs -> (* a*v = a1*v1 + a2*v2 + ...    ===   0 = -a*v +a1*v1 +a2*v2 +...*)
    let f = fun cstr (ai,vi) -> constr_add vi ai cstr in
    let addrow = (List.fold_left f (constr_add v (-a) Constr.empty) vs) in
    if addrow = Constr.empty
    then Trivial(l)
    else Eq0(addrow ,l)



let rec simplify: (constr list) -> (constr list) = (* At the moment, we just remove the trivial inequalities (unnecessary as they dont get printed)*)
  fun l -> simplify_aux [] l (* eta-reduktion eh nicht moeglich in OCAML wegen 'let REC' *)
and simplify_aux: (constr list) -> (constr list) -> (constr list) = (* At the moment, we just remove the trivial inequalities *)
  fun acc l -> 
    match l with
    | []    -> acc
    | x::xs -> 
	(
	 match x with
	 | Trivial(_) -> simplify_aux     acc  xs
	 | _          -> simplify_aux (x::acc) xs
	)

(*
   (* This one below is pretty inefficient, as filter preserves the order: *)
   let simplify: (constr list) -> (constr list) = (* At the moment, we just remove the trivial inequalities *)
     fun cs ->
       List.filter (function Trivial(_) -> false | _ -> true) cs
*)
(*
let simplify:  (constr list) -> (constr list) =
  List.fold_left (fun acc x -> match x with | Trivial(_) -> acc | _ -> x::acc) []
*)

let set_of_vars: (constr list) -> (cvar list) =
  let rec set_of_vars_aux: (constr list) -> (cvar list) =
    function 
      | []             -> []
      | Trivial(_) :: css -> set_of_vars_aux css
      | Eq0(tb,l)  :: css
      | Geq0(tb,l ):: css -> Constr.fold (fun k _ acc -> k::acc) tb (set_of_vars_aux css)
  in
  if !the_options.performance 
  then compose list_nub set_of_vars_aux  (* PERFORMANCE: list_nub is exepensive but reduced time for lp_solver!*)
  else fun cstrs -> 
    List.sort String.compare (list_nub  (set_of_vars_aux cstrs))



(* let sol_val: cvar -> solution -> float = fun v s -> Constr.find v s *)


let calc_row: row -> solution -> float =
  fun r sol ->
    let calc: cvar -> int -> float -> float =
      fun x a sum -> ((float_of_int a) *. (Constr.find x sol)) +. sum
    in Constr.fold calc r 0.


let filter_strict: (constr list) -> solution -> (constr list) =
  fun constrs sol ->
    let strict: constr -> bool =
      function 
	  Geq0(tb,l) -> 
	    let rowval = (calc_row tb sol) in
	    if (rowval = 0.) || (string_endswith l "_Ap0")
	    then false (* non-strict *)
	    else 
	      if (rowval <= 0.) 
	      then true  (* strict *)
	      else err ("Precision error: computed solution may not be valid for '"^l^"'!")
	| Eq0(tb,l)  -> 
	    if (calc_row tb sol) = 0. 
	    then false 
	    else err ("Precision error: computed solution may not be valid for '"^l^"'!")
	| Trivial(l) -> false
    in List.filter strict constrs



type enr_flag = Toplevel | Topleftofarrow | Nontoplevel

let rec enrich_type: (typdec list) -> enr_flag -> typ -> rich_typ = (* Der Boolwert gibt an ob toplevel variable bentigt werden *)
  (* Das notwendige durchreichen der typdec-list ist nervig! ConTyp sollte Anzahl der Konstruktoren enthalten! *)
 fun tds b typ -> match typ.v with
   | UnitTyp -> RUnitTyp
   | DiamondTyp -> RDiamantTyp
   | BoolTyp -> RBoolTyp
   | IntTyp  -> RIntTyp
   | FloatTyp -> RFloatTyp
   | ArrayTyp(styp) -> 
       (* Support.Error.notImplemented "Resource inference for array-types not possible." *)
       (* let _ = Support.Error.warning "Resource inference for array-types not possible. Treating array as unit type." in RUnitTyp *)
       raise (Invalid_argument "Arrays not implemented")
   | CharTyp  -> RCharTyp
   | StringTyp -> RStringTyp
   | TvarTyp(tvar) -> RTvarTyp(tvar)
   | ArrowTyp(styp1, styp2) -> 
       let b1 = 
	 match b with
	 | Toplevel -> Topleftofarrow
	 |  _ as id -> id
       in RArrowTyp((enrich_type tds b1 styp1),(enrich_type tds b styp2))
   | ConTyp(ts, id) -> 
       let constrs = get_constrs tds id in
       let f = fun (tb,ord) tc -> ((enrich_constructor tb tds b tc ord id), ord+1) in (* Hsslicher Hack, aber diese Funktionalitt in enrich_constructor einzubauen fand ich auch nicht so gut *)
       let (rtc,maxord) = List.fold_left f (ConTab.empty,1) constrs in
       let ets = List.map (enrich_type tds b) ts in
       RConTyp(ets, id, rtc)

and enrich_constructor: rt_contab -> (typdec list) -> enr_flag -> typcon -> int -> typeidentifier -> rt_contab = (* Ebenfalls hssliche Spezial-Funktion. Irgendwie schaffe ich es nicht nachtrglich schnen Programmcode einzuflicken. *)
  fun tb tds b tc ord owntid ->
    let cv = new_varname 
	(
	 match b with
	 | Toplevel        -> Datout
	 | Topleftofarrow  -> Datin
	 | Nontoplevel     -> HDat
	)
    in
    let TypCon(info, constr, csize, args) = tc in
    (* ENDLOSSCHLEIFEN-GEFAHR! Die Selbstreferenz muss abgefangen werden: *)
    let f = function
              | {v=(ConTyp(_,t))} when (t = owntid) -> RSelfTyp(t) 
              | (_ as ty)                           -> (enrich_type tds b ty)
    in
    let ats = List.map f args in
    let rci = {rcvar=cv; rorder=ord; rarg_types=ats ; rsize=csize } in (* !!! Be careful when using rsize !!! Check option '-uni' !!! Prepare for unspecified SIZES !!! *)
    ConTab.add constr rci tb


let rec set_to_zero_l: string -> rich_typ -> constr list =
  fun label rt ->
    match rt with
    | RUnitTyp 
    | RDiamantTyp
    | RBoolTyp 
    | RIntTyp 
    | RFloatTyp
    | RCharTyp 
    | RStringTyp     -> []
    | RTvarTyp(tvar) -> []
    | RSelfTyp(tid)  -> []
    | RArrowTyp(rta,rtb) -> 
	(* Requires all annotations to be zero: *)
	(* list_append (set_to_zero_l label rta) (set_to_zero_l label rtb) *) 
	(* Forces only righthandside to zero *)
	   set_to_zero_l label rtb                                           
    | RConTyp(rtargs, tid, rttab) -> 
	list_append 
	  (list_concat (List.rev_map (set_to_zero_l label) rtargs))
	  (
	   let folder: constructor -> rich_coninfo -> (constr list) -> (constr list) =
	     fun cnstr rci acc ->
	       list_append 
		 (
		   list_concat (List.map (set_to_zero_l label) rci.rarg_types)
		 )
	        (
		 (equal0_l label [(1,rci.rcvar)])::acc
		)
	   in
	   ConTab.fold folder rttab []
	  )


(* Aus Altersgrnden: Eigentlich sollte csize_to_num besser auf rich_coninfo arbeiten! *)
let csize_to_num: coninfo -> int = 
  fun ci -> Syntax.ifuni_size (!the_global.opt.uniform) ci 
(*
    if   (!the_global.opt.uniform)
    then Syntax.ifuni_size true ci
    else Syntax.ifuni_size false ci
*)(* 
   This is strange! The following definition is accepted, but does not work as desired. 
   The reference is checked only once - and too early!
   
   let csize_to_num: coninfo -> int = (Syntax.ifuni_size (!the_global.opt.uniform)) 
*)

   
let unfold: rich_typ -> rich_typ =
  function
    | (RConTyp(params, tid, rtt)) as rt -> 
	let uf: rich_typ -> rich_typ = 
	  function 
	    | (RSelfTyp(id)) when id = tid -> rt 
	    |  _  as other                 -> other
	in
	let rtt' = ConTab.map (fun rci -> { rci with rarg_types = List.map uf rci.rarg_types}) rtt
	in RConTyp(params, tid, rtt')
    | _ as rt -> (print_string "\n Warning: useless unfold! "); rt


(* Ursprnglich Version: *)
(*
let rec identif1y_args: rich_typ -> ((rich_typ list) * rich_typ) =     (* identify as many arguments as possible of a function type *) 
  fun rt -> identify_args_aux [] rt (* Ocaml does not allow eta here:
	    identify_args_aux [] 
				     *)
and identify_args_aux: (rich_typ list) -> rich_typ -> ((rich_typ list) * rich_typ) =
  fun acc rt -> match rt with
    | RArrowTyp(rta, rtb)  -> identify_args_aux (rta::acc) rtb (* Reverse accumulator at the end! *)
    | _ as rte             -> ((List.rev acc), rte)
*)


let identify_args: rich_typ -> ((rich_typ list) * rich_typ) =       (* identify as many arguments as possible of a function type *) 
  let rec identify_args_aux: (rich_typ list) -> rich_typ -> ((rich_typ list) * rich_typ) =
    fun acc rt -> match rt with
    | RArrowTyp(rta, rtb)  -> identify_args_aux (rta::acc) rtb (* Reverse accumulator at the end! *)
    | _ as rte             -> ((List.rev acc), rte)
  in
  identify_args_aux [] 



(* Enriched VAL-Deklarations *)


let complete_annval_rt: funcidentifier -> (typdec list) -> rich_typ -> rich_typ = (* Parsing an 'AnnValDEC' produces directly a rich_typ, but only an incomplete one. Validate and complete it! I know, the whole thing is an awful hack, instead of an rich_typ, there should be a new datatype pre_richtype or so... *)
  fun fid tds ->
    let error_tag: typeidentifier = "ErrorType" in
    let rec complete_annval_rt_aux: typeidentifier -> rich_typ -> rich_typ = (* 1.Arg: Remembers the supertype for updating RSelfTyp("???") *)
      fun super_typid rt ->
	match rt with
	| RConTyp(pars,tid,rcontab) ->
	    let pars' = List.map (complete_annval_rt_aux super_typid) pars in
	    (* Get Constructor-List for this type *)
	    let TypDec(info,tpars,_,tconstrs) = 
	      try  List.find (function TypDec(_,_,id,_) -> tid = id) tds (* Syntax.get_constrs does the same! Use it! *)
	      with Not_found -> Support.Error.err ("Unknown type in annotated val declaration for '"^fid^"': '"^tid^"'.")
	    in
	    let update: constructor -> rich_coninfo -> rich_coninfo =
	      fun consn rci -> 
		(* Get the Constructor Info *)
		let ((TypCon(info,_,csiz,cargs)), ord) = 
		  try  list_orderedfind (function TypCon(_,id,_,_) -> consn = id) tconstrs 
		  with Not_found -> Support.Error.err ("Unknown constructor in annotated val declaration for '"^fid^"': '"^consn^"'.")
		in 
		let rargt' = List.map (complete_annval_rt_aux tid) rci.rarg_types
		in (* BUGGY: We do not check wether rci.rarg_types fits cargs! *)
		let rcv' = new_constant 
		    (
		     try 
		       Some (float_of_string rci.rcvar)
		     with _ -> None
		    )
		in
		{  rcvar=rcv'; rorder= ord; rsize= csiz; rarg_types= rargt' }
	    in
	    let rcontab' = ConTab.mapi update rcontab 
	    in RConTyp(pars',tid,rcontab')
	| RArrowTyp(rta,rtb) -> 
	    let rta' = complete_annval_rt_aux super_typid rta in
	    let rtb' = complete_annval_rt_aux super_typid rtb in
	    RArrowTyp(rta',rtb')
	| aself when (aself = rSelfTypUnknown) -> 
	    if super_typid = error_tag
	    then Support.Error.err ("Misplaced '"^selfSymbol^"' encountered in annotated val declaration for '"^fid^"'.")
	    else RSelfTyp(super_typid)
	| other -> other
    in complete_annval_rt_aux error_tag
	  

let enrich_fun_val: valdec -> (typdec list) -> rich_valdec =
  function 
    | ValDec(info, fid, typ) -> 
	let din = new_varname Lhs in
	let dout = new_varname Rhs in
	(
	 function tds ->
          let etyp = enrich_type tds Toplevel typ in
	  RValDec(fid, din, etyp, dout) 
	)
    | AnnValDec(info, fid, fdin, frt, fdout) -> 
	let fdin'  = (new_constant fdin)  in
	let fdout' = (new_constant fdout) in
	( 
	  function tds ->
	    let frt'   = complete_annval_rt fid tds frt  in
	    RValDec(fid, fdin', frt', fdout')
	 )
	

let compile_rvaldec_aux: (typdec list) -> rvaltab -> (valdec list) -> rvaltab =
  fun tds -> 
    List.fold_left  (* ('a -> 'b -> 'a) -> 'a -> 'b list -> 'a *)
      (fun tbl vd -> 
	(	
	  match vd with
	  | ValDec(info,id,_)
	  | AnnValDec(info,id,_,_,_) ->
	      try
		let _ = 
		  if FunTab.mem id tbl 
		  then Support.Error.warningAt info ("Multiple declarations for function '"^id^"'. Dropping previous declaration.")
		in
		FunTab.add id (enrich_fun_val vd tds) tbl
              with
	      |  (Invalid_argument "Arrays not implemented") -> 
		  (* HWL: ignoring warning for demo version *)
		  (* let _ = Support.Error.warningAt info ("Resource inference for array-types not possible. Ignoring function '"^id^"'.")
		  in *) tbl 
	      |  (Invalid_argument "Typeidentifier not declared.") -> 
		  Support.Error.errAt info ("Unknown typeidentifier.")
	 )
      )
      
let compile_rvaldec: (typdec list) -> (valdec list) -> rvaltab =
  fun tds -> (compile_rvaldec_aux tds FunTab.empty)

	 
(* Enriched Contexts *)
module Context = Map.Make(struct type t = variable let compare = String.compare end)
type rich_context = rich_typ Context.t

let bind: rich_context -> variable -> rich_typ -> rich_context =
 fun ctxt v rt -> 
   if v <> "_" (* We simply ignore underscores *)
   then
     if (Context.mem v ctxt) 
     then cerr ("Variable '"^v^"' already bound in present context.")
	 (* THIS MUST BE AN ERROR RATHER THAN A WARNING!
            Otherwise we should use module Hashtable instead of Map for our context
	    in order to deal with hiding of variables.
	  *)
     else Context.add v rt ctxt
   else ctxt

let change_bind: rich_context -> variable -> rich_typ -> rich_context =
 fun ctxt v rt -> 
   if v <> "_" (* We simply ignore underscores *)
   then
     if not (Context.mem v ctxt) 
     then bug ("Changing binding of variable '"^v^"' in present context failes, variable not bound.")
     else Context.add v rt ctxt
   else ctxt


(* General functions needed for sharing *)

let rec collect_vars_valus: (value list) -> (variable list) = (* the resulting variable list may contain duplicates! *)
  function
    | [] -> []
    | valu::valus ->
	(
	 match valu.v with
	 | VarVal(x) 
	   -> 
	     if   is_function_call x !the_global.ft   (* Unschn! *)
	     then collect_vars_valus valus 
	     else x :: collect_vars_valus valus 
	 | UnaryOpVal( _ , valu1) 
	   -> collect_vars_valus (valu1::valus)
	 | BinaryOpVal( _ , valu1, valu2) 
	   -> collect_vars_valus (valu1::valu2::valus)
	 | _ 
	   -> collect_vars_valus valus
	)

let rec collect_vars_expr: expression -> (variable list) = (* the resulting variable list may contain duplicates! *)
  fun expr -> (collect_vars_expr_aux expr)
and collect_vars_expr_aux: expression -> (variable list) = (* the resulting variable list may contain duplicates! *)
  fun expr -> match expr.v with
  | ValueExp(valu)                      -> collect_vars_valus [valu]
  | ConstrExp(constr, valus, Reuse(d))  -> d::(collect_vars_valus valus)
  | ConstrExp(constr, valus, _)         -> collect_vars_valus valus
  | AppExp(id, valus)                -> collect_vars_valus valus
  | LetExp(letvar, expra, exprb)     -> 
      let vara  = collect_vars_expr_aux expra in
      let varb  = collect_vars_expr_aux exprb in
      list_append vara (list_remove letvar varb) (* letvar zaehlt hier nicht! *)
  | SeqExp(expra, exprb)             -> 
      let vara  = collect_vars_expr_aux expra in
      let varb  = collect_vars_expr_aux exprb in
      list_append vara varb 
  | IfExp(ifvalu, thenexpr, elseexpr) ->
      let varif  = collect_vars_valus [ifvalu] in
      let varth  = collect_vars_expr_aux thenexpr in
      let varel  = collect_vars_expr_aux elseexpr in
      list_concat [varif; varth; varel] 
  | MatchExp(mvar, mrules)           
(*| MatchPrExp(mvar, mrules)  *)      ->
      mvar :: (List.concat (List.map collect_vars_mrule_aux mrules)) 
(*        mvar :: (list_concat_map collect_vars_mrule_aux mrules) *)
  
and collect_vars_mrule_aux: matchrule -> (variable list) = (* the resulting variable list may contain duplicates! *)
  function
    | (Matchrule(info, constr, vars, is_destr, expr,Reuse(d))) ->
	list_diff (collect_vars_expr_aux expr) (d::vars)
    | (Matchrule(info, constr, vars, is_destr, expr,_)) ->
	list_diff (collect_vars_expr_aux expr) vars 

let collect_vars_mrules: (matchrule list) -> (variable list) = (* the resulting variable list may contain duplicates! *)
  fun mrules ->
    (List.concat (List.map collect_vars_mrule_aux mrules)) 
(*      (list_concat_map collect_vars_mrule_aux mrules) *)
	
let rec is_var_in_valus: variable -> (value list) -> bool =
  fun var valus ->
    match valus with
    | []         -> false
    | valu::rest ->
	(
	 match valu.v with
	 | VarVal(x) -> 
	     if var = x 
	     then true
	     else is_var_in_valus var rest
	 | UnaryOpVal(op,valu1) ->          (* Im Moment koennte man Operatoren eigentlich ignorieren, aber man weiss ja nie wie sichs entwickelt! *)
	     is_var_in_valus var (valu1::rest)
	 | BinaryOpVal(op, valu1, valu2) ->
	     is_var_in_valus var (valu1::valu2::rest)
	 | _ ->
	     is_var_in_valus var rest
	)

let rec is_shared: value -> (value list) -> bool =
  fun valu valus -> match valu.v with
  | VarVal(x) -> is_var_in_valus x valus
  | UnaryOpVal(op,valu1) ->                (* Im Moment koennte man Operatoren eigentlich ignorieren, aber man weiss ja nie wie sichs entwickelt! *)
      is_shared valu1 valus
  | BinaryOpVal(op, valu1, valu2) ->
      is_shared valu1 valus || is_shared valu2 valus ||	is_shared valu2 [valu1]
  | _ -> false



(* General Debug/output Utilities *)

let print_var_sol: solution -> cvar -> string =
  fun sol cvar ->
    if Constr.mem cvar sol
    then                              (* Print the value of this variable *)
      if !the_global.opt.diamond 
      then ("<"^(print_normal_float (Constr.find cvar sol))^">")
      else (print_normal_float (Constr.find cvar sol))
    else                              (* Print the unbounded variable's name: cvar *)
      if !the_global.opt.diamond 
      then ("<"^cvar^">")
      else (cvar)

let rec print_rt_sol: solution -> rich_typ -> string = (* For tracing rich_types *)
  fun sol t -> match t with
  | RUnitTyp              -> "unit"
  | RDiamantTyp           -> if !the_global.opt.diamond then "diamond" else "<>" (* diamond=true means we do the nice & fancy printout, thus there are already a lot of '<' and '>' symbols which might lead to confusion! *)
  | RBoolTyp              -> "bool"
  | RIntTyp               -> "int"
  | RFloatTyp             -> "float"
  | RCharTyp              -> "char"
  | RStringTyp            -> "string"
  | RTvarTyp(typvar)      -> "tvar("^typvar^")"
  | RConTyp(rts,id,rti)   -> (* argumente-list, typ-identifier und resource variable info. Bsp: int ilist[3], int bool tree[4,5], etc *)
      let params= List.fold_left (fun a x -> a ^ (print_rt_sol sol x)) "" rts in
      let cvars= 
	let raw = 
	  ConTab.fold 
	    (
	     fun cstr rcinf acc -> 
	       (* Sorting in this way might be dangerous, as arguments may be swapped according to solution... *)
	       (* let args  = List.sort (flip String.compare) (List.map (print_rt_sol sol) rinf.rarg_types) in *)
	       let args  = (List.map (print_rt_sol sol) rcinf.rarg_types) in 
	       let annota= print_var_sol sol rcinf.rcvar in 
	       let ord   = (string_of_int (rcinf.rorder+1))^"" in
(*	       (ord^(String.concat "," (List.append args [annota]))) :: acc    (* No Constructor Names, but Constructor Order "1","2",... *) *)
	       (ord^cstr^"("^(String.concat "," (List.append args [annota]))^")") :: acc
	    ) rti []
	in
	let sorted = (List.map (string_drop 2) (List.sort String.compare raw))
	in (* Order of Constructors as defined - remove 'ord^' if alphabetical order is desired *)
	if (!the_global.opt.debug || !the_global.opt.diamond)
	then String.concat "|"   sorted 
	else String.concat "|" (List.map (fun s -> string_rdrop 1 (string_drop 1 (string_drop_until '(' s))) 
 			         sorted)
      in 
      if (params = "")
      then id ^ "[" ^ cvars ^ "]" 
      else "(" ^ params ^ id ^ "[" ^ cvars ^ "]" ^ ")"
  | RArrowTyp(rtd,rtr)    -> (print_rt_sol sol rtd) ^" -> "^ (print_rt_sol sol rtr)   (* higher-order types kennen wir nicht: muss also rechtsgeklammert sein und somit annotationsfrei, da diese in der ValDec drin sind *)
  | RSelfTyp(id)          -> selfSymbol (* "Self("^id^")" *)

let print_rt: rich_typ -> string = 
  print_rt_sol nonempty_solution


let print_val: value -> string =
  fun v -> match v.v with 
    | VarVal(x) -> x
    | IntVal(x) -> string_of_int x
    | FloatVal(x) -> string_of_float x
    | CharVal(x) -> cerr "Internal Function 'print_val' still unfinished for characters"
    | StringVal(x) -> x
    | BoolVal(true) -> "True"
    | BoolVal(false) -> "False"
    | UnitVal -> "()"
    | _ -> cerr "Internal Function 'print_val' still unfinished for operators"

let print_val_ctxt: rich_context -> value -> string =
  fun c v -> match v.v with 
    | VarVal(x) -> print_rt 
	  (
	   try  Context.find x c
	   with Not_found -> cerrAt v.i ("Variable '"^x^"' not found in context.")
	  )
    | IntVal(x) -> string_of_int x
    | FloatVal(x) -> string_of_float x
    | CharVal(x) -> cerr "Internal Function 'print_val' still unfinished for characters"
    | StringVal(x) -> x
    | BoolVal(true) -> "True"
    | BoolVal(false) -> "False"
    | UnitVal -> "()"
    | _ -> cerr "Internal Function 'print_val' still unfinished for operators"


let pos_cerr: 'a list -> string -> unit = (* Raises exception if list argument is not empty *)
  function 
    | [] -> fun s -> ()
    | _  -> cerr 
let pos_cerrAt: 'a list -> info -> string -> unit = (* Raises exception if list argument is not empty *)
  function 
    | [] -> fun i s -> ()
    | _  -> cerrAt

   
let print_rvaltab_sol: solution -> rich_valdec -> string =
  fun sol (RValDec(fid,din,rt,dout)) ->
    (* THESE things for a nice line break are implemented poorly: just count the length of the output string and search for a break properly instead of guessing! *)
    let width:int = !the_global.opt.screen_width in  (* SCREEN-WIDTH tested at 'home': 130 *) 
    let alig:int  = ((width / 12) + 5) in            (* General guessed length of a function identifier for aligned printing *)
    (* the two above cannot be moved outside the "fun", for then ocaml will always use the default-width *)
    let len  = String.length fid in
    let fill = 
      if len < alig
      then String.make (alig - len) ' '
      else ""
    in
    let f_out_rt =
      if sol = Constr.empty 
      then
	("Constraints for this function are already infeasible on their own")  (*BAD STYLE, function may be used in different context! Use RAISE and TRY instead! *)
      else
	let out_rt = (print_rt_sol sol rt)
	in ((print_var_sol sol din) ^ ", " ^ out_rt ^ ", " ^ (print_var_sol sol dout))
    in
    let out_line = ("  "^fid^fill^": " ^ f_out_rt ^ ";\n") in
    try 
      string_break out_line width '-' ("\n"^(String.make (alig+6) ' '))
    with _ -> out_line

let print_rvaltab: rich_valdec -> string = 
  print_rvaltab_sol nonempty_solution

let print_constr: constr -> string =
  let pr_l: string -> string =
    function
      | ""     -> "     "
      | _ as l -> 
	  let lc = 
	    let len = String.length l in
	    if len > 26 (* lp_solve does not accept huge labels, and simply ignores inequalities with huge labels! *)
	    then 
	      let cut_sym = "$" in
	      try
		let cut = (String.rindex_from l (len - 5) '_') in
		let (st,en) = string_split l cut in
		let st' = string_rdrop (len - 24) st in
		(st'^cut_sym^en)
	      with 
		_ -> ((String.sub l 0 20)^cut_sym^(String.sub l (len-4) 4))   (* Just in case... *)
	    else l
	  in (lc^": ")
  in
  function 
    | Eq0(tb,l)  -> 
	if !the_global.opt.performance 
	then           (Constr.fold (fun cv fak acc -> (" "^(print_pretty_int' fak)^"*"^cv^" "^acc)) tb  "   = 0 ; \n")
	else ((pr_l l)^(Constr.fold (fun cv fak acc -> (" "^(print_pretty_int' fak)^"*"^cv^" "^acc)) tb  "   = 0 ; \n"))
    | Geq0(tb,l) -> 
	if !the_global.opt.performance 
	then           (Constr.fold (fun cv fak acc -> (" "^(print_pretty_int' fak)^"*"^cv^" "^acc)) tb  "  <= 0 ; \n")
	else ((pr_l l)^(Constr.fold (fun cv fak acc -> (" "^(print_pretty_int' fak)^"*"^cv^" "^acc)) tb  "  <= 0 ; \n"))
    | Trivial(l) -> 
	if !the_global.opt.debug 
	then ("/* " ^ (pr_l l) ^ "trivial ; */ \n")
	else ""

let print_constr_sol: solution -> constr -> string = (* Prints constraints containing unsatified inequalities *)
  fun sol constr -> 
    let vs = set_of_vars [constr] in
    if (List.for_all (fun v -> Constr.mem v sol) vs)
    then ""
    else print_constr constr


(* ---------------- Stuff for pickling rich types (kwxm) ---------------- *)

exception ConvertError of string

let toList t = ConTab.fold (fun s a l -> (s,a)::l) t []

let getSpace sol var = 
  if Constr.mem var sol
  then int_of_float (Constr.find var sol)
  else raise (ConvertError var)

let rec convert: solution -> rich_typ -> Camelot_absyn_ASDL.rich_ty
  = fun sol t -> match t with
  | RUnitTyp              -> Camelot_absyn_ASDL.RICH_UNITty
  | RDiamantTyp           -> Camelot_absyn_ASDL.RICH_DIAMONDty ""
  | RBoolTyp              -> Camelot_absyn_ASDL.RICH_BOOLty
  | RIntTyp               -> Camelot_absyn_ASDL.RICH_INTty
  | RFloatTyp             -> Camelot_absyn_ASDL.RICH_FLOATty
  | RCharTyp              -> Camelot_absyn_ASDL.RICH_CHARty
  | RStringTyp            -> Camelot_absyn_ASDL.RICH_STRINGty
  | RTvarTyp typvar       -> Camelot_absyn_ASDL.RICH_TVARty typvar
  | RArrowTyp(rtd,rtr)    -> Camelot_absyn_ASDL.RICH_ARROWty (convert sol rtd, convert sol rtr)
  | RSelfTyp(id)          -> Camelot_absyn_ASDL.RICH_SELFty
  | RConTyp(rts,id,rti)   -> 
      let twiddle (name,ci) = 
	let space = getSpace sol ci.rcvar in
	Camelot_absyn_ASDL.RICH_CON (name, List.map (convert sol) ci.rarg_types, space) in
      let args = toList rti in
      Camelot_absyn_ASDL.RICH_CONty (List.map (convert sol) rts, id, List.map twiddle args)


let convert_rvaldec: solution -> rich_valdec -> Camelot_absyn_ASDL.rich_valdec =
  fun sol (RValDec(fid,din,rt,dout)) ->
      if sol = Constr.empty 
      then raise (ConvertError "Constraints for this function are already infeasible on their own")
      else Camelot_absyn_ASDL.RICH_VALDEC (fid, getSpace sol din, convert sol rt, getSpace sol dout)


let pickle_rvaldec: out_channel -> solution -> rich_valdec -> unit =
  fun oc sol rv -> Camelot_absyn_ASDLUtil.write_rich_valdec (convert_rvaldec sol rv) oc

(* ---------------- End of additions ---------------- *)


let constr2file: global -> (constr list) -> bool -> (out_channel * (string option)) -> unit =
(* This function shall print the constraints in lp_solver-format (and is written in an iterative style...) *)
(* CURRENTLY WRITTEN FOR: "lp_solve" version 4.0.1.0 (ftp://ftp.es.ele.tue.nl/pub/lp_solve ,  LesserGNU public license) *)
  fun gl constrs int_lp (outch,oname) ->
    let write:string -> unit = output_string outch in
    let avs = (set_of_vars constrs) in 
    let _ = (write ("/*\n  This file is an automatically generated lp for '"^lp_solver^"'."));
            (
	     match oname with
	     | Some name -> write (" ("^name^") \n") 
	     | None      -> write (" \n") 
	    );
            (write ("  Contains "^(string_of_int (List.length constrs)) ^ " inequalties in "^(string_of_int (List.length(avs)))^" variables.\n" ));
            (write "*/\n\n");
    in
 (* Print enriched VAL-Deklarations as comment *)
    let _ = write "/* \n" in
    let _ = FunTab.iter (fun id rv -> write (print_rvaltab rv)) gl.vt in
    let _ = write "*/ \n\n" in
 (* Print Objective-Function *)
    let obj =
      let f = fun v -> 
	match (Char.escaped (String.get v 0)) with
	| x when x = (varname_convention Lhs)    -> " "^(!the_global.opt.obj_lhs)^"*"^v
	| x when x = (varname_convention Rhs)    -> " "^(!the_global.opt.obj_rhs)^"*"^v
	| x when x = (varname_convention Datin)  -> " "^(!the_global.opt.obj_datin)^"*"^v
	| x when x = (varname_convention Datout) -> " "^(!the_global.opt.obj_datout)^"*"^v
	| _                                      -> ""
      in List.map f avs
    in
    let _ = write ("MIN: "^(String.concat "" obj)^" ;\n\n") in
 (* Print all constraints *)
    let _ = List.iter (fun x -> write (print_constr x)) constrs in
 (* Print non-negativity constraints - All constants must be positive! *)
    let _ = List.iter (fun x -> write (x^" >= 0 ;\n")) avs in
 (* Print constants OR upper-bounds constraints - prohibits unbounded LPs *)
    let _ = 
      List.iter 
	(fun x ->
	  if (is_cvar_constant x) 
	  then (* A coded constant *)
	    try
	      let c = decode_constant x in (* Extract constant value from name *)
	      write (x^" =  "^c^" ;\n")
	    with (Failure "float_of_string") -> bug ("Floatconstant-variable '"^x^"' cannot be reverted to float.")
	  else (* An ordinary variable, not a constant *)
	    write (x^" <= "^(!the_global.opt.infinity)^" ;\n")
	) avs 
    in
 (* Print integer restrictions if desired *)
    let _ = if int_lp 
            then List.iter (fun x -> write ("int "^x^" ;\n")) avs 
    in
    ()


let rec file2solution: global -> (constr list) -> in_channel -> solution =
  fun gl constrs inch ->
    let read : unit -> string = function () -> (input_line inch) in
    try 
      if (read()) = "This problem is infeasible" 
      then
	(* let _ = print_string "\n\n   ---   LP is infeasible   --- \n\n" in *)
        Constr.empty
      else
	let avs =  (set_of_vars constrs) in 
    	let _ = while (read()) <> "Actual values of the variables:"  (* Header that usually precedes solution of lp_solver! *)
                do () done 
	in
	let rec parselines: solution -> solution =
	  fun tbl ->
	    try 
	      let line = read () in
	      let br = String.index line ' ' in
	      let var  = String.sub line  0 br  in
	      if (List.mem var avs) && not (Constr.mem var tbl)
	      then
		let valu = float_of_string (String.sub line (br+1) ((String.length line)-(br+1))) in
		if valu <= ((float_of_string (!the_global.opt.infinity)) -. (float_of_string (!the_global.opt.delta)))
		then parselines (constr_add_float var valu tbl)
		else parselines tbl
	      else 
		if (Constr.mem var tbl) 
		then
		  err ("LP_solve returned double variable '"^var^"'. See written file for input to '"^lp_solver^"'.")
		else
		  err ("LP_solve returned unknown variable '"^var^"'. See written file for input to '"^lp_solver^"'.")
		    
	    with 
	    | End_of_file -> tbl (* Shall we close here ? *)
	    |  _          -> cerr ("Problem parsing output of '"^lp_solver^"'. See written file for input to '"^lp_solver^"'.")
	in parselines Constr.empty
    with  End_of_file -> cerr ("Problem reading output of '"^lp_solver^"'. Probably no solution found.")



(* General Purpose Constraint Construction *)

let rec unify: rich_typ -> rich_typ -> (constr list) = (* Produce the necessary equalties. Mismatch in underlying types raises Invalid_argument *)
 fun r1 r2 -> match (r1,r2) with
  | (RUnitTyp, RUnitTyp)     -> [] 
  | (RBoolTyp, RBoolTyp)     -> []
  | (RIntTyp, RIntTyp)       -> []
  | (RFloatTyp, RFloatTyp)   -> []
  | (RCharTyp, RCharTyp)     -> []
  | (RStringTyp, RStringTyp) -> [] 
  | (RTvarTyp(tva), _)       -> notImplemented ("Unifying typevariables '"^tva^"'.")
  | (_, RTvarTyp(tvb))       -> notImplemented ("Unifying typevariables '"^tvb^"'.")
  | (RConTyp(rsa,ida,rtia), RConTyp(rsb,idb,rtib)) when ida = idb 
    -> let css = List.rev_map2 unify rsa rsb in
       let vss = conTab_fold2 (fun cstr a b acc -> list_append (unify_constructor_arguments a b) acc) rtia rtib [] in
       list_concat (vss::css)
  | (RArrowTyp(doa,raa), RArrowTyp(dob,rab)) -> list_append (unify dob doa) (unify raa rab)
  | (RSelfTyp(ida), RSelfTyp(idb)) when ida = idb -> [] 
  | _ -> cerr ("Arguments cannot be unified: \n   "^(print_rt r1)^"   AND   "^(print_rt r2))

and unify_constructor_arguments: rich_coninfo -> rich_coninfo -> (constr list) =
 fun rtia rtib ->
   if rtia.rorder = rtib.rorder 
   then
     let css = List.rev_map2 unify rtia.rarg_types rtib.rarg_types in
     (equal_l (!act_fkt^"_____"^"Uni") rtia.rcvar rtib.rcvar)::(list_concat css)
   else cerr "Mismatching Constructors while unifying. Code-Bug!"				    

(* Functions unify and subtype are almost identical -> merge into parameterized version ? *)

let rec subtype: rich_typ -> rich_typ -> (constr list) = (* Produce the necessary equalties. Mismatch in underlying types raises Invalid_argument *)
  (* subtype r1 r2 means that r1 is a subtype that shall fit in some bigger type - with respect to the resource annotations only *)
  fun r1 r2 -> 
    if not !the_global.opt.subtyping
    then unify r1 r2
    else
      match (r1,r2) with
      | (RUnitTyp, RUnitTyp)     
      | (RDiamantTyp, RDiamantTyp)      
      | (RBoolTyp, RBoolTyp)     
      | (RIntTyp, RIntTyp)       
      | (RFloatTyp, RFloatTyp)   
      | (RCharTyp, RCharTyp)     
      | (RStringTyp, RStringTyp) -> [] 
      | (RTvarTyp(tva), RTvarTyp(tvb)) when tva  = tvb  -> []
      | (RConTyp(rsa,ida,rtia), RConTyp(rsb,idb,rtib)) when ida = idb ->
	  let css = List.rev_map2 subtype rsa rsb in
	  let vss = conTab_fold2 (fun cstr a b acc -> list_append (subtype_constructor_arguments a b) acc) rtia rtib [] in
	  list_concat (vss::css)
      | (RArrowTyp(doa,raa), RArrowTyp(dob,rab)) -> list_append (subtype dob doa) (subtype raa rab)  (* Note: lhs is flipped  *)
      | (RSelfTyp(ida), RSelfTyp(idb)) when ida = idb -> [] 
      | _ -> cerr ("Subtyping error: \n   The type "^(print_rt r1)^" cannot be a subtype of "^(print_rt r2)^".")
	    
and subtype_constructor_arguments: rich_coninfo -> rich_coninfo -> (constr list) =
  fun rtia rtib ->
    if not !the_global.opt.subtyping (* sicherheitshalber *)
    then unify_constructor_arguments rtia rtib
    else
      if rtia.rorder = rtib.rorder 
      then
	let css = List.rev_map2 subtype rtia.rarg_types rtib.rarg_types in
	(* let sub = ("Sub_"^rtib.rcvar^"_"^rtia.rcvar) in (* DEBUG *) *)
	(geq_l (!act_fkt^"_____"^"Sub") (1, rtib.rcvar) [(1, rtia.rcvar)])::(List.flatten css)
      else cerr "Mismatching Constructors while subtyping. Code-Bug!"


let rec share: rich_typ -> (rich_typ * rich_typ * (constr list)) =
  function 
    | (RUnitTyp as x)
    | (RBoolTyp as x)
    | (RIntTyp  as x)
    | (RFloatTyp as x)
    | (RCharTyp  as x)
    | (RStringTyp  as x) 
    | ((RSelfTyp(_)) as x) -> (x,x,[])  (* Why is it correct to share self? Cant remember...?! *)
    | (RDiamantTyp as e)
    | ((RArrowTyp(_)) as e)   
    | ((RTvarTyp(_)) as e)   -> cerr ("Unable to share type '"^(print_rt e)^"'.")
    | RConTyp(rtpar,tid,argrt_tab) as x ->
	if !the_global.opt.sharing   (* Is sharing allowed at all? *)
	then
	  let (rtpara, rtparb, cnstrpar) = share_list rtpar in (* Not sure if we can/must really share the parameters? *)
	  let (argtaba, argtabb, cnstrarg) =
	    let f: constructor -> rich_coninfo -> (rt_contab * rt_contab * (constr list)) -> (rt_contab * rt_contab * (constr list)) =
	      fun con rci (tba, tbb, acc_cnstrs) ->
		let va = new_varname HDat in
		let vb = new_varname HDat in
		let sctr = 
		  (* let _ = print_string "Sharing...\n" in (*DEBUG*) *)
		  if !the_global.opt.sharestrict 
		  then equal0_l (!act_fkt^"_____"^"Shr") [(-1, rci.rcvar); (1,va); (1,vb)]
		  else geq_l    (!act_fkt^"_____"^"Shr")   (1, rci.rcvar) [(1,va); (1,vb)] 
		in 
		let (rtarga, rtargb, cnstrarg) = share_list rci.rarg_types in
		let rca = {rcvar = va; rorder = rci.rorder; rarg_types = rtarga; rsize = rci.rsize } in
		let rcb = {rcvar = vb; rorder = rci.rorder; rarg_types = rtargb; rsize = rci.rsize } in
		let tba' = ConTab.add con rca tba in
		let tbb' = ConTab.add con rcb tbb 
		in (tba',tbb', (list_append cnstrarg (sctr::acc_cnstrs)))
	    in  ConTab.fold f argrt_tab (ConTab.empty, ConTab.empty, [])
	  in 
	  (RConTyp(rtpara,tid,argtaba), RConTyp(rtparb,tid,argtabb), (list_append cnstrpar cnstrarg))
	else 
	  let _ = warning ("Variable-Sharing detected.") in
	  (x,x,[])

and share_list: (rich_typ list) -> ((rich_typ list) * (rich_typ list) * (constr list)) =
  fun rts ->
    let g = fun rt (ras,rbs,cs) -> 
      let (ra, rb, c) = share rt 
      in  (ra::ras, rb::rbs, (list_append c cs))
    in  List.fold_right g rts ([],[],[]) (* We must keep the order of rts, hence fold_right is required *)


let share_value_in: value -> rich_context -> (rich_context * rich_context * (constr list)) = 
  fun valu ctxt -> 
    match valu.v with
    | VarVal(var) ->
	let rt = 
	  try  Context.find var ctxt 
	  with Not_found -> cerrAt valu.i ("Identifier '"^var^"' not found in context. (Misplaced 0-arity Functioncall?)")
	in 
	let (rta, rtb, cshr) = share rt in
	let ctxt_r = Context.remove var ctxt in (* Vor 'bind' muss var entfernt werden, da klar ist das var in ctxt schon vorkommt! Direkt mit .add berschreiben ist sicherlich nicht so gut! *)
	let ctxta = bind ctxt_r var rta in 
	let ctxtb = bind ctxt_r var rtb in
	(ctxta, ctxtb, cshr)
    |  _  -> (ctxt, ctxt, [])


let rec split_by: ((variable list) * (variable list)) -> rich_context -> (rich_context * rich_context * (constr list)) = (* Share when necessary. The input lists may contain duplicates. *) 
  fun (varsa, varsb) ctxt ->
(*  (*DEBUGGING*)
    let _ = (print_string "\n\n ---> \n") in
    let _ = (List.iter (fun s -> print_string (s^"a> ")) varsa) in
    let _ = (print_string "\n") in 
    let _ = (List.iter (fun s -> print_string (s^"b> ")) varsb) in
    let _ = (print_string "\n";) in 
    let _ = (Context.iter (fun k rt -> print_string (k^"c< ")) ctxt) in
    let _ = (print_string "\n <--- \n") in
*)
    match (varsa, varsb) with
    | ([], [])      -> (Context.empty, Context.empty, [])
    | ([], (b::bs)) ->
	let b_rt = 
	  try  Context.find b ctxt 
	  with Not_found -> cerr ("Variable '"^b^"' not found in context!")
	in
	let bs' = list_remove b bs in  (* Not strictly necessary, but allows use of bind rather than Context.add *) 
	let (ctxta, ctxtb, cnstrs) = split_by ([], bs') ctxt in
	let ctxtb' = bind ctxtb b b_rt in
	(ctxta, ctxtb', cnstrs)
    | ((a::aas), bs) ->  (* Mind that "as" is a keyword *)
	let a_rt = 
	  try  Context.find a ctxt 
	  with Not_found -> cerr ("Variable '"^a^"' not found in context!")
	in
	let aas' = list_remove a aas in (* Necessary in else branch! Prohibits a later overwriting of this earlier sharing! Not necessary in then-branch, but allows use of bind rather than Context.add *) 
	if (not (List.mem a bs))
	then 
	  let (ctxta, ctxtb, cnstrs) = split_by (aas', bs) ctxt in
	  let ctxta' = bind ctxta a a_rt in
	  (ctxta', ctxtb, cnstrs)
	else
	  let bs' = list_remove a bs in 
	  let (ctxta, ctxtb, cnstrs) = split_by (aas',bs') ctxt in
	  let (a_rta, a_rtb, cshr) = share a_rt in (* Share checks the option and warns if necessary *)
	  let ctxta' = bind ctxta a a_rta in
	  let ctxtb' = bind ctxtb a a_rtb in
	  (ctxta', ctxtb', (list_append cshr cnstrs))


(* The mini TYPE inference --- we know that the program is well-typed, but we need to recover the type of a subterm sometimes *)
let rec infer_type: global -> rich_context -> expression -> (rich_typ * (variable list))= (* the resulting variable list may contain duplicates! *)
  fun gl ctxt expr -> 
    (*
       let _ = (*DEBUGGING HERE!*)
       print_string "\ninfer_type:";
       print_expr expr;
       print_newline
       in 
     *)
    match expr.v with
    | ValueExp(valu)                   -> 
	( match valu.v with     
        | VarVal(id) -> 
	    ( 
	      try  ((Context.find id ctxt),[id]) (* Catching a misparsed 0-arity function call again: *)
  	      with Not_found -> infer_type gl ctxt {i=expr.i; v=AppExp(id,[])} (* Can this ever happen? *)
	     )      
	| IntVal(i)     -> (RIntTyp, []) 
	| FloatVal(f)   -> (RFloatTyp, [])
	| CharVal(c)    -> (RCharTyp, []) 
	| StringVal(s)  -> (RStringTyp, [])
	| BoolVal(b)    -> (RBoolTyp, [])     
	| UnitVal       -> (RUnitTyp, [])
	| UnaryOpVal(uop,valu)       ->
	    ( 
	      let vars = collect_vars_valus [valu] in
	      match uop.v with
	      | NotOp     -> (RBoolTyp, vars)
	      | UMinusOp  -> (RIntTyp, vars)
	      | UFminusOp -> (RFloatTyp, vars)
	     )
	| BinaryOpVal(bop,valua,valub) ->
	    ( 
	      let vars = collect_vars_valus [valua; valub] in
	      match bop.v with
	      | TimesOp   
	      | DivOp      
	      | PlusOp     
	      | MinusOp   -> (RIntTyp, vars)
	      | FtimesOp   
	      | FdivOp     
	      | FplusOp    
	      | FminusOp  -> (RFloatTyp, vars)
	      | LessOp    
	      | LteqOp    
	      | GreaterOp 
	      | GteqOp    
	      | EqualOp   -> (RBoolTyp, vars)
	      | ConsOp    -> cerrAt bop.i "Not implemented: Cons-operator."
	      | AppendOp  -> (RStringTyp, vars)
	      | AndOp     
	      | OrOp      
	      | AndalsoOp 
	      | OrelseOp  -> (RBoolTyp, vars)
	      | ModOp     -> (RIntTyp, vars)
	     )
	 )
    | ConstrExp(constr, valus, dia)         -> 
	let coninfo = 
	  try  ConTab.find constr gl.ct
          with Not_found -> cerrAt expr.i ("Constructor identifier '"^constr^"' unknown.") 
	in  
	let d_var =
	  match dia with
	  | Reuse(d) -> [d]
	  | _ -> []
	in
	((enrich_type gl.td Nontoplevel coninfo.own_typ), (list_append d_var (collect_vars_valus valus)))
    | AppExp(id, valus)                ->
	let vars = (collect_vars_valus valus) in 
	if FunTab.mem id gl.vt 
	then 
	  let RValDec(fid,fdin,frt,fdout) = FunTab.find id gl.vt in
	  let (arg_rts, res_rt) = identify_args frt in
	  (res_rt, vars)
	else 
	  ( (* Must be a built-in function then... *)
	    let bif_rt = 
	      try rt_of_built_in_function id
	      with Invalid_argument s
		->  cerrAt expr.i ("Function identifier (or variable) '"^id^"' unknown! (while inferring a subterms type)")
	    in 
	    match bif_rt with
	    | RSelfTyp "built-in-function" ->
		let (rt, _ ) = infer_type gl ctxt {i=expr.i; v=ValueExp(get_last {i=expr.i; v=UnitVal} valus)} 
		in     (rt, vars)
	    | RSelfTyp unk -> bugAt expr.i ("Problems at inferring the rich type of built-in-function '"^id^"'. RSelfTag '"^unk^"' makes no sense.")
	    | _ -> (bif_rt, vars)
	   )
    | LetExp(letvar, expra, exprb)     -> 
	let (expart, vara) = infer_type gl ctxt expra in
	let (expbrt, varb) = infer_type gl (bind ctxt letvar expart) exprb in
	let vars = (list_append vara (list_remove letvar varb)) in (* letvar zaehlt hier nicht! *)
	(expbrt, vars) 
    | SeqExp(expra, exprb)             -> 
	let vara = collect_vars_expr expra in
	let (expbrt, varb) = infer_type gl ctxt exprb in
	let vars = (list_append vara varb) in 
	(expbrt, vars) 
    | IfExp(ifvalu, thenexpr, elseexpr) ->
	let varif = collect_vars_valus [ifvalu] in
	let (expart, varth) = infer_type gl ctxt thenexpr in
	let varel = collect_vars_expr elseexpr in
	let vars = (list_concat [varif; varth; varel]) in 
	(expart, vars) (* we know that the program is well typed, hence expart is enough! BUT: Could it be that (unify (infer_type thenexpr) (infer_type elseexpr) is not empty!???  Yes, but will be catched during inference... *)
    | MatchExp(mvar, mrules)           
  (*| MatchPrExp(mvar, mrules) *)       -> 
        let (rts, uvars) = infer_mrules gl ctxt mrules in
	let rt = 
	  (
	   match rts with
	   | x::xs -> x
	   | []    -> cerrAt expr.i "Match without rules encountered"
	  )
	in
	(rt, mvar::uvars) (*BUGFIX: "mvar::"*)
	  
and infer_mrules: global -> rich_context -> (matchrule list) -> ((rich_typ list)* (variable list)) =
  fun gl ctxt mrules ->
    let (rts, uvars) = List.split (List.map (infer_mrule gl ctxt) mrules) in
    let uvars' = (list_concat uvars) in
    (rts, uvars')
  
and infer_mrule: global -> rich_context -> matchrule -> (rich_typ * (variable list)) =
  fun gl ctxt (Matchrule(info, constr, vars, is_destr, expr, dia)) ->
    let coninfo = 
      try  ConTab.find constr gl.ct 
      with Not_found -> cerrAt info ("Constructor identifier '"^constr^"' unknown.") 
    in
    let zipargtyp: rich_context -> variable -> typ -> rich_context =
      (fun c v t -> bind c v (enrich_type gl.td Nontoplevel t))
    in
    let ctxt' = 
      try   List.fold_left2 zipargtyp ctxt vars coninfo.arg_types 
      with  Invalid_argument(_) -> errAt info ("Wrong number of arguments for '"^constr^"'-constructor.")
    in
    let (rt, uvars) = infer_type gl ctxt' expr in
    let vars' = match dia with Reuse(d) -> d::vars | _ -> vars in
    let uvars' = List.fold_right list_remove vars' uvars in
    (rt, uvars')


(* THE CONSTRAINT INFERRING FUNCTIONS: *)
(* iterm shall stand for "Infer constraints for TERM" *)

let runtime1 = ref 0.  (* for runtime measurement *)
let runtime2 = ref 0.  (* for runtime measurement *)
(*  MAIN - main - Main  *)
let rec iprogram: options -> program -> (out_channel * (string option)) -> unit =
 fun opts (Program(info, typdecs, valdecs, funblocks)) (outch,oname) ->
   let oname_string = (* Get string back from string option *)
     let s = print_some oname in
     if s = "" then "stdout" else s
   in
   let _ = runtime1 := Sys.time () in    (*** START TIME MEASUREMENT ***)
   let funt:funtab      = compile_fun (List.rev (List.flatten funblocks)) in   (* Function-Table          - all defined functions *)
   let rvdt:rvaltab     = compile_rvaldec typdecs valdecs in                   (* Enriched-ValDecl-Table  - all declared functions *)
   let cost:contab      = compile_contab typdecs in                            (* Constructor-User Cost Look-Up Table *)
   let gl:global        = { global_template with ft= funt; ct= cost; vt=rvdt; td=typdecs; opt=opts } in
   let _ = the_global := gl in (* Fuer alle die, dies sonst nicht bekommen. Global 'sollte' sich ab hier nicht mehr aendern! *)

   (* Building constraints *)
   let _ = if gl.opt.debug then print_string "\n\n Building constraints for function:" in
   let lp:(constr list) = (* The order of the constraints in the list does not matter *)
(*     try       (* general error-recovery !?? *)  *)
       let constr_defined_fun =
	 (* Constraints for all defined functions *)
	 (FunTab.fold (fun id fdef cs -> list_append (ifundef gl fdef) cs) funt [])                                    (*<-*) 
       in
       let constr_declared_fun =
	 (* Constraints for all declared functions *)
	 (FunTab.fold (fun id fdec cs -> list_append (ifundec gl fdec) cs) rvdt [])                                    
       in list_append constr_declared_fun constr_defined_fun
(*     with _ -> (* general error-recovery !?? *)
       let _ = flush stdout in
       let _ = print_string "Trying error-recovery to identify problem...\n" in
       []
*)
   in 

   let _ = runtime1 := (Sys.time ()) -. !runtime1 in   (*** END TIME MEASUREMENT ***)
   (* Print out some statistics here *)
   let _ = 
     if (lp <> [])
     then
       let len0 =
	 let pr_str0 = 
	   ("\n Resource constraints constructed")
	   ^(
	     if (!runtime1 >= 0.03 ) 
	     then (" in "^(print_normal_float !runtime1)^" seconds: ")
	     else (": ")
	    )
	 in
	 let _ = print_string pr_str0 in
         String.length pr_str0 
       in
       let _ =
	 if (not (opts.performance && opts.solvelp))
	 then (* This line is expensive: *)
	   (
	    let lp = 
	      if (opts.performance || opts.debug)   (*-debug shall also reveal all trivial inequalities *)
	      then lp 
	      else simplify lp (* lp is also simplified while being processed in constr2file / print_cnstr. But this line gets the statistics right. *)
	    in 
	    let len1 = 
	      let pr_str1 = ((string_of_int (List.length lp)) ^ " inequalties in "^(string_of_int (List.length(set_of_vars lp)))^" variables.") in
	      let _ = print_string pr_str1 in
	      String.length pr_str1
	    in
	    (* Print equalities to file here... *)
	    let _ = constr2file gl lp opts.integer_lp (outch,oname) in

	    let pr_str2 = (" Written to '"^(oname_string)^"'.\n\n") in 
	    let len2 = String.length pr_str2 in
	    let _ = if (!the_options.screen_width < (len0+len1+len2)) then print_newline () in
	    let _ = print_string pr_str2 in
	    ()
	   )
       in ()
   in   
(* Close print-out file in any case *)
   let _ = 
     try
       if outch <> stdout then
	 let _ = 
	   if !the_options.debug 
	   then print_string (" Closing file '"^(oname_string)^"'.\n") 
	 in close_out outch
     with _ -> ()   (*
		       This does not catch the error when closing stdout. 
		       Because the error is only raised when the next print command is called! 
		     *)
   in

   (* Call lp_solver via pipe...*)
   let _ = 
     if opts.solvelp && (lp <> [])
     then
       (* HWL: ignoring message for demo *)
       (* let _ = print_string (" Calling '"^lp_solver^"' via pipe... \n") in *)
       let _ = try flush stdout with _ -> () in (* Before we pipe around, lets see what happened so far... *)
       let _ = runtime2 := Sys.time () in
       let args = Array.make 2 "-S2" in  (* Achtung: Bei mehr als einem Argument reicht Array.make leider nicht mehr aus! *)
       let (lpout,lpin) = pipe2prog lp_solver args in 
       let  _  = constr2file  gl lp opts.integer_lp (lpout,oname) in  (* Write constraints to   lp_solver via pipe*)
       let  _  = try close_out lpout with _ -> bug "Closing lpout failed!" in
       let sol = file2solution gl lp lpin in                          (* Read  solution    from lp_solver via pipe*)
       let  _  = try close_in lpin with _ -> bug "Closing lpin failed!" in
       let _ = runtime2 := (Sys.time ()) -. !runtime2 in
       (* Verify that all constraints are indeed satisfied... *)
         (* ... *)
       let _ = 
	 (* Print enriched instantiated signature to screen... *)

(******** HERE ********)
	 if sol != Constr.empty 
	 then (* Print only if LP is feasible *)
	   let _ = print_string ("\n Solution from '"^lp_solver^"' yields the following enriched types:\n") in
	   let _ = FunTab.iter (fun id rv -> print_string (print_rvaltab_sol sol rv)) gl.vt in

	   let () = if !the_options.pickle (* kwxm *)
	   then
	     (* HWL was here: added dir to pickle file name *)
	     let pkl_out = open_out (!the_options.dir^"/valdecs.pkl") in
	     let _ = FunTab.iter (fun id rv -> pickle_rvaldec pkl_out sol rv) gl.vt in
	     close_out pkl_out 
	   else () in

	   let _ = print_string "\n" in 
	   let _ = 
	     List.iter (fun x -> print_string (print_constr_sol sol x)) lp 
	   in
	   let _ = 
	     (if (!runtime2 >= 0.03 ) 
	     then print_string (" Solution was found by '"^lp_solver^"' in "^(print_normal_float !runtime2)^" seconds.\n\n"))
	   in 
	   let _ = 
	     if not opts.performance then
	       let strict_cnstrs = filter_strict lp sol in
	       if strict_cnstrs <> [] then
		 let _ = print_string " Memory leak detected in the following branches of computation: \n" in
		 List.iter (fun c -> print_string ("  "^(print_constr c))) strict_cnstrs
(*	   in let _ = print_string "\n" *)
	   in ()
	 else (* RECOVERY! *)
	   let _ = print_string "\n\n     -----     LP FOR THE WHOLE PROGRAM IS INFEASIBLE !!!    -----   \n\n" in    
	   let _ = 
	     if not opts.performance
	     then
	       let _ = print_string ("\n Trying to solve constraints for each defined function individually. ") in
	       let _ = print_string ("\n   (This may take a while. Need to recompute discarded information) \n") in
	       let _ = print_string ("\n NOTE:  while some functions may produce feasible LPs on their own, plugging") in
	       let _ = print_string ("\n these functions together into a program may lead to an infeasible LP as well.\n\n") in
	       let proc_f: out_channel option -> funcidentifier -> fundef -> unit =
		 fun ocopt fid fdef ->
		   let flp = ifundef gl fdef in
		   if flp <> [] 
		   then 
		     let args = Array.make 2 "-S2" in  
                        (* Achtung: Bei mehr als einem Argument reicht Array.make leider nicht mehr aus! *)
		     let (flpout,flpin) = pipe2prog lp_solver args in
		     let  _  = constr2file  gl flp opts.integer_lp (flpout,None) in  
                         (* Write constraints to   lp_solver via pipe*)
		     let  _  = try close_out flpout with _ -> () in
		     let fsol = file2solution gl flp flpin in
                         (* Read  solution    from lp_solver via pipe*)
		     let  _  = try close_in flpin with _ -> ()  in
		     let  _  =
		       try
			 let rv = FunTab.find fid gl.vt in
			 let () = match ocopt with (* kwxm:  difficult to get stream into function *)
			   Some oc -> pickle_rvaldec oc fsol rv 
			 | None -> () in
			 print_string (print_rvaltab_sol fsol rv)
		       with 
			 Not_found -> print_string ("  Constraints for function '"^fid^"' cannot be checked.\n")
		     in ()
  	       (* fi *)
	       in let _ = if !the_options.pickle (* kwxm: if so,  we have to open the file, iterate, then close it *)
	          then let pkl_out = open_out "valdecs.pkl"
		  in let _ = FunTab.iter (proc_f (Some pkl_out)) funt
		  in close_out pkl_out 
	       else
		  FunTab.iter (proc_f None) funt

	       in print_string "\n\n     -----     LP FOR THE WHOLE PROGRAM IS INFEASIBLE !!!    -----   \n\n" 
	   in print_string " \n "
       in ()
   in ()

and ifundef: global -> fundef -> (constr list) =   (* Given a function, we construct all constraints of its defining body *)
  fun gl (FunctionDef(info, fid, vars, expr)) ->
    try
      let RValDec(_, din, rtyp, dout) = FunTab.find fid gl.vt 
      in
      let (arg_rts, res_rt) = identify_args rtyp in
      let ctxt = 
	try List.fold_left2 bind  Context.empty vars arg_rts 
	with Invalid_argument s -> cerrAt info ("Mismatch between type and number of arguments for function " ^ fid ^ ".")
      in  
      let _ = if gl.opt.debug then (print_string ("\n\n"^fid^" : "^(print_rt res_rt))) in    (* DEBUGGING HERE  *) 
      let _ = act_fkt := fid in
      iexpression gl ctxt din expr res_rt dout 
    with 
    | Not_found -> 
        (* HWL: ignore warning for demo version
	let _ = Support.Error.warningAt info ("No Val declaration for function '" ^ fid ^ "'. Ignoring function.") 
	in *) []
    | Invalid_argument s when (s="Arrays not implemented") ->
	let _ = Support.Error.warningAt info ("Function '" ^ fid ^ "' uses array types. Arrays are not implemented. Ignoring function.") 
	in []
       	
 
and ifundec: global -> rich_valdec -> (constr list) =   (* Constraints for undefined (maybe external) functions *)
  fun gl (RValDec(fid, fdin, frt, fdout)) ->
    if (FunTab.mem fid gl.ft)
    then (* Functions is also defined *)
      (* Constraints are derived via ifundef *)
      (* Maybe check for an override here *)
      []
    else (* Function is not defined, only declared *)
      (* Check if AnnValDec was present... *)
      if (is_cvar_constant fdin)
      then (* accept provided annotations *)
       [] (* Nothing to do here, since if values are used, the constr2file print the necessary equalities *)
      else (* set fdin = fdout and all rhs-annotations to zero *)
	let fid_zero = (fid^"_"^"Zero") in
	(geq_l fid_zero (1,fdin) [(1,fdout)])::(set_to_zero_l fid_zero frt)

(* Template for iterm:
   let iterm: global -> rich_context -> cvar -> expression -> rich_typ -> cvar -> (constr list)
   Idea: Context, Diamond-In-variable, the term, result type, Diamond-Out-variable.
*)

and iexpression: global -> rich_context -> cvar -> expression -> rich_typ -> cvar -> (constr list) =
  fun gl ctxt din expr rt dout ->
    let debug_info: string -> unit = 
      if gl.opt.debug 
      then fun s -> print_string ("\n" ^ (printInfoNF' expr.i) ^ " " ^ s ^ " ")
      else fun _ -> ()
    in
(*    print_string ("\n Debug: "^(print_rt rt));   (* DEBUGGING HERE*)   *)
(*   (print_string " Hi!";[])                      (* DEBUGGING HERE*)   *)
    match expr.v with
    | ValueExp(valu) -> 
	let _ = debug_info "VAL" in	  
	( match	valu.v with     (* Catch misparsed 0-arity function calls here: *)
           | VarVal(id) when not (Context.mem id ctxt)   (* Not necessarily true that (FunTab.mem id ft) holds, as it could be a built-in function (currently not inside ft) *)
               -> iexpression gl ctxt din {i=expr.i; v=AppExp(id,[])} rt dout 
           | _ -> (geq_l (!act_fkt^(linetag expr.i)^"Val") (1,din) [(1,dout)])::(ivalue gl ctxt valu rt)
	)
    | ConstrExp(constr, valus, _ ) ->  (* We could check that the diamond argument is in the context, but then again, we assume that the program type-checks anyway *)
	let _ = debug_info "CONSTR" in	  
	(
	 let cinfo = 
	   try  ConTab.find constr gl.ct 
	   with Not_found -> cerrAt expr.i ("Constructor identifier '"^constr^"' unknown.")
	 in
	 
	 match (unfold rt) with
	   | RConTyp(pararts, tid, argrt_tab) when tid = cinfo.typid
	     -> 
	       let _ = pos_cerrAt pararts expr.i "Parameterized types not supported yet." in      (* ToDo: what shall we do with pararts, once we accept type-variables? *)

	       (* Eigentlich reicht hier auch rcinfo.rsize, aber csize_to_num gabs schon. Vielleicht Datenabgleich durchfhren? -> Oder Redundanz in Datenstrukturen vermeiden! *)
	       let rcinfo = ConTab.find constr argrt_tab in

(*	       let   = rcinfo.rsize (* of type constructor_size *) *)
	       let csiz = csize_to_num cinfo in

	       let arg_constrs = iargs gl ctxt expr.i valus rcinfo.rarg_types in
	       let base_constr = geq_l (!act_fkt^(linetag expr.i)^"Con") (1,din) [(csiz, cv_one); (1, rcinfo.rcvar); (1, dout)] in
	       base_constr :: arg_constrs   (* No unificaton with rt needed, as we do not know any other type *)
	   | RConTyp(_, tid, _) when not(tid = cinfo.typid) -> 
	          cerrAt expr.i ("Type mismatch: Constructor-expression with type '"^tid^"' encountered, but type '"^cinfo.typid^"' is required.")
	   | _ -> cerrAt expr.i ("Type mismatch: Constructor-expression with non-constructor-type encountered (type '"^cinfo.typid^"' required).")
	)
    | AppExp(id, valus) -> 
	let _ = debug_info "APP" in	  
	if FunTab.mem id gl.vt 
	then 
	  let RValDec(fid,fdin,frt,fdout) = FunTab.find id gl.vt in
	  let (arg_rts, res_rt) = identify_args frt in
	  let arg_cnstrs = iargs gl ctxt expr.i valus arg_rts in (* iargs takes care of possible sharing *)
	  let uni_cnstrs = subtype rt res_rt in (* VERIFY the order of arguments to subtype! It might be WRONG! *)
	  let cal_cnstrs = [(geq_l (!act_fkt^(linetag expr.i)^"Ap0") (1,din) [(1,fdin)]); (geq_l (!act_fkt^(linetag expr.i)^"App") (1,din) [(1,fdin); (-1,fdout); (1,dout)])] in
	  list_concat [cal_cnstrs; uni_cnstrs; arg_cnstrs]
	else 
	  ( (* Must be a built-in function then... *)
	    if (is_built_in_function id) 
	    then 
	      (* din = dout *)
	      let cal_cnstrs = [(geq_l (!act_fkt^(linetag expr.i)^"BiF") (1,din) [(1,dout)])] in 
	      (* annotations in type: *)
	      match rt_of_built_in_function id with
	      | RSelfTyp(tid) -> (* POLYMORPHIC: Result is always the same type as its last argument *)  
		  let lval = get_last {i=UNKNOWN; v=UnitVal} valus in
		  let (lval_rt,_) = infer_type gl ctxt {i=expr.i; v=ValueExp(lval)} in
		  list_append (subtype rt lval_rt) cal_cnstrs (* Subtype is verified *)
	      | RConTyp(parart, tid, rtc) -> (* Currently no built-in exists *)
		  errAt expr.i "Unexpected type of built-in function encountered."
	      | RArrowTyp(rta,rtb) ->        (* Currently no built-in exists *)
		  errAt expr.i "Unexpected type of built-in function encountered."
	      | RTvarTyp(tv) ->              (* Currently no built-in exists *)
		  errAt expr.i "Unexpected type of built-in function encountered."
	      | this_rt -> list_append (subtype rt this_rt) cal_cnstrs 
	    else  cerrAt expr.i ("Function identifier (or variable) '"^id^"' unknown.")
	   )
    | LetExp(letvar, expra, exprb) -> 
	let _ =
	  if gl.opt.debug 
	  then (*DEBUGGING HERE!!!!*)
	    (
	     (debug_info "LET");
	     (print_string (" Let-Var: "^letvar));
	     (print_string ";  Ctxt: ");
	     (print_string (String.concat "," (Context.fold (fun v rt acc -> v::acc) ctxt [])))
	    )
	in
	let daux = new_varname Aux in 
        let (rt_letvar, uvarsa)  = infer_type gl ctxt expra in
        let uvarsb  = collect_vars_expr exprb in
	let uvarsb' = list_remove letvar uvarsb in
	let _ = 
	  if gl.opt.debug 
	  then (*DEBUGGING HERE!!!!*)
	    (
	     (print_string ";  Let-Ctxt: ");
	     (print_string (String.concat "," uvarsa));
	     (print_string ";  In-Ctxt: ");	    
	     (print_string (String.concat "," uvarsb'));
	     (print_string ";")
	    )
	in
	let (ctxta, ctxtb, cshr) = split_by (uvarsa, uvarsb') ctxt in  (* Splitting means sharing! *)
	let ctxtb' = bind ctxtb letvar rt_letvar in 
	list_concat [cshr;
		     (iexpression gl ctxta  din  expra rt_letvar daux); (* Wichtig: Der inferierte Typ kann noch zu grob fr den Term sein! *) 
		     (iexpression gl ctxtb' daux exprb rt        dout)
		   ]
    | SeqExp(expra, exprb) -> 
	let _ = debug_info "SEQ" in	  
	let daux = new_varname Aux in 
        let (rt_letvar, uvarsa)  = infer_type gl ctxt expra in
        let uvarsb  = collect_vars_expr exprb in
	let _ = 
	  if gl.opt.debug 
	  then (*DEBUGGING HERE!!!!*)
	    (
	     (print_string " Ctxt: ");
	     (print_string (String.concat ", " (Context.fold (fun v rt acc -> v::acc) ctxt [])));
	     (print_string ";  Let-Ctxt: ");
	     (print_string (String.concat ", " uvarsa));
	     (print_string ";  In-Ctxt: ");	    
	     (print_string (String.concat ", " uvarsb));
	     (print_string ";")
	    )
	in
	let (ctxta, ctxtb, cshr) = split_by (uvarsa, uvarsb) ctxt in  (* Splitting means sharing! *)
	list_concat [cshr;
		     (iexpression gl ctxta  din  expra rt_letvar daux); (* Wichtig: Der inferierte Typ kann noch zu grob fr den Term sein! *) 
		     (iexpression gl ctxtb  daux exprb rt        dout)
		   ]
    | IfExp(ifvalu, thenexpr, elseexpr) ->
	let _ = debug_info "IF" in	  
	list_concat 
	  [
	   (ivalue gl ctxt ifvalu RBoolTyp);
	(* NO need to remove ifvar from context: booleans (and basic values) may be shared! *)
	(* let ctxt' = Context.remove ifvar ctxt in  *)
	   (iexpression gl ctxt din thenexpr rt dout) ;
           (iexpression gl ctxt din elseexpr rt dout)
	 ]
    | (MatchExp(  mvar, mrules) as empr) ->
	let _ = debug_info "MATCH" in	  
	let mvrt = 
	  try  (Context.find mvar ctxt)  (* do not unfold here *)
          with Not_found -> cerrAt expr.i ("Variable '"^mvar^"' not found in context.")
	in 
	List.fold_left (fun acc rul -> list_append (imrule gl ctxt din (expr.i, mvar, mvrt, rul) rt dout) acc) [] mrules

 (*** 
    Old Code, when match/match' was in place. Now changed to Camelto-Style via Constructor@_ in each case 
    | (MatchPrExp(mvar, mrules) as empr)  -> 
	let pr = (* This saves some code-redundancy *)
	  if (match empr with MatchPrExp(_) -> true | _ -> false) 
	  then 
	    if gl.opt.sharing 
	    then 
	      true
	    else 
	      let _ = warningAt expr.i "Non-destructive match encountered. Treated as destructive, since sharing is prohibited." in
	      false
	  else 
	    false
	in 
	let mvrt = try  Context.find mvar ctxt
	           with Not_found -> cerrAt expr.i ("Variable '"^mvar^"' not found in context.")
	in 
	let (ctxt', mvrt', cnstr_shr) = 
	  if 
	    pr (* No sharing necessary: mvar is destroyed! *) 
	      && 
	    (List.mem mvar (collect_vars_mrules mrules))
	  then  (* Share mvar: *)
	    let (mvrta, mvrtb, cshr) = share mvrt in
	    ((change_bind ctxt mvar mvrtb), mvrta, cshr)
	  else  (* mvar is destroyed or not used in mrules *) 
	    ((Context.remove mvar ctxt),    mvrt,  [])
	in List.fold_left (fun acc rul -> list_append (imrule gl ctxt' din (pr,expr.i, mvar, mvrt', rul) rt dout) acc) cnstr_shr mrules
***)

and imrule: global -> rich_context -> cvar -> (info * variable * rich_typ * matchrule) -> rich_typ -> cvar -> (constr list) = (* If bool = true then MatchPrime, else Match *)
  fun gl ctxt din (info, mvar, mvrt, rule) rt dout ->
    let Matchrule(minfo, constr, argvars, pr, expr,dia) = rule in
    let ctxt_d = 
      match dia with 
      | Reuse(d) -> bind ctxt d RDiamantTyp
      |  _       -> ctxt
    in
    let (pr', ctxt', mvrt', cnstr_shr) = (* Sharing must be decided before mvrt is used *)
      if pr && gl.opt.sharing  
      then (* mvar is not destroyed, hence we might need to share mvar again *)
        if (List.mem mvar (collect_vars_mrules [rule])) (* Using collect_vars_mrules rather than collect_vars_expr removes also all argvars from list *)
        then (* Share mvar *)
          let (mvrta, mvrtb, cshr) = share mvrt in
          (true, (change_bind ctxt_d mvar mvrtb), mvrta, cshr)
	else (* No sharing necessary, as mvar does not occur later in this branch *)
	  (true, ctxt_d, mvrt, [])
      else (* mvar is destroyed, thus remove mvar from ctxt *)
	let _ = (if pr then warningAt info "Non-destructive match encountered. Treated as destructive, since sharing is prohibited.") in
	(false, (Context.remove mvar ctxt_d), mvrt, [])
    in
    match (unfold mvrt') with 
    | RConTyp((p::ps as pararts), tid, argrt_tab) 
      -> cerrAt info "Parameterized types not supported yet." 
	  (* ToDo: what shall we do with pararts, once we accept type-variables? *)
    | RConTyp([], tid, argrt_tab) 
      -> 
	let cinfo = 
	  try  ConTab.find constr gl.ct  (* Not really needed, but we still need cinfo with csize_to_num, although one should change csize_to_num in order to deal with rich_coninfo's *)
          with Not_found -> cerrAt expr.i ("Constructor identifier '"^constr^"' unknown.")
	in
	let csiz = csize_to_num cinfo in (* Eigentlich reicht hier auch rcinfo.rsize, aber csize_to_num gabs schon. Vielleicht Datenabgleich durchfhren? -> Oder Redundanz in Datenstrukturen vermeiden! *)
	let _ = 
	  if cinfo.typid <> tid 
	  then cerrAt minfo ("Constructor '"^constr^"' does not fit type of match-variable '"^mvar^"'.") in (* Not really needed, as also checked indirectly below: *)
	let rcti = 
	  try  ConTab.find constr argrt_tab
	  with Not_found -> cerrAt minfo ("Constructor '"^constr^"' does not fit type of match-variable '"^mvar^"'.")
	in  
	let daux = new_varname Aux in
	let ord  = string_of_int (rcti.rorder) in
	let base_constr = 
	  if pr' (* Destructive or not? *)
	  then geq_l (!act_fkt^(linetag minfo)^"M'"^ord) (1,din) [                (-1,rcti.rcvar); (1,daux)] (* Non-destructive *)
	  else geq_l (!act_fkt^(linetag minfo)^"Ma"^ord) (1,din) [(-csiz,cv_one); (-1,rcti.rcvar); (1,daux)] 
	in
	let ctxt'' = 
	  try List.fold_left2 bind ctxt' argvars rcti.rarg_types 
	  with  Invalid_argument(_) -> errAt minfo ("Wrong number of arguments for '"^constr^"'-constructor.")
	in
	let expr_constrs = iexpression gl ctxt'' daux expr rt dout in
	list_append cnstr_shr (base_constr :: expr_constrs)
    | _ -> cerrAt info ("Type mismatch: Match-Variable "^mvar^" with non-constructor-type encountered.")   (* This is awkward! Cant we just restrict to subtypes? *)

and iargs: global -> rich_context -> info -> (value list) -> (rich_typ list) -> (constr list) =
  fun gl ctxt i vall rtl ->
    match (vall, rtl) with
    | ([], [])           -> []
    | (v::vals, rt::rts) -> 
	let (ctxta, ctxtb, cshr) = 
	  if is_shared v vals 
	  then
	    if gl.opt.sharing 
	    then
	      (* We cannot use 'split' here once, as variables could be shared multiple times here... *)
	      share_value_in v ctxt (* All variables are shared, other values wont do anything... *)
	    else 
	      (* Warnung wird normalerweise in share ausgegeben, aber hier haben wir ein info zur Verfuegung... *)
	      let _ = warningAt v.i ("Sharing of "^(print_val v)^" detected.") in
	      (ctxt, ctxt, [])
	  else
	    (ctxt, ctxt, [])
	in
	let vconstr = ivalue gl ctxta v rt in
	let rconstr = iargs gl ctxtb i vals rts  in
	list_concat [cshr; vconstr; rconstr]
    |    _     -> cerrAt i "Mismatching number of arguments for function call or constructor application."

(* IVALUE:
    - Es knnen nur Constraints durch unifizierung mit dem Ergebnistyp entstehen
       -> din, dout nicht notwendig dazu!
       -> Operatoren brauchen nicht betrachtet zu werden, da der 'soll'-Typ der Argumente eh unbekannt ist.
and ivalue: global -> rich_context -> cvar -> value -> rich_typ -> cvar -> (constr list) = (* DEPRECATED, use ivalue_aux instead *)
  fun gl ctxt din value rt dout -> (geq_l (!act_fkt^"_"^"Val") (1,din) [(1,dout)]::(ivalue_aux gl ctxt value rt)
  (* Es kann auch sein, das hier nochmal 0-Arity Function-Calls abgefangen werden mssen, dann braucht man din,dout! *)						       
*)

and ivalue: global -> rich_context -> value -> rich_typ -> (constr list) =
  fun gl ctxt value rt ->
    match value.v with
    | VarVal(v) -> 
	let crt = try  Context.find v ctxt 
                  with Not_found -> cerrAt value.i ("Identifier '"^v^"' not found in context. (Misplaced 0-arity Functioncall?)")
	in  subtype rt crt
    | _ -> []  (* We do not check the types here, 
		  as all built in ops operate only on unannotated types,
		  hence cannot lead to unfication/subtyping.
		*)





