(* 

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: common.ml,v $
   Id:       $Id: common.ml,v 1.9 2004/01/21 14:58:38 sjost Exp $ 

	

   What this File is all about:
   ----------------------------
   This file captures commonly used functions and definitions or general functions,
   it _shall_ not relate to relate to anything specific for lfd_infer.
   
   Sometimes we add/replace functions of the OCaml library, e.g. 
   like "List.append" which is 'replace' by "list_append".
   This is useful for changing them easily later, like
   using List.rev_append instead for performace issues, etc.

*)

(* ToDos / ToReconsiders: 

    - rewrite functions tagged with 'rewrite efficiently', eg. 'list_remove_last'

*)


(* UNIX macht Spass *)
let pipe2prog: string -> (string array) -> (out_channel * in_channel) =
  fun prog args ->
    let (  toext_read,   toext_write) = Unix.pipe () in (* Pipe from Parent zum Child *)
    let (fromext_read, fromext_write) = Unix.pipe () in (* Pipe from Child zum Parent *)
    if (Unix.fork ()) <> 0                           (* Jetzt bin ich zweimal da! *)
    then (* Parent Process *)
      (
       Unix.close    toext_read;       (* Der Vater hoert sich nicht selbst zu *)
       Unix.close  fromext_write;      (* Der Vater spricht sich nicht selbst an *)
       let  inch = Unix.in_channel_of_descr  fromext_read  in
       let outch = Unix.out_channel_of_descr   toext_write in
       set_binary_mode_in   inch false;
       set_binary_mode_out outch false;
       (outch, inch)
      )
    else (* Child Process *)
      (
       Unix.dup2   toext_read Unix.stdin;   (* Das Kind soll dem Vater ueber stdin zuhoeren *)
       Unix.close  toext_read;
       Unix.close  toext_write;               (* Das Child soll nicht zu sich reden *)
       Unix.dup2   fromext_write Unix.stdout; (* Das Kind soll dem Vater ueber stdout ansprechen *)
       Unix.close  fromext_write;
       Unix.close  fromext_read;            (* Das Child soll sich nicht selbst zuhoeren *)
       try 
	 Unix.execvp prog args;
	 (* This line is never reached *)
	 assert false
       with _ -> raise (Sys_error ("Calling the external program '"^prog^"' failed."))
      )


(* General functions *)

let pr_snd: 'a -> 'b -> 'b = (* Curried second projection *)
  fun a b -> b

let compose: ('b -> 'c) -> ('a -> 'b) -> ('a -> 'c) =
  fun f g x -> f(g(x))

let flip: ('a -> 'b -> 'c) -> 'b -> 'a -> 'c =
  fun f x y -> f y x


let save_increment: int -> int = (* Increments and checks for overflow *)
  fun x -> 
    if x >= x+1 
    then
      failwith "Integer overflow detected!"  
    else x+1

let save_ref_increment: int ref -> int = (* Increments references and checks for overflow *)
  fun x ->
    let old_val = !x in
    let _ = x := !x + 1 in
    if old_val >= !x 
    then 
      failwith "Integer overflow detected!"  
    else !x

  


(* General functions on lists *)
let mapfold: (('a * 'b) -> ('a * 'c)) -> ('a * ('b list)) -> ('a * ('c list)) =
  fun f (ini,l) ->
    let (res,l') =
      List.fold_left (fun (a,cs) b -> let (a',c) = f (a,b) in (a',c::cs)) (ini,[]) l
    in (res, List.rev l')
      
let mapfold': (('a * 'b) -> ('a * 'c)) -> ('a * ('b list)) -> ('a * ('c list)) =
  fun f (ini,l) ->
    List.fold_right (fun b (a,cs) -> let (a',c) = f (a,b) in (a',c::cs)) l (ini,[])  
      
      (* Example usage:
	 # mapfold  (fun (a,b) -> (a+1,(a,b))) (1,[1;2;3]);;
	 - : int * (int * int) list = (4, [(1, 1); (2, 2); (3, 3)])
	 # mapfold' (fun (a,b) -> (a+1,(a,b))) (1,[1;2;3]);;
	 - : int * (int * int) list = (4, [(3, 1); (2, 2); (1, 3)])
       *)

let rec list_snoc: 'a list -> 'a -> 'a list = (* Append an element at the end of a list *)
  fun l e ->
    match l with
    | []    -> [e]
    | x::xs -> x::(list_snoc xs e)

let rec list_append: 'a list -> 'a list -> 'a list = (* List.rev_append *) (* Beware: Lists get reversed, hence verifiy that it is only applied to constr lists, where the order does not matter! *)
   fun xs1 xs2 ->
    match xs1 with
    | []       -> xs2
    | x::xs1'  -> list_append xs1' (x::xs2) 


let rec list_concat: (('a list) list) -> 'a list = (* List.concat *)
   fun xss -> list_concat_aux [] xss 
       
and list_concat_aux: 'a list -> (('a list) list) -> 'a list =
  fun acc xss ->
    (
     match xss with
     | [] -> acc
     | x::xs -> 
	 let acc' = list_append x acc in
	 list_concat_aux acc' xs
    )


let list_concat_map: ('a -> 'b list) -> 'a list -> 'b list =
  fun f -> List.fold_left (fun acc x -> List.append acc (f x)) [] 
 (* Is this more efficient? It seems that inlining the code below  is more efficient *)
(* fun f l -> List.concat (List.map f l) *)



let list_remove: 'a -> 'a list -> 'a list = (* Remove _all_ occurences from list *)
  fun e -> List.filter ((<>) e) 

let rec list_remove_once: 'a -> 'a list -> 'a list = (* Removes given element exactly once *)
  fun a l ->
    match l with
    | []     -> raise Not_found
    | x::xs  -> 
	if a = x 
	then xs
	else x::(list_remove_once a xs)

let rec list_nub: 'a list -> 'a list = (* Remove all duplicates from a list - Very Expensive ! *)
  fun xs -> list_nub_aux [] xs 
and list_nub_aux: 'a list -> 'a list -> 'a list = 
  fun acc l ->
    match l with
    | []     -> acc
    | x::xs  -> 
	if   (List.mem x acc)
	then list_nub_aux acc xs
	else list_nub_aux (x::acc) xs

(* The one BELOW IS quite INEFFICIENT, but preserves order..
let rec list_nub: 'a list -> 'a list = (* Remove all duplicates from a list *)
  function 
    | []     -> []
    | x::xs  -> x :: (list_nub (list_remove x xs))
*)

		      
(* Same problem as in GET_LAST here: eta-expansion is required! *)
(*
let list_diff: 'a list -> 'a list -> 'a list = 
  List.fold_left (fun xs y -> list_remove y xs) 
*)

let rec list_diff: 'a list -> 'a list -> 'a list = (* List-Difference, always removes all occurences *)
  function 
    | []    -> fun id -> []
    | x::xs -> fun ys -> 
	if   List.mem x ys
	then     list_diff xs ys
	else x::(list_diff xs ys)

let list_intersection: 'a list -> 'a list -> 'a list =
  fun xs -> List.filter (fun y -> List.mem y xs)

let list_splitat: 'a list -> int -> ('a list * 'a list) = (* Rewrite more efficently or using fold *)
  fun l i ->
    let rec lsplit_aux: 'a list -> 'a list -> int -> ('a list * 'a list) =
      fun acc l1 i1 ->
	if i1 = 0 
	then (acc, l1)
	else lsplit_aux ((List.hd l1)::acc) (List.tl l1) (i1-1) 
    in try
    let (f,s) = lsplit_aux [] l i in
    ((List.rev f),s)
    with (Failure _) -> raise (Invalid_argument "list_splitat")
	
let rec list_unordered_equal: 'a list -> 'a list -> bool = (* Compares list like _unordered_ sets -- propably inefficient using list_remove_once! *)
  fun la lb ->
    match (la,lb) with
    | ([],[])        -> true
    | (x::xs, y::ys) ->
	if x=y 
	then (list_unordered_equal xs ys)
	else 
	  (
 	   try  list_unordered_equal xs (list_remove_once x (y::ys))
	   with Not_found -> false
	  )
    | _              -> false     
	  
let sum: int list -> int =
  List.fold_left (+) 0

let list_remove_last: 'a list -> 'a list = (* Rewrite efficently! *)
  fun a -> List.rev(List.tl(List.rev a))

let get_last: 'a -> 'a list -> 'a = (* Get last element if it exists, otherwise use default... *)
    fun a b -> (List.fold_left pr_snd a b) (* Why is eta-reduction prohibited here ?*) 
(*  fun a   ->  List.fold_left pr_snd a  (* This works... *) *)
(*             (List.fold_left pr_snd)   (* This doesnt work: *)
					    The type of this expression, '_a -> '_a list -> '_a,
					    contains type variables that cannot be generalized
                                          *) 

let list_orderedfind: ('a -> bool) -> 'a list -> ('a * int) =  (* Yields searched element and its place within the list. First element has address 1! *)
  fun p l ->
    let rec lof_aux: ( int * 'a list ) -> ('a * int) = 
      function 
	| (_,[])     -> raise Not_found
	| (i, x::xs) -> 
	    if   p x 
	    then (x,i)
	    else lof_aux ((i+1), xs)
    in lof_aux (1,l) 

(* Strings and Things (aka Files) *)
(* I hate it that ocaml doesnt treat a string as a 'char list'. This simply produces duplicated code parts! *)

let string_of_char: char -> string = String.make 1
    
let string_endswith: string -> string -> bool =
fun mas sub ->
  let n_mas = String.length mas in
  let n_sub = String.length sub in
  try (String.sub mas (n_mas - n_sub) n_sub) = sub 
  with Invalid_argument("String.sub") -> false      (* z.b. Knnte n_mas < n_sub sein *)

let string_drop: int -> string -> string =  (* Drops i characters from left *)
  fun i s ->
    let n = String.length s in
    try String.sub s i (n-i)
    with Invalid_argument("String.sub") -> raise (Invalid_argument "string_drop")

let string_rdrop: int -> string -> string = (* Drops i characters from right *)
  fun i s ->
    let n = String.length s in
    try String.sub s 0 (n-i)
    with Invalid_argument("String.sub") -> raise (Invalid_argument "string_drop")

let string_drop_until: char -> string -> string =
  fun c s ->
    try
      let p = String.index s c in
      string_drop p s
    with
      Not_found -> s (* Char does not occur, hence string is not altered *)

let string_split: string -> int -> (string * string) =
  fun s i ->
    let n   = String.length s in
    let fst = String.sub s 0 i in
    let snd = String.sub s i (n-i) in
    (fst,snd)

let string_chop: string -> string list = (* Returns the a list of strings of length 1  --- should produce a char list, but string list is handy right now *)
  fun s ->
    let slen = (String.length s) - 1 in
    let rec chop_aux: string list -> int -> string list =
      fun acc d ->
	if (d > slen)
	then List.rev acc
	else chop_aux ((string_of_char (String.get s d))::acc) (d+1)
    in chop_aux [] 0

let string_replace: string -> char -> char -> string = (* Returns a copy of the input string with _all_ occurences of a character replaces by another character. *)
  fun s o n ->
    let s' = String.copy s in (* No side-effects! *)
    let _  =
      for i = 0 to (String.length s' - 1) do
	if o = (String.get s' i)
	then String.set s' i n
      done
    in s'

let string_captured: string -> string = (* Inverse to String.escaped, except for numeral characters (written like \123) *)
  fun s ->
    let out = ref "" in
    let last_c = ref false in
    let add_char: char -> unit =
      fun c -> out := (!out^(string_of_char c))
    in
    let cap_aux: char -> unit =
      fun act_c ->
	if !last_c 
	then 
	  begin
	    last_c := false;
	    match act_c with
	    | '\\' 
	    | '\"' -> add_char act_c 
	 (* | '\'' does not occur as escaped character in strings *)
	 (* | '0'    These number probably dont work correctly yet...
	    | '1'
	    | '2'
	    | '3'
	    | '4'
	    | '5'
	    | '6'
	    | '7'
	    | '8'
	    | '9'  -> add_char act_c *)
	    | 'n'  -> add_char '\n'
	    | 'r'  -> add_char '\r'
	    | 't'  -> add_char '\t'
	    | 'b'  -> add_char '\b'
	    | _    -> begin add_char '\\'; add_char act_c end
	  end
	else
	  match act_c with
	  | '\\'  -> last_c := true
	  | _     -> add_char act_c
    in
    let _ = String.iter cap_aux s in
    !out
      
let string_intersperse: string -> int -> string -> string = (* Every i chars, insert the third argument *)
  fun s i sep ->
    let rec inters_aux: string list -> string -> string list =
      fun acc rs ->
	if ((String.length rs) > i)
	then 
	  let (h,t) = string_split rs i in
	  inters_aux (h::acc) t
	else 
	  List.rev (rs::acc)
    in String.concat sep (inters_aux [] s)

let string_break: string -> int -> char -> string -> string = (* Trys to break 'the_s' into chunks with at most 'width' parts, breaking only before each 'break_c' character, inserting 'sep_s' as the break *)
  fun the_s width break_c sep_s ->
    let rec br_aux: string list -> string -> string list =
      fun acc rs ->
	if ((String.length rs) > width)
	then
	  try
	    let indx = (String.rindex_from rs width break_c) in
	    let (fst, snd) = string_split rs indx in
	    if indx <> 0 
	    then br_aux (fst::acc) snd
	    else raise Not_found
	  with Not_found -> (* No suitable breakpoint found *)
	    begin 
	      try
		let indx = (String.index_from rs width break_c) in
		let (fst, snd) = string_split rs indx in
		if indx <> 0 
		then br_aux (fst::acc) snd
		else raise Not_found
	      with Not_found -> (* No breakpoint at all *)
		List.rev (rs::acc)
	    end
	else List.rev (rs::acc)
    in String.concat sep_s (br_aux [] the_s)

let filename_has_suffix: string -> bool =
  fun s -> 
    try  String.rcontains_from s ((String.length s) - 2) '.'
    with _ -> false

let print_some: string option -> string = (* Prints names if any... *) 
  function
    | None   -> ""
    | Some s -> s


(* Some printing functions for printing aligned integers:
   - print_pretty:    always signed,
   - print_aligned:   only negative integers singned 
   - primed-versions just return a string,
   - unprimed-versions print directly via print_string
*)
let print_pretty_int': int -> string =
  function
    | x when x < (-9) -> (string_of_int x)
    | x when x <   0  -> (" "^(string_of_int x))
    | x when x <  10  -> (" +"^(string_of_int x))
    | x               -> ("+"^(string_of_int x))

let print_pretty_int_ext': int -> string =
  function
    | x when x < (-99) -> (string_of_int x)
    | x when x <  (-9) -> (" "^(string_of_int x))
    | x when x <    0  -> ("  "^(string_of_int x))
    | x when x <   10  -> ("  +"^(string_of_int x))
    | x when x <  100  -> (" +"^(string_of_int x))
    | x                -> ("+"^(string_of_int x))

let print_aligned_int': int -> string =
  function
    | x when x < (-9) -> (string_of_int x)
    | x when x <   0  -> (" "^(string_of_int x))
    | x when x <  10  -> ("  "^(string_of_int x))
    | x when x < 100  -> (" "^(string_of_int x))
    | x               -> (string_of_int x)

let print_aligned_int_ext': int -> string =
  function
    | x when x < (-99) -> (string_of_int x)
    | x when x <  (-9) -> (" "^(string_of_int x))
    | x when x <    0  -> ("  "^(string_of_int x))
    | x when x <   10  -> ("   "^(string_of_int x))
    | x when x <  100  -> ("  "^(string_of_int x))
    | x when x < 1000  -> (" "^(string_of_int x))
    | x                -> (string_of_int x)
    

let lined_aligned_int: int -> string =
  function
    | x when x <  10  -> ("__"^(string_of_int x))
    | x when x < 100  -> ("_"^(string_of_int x))
    | x               -> (string_of_int x)

let print_pretty_int: int -> unit =
  compose print_string print_pretty_int'

let print_pretty_int_ext: int -> unit =
  compose print_string print_pretty_int_ext'

let print_aligned_int: int -> unit =
  compose print_string print_aligned_int'

let print_aligned_int_ext: int -> unit =
  compose print_string print_aligned_int_ext'

let print_normal_float: float -> string =
  fun f ->
    let raw = 
      if (f = 0.0) (* to prevent "-0" print-outs *)
      then "0"
      else string_of_float f 
    in
    let n   = (String.length raw) - 1 in
    if   (String.get raw n) = '.' 
    then String.sub raw 0 n
    else raw

let rec print_pretty_int_list: int list -> string = (* Printing integer lists... *)
  function 
    | [] -> "[]"
    | h::t -> ("["^(string_of_int h)^(print_pretty_int_list_aux t))
and print_pretty_int_list_aux: int list -> string =
  function
    | []   -> "]"
    | h::t -> (","^(string_of_int h)^(print_pretty_int_list_aux t))


let print_pretty_float : float -> string = (* removes a trailing '.' *)
  fun f ->
    let o = string_of_float f in
    let l = String.get o (String.length o - 1) in
    if l = '.' 
    then String.sub o 0 (String.length o - 1)
    else o

let print_unsigned_pretty_float : float -> string = (* Print float without sign and without a trailing dot *)
  fun f ->
    let o = print_pretty_float f in
    match (String.get o 0) with
    | '*'
    | '+'  
    | '-' -> String.sub o 1 ((String.length o)-1)
    |  _  -> o

let string_contains_operator : string -> bool =
  fun s -> (* We would need a fold-operator on strings for an efficient implementation! *)
    String.contains s '+' ||
    String.contains s '-' ||
    String.contains s '*' ||
    String.contains s '/' ||
    String.contains s '=' ||
    String.contains s '>' ||
    String.contains s '<' ||
    String.contains s '^' 


 
(*
      (* Practising a very bad style of recursion: *)
exception MTry
let rec mtry: int -> int -> unit =
  fun x y ->
    let _ = (print_int x; print_string "-"; print_int y; print_string "  #  ") in
    if x > 9 then raise MTry
    else if y > 9 then raise MTry
    else 
      (* First version: *)
      try mtry x (y+1)
      with _ -> 
	begin
	  try mtry (x+1) 0
	  with _ -> ()
	end
      (*
      (* Alternate and equivalent version: *)
	try
	  begin
	    try
	      mtry x (y+1)
	    with _ -> mtry (x+1) 0
	  end
	with _ -> ()
       *)
*)	

let ext_bool_of_string: string -> bool =
  fun in_s ->
    let s = String.lowercase in_s in
    match s with
    | "t" -> true
    | "f" -> false
    |  _  -> bool_of_string s
	  
