(**************************************************************************************

   Author:   Steffen Jost  <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: argument.ml,v $
   Id:       $Id: argument.ml,v 1.22 2005/03/22 21:08:08 a1hloidl Exp $ 

   This module shall parse all argument options.


   ToDos: 
   ------
   
   - rewrite some things: Instead of using all these let-thingys and the_options,
     better use either one of them ie. "Argument.name" suffices already for access!
   

**************************************************************************************)

open Common
open Support.Error
open Syntax
open Arg (* the OCAML Library for paring commandlines *)

let author_info = "Steffen Jost <jost@tcs.informatik.uni-muenchen.de>"
let cvs_revision = "$Revision: 1.22 $"
let cvs_date     = "$Date: 2005/03/22 21:08:08 $"
let version_info = "\nThis is LFD_infer  V1.16   ---   3/2004"  
(*
  Version-numbers containing a star shall not be commited to the CVS-Repository!
  The last committed version is the one without the star! 
*)

let ext_camelot:string      = ".cmlt"
let ext_lf_diamond:string   = ".lfd"
let ext_constraints:string  = ".constraints"

let lp_solver = "/usr/local/bin/lp_solve" (* Name of an external lp_solver that can be called over the actual path (or include path here) *)

type options = 
    {
     integer_lp: bool;   (* Do we want integer annotations? If no, we accept rational annotations *)
     main: string;       (* Name of function to be executed *)
     argmain: (string list); (* UNPARSED Argument(s) for function main. *)
     execute: bool;      (* Shall we sandbox 'main'? *)
     inference:bool;     (* Shall we infer annotations? *) 
     solvelp: bool;      (* Shall we call the lp_solver? *)
     infinity: string;   (* Upper bound for each constraint variable *)
     delta: string;      (* Minimum distance to infinity *)
     obj_lhs: string;    (* Factor for  left-hand function type  annotations in objective function. *)
     obj_rhs: string;    (* Factor for right-hand function type  annotations in objective function. *)
     obj_datin: string;  (* Factor for  left-hand toplevel datatype annotations in objective function. *)
     obj_datout: string; (* Factor for right-hand toplevel datatype annotations in objective function. *)
     sharing: bool;      (* Is sharing allowed? *)
     sharestrict: bool;  (* Sharing uses "=" or ">=" ? *)
     subtyping: bool;    (* Is subtyping via sharing allowed? *)
     debug: bool;        (* Shall we display debug information? *)
     diamond: bool;      (* Print resource values like "<3.5>" instead of "3.5" *)
     uniform: bool;      (* Set all Sizes to one, i.e. uniform-sized diamonds as in LFPL *)
     inplace: bool;      (* Determines whether the sandbox shall reuse heap-locations *)
     clap_sort: bool;    (* If true,  then constructors with more arguments are preferred at Command Line Argument Parsing *)
     clap_ord: bool;     (* If false, then order of constructors is reversed at Command Line Argument Parsing *)
     screen_width: int;  (* Maybe used for a neat screen-printing, but is ignored currently in most printing routines. Hence no argument switch at present. *)
     performance: bool;  (* Increases performance of constraint generation, but is less verbose and produces unreadable files, etc. *)
     pickle: bool;       (* If true,  write rich valdecs in asdl binary format to file "valdecs.pkl" *)
     dir: string;        (* Output dir, i.e. where .lfd etc files will be created *)
   }

(* Defaults: change only THESE values here if different defaults are desired! *)
let default: options = 
  {
   integer_lp = false;
   main = "start";  (* NOTE: if no function of this name is found, a second default is used; it is defined below. Does not affect constraint.ml, as the constraint construction doesent care which function is 'main'. *)
   argmain = [];    (* Main is specified have ONE argument, but this may be a list type, hence we pass a string list here. Note that these strings may actually represent any element of any other LFD-type. exec.ml parses this list. *)
   execute = false; 
   inference = true; 
   solvelp = true; 
   infinity = "10000";  (* Upper-bound for integers in OCaml is at least: 1073741823 *)
   delta ="0";
   obj_lhs    = "+2"; (* must be signed! *)
   obj_rhs    = "-1";
   obj_datin  = "+4";
   obj_datout = "-2";
   sharing = true;
   sharestrict = true;
   subtyping = true;
   debug = false;
   diamond = false;     (* We are using the compressed style in papers, so it should be the default now, althugh it is ambigiuos if there are constructors with identical arguments *)
   uniform = false;
   inplace = true;
   clap_sort = true;
   clap_ord  = true;
   screen_width = 82;   (* Changed to fit smaller screens by default *)
   performance = false; (* Dont set it to true by default! *)
   pickle = false;
   dir = ".";
 }

let fct_main_alt_default = "main" (* Alternate default name for the main function, if the name provided by argument.ml does not exist. *)

let integer_lp_ref   = ref default.integer_lp
let execute_ref      = ref default.execute
let inference_ref    = ref default.inference
let solvelp_ref      = ref default.solvelp
let main_ref         = ref default.main
let argmain_ref      = ref default.argmain
let infinity_ref     = ref default.infinity           
let delta_ref        = ref default.delta              
let obj_lhs_ref      = ref default.obj_lhs            
let obj_rhs_ref      = ref default.obj_rhs            
let obj_datin_ref    = ref default.obj_datin          
let obj_datout_ref   = ref default.obj_datout
let sharing_ref      = ref default.sharing
let sharestrict_ref  = ref default.sharestrict
let subtyping_ref    = ref default.subtyping
let debug_ref        = ref default.debug
let diamond_ref      = ref default.diamond
let uniform_ref      = ref default.uniform
let inplace_ref      = ref default.inplace
let clap_sort_ref    = ref default.clap_sort
let clap_ord_ref     = ref default.clap_ord
let screen_width_ref = ref default.screen_width
let performance_ref  = ref default.performance
let pickle_ref       = ref default.pickle
let dir_ref          = ref default.dir

(* For the anonymous options: *)
let infile_ref     = ref "_"
let outfile_ref    = ref "@"
let argc           = ref 1

(* GLOBAL: make options public - this is certainly not how it should be done in OCAML... *)
let the_options: options ref = ref default

let debug_string: string -> unit =
  fun s -> if !the_options.debug then print_string s

let update_options: unit -> options = (* updates the_options as a side-effect *)
  fun () ->
    let act_opt =
      { 
	integer_lp   = !integer_lp_ref; 
	main         = !main_ref;
	argmain      = !argmain_ref;
	execute      = !execute_ref;
	inference    = !inference_ref; 
	solvelp      = !solvelp_ref;
	infinity     = !infinity_ref;
	delta        = !delta_ref;
	obj_lhs      = !obj_lhs_ref;
	obj_rhs      = !obj_rhs_ref;
	obj_datin    = !obj_datin_ref;
	obj_datout   = !obj_datout_ref;
	sharing      = !sharing_ref;
	sharestrict  = !sharestrict_ref;
	subtyping    = !subtyping_ref;
	debug        = !debug_ref;
	diamond      = !diamond_ref;
	uniform      = !uniform_ref;
	inplace      = !inplace_ref;
	clap_sort    = !clap_sort_ref;
	clap_ord     = !clap_ord_ref;
	screen_width = !screen_width_ref;
	performance  = !performance_ref;
	pickle       = !pickle_ref;
	dir          = !dir_ref;
      }
    in
    let _ = the_options := act_opt in
    act_opt

let my_parse_arg: unit -> ((in_channel * string option) * (out_channel * string option) * options) =
  fun () -> 
    let usage = 
      (
       version_info ^
       "\n\nUsage: lfd_infer ["^ext_lf_diamond^"-infile] [constraint-outfile] <options> \n   \
                    where '_' stands for stdin/stdout,\n   \
                    and   '@' stands for the default constraint-outfile-name, having extension '"^ext_constraints^"'. \
		    \n\n\
	            Options  ( * marks current default):"  
      )
    in
    let defprint = fun b -> if b then "  (*)" else "" in
    let main_opt1     = ("-main", String(fun s -> (execute_ref := true); (main_ref := s)), (" [String]       The name of the function to be executed when sandboxing  ('"^(!main_ref)^"')")) in
    let mainarg_opt1  = ("-arg" , Rest(fun s -> (execute_ref := true); (argmain_ref := list_snoc !argmain_ref s)), 
			 ("  [String List]  A sequence of arguments for the main function. Must be the last option,\n         \
                             as all following command line symbols are parsed as arguments!")) in
    let int_opt1 = ("-int", Set(integer_lp_ref), ("  Restrict to integer annotations"^defprint(!integer_lp_ref))) in
    let rat_opt1 = ("-rat", Clear(integer_lp_ref), ("  Allow rational annotations"^defprint(not !integer_lp_ref))) in
    let exec_opt1 = ("-ex", Set(execute_ref), ("   Execute function '"^(default.main)^"' sandboxed"^defprint(!execute_ref))) in
    let noexec_opt1 = ("-nex", Clear(execute_ref), ("  Do not execute function '"^(default.main)^"'"^defprint(not !execute_ref))) in
    let noinfer_opt1 = ("-ninf", Clear(inference_ref), (" Do not infer any annotations (no constraints are derived)"^defprint(not !inference_ref))) in
    let infer_opt1 = ("-inf", Set(inference_ref), ("  Infer type annotations (just build the constraints)"^defprint(!inference_ref))) in
    let nosolve_opt1 = ("-nlp", Clear(solvelp_ref), ("  Do not call '"^lp_solver^"', merely write constraints to file"^defprint(not !solvelp_ref))) in
    let solve_opt1 = ("-lp", Set(solvelp_ref), ("   Call '"^lp_solver^"' on inferred constraints and print instantiated types"^defprint(!solvelp_ref))) in
    let subtyping_opt1 = ("-sub", Set(subtyping_ref), ("  Allow annotation subtyping"^defprint(!subtyping_ref))) in
    let nosubtyping_opt1 = ("-nsub", Clear(subtyping_ref), (" Prohibit annotation subtyping"^defprint(not !subtyping_ref))) in
    let sharing_opt1 = ("-shr", Set(sharing_ref), ("  Allow sharing of variables"^defprint(!sharing_ref))) in
    let nosharing_opt1 = ("-nshr", Clear(sharing_ref), (" Restrict non-base types to linear use"^defprint(not !sharing_ref))) in
    let infinity_opt1 = ("-infty", Int(fun i -> infinity_ref := (string_of_int i)), ("[Int]   Upper bound for each constraint variable  ("^(!infinity_ref)^")")) in
    let delta_opt1    = ("-delta", Int(fun i -> delta_ref    := (string_of_int i)), ("[Int]   Minimum distance to infty; affects screen print only ("^(!delta_ref)^")")) in
    let obj_lhs_opt1    = ("-olhs", Int(fun i -> obj_lhs_ref := (print_pretty_int' i)), (" [Int]   Objective Function(min): Factor for fixed  input  ("^(!obj_lhs_ref)^")")) in
    let obj_rhs_opt1    = ("-orhs", Int(fun i -> obj_rhs_ref := (print_pretty_int' i)), (" [Int]   Objective Function(min): Factor for fixed output  ("^(!obj_rhs_ref)^")")) in
    let obj_datin_opt1  = ("-odin", Int(fun i -> obj_datin_ref := (print_pretty_int' i)), (" [Int]   Objective Function(min): Factor for data   input  ("^(!obj_datin_ref)^")")) in
    let obj_datout_opt1 = ("-odout", Int(fun i -> obj_datout_ref := (print_pretty_int' i)), ("[Int]   Objective Function(min): Factor for data  output  ("^(!obj_datout_ref)^")")) in
    let debug_opt1 = ("-debug", Set(debug_ref), ("Demand a more verbose computation"^defprint(!debug_ref))) in
    let nodia_opt1 = ("-ndia", Clear(diamond_ref), (" Print annotated types in a condensed, non-fancy style"^defprint(not !diamond_ref))) in
    let dia_opt1 =   ("-dia",   Set(diamond_ref), ("  Print annotated types more verbosely and mark resource annotations with angle brackets '< >'"^defprint(!diamond_ref))) in
    let scrw_opt1 = ("-width", Int(fun i -> screen_width_ref := i), ("[Int]   Maximal number of characters per line on your screen  ("^(print_aligned_int' (!screen_width_ref))^")")) in
    let uniform_opt1 = ("-uni", Set(uniform_ref), ("  Uniform (LFPL-style) constructor sizes override (=1), use with caution"^defprint(!uniform_ref))) in
    let inplace_opt1 = ("-ipl", Set(inplace_ref), ("  Sandboxed execution uses in-place updates (not all dangling pointers might be encountered)."^defprint(!inplace_ref))) in
    let noinplace_opt1 = ("-nipl", Clear(inplace_ref), (" Sandboxed execution never reuses a heap address (does not affect statistics)."^defprint(not !inplace_ref))) in
    let clap_sort_opt1   = ("-asor",  Set(clap_sort_ref),  (" CmdLineArgParser: Prefer constructors with many arguments."^defprint(!clap_sort_ref))) in
    let noclap_sort_opt1 = ("-nasor", Clear(clap_sort_ref),("CmdLineArgParser: Try constructors in the order as declared."^defprint(not !clap_sort_ref))) in
    let clap_ord_opt1    = ("-arev",  Clear(clap_ord_ref), (" CmdLineArgParser: Reverse order of (maybe sorted) constructors tried."^defprint(not !clap_ord_ref))) in
    let noclap_ord_opt1  = ("-narev", Set(clap_ord_ref),   ("CmdLineArgParser: Try constructors in natural (or sorted) order first."^defprint(!clap_ord_ref))) in
    let perf_opt1 = ("-perf", Set(performance_ref), (" Increases performance at the cost of verbosity and readability."^defprint(!performance_ref))) in
    let noperf_opt1 = ("-nperf", Clear(performance_ref), ("Decreases performance but (among other things) protocols constraints to specified file."^defprint(not !performance_ref))) in
    let pickle_opt1 = ("-pkl", Set(pickle_ref), ("  Pickle rich types in valdecs.pkl."^(defprint(!pickle_ref)))) in
    (* HWL was here: -d option to set working dir for outputs *)
    let dir_opt1 = ("-d", String(fun str -> dir_ref := str), ("Setting output directory to "^(!dir_ref))) in
(*  Does not work properly anyway, as CVS-Numbers might diverge:
    let cvs_ver = ("-cvs", Rest(fun s ->  
      let v = Common.string_rdrop 1 (Common.string_drop 11 cvs_revision) in
      let d = Common.string_rdrop 1 (Common.string_drop 7  cvs_date)     in
      let m = ("CVS Version V"^v^" and Date "^d^" for Argument-Module")   in
      let _ = print_string m in raise (Arg.Bad m)),  "  Print version number according to CVS to screen.") 
    in
*)
    let opts =
       rat_opt1::
       int_opt1::
       infer_opt1::
       noinfer_opt1::
       solve_opt1::
       nosolve_opt1::
       uniform_opt1::
       subtyping_opt1::
       nosubtyping_opt1::
       sharing_opt1::
       nosharing_opt1::
       obj_lhs_opt1::obj_rhs_opt1::obj_datin_opt1::obj_datout_opt1::
       infinity_opt1::delta_opt1::
       exec_opt1::noexec_opt1::
       inplace_opt1::noinplace_opt1::
       main_opt1::mainarg_opt1::
       clap_sort_opt1::noclap_sort_opt1::clap_ord_opt1::noclap_ord_opt1::
       perf_opt1::noperf_opt1::
       pickle_opt1::
       debug_opt1::
       nodia_opt1::dia_opt1::
       scrw_opt1::
       dir_opt1::
       []
    in
    let pfile: string -> unit =
      fun s -> 
	(
	 match !argc with
	 | 1 -> (* Infile-Argument *)
	     if 
	       (
		filename_has_suffix s                   (* We accept any suffix *)
  		  (* string_endswith s ext_lf_diamond *)     (* We accept only our own suffix *)
	      || 
		s = "_"
 	)
	     then infile_ref := s 
	     else infile_ref := (s^ext_lf_diamond)
         | 2 -> (* Outfile-Argument *)
	     if 
	       (string_endswith s ext_lf_diamond) 
	   ||
	     (string_endswith s ext_camelot) 
	     then raise (Arg.Bad ("Outputfile should not end with '"^ext_lf_diamond^"' or '"^ext_camelot^"' !"))
             else outfile_ref :=s
	 | _ -> raise (Arg.Bad "Too many command line arguments.")
	); argc := 1 + !argc; () 
    in
    let () = Arg.parse opts pfile usage in
    let (inchan, inname) =
      if !infile_ref = "_"
      then
	let _ = 
	  if !argc <= 1 then 
	    let _ = print_string "\n Awaiting input program via stdin...\n"
	    in flush stdout
	in (stdin,None)
      else 
	try  (open_in !infile_ref, Some !infile_ref) 
	with Sys_error _ -> err ("Can't open " ^ !infile_ref) 
    in
    let (outchan,outname) =
      match !outfile_ref with
      | "_"                           -> (stdout,None)
      |  _  when (not !inference_ref) -> (stdout,None)
      | "@" when (!infile_ref = "_")  -> (stdout,None)
      | "@" -> 
	  (
	   let outfile = 
             try (Filename.chop_extension !infile_ref)^ext_constraints
	     with Invalid_argument s -> !infile_ref^ext_constraints
	   in  
	   try  (open_out outfile, Some outfile) 
           with Sys_error _ -> err ("Can't open " ^ outfile)
	)
      |  _  -> 
	  try  (open_out !outfile_ref, Some !outfile_ref) 
	  with Sys_error _ -> err ("Can't open " ^ !outfile_ref)
    in   
    let _ = solvelp_ref := !solvelp_ref && !inference_ref 
    in
    (
     (inchan, inname),
     (outchan, outname),  
     (update_options ())
    )
