(*  
   File:        ToyVCGtest.ML
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyVCGtest.thy,v 1.1 2003/06/24 23:20:56 da Exp $

   Some tests of the VCG.
   
*)

theory ToyVCGtest = ToyVCG:

section {* Testing the VCGen *}

subsection {* Sanity checks from ToyHLbasic *}

subsubsection {* Primop *}

lemma "x1 \<noteq> x2 \<Longrightarrow>
	\<Turnstile> {(z, s). s<x1> = 2 \<and> s<x2> = 3}
	 (Primop (op+) x1 x2)
         {(z, s, result). result = IVal 5}"
apply hoare_simp
done

subsubsection {* GetF *}

lemma "\<Turnstile> {(obj, s). s\<lfloor>vn\<rfloor> = Ref a \<and> 
		  fmap_lookup (heap s) a = Some obj}
	   (GetFi vn f)
         {(obj, s, result). result = IVal (fst (snd obj) f)}"
apply hoare_simp
done

lemma "\<Turnstile> {(obj, s). s\<lfloor>vn\<rfloor> = Ref a \<and> 
		  fmap_lookup (heap s) a = Some obj}
	   (GetFr vn f)
         {(obj, s, result). result = RVal (snd (snd obj) f)}"
apply hoare_simp
done

subsubsection {* PutF *}

lemma "\<Turnstile> {(z, s). \<exists> a cn ifields rfields. 
	s\<lfloor>vn\<rfloor> = Ref a \<and> fmap_lookup (heap s) a = Some (cn,ifields,rfields) \<and> s<valv> = z}
	   (PutFi vn f valv)
         {(z, s, v) . v = IVal z}"
apply hoare_simp
done

(* this uses short-hand notation for heap lookup *)
lemma "\<Turnstile> {(z, s). \<exists> a cn ifields rfields. 
	s\<lfloor>vn\<rfloor> = Ref a \<and> s\<lless>a\<ggreater> = Some (cn,ifields,rfields) \<and> s\<lfloor>valv\<rfloor> = z}
	   (PutFr vn f valv)
         {(z, s, v) . v = RVal z}"
apply hoare_simp
done

subsubsection {* New *}

lemma "\<Turnstile> {(z, s). freshloc (fmap_dom (heap s)) = a}
	   (New c)
         {(z, s, v) . v = RVal (Ref a)}"
apply hoare_simp
done

subsubsection {* IF *}

lemma "\<Turnstile> {(z,s). s<b> = i \<and> (i=0 \<or> i=1) \<and>
	        s<x1> = 3 \<and> s<x2> = 5}
        (IF b THEN IVar x1 ELSE IVar x2)
         {(z, s, result). \<exists> j. result = IVal j \<and> (j = 3 \<or> j =5)}"
apply hoare_simp
done

lemma "takestimelt ?k (IF b THEN (IF b' THEN IVar x ELSE IVar y) ELSE IVar z)"
apply (simp add: takestimelt_def)
apply hoare_simp
apply (auto)    (* instantiates ?k to 4 *)
done

lemma "takestimelt 4 (IF b THEN (IF b' THEN IVar x ELSE IVar y) ELSE IVar z)"
apply (simp add: takestimelt_def)
apply hoare_simp
done

lemma "takestimelt ?k (IF b THEN IVar z ELSE (IF b' THEN IVar x ELSE IVar y))"
apply (simp add: takestimelt_def)
apply hoare_simp
apply (auto)? (* NB: this FAILS, probably because k gets instantiated to 2 too soon *)
oops

lemma "takestimelt 4 (IF b THEN IVar z ELSE (IF b' THEN IVar x ELSE IVar y))"
apply (simp add: takestimelt_def)
apply hoare_simp
done

subsubsection {* LET *}

lemma "x~=y \<Longrightarrow> \<Turnstile> {(z, s). True}
		 LET x=expr.Int 5; y=expr.Int 7 IN (Primop (op+) x y) END 
                {(z, s, result). (result = IVal 12)}"
apply hoare_simp
done


constdefs
  clock_less_than  :: "(state \<times> state) set"
  "clock_less_than \<equiv> inv_image less_than (clock::state\<Rightarrow>nat)"

lemma wf_clock_less_than [intro]: "wf (clock_less_than)"
by (simp add: clock_less_than_def,
       rule wf_inv_image, rule wf_less_than)


subsection {* \texttt{id}: identity function *}

(* id: inline *)

locale identity_example =
  fixes    j'1 :: iname
    and	   k :: iname
    and	   idd    :: funame
    and	   iddbdy :: expr
  defines  "iddbdy == LET k = expr.IVar j'1
		       IN
                         IVar k
		       END"
  assumes  iddfnbdy [simp]:  "funtable idd = iddbdy"
      and  vardistinct:      "distinct [j'1,k]"
      and  wfmeasure [simp]: "fun_wfmeasure_table idd = clock_less_than"

declare (in identity_example) iddbdy_def [simp]

(* id: inline *)
lemma (in identity_example) "\<Turnstile> {(z,s). s<j'1> = z}
                  iddbdy
                  {(z, s, result). result=IVal z}"
apply (simp)       (* to unfold incbdy *)
apply hoare_simp
done


(* id: CALL *)
lemma (in identity_example) "\<Turnstile> {(z,s). s<j'1> = z}
                  (CALL idd)
                  {(z, s, result). result=IVal z}"
apply hoare_simp
apply (insert wf_clock_less_than, auto)
done


subsection {* \texttt{inc}: increment example (over variable) *}

locale increment_example =
  fixes    one :: iname
    and	   j :: iname
    and	   inc    :: funame
    and	   incbdy :: expr
  defines  "incbdy == LET one = expr.Int 1;
			  j = Primop (% x y. x+y) j one
		       IN
                         IVar j
		       END"
  assumes  incfnbdy[simp]:     "funtable inc = incbdy"
      and  vardistinct[intro]: "distinct [one,j,k]"

declare (in increment_example) incbdy_def [simp]
(* NB: adding vardistinct to simp set doesn't help, it isn't
   a simplification.  We need to find some other way to feed it into
   the VCG tactic. *)


(* This time we don't bother with a wf measure: just add as an
   assumption that it's wf'ded. *)

(* inc: inline *)
lemma (in increment_example) "wf (fun_wfmeasure_table inc) \<Longrightarrow>
		  \<Turnstile> {(z,s). s<j> = 3}
                  incbdy
                  {(z, s, result). result=IVal 4}"
apply (simp)       (* to unfold incbdy *)
apply (insert vardistinct)
apply hoare_simp
done

(* inc: CALL *)
lemma (in increment_example) "wf (fun_wfmeasure_table inc) \<Longrightarrow>
	         \<Turnstile> {(z,s). s<j> = 3}
                  (CALL inc)
                  {(z, s, result). result=IVal 4}"
apply (insert vardistinct)
apply hoare_simp
apply(auto)?
done

lemma (in increment_example) "wf (fun_wfmeasure_table inc) \<Longrightarrow>
                  \<Turnstile> {(z,s). s<j> = 3}
                  (CALL inc)
                  {(z, s, result). result=IVal 4}"
apply (insert vardistinct)
apply hoare_simp
apply(auto)?
done

lemma (in increment_example) "wf (fun_wfmeasure_table inc) \<Longrightarrow>
                  \<Turnstile> {(z,s). s<j> = z}
                  (CALL inc)
                  {(z, s, result). result\<noteq>IVal z}"
apply (insert vardistinct)
apply hoare_simp
apply(auto)?
done

lemma (in increment_example) "wf (fun_wfmeasure_table inc) \<Longrightarrow>
	\<Turnstile> {(z,s). s<j> = z \<and> (2 dvd z)}
        (CALL inc)
         {(z, s, result). \<forall> R. result = (IVal R) \<longrightarrow> ~(2 dvd R)}"
apply (insert vardistinct)
apply (simp add: dvd_def)
apply hoare_simp
apply(auto)?
done

subsection {* \texttt{twice}: testing CALL *}

locale twice_example =
  fixes    i :: iname
    and	   j :: iname
    and	   k :: iname
    and	   twice    :: funame
    and	   twicebdy :: expr
  defines  "twicebdy == LET k = IVar i;
                          j = Primop (% x y. x+y) i k
		      IN
                         IVar j
		      END"
  assumes  twicefnbdy:  "funtable twice = twicebdy"
      and  vardistinct: "distinct [i,j,k]"

declare (in twice_example) twicebdy_def [simp]
declare (in twice_example) twicefnbdy   [simp]


(* twice  *)
lemma (in twice_example) "wf (fun_wfmeasure_table twice) \<Longrightarrow> \<Turnstile> {(z,s). s<i> = 2}
                  (CALL twice)
                  {(z, s, result). result=IVal 4}"
apply (insert vardistinct)
apply(simp)
apply hoare_simp
done

subsection {* \texttt{swap}: testing GetFi and PutFi *}

locale swap_example =
  fixes    m :: iname
    and	   n :: iname
    and	   q :: iname
    and	   z1 :: iname
    and	   z2 :: iname
    and	   lesser    :: ifldname
    and	   greater   :: ifldname
    and	   swap      :: funame
    and    SwapClass :: cname
    and	   swapBody  :: expr
    and    l1 :: locn
    and    M :: int    
    and    N :: int    
    and    M' :: int    
    and    N' :: int    
    and    curr_obj :: "int => int => obj"
  defines "swapBody \<equiv>  LET 
                m  = GetFi self lesser ;
                n  = GetFi self greater ;
                q  = Primop (\<lambda> x y . if y<x then (1::int) else (0::int)) m n 
              IN
                IF q 
                  THEN LET
                         z1 = PutFi self lesser n ; 
                         z2 = PutFi self greater m 
                       IN
                         expr.Int 1
                       END 
                  ELSE expr.Int 0
              END"
      and  "curr_obj x y == (SwapClass, (emptyi(lesser:=x))(greater:=y), emptyr)"
  assumes  swapfnbdy: "funtable swap = swapBody"
      and  swapClass: "classtable SwapClass = \<lparr> iflds = [lesser,greater], rflds = [], meths = \<lambda> mn. K mn \<rparr>"
      and  vardistinct: "distinct [m,n,q,z1,z2]"
      and  flddistinct: "distinct [lesser,greater]"

declare (in swap_example) swapBody_def [simp]
declare (in swap_example) swapfnbdy   [simp]

lemma (in swap_example) 
   "\<Turnstile> {(z,s). clock s = z \<and> N < M \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> fmap_lookup (heap s) l1 = Some (curr_obj M N)}	
	swapBody
      {(z,s,v). s\<lfloor>self\<rfloor> = Ref l1 \<and> fmap_lookup (heap s) l1 = Some (curr_obj N M) }"
apply (insert vardistinct flddistinct)
apply (simp add: curr_obj_def) (* unfold body *)
apply hoare_simp
done

(* same as above but using short-hand notation for heap access *)
lemma (in swap_example) 
   "\<Turnstile> {(z,s). clock s = z \<and> N < M \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and>  s\<lless>l1\<ggreater> = Some (curr_obj M N)}	
	swapBody
      {(z,s,v). s\<lfloor>self\<rfloor> = Ref l1 \<and>  s\<lless>l1\<ggreater> = Some (curr_obj N M) }"
apply (insert vardistinct flddistinct)
apply (simp add: curr_obj_def) (* unfold body *)
apply hoare_simp
done


(* case distinction in post-condition *)
lemma (in swap_example) 
   "\<Turnstile> {(z,s). clock s = z \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> fmap_lookup (heap s) l1 = Some (curr_obj M N)}	
	swapBody
      {(z,s,v). s\<lfloor>self\<rfloor> = Ref l1 \<and> 
                (M<N \<longrightarrow> fmap_lookup (heap s) l1 = Some (curr_obj M N)) \<and> 
                (N<M \<longrightarrow> fmap_lookup (heap s) l1 = Some (curr_obj N M)) \<and> 
                (N=M \<longrightarrow> fmap_lookup (heap s) l1 = Some (curr_obj N N))}"
apply (insert vardistinct flddistinct)
apply (simp add: curr_obj_def)
apply hoare_simp
done

section {* \texttt{inc}: simple recursion over variable with functions *}

constdefs iszero::"int \<Rightarrow> int"
"iszero x == (if x = 0 then 1 else 0)"

locale count_example =
(*fun countdown(n) = let n = n -1
                     in if n < 1 then 0 else countdown(n) end
 *)
  fixes    m :: iname
    and	   n :: iname
    and	   b :: iname
    and	   countfn    :: funame
    and	   countfnbdy :: expr
  defines  "countfnbdy == LET m = expr.Int 1;
			      n = Primop (% x y. x-y) n m;
			      b = Primop (% x y. if x < 1 then 0 else 1) n n 
			   IN 

			      IF b THEN CALL countfn ELSE expr.Int 0
  			   END"
  assumes  countfnbdy[simp]:  "funtable countfn = countfnbdy"
      and  wfmeasure [simp]:  "fun_wfmeasure_table countfn = inv_image less_than (\<lambda> s. nat (get_ivar s n))"
      and  vardistinct:       "distinct [m,n,b]"
      and  preAss1:  "fun_preassn_table countfn == {(z,s). 0 < (s<n>)}"
      and postAss: "fun_postassn_table countfn == {(z,s,v). v=IVal 0 \<and> s<n>=0 \<and> (\<exists> K. clock s = K)}"

declare (in count_example) countfnbdy_def [simp]  (* unfold def *)
declare (in count_example) preAss1    [simp]
declare (in count_example) postAss   [simp]

lemma (in count_example) 
   "\<Turnstile> {(z,s). 0 < (s<n>)}	
	(CALL countfn) 
      {(z,s,v). v=IVal 0 \<and> (s<n>=0)}"
apply (insert vardistinct)
apply hoare_simp
apply (erule_tac thin_rl)  (* remove used IH assumption *)
apply (simp add: inv_image_def)		     (* termination *)
apply (erule thin_rl, insert wfmeasure, simp)
apply (rule wf_inv_image, rule wf_less_than) (* WF measure *)
done

lemma (in count_example) 
   "\<Turnstile> {(z,s). 0 < (s<n>)}	
	(CALL countfn) 
      {(z,s,v). \<exists> K. clock s = K}"
apply (insert vardistinct)
apply hoare_simp
apply (erule_tac thin_rl)  (* remove used IH assumption *)
apply (simp add: inv_image_def)		     (* termination *)
apply (erule thin_rl, insert wfmeasure, simp)
apply (rule wf_inv_image, rule wf_less_than) (* WF measure *)
done

lemma (in count_example) 
   "\<Turnstile> {(z,s). 0 < z \<and> z = (s<n>)}	
	(CALL countfn) 
      {(z,s,v). \<exists> K. clock s = K}"
apply (insert vardistinct)
apply hoare_simp
apply (erule_tac thin_rl)  (* remove used IH assumption *)
oops

constdefs HSize ::"state \<Rightarrow> nat"
"HSize s == card (fmap_dom (heap s))"

(*Version with a better invariant*)
locale count_example3 =
(*fun countdown(n) = let n = n -1
                     in if n < 1 then 0 else countdown(n) end
 *)
  fixes    m :: iname
    and	   n :: iname
    and	   b :: iname
    and	   countfn    :: funame
    and	   countfnbdy :: expr
  defines  "countfnbdy == LET m = expr.Int 1;
			      n = Primop (% x y. x-y) n m;
			      b = Primop (% x y. if x < 1 then 0 else 1) n n 
			   IN 
			      IF b THEN CALL countfn ELSE expr.Int 0
  			   END"
  assumes  countfnbdy[simp]:  "funtable countfn = countfnbdy"
      and  wfmeasure [simp]:  "fun_wfmeasure_table countfn = inv_image less_than (\<lambda> s. nat (get_ivar s n))"
      and  vardistinct:       "distinct [m,n,b]"
      and  preAss:  "fun_preassn_table countfn = {((InitClock,InitSN,InitHS),s). 
                                                   clock s = InitClock + 12 * (InitSN - (nat (s<n>))) \<and> 
                                                   0 < s<n> \<and>
                                                   HSize s = nat InitHS}"

declare (in count_example3) countfnbdy_def [simp]  (* unfold def *)
declare (in count_example3) preAss    [simp]

(*First proof: functional correctness*)
lemma (in count_example3) 
   "\<Turnstile> {((InitClock,InitSN,InitHS),s). clock s = InitClock + 12 * (InitSN - (nat (s<n>))) \<and> 0 < s<n>}
	(CALL countfn) 
      {(z,s,v). v=IVal 0 \<and> (s<n>=0)}"
apply (insert vardistinct)
apply(hoare_simp)
apply(auto)
defer 1
apply (simp add: inv_image_def)		     (* termination *)
oops (*this is now only a matter of arithmetic\<dots>*)

(*Second proof: time consumption*)
lemma (in count_example3) 
   "\<Turnstile> {((InitClock,InitSN,InitHS),s). clock s = InitClock + 12 * (InitSN - (nat (s<n>))) \<and> 0 < s<n>}
	(CALL countfn) 
      {((InitClock, InitSN,InitHS),s,v). clock s = InitClock + 12 * (InitSN - 1) + 13}"
apply (insert vardistinct)
apply(hoare_simp)
apply(auto)
apply(subgoal_tac "s'<n> = 1")
apply (erule_tac thin_rl)  (* remove used IH assumption *)
apply(simp (no_asm_simp))
apply(simp)
prefer 2
apply (simp add: inv_image_def)		     (* termination *)
oops (*only arithmetic left*)

(*Third proof: heap consumption*)
lemma (in count_example3) 
   "\<Turnstile> {((InitClock,InitSN,InitHS),s). clock s = InitClock + 12 * (InitSN - (nat (s<n>))) \<and> 0 < s<n> \<and> HSize s = nat InitHS}
	(CALL countfn) 
      {((InitClock,InitSN,InitHS),s,v). HSize s = nat InitHS}"
apply (insert vardistinct)
apply(hoare_simp)
apply(auto)
apply (simp_all add: HSize_def)
prefer 2
apply (simp add: inv_image_def)		     (* termination *)
oops (*only arithmetic left*)

(*Fourth proof: functional correctness, time and heap consumption*)
lemma (in count_example3) 
   "\<Turnstile> {((InitClock,InitSN,InitHS),s). clock s = InitClock + 12 * (InitSN - (nat (s<n>))) \<and>
                                      0 < nat (s<n>) \<and> 
                                      HSize s = nat InitHS}
	(CALL countfn) 
      {((InitClock,InitSN,InitHS),s,v). v=IVal 0 \<and> (s<n>=0) \<and> clock s = InitClock + 12 * (InitSN - 1) + 13 \<and> HSize s = nat InitHS}"
apply (insert vardistinct)
apply(hoare_simp)
apply(auto)
apply (simp_all add: HSize_def)
apply(subgoal_tac "s'<n> = 1")
apply (erule_tac thin_rl)  (* remove used IH assumption *)
apply(simp (no_asm_simp))
apply(simp)
prefer 2
apply (simp add: inv_image_def)		     (* termination *)
oops (*only arithmetic left*)

(*----------------------------------------------------------------*)
(*the intention of this modified program is to prove that we create max (0, s<n>) objects*)
locale countNEW_example =
(*fun countdown(n) = let n = n -1
                         p = new LIST
                     in if n<1 then 0 else countdown(n) end
 *)
  fixes    m :: iname
    and	   n :: iname
    and	   b :: iname
    and    p :: rname
    and    C :: cname
    and	   countfn    :: funame
    and	   countfnbdy :: expr
  defines  "countfnbdy == LET m = expr.Int 1;
			      n = Primop (% x y. x-y) n m;
			      b = Primop (% x y. if x < 1 then 1 else 0) n n 
                          IN LETR p = NEW C
			     IN 
			      IF b THEN expr.Int 0 ELSE CALL countfn 
                             END
  			  END"
  assumes  countfnbdy[simp]:  "funtable countfn = countfnbdy"
      and  wfmeasure [simp]:  "fun_wfmeasure_table countfn = inv_image less_than (\<lambda> s. nat (get_ivar s n))"
      and  vardistinct:       "distinct [m,n,b]"
      and  preAss:  "fun_preassn_table countfn = {((InitClock,InitSN,InitHS),s). 
                                                   clock s = InitClock + 14 * (InitSN - (nat (s<n>))) \<and> 
                                                   0 < s<n> \<and>
                                                   HSize s = nat InitHS + (InitSN - (nat (s<n>)))}"

declare (in countNEW_example) countfnbdy_def [simp]  (* unfold def *)
declare (in countNEW_example) preAss    [simp]

lemma SizeInsert: "card (insert (freshloc H) H) = Suc (card H)"
sorry

lemma (in countNEW_example) 
   "\<Turnstile> {((InitClock,InitSN,InitHS),s). clock s = InitClock + 14 * (InitSN - (nat (s<n>)))\<and> 
                                      0 < s<n> \<and> 
                                      HSize s = nat InitHS}
	(CALL countfn) 
      {((InitClock,InitSN,InitHS),s,v). v=IVal 0 \<and> (s<n>=0)}"
apply (insert vardistinct)
apply(hoare_simp)
apply(auto)
apply (simp_all add: HSize_def SizeInsert)
prefer 5
apply (simp add: inv_image_def)		     (* termination *)
oops (*only arithmetic left*)


lemma
(*-----------------------------------------------------------------*)
lemma
subsection {* \texttt{dec}: decrement a field *}

locale dec_example =
  fixes    m         :: iname
    and	   n         :: iname
    and	   z1        :: iname
    and	   q1        :: iname
    and	   zero      :: iname
    and    l1        :: locn
    and    N         :: int    
    and    count     :: ifldname
    and	   dec      :: funame
    and    DecClass :: cname 
    and	   decBody  :: expr
  defines "decBody \<equiv>  LET 
                m  = GetFi self count ;
                n  = Primop (\<lambda> x y . x - 1) m m ;
                z1 = PutFi self count n ; 
                zero = expr.Int 0 ;
                q1 = Primop (\<lambda> x y . if y<x then (1::int) else (0::int)) n zero
              IN
                IF q1 
                  THEN IVar zero
                  ELSE IVar z1
              END"
  assumes  DecClass:   "classtable DecClass = \<lparr> iflds = [count], rflds = [], meths = \<lambda> mn. K mn \<rparr>"
      and  vardistinct: "distinct [m,n,z1,q1,zero]"
      and  vardistinct': "zero ~= q1"
      and  dec_fnbdy:   "funtable dec = decBody"

declare (in dec_example) decBody_def[simp]
declare (in dec_example) dec_fnbdy[simp]

lemma (in dec_example) 
   "\<Turnstile> {(z,s). clock s = z \<and> 0 < N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
        fmap_lookup (heap s) l1 = Some (DecClass, emptyi(count:=N), emptyr)}	
	decBody
      {(z,s,v). v = IVal 0 }"
apply (insert vardistinct, insert vardistinct')
apply (simp)
apply hoare_simp
done

lemma (in dec_example) 
   "[| wf (fun_wfmeasure_table dec) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
         s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count:=N), emptyr)}	
      (CALL dec)
      {(z,s,v). v = IVal 0 \<and>
         s\<lfloor>self\<rfloor> = Ref l1 \<and> s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count:=(N - 1)), emptyr) }"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
apply(auto)
oops
(*apply (assumption)
done
*)

subsection {* dec2 (non-rec, but with a CALL in the ELSE branch) *}

consts the_locn :: "ref => locn"
primrec 
"the_locn (Ref l) = l"

constdefs the_ifld :: "rname => ifldname => state => nat"
"the_ifld x y s == nat ((fst (snd (the (fmap_lookup (heap s) (the_locn (s\<lfloor>x\<rfloor>)))))) y)"

(*                    Ref r => nat (snd (the (fmap_lookup (heap s) r)) count) *)

locale dec2_example =
  fixes    m         :: iname
    and	   n         :: iname
    and	   z1        :: iname
    and	   q1        :: iname
    and	   zero      :: iname
    and	   zero'     :: iname
    and    l1        :: locn
    and    N         :: int    
    and    count     :: ifldname
    and	   dec      :: funame
    and    DecClass :: cname 
    and	   decBody  :: expr
    and    const     :: funame
    and	   constBody  :: expr
  defines "decBody \<equiv>  LET 
                m  = GetFi self count ;
                n  = Primop (\<lambda> x y . x - 1) m m ;
                z1 = PutFi self count n ; 
                zero = expr.Int 0 ;
                q1 = Primop (\<lambda> x y . if \<not> y<x then (1::int) else (0::int)) n zero
              IN
                IF q1 
                  THEN IVar zero
                  ELSE CALL const
              END"
  and     "constBody \<equiv>  IVar n"
  assumes  DecClass:   "classtable DecClass = \<lparr> iflds = [count], rflds = [], meths = \<lambda> mn. K mn \<rparr>"
      and  vardistinct: "distinct [m,n,z1,q1,zero,zero']"
      and  dec_fnbdy:   "funtable dec = dec2Body"
      and  const_fnbdy:   "funtable const = constBody "
      and  dec_wfmeasure: "fun_wfmeasure_table dec = inv_image less_than (the_ifld self x)"
      and  const_wfmeasure: "fun_wfmeasure_table const = inv_image less_than (\<lambda>s . nat (s<n>))"
      and  const_preinv: "fun_preassn_table const == {((z::int), s). s<n>=z \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count := N), emptyr)}"
      and  const_postinv: "fun_postassn_table const == {((z::int), s, v). v = IVal z \<and> s<n>=z \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count := N), emptyr)} "
(*
      and const_inv: "\<Turnstile> {(z, s). s<n>=N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and>
             s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count := N), emptyr)} 
            constBody
            {(z, s, v).
             v = IVal N \<and> s<n>=N \<and>
             s\<lfloor>self\<rfloor> = Ref l1 \<and>
             s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count := N), emptyr)}"
*)
(*       and const_inv: "\<Turnstile> ((fst fun_assn_table) const) CALL const ((snd fun_assn_table) const)" *)

declare (in dec2_example) decBody_def[simp]
declare (in dec2_example) dec_fnbdy[simp]
declare (in dec2_example) constBody_def[simp]
declare (in dec2_example) const_fnbdy[simp]
declare (in dec2_example) dec_wfmeasure[simp]
declare (in dec2_example) const_preinv[simp]
declare (in dec2_example) const_postinv[simp]

(* const alone is ok *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=N}	
      (CALL const)
      {(z,s,v). v = IVal N \<and> s<n>=N}"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
oops
(*apply(auto)
done
*)

(* combining both sucks *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        CALL const
      END
      {(z,s,v). v = IVal N}"
apply (insert vardistinct)
apply (simp)
apply hoare_rec_step (* LET *)
defer 1
(* CALL *)
apply (tactic {* res_inst_tac [("Q'","fun_postassn_table const")] (thm "HCallAux0") 1 *})
(*
apply (tactic {* resolve_tac [(thm "HCallAux0")] 1 *})
apply (tactic {* instantiate_tac [("Q'3","fun_postassn_table const")] *})
*)
apply (tactic {* rtac (thm "conjI") 1 *})
(*
apply (tactic {* simp_tac (simpset() addsimps [(thm "const_preinv_blessed"),(thm "const_postinv_blessed")]) 2 *})
apply (tactic {* simp_tac (simpset() addsimps [(thm "const_preinv_blessed"),(thm "const_postinv_blessed")]) 1 *})
*)
defer 1
apply (tactic {* rtac (thm "subsetI") 1 *})
defer 1
apply hoare_rec_step
apply clarsimp
defer 1
apply (simp add: const_postinv)
oops

(* same if the CALL is nested in an IF *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s<n>=N \<and> s<z1>=0}	
      IF z1
        THEN IVar n
        ELSE CALL const
      {(z,s,v). v = IVal N}"
apply (insert vardistinct)
apply (simp)
apply hoare_simp
apply (auto)
oops

(* const using CallRec *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=z}	
      (CALL const)
      {(z,s,v). v = IVal z \<and> s<n>=z}"
apply (insert vardistinct)
apply (simp)
apply (rule HSP)
apply (rule HCall9)
apply (simp)
apply (rule subsetI)
apply (simp)
apply (subgoal_tac "x=(z0,s0)")
apply (simp)
apply (rule impI)
apply (rule allI)+
apply (rule impI)
apply (rule conjI)
(* petaQ !!!! *)
oops

lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=z}	
      (CALL const)
      {(z,s,v). v = IVal z \<and> s<n>=z}"
apply (insert vardistinct)
apply (simp)
apply hoare_rec
(* apply (rule conjI) *)
oops


(* LET alone is ok *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        IVar n
      END
      {(z,s,v). v = IVal N \<and> s<n>=N}"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
done

(* CALL nested in LET is a problem *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=z}	
      LET 
        zero' = expr.Int 0
      IN 
        CALL const
      END
      {(z,s,v). s<n>=z}"
apply (insert vardistinct)
apply (simp)
(* doing it manually; no VCG for now *)
apply (rule HLet)
apply (rule HSP)
apply (rule HInt0)
apply (simp)
apply (rule subsetI)
(* done with let header; ?R1 is post-assn for body, after "system" work in let *)
defer 1
(* (1) *)
apply (rule HCall9)
apply (simp)
apply (auto)
(* (2) *)
(* apply (rule HCall0) *)
(*
apply (rule allI)
apply (rule HSP)
apply (rule HVar0)
apply (rule subsetI)
apply (simp)
apply (fastsimp)
*)
(* False !? *)
oops

lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        CALL const
      END
      {(z,s,v). v = IVal N}"
apply (insert vardistinct)
apply (simp)
apply (rule HLet)
defer 1
apply (rule HCall0)
apply (rule allI)
defer 1
apply (rule HSP)
apply (rule HInt0)
apply (simp)
apply (rule subsetI)
apply (simp)
defer 1
apply (simp)
apply (rule HVar)
apply (rule subsetI)
apply (simp)
apply (clarify)
apply (simp)
defer 1
apply (simp add: state_functions)
apply (auto)
oops

(* dropping from Hoare-level to foundational-level *)
(* apply (unfold hoare_valid_def)
apply (rule allI)+
apply simp
apply (rule impI)
apply (rule impI)
apply (rule allI)
apply (rule_tac x="0" in exI)
apply (rule conjI)
defer 1
defer 1
*)
(* incomplete but looks ok on this level *)
(*
apply (rule allI)+
apply (rule impI)
apply (rule allI)+
apply (rule impI)
apply simp
apply (rule conjI)
*)


lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        CALL const
      END
      {(z,s,v). v = IVal N \<and> s<n>=N}"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
apply (auto)
(* seems auto throws too much info away *)
oops

(* without the CALL it works fine *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        IVar n
      END
      {(z,s,v). v = IVal N \<and> s<n>=N}"
apply (insert vardistinct)
apply (simp)
apply hoare_simp_step  (* LET *)
defer 1
apply hoare_simp_step  (* IVar *)
apply hoare_simp_step  (* Int 0 *)
done

(* with the call we get stuck *)
lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). s<n>=N}	
      LET 
        zero' = expr.Int 0
      IN 
        CALL const
      END
      {(z,s,v). v = IVal N \<and> s<n>=N}"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
defer 1
defer 1
apply (simp add: const_wfmeasure inv_image_def)
apply (erule thin_rl)
defer 1
defer 1
apply (auto)
(* again that s'<n> = N left to prove *)
oops

lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table dec) ; wf (fun_wfmeasure_table const) ; zero \<noteq> q1 |] ==>  
    \<Turnstile> {(z,s). clock s = z \<and> 0 < N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
         s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count:=N), emptyr)}	
	decBody
      {(z,s,v). v = IVal (N - 1) }"
apply (insert vardistinct)
apply (simp)
apply (hoare_simp)
apply (auto)
oops

lemma (in dec2_example) 
   "[| wf (fun_wfmeasure_table dec) ; wf (fun_wfmeasure_table const) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
         s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count:=N), emptyr)}	
      decBody
      {(z,s,v). v = IVal (N - 1) \<and>
         s\<lfloor>self\<rfloor> = Ref l1 \<and> s\<lless>l1\<ggreater> = Some (DecClass, emptyi(count:=(N - 1)), emptyr) }"
apply (insert vardistinct)
apply (insert const_inv)
apply (simp add: const_inv)
apply (insert const_inv)
apply (rotate_tac -1)
apply (hoare_simp)
apply (auto)
oops

subsection {* \texttt{ping}: recursive CALL version *}

locale ping_example =
  fixes    m         :: iname
    and	   n         :: iname
    and	   z1        :: iname
    and	   q1        :: iname
    and	   zero      :: iname
    and    l1        :: locn
    and    N         :: nat    
    and    count     :: ifldname
    and	   ping      :: funame
    and    PingClass :: cname 
    and	   pingBody  :: expr
  defines "pingBody \<equiv>  LET 
                m  = GetFi self count ;
                n  = Primop (\<lambda> x y . x - 1) m m ;
                z1 = PutFi self count n ; 
                zero = expr.Int 0 ;
                q1 = Primop (\<lambda> x y . if \<not> y<x then (1::int) else (0::int)) n zero
              IN
                IF q1 
                  THEN IVar zero
                  ELSE CALL ping
              END"
  assumes  PingClass:   "classtable PingClass = \<lparr> iflds = [count], rflds = [], meths = \<lambda> mn. K mn \<rparr>"
      and  vardistinct: "distinct [m,n,z1,q1,zero]"
      and  pingfnbdy:   "funtable ping = pingBody"
      and  wfmeasure [simp]: "fun_wfmeasure_table ping = inv_image less_than (the_ifld self x)"
      (* and  wfmeasure [simp]:  "fun_wfmeasure_table countfn = inv_image less_than (\<lambda> s. snd (the (fmap_lookup (heap s) (case (s\<lfloor>self\<rfloor>) of Ref r => r))) count)" *)
      (* and  wfmeasure:   "fun_wfmeasure_table countfn = {(s1,s2). (s1<n>) < (s2<n>)}" *)

(*
constdefs (in ping_example) measure_by_ifld :: "rname => (state \<times>) state set"
"measure_by_ifld x == inv_image less_than (the_ifld self x)"
*)

declare (in ping_example) pingBody_def[simp]
declare (in ping_example) pingfnbdy[simp]

(* probably need more assumptions: many VCs don't simplify *)
lemma (in ping_example) 
   "[| wf (fun_wfmeasure_table ping) |] ==>  
    \<Turnstile> {(z,s). 0 < N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
         s\<lless>l1\<ggreater> = Some (PingClass, emptyi(count:=(int N)), emptyr)}	
      (CALL ping)
      {(z,s,v). v = IVal 0 \<and>
         s\<lfloor>self\<rfloor> = Ref l1 \<and> s\<lless>l1\<ggreater> = Some (PingClass, emptyi(count:=0), emptyr) }"
apply (insert vardistinct)
apply (simp)
apply hoare_simp
(* 2 subgoals, both VCs, hurray *)
(* Plan A: take VCs apart and check what's really needed to be proven *)
apply (rule conjI)
apply (rule impI)
apply (rule conjI)
oops

(*
apply (rule impI)
apply (rule conjI)
defer 1
apply (rule impI)
apply (rule conjI)
defer 1
apply (rule impI)
apply (rule conjI)
defer 1
apply (rule_tac x="l1" in exI)
apply (rule conjI)
defer 1
apply (rule_tac x="PingClass" in exI)
apply (rule_tac x="emptyi(count := N)" in exI)
apply (rule conjI)
apply (rule impI)
apply (rule conjI)
*)(* argh, that order-in-vardistinct problem again *)(*
defer 1
apply (rule impI)
apply (rule conjI)
apply (rule_tac x="emptyr" in exI)
apply (simp)
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
apply (rule_tac x="l1" in exI)
apply (rule conjI)
apply simp
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
*)(* instantiate components of the obj *)(*
apply (rule_tac x="PingClass" in exI)
apply (rule_tac x="emptyi(count := N)" in exI)
apply (rule_tac x="emptyr" in exI)
apply (simp)
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
apply (rule impI)
apply (rule conjI)
apply (rule_tac x="emptyr" in exI)
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
apply (rule_tac x="l1" in exI)
apply (rule conjI)
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
apply (rule_tac x="PingClass" in exI)
apply (rule_tac x="emptyi(count := N)" in exI)
apply (rule_tac x="emptyr" in exI)
apply (rule conjI)
apply (simp)
*)(* argh, lost the structure theorem on l1 *)(*
defer 1
apply (simp)
apply (rule conjI)
apply (simp add: state_functions)
oops
*)
(* Plan B: use the sledgehammer *)
(* apply (auto) *)


lemma (in ping_example) 
   "[| wf (fun_wfmeasure_table ping) |] ==>  
    \<Turnstile> {(z,s). clock s = z \<and> 0 = N \<and> s\<lfloor>self\<rfloor> = Ref l1 \<and> 
        fmap_lookup (heap s) l1 = Some (PingClass, emptyi(count:=(int N)), emptyr)}	
	pingBody
      {(z,s,v). v = IVal 0 }"
apply (insert vardistinct)
apply (simp)
apply hoare_simp
(* this fails to expand the IF in the let body #$#$@ *)
(* apply (simp add: state_functions) *)
(**)
defer 1
oops


subsection {* \texttt{coast/pedal} example (mut-rec CALL) taken from~\cite{Nipkow-CSL02} *}

constdefs ISZERO :: "iname => expr"
 "ISZERO n == Primop (% x y. if x=0 then 1 else 0) n n"
constdefs DEC :: "iname => expr"
 "DEC n == Primop (% x y. x - 1) n n"
constdefs MYAND :: "iname => iname => expr"
 "MYAND m n == Primop (% x y. x*y) m n"
constdefs MYLT :: "iname => iname => expr"
 "MYLT m n == Primop (% x y. if x<y then 1 else 0) m n"

locale coast_pedal_example =
    fixes    m  :: iname
      and    n  :: iname
      and    q1 :: iname
      and    q2 :: iname
      and    q3 :: iname
      and    q4 :: iname
      and    q5 :: iname
      and    M :: int
      and    N :: int
      and pedal :: funame
      and coast :: funame
      and pedalBody :: expr
      and coastBody :: expr
  defines "pedalBody == LET
                         q1 = ISZERO n ;
                         q2 = ISZERO m ;
                         q3 = MYAND q1 q3
                       IN
                       	 IF q3 
                       	   THEN IVar q3 
                       	   ELSE LET
                       	          q4 = MYLT n m
                       	        IN
                       	          IF q4
                       	            THEN LET
                       	                   n = DEC n;
                       	                   m = DEC m
                       	                 IN 
                       	                   CALL coast
                       	                 END
                       	             ELSE LET
                       	                   n = DEC n
                       	                  IN 
                       	                   CALL pedal
                       	                  END
	                        END
                       END"
      and "coastBody == LET
                         q5 = MYLT n m
                       IN
                         IF q5
                           THEN LET
                                  m = DEC m
                                IN
                                  CALL coast
                                END
                           ELSE CALL pedal
                       END"
  assumes pedalfnbdy : "funtable pedal = pedalBody"
      and coastfnbdy : "funtable coast = coastBody"
      and vardistinct : "distinct [m,n,q1,q2,q3,q4,q5]"

      (* ToDo: use lexicographic ordering on (n,m) as measure
         and  wfmeasure [simp]:  "fun_wfmeasure_table countfn = inv_image less_than (\<lambda> s. nat (get_ivar s n))"*)

declare (in coast_pedal_example) pedalfnbdy [simp]  (* function body lookup *)
declare (in coast_pedal_example) coastfnbdy [simp]  (* function body lookup *)
declare (in coast_pedal_example) pedalBody_def [simp]  (* unfold function body *)
declare (in coast_pedal_example) coastBody_def [simp]  (* unfold function body *)
declare (in coast_pedal_example) ISZERO_def [simp]  (* unfold basic expr constructor *)
declare (in coast_pedal_example) MYAND_def [simp]   (* unfold basic expr constructor *)
declare (in coast_pedal_example) MYLT_def [simp]    (* unfold basic expr constructor *)

lemma (in coast_pedal_example) 
   "\<Turnstile> {(z,s). (s<m>=M) \<and> (s<n>=N) \<and> (0<N) \<and> (0<M)} 
	pedalBody
      {(z,s,v). (v = IVal 1) \<and> (s<m>=0) \<and> (s<n>=0)}"
apply (insert vardistinct)
apply (simp)
oops
(* apply hoare_simp*) (* loops *)

subsection {* ping (INVOKE version) *}

(*
locale example_ping =
  fixes    m :: iname
    and	   n :: iname
    and	   z1 :: iname
    and	   q1 :: iname
    and	   zero :: iname
    and	   dummyarg :: iname
    and	   count    :: ifldname
    and	   ping     :: mname
    and	   countfn  :: funame
    and    PingClass :: cname 
    and	   pingBody :: expr
  defines "pingBody \<equiv>  LET 
                m  = GetFi self count ;
                n  = Primop (\<lambda> x y . x - 1) m m ;
                z1 = PutFi self count n ; 
                zero = expr.Int 0 ;
                q1 = Primop (\<lambda> x y . if y<x then (1::int) else (0::int)) n zero
              IN
                IF q1 
                  THEN Invoke self ping dummyarg
                  ELSE Var z1
              END"
  assumes  PingClass: "classtable PingClass = \<lparr> flds = [count], meths = empty ( ping \<mapsto> ({m,n,z1,q1,zero}, pingBody) ) \<rparr>"
      and  vardistinct: "distinct [m,n,z1,q1,zero]"
*)
end

(* -- doing single stepping through the derivation *)
apply hoare_simp_step (* step 1 *)
defer 1
apply hoare_simp_step (* step 2 *)
defer 1
apply hoare_simp_step (* step 3 *)
defer 1
apply hoare_simp_step (* step 4 *)
defer 1
apply hoare_simp_step (* step 5 *)
defer 1
apply hoare_simp_step
apply (tactic {* HoareRecWFCall 1 *})
apply (tactic {* rtac allI 1 *})
apply (tactic {* rtac impI 1 *})
apply (tactic {* rtac (thm "HCall0") 1 *})
apply (tactic {* rtac allI 1 *})
apply (simp)
(* body now unfolded *)
apply hoare_simp_step (* step 1 *)
defer 1
apply hoare_simp_step (* step 2 *)
defer 1
apply hoare_simp_step (* step 3 *)
defer 1
apply hoare_simp_step (* step 4 *)
defer 1
apply hoare_simp_step (* step 5 *)
defer 1
apply hoare_simp_step
(* at CALL again; we should be able to apply assumption now *)
apply (tactic {* (assume_tac 1) *})
apply hoare_simp_step (* IVar *)
defer 1
apply hoare_simp_step (* IVar *)
apply hoare_simp_step (* GetFi self count *)
(* needs simp *)
(* apply (tactic {* (BasicSimpTac tac 1) *} *)
defer 1
apply hoare_simp_step (* Primop *)
apply hoare_simp_step (* GetFi self count *)
apply hoare_simp_step (* Int 0 *)
apply hoare_simp_step (* Primop *)
apply hoare_simp_step (* GetFi self count *)
(* needs simp *)
defer 1
apply hoare_simp_step (* Primop *)
apply hoare_simp_step (* GetFi self count *)
apply hoare_simp_step (* Int 0 *)
apply hoare_simp_step (* Primop *)
(* wf-ness follows from assumption *)
apply (assumption)
(* -- now only VCs left *)
apply (auto)
(* False; petaQ!!!! *)
oops

(*ping*)
apply hoare_simp_step (* IF *)
(* the next step should introduce an ind hyp *)
apply hoare_simp_step (* IF *)
(* needs debugging to figure out where the VCG gets stuck *)
apply (tactic {* HoareRecWFCall 1 *})
apply (tactic {* rtac allI 1 *})
apply (tactic {* rtac impI 1 *})
apply (tactic {* rtac (thm "HCall0") 1 *})
apply (tactic {* rtac allI 1 *})
(* apply (tactic {* rtac (thm "subset_refl") 1 *}) *)
(* apply (tactic {* BasicSimpTac all_tac 1 *}) *)
apply (simp)
apply hoare_simp
apply (fastsimp)
defer 1
apply simp
defer 1
apply (simp add: state_functions)
apply hoare_simp_step
apply (fastsimp)
apply hoare_simp_step
defer 1
apply hoare_simp_step
apply (fastsimp)


apply (rule HCall)

apply hoare_simp_step
apply (simp)

apply (simp add: state_functions)
*)
