(* This file is temporarily defunct in favour of ToyHLbasic, ToyHLproc *)

(* Nipkow-style Hoare logic, using sets as pre-post conditions *)

(* Sets seem to be easier to deal with than predicates. 
   They also look a bit nicer to write. *)

header "Hoare logic for Toy Grail"

theory ToyHLset = ToyGrailLemmas:

text {* Assertions take two forms: pre-conditions are predicates
  on pre-states; post-conditions are predicates on post-states
  and also the resulting value.  Additionally we include 
  auxiliary variables related in the pre and post condition.
  Auxiliary variables can have any type. *}


types
'a preassn = "('a \<times> state) set"
'a postassn = "('a \<times> state \<times> val) set"

constdefs 
  hoare_valid :: "'a preassn \<Rightarrow> expr \<Rightarrow> 'a postassn \<Rightarrow> bool"  ("|= (1_)/ (_)/ (1_)" 50)
   "|= P e Q \<equiv> \<forall> s t v. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> 
				  (\<forall> z. (z, s) \<in>  P \<longrightarrow> (z, t, v) \<in> Q)"

syntax (xsymbols)
  hoare_valid :: "'a preassn \<Rightarrow> expr \<Rightarrow> 'a postassn \<Rightarrow> bool"  ("\<Turnstile> (1_)/ (_)/ (1_)" 50)



text {* Here are some shorthands for constructing triples over an expression. *}

constdefs
  takestime  :: "nat \<Rightarrow> expr \<Rightarrow> bool"   (* uses an auxiliary variable! *)
  "takestime n e   \<equiv>  \<Turnstile> {(z,s). clock s = z} e {(z,s,v). clock s = z+n}"

  takestimeleq  :: "nat \<Rightarrow> expr \<Rightarrow> bool"
  "takestimeleq n e   \<equiv>  \<Turnstile> {(z,s). clock s = z} e {(z,s,v). clock s <= z+n}"
  
  takesspace :: "nat \<Rightarrow> expr \<Rightarrow> bool"
  "takesspace n e   
    \<equiv>  \<Turnstile> {(z,s). size (dom (heap s)) = z} e {(z,s,v). size (dom (heap s)) = z+n}"



text {* The following subsections define the basic rules with some trivial
  examples to exercise them. *}

(********************************************************************************)

subsection {* Consequence rules: HSP, HWC, HConseq, HConseqProc *}

(* strengthening precedent (SP) *)
lemma HSP :  "\<lbrakk> \<Turnstile> P' e Q; P \<subseteq> P' \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
apply(simp add: hoare_valid_def)
apply(auto)
done

(* weakening consequent (WC) *)
lemma HWC :  "\<lbrakk> \<Turnstile> P e Q'; Q' \<subseteq>  Q \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
apply(simp add: hoare_valid_def)
apply (auto)
done

(* doing both (consequence rule) *)
lemma HConseq: "\<lbrakk> P \<subseteq> P'; \<Turnstile> P' e Q'; Q' \<subseteq>  Q \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
apply(simp add: hoare_valid_def)
apply (fastsimp)
done

text {*  The following consequence rule is useful (allegedly) when
  reasoning about procedures. *}

lemma HConseqProc: "\<lbrakk> \<Turnstile> P' e Q'; 
		       \<forall> s t v. 
		        (\<forall> z. (z,s)\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			\<longrightarrow>
		        (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk>
		   \<Longrightarrow> \<Turnstile> P e Q"
apply (simp add: hoare_valid_def)
apply (fastsimp)
done

(*
lemma basic: "(\<forall> s. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>F s e,G s e\<rangle>) \<Longrightarrow>  \<Turnstile> {(z,s).(z,G s e,F s e) \<in> P } e P"
apply (simp add: hoare_valid_def)
apply (auto)
done 
*)

(********************************************************************************)

subsection {* HVoid *}

lemma HVoid: "\<Turnstile> {(z,s). (z,tickn 1 s, val.Void) \<in> Q} expr.Void Q"
apply (simp add: hoare_valid_def)
apply (auto)
done

lemma HVoid': "\<Turnstile> P expr.Void {(z,s,v). \<exists> s'. (z,s') \<in> P \<and> s=tickn 1 s' \<and> v=val.Void}"
apply (simp add: hoare_valid_def)
apply (auto)
done

(* Tests *)
lemma "\<Turnstile> {(z, s). True}
	  (expr.Void)
         {(z, s, result). result = val.Void}"
apply (rule HSP)
apply (rule HVoid)
apply (simp)
done

lemma "\<Turnstile> {(z, s). True}
	  (expr.Void)
         {(z, s, result). result = val.Void}"
apply (rule HWC)
apply (rule HVoid')
apply (auto)
done

lemma "takestime 1 expr.Void"
apply (simp add: takestime_def)
apply (rule HSP)
apply (rule HVoid)
apply (simp add: tickn_def)
done

lemma "takesspace 0 expr.Void"
apply (simp add: takesspace_def)
apply (rule HSP)
apply (rule HVoid)
apply (simp)
done

(********************************************************************************)

subsection {* HNull *}

lemma HNull: 
  "\<Turnstile> {(z, s). (z,tickn 1 s, val.Null)\<in> Q} expr.Null Q"
apply (simp add: hoare_valid_def)
apply (auto)
done


(********************************************************************************)

subsection {* HInt *}

lemma HInt: "\<Turnstile> {(z, s). (z,tickn 1 s,val.Int i) \<in> Q} (expr.Int i) Q"
apply (simp add: hoare_valid_def)
apply (auto)
done


(* Test HInt *)
lemma "\<Turnstile> {(z, s). True}
	  (expr.Int 5)
         {(z, s, result). result = val.Int 5}"
apply(rule HSP)
apply(rule HInt)
apply (simp)
done


(********************************************************************************)

subsection {* HVar *}


lemma HVar: "\<Turnstile> {(z, s). (z, tickn 1 s, the (s<vn>)) \<in> Q}
		(expr.Var vn) 
	        Q"
apply (simp add: hoare_valid_def)
apply (auto)
done

lemma HVarNothe: "\<Turnstile> {(z, s). \<exists> v. s<vn> = Some v \<and> (z, tickn 1 s, v) \<in> Q}
		  (expr.Var vn) 
	          Q"
apply (simp add: hoare_valid_def)
apply (auto)
done


lemma HVar': "\<Turnstile> P
		(expr.Var vn) 
	        {(z, s, v). \<exists> s'. (z,s') \<in> P \<and> s'<vn> = Some v \<and>  s=tickn 1 s'}"
apply (simp add: hoare_valid_def)
apply (auto)
done




(********************************************************************************)

subsection {* HPrimop *}

consts
 theival :: "val option \<Rightarrow> int"
recdef theival "measure(\<lambda> vo.1)"
 "theival  (Some (val.Int i)) = i"

lemma HPrimop: 
  "\<Turnstile> {(z, s). (z,tickn 3 s, val.Int (f (theival(s<vn1>)) (theival(s<vn2>)))) \<in> P}
      (expr.Primop f vn1 vn2)
      P"
apply (simp add: hoare_valid_def)
apply (auto)
done


(* in this rule we know post state preserves vars; but preconds strong for PC *)
lemma HPrimopNothe: "\<Turnstile> {(z, s). \<exists> i1 i2. s<vn1> = Some (val.Int i1) \<and>
			            s<vn2> = Some (val.Int i2) \<and>
				   (z,tickn 3 s, val.Int (f i1 i2)) \<in> P}
                  (expr.Primop f vn1 vn2)
		   P"
apply (simp add: hoare_valid_def)
apply (auto)
done

(* this version better for PC, in that we don't need to show vn1, vn2 bound
 before using it. *)
lemma HPrimop': "\<Turnstile> P
                  (expr.Primop f vn1 vn2)
		   {(z, s, v). \<exists> s' i1 i2. (z,s') \<in> P \<and>
					    s'<vn1> = Some (val.Int i1) \<and>
			                    s'<vn2> = Some (val.Int i2) \<and>
					    s = tickn 3 s' \<and>
					    v = val.Int (f i1 i2)}"
apply (simp add: hoare_valid_def)
apply (auto)
done


(* Test HPrimop *)

(* NB: assumption x1~=x2 is not used *)
lemma "x1 ~= x2 \<Longrightarrow>
	\<Turnstile> {(z, s). s<x1> = Some (val.Int 2) \<and> s<x2> = Some (val.Int 3)}
	   (Primop (op+) x1 x2)
         {(z, s, result). result = val.Int 5}"
apply (rule HSP)
apply (rule HPrimop)
apply (auto)
done

lemma "\<Turnstile> {((i,j), s). s<x1> = Some (val.Int i) \<and> s<x2> = Some (val.Int j)}
	   (Primop (op+) x1 x2)
         {((i,j), s, result). result = val.Int (i+j)}"
apply (rule HWC)
apply (rule HPrimop')
apply (auto)
done

(* Ask Isabelle how long a primop takes: *)
lemma "takestime ?k (expr.Primop f v1 v2)"
apply (simp add: takestime_def)
apply (rule HSP)
apply (rule HPrimop)
apply (simp add: tickn_def)
apply (auto)
done

lemma "takestime ?k (expr.Primop f v1 v2)"
apply (simp add: takestime_def)
apply (rule HWC)
apply (rule HPrimop')
apply (simp add: tickn_def)
apply (auto)
done



(********************************************************************************)

subsection {* HGetF *}

lemma HGetF: "\<Turnstile> {(z, s). \<exists>  a obj. s<vn> = Some(Ref a) \<and> 
			            heap s a = Some obj \<and> 
			            snd obj f = Some v \<and>
				    (z, tickn 2 s, v) \<in> P}
	        (GetF vn f)
		P"
apply (simp add: hoare_valid_def)
apply (auto)
done


(********************************************************************************)

subsection {* HPutF *}

lemma HPutF: 
  "\<Turnstile> {(z, s). 
         \<exists>  a obj rtv. 
		(s<vn> = Some(Ref a) \<and> 
			 heap s a = Some obj \<and> 
		         s<valv> = Some rtv \<and>
			 f mem (flds (classtable (fst obj))) \<and>
			 (z, tickn 3 (objfieldupdate s a obj f rtv), val.Void) \<in> P)}
	        (PutF vn f valv)
		P"
apply (unfold hoare_valid_def)
apply (auto)
done



(********************************************************************************)

subsection {* HPutF *}

lemma HNew: "\<Turnstile> {(z, s). \<exists> a. a = freshloc (dom (heap s)) \<and>
			(z, tickn 1 (newobj s a c), val.Ref a) \<in> P}
	        (New c)
		P"
apply (unfold hoare_valid_def)
apply (auto)
done

lemma HNew': "\<Turnstile> P
	        (New c)
		{(z, s, v). \<exists> a s'. (z, s') \<in> P \<and> 
				    a = freshloc (dom (heap s')) \<and>
				    s = tickn 1 (newobj s' a c) \<and> 
				    v = val.Ref a}"
apply (unfold hoare_valid_def)
apply (auto)
done

(* need to assume finiteness of heap here and also some lemmas about insert 
lemma "takesspace 1 (New c)"
apply (unfold takesspace_def)
apply (rule HWC)
apply (rule HNew')
apply (simp add: newobj_def freshloc_def)
apply (auto)
done
*)



(********************************************************************************)

subsection {* HIf *}


lemma HIf : "\<lbrakk> \<Turnstile> {(z,s). s<x>=Some (val.Int 1) \<and> (\<exists> s'. s=tickn 1 s' \<and> (z,s')\<in> P)} e Q;
	      \<Turnstile> {(z,s). s<x>=Some (val.Int 0) \<and> (\<exists> s'. s=tickn 1 s' \<and> (z,s')\<in> P)} e' Q \<rbrakk>
       \<Longrightarrow> \<Turnstile> P (IF x THEN e ELSE e') Q"
apply (unfold hoare_valid_def)
apply (fastsimp)
done

(* Test HIf *)

lemma "\<Turnstile> {(z,s). s<b> = Some (val.Int i) \<and> (i=0 \<or> i=1) \<and>
	       s<x1> = Some (val.Int 3) \<and> s<x2> = Some (val.Int 5)}
        (IF b THEN Var x1 ELSE Var x2)
         {(z, s, result). \<exists> j. result = val.Int j \<and> (j = 3 \<or> j =5)}"
apply (rule HIf)
apply (rule HSP)
apply (rule HVar)
apply (auto)
apply (rule HSP)
apply (rule HVar)
apply (auto)
done

lemma "takestime ?k (IF b THEN Var x ELSE Var y)"
apply (simp add: takestime_def)
apply (rule HIf)
apply (rule HWC)
apply (rule HVar')
apply (simp add: tickn_def)
apply (auto)
apply (rule HWC)
apply (rule HVar')
apply (simp add: tickn_def)
apply (auto)
done

(* This one is more of a challenge: takestimeleq guesses right only
   because the longer branch comes first.*)
lemma "takestimeleq ?k (IF b THEN (IF b' THEN Var x ELSE Var y) ELSE Var z)"
apply (simp add: takestimeleq_def)
apply (rule HIf)+
apply (rule HWC)
apply (rule HVar')
apply (simp add: tickn_def, auto)
apply (rule HWC)
apply (rule HVar')
apply (simp add: tickn_def)
apply (auto)
apply (rule HWC)
apply (rule HVar')
apply (simp add: tickn_def)
apply (auto)
done





(********************************************************************************)

subsection {* HLet *}

lemma HLet0: "\<lbrakk> \<Turnstile> T (e') Q;
                \<forall> z s v. (z,s,v) \<in> R \<longrightarrow> (z, varupdate (tickn 1 s) x v) \<in> T;
	        \<Turnstile> P e R
                \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
apply(simp add: hoare_valid_def)
apply(auto)
done


lemma HLet: "\<lbrakk> \<Turnstile> P e R;  
	       \<Turnstile> {(z,s).\<exists> v s'. (z,s',v) \<in> R \<and> s=varupdate (tickn 1 s') x v}  e' Q \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
apply(simp add: hoare_valid_def)
apply(auto)
apply (fastsimp)
done


(* Test HLet *)

(* Instantiation is messy...*)
lemma "\<Turnstile> {(z, s). True}
	 LET x=(expr.Int 5) IN expr.Var x END
         {(z, s, result). (result = val.Int 5)}"
apply (rule HLet)
apply (rule HSP)
apply (rule HInt)
apply (tactic {* instantiate_tac [("R","{(z,s,v). v=val.Int 5}")] *})
apply (simp)
apply (rule HSP)
apply (rule HVar)
apply (auto)
done

(* Here's a rule for doing the matching to instantiate automatically:
   at least in these simple cases. *)
lemma Hinstantiate:
  "\<lbrakk> R = {(z,s,v) . v=V \<and> (\<exists> s'. F s' = s)} \<rbrakk> \<Longrightarrow> 
   {(z, s). True} \<subseteq> {(z, s). (z, F s, V) \<in> R}"
apply (auto)
done

lemma "\<Turnstile> {(z, s). True}
	 LET x=(expr.Int 5) IN expr.Var x END
         {(z, s, result). (result = val.Int 5)}"
apply (rule HLet)
apply (rule HSP)
apply (rule HInt)
apply (rule Hinstantiate)
apply (simp)
apply (rule HSP)
apply (rule HVar)
apply (auto)
done


lemma "x~=y \<Longrightarrow> \<Turnstile> {(z, s). True}
	 LET x=(expr.Int 5); y=(expr.Int 7) IN  (Primop (op+) x y) END 
         {(z, s, result). (result = val.Int 12)}"
apply (rule HLet)
apply (rule HSP)
apply (rule HInt)
apply (rule Hinstantiate)
apply (simp)
apply (rule HLet)
apply (rule HSP)
apply (rule HInt)
apply (auto)
apply (rule HSP)
apply (rule HPrimop)
apply (auto)
oops

(*
apply (fastsimp)
apply (rule HLet)
prefer 3
apply (simp)

apply (tactic {* instantiate_tac [("R1","{(z,s,v). (EX i1. varupdate s y v<x> = Some (val.Int i1) \<and> (EX i2. v = val.Int i2 \<and> i1 + i2 = 12))}")] *})

apply (rule HLet)
apply (rule HPrimop)
apply (simp)
apply (simp)
apply (rule HInt)
apply (simp)
apply (tactic {* instantiate_tac [("R","{(z,s,v). varupdate (varupdate s x v) y (val.Int 7)<x> = Some (val.Int 5)}")] *})
apply (simp)
done
*)



(********************************************************************************)

lemma HCall: 
 assumes 
  PQcallbody:
   "(\<Turnstile> {(z,s).\<exists> s'. (z,s') \<in> P \<and> s=tickn 1 (incrcallcount (tickn 1 (incrcallcount s')))} 
   (CALL fn) Q) \<longrightarrow>
    (\<Turnstile> {(z,s).\<exists> s'. (z,s') \<in> P \<and> s=tickn 1 (incrcallcount s')} (funtable fn) Q)"
   (is "?PQCallsuc \<longrightarrow> ?PQbody")
  shows "\<Turnstile> P (CALL fn) Q"
proof -
  assume "?PQCallsuc \<longrightarrow> ?PQbody" 
  

qed

H: "(\<Turnstile> {(z,s).\<exists> s'. (z,s') \<in> P \<and> s=tickn 1 (incrcallcount (tickn 1 (incrcallcount s')))} 
             (CALL fn) Q) \<longrightarrow>  
             (\<Turnstile> {(z,s).\<exists> s'. (z,s') \<in> P \<and> s=tickn 1 (incrcallcount s')} (funtable fn) Q)"
  

apply (unfold hoare_valid_def)
apply (clarify)
apply (subgoal_tac "\<langle>?s,funtable fn\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow>
            \<forall>z. (z, ?s) \<in> {(z, s). \<exists>s'. (z, s') \<in> P \<and> s = tickn 1 (incrcallcount s')} \<longrightarrow>
                 (z, t, v) \<in> Q)")
apply (erule mp)
done


(* We want to apply hypothesis: make a subgoal from the result of doing that,
    instantiating. *)
apply (subgoal_tac "\<forall>s t v.
           \<langle>s,funtable fn\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow>
           (\<forall>z. (z, s) \<in> {(z, s). \<exists>s'. (z, s') \<in> P \<and> s = tickn 1 (incrcallcount s')} \<longrightarrow>
                (z, t, v) \<in> Q)")
(* Now use generation for the evaluation rule on CALL fn *)
(* Let's solve the subgoal first otherwise later autos will screw it up *)
prefer 2
apply (erule mp)


apply (erule evalCall_cases)
(* This gives: \<langle>tickn 1 (incrcallcount s),funtable fn\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> 
   Plug this instance of funtable fn evaluating into the subgoal above: *)
apply (subgoal_tac "(z, tickn 1 (incrcallcount s)) \<in> 
		     {(z, s). \<exists>s'. (z, s') \<in> P \<and> s = tickn 1 (incrcallcount s')} \<longrightarrow>
                     (z, t, v) \<in> Q")
(* pick this implication and solve subgoal *)
apply (erule mp)
apply (safe)
apply (rule_tac x = s in exI)




apply (subgoal_tac "\<forall>s t v.
           \<langle>s,funtable fn\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow>
           (\<forall>z. (z, s) \<in> P \<longrightarrow> (z, tickn (Suc 0) (incrcallcount t), v) \<in> Q)")
apply (erule evalCall_cases)
apply (subgoal_tac "(\<forall>z. (z, tickn 1 (incrcallcount s)) \<in> P \<longrightarrow> (z, tickn (Suc 0) (incrcallcount t), v) \<in> Q)")

done

(* FIXME: really need isar tactic that says apply assumption 1 to assumption 2 *)

(********************************************************************************)


subsection {* Interlude *}

consts disjvars :: "vname list \<Rightarrow> bool"
primrec
  "disjvars(x#xs) = (\<not> (x  mem xs) \<and> disjvars(xs))"
  "disjvars []    = True"








(*********************************************************************************)

section {* Validity within a context *}

text {* We formalise contextual validity, following Nipkow. 
  A context is a set of hoare triples. *}


constdefs set_valid :: "('a etriple) set \<Rightarrow> bool"   ("\<parallel>= _" 50)
   "\<parallel>= C  \<equiv> \<forall> (P,e,Q) \<in> C. \<Turnstile> P e Q"

constdefs conseqset_valid :: "('a etriple) set \<Rightarrow> ('a etriple) set \<Rightarrow> bool"   ("_ \<parallel>= _" 51)
   "C \<parallel>= D  \<equiv> (\<parallel>= C) \<longrightarrow> (\<parallel>= D)"

(* IDEA: think of having a map from function names to pre,post pairs here instead. *)

lemma HLiftCtxt: "(\<Turnstile> P e Q) \<Longrightarrow> ({} \<parallel>= {(P,e,Q)})"
by (unfold set_valid_def conseqset_valid_def, fastsimp)

lemma HSet: "(\<forall> (P,e,Q)\<in> D. (C \<parallel>= {(P,e,Q)}))  \<longrightarrow> (C \<parallel>= D)"
by (unfold set_valid_def conseqset_valid_def, auto)

lemma HSingle: "\<lbrakk> C \<parallel>= D ; (P,e,Q) \<in> D \<rbrakk> \<Longrightarrow> (C \<parallel>= {(P,e,Q)})"
by (unfold set_valid_def conseqset_valid_def, fastsimp)

lemma HEmptyCtxt: "({} \<parallel>= {(P, e, Q)}) \<Longrightarrow>  (\<Turnstile> P e Q)"
by (unfold set_valid_def conseqset_valid_def, fastsimp)


subsection {* Experiment with direct derivability of CALL rule *}

lemma HCallBase: "(P, CALL fn, Q) \<in> C \<longrightarrow> (C \<parallel>= {(P, CALL fn, Q)})"
by (unfold conseqset_valid_def set_valid_def, auto)


lemma HCall: 
 "({(P,CALL fn,Q)} \<parallel>= {(P, funtable fn, {(z,s,v). (z,tickn 1 (incrcallcount s),v)\<in> Q})})
	   \<Longrightarrow>  {} \<parallel>= {(P, CALL fn, Q)}"
apply (unfold conseqset_valid_def)
(*apply (erule_tac HCalln [then simp: add validiffvalidn])*)
sorry





subsection {*  Evalution within a restricted call depth *}

text {* Following Nipkow: the soundness of call and invoke rules
  requires induction on the maximum number of recursive calls.  The
  call counting is built into the operational semantics, so we
  just introduce a derived form for bounding number of recursions. *}

constdefs
  evalexprn     :: "[state, expr, nat, val, state] \<Rightarrow> bool" ("\<langle>_,_\<rangle> \<longrightarrow>_ \<langle>_,_\<rangle>")
  "\<langle>s,e\<rangle> \<longrightarrow>n \<langle>v,t\<rangle> == \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>  \<and> (callcount t <= callcount s + n)
					 \<and> (invokecount t <= invokecount s + n)"

lemma evaliffevaln: "(\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>)  = (\<exists> n. \<langle>s,e\<rangle> \<longrightarrow>n \<langle>v,t\<rangle>)"
apply (simp add: evalexprn_def)
apply (auto)
apply (erule evalexpr.induct)
apply (simp_all add: state_functions)
apply (auto)
apply (rule_tac x = "Suc n" in exI, simp)
apply (rule_tac x = "Suc n" in exI, simp)
apply (rule_tac x = "n + na" in exI, simp)
done

(* rather than generalising all the defs here, could we just 
restrict pre conditions to specifying the n ?  *)

constdefs
  hoare_validn :: "nat \<Rightarrow> 'a preassn \<Rightarrow> expr \<Rightarrow> 'a postassn \<Rightarrow> bool"  ("\<Turnstile>:_ (1_)/ (_)/ (1_)" 50)
   "\<Turnstile>:n P e Q \<equiv> \<forall> s t v. \<langle>s,e\<rangle> \<longrightarrow>n \<langle>v,t\<rangle> \<longrightarrow> 
				  (\<forall> z. (z, s) \<in>  P \<longrightarrow> (z, t, v) \<in> Q)" 

lemma validiffvalidn: "(\<Turnstile> P e Q) =  (\<forall> n. (\<Turnstile>:n P e Q))"
apply (unfold hoare_valid_def hoare_validn_def)
apply (simp add: evaliffevaln)
apply (auto)
done

constdefs
  set_validn :: "nat \<Rightarrow> ('a etriple) set \<Rightarrow> bool"   ("\<parallel>=:_ _" 50)
   "(\<parallel>=:n C)  \<equiv> \<forall> (P,e,Q) \<in> C. (\<Turnstile>:n P e Q)"

constdefs conseqset_validn :: "('a etriple) set \<Rightarrow> nat \<Rightarrow> ('a etriple) set \<Rightarrow> bool"   
			       ("_ \<parallel>=:_ _" 51)
   "C \<parallel>=:n D  \<equiv> (\<parallel>=:n C) \<longrightarrow> (\<parallel>=:n D)"


subsection {* The CALL rules *}


text {* The base case for proving derivability *}

lemma Call0: "\<not> \<langle>s,CALL fn\<rangle> \<longrightarrow>0 \<langle>v,t\<rangle>"
apply (simp add: evalexprn_def)
apply rule+
apply (erule evalCall_cases)
apply (simp add: incrcallcount_def tickn_def)
apply (insert callcount_mono [of "s\<lparr>callcount := Suc (callcount s), clock := Suc (clock s)\<rparr>" "funtable fn" v t])
apply (auto)
done


(* Here's the step case:
\<And>n na. {(P, CALL fn,
             Q)} \<parallel>=:na {(P, funtable fn,
                         {u. (\<lambda>(z, u). (\<lambda>(s, v). (z, tickn 1 (incrcallcount s), v) \<in> Q) u)
                              u})} \<longrightarrow>
           {} \<parallel>=:na {(P, CALL fn, Q)} \<Longrightarrow>
           {(P, CALL fn,
             Q)} \<parallel>=:Suc na {(P, funtable fn,
                             {u. (\<lambda>(z, u). (\<lambda>(s, v). (z, tickn 1 (incrcallcount s), v) \<in> Q) u)
                                  u})} \<longrightarrow>
           {} \<parallel>=:Suc na {(P, CALL fn, Q)}
*)


lemmas state_simps = tickntickn storetickn heaptickn getvartickn getvar_same getvar_other varupdate_tickn updatediff_commute storestoresimp updatesame maxstacktick maxstackfieldupd maxstacknew invokecount_mono callcount_mono maxstack_mono heap_mono


text {* The rule for restricted call depth. *}

lemma HCalln: 
"\<forall> n. ({(P,CALL fn,Q)} 
	  \<parallel>=:n {(P, funtable fn, {(z,s,v). (z,tickn 1 (incrcallcount s),v)\<in> Q})})
    \<longrightarrow>  {} \<parallel>=:n {(P, CALL fn, Q)}"
apply (rule)
apply (rule nat.induct)
apply (simp only: set_validn_def conseqset_validn_def hoare_validn_def)
apply (simp add: Call0)
apply (unfold set_validn_def conseqset_validn_def hoare_validn_def evalexprn_def)
apply (unfold tickn_def incrcallcount_def)
(* induction step *)
(*apply (simp assms only: tickn_def incrcallcount_def state_simps) *)
sorry





(* Phew!  This one is hairy but again says nothing. 
   We ask that the pre-condition of the method specifies the existence 
   of the object and the method in the class upon which it is being invoked.
   Again a table would seem better. *)

lemma HMethInd: 
"( (\<forall> (P,e,Q) \<in> C. \<exists> objn m a. (e = Invoke objn m a)) &
   (C \<parallel>= {(P,mbdy,Q) . 
	   (\<exists> objn m arg lvs. (P, Invoke objn m arg, Q) \<in> C 
	    \<and> (\<forall> z s. (z, s) \<in> P \<longrightarrow> 
	        (\<exists> a lvs obj. s<objn> = Some(Ref a) \<and> heap s a = Some obj \<and>
		              meths (classtable (fst obj)) mn = Some (lvs, mbdy))))}))
	\<longrightarrow> ({} \<parallel>= C)"
apply (simp add: set_valid_def conseqset_valid_def)
done

lemma HMethBase: "(P, Invoke objn m a, Q) \<in> C \<longrightarrow> (C \<parallel>= {(P, Invoke objn m a, Q)})"
apply (simp add: conseqset_valid_def)
done


(* ----------------------------------------------------------------- *)


(*First example:
  fun f(n,m) = let m = m+n
                   n = n -1
               in if n=0 then m else f(n,m)
  satisfies {\<lambda> [N,M] i s. s<n> = N & N > 0 & clock s = i}
            Result Call f
            {\<lambda> [N,M] i v s. clock s = i + factor N}
*) 
(*The program*)
consts fn::funame       
       m:: vname
       n:: vname
       k:: vname

constdefs iszero::"int \<Rightarrow> int"
"iszero x == (if x = 0 then 1 else 0)"

(*in the second and third primops, m is only used as a dummy*)
constdefs fnbody::expr
"fnbody == Let m (Primop (\<lambda> x y . x + y) m n)
           (Let n (Primop (\<lambda> x y . x - 1) n m)
           (Let k (Primop (\<lambda> x y. iszero x) n m)
           (If_ k (Var m) (Call fn))))"
constdefs factor::nat
"factor == 14"

(*"Axiom", also refining the const body*)
constdefs AXIOM:: "bool"
"AXIOM == n ~= m & m ~= n & m ~= k & k ~= m & n ~= k & k ~= n & funtable fn = fnbody"

(*Constructing the hoare triple - value N is a "global" input"*)
constdefs TRIPLE1::"nat \<Rightarrow> bool"
"TRIPLE1 N ==
   \<Turnstile> {(z,s). clock s=z  \<and>  s<n>=Some (val.Int (int N)) \<and> 0 < N}
           (Call fn)
     {(z,s,v). clock s = ((Suc z) + factor * N)}"

lemma HVacuous: "\<Turnstile> {(z,s). False} e P" 
apply (simp add: hoare_valid_def)
done


constdefs funIH :: "(nat \<times> nat) etriple"
  "funIH ==  ({((z,na), s). clock s = z \<and> s<n> = Some (val.Int (int na)) \<and> 0 < na}, 
	      CALL fn,
             {((z,na),s,v). clock s = Suc (z + factor * na)})"

(*Semantic validity of the triple -- independent from any input! Proof is by
  induction on the external parameter N"*)
lemma Triple1_Valid: "AXIOM \<Longrightarrow> TRIPLE1 N"
apply (induct_tac N)
apply (simp add: AXIOM_def TRIPLE1_def, rule HVacuous)
apply (simp add: AXIOM_def TRIPLE1_def)




apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def AXIOM_def fnbody_def varupdate_def iszero_def,
      auto)+
apply(case_tac "na = 0")
apply(auto)
apply(simp add: time_def factor_def)
apply(case_tac "na = 0")
apply(auto)
apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def fnbody_def varupdate_def iszero_def,
      auto)+
apply(case_tac "int na - 1 = 0")
apply(auto)
apply(case_tac "na")
apply(auto)
apply(simp add: time_def factor_def)
apply(case_tac "int na - 1 = 0")
apply(auto)
apply(case_tac "na")
apply(auto)






end
