theory ToyHL = ToyGrailDef:
(*Lennart's early experiments with reasoning about time\<dots>
Part I. definition of Hoare triples with auxiliary variables, for exprs and letexprs,
        and of corresponding validity predicates;
        some auxiliary functions for constructing such triples
Part II. Some very early Hoare rules, a la Nipkow's "Hoare logics in Isabelle/HOL" paper
         additional early rules for If and Let, and lemmas justifying them
Part III: Verification of a simple program in foundational style (i.e. not using
          the stuff from Part II), but abstracting from input
Part IV: HIGHLY experimental stuff, aiming at (amongst others) getting a derivation 
         for the example prog using the rules from Part II
*)

(*----------------------------PART I: assertions --------------------------*)
types preAss = "state \<Rightarrow> bool"
types postAss = "val \<Rightarrow> state \<Rightarrow> bool"

(*assertions with auxiliary variables/values*)
types
'a preAssAV = "'a \<Rightarrow> preAss"
'a postAssAV = "'a \<Rightarrow> postAss"

lemma updateStoreTwice[simp]: "s\<lparr>store := v\<rparr>\<lparr>store := w\<rparr> = s\<lparr>store := w\<rparr>"
apply(auto)
done

lemma updateClockTwice[simp]: "s\<lparr>clock := v\<rparr>\<lparr>clock := w\<rparr> = s\<lparr>clock := w\<rparr>"
apply(auto)
done

lemma updateMaxstackTwice[simp]: "s\<lparr>maxstack := v\<rparr>\<lparr>maxstack := w\<rparr> = s\<lparr>maxstack := w\<rparr>"
apply(auto)
done

lemma updateFramestackTwice[simp]: "s\<lparr>framestack := v\<rparr>\<lparr>framestack := w\<rparr> = s\<lparr>framestack := w\<rparr>"
apply(auto)
done

lemma updateHeapTwice[simp]: "s\<lparr>heap := v\<rparr>\<lparr>heap := w\<rparr> = s\<lparr>heap := w\<rparr>"
apply(auto)
done

(*Hoare triples for expressions and letexpressions, and their
  semantic validity (partial correctness)*)
datatype 'a Etriple = etriple "'a preAssAV" "expr" "'a postAssAV"

consts Evalid::"'a Etriple \<Rightarrow> bool"
primrec
"Evalid (etriple P e Q) = (\<forall> s t v. (\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. (P z s \<longrightarrow> Q z v t))))" 

datatype 'a Ltriple = ltriple "'a preAssAV" "letexpr" "'a postAssAV"
 
consts Lvalid:: "('a Ltriple) \<Rightarrow> bool"
primrec
"Lvalid (ltriple P l Q) = (\<forall> s t v. (\<langle>s,l\<rangle> \<longrightarrow>l \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. (P z s \<longrightarrow> Q z v t))))"

(*A simple predicate for time, auxiliary functions for producing simple
  Etriples for all expressions with the clock ticks as given in the op sem.
  Each constructed etriple is proven semantically valid.*)

constdefs time::"nat \<Rightarrow> state \<Rightarrow> bool"
"time i s == clock s = i"

constdefs timeVoid::"nat \<Rightarrow> 'a Etriple"
"timeVoid i == etriple (\<lambda> z. time i) (expr.Void) (\<lambda> z v. time (Suc i))"

lemma LemmaEvalidTimeVoid: "Evalid (timeVoid i)"
apply(simp add: timeVoid_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

constdefs timeNull::"nat \<Rightarrow> 'a Etriple"
"timeNull i == etriple (\<lambda> z. time i) (expr.Null) (\<lambda> z v. time (Suc i))"

lemma  LemmaEvalidTimeNull: "Evalid (timeNull i)"
apply(simp add: timeNull_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

constdefs timeInt::"int \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timeInt j i == etriple (\<lambda> z. time i) (expr.Int j) (\<lambda> z v. time (Suc i))"

lemma  LemmaEvalidTimeInt: "Evalid (timeInt j i)"
apply(simp add: timeInt_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

constdefs timeVar::"vname \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timeVar x i == etriple (\<lambda> z. time i) (expr.Var x) (\<lambda> z v. time (Suc i))"

lemma  LemmaEvalidTimeVar: "Evalid (timeVar x i)"
apply(simp add: timeVar_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

constdefs timePrimop::"(int \<Rightarrow> int \<Rightarrow> int) \<Rightarrow> vname \<Rightarrow> vname \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timePrimop f x y i == etriple (\<lambda> z. time i) (expr.Primop f x y) (\<lambda> z v. time (i + 3))"

lemma  LemmaEvalidTimePrimop: "Evalid (timePrimop f x y i)"
apply(simp add: timePrimop_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

(*constdefs time Invoke: to be done*)

constdefs timeGetF::"vname \<Rightarrow> fldname \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timeGetF x f i == etriple (\<lambda> z. time i) (expr.GetF x f) (\<lambda> z v. time (i + 2))"

lemma  LemmaEvalidTimeGetF: "Evalid (timeGetF x f i)"
apply(simp add: timeGetF_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

constdefs timePutF::"vname \<Rightarrow> fldname \<Rightarrow> vname \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timePutF x f y i == etriple (\<lambda> z. time i) (expr.PutF x f y) (\<lambda> z v. time (i + 3))"

lemma  LemmaEvalidTimePutF: "Evalid (timePutF x f y i)"
apply(simp add: timePutF_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def objfieldupdate_def)
done

constdefs timeNew::"cname \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"timeNew c i == etriple (\<lambda> z. time i) (expr.New c) (\<lambda> z v. time (Suc i))"

lemma  LemmaEvalidTimeNew: "Evalid (timeNew c i)"
apply(simp add: timeNew_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def newobj_def)
done

(*TimeEtriple combines all the above auxiliary functions.*)
consts TimeEtriple::"expr \<Rightarrow> nat \<Rightarrow> 'a Etriple"
primrec
"TimeEtriple expr.Void i = timeVoid i"
"TimeEtriple expr.Null i = timeNull i"
"TimeEtriple (expr.Int j) i = timeInt j i"
"TimeEtriple (expr.Var x) i = timeVar x i"
"TimeEtriple (expr.Primop f x y) i = timePrimop f x y i"
"TimeEtriple (expr.New c) i = timeNew c i"
"TimeEtriple (expr.GetF x f) i = timeGetF x f i"
"TimeEtriple (expr.PutF x f y) i = timePutF x f y i"
(*"TimeEtriple (expr.Invoke x m y) i = timeVoid i"*)


(*Semantic validity of TimeEtriple -- follows from the above lemmas*)
theorem TheoremEvalidTimeEtriple:"(\<forall> x m y. e ~= expr.Invoke x m y) \<longrightarrow> Evalid (TimeEtriple e i)"
apply(case_tac e)
apply(auto)
apply(simp_all add: LemmaEvalidTimeVoid LemmaEvalidTimeNull LemmaEvalidTimeInt 
                    LemmaEvalidTimeVar LemmaEvalidTimePrimop LemmaEvalidTimeNew 
                    LemmaEvalidTimeGetF LemmaEvalidTimePutF)
done

(*A more modular way of defining the time etriples is costE - the increments again
  stem from the op sem*)
consts costE::"expr \<Rightarrow> nat"
primrec
"costE expr.Void = 1"
"costE expr.Null = 1"
"costE (expr.Int i) = 1"
"costE (expr.Var x) = 1"
"costE (expr.Primop f x y) = 3"
"costE (expr.New c) = 1"
"costE (expr.GetF x fn) = 2"
"costE (expr.PutF x fn y) = 3"
(*"costE expr.Invoke x m y = 1"*)

(*A single function creates etriples accoring to costE*)
constdefs costEtriple::"expr \<Rightarrow> nat \<Rightarrow> 'a Etriple"
"costEtriple e i == etriple (\<lambda> z. time i) e (\<lambda> z v. time (i + costE e))"

(*In fact, the two constructions yield the same triples*)
lemma costEtripleEqualsTimeEtriple:
  "(\<forall> x m y. e ~= expr.Invoke x m y) \<longrightarrow> costEtriple e i = TimeEtriple e i" 
apply(rule)
apply(case_tac e)
apply(auto)
apply(simp_all add: costEtriple_def timeVoid_def timeNull_def 
                    timeInt_def timeVar_def timePrimop_def 
                    timeNew_def timeGetF_def timePutF_def)
done

(*Consequently, costEtriple also generates valid triples --
  before adding auxiliaries z to the type of etriples this proof was 
  MUCH shorter, essentially and just invoking TheoremEvalidTimeEtriple
  Why does this not work any longer?*)
theorem  TheoremEvalidCostEtriple:
  "(\<forall> x m y. e ~= expr.Invoke x m y) \<longrightarrow> Evalid (costEtriple e i)"
apply(clarify)
apply(case_tac e)
apply(simp_all add: costEtriple_def timeVoid_def timeNull_def 
                    timeInt_def timeVar_def timePrimop_def 
                    timeNew_def timeGetF_def timePutF_def time_def)
(*apply(insert costEtripleEqualsTimeEtriple  TheoremEvalidTimeEtriple)*)
apply(insert TheoremEvalidTimeEtriple)
apply(auto)
apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def newobj_def objfieldupdate_def)+
done

(*Another auxiliary function - created the default ltriple for Ret*)
constdefs timeRet::"vname \<Rightarrow> nat \<Rightarrow> (val list) Ltriple"
"timeRet x i == ltriple (\<lambda> vs. (time i)) (Ret x) (\<lambda> vs. (\<lambda> v. time (Suc i)))"

(*Semantic validity*)
lemma "Lvalid (timeRet x i)"
apply(simp add: timeRet_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done

(*The function cost is intended to allow one to expresses the clock change 
  depending on the initial values, like in cost [N,M] = 12 * N if N was the initial value of
  some variable.
  The programmner has to ensure that the two val lists mentioned in the type of 
  timeCall are of equal length. Often, this length will be the number of arguments of f!!*)

(* da: I think the next def is not quite right either *)

constdefs timeCall::"funame \<Rightarrow> nat \<Rightarrow> (val list \<Rightarrow> nat) \<Rightarrow> (val list) Ltriple"
"timeCall f i cost == ltriple (\<lambda> vs. (time i)) (Call f) (\<lambda> vs v. time (Suc (cost vs) + i))"

(*The following lemma probably doesn't hold due to wrong binding *)
lemma "\<forall> f b cost i. Lvalid (ltriple (\<lambda> vs s. funtable f = b) b (\<lambda> vs v. time (cost vs + i)))
       \<longrightarrow>
       Lvalid (timeCall f i cost)"
apply(simp add: timeCall_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def,
      auto)+
sorry

(*relate to cost of e??*)
(*constdefs timeLet::"vname \<Rightarrow> expr \<Rightarrow> letexpr \<Rightarrow> nat \<Rightarrow> (val list \<Rightarrow> nat) \<Rightarrow> (val list) Ltriple"
"timeLet x e l i cost == ltriple (\<lambda> vs. time i) (Let x e l) (\<lambda> vs v. time (Suc (cost vs) + i))"
lemma "(\<forall> x m y. e ~= Invoke x m y) \<longrightarrow> Lvalid (timeLet x e l i cost)"
*)



(*----------------Part II Nipkow-style Hoare logic--------------------*)

(*TS: triple set, used in both components of derived judgements*)
types TS = "((val list) Ltriple) set"

(*Auxiliary function used in rule CALL, see Nipkow paper*)
constdefs mkD:: "TS \<Rightarrow> TS"
"mkD C == {T . \<exists> P fn Q. (T = ltriple P (funtable fn) Q & ((ltriple P (Call fn) Q) \<in> C))}"

(*Lderiv is the relation modelling the Hoare derivation system, i.e. (C,D) \<in> Lderiv is the
  relation defined in Section 6.1 of Nipkow*)
consts Lderiv::"(TS \<times> TS) set"
(* In the rules Let and If we currently use semantic side conditions involving
   Evalid and Lvalid. Of course, these should later be replaced by Ederiv and Lderiv.
   By the above theorem, Eedriv already yields Evalid, so we could perform the
   replacement now.
   But for Lderiv we can't do that yet (that's the soundness proof), and the current
   formalistion is closer to the "Justification lemmas below".

   Also, one might later want to use a generalisation rule instead of implication of predicates.
   But we still need some implications for converting between post- and preconditions.*)
inductive Lderiv intros
Call: "\<lbrakk>\<forall> P l Q. ((ltriple P l Q) \<in> C \<longrightarrow> (\<exists> fn. (l= Call fn)));
        (C, (mkD C)) \<in> Lderiv\<rbrakk>
       \<Longrightarrow> 
       ({}, C) \<in> Lderiv"
Assumption: "\<lbrakk>ltriple P (Call fn) Q \<in> C\<rbrakk> \<Longrightarrow> (C, {ltriple P (Call fn) Q}) \<in> Lderiv"
Struct1: "\<lbrakk>\<forall> P l Q. (ltriple P l Q \<in> D \<longrightarrow> ((C, {ltriple P l Q}) \<in> Lderiv))\<rbrakk> \<Longrightarrow> (C,D) \<in> Lderiv"
Struct2: "\<lbrakk>(C,D) \<in> Lderiv; ltriple P l Q \<in> D\<rbrakk> \<Longrightarrow> (C, {ltriple P l Q}) \<in> Lderiv"
Let:  "\<lbrakk>(\<exists> P R T Q. 
        ((\<forall> z s. U z s \<longrightarrow> P z s) &
         (Evalid (etriple P e R)) & 
         (\<forall> z v s. R z v s \<longrightarrow> T z (varupdate s x v)) &
         (Lvalid (ltriple T l Q)) &
         (\<forall> z v s. Q z v s \<longrightarrow> S z v (tickn 1 s))))\<rbrakk>
       \<Longrightarrow> (C, {ltriple U (Let x e l) S}) \<in> Lderiv"
If: "\<lbrakk>(\<forall> z s. U z s \<longrightarrow> P z s);
      (Evalid (etriple P (expr.Var boolv) R));
      (\<forall> z s. R z (val.Int 1) s \<longrightarrow> T1 z s);
      (\<forall> z s. R z (val.Int 0) s \<longrightarrow> T2 z s);
      (Lvalid (ltriple T1 res1 Q1));
      (Lvalid (ltriple T2 res2 Q2));
      (\<forall> z v s. Q1 z v s \<longrightarrow> S z v (tickn 1 s));
      (\<forall> vs v s. Q2 vs v s \<longrightarrow> S vs v (tickn 1 s))\<rbrakk> 
     \<Longrightarrow> (C, {ltriple U (If_ boolv res1 res2) S}) \<in> Lderiv"

(*Justification of the Let rule -- might be the basis of a formal soundness theorem:*)
lemma "((\<forall> z s. U z s \<longrightarrow> P z s) &
         (Evalid (etriple P e R)) &
         (\<forall> z v s. R z v s \<longrightarrow> T z (varupdate s x v)) &
         (Lvalid (ltriple T l Q)) &
         (\<forall> z v s. Q z v s \<longrightarrow> S z v (tickn 1 s)))
       \<longrightarrow> Lvalid (ltriple U (Let x e l) S)"
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def varupdate_def)
apply(auto)
sorry

(*Justification of the If rule:*)
lemma "\<forall> res1 res2 e U S P R T1 T2 Q1 Q2.
       ( ((\<forall> z s. U z s \<longrightarrow> P z s) &
        (Evalid (etriple P (expr.Var boolv) R)) &
        (\<forall> z s. R z (val.Int 1) s \<longrightarrow> T1 z s) &
        (\<forall> z s. R z (val.Int 0) s \<longrightarrow> T2 z s) &
        (Lvalid (ltriple T1 res1 Q1)) &
        (Lvalid (ltriple T2 res2 Q2)) &
        (\<forall> z v s. Q1 z v s \<longrightarrow> S z v (tickn 1 s)) &
        (\<forall> z v s. Q2 z v s \<longrightarrow> S z v (tickn 1 s)))
       \<longrightarrow> Lvalid (ltriple U (If_ boolv res1 res2) S))"
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
sorry    (* da: very sorry, I have broken this rule: probably can be simplified
	    now instead of having expr.Var boolv, just lookup boolv in
            starting state.  I don't really understand the rule.  *)

(*apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done
*)

(*-------------------PART III: Foundational example ---------------------*)
(*First example:
  fun f(n,m) = let m = m+n
                   n = n -1
               in if n=0 then m else f(n,m)
  satisfies {\<lambda> [N,M] i s. s<n> = N & N > 0 & clock s = i}
            Result Call f
            {\<lambda> [N,M] i v s. clock s = i + factor N}
*) 
(*The program*)
consts fn::funame       
       m:: vname
       n:: vname
       k:: vname

constdefs iszero::"int \<Rightarrow> int"
"iszero x == (if x = 0 then 1 else 0)"

(*in the second and third primops, m is only used as a dummy*)
constdefs fnbody::letexpr
"fnbody == Let m (Primop (\<lambda> x y . x + y) m n)
           (Let n (Primop (\<lambda> x y . x - 1) n m)
           (Let k (Primop (\<lambda> x y. iszero x) n m)
           (If_ k (Ret m) (Call fn))))"
constdefs factor::nat
"factor == 14"

(*"Axiom", also refining the const body*)
constdefs AXIOM:: "bool"
"AXIOM == n ~= m & m ~= n & m ~= k & k ~= m & n ~= k & k ~= n & funtable fn = fnbody"

(*Constructing the hoare triple - value N is a "global" input"*)
constdefs TRIPLE1::"nat \<Rightarrow> nat \<Rightarrow> (val list) Ltriple"
"TRIPLE1 N i ==
   ltriple (\<lambda> z s. time i s \<and> s<n>=Some (val.Int (int N)) \<and> 0 < N)
           (Call fn)
           (\<lambda> z v s. time ((Suc i) + factor * N) s)"

(*Semantic validity of the triple -- independent from any input! Proof is by
  induction on the external parameter N"*)
lemma Triple1_Valid: "\<forall> i. AXIOM \<longrightarrow> Lvalid (TRIPLE1 N i)"
apply(induct_tac N)
apply(auto)
apply(simp add: TRIPLE1_def)
apply(simp add: TRIPLE1_def)
apply(auto)
apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def AXIOM_def fnbody_def varupdate_def iszero_def,
      auto)+
apply(case_tac "na = 0")
apply(auto)
apply(simp add: time_def factor_def)
apply(case_tac "na = 0")
apply(auto)
apply(erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def fnbody_def varupdate_def iszero_def,
      auto)+
apply(case_tac "int na - 1 = 0")
apply(auto)
apply(case_tac "na")
apply(auto)
apply(simp add: time_def factor_def)
apply(case_tac "int na - 1 = 0")
apply(auto)
apply(case_tac "na")
apply(auto)
sorry(*
apply(subgoal_tac "\<exists> x. (store (sa\<lparr>clock := clock sa + 3, clock := 4 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat))), clock := 7 + clock sa,
                clock := 8 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat)),
                clock := 11 + clock sa, clock := 12 + clock sa,
                store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0), clock := 13 + clock sa,
                clock := 16 + clock sa, clock := 17 + clock sa,
                store := store sa(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1))), clock := 20 + clock sa,
                clock := 21 + clock sa, store := store sa(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat)),
                clock := 24 + clock sa, clock := 25 + clock sa,
                store := store sa(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat))(k\<mapsto>val.Int 0),
                clock := 26 + clock sa\<rparr>) n = Some (val.Int x) & int nat = 1 + x)")
apply(clarsimp)
apply(rule)
apply(rule)
prefer 2
apply(case_tac "nat")
apply(auto)
apply(subgoal_tac "time ((Suc 26) + (clock sa) + factor factor * nat) s2a")
apply(simp add: time_def factor_def)
apply(simp add: time_def)
apply(auto)
apply(subgoal_tac 
  "clock s2a = Suc (clock ((sa\<lparr>clock := clock sa + 3, clock := 4 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat))), clock := 7 + clock sa,
                clock := 8 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat)),
                clock := 11 + clock sa, clock := 12 + clock sa,
                store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0), clock := 13 + clock sa,
                clock := 16 + clock sa, clock := 17 + clock sa,
                store := store sa(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1))), clock := 20 + clock sa,
                clock := 21 + clock sa, store := store sa(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat)),
                clock := 24 + clock sa, clock := 25 + clock sa,
                store := store sa(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat))(k\<mapsto>val.Int 0),
                clock := 26 + clock sa\<rparr>)) + factor * nat)")
apply(simp)
apply(subgoal_tac 
  "store (sa\<lparr>clock := clock sa + 3, clock := 4 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat))), clock := 7 + clock sa,
                clock := 8 + clock sa, store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat)),
                clock := 11 + clock sa, clock := 12 + clock sa,
                store := store sa(m\<mapsto>val.Int (i1 + (2 + int nat)))(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0), clock := 13 + clock sa,
                clock := 16 + clock sa, clock := 17 + clock sa,
                store := store sa(n\<mapsto>val.Int (1 + int nat))(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1))), clock := 20 + clock sa,
                clock := 21 + clock sa, store := store sa(k\<mapsto>val.Int 0)(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat)),
                clock := 24 + clock sa, clock := 25 + clock sa,
                store := store sa(m\<mapsto>val.Int (3 + (2 * int nat + i1)))(n\<mapsto>val.Int (int nat))(k\<mapsto>val.Int 0),
                clock := 26 + clock sa\<rparr>
         ) n = Some (val.Int (int nat))")
prefer 2
apply(simp)
apply(blast)
done
*)

(*Alternative triple, modelling N internally *)
constdefs TRIPLE2::"nat \<Rightarrow> vname Ltriple"
"TRIPLE2 i ==
   ltriple (\<lambda> z s. (\<exists> N. (time i s \<and> 
                           s<n>=s<z> \<and> s<z>=Some (val.Int (int N)) \<and> 
                           z ~= n \<and> z ~= m \<and> 0 < N)))
           (Call fn)
           (\<lambda> z v s. (\<exists> N. (s<z> = Some (val.Int (int N)) \<and> time ((Suc i) + factor * N) s)))"

(*A lemma which is needed in the following proof*)
lemma AuxVarsRemainUnaffected: 
  "\<forall> x v s w t. (x ~= n & x ~= m & s<x> = Some v & \<langle>s,Call fn\<rangle> \<longrightarrow>l \<langle>w,t\<rangle> \<longrightarrow> 
                 t<x> = Some v)"
(*should follow from a more general lemma about auxiliary variables:
the content of a auxiliary variable is never changed during program execution*)
sorry

(*Validity of second triple -- not yet quite working\<dots>*)
lemma "\<forall> i. AXIOM \<longrightarrow> Lvalid (TRIPLE2 i)"
apply(simp add: TRIPLE2_def)
apply(auto)
apply(rule)
apply(subgoal_tac "t<z> = Some (val.Int (int N)) \<and> time (Suc (i + factor * N)) t")
apply(assumption)
apply(auto)
apply(simp add: AXIOM_def)
apply(insert AuxVarsRemainUnaffected)
apply(blast)
apply(subgoal_tac "Lvalid (TRIPLE1 N i)")
prefer 2
apply(insert Triple1_Valid)
apply(auto)
apply(simp add: time_def factor_def)
(*apply(blast)*)
(*proof currently broken\<dots>*)
sorry

(*----------------------------Second example: Factorial, via method recursion-----------------*)
(*method FAC =
  let one = 1
      fun FACrecurse(param,one) = let p = param - one
                              r = invoke FAC(p) 
                              r = r * param
                          in r end
  in if param=0 then one else FACrecurse(param,one)
*)

consts FACrecurse::funame
       FACmname::mname
       FACcname::cname
       one:: vname
       r::vname
       p::vname
constdefs FACrecursebody::letexpr
"FACrecursebody == Let p (Primop (\<lambda> x y . x - y) param one)
                    (Let r (Invoke self FACmname p)
                    (Let r (Primop (\<lambda> x y. x * y) r param)
                    (Ret r)))"

constdefs FACbody::letexpr
"FACbody == Let one (expr.Int 1)
            (Let k (Primop (\<lambda> x y. iszero x) param one)
            (If_ k (Ret one) (Call FACrecurse)))"

constdefs FACmethbody::methbody
"FACmethbody == ({p,r,one},FACbody)"

(* note: could use this plus some rules *)
consts disjvars :: "vname list \<Rightarrow> bool"
primrec
  "disjvars(x#xs) = (\<not> (x  mem xs) \<and> disjvars(xs))"
  "disjvars []    = True"

constdefs FAC_AXIOM::"state \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> bool"
"FAC_AXIOM s i N == p ~= r & r ~= p & p ~= one & one ~= p & p ~= param & param ~= p & p ~= self & self ~= p & p ~= k & k ~= p &
                    r ~= one & one ~= r & r ~= param & param ~= r & r ~= self & self ~= r & r ~= k & k ~= r &
                    one ~= param & param ~= one & one ~= self & self ~= one & one ~= k & k ~= one &
                    param ~= self & self ~= param & param ~= k & k ~= param &
                    self ~= k & k ~= self &
                    funtable FACrecurse = FACrecursebody &
                    classtable FACcname = \<lparr>flds = [], meths = empty(FACmname \<mapsto> FACmethbody)\<rparr> &
                    heap s = empty(1 \<mapsto> (FACcname, empty)) &
                    s<self> = Some (Ref 1) & 
                    s<p> = Some (val.Int (int N)) &
                    clock s = i &
                    length (framestack s) <= maxstack s"

lemma LemmaFac0:
  "FAC_AXIOM s i 0 \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (v= val.Int 1 & t = s\<lparr> maxstack := max (Suc (length (framestack s))) (maxstack s), clock := (i + 12)\<rparr>)"
apply(clarify)
apply(simp add: FAC_AXIOM_def empty_def)
apply(clarify)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def FACbody_def FACrecursebody_def FACmethbody_def newframe_def,
      erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def varupdate_def,
      clarify)+
apply(simp)
done

consts facspec::"nat \<Rightarrow> nat"
recdef facspec "measure(\<lambda> N. N)"
"facspec 0 = 1"
"facspec (Suc N) = (facspec N) * (Suc N)"
lemma "facspec (Suc (Suc 0)) = 2 \<and> facspec (Suc (Suc (Suc 0))) = 6 \<and> facspec (Suc (Suc (Suc (Suc 0)))) = 24"
apply(auto)
done

lemma NatDistr1: "\<forall> a b c. ((a * (b + c)::nat) = (a * b + a * c))"
apply(auto)
apply(induct_tac a)
apply(auto)
done

lemma IntDistr1:" (int a) * (1 + (int c)) = (int a) + (int (a * c))"
sorry

lemma FACDistr1:
"int(facspec N) * (1 + int N) = (int (facspec N)) + (int ((facspec N) * N))"
sorry


constdefs XX::nat
"XX == 34"

lemma LemmaFac1:
  "FAC_AXIOM s i 1 \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (v= val.Int 1 & t = s\<lparr> maxstack := max (1 + Suc (length (framestack s))) (maxstack s), clock := (i + XX)\<rparr>)"
apply(clarify)
apply(simp add: FAC_AXIOM_def empty_def)
apply(clarify)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def FACbody_def FACrecursebody_def FACmethbody_def newframe_def,
      erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def varupdate_def,
      clarify)+
apply(auto)
apply(rule)
apply(simp_all add: max_def XX_def)
done

constdefs XX2::nat
"XX2 == 56"

lemma LemmaFac2:
  "FAC_AXIOM s i 2 \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (v= val.Int 2 & t = s\<lparr> maxstack := max (2 + Suc (length (framestack s))) (maxstack s), clock := (i + XX2)\<rparr>)"
apply(clarify)
apply(simp add: FAC_AXIOM_def empty_def)
apply(clarify)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def FACbody_def FACrecursebody_def FACmethbody_def newframe_def,
      erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def varupdate_def,
      clarify)+
apply(auto)
apply(rule)
apply(simp_all add:XX2_def max_def)
done

constdefs XX3::nat
"XX3 == 78"

lemma LemmaFac3:
  "FAC_AXIOM s i 3 \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (v= val.Int 6 & t = s\<lparr> maxstack := max (3 + Suc (length (framestack s))) (maxstack s), clock := (i + XX3)\<rparr>)"
apply(clarify)
apply(simp add: FAC_AXIOM_def empty_def)
apply(clarify)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def FACbody_def FACrecursebody_def FACmethbody_def newframe_def,
      erule evalexpr_evallet.elims,
      simp_all add: tickn_def get_var_def varupdate_def,
      clarify)+
apply(auto)
apply(rule)
apply(simp_all add:XX3_def max_def)
done

theorem TheoremFac:
  "\<forall> s i v t. 
   (FAC_AXIOM s i N \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (t = s\<lparr> maxstack := max (N + Suc (length (framestack s))) (maxstack s), 
              clock := ((i + 12) + 22 * N)\<rparr>))"
apply(induct_tac N)
apply(clarsimp)
apply(insert LemmaFac0)(* [of "s" "i" "v" "t"])*)
apply(simp)
(*Step case*)
apply(clarify)
apply(simp_all add: FAC_AXIOM_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def 
                    iszero_def FACbody_def FACrecursebody_def 
                    FACmethbody_def newframe_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
(*check the conditions for indhyp are satisfied*)
apply(subgoal_tac "heap (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) = empty(Suc 0\<mapsto>(FACcname, empty))")
prefer 2
apply(simp)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) self = Some (Ref (Suc 0))")
prefer 2
apply(simp)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) p = Some (val.Int (int n))")
prefer 2
apply(simp)
apply(subgoal_tac "length (framestack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)) \<le> maxstack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)")
(*now apply indhyp*)
apply(subgoal_tac "s1 = (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)\<lparr>maxstack := max (Suc (n + length (framestack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)))) (maxstack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)), clock := clock (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) + 12 + 22 * n\<rparr>")
(*first discharge the indhyp*)
prefer 2
apply(blast)
(*then discharge the last condition for the induction hypothesis*)
prefer 2
apply(simp)
apply(case_tac "Suc (length (framestack sa)) \<le> maxstack sa")
apply(simp add: max_def)
apply(simp add: max_def)
(*now prove the claim*)
apply(rotate_tac -6)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(rotate_tac -1)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(rule)
apply(simp_all add: max_def)
done

lemma facspecProperty: "int (facspec N) * (1 + int N) = int (facspec N) + int (facspec N * N)"
sorry

theorem TheoremFac:
  "\<forall> s i v t. 
   (FAC_AXIOM s i N \<longrightarrow> 
       \<langle>s, Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v, t\<rangle> \<longrightarrow> 
       (\<exists> NN. (v= val.Int NN & NN = int (facspec N) & 
          t = s\<lparr> maxstack := max (N + Suc (length (framestack s))) (maxstack s), 
              clock := ((i + 12) + 22 * N)\<rparr>)))"
apply(induct_tac N)
apply(clarsimp)
apply(insert LemmaFac0)(* [of "s" "i" "v" "t"])*)
apply(simp)
(*Step case*)
apply(clarify)
apply(simp_all add: FAC_AXIOM_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def 
                    iszero_def FACbody_def FACrecursebody_def 
                    FACmethbody_def newframe_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
(*check the conditions for indhyp are satisfied*)
apply(subgoal_tac "heap (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) = empty(Suc 0\<mapsto>(FACcname, empty))")
prefer 2
apply(simp)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) self = Some (Ref (Suc 0))")
prefer 2
apply(simp)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) p = Some (val.Int (int n))")
prefer 2
apply(simp)
apply(subgoal_tac "length (framestack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)) \<le> maxstack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)")
(*now apply indhyp*)
apply(subgoal_tac "rtv1 = val.Int (int (facspec n)) \<and>
                   s1 = (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)\<lparr>maxstack := max (Suc (n + length (framestack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)))) (maxstack (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)), clock := clock (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)),
                maxstack := max (Suc (length (framestack sa))) (maxstack sa), clock := Suc (Suc (clock sa)),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 6 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 11 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) + 12 + 22 * n\<rparr>")
(*first discharge the indhyp*)
prefer 2
apply(blast)
(*then discharge the last condition for the induction hypothesis*)
prefer 2
apply(simp)
apply(case_tac "Suc (length (framestack sa)) \<le> maxstack sa")
apply(simp add: max_def)
apply(simp add: max_def)
(*now prove the claim*)
apply(rotate_tac -6)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(rotate_tac -1)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(clarsimp)
apply(rule)
apply(simp add: facspecProperty)
apply(rule)
apply(simp_all add: max_def)
done


lemma
"later:"

constdefs FACfactor :: nat
"FACfactor == 12"

(*Constructing the hoare triple - value N is a "global" input"*)
constdefs FAC_TRIPLE::"nat \<Rightarrow> nat \<Rightarrow> (val list) Etriple"
"FAC_TRIPLE N i ==
   etriple (\<lambda> z s. time i s \<and> 
                   heap s 1 = Some(FACcname, empty) \<and> 
                   s<self> = Some(Ref 1) \<and> 
                   s<p> = Some (val.Int (int N)) \<and> 
                   0 <= N)
           (Invoke self FACmname p)
           (\<lambda> z v s. time (i + FACfactor + FACfactor * N) s)"

(*Semantic validity of the triple -- independent from any input! Proof is by
  induction on the external parameter N"*)
lemma Triple1_Valid: "\<forall> i. FAC_AXIOM \<longrightarrow> Evalid (FAC_TRIPLE N i)"
apply(induct_tac N)
apply(auto)
apply(simp add: FAC_TRIPLE_def)
apply(auto)
apply(simp add: FAC_AXIOM_def FACinitstate_def empty_def)
apply(auto)
apply(simp_all add: tickn_def get_var_def FACrecursebody_def FACmethbody_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(simp_all add: newframe_def  FACbody_def tickn_def get_var_def varupdate_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(case_tac "param = one")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(simp add: FACfactor_def time_def)
apply(simp add: FAC_TRIPLE_def)
apply(auto)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(simp add: newframe_def FAC_AXIOM_def FACmethbody_def FACrecursebody_def FACbody_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(case_tac "param = one")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
(*now the application of the induction hypothesis*)
apply(subgoal_tac "time ((7 + clock sa) + FACfactor + FACfactor * n) s1")
apply(simp add: time_def factor_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(case_tac "param = r")
apply(auto)
apply(case_tac "param = p")
apply(auto)
apply(simp add: FACfactor_def newframe_def time_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
apply(case_tac "self = p")
apply(auto)
apply(case_tac "self = one")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def iszero_def)
apply(auto)
oops
apply(simp add: FACfactor_def newframe_def time_def)
apply(auto)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)), clock := Suc (clock sa),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 4 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 7 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) p = Some (val.Int (int n))")
prefer 2
apply(simp)
apply(subgoal_tac "store (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)), clock := Suc (clock sa),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 4 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 7 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) self = Some (Ref (Suc 0))")
prefer 2
apply(simp)
apply(case_tac "self = param")
apply(simp)
apply(rule)
apply(simp)
apply(rule)
apply(rule)
apply(simp)
apply(case_tac "self = one")
apply(simp)
apply(simp)
apply(rule)
apply(simp)
apply(case_tac "self = p")
apply(simp)
apply(simp)
apply(subgoal_tac "heap (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)), clock := Suc (clock sa),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 4 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 7 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>) (Suc 0) = Some (FACcname, empty)")
prefer 2
apply(simp)
apply(subgoal_tac "time (7 + clock sa) (sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)), clock := Suc (clock sa),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 4 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 7 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)")
prefer 2
apply(simp add: time_def)
apply(blast)
apply(subgoal_tac "\<exists>v. \<langle>s,Invoke self FACmname p\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>)")
apply
apply


apply
(sa\<lparr>framestack :=
                (FACmname, dummyvar,
                 LET one = expr.Int 1; k = Primop (\<lambda>x y. if x = 0 then 1 else 0) param one
                 IN IF k THEN RETURN one  ELSE CALL FACrecurse END,
                 store sa) #
                framestack sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n)), clock := Suc (clock sa),
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1), clock := 4 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0), clock := 7 + clock sa,
                store := empty(self\<mapsto>Ref (Suc 0))(param\<mapsto>val.Int (1 + int n))(one\<mapsto>val.Int 1)(k\<mapsto>val.Int 0)(p\<mapsto>
                  val.Int (int n))\<rparr>)

apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def AXIOM_def fnbody_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(case_tac "m=n")
apply(auto)
apply(simp_all add: iszero_def)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(case_tac "na = 0")
apply(auto)
apply(simp add: time_def factor_def)
apply(case_tac "na = 0")
apply(auto)
apply(subgoal_tac "time ((Suc 9) + (clock sa) + factor * na) s2a")
apply(simp add: time_def factor_def)
apply(simp add: time_def)
apply(auto)
apply(subgoal_tac 
  "clock s2a = Suc (clock ((sa\<lparr>clock := clock sa + 3, store := store sa(m\<mapsto>val.Int (i1 + (1 + int na))), clock := 6 + clock sa,
                store := store sa(m\<mapsto>val.Int (i1 + (1 + int na)))(n\<mapsto>val.Int (int na)), clock := 9 + clock sa,
                store := store sa(m\<mapsto>val.Int (i1 + (1 + int na)))(n\<mapsto>val.Int (int na))(k\<mapsto>val.Int 0)\<rparr>)) + factor * na)")
apply(simp)
apply(subgoal_tac 
  "store (sa\<lparr>clock := clock sa + 3, store := store sa(m\<mapsto>val.Int (i1 + (1 + int na))), clock := 6 + clock sa,
                           store := store sa(m\<mapsto>val.Int (i1 + (1 + int na)))(n\<mapsto>val.Int (int na)), clock := 9 + clock sa,
                           store := store sa(m\<mapsto>val.Int (i1 + (1 + int na)))(n\<mapsto>val.Int (int na))(k\<mapsto>val.Int 0)\<rparr>
         ) n = Some (val.Int (int na))")
prefer 2
apply(simp)
apply(blast)
done
a

apply(simp_all add: varupdate_def tickn_def get_var_def)
apply(rule)
apply(case_tac "self = p")
apply(simp_all)
apply(rule)
apply(simp)
apply(simp)
apply(simp)
apply(rule)
apply(simp)
apply(simp)
apply(simp add: newframe_def FACbody_def)
apply(rule evalLet)
apply(rule evalInt)
apply(simp_all add: varupdate_def tickn_def get_var_def)
apply(rule evalLet)
apply(rule evalPrimop)
apply(simp_all add: varupdate_def tickn_def get_var_def)
apply(subgoal_tac "(param = one \<longrightarrow> 0 = 1) \<and> (param \<noteq> one \<longrightarrow> 0 = 0)")
apply(assumption)
apply(auto)
apply(simp_all add: iszero_def varupdate_def tickn_def get_var_def)
apply(rule evalIf_True)
apply(simp_all add: varupdate_def tickn_def get_var_def)
apply(rule evalRet)
apply(simp_all add: varupdate_def tickn_def get_var_def)
apply(rule evalRet)
apply(simp_all add: varupdate_def tickn_def get_var_def time_def)

apply(simp add: TRIPLE1_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def AXIOM_def fnbody_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def varupdate_def)
apply(auto)
apply(case_tac "m=n")
apply(auto)
apply(simp_all add: iszero_def)
apply(erule evalexpr_evallet.elims)

done

(*--------------------------------------------PART IV: Experimental stuff (commented out)---------*)
(*
lemma "AXIOM \<longrightarrow> ({},{fnInv i}) \<in> Lderiv"
apply(auto)
apply(simp add: fnInv_def)
apply(rule Call)
apply(auto)
apply(simp add: mkD_def AXIOM_def fnbody_def)
apply(auto)
apply(rule Let)
apply(rule)
apply(rule)
apply(rule)
apply(rule)
apply(rule)
apply(subgoal_tac 
  "\<forall>z s. funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> time i s \<and> (\<exists>N. s<n> = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N) \<longrightarrow>  funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and>
          time i s \<and> (\<exists>N. s<n> = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N)")
apply(assumption)
apply(clarify)
apply(rule)
apply(subgoal_tac 
  "Evalid (etriple (\<lambda>z s . funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> time i s \<and> (\<exists>N. s<n> = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N))
                   (Primop (op +) m n) 
                   (\<lambda>z v s . funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> time (i+3) s \<and> 
                    (\<exists>N. s<n> = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N)))") 
apply(assumption)
apply(simp add: time_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(rule)
apply(subgoal_tac 
  "\<forall>z v s. funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> time (i + 3) s \<and> 
           (\<exists>N. store s n = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N) \<longrightarrow>
           (\<lambda> z v s . (funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> time (i + 3) s \<and> 
                      (\<exists>N. store s n = Some (val.Int N) \<and> val.Int N = hd z \<and> 0 \<le> N))) 
           z v (s\<lparr>store := store s(m\<mapsto>v)\<rparr>)")
apply(assumption)
apply(simp add: time_def)
apply(rule)
apply(clarify)
apply(subgoal_tac "(\<lambda> z v s. funs s fn =
             Some (letexpr.Let m (Primop op + m n)
                    (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
                      (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))) \<and> 
                 (\<exists> N M. store s n = Some (val.Int M) \<and> 
                              val.Int N = hd z & 0 <= N & M + 1 = N &
                              time (i + factor * nat N) s)) Z v t")
apply(assumption)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(case_tac "m=n")
apply(auto)
apply(simp add:iszero_def time_def)
apply(case_tac "i1 - 1 = 0")
apply(auto)
prefer 2
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(simp add: iszero_def)
apply(clarify)
apply(case_tac "m=n")
apply(auto)
apply(simp add: time_def iszero_def)
apply(case_tac "i1 - 1 = 0")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(case_tac "-2 + i1 = 0")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def get_var_def)
apply(clarify)
apply(case_tac "-2 + i1 = 0")
apply(auto)
oops
apply(rule)
apply(rule)
apply(subgoal_tac "Evalid (etriple (time i) (Primop op + m n) (\<lambda> v s. time (i+3) s))")
apply(assumption)
prefer 2
apply(rule)

apply(clarify)
apply(simp add:Evalid_def)
apply(clarify)
apply(subgoal_tac "
(\<forall>vs s. ((time i) s \<and> (\<exists>N. s<n> = Some (val.Int N)) \<longrightarrow> (time i) s)) \<and>
       (Evalid (etriple (time i) (Primop (op +) m n) (\<lambda> v. time (i+3)))) \<and>
       (\<forall>vs v s. (\<lambda> v. (time (i+3)) v s \<longrightarrow> (time (i+3)) vs (s\<lparr>store := store s(m\<mapsto>v)\<rparr>))) \<and>
       Lvalid
        (ltriple (time (i+3))
          (letexpr.Let n (Primop (\<lambda>x y. x - 1) n m)
            (If_ (Primop (\<lambda>x y. ToyHL.iszero x) n m) (Ret m) (Call fn)))
          (\<lambda> Z v s. (hd Z = val.Int N & time (i + factor * nat N)))) \<and>
       (\<forall>vs v s. ((\<lambda> Z v s. (hd Z = val.Int N & time (i + factor * nat N))) vs v s \<longrightarrow> (time i) (tick s)))")
apply(auto)
apply(subgoal_tac "\<forall>vs s. time i s \<and> (\<exists>N. s<n> = Some (val.Int N)) \<longrightarrow> time i s")
apply(auto)
apply(rule Let)
apply(auto)
sorry
*)

(* some other earlier stuff\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>\<dots>*)
(*
constdefs Ex1triple::"nat \<Rightarrow> (val list) Ltriple"
"Ex1triple i == 
  ltriple (\<lambda> NM s. s<n> = Some (hd NM) & (\<exists> N. hd NM = val.Int (int N) & 0 <= N) & time i s)
          (Call f)
          (\<lambda> NM v s. (\<exists> N. (hd NM = val.Int (int N) & 0 <= N & time (i + factor * N) s)))"

lemma "Lvalid (Ex1triple i)"
apply(simp add: Ex1triple_def)
apply(auto)
sorry

consts preF1::"funame \<Rightarrow> nat \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> state \<Rightarrow> bool"
consts postF1::"funame \<Rightarrow> nat \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> val \<Rightarrow> state \<Rightarrow> bool"

constdefs timeCall1::"funame \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> nat \<Rightarrow> Rtriple"
"timeCall1 f x V i == rtriple (preF1 f i x V) (Call f) (postF1 f i x V)"

consts preF2::"funame \<Rightarrow> nat \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> state \<Rightarrow> bool"
consts postF2::"funame \<Rightarrow> nat \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> val \<Rightarrow> state \<Rightarrow> bool"

constdefs timeCall2::"funame \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> vname \<Rightarrow> val \<Rightarrow> nat \<Rightarrow> Rtriple"
"timeCall2 f x1 V1 x2 V2 i == 
   rtriple (preF2 f i x1 V1 x2 V2) (Call f) (postF2 f i x1 V1 x2 V2)"


"Axioms"
constdefs PREf2:: "bool"
"PREf2 == (preF2 fn = (\<lambda> i x N y M s . (s<x> = Some N & s<y> = Some M & time i s & (\<exists> NN. N = val.Int NN & 0 <= NN) & funs s fn = Some fnbody)))"

constdefs factor::nat
"factor == 14"

constdefs POSTf2::"bool"
"POSTf2 == (postF2 fn = (\<lambda> i x N y M V s. (\<exists> NN. N = val.Int NN & time (i + (factor * (nat NN))) s)))"

consts Rvalid::"Rtriple \<Rightarrow> bool"
primrec
"Rvalid (rtriple P r Q) = (\<forall> s t v. P s \<longrightarrow> \<langle>s,r\<rangle> \<longrightarrow>r \<langle>v,t\<rangle> \<longrightarrow> Q v t )"

lemma "PREf2 & POSTf2 & n ~=m  \<longrightarrow> Rvalid (timeCall2 fn n N m M i)"
apply(induct_tac N)
apply(simp_all add:PREf2_def)
apply(auto)
apply(simp_all add: timeCall2_def PREf2_def POSTf2_def)
apply(induct_tac "int")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(simp add: fnbody_def tickn_def)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(simp add: tickn_def get_var_def)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(simp add: tickn_def get_var_def)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(simp_all add: tickn_def get_var_def iszero_def time_def)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(simp_all add: tickn_def get_var_def iszero_def time_def)
apply(case_tac "m=n")
apply(auto)
apply(case_tac "i1a - 1 = 0")
apply(auto)
apply(simp add: factor_def)
what do here? simple arithmetic
prefer 2
oops
apply(auto)
apply(case_tac "nat i1a")
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(auto)

datatype 
Ltriple = ltriple preAss letexpr postAss

consts Lvalid::"Ltriple \<Rightarrow> bool"
primrec
"Lvalid (ltriple P e Q) = (\<forall> s t v. P s \<longrightarrow> \<langle>s,e\<rangle> \<longrightarrow>l \<langle>v,t\<rangle> \<longrightarrow> Q v t )"

lemma "(Rvalid (rtriple (\<lambda> s. Q (val.Int 1) s) r1 (\<lambda> i s. (R i (tick s)))) &
        Rvalid (rtriple (\<lambda> s. Q (val.Int 0) s) r2 (\<lambda> i s. (R i (tick s)))) &
        Evalid (etriple P e Q)
       ) \<longrightarrow> 
       Lvalid (ltriple P (If e r1 r2) R)"
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
 modulo the different implications we want to do this:
apply(subgoal_tac "Q (val.Int 1) s1")
apply(auto)
apply(subgoal_tac "Q (val.Int 0) s1")
apply(auto)

sorry

constdefs timeIf::"expr \<Rightarrow> res \<Rightarrow> res \<Rightarrow> nat \<Rightarrow> Ltriple"
"timeIf e r1 r2 i == ltriple (time i) (If e r1 r2) (\<lambda> v. time (i + 3))"

lemma "Evalid (timePutF x f y i)"
apply(simp add: timePutF_def time_def)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(simp_all add: tickn_def)
done



consts ederiv::"Etriple set"
I guess that ederivVar1 is better for total correctness, while ederivVar2 is for 
  partial one, as intended here. Both are sound
inductive ederiv intros
ederivVoid: "etriple (\<lambda> s. Q (val.Void) (tick s)) expr.Void Q : ederiv"
ederivNull: "etriple (\<lambda> s. Q (val.Null) (tick s)) expr.Null Q : ederiv"
ederivInt: "etriple (\<lambda> s. Q (val.Int i) (tick s)) (expr.Int i) Q : ederiv"
ederivVar1: "etriple (\<lambda> s. (\<exists> rtv. (s<v> = Some rtv & Q rtv (tick s)))) (Var v) Q : ederiv"
ederivVar2: "etriple (\<lambda> s. (\<forall> rtv. (Q rtv (tick s)))) (Var v) Q: ederiv"
ederivPrimop1: "etriple (\<lambda> s. (\<exists> i1 i2. (s<vn1> = Some(val.Int i1) & 
                                         s<vn2> = Some(val.Int i2) & 
                                         (Q (val.Int (f i1 i2)) (tickn 3 s)))))
                        (Primop f vn1 vn2) 
                        Q: ederiv"
ederivPrimop2: "etriple (\<lambda> s. (\<forall> i1 i2. (Q (val.Int (f i1 i2)) (tickn 3 s))))
                        (Primop f vn1 vn2) 
                        Q: ederiv"
ederivPrimop3: "etriple (\<lambda> s. (\<forall> i. (Q (val.Int i) (tickn 3 s))))
                        (Primop f vn1 vn2) 
                        Q: ederiv"
ederivCons: "\<lbrakk>\<forall> s. (P' s \<longrightarrow> P s); etriple P e Q:ederiv; \<forall> v s. (Q v s \<longrightarrow> Q' v s)\<rbrakk>
             \<Longrightarrow> etriple P' e Q':ederiv"

lemma eDerivSound: "tr: ederiv \<longrightarrow> Evalid tr"
apply(clarify)
apply(rule ederiv.induct)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
done

datatype 
Rtriple = rtriple preAss res postAss

consts Rvalid::"Rtriple \<Rightarrow> bool"
primrec
"Rvalid (rtriple P r Q) = (\<forall> s t v. P s \<longrightarrow> \<langle>s,r\<rangle> \<longrightarrow>r \<langle>v,t\<rangle> \<longrightarrow> Q v t )"

consts rderiv::"Rtriple set"
inductive rderiv intros
rderivExpr: "\<lbrakk>etriple P e Q: ederiv \<rbrakk> \<Longrightarrow> rtriple P (Expr e) Q : rderiv"
rderivCall: "\<lbrakk>needs context etriple \<rbrakk> \<Longrightarrow> rtriple P (Call f) Q: rderiv"

lemma rDerivSound: "tr: rderiv \<longrightarrow> Rvalid tr"
apply(clarify)
apply(rule rderiv.induct)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(subgoal_tac "Evalid (etriple P e1 Q)")
apply(auto)
apply(simp add: eDerivSound) [of "(etriple P e1 Q)"])
apply(auto)

apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
apply(erule evalexpr_evallet.elims)
apply(auto)
done

constdefs wpE:: "expr \<Rightarrow> postAss \<Rightarrow> preAss"
"wpE e Q == (\<lambda> s. \<forall> t v. \<langle>s, e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> Q v t)"

lemma wpE_lemma: "wpE expr.Void Q = (\<lambda> s. Q (val.Void) (tick s))"
apply(simp add: wpE_def)
apply(rule)
apply(rule)
sorry

theorem "Evalid tr \<longrightarrow> tr: ederiv"
apply(case_tac "tr")
apply(auto)
apply(insert wpE_lemma)
apply(simp add: wpE_def)
sorry
*)

end
