(*  
   File:	ToyGrailDef.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyGrailDef.thy,v 1.2 2003/06/24 23:25:05 da Exp $

   This is an attempt to reduce Grail to an absolute minimum to
   study and design the bytecode logic of resources.
*)   

header {* The definition of Toy Grail *}

theory ToyGrailDef = Finmap + Main:

text {* 
  Toy Grail is an attempt to reduce Grail to an absolute minimum to
 study and design the bytecode logic of resources. 

  Simplifications include:
  \begin{itemize}
   \item two types: integer and reference, distinguished in syntax
   \item much type information is lost
   \item no static fields 
   \item virutal methods have only a single parameter
   \item a single syntactic category for terms (function bodies)
   \item total maps to model local variables and objects
  \end{itemize}
*}


(****************************************************************************
 *
 * Section 2: SYNTAX
 *
 ****************************************************************************)


section {* Syntax of Toy Grail *}

subsection {* Prelude: Machine Model preliminaries *}

text {* First we declare some abstract types. *}

typedecl
  iname		 --  {* names of integer variables *}
typedecl
  rname		 --  {* names of reference variables *}

consts
  self     :: rname -- {* the self object *}
  param    :: rname -- {* the method parameter *}

typedecl
  funame	-- {* names of functions *}

typedecl 
  cname		-- {* names of classes *}

typedecl
  ifldname	-- {* names of integer fields *}

typedecl
  rfldname	-- {* names of reference fields *}

typedecl
  mname		-- {* names of methods *}

types
  locn = nat		    (* model locations as natural numbers\<dots> *)

(* moved up here because ref needed in RPrimop -- HWL *)

datatype ref = Nullref | Ref locn

constdefs
  freshloc  :: "locn set \<Rightarrow> locn"       (* so it's easy to define new *)
  "freshloc L == Suc (fold max 0 L)"
(*  "freshloc L == Suc (Max (insert 0 L))"  Isabelle 2003 version. *)

text {* The next property is an axiom for the moment: we will prove it
  when Isabelle 2003 introduces @text{Max} of a finite set. *}

axioms  
 freshloc : "finite L \<Longrightarrow> freshloc L \<notin> L"

subsection {* Syntax proper *}

text {* Simple expressions are the basic unit of execution,
  and denote values.  They are built from operators, constants,
  and previously computed values (names).  An expression may have a
  side effect, but there are no directly nested expressions.
  We model untyped expressions. 

  Compound expressions are built using let expressions.  
  A function body is a let expression.  A function ends with
  tail calls to other functions or method returns.  

  The natural thing to do is to have two syntactic categories
  of terms: one for simple expressions and one for compound
  expressions.  However this makes the whole development 
  more tedious; so instead we define a predicate
  @{text SimpleExpr} which restricts the operational
  semantics in the single case that matters (the
  binding in a let expressions).
*}

datatype expr =
   Null				  ("NULL")
 | Int	  int			   -- {* integer constant *}
 | IVar   iname			   -- {* integer variable *}
 | RVar   rname			   -- {* reference variable *}
 | Primop "int \<Rightarrow> int \<Rightarrow> int"	   -- {* arithmetic/boolean operation *}
	  iname iname
 | RPrimop "ref \<Rightarrow> ref \<Rightarrow> int"	   -- {* operations over references *}
	  rname rname
 | New    cname			  ("NEW _")
 | GetFi  rname ifldname	  ("_\<bullet>_" [65,1000] 65)
 | GetFr  rname rfldname	  ("_\<bullet>_" [65,1000] 65)
 | PutFi  rname ifldname iname	  ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | PutFr  rname rfldname rname	  ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | Invoke rname mname rname       ("_\<bullet>_'(_')" [70,1000,65] 61) 
 | InvokeStatic cname mname rname ("[class _]\<bullet>_'(_')" [70,1000,65] 61) 
 | Leti   iname expr expr        
 | Letr   rname expr expr        
 | If_    iname expr expr 	  ("(1IF _/ THEN _/ ELSE _)"  [0,0,0] 61)
 | Call   funame		  ("(CALL _)" [0] 60)

text {* In the method invocation, the second variable name is the actual
  parameter.  *}

text {* The following black magic is Isabelle gobbledygook to
  introduce some nice concrete syntax. *}

(* let syntax *)
nonterminals
 gletbind gletbinds

syntax
  "_Gbind"      :: "[id, expr] => gletbind"             ("(2_ =/ _)" [0,60] 100)
  "_Grefbind"   :: "[id, expr] => gletbind"             ("(2rf _ =/ _)" [0,60] 100)
  ""            :: "gletbind => gletbinds"              ("_")
  "_gbinds"     :: "[gletbind, gletbinds] => gletbinds" ("_;/ _")
  "_GLets"      :: "[gletbinds,  expr] => expr" ("(LET (_)/ IN (_) END)" 100)
(* FIXME: "LETR" is backward compatibility *)
  "_GLetsr"      :: "[gletbinds,  expr] => expr" ("(LETR (_)/ IN (_) END)" 100)

translations
  "_GLets (_gbinds b bs) e"  == "_GLets b (_GLets bs e)"
  "LET  v = e IN l END"      == "Leti v e l"
  "LET rf v = e IN l END"    == "Letr v e l"
  "LETR v = e IN l END"      == "Letr v e l"  (* backward compat *)



text {* Apart from being well-typed, a good program
  must only assign let-bound variables to simple expressions.  In
  particular, it must not have any non-tail calls, which can arise in
  the syntax above, for example, as @{text "LET v=CALL f IN e END"}.
  Programs that break this rule will get stuck in the operational
  semantics. *}

consts
  SimpleExpr :: "expr \<Rightarrow> bool"
primrec
  "SimpleExpr Null = True"
  "SimpleExpr (expr.Int i) = True"
  "SimpleExpr (IVar v) = True"
  "SimpleExpr (RVar v) = True"
  "SimpleExpr (Primop f u v) = True"
  "SimpleExpr (RPrimop f u v) = True"
  "SimpleExpr (New c) = True"
  "SimpleExpr (GetFi v f) = True"
  "SimpleExpr (GetFr v f) = True"
  "SimpleExpr (PutFi v f u) = True"
  "SimpleExpr (PutFr v f u) = True"
  "SimpleExpr (Invoke v m u) = True"
  "SimpleExpr (InvokeStatic c m u) = True"
  "SimpleExpr (Leti v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (Letr v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (If_ v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (Call f) = False"



subsection {* Functions *}

text {* A Grail function is notionally parameterised on
  a subset of the current variables in scope.  Variables
  not in the formal parameters should not be accessible
  in the body.  However, a new stack frame is not 
  allocated.  For now we model this simply by ignoring
  the formal parameters, so a Grail function is just
  its body, which is a expr.  

  A Grail method body is a collection of function bindings,
  together with a main body and a set of local variables used
  within.

  To simplify the representation for ToyGrail, we consider a
  global mapping from function names to function bodies (see later
  below, in class table).
  This means that we do not need to keep a record of the current
  program position (current method or current functions) in the state.

  Conceptually, methods are parameterised on two things: first the
  self object (self) invoking the method, and then the single formal
  parameter.  Both parameter names are fixed and left implicit.

  When a method is invoked, we might want to know the size of the frame
  to calculate the size of the stack, which is why we record the 
  total set of local variables.
  (Although we do not initialize these).
 *}

types
  methbody = "(iname set \<times> rname set) \<times> expr"

translations
  "methbody" <= (type) "(iname set \<times> rname set) \<times> expr"




(****************************************************************************
 *
 * Section 2: MACHINE MODEL
 *
 ****************************************************************************)


section {* Machine model *}


text {* 
  Objects consist of a class name together with a mapping of
  field names to values; object initialization will simply
  set this mapping to be empty, it is a static condition
  that initialization is followed by putfields to the
  fields specified in the class.

  The heap is a map from locations to objects; the domain
  of a heap has to be available for reasoning.  
  Stores are also modelled as maps rather than total 
  functions; this is because we need to intitialize
  stores during method invocation; we model a language
  with static scoping.  

  A frame is a method name (which can
  be used for profiling), together with a return point
  (@{text expr}), a place to store the return value
   (@{text vname}) and the store of the calling frame.
 *}



types 
  obj      = "cname \<times> (ifldname \<Rightarrow> int) \<times> (rfldname \<Rightarrow> ref)"
  heap     = "locn \<leadsto>\<^sub>f obj"             (* NB: heaps should be finite *)
  istore   = "iname \<Rightarrow> int"	       (* integer store *)
  rstore   = "rname \<Rightarrow> ref"            (* reference store *)
  store    = "istore \<times> rstore"
  frame    = "mname \<times> store"

text {* Fold up some of the type definitions during printing. *}

(* NB: these translations take place on fully expanded types *)
translations
  "obj"	   <= (type) "cname \<times> (ifldname \<Rightarrow> int) \<times> (rfldname \<Rightarrow> ref)"
  "heap"   <= (type) "nat \<leadsto>\<^sub>f obj"
  "istore"  <= (type) "iname \<Rightarrow> int"
  "rstore"  <= (type) "rname \<Rightarrow> ref"
  "store"   <= (type) "istore \<times> rstore"
  "frame"   <= (type) "mname \<times> store"


(* the maxstack could be a more accurate measurement of the size of the stack? *)

text {* The state contains several resource-aware components:
  \begin{itemize}
  \item maxstack
  \item callcount
  \item invokecount
  \item clock
  \end{itemize}
*}

record state =
	heap        :: heap		 
	istore      :: istore	         -- {* integer store on top of stack *}
	rstore      :: rstore	         -- {* ref store on top of stack     *}
        framestack  :: "frame list"      -- {* frame stack		     *}
        maxstack    :: int		 -- {* maximum depth of stack        *}  
        callcount   :: int		 -- {* count of CALL depth	     *}  
        invokecount :: int		 -- {* count of INVOKE depth	     *}  
	clock	    :: int		 -- {* instruction counter	     *}

translations
  "state" <= (type) "\<lparr> heap::heap, istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int\<rparr>"
  "state" <= (type) "\<lparr> heap::heap, istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int,\<dots>::'a \<rparr>"






subsection {* Programs *}

text {* The class table represents the program as a global parameter. 
   When doing proofs we can make assumptions about the
   definitions of particular classes. 
   By convention, static and dynamic methods do not have the same name. *}

record class =
	iflds :: "ifldname list"
	rflds :: "rfldname list"
	meths :: "mname \<Rightarrow> methbody"

translations
  "class" <= (type) "\<lparr> iflds::ifldname list, rflds::rfldname list,
		       meths::mname \<Rightarrow> methbody \<rparr>"
  "class" <= (type) "\<lparr> iflds::ifldname list, rflds::rfldname list,
		       meths::mname \<Rightarrow> methbody, \<dots>::'a \<rparr>"


consts
  classtable   ::  "cname \<Rightarrow> class"
  methtable     :: "mname \<Rightarrow> expr"
  funtable     ::  "funame \<Rightarrow> expr"
  

subsection {* Syntactic abbreviations for accessing the state *}

syntax
 get_ivar     :: "state \<Rightarrow> iname \<Rightarrow> int"    ("_<_>" [65,68] 65)
translations
 "get_ivar s x" == "istore s x"

syntax
 get_rvar     :: "state \<Rightarrow> rname \<Rightarrow> ref"    ("_\<lfloor>_\<rfloor>" [65,68] 65)
translations
 "get_rvar s x" == "rstore s x"

syntax
 get_obj  :: "state \<Rightarrow> locn \<Rightarrow> obj option"  ("_\<lless>_\<ggreater>" [65,68] 65)
translations
 "get_obj s l" == "fmap_lookup (heap s) l"

subsection {* Functions for updating the state *}

constdefs
 tickn	     :: "int \<Rightarrow> state \<Rightarrow> state"
 "tickn n s \<equiv>  s \<lparr> clock := (clock s) + n \<rparr>"
syntax
 tick	     :: "state \<Rightarrow> state"
translations
 "tick s"    == "tickn 1 s"


constdefs
  ivarupdate :: "state \<Rightarrow> iname \<Rightarrow> int \<Rightarrow> state"  ("_<_:=_>" [1000,1000,0] 1000)
  "ivarupdate s v val \<equiv> (s \<lparr> istore := (istore s)(v := val) \<rparr>)"

constdefs
  rvarupdate :: "state \<Rightarrow> rname \<Rightarrow> ref \<Rightarrow> state"  ("_\<lfloor>_:=_\<rfloor>" [1000,1000,0] 1000)
  "rvarupdate s v val \<equiv> (s \<lparr> rstore := (rstore s)(v := val) \<rparr>)"


text {* 
  Object initialization simply sets an object with an empty map.  To
  do something more sensible we would need to record type information
  for fields (at least integer vs object).  Instead we may assume that
  the programmer correctly initializes fields immediately after making
  an object, as a condition for static correctness.
 *}


constdefs
  (* DA note: do these empty items actually need definitions?  
     We never want to look up in them. *)
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"
 
  emptyobj  :: "cname \<Rightarrow> cname \<times> (ifldname \<Rightarrow> int) \<times> (rfldname \<Rightarrow> ref)"
  "emptyobj c  \<equiv> (c, emptyi, emptyr)"

  newobj    :: "state \<Rightarrow>  locn \<Rightarrow> cname \<Rightarrow> state"
  "newobj s a c \<equiv> s \<lparr> heap := (heap s)(a \<mapsto>\<^sub>f emptyobj c)\<rparr>"

constdefs
  obj_ifieldupdate :: "state \<Rightarrow> locn \<Rightarrow> obj \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> state"  ("_<_\<bullet>_:=_>" [1000,80,1000,80] 1000)
  "obj_ifieldupdate s a obj f rtv \<equiv> s \<lparr> heap := (heap s)(a \<mapsto>\<^sub>f (fst obj, 
					                       (fst (snd obj))(f:=rtv),
							       (snd (snd obj)))) \<rparr>"

  obj_rfieldupdate :: "state \<Rightarrow> locn \<Rightarrow> obj \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> state"  ("_\<lfloor>_\<bullet>_:=_\<rfloor>" [1000,80,1000,80] 1000)
  "obj_rfieldupdate s a obj f rtv \<equiv> s \<lparr> heap := (heap s)(a \<mapsto>\<^sub>f (fst obj, 
					                       fst (snd obj),
							       (snd (snd obj))(f:=rtv))) \<rparr>"

text {* A method invocation allocates a new frame on the frame stack. 
        This function adjusts the state accordingly, given a reference
        to the invoking object, and the parameter.
        The new store contains only bindings for the self object
	and the method parameter.  (We might also give initial
        values to the local variables for the method).

	Note that if we are invoking a static method, then the
	self variable will be set to null.
*}

constdefs
  newframe :: "state \<Rightarrow> mname \<Rightarrow> ref \<Rightarrow> ref \<Rightarrow> state"
  "newframe s m objref arg \<equiv> 
	s \<lparr> framestack := (m,(istore s, rstore s))#(framestack s),
	    istore := emptyi,
	    rstore := ((emptyr(self := objref))(param := arg)),
            maxstack := max ((int (length (framestack s))) + 1) (maxstack s) \<rparr>"

constdefs
  incrcallcount :: "state \<Rightarrow> state"
  "incrcallcount s \<equiv> s \<lparr> callcount := (callcount s)+1 \<rparr>"

constdefs
  incrinvokecount :: "state \<Rightarrow> state"
  "incrinvokecount s \<equiv> s \<lparr> invokecount:= (invokecount s)+1 \<rparr>"


subsection {* Initialisation *}

text {* Initial values values for all structures in the state. Use these to be
 independent of the representation. *}

constdefs
  emptyIstore    :: "iname \<Rightarrow> int"
  "emptyIstore \<equiv> emptyi"

  emptyRstore    :: "rname \<Rightarrow> ref"
  "emptyRstore \<equiv> emptyr"

  emptyHeap    :: heap
  "emptyHeap \<equiv> emptyfinmap"

  emptyFramestack :: "frame list"
  "emptyFramestack \<equiv> []"

  emptyFS :: "frame list"
  "emptyFS \<equiv> []"

  emptyState :: state
  "emptyState \<equiv> \<lparr>
	heap        =  emptyHeap,
	istore      =  emptyIstore,
	rstore      =  emptyRstore,
        framestack  =  [],
        maxstack    =  0,
        callcount   =  0,
        invokecount =  0,
	clock	    =  0
   \<rparr>"

declare  emptyIstore_def    [simp]
declare  emptyRstore_def    [simp]
declare  emptyHeap_def      [simp]
declare  emptyFramestack_def [simp]
declare  emptyFS_def [simp]
declare  emptyState_def [simp]
                  
(* loops in evalexpr if we use that; what a great system
translations
  "emptyIstore" => "emptyi"
  "emptyRstore" => "emptyr"
  "emptyHeap" => "emptyfinmap"
  "emptyFramestack" => "[]"
  "emptyFS" => "[]"
*)

(****************************************************************************
 *
 * Section 3: NATURAL SEMANTICS
 *
 ****************************************************************************)

section {* Big-step semantics *}

subsection {* Inductively defined set *}

datatype val = IVal int | RVal ref

consts
  evalexpr     :: "(state \<times> expr \<times> val \<times> state) set"

syntax
 evalexpr_  :: "[state, expr, val, state] \<Rightarrow> bool"      ("\<langle>_,_\<rangle> \<longrightarrow>e \<langle>_,_\<rangle>")

translations
 "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>" == "(s,e,v,s') : evalexpr"


subsection {* Evaluation relation *}

inductive evalexpr intros
 evalNull:    "\<langle>s, expr.Null\<rangle>    \<longrightarrow>e  \<langle>RVal Nullref, tick s\<rangle>"

 evalInt:     "\<langle>s, expr.Int i\<rangle> \<longrightarrow>e \<langle>IVal i, tick s\<rangle>"

 evalIVar:    "\<langle>s, IVar v\<rangle>   \<longrightarrow>e  \<langle>IVal (s<v>), tick s\<rangle>"

 evalRVar:    "\<langle>s, RVar v\<rangle>   \<longrightarrow>e  \<langle>RVal (s\<lfloor>v\<rfloor>), tick s\<rangle>"

 evalPrimop:  "\<langle>s, Primop f vn1 vn2\<rangle>   \<longrightarrow>e  \<langle>IVal (f (s<vn1>) (s<vn2>)), tickn 3 s\<rangle>"

 evalRPrimop: "\<langle>s, RPrimop f vn1 vn2\<rangle>  \<longrightarrow>e  \<langle>IVal (f (s\<lfloor>vn1\<rfloor>) (s\<lfloor>vn2\<rfloor>)), tickn 3 s\<rangle>"

-- {* To explain the clock count here: vn1 = 1, vn2 = 1, frame
   push/pop = 2, implicit return = 1. 
   FIXME: could break up increment of ticks into before/after call. *}

 evalInvoke: 
  "\<lbrakk> s\<lfloor>vn1\<rfloor> = Ref a; 
     fmap_lookup (heap s) a = Some obj;
     \<langle>newframe (incrinvokecount s) mn (Ref a) (s\<lfloor>vn2\<rfloor>), 
     snd (meths (classtable (fst obj)) mn)\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
    \<langle>s , Invoke vn1 mn vn2\<rangle> \<longrightarrow>e   
    \<langle>rtv, tickn 5 (s'\<lparr>istore := istore s,  rstore := rstore s, 
	              framestack := (framestack s) \<rparr>) \<rangle>"
 evalInvokeStatic: 
  "\<lbrakk> \<langle>newframe (incrinvokecount s) mn Nullref (s\<lfloor>vn2\<rfloor>), 
     snd (meths (classtable c) mn)\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
    \<langle>s , InvokeStatic c mn vn2\<rangle> \<longrightarrow>e   
    \<langle>rtv, tickn 4 (s'\<lparr>istore := istore s,  rstore := rstore s, 
	              framestack := (framestack s) \<rparr>) \<rangle>"

 evalGetFi: "\<lbrakk> s\<lfloor>vn\<rfloor> = Ref a; 
              fmap_lookup (heap s) a = Some obj; 
              fst (snd obj) f = rtv \<rbrakk>
            \<Longrightarrow>
            \<langle>s,  GetFi vn f\<rangle> \<longrightarrow>e \<langle>IVal rtv, tickn 2 s\<rangle>"

 evalGetFr: "\<lbrakk> s\<lfloor>vn\<rfloor> = Ref a; 
              fmap_lookup (heap s) a = Some obj; 
              snd (snd obj) f = rtv \<rbrakk>
            \<Longrightarrow>
            \<langle>s,  GetFr vn f\<rangle> \<longrightarrow>e \<langle>RVal rtv, tickn 2 s\<rangle>"

(* Could add condition below to check for definedness of fieldname f in the 
  definition of obj's (dynamic) class:  
              f mem (fst (flds (classtable (fst obj))))  *)

 evalPutFi: "\<lbrakk> s\<lfloor>vn1\<rfloor> = Ref a; 
              fmap_lookup (heap s) a = Some obj;
              s<vn2> = i \<rbrakk> 
            \<Longrightarrow>
            \<langle>s, PutFi vn1 f vn2\<rangle> \<longrightarrow>e \<langle> IVal i, tickn 3 (obj_ifieldupdate s a obj f i) \<rangle> "

 evalPutFr: "\<lbrakk> s\<lfloor>vn1\<rfloor> = Ref a; 
              fmap_lookup (heap s) a = Some obj;
              s\<lfloor>vn2\<rfloor> = r \<rbrakk> 
            \<Longrightarrow>
            \<langle>s, PutFr vn1 f vn2\<rangle> \<longrightarrow>e \<langle> RVal r, tickn 3 (obj_rfieldupdate s a obj f r) \<rangle> "

 evalNew:  "\<lbrakk> a = freshloc (fmap_dom (heap s)) \<rbrakk> 
            \<Longrightarrow>
            \<langle>s, New c\<rangle> \<longrightarrow>e \<langle>RVal (Ref a), tick (newobj s a c) \<rangle>"

 evalCall: "\<lbrakk> \<langle>tick (incrcallcount s), funtable fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
            \<Longrightarrow>
            \<langle>s, Call fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>" 

-- {* The rule for evaluating Let contains an additional side condition
      that the expression being bound does not itself contain a CALL (non-tail
      position). *}

 evalLeti: "\<lbrakk> \<langle>s, e\<rangle> \<longrightarrow>e \<langle>IVal i, s1\<rangle> ; 
	      \<langle>ivarupdate (tick s1) vn i, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>;
	     SimpleExpr e \<rbrakk>
           \<Longrightarrow>
           \<langle>s, Leti vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

 evalLetr: "\<lbrakk> \<langle>s, e\<rangle> \<longrightarrow>e \<langle>RVal r, s1\<rangle> ; 
	      \<langle>rvarupdate (tick s1) vn r, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>;
	     SimpleExpr e \<rbrakk>
           \<Longrightarrow>
           \<langle>s, Letr vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

-- {* we might tickn 2 in the If rules, but lets leave it at one to
   allow for extra store/retrieve which may not be necessary in 
   some cases on the real machine. *}

 evalIf_True: "\<lbrakk> s<v> = 1; \<langle>tick s, l1\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
               \<Longrightarrow>
               \<langle>s, If_ v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>"

 evalIf_False: "\<lbrakk> s<v> = 0; \<langle>tick s, l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
               \<Longrightarrow>
               \<langle>s, If_ v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv,s1\<rangle>"

end 
