(*  
   File:	ToyGrailLemmas.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyGrailLemmas.thy,v 1.1 2003/05/22 21:30:36 a1hloidl Exp $

   Useful properties, simplification rules for ToyGrail.
*)   


header  {* Toy Grail Lemmas *}

theory ToyGrailLemmas = ToyGrailDef:

subsection {* Trivial lemma about heaps *}

lemma [simp]: "finite (fmap_dom (heap s))"
apply auto
done


subsection {* Some rules for simplifying states *}

text {* To maintain the abstract constants for updating states
  rather than immediately expanding them, we need a bunch of  
  simplifications rules for simplifying state expressions.
*}
 
(* NB: The 9 state updaters are: 

   clock:  tickn 
   stores: ivarupdate rvarupdate
   heap:   newobj obj_ifieldupdate obj_rfieldupdate
   frames: newframe incrcallcount incrinvokecount
*)


text {* First, collect together the definitions of semantic functions. *}

lemmas state_functions [simp] = 
   tickn_def ivarupdate_def rvarupdate_def newobj_def 
   obj_ifieldupdate_def obj_rfieldupdate_def 
   newframe_def incrcallcount_def incrinvokecount_def


subsubsection {* Tick *} 

text {* We pull all ticks to the outside, and sum them. *}

lemma [simp]: "tickn i (tickn j s) = tickn (i+j) s"    			  by simp
lemma [simp]: "ivarupdate (tickn i s) x v = tickn i (ivarupdate s x v)"   by simp
lemma [simp]: "rvarupdate (tickn i s) x v = tickn i (rvarupdate s x v)"   by simp
lemma [simp]: "newobj (tickn i s) l c = tickn i (newobj s l c)"    	  by simp
lemma [simp]: "obj_ifieldupdate (tickn i s) a obj f rtv = 
                tickn i (obj_ifieldupdate s a obj f rtv)"  		  by simp
lemma [simp]: "obj_rfieldupdate (tickn i s) a obj f rtv = 
                tickn i (obj_rfieldupdate s a obj f rtv)"  		  by simp
lemma [simp]: "newframe (tickn i s) m l r = tickn i (newframe s m l r)"   by simp
lemma [simp]: "incrcallcount (tickn i s) = tickn i (incrcallcount s)"	  by simp
lemma [simp]: "incrinvokecount (tickn i s) = tickn i (incrinvokecount s)" by simp


subsection {* Projecting components of the state *}

subsubsection {* Projecting istore *}

lemma [simp]: "istore (tickn k s) = istore s"			   by simp
lemma [simp]: "istore (rvarupdate s v val) = istore s"		   by simp
lemma [simp]: "istore (newobj s a c) = istore s"		   by simp
lemma [simp]: "istore (obj_ifieldupdate s a obj f rtv) = istore s" by simp
lemma [simp]: "istore (obj_rfieldupdate s a obj f rtv) = istore s" by simp
lemma [simp]: "istore (incrcallcount s) = istore s"		   by simp
lemma [simp]: "istore (incrinvokecount s) = istore s"              by simp

lemma [simp]: "istore (ivarupdate s v val) = (istore s)(v:=val)"   by simp
lemma [simp]: "istore (newframe s m l r) = emptyi"		   by simp

subsubsection {* Projecting rstore *}

lemma [simp]: "rstore (tickn k s) = rstore s"			   by simp
lemma [simp]: "rstore (ivarupdate s v val) = rstore s"		   by simp
lemma [simp]: "rstore (obj_ifieldupdate s a obj f rtv) = rstore s" by simp
lemma [simp]: "rstore (obj_rfieldupdate s a obj f rtv) = rstore s" by simp
lemma [simp]: "rstore (newobj s a c) = rstore s"		   by simp
lemma [simp]: "rstore (incrcallcount s) = rstore s"		   by simp
lemma [simp]: "rstore (incrinvokecount s) = rstore s"		   by simp

lemma [simp]: "rstore (rvarupdate s v val) = (rstore s)(v:=val)"   by simp
lemma [simp]: "rstore (newframe s m l r) = emptyr(self:=l, param := r)"
 by simp


subsubsection {* Projecting the heap *}

lemma [simp] :"heap (tickn i s) = heap s"	   by simp
lemma [simp]: "heap (ivarupdate s x v) = heap s"   by simp
lemma [simp]: "heap (rvarupdate s x v) = heap s"   by simp
lemma [simp]: "heap (newframe s mn  a v) = heap s" by simp
lemma [simp]: "heap (incrcallcount s) = heap s"	   by simp
lemma [simp]: "heap (incrinvokecount s) = heap s"  by simp

lemma [simp]: "heap (newobj s a c) = (heap s)(a \<mapsto>\<^sub>f emptyobj c)"	    by simp

lemma [simp]: "heap (obj_ifieldupdate s a obj f rtv) = 
			(heap s)(a \<mapsto>\<^sub>f (fst obj, (fst (snd obj))(f:=rtv),
					         (snd (snd obj))))" by simp

lemma [simp]: "heap (obj_rfieldupdate s a obj f rtv) = 
			(heap s)(a \<mapsto>\<^sub>f (fst obj, (fst (snd obj)),
					         (snd (snd obj))(f:=rtv)))" by simp



subsubsection {* Projecting the framestack *}

lemma [simp]: "framestack (tickn n s) = framestack s"		by simp
lemma [simp]: "framestack (ivarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (rvarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (newobj s a c) = framestack s"        by simp
lemma [simp]: "framestack (obj_ifieldupdate s a obj f rtv) = framestack s" by simp
lemma [simp]: "framestack (obj_rfieldupdate s a obj f rtv) = framestack s" by simp
lemma [simp]: "framestack (incrcallcount s) = framestack s"     by simp
lemma [simp]: "framestack (incrinvokecount s) = framestack s"   by simp

lemma [simp]: "framestack (newframe s m objadr arg) = 
			(m,(istore s, rstore s))#(framestack s)" by simp



subsubsection {* Projecting the maxstack *}

lemma [simp]: "maxstack (tickn n s) = maxstack s"		       by simp
lemma [simp]: "maxstack (ivarupdate s v x) = maxstack s"	       by simp
lemma [simp]: "maxstack (rvarupdate s v x) = maxstack s"	       by simp
lemma [simp]: "maxstack (obj_ifieldupdate s a obj f rtv) = maxstack s" by simp
lemma [simp]: "maxstack (obj_rfieldupdate s a obj f rtv) = maxstack s" by simp
lemma [simp]: "maxstack (newobj s a c) = maxstack s"		       by simp
lemma [simp]: "maxstack (incrcallcount s) = maxstack s"		       by simp
lemma [simp]: "maxstack (incrinvokecount s) = maxstack s"	       by simp

lemma [simp]: "maxstack (newframe s m adr arg) = 
		max (int (length (framestack s)) + 1) (maxstack s)"    by simp



subsubsection {* Projecting the clock *}

lemma [simp]: "clock (ivarupdate s v x) = clock s"   		 by simp
lemma [simp]: "clock (rvarupdate s v x) = clock s"   		 by simp     
lemma [simp]: "clock (obj_ifieldupdate s a obj f rtv) = clock s" by simp
lemma [simp]: "clock (obj_rfieldupdate s a obj f rtv) = clock s" by simp
lemma [simp]: "clock (newobj s a c) = clock s"       		 by simp
lemma [simp]: "clock (incrcallcount s) = clock s"    		 by simp
lemma [simp]: "clock (incrinvokecount s) = clock s"  		 by simp
lemma [simp]: "clock (newframe s mn  a v) = clock s" 		 by simp     

lemma [simp]: "clock (tickn i s) = i + (clock s)"    		 by simp



subsection {* Monotonicity of resource parameters *}

text {* Here are the functions which are used in the operational
  semantics to change state. *}

(* Q: should these mono props really be added to simp sets? *)

lemma callcount_mono [simp] : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> callcount s <= callcount t"
 by (rule, erule evalexpr.induct, simp_all)

lemma invokecount_mono [simp] : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> invokecount s <= invokecount t"
 by (rule, erule evalexpr.induct, simp_all)

lemma maxstack_mono [simp] : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> maxstack s <= maxstack t"
 by (rule, erule evalexpr.induct, simp_all)

(* clock_mono actually says that clock *increases* on each evaluation *)
lemma clock_mono [simp] : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> clock s < clock t"
by (rule, erule evalexpr.induct, simp_all add: state_functions)

(* proof of domsub_lam in Isabelle2003:
lemma domsub_lam: "dom (f) \<subseteq> dom (\<lambda> x. if x=u then Some a else f x)"
by (auto)
*)

lemma fmapdomsub_lam: "fmap_dom (f) \<subseteq> fmap_dom (f(a \<mapsto>\<^sub>f b))"
apply (auto)
done

lemma heap_mono [simp] : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (fmap_dom (heap s) \<subseteq> fmap_dom (heap t))"
by (rule, erule evalexpr.induct, simp_all, auto)

text {* Now we remove the raw definitions of the state functions
  from the simplifier set. *}

declare state_functions [simp del]


subsection {* Elimination rules for the operational semantics *}

inductive_cases evalNull_cases 	 : "(s, expr.Null, v, s') \<in> evalexpr"
inductive_cases evalIVar_cases   : "(s, expr.IVar vn, v, s') \<in> evalexpr"
inductive_cases evalRVar_cases   : "(s, expr.RVar vn, v, s') \<in> evalexpr"
inductive_cases evalInt_cases  	 : "(s, expr.Int i, v, s') \<in> evalexpr"
inductive_cases evalPrimop_cases : "(s, expr.Primop f vn1 vn2, v, s') \<in> evalexpr"
inductive_cases evalInvoke_cases : "(s, expr.Invoke vn1 mn vn2, v, s') \<in> evalexpr"
inductive_cases evalInvokeStatic_cases : "(s, expr.InvokeStatic c mn vn, v, s') \<in> evalexpr"
inductive_cases evalGetFi_cases	 : "(s, expr.GetFi vn f, v, s') \<in> evalexpr"
inductive_cases evalGetFr_cases	 : "(s, expr.GetFr vn f, v, s') \<in> evalexpr"
inductive_cases evalPutFi_cases	 : "(s, expr.PutFi vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalPutFr_cases	 : "(s, expr.PutFr vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalNew_cases  	 : "(s, New c, v, s') \<in> evalexpr"
inductive_cases evalLeti_cases 	 : "(s, Leti vn e ls, v, s') \<in> evalexpr"
inductive_cases evalLetr_cases 	 : "(s, Letr vn e ls, v, s') \<in> evalexpr"
inductive_cases evalCall_cases 	 : "(s, Call fn, v, s') \<in> evalexpr"
inductive_cases evalIf_cases     : "(s, If_ vn l1 l2, v, s') \<in> evalexpr"

text {* We could declare these elimination rules to the automated
  solvers, but they might sometimes get expanded when we don't want it.  
  Instead, let's make a set.
  *}

lemmas eval_cases = evalNull_cases evalIVar_cases evalRVar_cases
  evalPrimop_cases evalInvoke_cases evalInvokeStatic_cases 
  evalGetFi_cases evalGetFr_cases
  evalPutFi_cases evalPutFr_cases evalNew_cases evalLeti_cases     
  evalLetr_cases evalCall_cases evalIf_cases                       

(*Lenb: Some lemmas which simplify states by contracting subsequent
        updates - the components are ordered in the order given in the
        definition of states*)
lemma HeapUpdHeapUpd[simp]: "s\<lparr>heap := A\<rparr>\<lparr>heap := B\<rparr> = s\<lparr>heap := B\<rparr>"
by(simp)
lemma TickHeap[simp]:"tickn i (s\<lparr>heap := A\<rparr>) = (tickn i s)\<lparr>heap := A\<rparr>"
by (simp add: tickn_def)
lemma TickIstore[simp]: "tickn i (s\<lparr>istore := A\<rparr>) = (tickn i s)\<lparr>istore := A\<rparr>"
by (simp add: tickn_def)
lemma TickRstore[simp]: "tickn i (s\<lparr>rstore := A\<rparr>) = (tickn i s)\<lparr>rstore := A\<rparr>"
by (simp add: tickn_def)
lemma IstoreHeap[simp]: "s\<lparr>istore := A, heap := B\<rparr> = s\<lparr>heap := B, istore :=A\<rparr>"
by simp
lemma IstoreUpdIstoreUpd[simp]: "s\<lparr>istore := A\<rparr>\<lparr>istore := B\<rparr> = s\<lparr>istore := B\<rparr>"
by(simp)
lemma RstoreHeap[simp]: "s\<lparr>rstore := A, heap := B\<rparr> = s\<lparr>heap := B, rstore :=A\<rparr>"
by simp
lemma RstoreIstore[simp]: "s\<lparr>rstore := A, istore := B\<rparr> = s\<lparr>istore := B, rstore :=A\<rparr>"
by simp
lemma RstoreUpdRstoreUpd[simp]: "s\<lparr>rstore := A\<rparr>\<lparr>rstore := B\<rparr> = s\<lparr>rstore := B\<rparr>"
by(simp)
lemma CallcountHeap[simp]: "s\<lparr>callcount := A, heap := B\<rparr> = s\<lparr>heap := B, callcount :=A\<rparr>"
by simp
lemma CallcountIstore[simp]: "s\<lparr>callcount := A, istore := B\<rparr> = s\<lparr>istore := B, callcount :=A\<rparr>"
by simp
lemma CallcountRstore[simp]: "s\<lparr>callcount := A, rstore := B\<rparr> = s\<lparr>rstore := B, callcount :=A\<rparr>"
by simp
lemma CallcountUpdCallcountUpd[simp]: "s\<lparr>callcount := A\<rparr>\<lparr>callcount := B\<rparr> = s\<lparr>callcount := B\<rparr>"
by(simp)
end
