(* 				 
   File:	$RCSfile: Semantics.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Semantics.thy,v 1.1 2003/11/19 18:08:31 lenb Exp $

   Functional semantics for a (restricted version of) Grail.
*)

header {* Big-step semantics *}

theory Semantics = Expr + FunMachine:


text {*
  We define a big-step semantics for  Grail.  For the purpose of later
  proofs, the relation is also indexed by the height of the derivation.
  *}

subsection{* Preliminaries: resource environments and abbreviations *}

text {* 
We define resource tuples to model resource consumption during the execution.
Currently, a resource tuple is a record with 4 components, counting execution time, 
function and method calls, and measuring maximal depth of method invocations.
*}

(*Changed from nats to ints*)
record rescomp = clock :: int        -- {* execution time *}
                 callc :: int        -- {* call counter *}
                 invkc :: int        -- {* invoke counter *}
                 invkdpth :: nat     -- {* max invoke depth *}

text {*
To add up resource consumption, we use the following two combinators over resource tuples.
*}

translations 
 "rescomp" <=(type) "\<lparr> clock :: int, callc :: int, invkc :: int, invkdpth :: nat \<rparr>"

constdefs rescomp_plus:: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp"  (infixr "\<oplus>" 0)
"rescomp_plus p1 p2 == \<lparr>clock = clock p1 + clock p2,
                     callc = callc p1 + callc p2,
                     invkc = invkc p1 + invkc p2,
                     invkdpth = invkdpth p1 + invkdpth p2\<rparr>"

constdefs rescomp_cup :: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp" (infixr "\<smile>" 0)
"rescomp_cup p1 p2 == \<lparr>clock = clock p1 + clock p2,
                    callc = callc p1 + callc p2,
                    invkc = invkc p1 + invkc p2,
                    invkdpth = max (invkdpth p1) (invkdpth p2)\<rparr>" 

text {* Some nicer syntax for creating and manipulating resource tuples. *}

(*constdefs*) 
syntax mkRescomp:: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp"
translations
"mkRescomp C CC IC ID" == "\<lparr>clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax mkRescomp_HWL :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp" ("\<langle>_ _ _ _\<rangle>")
translations
"\<langle>C CC IC ID\<rangle>" == "\<lparr>rescomp.clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

(*avoid name clash with tickn and tick*)
(*constdefs*)
syntax tkn ::"int \<Rightarrow> rescomp \<Rightarrow> rescomp"
translations
"tkn n p" == "((mkRescomp n 0 0 0) \<smile> p)"

(*constdefs*)
syntax tk::"rescomp \<Rightarrow> rescomp"
translations
"tk p" == "tkn 1 p"

(*constdefs*)
syntax tkcall::"rescomp \<Rightarrow> rescomp"
translations
"tkcall p" == "((mkRescomp 1 1 0 0) \<smile> p)"

(*a resource environment with a single clocktick*)
(*constdefs*)
syntax tickRo::"rescomp"
translations
"tickRo" == "mkRescomp 1 0 0 0"

declare rescomp_plus_def [simp add]
declare rescomp_cup_def [simp add]

text {*
Function for creating a new object on the heap and initialising its fields with the
given values.
*}

constdefs
  newObj    :: "heap \<Rightarrow> locn \<Rightarrow> env \<Rightarrow> cname \<Rightarrow> (ifldname \<times> iname) list \<Rightarrow> (rfldname \<times> rname) list \<Rightarrow> heap"
  "newObj h l e c ifldvals rfldvals \<equiv> 
       \<lparr> oheap = (heap.oheap h)(l \<mapsto>\<^sub>f c),
         iheap = iheapflds ifldvals l (heap.iheap h) (env.ienv e),
         rheap = rheapflds rfldvals l (heap.rheap h) (env.renv e) \<rparr>"

subsection {* Operational semantics *}

text {*
We now define the operational semantics of Grail as an inductively defined set, 
connecting, for a given expression, all possible pre-states with its post-states.
Note that we define the semantics with a counter of derivation height to use this
counter later on in the soundness  proof to do induction over. The counter is \emph{not}
used to capture resource consumption (we have a separate field in the resource tuple 
for that). 
*}

consts
  semn  :: "(env \<times> heap \<times> expr \<times> nat \<times> heap \<times> val \<times> rescomp) set" -- {* operational semantics *}

syntax
 semn_  :: "[env, heap, expr, nat, heap, val, rescomp] \<Rightarrow> bool"   ("_ \<turnstile> _ , _ \<Down>_ '(_ , _ , _')")

translations
 "E \<turnstile> h,e \<Down>n (hh,v,p)" == "(E,h,e,n,hh,v,p) : semn"

inductive semn intros
 semNull:    "E \<turnstile> h,(expr.Null) \<Down>1 (h,RVal Nullref, tickRo)"

 semInt:     "E \<turnstile> h,(expr.Int i) \<Down>1 (h,IVal i, tickRo)"

 semIVar:    "E \<turnstile> h,IVar v \<Down>1 (h,IVal (E<v>), tickRo)"

 semRVar:    "E \<turnstile> h,RVar v \<Down>1 (h,RVal (E\<lfloor>v\<rfloor>), tickRo)"

 semPrimop:  "E \<turnstile> h, (Primop f x y) \<Down>1 (h,IVal (f (E<x>) (E<y>)), \<langle>3 0 0 0\<rangle>)"

 semRPrimop:  "E \<turnstile> h, (RPrimop f x y) \<Down>1 (h,IVal (f (E\<lfloor>x\<rfloor>) (E\<lfloor>y\<rfloor>)), \<langle>3 0 0 0\<rangle>)"

 semGetfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a\<rbrakk> \<Longrightarrow>
            E \<turnstile> h,(GetFi x f) \<Down>1 (h, IVal ((heap.iheap h) f a), \<langle>2 0 0 0\<rangle>)"

 semGetfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a\<rbrakk> \<Longrightarrow>
            E \<turnstile> h,(GetFr x f) \<Down>1 (h, RVal ((heap.rheap h) f a), \<langle>2 0 0 0\<rangle>)"

 semPutfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a\<rbrakk> \<Longrightarrow>
            E \<turnstile> h, (PutFi x f y) \<Down>1 (h<a\<bullet>f:=E<y>>, arbitrary, \<langle>3 0 0 0\<rangle>)"

 semPutfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a\<rbrakk> \<Longrightarrow>
            E \<turnstile> h, (PutFr x f y) \<Down>1 (h\<lfloor>a\<diamondsuit>f:=E\<lfloor>y\<rfloor>\<rfloor>, arbitrary,\<langle>3 0 0 0\<rangle>)"

 semNew: "\<lbrakk> l = freshloc (fmap_dom (heap.oheap h))\<rbrakk> \<Longrightarrow>
          E \<turnstile> h,(New c ifldvals rfldvals) \<Down>1 (newObj h l E c ifldvals rfldvals, RVal (Ref l), tickRo)"

 semIf_True:  "\<lbrakk>E<x> = grailbool True; E \<turnstile> h, e1 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
               E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (tkn 2 p))"

 semIf_False:  "\<lbrakk>E<x> = grailbool False; E \<turnstile> h, e2 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
                E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (tkn 2 p))"

 semLeti: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,IVal i,p1);  E<x:=i> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Leti x e1 e2) \<Down>((max n m) + 1) (h2,v, tk (p1 \<smile> p2))"

 semLetr: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,RVal r,p1);  E\<lfloor>x:=r\<rfloor> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letr x e1 e2) \<Down>((max n m) + 1) (h2,v, tk (p1 \<smile> p2))"

 semLetv: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,w,p1);  E \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letv e1 e2) \<Down>(max n m) (h2,v, p1 \<smile> p2)"

 semCall: "\<lbrakk>E \<turnstile> h, funtable f \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h,(Call f) \<Down>(n+1) (h1, v, tkcall p)"

 semInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;  
              fmap_lookup (heap.oheap h) a = Some C; 
              (newframe_env (Ref a) (E\<lfloor>y\<rfloor>)) \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
             E \<turnstile> h, (Invoke x mn y) \<Down>(n + 1) (h1,v,\<langle>5 0 1 1\<rangle> \<oplus> p)"

 semInvokeStatic: "\<lbrakk>(newframe_env Nullref (E\<lfloor>y\<rfloor>)) \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                   E \<turnstile> h, (InvokeStatic C mn y) \<Down>(n + 1) (h1,v, \<langle>3 0 1 1\<rangle> \<oplus> p)"

 semMHInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;
                fmap_lookup (heap.oheap h) a = Some C; 
                E\<lfloor>self := Ref a\<rfloor>  \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
               E \<turnstile> h, (MH_Invoke x mn) \<Down>(n + 1) (h1,v,\<langle>4 0 1 1\<rangle> \<oplus> p)"

 semMHInvokeStatic: "\<lbrakk>E \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                     E \<turnstile> h, (MH_InvokeStatic C mn) \<Down>(n + 1) (h1,v, \<langle>2 0 1 1\<rangle> \<oplus> p)"

text {* Operational semantics without height index *}

constdefs
 sem :: "[env, heap, expr, heap, val, rescomp] \<Rightarrow> bool"  ("_ \<turnstile> _ , _ \<Down> _ , _ , _" 1000)
 "E \<turnstile> h,e \<Down> hh,v,p \<equiv> (\<exists> n. (E \<turnstile> h,e \<Down>n (hh,v,p)))"

text {*
  Comentary:
  \begin{enumerate}
  \item in MHInvokeStatic: static condition: method body does not refer to ''this'' - no need
    to add the binding self $\mapsto$ Nullref to E
  \item Put instructions in rules @{text evalPutFi} and @{text
  evalPutFr} are only evaluated
  for their side-effect and should only be used with @{text Letv}.  
  To emphasise this, they return the undetermined value @{text arbitrary}.

  \item Annotations are ignored in the operational semantics.
    
  \item To explain the resource counting in Invoke: vn1 = 1, vn2 = 1,
  frame push/pop = 2, and the implicit return = 1.
  NB: we could break up the increment of ticks into before/after 
  CALL.

  \item Many rules omit type-checking premises.
  For example, in get field and put field rules, we should
  check that the object belongs to a class that indeed has
  the fields being accessed.  

  \item
  Object initialization in the New rule simply sets an object
  component of the heap.
  The iheap and rheap are left alone.  For the logic (and
  by comparison with Grail), it might be better to have a
  New operator which also initialises the fields of an
  object.  However, if we admit in-place-update in objects,
  it seems difficult to avoid dealing with partly-defined
  data structures.

  \item 
   We might tickn 2 in the If rules, but we leave it at one to
   allow for extra store/retrieve which may not be necessary in 
   some cases on the real machine.

 \item The rule for evaluating Let once contained an additional side
  condition that the expression being bound does not itself contain a
  CALL (non-tail position).  So evaluation would get stuck for non-Grail
  programs.  However, there seems to be no harm in considering the
  more general semantics for a slightly richer language, since this
  side condition is not used in the logic.  (This restriction could
  therefore be considered as part of the type-checking phase for Toy Grail).

  \end{enumerate}
*}

subsection {* Elimination rules *}

inductive_cases eval_cases: 
 "E \<turnstile> h,(expr.Null) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(expr.Int i) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(IVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Primop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RPrimop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFi x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFr x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFi x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFr x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(New c ifldvals rfldvals) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Ifg x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Leti x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letr x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letv e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Call f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Invoke x m y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(InvokeStatic C m y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(MH_Invoke x m) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(MH_InvokeStatic C m) \<Down>n  (h1,v,p)"

subsection {* Miscellaneous lemmas *}

lemma no_zero_height_derivsAux: "\<forall> E h h1 v p . (E \<turnstile> h, e \<Down>0  (h1,v,p) \<longrightarrow> False)"
apply (induct_tac e)
apply (simp_all)
apply(clarsimp, elim eval_cases, simp)+
defer 1
apply(clarsimp, elim eval_cases, simp+)
apply(clarsimp, elim eval_cases, simp)+
apply (subgoal_tac "n=0 & m=0")
apply clarsimp
apply (simp add: max_def)
apply (case_tac "n \<le> m", auto)
done

lemma no_zero_height_derivs: "E \<turnstile> h, e \<Down>0  (h1,v,p) \<Longrightarrow> False"
by (insert no_zero_height_derivsAux, fastsimp)
end
