(*<*)
theory FunMachine = MachineBasic:
(*>*)

subsection {* Heaps and environments *}

text {* 
  The heap is a map from locations to objects.  
  Conceptually, objects consist of a class name together with a
  mapping of field names to values.  We use Burstall's
  idea of treating records (i.e. objects) on the heap as separate
  ``mini''-heaps themselves, so that a heap is split into
  three components: the object heap, the integer heap,
  and the reference heap.  The latter two components are
  modelled as total functions, which simplifies
  the formalization.  Of course, the semantics only examines
  the contents of heaps at non-empty locations.
*}

types 
  oheap    = "locn \<leadsto>\<^sub>f cname"           -- {* object heap; a finite map *}
  iheap    = "ifldname \<Rightarrow> locn \<Rightarrow> int"  -- {* integer heap for each field *}
  rheap    = "rfldname \<Rightarrow> locn \<Rightarrow> ref"  -- {* reference heap for each field *}

text {*
  Translations are included to fold up type definitions in printing.
  *}

translations
  "oheap"   <= (type)  "nat \<leadsto>\<^sub>f cname"
  "iheap"   <= (type)  "ifldname \<Rightarrow> nat \<Rightarrow> int"
  "rheap"   <= (type)  "rfldname \<Rightarrow> nat \<Rightarrow> ref"

text {*The three components of heaps are combined in a record.*}
record heap = 
	oheap   :: oheap		 -- {* object heap     *}
	iheap   :: iheap		 -- {* integer heap    *}
	rheap   :: rheap		 -- {* references heap *}

text {*
  Environments are modelled as pairs of total functions, with
  one component for integer variables and one for object variables.
  If the source program is well-typed, it should never attempt to
  access an undefined part of the env. 
 *}

types
  ienv   = "iname \<Rightarrow> int" -- {* the integer environment *}
  renv   = "rname \<Rightarrow> ref" -- {* the reference environment *}

text {*Again, translations fold up type definitions in printing*}

translations
  "ienv"  <= (type)  "iname \<Rightarrow> int"
  "renv"  <= (type)  "rname \<Rightarrow> ref"

text {* and a record bundles the two components.*}

record env =  (* or maybe just a pair *)
	ienv    :: ienv	         -- {* integer env  *}
	renv    :: renv	         -- {* ref env      *}

text {*Some more syntac translations for pretty-printing.*}
translations
  "env" <= (type) "\<lparr>ienv :: ienv, renv :: renv\<rparr>"
  "env" <= (type) "\<lparr>ienv :: ienv, renv :: renv, \<dots>::'a\<rparr>"
  "heap" <= (type) "\<lparr>oheap :: oheap, iheap :: iheap, rheap :: rheap\<rparr>"
  "heap" <= (type) "\<lparr>oheap :: oheap, iheap :: iheap, rheap :: rheap, \<dots>::'a\<rparr>"

syntax objhp::"heap \<Rightarrow> oheap"
translations "objhp" == "heap.oheap"
syntax inthp::"heap \<Rightarrow> iheap"
translations "inthp" == "heap.iheap"
syntax refhp::"heap \<Rightarrow> oheap"
translations "refhp" == "heap.rheap"
syntax Dom::"heap \<Rightarrow> locn set"
translations "Dom h" == "fmap_dom(oheap h)"

subsection  {* Values *}
text {* Values may be either integer values or reference values. *}

datatype val = IVal int | RVal ref

text {*Partial project functions:*}
consts
  theival :: "val \<Rightarrow> int"
  therval :: "val \<Rightarrow> ref"
primrec "theival  (IVal i) = i"
primrec "therval  (RVal r) = r"

subsection {* Access functions *}

text {* We define functions for accessing components of the heaps and environments.
        These are treated as syntactic abbreviations, rather than as functions. *}

(* NB: only priority of 1000 is high enough to override curried application in 
  writing "f s<x>".  We might try unifying these projections using axiomatic
  type class hack -- see HOL/Bali --- but that's pretty ugly. 
  Another improvement might be to restrict to using identifiers in the
  syntax: certainly for field names and variable names. *)

syntax "_get_ivar"     :: "env \<Rightarrow> iname \<Rightarrow> int"		("_<_>" [1000,1000] 1000)
translations "_get_ivar e x" == "env.ienv e x"

syntax "_get_rvar"     :: "env \<Rightarrow> rname \<Rightarrow> ref"		("_\<lfloor>_\<rfloor>" [1000,1000] 1000)
translations "_get_rvar e x" == "env.renv e x"

syntax "_get_rvar_nonnil"     :: "env \<Rightarrow> rname \<Rightarrow> locn"		("_\<lceil>_\<rceil>" [1000,1000] 1000)
translations "_get_rvar_nonnil e x" == "theloc (env.renv e x)"

syntax "_get_ifld" ::	  "[heap, locn, ifldname] \<Rightarrow> int"      ("_<_\<bullet>_>" [1000,1000,1000] 1000)
translations "_get_ifld h loc inm" == "iheap h inm loc"

syntax "_get_rfld" ::	  "[heap, locn, rfldname] \<Rightarrow> ref"      ("_\<lfloor>_\<diamondsuit>_\<rfloor>" [1000,1000,1000] 1000)
translations "_get_rfld h loc rnm" == "rheap h rnm loc"

syntax "_get_class" :: "[heap, locn] \<Rightarrow> cname option"         ("_ @@ _" [1000,1000] 1000)
translations "_get_class h a" == "(fmap_lookup (heap.oheap h) a)"

subsection {* Update functions *}
text {* Update functions are treated as abstract operations on heaps and environments.
        This makes propositions and intermediate assertions more readable, at the 
        expense of needing to add some simplification rules.*}

-- {* Env and heap updates *}
constdefs 
ivarupdate :: "env \<Rightarrow> iname \<Rightarrow> int \<Rightarrow> env"	("_<_:=_>" [1000,1000,0] 1000)
  "ivarupdate e v val \<equiv> (e \<lparr> ienv := (env.ienv e)(v := val) \<rparr>)"

rvarupdate :: "env \<Rightarrow> rname \<Rightarrow> ref \<Rightarrow> env"        ("_\<lfloor>_:=_\<rfloor>" [1000,1000,0] 1000)
  "rvarupdate e v val \<equiv> (e \<lparr> renv := (env.renv e)(v := val) \<rparr>)"

obj_ifieldupdate :: "heap \<Rightarrow> locn \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> heap"   ("_<_\<bullet>_:=_>" [1000,80,1000,80] 1000)
  "obj_ifieldupdate h a f rtv \<equiv> h \<lparr> iheap := (iheap h) (f:= (iheap h f)(a:=rtv)) \<rparr>"

obj_rfieldupdate :: "heap \<Rightarrow> locn \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> heap"   ("_\<lfloor>_\<diamondsuit>_:=_\<rfloor>" [1000,80,1000,80] 1000)
  "obj_rfieldupdate h a g rtv \<equiv> 
     h \<lparr> rheap := (rheap h) (g:= (rheap h g)(a:=rtv)) \<rparr>"

(*<*)
lemmas update_lemmas = ivarupdate_def rvarupdate_def obj_ifieldupdate_def obj_rfieldupdate_def
(*>*)

-- {* New objects *}
text {* 
  Object initialization simply sets the class of an object. We may update the 
  fields with empty maps, but assume that type-checking guarantees that
  no uninitialised fields are accessed.
 *}
consts iheapflds :: "[(ifldname \<times> iname) list, locn, iheap, ienv] \<Rightarrow> iheap"
primrec
  "iheapflds [] a ihp st = ihp"
  "iheapflds (ii#iis) a ihp st = iheapflds iis a (ihp ((fst ii):=((ihp (fst ii))(a:= (st (snd ii)))))) st"

consts rheapflds :: "[(rfldname \<times> rname) list, locn, rheap, renv] \<Rightarrow> rheap"
primrec
  "rheapflds [] a rhp st = rhp"
  "rheapflds (rr#rrs) a rhp st = rheapflds rrs a (rhp ((fst rr):=((rhp (fst rr))(a:=(st (snd rr)))))) st"

constdefs
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"

text {* Object creation initialises the fields of an object ata a particular location
        (a fresh location in the case of object creation) with the given values.
*}

constdefs
  newObj :: "heap \<Rightarrow> locn \<Rightarrow> env \<Rightarrow> cname \<Rightarrow> (ifldname \<times> iname) list \<Rightarrow> (rfldname \<times> rname) list \<Rightarrow> heap"
  "newObj h l e c ifldvals rfldvals \<equiv> 
       \<lparr> oheap = (heap.oheap h)(l \<mapsto>\<^sub>f c),
         iheap = iheapflds ifldvals l (heap.iheap h) (env.ienv e),
         rheap = rheapflds rfldvals l (heap.rheap h) (env.renv e) \<rparr>"

subsection {*Frames*}
text {* A method invocation allocates a new frame on the frame stack. 
        This function creates the appropriate environment, given a reference
        to the invoking object and the parameters.
        The environment contains only bindings for the self object
	and the method parameters.  (We might also give initial
        values to the local variables for the method).
	If we are invoking a static method, then the
	self variable will be set to null.*}

consts makeFrame :: "(PARAMTYPE \<times> ARGTYPE \<times> env \<times> env) \<Rightarrow> env"
recdef makeFrame "measure (\<lambda> (p, a, E, EE) . length p)"
"makeFrame ([], [], E, EE) = EE"
"makeFrame ((RNAME p) # params, (RNAME a) # args, E, EE) = makeFrame (params, args, E, EE\<lfloor>p:=E\<lfloor>a\<rfloor>\<rfloor>)"
"makeFrame ((INAME p) # params, (INAME a) # args, E, EE) = makeFrame (params, args, E, EE<p:=E<a>>)"

(*
constdefs  newframe_env :: "ref \<Rightarrow> ref \<Rightarrow> env"
  "newframe_env objref arg \<equiv>   \<lparr> ienv = emptyi, 
 	   	                 renv = ((emptyr(self := objref))(param := arg)) \<rparr>"
*)
constdefs newframe_env ::"ref \<Rightarrow> PARAMTYPE \<Rightarrow> ARGTYPE \<Rightarrow> env \<Rightarrow> env"
"newframe_env objref params args E \<equiv> 
  makeFrame (params, args, E, \<lparr> ienv = emptyi, renv = (emptyr(self := objref)) \<rparr>)"

subsection {* Heap operations: HSize, same and HpMinus*}
text {* The size of the heap is approximated as the size of the domain of @{text oheap}. *}

constdefs HSize ::"heap \<Rightarrow> int"
"HSize h == int (card (Dom h))"
declare HSize_def [simp]

text {*It is easy to prove that the heap size increments with each fresh location.*}
lemma SizeInsert[simp]: "int (card (insert (freshloc (fmap_dom H)) (fmap_dom H))) = 
                         int (card (fmap_dom H)) + 1"
(*<*)
apply (subgoal_tac "card (insert (freshloc (fmap_dom H)) (fmap_dom H)) =
Suc (card (fmap_dom H))")
apply simp
apply (rule card_insert_disjoint)
apply fastsimp
apply (rule freshloc, fastsimp)
done
(*>*)

text {* The following predicate requires two heaps to coincide 
        pointwise on a given set of locations.*}
constdefs same::"locn set \<Rightarrow> heap \<Rightarrow> heap \<Rightarrow> bool"
"same X h hh == \<forall> l. (l \<in> X \<longrightarrow> (h@@l = hh@@l \<and> 
                                  (\<forall> ifield . h<l\<bullet>ifield> = hh<l\<bullet>ifield>) \<and> 
                                  (\<forall> rfield . h\<lfloor>l\<diamondsuit>rfield\<rfloor> = hh\<lfloor>l\<diamondsuit>rfield\<rfloor>)))"

text {* Another predicate for removing one object from a heap.*}
constdefs HpMinus ::"heap \<Rightarrow> locn \<Rightarrow> heap \<Rightarrow> bool"
"HpMinus h l hh == (same (Dom hh) hh h \<and> Dom hh = Dom h - {l} \<and> l : Dom h)"


text {* Values of booleans are converted into integers *}
syntax grailbool :: "bool \<Rightarrow> int"
translations "grailbool b" == "if b then (1::int) else (0::int)"
(*<*)
end
(*>*)
