header {* Machine model for Grail *}

theory Machine = Finmap + MachineBasic:

subsection {* Heaps *}

text {* 
  The heap is a map from locations to objects.  
  Conceptually, objects consist of a class name together with a
  mapping of field names to values.  We use Burstall's
  idea~\cite{NipkowXX} of treating
  records (i.e. objects) on the heap as separate
  ``mini''-heaps themselves, so that a heap is split into
  three components: the object heap, the integer heap,
  and the reference heap.  The latter two heaps are
  modelled as total functions, which simplifies
  the formalization.  Of course, we must check
  that the semantics only examines
  the contents of heaps at non-empty locations.
  *}

types 
  oheap    = "locn \<leadsto>\<^sub>f cname"            -- {* object heap; a finite map *}
  iheap    = "ifldname \<Rightarrow> locn \<Rightarrow> int"  -- {* integer heap for each field *}
  rheap    = "rfldname \<Rightarrow> locn \<Rightarrow> ref"  -- {* reference heap for each field *}


text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "oheap"   <= (type)  "nat \<leadsto>\<^sub>f cname"
  "iheap"   <= (type)  "ifldname \<Rightarrow> nat \<Rightarrow> int"
  "rheap"   <= (type)  "rfldname \<Rightarrow> nat \<Rightarrow> ref"



subsection {* Stores *}

text {*
  Like heaps, stores are modelled as total functions for simplicity.
  If the source program is well-typed, it should never attempt to
  access an undefined part of the store. 

  A frame is a method name (which can
  be used for profiling), together with a store.
  For a small-step semantics we would also need 
  a return point and a place to store the return value.
  We do not need to include the class of the currently
  executing method because this can be retrieved from
  the @{text self} variable.  When executing a static
  method, @{text self} is set to null.
 *}

types
  istore   = "iname \<Rightarrow> int"	        -- {* an integer   store *}
  rstore   = "rname \<Rightarrow> ref"          -- {* a reference store *}
  store    = "istore \<times> rstore"
  frame    = "mname \<times> store"

text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "istore"  <= (type)  "iname \<Rightarrow> int"
  "rstore"  <= (type)  "rname \<Rightarrow> ref"
  "store"   <= (type)  "istore \<times> rstore"
  "frame"   <= (type)  "mname \<times> store"


subsection {* States *}

text {* The state contains several resource-aware components:
  \begin{itemize}
  \item maxstack
  \item callcount
  \item invokecount
  \item clock
  \end{itemize}
  These are all represented as integers for ease of reasoning
  in combination with integer program variables, because Isabelle
  doesn't handle the subtyping Nat<Int very well.
*}

record heap = 
	objhp    :: oheap		 -- {* object heap *}
	inthp    :: iheap		 -- {* integer heap *}
	refhp    :: rheap		 -- {* references heap *}

record state =
	oheap       :: oheap		 -- {* object heap *}
	iheap	    :: iheap		 -- {* integer heap *}
	rheap	    :: rheap		 -- {* references heap *}
	istore      :: istore	         -- {* integer store on top of stack *}
	rstore      :: rstore	         -- {* ref store on top of stack     *}
        framestack  :: "frame list"      -- {* frame stack		     *}
        maxstack    :: int		 -- {* maximum depth of stack        *}  
        callcount   :: int		 -- {* count of CALL depth	     *}  
        invokecount :: int		 -- {* count of INVOKE depth	     *}  
	clock	    :: int		 -- {* instruction counter	     *}

(* NB: field names used more than once will override earlier use *)

(* NB: maybe could use extensible records above *)

constdefs
 getheap :: "state \<Rightarrow> heap"
 "getheap s \<equiv>  \<lparr> objhp = oheap s, inthp = iheap s, refhp = rheap s \<rparr>"
 

translations
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int\<rparr>"
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int,\<dots>::'a \<rparr>"

text {* The size of the heap can be recovered from the state.
  It's approximated simply as the size of the domain of @{text oheap}. *}

constdefs hpsize ::"state \<Rightarrow> int"
  "hpsize s == int (card (fmap_dom (oheap s)))"


subsection  {* Values *}

text {* Values and assertions are introduced here because
  the syntax of programs can include them.  *}

text {* Values may be either integer values or reference values. *}

datatype val = IVal int | RVal ref
consts
  theival :: "val \<Rightarrow> int"
  therval :: "val \<Rightarrow> ref"
primrec
  "theival  (IVal i) = i"
primrec
  "therval  (RVal r) = r"


section {* Accessing and manipulating states *}

text {* We define some accessor functions for accessing components
  of the state.  These are treated as syntactic abbreviations,
  rather than functions. *}

(* NB: only priority of 1000 is high enough to override curried application in 
  writing "f s<x>".  We might try unifying these projections using axiomatic
  type class hack -- see HOL/Bali --- but that's pretty ugly. 
  Another improvement might be to restrict to using identifiers in the
  syntax: certainly for field names and variable names. *)

syntax
 "_get_ivar"     :: "state \<Rightarrow> iname \<Rightarrow> int"		("_<_>" [1000,1000] 1000)
translations
 "_get_ivar s x" == "istore s x"

syntax
 "_get_rvar"     :: "state \<Rightarrow> rname \<Rightarrow> ref"		("_\<lfloor>_\<rfloor>" [1000,1000] 1000)
translations
 "_get_rvar s x" == "rstore s x"

syntax
 "_get_rvar_nonnil"     :: "state \<Rightarrow> rname \<Rightarrow> locn"		("_\<lceil>_\<rceil>" [1000,1000] 1000)
translations
 "_get_rvar_nonnil s x" == "theloc (rstore s x)"

(* da: adjusted concrete syntax of next one because \<guillemotleft> much easier to type! *)
syntax  
 "_get_obj"  ::      "[state, locn] \<Rightarrow> cname option"	("_\<guillemotleft>_\<guillemotright>" [1000,1000] 1000)
translations
 "_get_obj s l" == "fmap_lookup (oheap s) l"

(* HWL: GNU emacs with x-symbols has problems with the \<guillemotleft> \<guillemotright> etc so I use this syntax *)
syntax  
 "_get_obj_HWL"  ::      "[state, locn] \<Rightarrow> cname option"	("_\<lless>_\<ggreater>" [1000,1000] 1000)
translations
 "_get_obj_HWL s l" == "fmap_lookup (oheap s) l"

syntax
 "_get_ifld" ::	  "[state, locn, ifldname] \<Rightarrow> int"      ("_<_\<bullet>_>" [1000,1000,1000] 1000)
translations
 "_get_ifld s loc inm" == "iheap s inm loc"

syntax
 "_get_rfld" ::	  "[state, locn, rfldname] \<Rightarrow> ref"      ("_\<lfloor>_\<diamondsuit>_\<rfloor>" [1000,1000,1000] 1000)
translations
 "_get_rfld s loc rnm" == "rheap s rnm loc"

syntax
 "freshlocst"  :: "state \<Rightarrow> locn"
translations
 "freshlocst s" == "freshloc (fmap_dom (oheap s))"


subsection {* Functions for updating the state *}

text {* The state updating functions are treated as abstract operations
  on states.  This makes propositions and intermediate assertions
  more readable, at the expense of needing to add numerous
  simplification rules to deal with them. *}



constdefs
 tickn	     :: "int \<Rightarrow> state \<Rightarrow> state"
 "tickn n \<equiv>  \<lambda> s. s \<lparr> clock := (clock s) + n \<rparr>"
syntax
 tick	     :: "state \<Rightarrow> state" -- {* a handy abbreviation *}
translations
 "tick"    == "tickn 1"

constdefs
  ivarupdate :: "state \<Rightarrow> iname \<Rightarrow> int \<Rightarrow> state"	("_<_:=_>" [1000,1000,0] 1000)
  "ivarupdate s v val \<equiv> (s \<lparr> istore := (istore s)(v := val) \<rparr>)"

constdefs
  rvarupdate :: "state \<Rightarrow> rname \<Rightarrow> ref \<Rightarrow> state"        ("_\<lfloor>_:=_\<rfloor>" [1000,1000,0] 1000)
  "rvarupdate s v val \<equiv> (s \<lparr> rstore := (rstore s)(v := val) \<rparr>)"


constdefs
  obj_ifieldupdate :: "state \<Rightarrow> locn \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> state"   ("_<_\<bullet>_:=_>" [1000,80,1000,80] 1000)
  "obj_ifieldupdate s a f rtv \<equiv> s \<lparr> iheap := (iheap s) (f:= (iheap s f)(a:=rtv)) \<rparr>"

  obj_rfieldupdate :: "state \<Rightarrow> locn \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> state"   ("_\<lfloor>_\<diamondsuit>_:=_\<rfloor>" [1000,80,1000,80] 1000)
  "obj_rfieldupdate s a f rtv \<equiv> s \<lparr> rheap := (rheap s) (f:= (rheap s f)(a:=rtv)) \<rparr>"


constdefs
  incrcallcount :: "state \<Rightarrow> state"
  "incrcallcount s \<equiv> s\<lparr> callcount := (callcount s)+1 \<rparr>"


syntax
  tickcall :: "state \<Rightarrow> state"   -- {* the effect of procedure calling *}
translations
  "tickcall"   == "tick o incrcallcount"
(* NB: this can be slightly troublesome: unification fails with
   matching composition tick o incrcallcount against tick (incrcallcount s) *)


text {* New objects\<dots> *}

consts
  iheapflds :: "[(ifldname \<times> iname) list, locn, iheap, istore] \<Rightarrow> iheap"
primrec
  "iheapflds [] a ihp st = ihp"
  "iheapflds (ii#iis) a ihp st = iheapflds iis a (ihp ((fst ii):=((ihp (fst ii))(a:= (st (snd ii)))))) st"

consts
  rheapflds :: "[(rfldname \<times> rname) list, locn, rheap, rstore] \<Rightarrow> rheap"
primrec
  "rheapflds [] a rhp st = rhp"
  "rheapflds (rr#rrs) a rhp st = rheapflds rrs a (rhp ((fst rr):=((rhp (fst rr))(a:=(st (snd rr)))))) st"

constdefs
  newobj    :: "state \<Rightarrow> cname \<Rightarrow> (ifldname \<times> iname) list \<Rightarrow> (rfldname \<times> rname) list \<Rightarrow> state"
  "newobj s c ifldvals rfldvals \<equiv> 
  (s \<lparr> oheap := (oheap s)((freshlocst s) \<mapsto>\<^sub>f c),
       iheap := iheapflds ifldvals (freshlocst s) (iheap s) (istore s),
       rheap := rheapflds rfldvals (freshlocst s) (rheap s) (rstore s) \<rparr>)"



text {* 
  Object initialization simply sets the class of an object.
  We might also update the fields with empty maps, defined
  below.  But it should be a guaranteed by type-checking that
  no uninitialised field is accessed.
 *}

constdefs
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"

text {* A method invocation allocates a new frame on the frame stack. 
        This function adjusts the state accordingly, given a reference
        to the invoking object, and the parameter.
        The new store contains only bindings for the self object
	and the method parameter.  (We might also give initial
        values to the local variables for the method).
	Note that if we are invoking a static method, then the
	self variable will be set to null.
*}

constdefs
   newframe :: "[state, mname, ref, ref] \<Rightarrow> state"
  "newframe s m objref arg \<equiv> 
	s \<lparr> framestack := (m,(istore s, rstore s))#(framestack s),
	    istore := emptyi,
	    rstore := ((emptyr(self := objref))(param := arg)),
	    invokecount:= (invokecount s) + 1,
            maxstack := max ((int (length (framestack s))) + 1) (maxstack s) \<rparr>"

text {* Restore the old frame from a previous state.  This trick is used
  in the rules instead of modelling a real stack of frames. *}

constdefs  (* first argument is state which is being popped *)
  oldframe :: "[state,state] \<Rightarrow> state"
  "oldframe s' s \<equiv> s'\<lparr>istore := istore s,  rstore := rstore s, 
	                framestack := (framestack s) \<rparr>"


text {* Values of booleans as integers *}

syntax
 grailbool :: "bool \<Rightarrow> int"
translations
 "grailbool b" == "if b then (1::int) else (0::int)"



end
