(* 				 
   File:	$RCSfile: Expr.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Expr.thy,v 1.6 2003/10/01 20:45:50 a1hloidl Exp $

   Abstract Syntax of Grail
*)

header {* Abstract Syntax of Grail *}

(*<*)
theory Expr = MachineBasic:
(*>*)

text {* Simple expressions are the basic unit of execution,
  and denote values.  They are built from operators, constants,
  and previously computed values (names).  An expression may have a
  side effect, but there are no directly nested expressions.
  We model untyped expressions. 

  Compound expressions are built using let expressions.  
  A function body is a let expression.  A function ends with
  tail calls to other functions or method returns.  
  The natural thing to do is to have two syntactic categories
  of terms: one for simple expressions and one for compound
  expressions.  However this makes the whole development 
  more tedious; so instead we define a predicate
  @{text SimpleExpr} which restricts the operational
  semantics in the single case that matters (the
  binding in a let expressions).

  Expressions are augmented with logical annotations, for
  pre-conditions, post-conditions, and termination
  measures.  They are polymorphic in the type of auxiliary
  variables.
*}


datatype 'a expr =
   Int	    int			
 | IVar     iname		
 | Primop   "int \<Rightarrow> int \<Rightarrow> int" iname iname
 | Null	          		           ("NULL")
 | RVar     rname	
 | RPrimop   "ref \<Rightarrow> ref \<Rightarrow> int" rname rname
 | New      "cname" "(ifldname \<times> iname) list" "(rfldname \<times> rname) list" ("NEW <_> '(_,_')")
 | GetFi    rname ifldname	  	   ("_\<bullet>_" [65,1000] 65)
 | GetFr    rname rfldname	  	   ("_\<diamondsuit>_" [65,1000] 65)
 | PutFi    rname ifldname iname	   ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | PutFr    rname rfldname rname	   ("(2_\<diamondsuit>_ :=/ _)" [70,1000,65] 61)
 | Invoke   rname mname rname              ("_\<diamondsuit>_'(_')" [70,1000,65] 61) 
 | InvokeStatic cname mname rname          ("_\<bullet>_'(_')" [70,1000,65] 61) 
 | MH_Invoke   rname mname                 ("_\<diamondsuit>\<diamondsuit>_" [70,1000] 61) 
 | MH_InvokeStatic cname mname             ("_\<bullet>\<bullet>_" [70,1000] 61) 
 | Leti     iname "'a expr" "'a expr"        
 | Letr     rname "'a expr" "'a expr"
 | Letv     "'a expr" "'a expr"
 | Ifg      iname "'a expr" "'a expr"	   ("(1IF _/ THEN _/ ELSE _)"  [0,0,0] 61)
 | Call     funame			   ("(CALL _)" [0] 60)

(*  | Ann	    "'a" "'a expr"  *)



text {* Notes:
  \begin{enumerate}
  \item Primitive operations (\texttt{RPrimop}) on references might
  include a test for null, and (perhaps) a pointer equality test.

  \item In method invocations, the first argument is either a
  variable pointing to an object (instance methods), or
  a class name (static/class methods).  The second argument
  is the method name, and the third argument is the
  actual parameter, a single reference variable.

  \item If we treated imperative Grail, we might instead use the traditional
  imperative language of commands together with non-side effecting
  expressions.  That would be considerably simpler;
  for example, types of pre-assertions and post-assertions become
  the same, and the syntax needs only one kind of assertion.
  Lets are broken into assignment and composition.
  But that treatment de-emphasises the functional view.
  \end{enumerate}
*}
  


text {* Introduce some nice concrete syntax. *}

(* The following black magic is Isabelle gobbledygook to
   introduce some nice concrete syntax. *)

(* let syntax *)
nonterminals
 gletbind gletbinds

syntax
  "_Gbind"      :: "[id, 'a] => gletbind"                ("(2_ =/ _)" 10)
  "_Grefbind"   :: "[id, 'a] => gletbind"                ("(2rf _ =/ _)" 10)
  "_Gvoidbind"  :: "'a => gletbind"                      ("(2 '_ =/ _)" 10)
  ""            :: "gletbind => gletbinds"               ("_")
  "_gbinds"     :: "[gletbind, gletbinds] => gletbinds"  ("_;/ _")
  "_GLets"      :: "[gletbinds, 'a expr] => 'a expr"     ("(LET (_)/ IN (_) END)" 60)

translations
  "_GLets (_gbinds b bs) e"  == "_GLets b (_GLets bs e)"
  "LET    v = e IN l END"    == "Leti v e l"
  "LET    _ = e IN l END"    == "Letv e l"
  "LET rf v = e IN l END"    == "Letr v e l"

subsection {* Functions and methods *}

text {* A Grail function is notionally parameterised on
  a subset of the current variables in scope.  Variables
  not in the formal parameters should not be accessible
  in the body.  However, a new stack frame is not 
  allocated.  We model this simply by ignoring
  the formal parameters, so a Grail function is just
  its body, which is an @{text expr}.  

  A Grail method body is a collection of function bindings,
  together with a main body and a set of local variables used
  within.

  To simplify the representation for Grail, we consider a
  global mapping from function names to function bodies (see later
  below, in class table).
  This means that we do not need to keep a record of the current
  program position (current method or current functions) in the state.

  Methods are parameterised on two things: first the
  self object (self) invoking the method, and then the single formal
  parameter.  Both parameter names are fixed, however, so
  are left implicit.

  When a method is invoked, we might want to know the size of the frame
  to calculate the size of the stack, which is why we record the 
  total set of local variables.
  (Although we do not initialize these).
 *}

types
  'inv methbody = "(iname set \<times> rname set) \<times> 'inv expr"

translations
  "inv methbody" <= (type) "(iname set \<times> rname set) \<times> inv expr"

subsection {* Programs *}

text {* The class table represents the program as a global parameter. 
   Rather than a single table, we split it into four tables,
  representing: functions (global), i/r-fields, and methods. *}

consts
  funtable      :: "funame \<Rightarrow> 'a expr"           -- {* global function names *}
  ifieldtable   :: "cname \<Rightarrow> ifldname list"      -- {* int fields by class *}
  rfieldtable   :: "cname \<Rightarrow> rfldname list"      -- {* ref fields by class *}
  methtable     :: "cname \<Rightarrow> mname \<Rightarrow> 'a expr"    -- {* methods by class *}

(*<*)
end
(*>*)

