(* 				 
   File:	$RCSfile: VDMVanilla.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: VDMVanilla.thy,v 1.1 2003/11/18 12:03:20 a1hloidl Exp $

   VDM-style rules for the program logic.
*)

header {* VDM-style Program Logic  *}

(*<*)
theory VDMVanilla = Semantics + Lemmas:

subsection {* Basic definitions *}

text {* For dynamic method invocation we have to know the names of all possible classes *}
consts all_classes :: "cname set"

axioms finclasses: "\<forall> (C::cname). C \<in> all_classes \<and> finite all_classes"
(*>*)

(*derivation system for VDM*)

subsection {* Basic types *}

text {* 
 VDM-style assertions are modelled as sets, connecting pre-states with post-states,
 the result value of the execution and the resources consumed during execution.
 Since we have split up the components of the state, we get a 5-tuple (E,h,hh,v,p).
 With this type, the informal statement ``assertion $P$ is fulfilled in pre-state
 ($E$, $h$), post-state ($E$, $hh$) with result value $v$ and resource consumption $p$''
 is written formally as set membership, i.e.\
 \[
 (E,h,hh,v,p) \in P
 \]

 A VDM-style context, collects specifications of functions and methods. A context is
 therefore a set of tuples, with an expression as first and an assertion as second
 argument.
*}

types
  "vdmassn" = "(env \<times> heap \<times> heap \<times> val \<times> renv) set"

types
  'a vdmtuple = "'a expr \<times> vdmassn"

types
  'a vdmcontext = "('a vdmtuple) set"

consts vdm_proof :: "('a vdmcontext \<times> 'a vdmtuple) set"

syntax
  vdm_deriv :: "'a vdmcontext \<Rightarrow> 'a expr \<Rightarrow> vdmassn \<Rightarrow> bool"
		  ("_ \<rhd>  _ : _ " [900,100,100] 50)

translations
 "G \<rhd> e : P" == "(G,e,P) \<in> vdm_proof"

syntax
  vdm_emptyctx :: "'a expr \<Rightarrow> vdmassn \<Rightarrow> bool" ("\<rhd> _ : _ " 40)
translations
  "\<rhd> e : P " == "{} \<rhd> e : P"

(* Now all in Adapt.thy (not used in logic rules themselves!!)
subsection {* Definitions *}

subsection {* Substitutions on assertions *}
text {* 
Substitutions on assertions are needed to adapt assertions in context to the body
when we meet a recursive call. Since we use a shallow embedding of assertions, all
we have to do is to modify the environment to take the substituter instead of the
substitee. Since we only need this to substitute argument for formal parameter,
we only have to define the case of substituting rname for rname.
*}

constdefs subst :: "vdmassn \<Rightarrow> rname \<Rightarrow> rname \<Rightarrow> vdmassn"
"subst P y x \<equiv> {(E,h,hh,v,p). (E\<lfloor>x := E\<lfloor>y\<rfloor>\<rfloor>,h,hh,v,p) \<in> P}"

constdefs nuke :: "rname \<Rightarrow> vdmassn \<Rightarrow> vdmassn"
"nuke y P \<equiv> {(E,h,hh,v,p). (E\<lfloor>y := Nullref\<rfloor>,h,hh,v,p) \<in> P}"

constdefs not_free_in_assn :: "rname \<Rightarrow> vdmassn \<Rightarrow> bool"
"not_free_in_assn y Q \<equiv> \<forall> x E h hh v p. (E,h,hh,v,p) \<in> Q \<longrightarrow> (E\<lfloor>y:=E\<lfloor>x\<rfloor>\<rfloor>,h,hh,v,p) \<in> Q"

lemmas petaQ_lemmas = subst_def nuke_def not_free_in_assn_def

*)
subsection {* Auxiliary functions and lemmas *}

text {* 
 For defining rules for method invocation, we have to define the notion of
 a well-structured heap at a given location.

 Well-structured heap (qach\_QaQ) means "in an environment E and heap h, the
 r-variable points to location a in the heap, where we find an object of class C.
*}

(* @@ HWL: I haven't changed the Klingon name of the function; modify if you 
           fell strongly against it! *)

constdefs qach_QaQ :: "env \<Rightarrow> heap \<Rightarrow> locn \<Rightarrow> rname \<Rightarrow> cname \<Rightarrow> bool"
 "qach_QaQ E h a x C == (E\<lfloor>x\<rfloor> = Ref a \<and> (fmap_lookup (heap.oheap h) a = Some C))" 

text {*
 The following lemmas over this function avoid expanding its definition and keep
 intermediate subgoals in the proof short.
 Note, that by default the definition of qach\_QaQ is not added to the simpset.
*}

lemma qach_QaQ_inj_arg: "\<lbrakk> qach_QaQ E h a r C ; qach_QaQ E h a' r C \<rbrakk> \<Longrightarrow> a = a'"
by (simp add: qach_QaQ_def)

(*<*)
lemma qach_QaQ_rvarIUpd: "qach_QaQ E<x:=n> h a r C = qach_QaQ E h a r C"
by (simp add: qach_QaQ_def)

lemma qach_QaQ_rvarRUpdSame: "qach_QaQ E\<lfloor>r:=Ref a\<rfloor> h a r C = (fmap_lookup (heap.oheap h) a = Some C)"
by (simp add: qach_QaQ_def)

lemma qach_QaQ_rvarRUpdOther: "r \<noteq> x  \<Longrightarrow> qach_QaQ E\<lfloor>x:=b\<rfloor> h a r C = qach_QaQ E h a r C"
by (simp add: qach_QaQ_def)

lemma qach_QaQ_heap1: "qach_QaQ E \<lparr> oheap = oheap h, iheap = foo, rheap = rheap h \<rparr> a r C = qach_QaQ E h a r C"
by (simp add: qach_QaQ_def)

lemma qach_QaQ9: "\<lbrakk> E\<lfloor>x\<rfloor> = E\<lfloor>y\<rfloor> ; qach_QaQ E h a x C \<rbrakk> \<Longrightarrow>  qach_QaQ E h a y C"
by (simp add: qach_QaQ_def)

lemma qach_QaQ8: "\<lbrakk> qach_QaQ E\<lfloor>x:=E\<lfloor>y\<rfloor>\<rfloor> h a x DecClass \<rbrakk> \<Longrightarrow> qach_QaQ E h a y DecClass"
by (simp add: qach_QaQ_def)

lemmas qach_QaQ_lemmas = qach_QaQ_inj_arg qach_QaQ_rvarIUpd qach_QaQ_rvarRUpdSame qach_QaQ_rvarRUpdOther qach_QaQ_heap1 qach_QaQ9 qach_QaQ8
(*>*)
(* --------------------------------------------------------------------------- *)

subsection {* Vanilla syntax *}

text {*
 Plug resource tuple into an assertion.
*}

constdefs renv_assn :: "vdmassn \<Rightarrow> renv \<Rightarrow> vdmassn"  ("_$_")
 "renv_assn Q q \<equiv> {(E,h,hh,v,p) . p = q \<and> (\<exists> q' . (E,h,hh,v,q') \<in> Q)}"

lemma "\<lbrakk> (E,h,hh,v,p) \<in> (P $ q) \<rbrakk> \<Longrightarrow> p=q"
by (simp add: renv_assn_def)

text {*
 Add resource tuple to an assertion.
*}

constdefs tick_assn :: "vdmassn \<Rightarrow> renv \<Rightarrow> vdmassn"  ("_\<oplus>_")
 "tick_assn Q q \<equiv> {(E,h,hh,v,p') . \<forall> p . (E,h,hh,v,p) \<in> Q \<longrightarrow> p' = (p \<oplus> q)}"

lemma "\<lbrakk> (E,h,hh,v,p) \<in> P ; (E,h,hh,v,p') \<in> P\<oplus>\<langle>1 0 0 0\<rangle> \<rbrakk> \<Longrightarrow> (clock p) < (clock p')"
by (simp add: tick_assn_def)

text {* Set result value in assertion *}
constdefs val_assn :: "vdmassn \<Rightarrow> val \<Rightarrow> vdmassn"  ("_\<cdot>=_")
 "val_assn Q w \<equiv> {(E,h,hh,v',p) . v' = w \<and> (\<exists> v' . (E,h,hh,v',p) \<in> Q)}"

text {* set integer result in assertion *}
constdefs ival_assn :: "vdmassn \<Rightarrow> int \<Rightarrow> vdmassn"  ("_\<bullet>=_")
 "ival_assn Q i \<equiv> {(E,h,hh,v',p) . v' = (IVal i) \<and> (\<exists> v' . (E,h,hh,v',p) \<in> Q)}"

text {* set reference result in assertion *}
constdefs rval_assn :: "vdmassn \<Rightarrow> ref \<Rightarrow> vdmassn"  ("_\<diamondsuit>=_")
 "rval_assn Q r \<equiv> {(E,h,hh,v',p) . v' = (RVal r) \<and> (\<exists> v' . (E,h,hh,v',p) \<in> Q)}"

text {* set integer result in assertion, with pre-state lookup *}
constdefs ival_f_assn :: "vdmassn \<Rightarrow> ((env \<times> heap) \<Rightarrow> int) \<Rightarrow> vdmassn"  ("_\<bullet>-_")
 "ival_f_assn Q f \<equiv> {(E,h,hh,v',p) . v' = (IVal (f (E,h))) \<and> (\<exists> v' . (E,h,hh,v',p) \<in> Q)}"

text {* set reference result in assertion, with pre-state lookup *}
constdefs rval_f_assn :: "vdmassn \<Rightarrow> ((env \<times> heap) \<Rightarrow> ref) \<Rightarrow> vdmassn"  ("_\<diamondsuit>-_")
 "rval_f_assn Q f \<equiv> {(E,h,hh,v',p) . v' = (RVal (f (E,hh))) \<and> (\<exists> v' . (E,h,hh,v',p) \<in> Q)}"
lemma "\<lbrakk> (E,h,hh,v,p) \<in> P\<bullet>=w \<rbrakk> \<Longrightarrow> v=(IVal w)"
by (simp add: ival_assn_def)

lemma "\<lbrakk> (E,h,hh,v,p) \<in> P\<diamondsuit>=w \<rbrakk> \<Longrightarrow> v=(RVal w)"
by (simp add: rval_assn_def)

lemma "(P\<bullet>=w)\<bullet>=w' = P\<bullet>=w'"
by (simp add: ival_assn_def)

constdefs ungrailbool :: "int \<Rightarrow> bool"
 "ungrailbool i \<equiv> (if (i = (1::int)) then True else False)"

text {* conditional on assertions, modifying the first argument assertion *}
constdefs myif :: "vdmassn \<Rightarrow> ((env \<times> heap) \<Rightarrow> int) \<Rightarrow> vdmassn \<Rightarrow> vdmassn \<Rightarrow> vdmassn" ("ifA _ _ thenA _ elseA _")
 "myif p f p1 p2 \<equiv> {(E,h,hh,v,p) . if (ungrailbool (f (E,h))) then (E,h,hh,v,p) \<in> p1 else (E,h,hh,v,p) \<in> p2}"

text {* seq composition on assertions *}
constdefs myiseq :: "vdmassn \<Rightarrow> iname \<Rightarrow> vdmassn \<Rightarrow> vdmassn"   ("_ ;_\<bullet> _")
 "myiseq P1 x P2 \<equiv> {(E,h,hh,v,p) . \<exists> p1 p2 h1 i . (E,h,h1,IVal i,p1) \<in> P1 \<and> (E<x:=i>,h1,hh,v,p2) \<in> P2 \<and> p = (p1 \<smile> p2)}"

constdefs myrseq :: "vdmassn \<Rightarrow> rname \<Rightarrow> vdmassn \<Rightarrow> vdmassn"   ("_ ;_\<diamondsuit> _")
 "myrseq P1 x P2 \<equiv> {(E,h,hh,v,p) . \<exists> p1 p2 h1 r . (E,h,h1,RVal r,p1) \<in> P1 \<and> (E\<lfloor>x:=r\<rfloor>,h1,hh,v,p2) \<in> P2 \<and> p = (p1 \<smile> p2)}"

constdefs myvseq :: "vdmassn \<Rightarrow> vdmassn \<Rightarrow> vdmassn"   ("_ ;\<cdot> _")
 "myvseq P1 P2 \<equiv> {(E,h,hh,v,p) . \<exists> p1 p2 h1 . (E,h,h1,arbitrary,p1) \<in> P1 \<and> (E,h1,hh,v,p2) \<in> P2 \<and> p = (p1 \<smile> p2)}"

(*
constdefs myoplus :: "(vdmassn \<times> renv>) \<Rightarrow> ((vdmassn \<times> renv) \<Rightarrow> renv) \<Rightarrow> vdmassn" ("_ \<oplus>\<bullet> _")
 "myoplus P' p' = (case P' of (P,p) \<Rightarrow> P \<oplus> (p \<smile> p'))"
*)

text {* heap unchanged, no other restriction *}
constdefs Pid :: vdmassn
 "Pid \<equiv> {(E,h,hh,v,p). hh = h}"

text {* Add definitions to simpset. *}

declare renv_assn_def [simp]
declare tick_assn_def [simp]

declare val_assn_def [simp]
declare ival_assn_def [simp]
declare rval_assn_def [simp]
declare Pid_def [simp]
declare myif_def [simp]
declare myiseq_def [simp]
declare myrseq_def [simp]
declare myvseq_def [simp]

subsection {* Program Logic Rules *}

text {*
 The program logic is inductively defined as a set, with context, expression and assertion as its components.
*}

(*
vdm_getfi: "G \<rhd> GetFi x f : (Pid\<bullet>-(\<lambda> (E,h) . (heap.iheap hh) f (THE a . E\<lfloor>x\<rfloor> = Ref a))) 
                            \<oplus> \<langle>2 0 0 0\<rangle>"

vdm_getfr: "G \<rhd> GetFr x f : (Pid\<diamondsuit>-(\<lambda> (E,h) . (heap.rheap hh) f (THE a . E\<lfloor>x\<rfloor> = Ref a))) 
                            \<oplus> \<langle>2 0 0 0\<rangle>"

*)
(*no rule for Ann A e yet*) (* bloody right, death to annotations!! *)
inductive vdm_proof intros
vdm_conseq: "\<lbrakk>G \<rhd> e : P; P \<subseteq> Q\<rbrakk> \<Longrightarrow> G \<rhd> e : Q"

vdm_null: "G \<rhd> NULL : (Pid\<diamondsuit>=Nullref) $ \<langle>1 0 0 0\<rangle>"

vdm_int: "G \<rhd> expr.Int i : (Pid\<bullet>=i) $ \<langle>1 0 0 0\<rangle>"

vdm_ivar: "G \<rhd> IVar x : (Pid\<bullet>-(\<lambda> (E,h) . E<x>)) $ \<langle>1 0 0 0\<rangle>"

vdm_rvar: "G \<rhd> RVar x : (Pid\<diamondsuit>-(\<lambda> (E,h) . E\<lfloor>x\<rfloor>)) $ \<langle>1 0 0 0\<rangle>"

vdm_prim: "G \<rhd> Primop f x y : {(E,h,hh,v,p) . hh = h \<and> v = IVal (f (E<x>) (E<y>)) \<and> p = \<langle>3 0 0 0\<rangle>}"

vdm_rprim: "G \<rhd> RPrimop f x y : {(E,h,hh,v,p) . hh = h \<and> v = IVal (f (E\<lfloor>x\<rfloor>) (E\<lfloor>y\<rfloor>)) \<and> p = \<langle>3 0 0 0\<rangle>}"

vdm_getfi: "G \<rhd> GetFi x f : {(E,h,hh,v,p) . \<exists> a . E\<lfloor>x\<rfloor> = Ref a \<and> hh = h \<and>  
                                                   v = IVal ((heap.iheap hh) f a) \<and> 
                                                   p = \<langle>2 0 0 0\<rangle> }"
vdm_getfr: "G \<rhd> GetFr x f : {(E,h,hh,v,p) . \<exists> a . E\<lfloor>x\<rfloor> = Ref a \<and> hh = h \<and> 
                                                   v = RVal ((heap.rheap hh) f a) \<and> 
                                                   p = \<langle>2 0 0 0\<rangle>}"

vdm_putfi: "G \<rhd> PutFi x f y : 
            ({(E,h,hh,v,p) . \<exists> a . E\<lfloor>x\<rfloor>  = Ref a \<and> hh = h<a\<bullet>f:=E<y>>}\<cdot>=(arbitrary::val))
            \<oplus> \<langle>3 0 0 0\<rangle>"

vdm_putfr: "G \<rhd> PutFr x f y : 
            ({(E,h,hh,v,p) . \<exists> a . E\<lfloor>x\<rfloor>  = Ref a \<and> hh = h\<lfloor>a\<diamondsuit>f:=E\<lfloor>y\<rfloor>\<rfloor>}\<cdot>=(arbitrary::val))
            \<oplus> \<langle>3 0 0 0\<rangle>"

vdm_new: "G \<rhd> New c ifldvals rfldvals :
              {(E,h,hh,v,p) . \<exists> l . l = freshloc (fmap_dom (heap.oheap h)) \<and> 
                                    hh = newObj h l E c ifldvals rfldvals \<and>
                                    v = RVal (Ref l) \<and>
                                    p = tickRo}"

vdm_if: "\<lbrakk>G \<rhd> e1 : P1; G \<rhd> e2 : P2\<rbrakk> \<Longrightarrow> 
         G \<rhd> (IF x THEN e1 ELSE e2) :
          (ifA Pid (\<lambda> (E,h) . E<x>) thenA P1 elseA P2) \<oplus> \<langle>2 0 0 0\<rangle>"


vdm_leti: "\<lbrakk>G \<rhd> e1 : P1; G \<rhd> e2 : P2\<rbrakk> \<Longrightarrow>
           G \<rhd> (Leti x e1 e2) :
           (P1 ;x\<bullet> P2) \<oplus> \<langle>1 0 0 0\<rangle>"

vdm_letr: "\<lbrakk>G \<rhd> e1 : P1; G \<rhd> e2 : P2\<rbrakk> \<Longrightarrow>
           G \<rhd> (Letr x e1 e2) :
           (P1 ;x\<diamondsuit> P2) \<oplus> \<langle>1 0 0 0\<rangle>"

vdm_letv: "\<lbrakk>G \<rhd> e1 : P1; G \<rhd> e2 : P2\<rbrakk> \<Longrightarrow>
           G \<rhd> (Letv e1 e2) :
           (P1 ;\<cdot> P2) \<oplus> \<langle>0 0 0 0\<rangle>"

vdm_call: "\<lbrakk>(G \<union> {(Call f,P)}) \<rhd> (funtable f) : P \<oplus> \<langle> 1 1 0 0\<rangle> \<rbrakk> \<Longrightarrow>
           G \<rhd> (Call f) : P"

vdm_mhinvokestatic:
  "\<lbrakk> (G \<union> {(MH_InvokeStatic C mn,P)}) \<rhd> 
    (methtable C mn) :  
    {(E,h,hh,v,p) . (E,h,hh,v,\<langle>2 0 1 1\<rangle> \<oplus> p) \<in> P} \<rbrakk> \<Longrightarrow>
   G \<rhd> (MH_InvokeStatic C mn) : P"

(* OLD rule; only left in here to ensure some older proofs in VDMderived go through 
vdm_mhinvoke:
  "\<lbrakk> (E'::env)\<lfloor>x\<rfloor> = Ref a ; fmap_lookup (heap.oheap h') a = Some C ;
     (G \<union> {(MH_Invoke x mn,P)}) \<rhd> 
     (methtable C mn) :  
     {(E,h,hh,v,p) . E = E'\<lfloor>self := Ref a\<rfloor> \<and> h = h' \<and> (E',h,hh,v,\<langle>4 0 1 1\<rangle> \<oplus> p) \<in> P} \<rbrakk> \<Longrightarrow>
   G \<rhd> (MH_Invoke x mn) : P"
*)
(* NEW Invoke rule; we need quantification over E' h'
vdm_mhinvoke:
  "\<lbrakk> \<forall> E' h' a C. 
     (qach_QaQ E' h' a x C \<longrightarrow>
      (G \<union> {(MH_Invoke x mn,P)}) \<rhd> 
      (methtable C mn) :  
      {(E,h,hh,v,p) . (E = E'\<lfloor>self := Ref a\<rfloor> \<and> h = h') \<longrightarrow> (E',h,hh,v,\<langle>4 0 1 1\<rangle> \<oplus> p) \<in> P}) \<rbrakk> \<Longrightarrow>
   G \<rhd> (MH_Invoke x mn) : P"
*)

vdm_mhinvoke:
  "\<lbrakk> \<forall> E' h' a C. 
     (qach_QaQ E' h' a x C \<longrightarrow>
      (G \<union> {(MH_Invoke x mn,P)}) \<rhd> 
      (methtable C mn) :  
      {(E,h,hh,v,p) . (E = E'\<lfloor>self := Ref a\<rfloor> \<and> h = h') \<longrightarrow> (E',h,hh,v,\<langle>4 0 1 1\<rangle> \<oplus> p) \<in> P}) \<rbrakk> \<Longrightarrow>
   G \<rhd> (MH_Invoke x mn) : P"


vdm_invokestatic:
  "\<lbrakk> \<forall> E'. 
    (G \<union> {(InvokeStatic C mn y,P)}) \<rhd> 
    (methtable C mn) :  
    {(E,h,hh,v,p) . E = newframe_env Nullref (E'\<lfloor>y\<rfloor>)  \<longrightarrow>
                    (E',h,hh,v,\<langle>3 0 1 1\<rangle> \<oplus> p) \<in> P} \<rbrakk> \<Longrightarrow>
   G \<rhd> (InvokeStatic C mn y) : P"

vdm_invoke:
  "\<lbrakk> \<forall> E' h' a C. 
     qach_QaQ E' h' a x C \<longrightarrow>
     (G \<union> {(Invoke x mn y,P)}) \<rhd> 
     (methtable C mn) :  
    {(E,h,hh,v,p) . E = newframe_env (E'\<lfloor>x\<rfloor>) (E'\<lfloor>y\<rfloor>) \<and> h = h' \<longrightarrow>  (E',h,hh,v,\<langle>5 0 1 1\<rangle> \<oplus> p) \<in> P} \<rbrakk> \<Longrightarrow>
   G \<rhd> (Invoke x mn y) : P"

vdm_ax: "\<lbrakk> (e,P) \<in> G \<rbrakk> \<Longrightarrow> G \<rhd> e : P"

subsection {* Semantic validity *}

text {*
 These definitions give meaning to the judgements of a derivation.
 As everywhere in the logic, we have two versions: one with and one without counter.
*}

constdefs  vdm_validn :: "nat \<Rightarrow> 'a expr \<Rightarrow> vdmassn \<Rightarrow> bool"  ("\<Turnstile>\<^sub>_ _ : _" 50)
"\<Turnstile>\<^sub>n e : P \<equiv> (\<forall> m . m \<le> n \<longrightarrow> (\<forall> E h hh v p . ((E \<turnstile> h,e \<Down>m (hh,v,p)) \<longrightarrow> (E,h,hh,v,p) \<in> P)))"
(*
constdefs reccount::"renv \<Rightarrow> nat"
"reccount r == (callc r) + (invkc r)"

constdefs  vdm_validn :: "nat \<Rightarrow> 'a expr \<Rightarrow> vdmassn \<Rightarrow> bool"  ("\<Turnstile>\<^sub>_ _ : _" 50)
"\<Turnstile>\<^sub>n e : P \<equiv> (\<forall> E h hh v p . ((E \<turnstile> h,e \<Down> hh,v,p) \<and> (reccount p = n) \<longrightarrow> (E,h,hh,v,p) \<in> P))"
*)
constdefs  vdm_valid :: "'a expr \<Rightarrow> vdmassn \<Rightarrow> bool"  ("\<Turnstile> _ : _" 50)
"\<Turnstile> e : P \<equiv> (\<forall> E h hh v p . ((E \<turnstile> h,e \<Down> hh,v,p) \<longrightarrow> (E,h,hh,v,p) \<in> P))"

constdefs vdm_context_validn::"nat \<Rightarrow> 'a vdmcontext \<Rightarrow> bool"  ("|\<Turnstile>\<^sub>_ _" 60)
"|\<Turnstile>\<^sub>n G \<equiv> (\<forall> (e,P) \<in> G . \<Turnstile>\<^sub>n e : P)"

constdefs vdm_context_valid::"'a vdmcontext \<Rightarrow> bool"  ("|\<Turnstile> _" 60)
"|\<Turnstile> G \<equiv> (\<forall> (e,P) \<in> G . \<Turnstile> e : P)"

text {* 
 Several useful lemmas on this notion of validity.
*}

(* All proofs of these simple lemmas are hidden now *)

lemma valid_validn: "\<Turnstile> e : P \<Longrightarrow>  \<Turnstile>\<^sub>n e : P"
(*<*)
by (simp add: vdm_valid_def vdm_validn_def sem_def, fastsimp)
(*by (simp add: vdm_valid_def vdm_validn_def)*)
(*>*)

lemma validn_valid: "(\<forall> n . \<Turnstile>\<^sub>n e : P) \<Longrightarrow> \<Turnstile> e : P"
(*<*)
by (simp add: vdm_valid_def vdm_validn_def sem_def, fastsimp)
(*>*)

lemma ctxt_valid_validn: "( |\<Turnstile> G) \<Longrightarrow> (\<forall> n.( |\<Turnstile>\<^sub>n G))"
(*<*)
apply (simp add: vdm_context_validn_def vdm_context_valid_def)
apply clarsimp
apply (auto elim: valid_validn)
done
(*>*)

text {*
 Useful lemmas over the contexts.
*}

lemma emptyctxn [simp]: "|\<Turnstile>\<^sub>n {}"
(*<*)
by (simp add: vdm_context_validn_def)
(*>*)

lemma ctx_projn[rule_format]: "(e,P) \<in> G \<and> |\<Turnstile>\<^sub>n G \<Longrightarrow> (\<Turnstile>\<^sub>n e : P)"
(*<*)
by (simp add: vdm_context_validn_def, fastsimp)
(*>*)

lemma ctxt_drop: "( |\<Turnstile> {(e,P)} \<union> G) \<Longrightarrow> |\<Turnstile> G"
(*<*)
by (simp add: vdm_context_valid_def)
(*>*)

lemma ctxt_ext[simp]: "( |\<Turnstile> {(e,P)} \<union> G) = ((\<Turnstile> e : P ) \<and> ( |\<Turnstile> G))"
(*<*)
by (simp add: vdm_context_valid_def)
(*>*)

lemma emptyctx [simp]: "|\<Turnstile> {}"
(*<*)
by (simp add: vdm_context_valid_def)
(*>*)

lemma ctx_subn[rule_format]:  "H \<subseteq> G \<and> |\<Turnstile>\<^sub>n G \<Longrightarrow> |\<Turnstile>\<^sub>n H"
(*<*)
by (simp add: vdm_context_validn_def, auto)
(*>*)

lemma ctxt_cons: "\<lbrakk> |\<Turnstile> G; \<Turnstile> e : P\<rbrakk> \<Longrightarrow> |\<Turnstile> G \<union> {(e,P)}"
(*<*)
by (simp add: vdm_context_valid_def)
(*>*)

lemma contxt_validn_valid[rule_format]: "(\<forall> n. |\<Turnstile>\<^sub>n G) \<Longrightarrow> |\<Turnstile> G"
(*<*)
by (simp add: vdm_context_validn_def vdm_context_valid_def 
              vdm_valid_def vdm_validn_def sem_def,
    fastsimp)
(*>*)

lemma contxt_valid_validn: "|\<Turnstile> G \<Longrightarrow> (\<forall> n. |\<Turnstile>\<^sub>n G)"
(*<*)
by (simp add: vdm_context_validn_def vdm_context_valid_def 
              vdm_valid_def vdm_validn_def sem_def,
    fastsimp)
(*>*)

lemma lowerm: "\<lbrakk>  m < n; \<Turnstile>\<^sub>n e:P \<rbrakk> \<Longrightarrow> \<Turnstile>\<^sub>m e : P"
(*<*)
apply (simp add: vdm_validn_def)
apply clarsimp
apply (subgoal_tac "ma < n")
prefer 2 apply simp
apply (erule_tac thin_rl)
apply (erule_tac x="ma" in allE)
apply simp
done
(*>*)

(* lemmas lowerm_suc = lowerm [of n "Suc n"] *)
 
(* yes, I know this is sad; so what? *)
lemma lowerm_suc: "\<lbrakk>  \<Turnstile>\<^sub>Suc n e:P \<rbrakk> \<Longrightarrow> \<Turnstile>\<^sub>n e : P"
(*<*)
apply (simp add: vdm_validn_def)
apply clarsimp
apply (erule_tac x="m" in allE)
apply simp
done
(*>*)

lemma ctxt_lower: "\<lbrakk> |\<Turnstile>\<^sub>n G; m<n \<rbrakk> \<Longrightarrow> |\<Turnstile>\<^sub>m G"
(*<*)
apply (simp add: vdm_context_validn_def)
apply clarsimp
apply (rule lowerm)
apply fastsimp+
done
(*>*)

(* lemmas ctxt_lower_suc = ctxt_lower [of "Suc n" G n]; *)

(* yes, I know this is sad; so what? *)
lemma ctxt_lower_suc: "\<lbrakk> |\<Turnstile>\<^sub>Suc n G \<rbrakk> \<Longrightarrow> |\<Turnstile>\<^sub>n G"
(*<*)
apply (simp add: vdm_context_validn_def)
apply clarsimp
apply (rule lowerm)
apply fastsimp+
done
(*>*)

lemma ctxt_consn: "\<lbrakk> |\<Turnstile>\<^sub>n G; \<Turnstile>\<^sub>n e:P \<rbrakk> \<Longrightarrow> ( |\<Turnstile>\<^sub>n {(e,P)} \<union> G)"
(*<*)
by (simp add: vdm_context_validn_def)
(*>*)

lemma ctxt_insertn: "\<lbrakk> |\<Turnstile>\<^sub>n G; \<Turnstile>\<^sub>n e:P \<rbrakk> \<Longrightarrow> ( |\<Turnstile>\<^sub>n (insert (e,P) G))"
(*<*)
by (simp add: vdm_context_validn_def)
(*>*)

subsection {* Validity in context *}

text {* 
 This is the top level definition of validity we use: it is defined over context,
 expression and assertion.
*}

constdefs
vdm_valid_in_ctxt :: "'a vdmtuple set \<Rightarrow> 'a expr \<Rightarrow> vdmassn \<Rightarrow> bool" ("_ \<Turnstile> _ : _" 75)
  "G \<Turnstile> e : P \<equiv>  ( |\<Turnstile> G ) \<longrightarrow> (\<Turnstile> e:P)"

constdefs
vdm_valid_in_ctxt_n :: "'a vdmtuple set \<Rightarrow> nat \<Rightarrow> 'a expr \<Rightarrow> vdmassn \<Rightarrow> bool" ("_ \<Turnstile>\<^sub> _ _ : _" 75)
  "G \<Turnstile>\<^sub>n e : P \<equiv>  ( |\<Turnstile>\<^sub>n G ) \<longrightarrow> ( \<Turnstile>\<^sub>n e:P)"
(*  "G \<Turnstile>\<^sub>n e : P \<equiv>  \<forall> m . m \<le> n \<longrightarrow> ( |\<Turnstile>\<^sub>m G ) \<longrightarrow> ( \<Turnstile>\<^sub>m e:P)"*)

subsection {* Collecting rules *}

text {*
 We give names to some sensible collections of rules from the program logic.
*}

lemmas vdm_basics = vdm_null vdm_int vdm_ivar vdm_rvar vdm_prim vdm_rprim vdm_getfi vdm_getfr vdm_putfi vdm_putfr vdm_new vdm_if vdm_leti vdm_letr vdm_letv 
lemmas vdm_calls = vdm_call vdm_mhinvokestatic vdm_mhinvoke 

(*<*)
lemmas vdm_pathetic = vdm_null vdm_int vdm_ivar vdm_rvar vdm_prim vdm_rprim vdm_getfi vdm_getfr vdm_putfi vdm_putfr vdm_new 
(*>*)

(*<*)
end
(*>*)

