(*  
   File:	$RCSfile: ToyHLbasic0.thy,v $
   Name:        $Name:  $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyHLbasic0.thy,v 1.1 2003/05/29 11:47:37 a1hloidl Exp $

   Basic rules of Hoare Logic for Toy Grail, with test examples.

   TODO: 
    -- add test examples for invoke

   Aux = state
*)   

header {* Hoare logic for Toy Grail: Basic Rules *}

theory ToyHLbasic0 = ToyGrailLemmas:

subsection {* Basic types and definitions *}

text {* Assertions take two forms: pre-conditions are predicates
  on pre-states; post-conditions are predicates on post-states
  and also the resulting value.  Additionally we include 
  auxiliary variables related in the pre and post condition.
  Auxiliary variables can have any type. *}

(* This code is sponsored by the Association for the Elimination of Polymorphism *)
types
 a_preassn  = "(state \<times> state) set"
 a_postassn = "(state \<times> state \<times> val) set"

types
 a_cntxt = "(a_preassn \<times> expr \<times> a_postassn) set"

constdefs 
  hoare_valid :: "a_preassn \<Rightarrow> expr \<Rightarrow> a_postassn \<Rightarrow> bool"  ("|= (1_)/ (_)/ (1_)" 50)
   "|= P e Q \<equiv> \<forall> s t v. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> 
				  (\<forall> z. (z, s) \<in>  P \<longrightarrow> (z, t, v) \<in> Q)"

syntax (xsymbols)
  hoare_valid :: "a_preassn \<Rightarrow> expr \<Rightarrow> a_postassn \<Rightarrow> bool"  ("\<Turnstile> (1_)/ (_)/ (1_)" 50)

(* extended to sets (for mutual recursion) *)
constdefs
 hoare_valids :: "a_cntxt \<Rightarrow> bool"               ("|\<Turnstile> _" 50)
 "|\<Turnstile> C \<equiv> \<forall>(P,e,Q) \<in> C. \<Turnstile> P e Q"

constdefs
 cntxt_hoare_valid :: "[a_cntxt, a_preassn, expr, a_postassn]  \<Rightarrow> bool" ("(_ /\<Turnstile> (1_)/ (_)/ (1_))" 50)
 "C \<Turnstile> P e Q \<equiv> |\<Turnstile> C \<longrightarrow> \<Turnstile> P e Q"

 cntxt_hoare_valids :: "[a_cntxt, a_cntxt] \<Rightarrow> bool" ("_ |\<Turnstile>/ _" 50)
 "C |\<Turnstile> D  \<equiv>  |\<Turnstile> C \<longrightarrow> |\<Turnstile> D"


subsection {* Global tables of assertions and derived rules *}

(* moved up here to use in earlier rules -- HWL *)

(*
  Notes:
  1. We assume that funames are different in diff meths 
  2. Post assertion for functions may be same as post-assertion as whole
     method, but it may be convenient to have something different;
     in any case, adding post assertions to fun_assn_table may be
     convenient.
*)

(* changed types of {fun,meth}_assn_table to match those in ToyHlwp.thy -- HWL *)
(* I'm having problems feeding rewrite rules for fun_assn_table to the simplifier
   if it's type is polymorphic over the aux var; for now I have specialised it,
   but have to find a general solution (see ToyVCGtest0.thy) -- HWL *)

consts
 fun_preassn_table   ::  "funame \<Rightarrow> a_preassn" 
 fun_postassn_table  ::  "funame \<Rightarrow> a_postassn" 
 fun_assn_table      ::  "funame \<Rightarrow> (a_preassn \<times> a_postassn)" 
 fun_wfmeasure_table ::  "funame \<Rightarrow> (state \<times> state) set"

 meth_preassn_table   ::  "mname \<Rightarrow> a_preassn"
 meth_postassn_table  ::  "mname \<Rightarrow> a_postassn"
 meth_assn_table      ::  "mname \<Rightarrow> (a_preassn \<times> a_postassn)"
 meth_wfmeasure_table ::  "mname \<Rightarrow> (state \<times> state) set"

subsection {* Resource predicates *}

text {* Here are some shorthands for constructing triples over an expression. *}

constdefs
  takestime  :: "nat \<Rightarrow> expr \<Rightarrow> bool"   (* uses an auxiliary variable! *)
  "takestime n e   \<equiv>  \<Turnstile> {(z,s). s = z} e {(z,s,v). clock s = (clock z)+(int n)}"

  takestimelt  :: "nat \<Rightarrow> expr \<Rightarrow> bool"
  "takestimelt n e   \<equiv>  \<Turnstile> {(z,s). s = z} e {(z,s,v). clock s < (clock z)+(int n)}"
  (*
  takesspace :: "nat \<Rightarrow> expr \<Rightarrow> bool"
  "takesspace n e   
    \<equiv>  \<Turnstile> {(z,s). finite(fmap_dom (heap s)) \<and> card (fmap_dom (heap s)) = z} e 
	{(z,s,v). finite(fmap_dom (heap s)) \<and> card (fmap_dom (heap s)) = z+n}"
  *)

constdefs relTakestime:: "state \<Rightarrow> nat \<Rightarrow> expr \<Rightarrow> bool"
"relTakestime S n e \<equiv>  \<Turnstile> {(z,s). s = z \<and> s = S} e {(z,s,v). clock s = (clock z)+(int n)}"

constdefs relTakestimelt:: "state \<Rightarrow> nat \<Rightarrow> expr \<Rightarrow> bool"
"relTakestimelt S n e \<equiv>  \<Turnstile> {(z,s). s = z \<and> s = S} e {(z,s,v). clock s < (clock z)+(int n)}"

(*
constdefs relTakesspace :: "state \<Rightarrow> nat \<Rightarrow> expr \<Rightarrow> bool"
"relTakesspace S n e   
    \<equiv>  \<Turnstile> {(z,s). finite(fmap_dom (heap s)) \<and> card (fmap_dom (heap s)) = z \<and> s = S} e 
	{(z,s,v). finite(fmap_dom (heap s)) \<and> card (fmap_dom (heap s)) = z+n}"
*)

subsection {* Rules *}

text {* Some basic rules for introducing or eliminating a triple. 
  These are helpful for deriving the Hoare proof rules, but we should 
  \emph{not} use them when doing proofs with the Hoare rules! *}

lemma HoareI [intro]: 
  "(\<forall> s v t. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. ((z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q))) \<Longrightarrow> \<Turnstile> P e Q"
by (unfold hoare_valid_def, auto)

lemma HoareE [elim!]: "\<lbrakk> \<Turnstile> P e Q;  \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>;  (z,s)\<in> P \<rbrakk> \<Longrightarrow> (z,t,v)\<in> Q"
by (unfold hoare_valid_def, auto)

lemma presubset_pred [intro]: "\<lbrakk> \<And>z s. P z s \<Longrightarrow> Q z s \<rbrakk> \<Longrightarrow> {(z,s). P z s} \<subseteq> {(z,s). Q z s}"
apply (auto)
done

text {* The following subsections define the basic rules with some trivial
  examples to exercise them. *}

(********************************************************************************)

subsection {* Consequence rules: HSP, HWC, HConseq, HConseqProc *}

(* strengthening precedent (SP) *)
lemma HSP :  "\<lbrakk> \<Turnstile> P' e Q; P \<subseteq> P' \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
by (auto)

(* weakening consequent (WC) *)
lemma HWC :  "\<lbrakk> \<Turnstile> P e Q'; Q' \<subseteq>  Q \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
by (auto)

(* doing both (consequence rule) *)
lemma HConseq: "\<lbrakk> \<Turnstile> P' e Q'; P \<subseteq> P'; Q' \<subseteq>  Q \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"
by (fastsimp)

text {*  The following consequence rule is useful (allegedly) when
  reasoning about procedures. *}

lemma HConseqProc: "\<lbrakk> \<Turnstile> P' e Q'; 
		       \<forall> s t v. 
		        (\<forall> z. (z,s)\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			\<longrightarrow>
		        (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk>
		   \<Longrightarrow> \<Turnstile> P e Q"
by (fastsimp)

text {* The above rule is the one used by Kleymann. Nipkow uses a slightly different 
version of  precondition strengthening  that takes care of
the auxiliary state @{term z}: @{prop"\<forall>s. (\<exists>z. P z s) \<longrightarrow> (\<exists>z. P' z s)"}. *}


lemma HConseqProc1: "\<lbrakk> \<Turnstile> P' e Q'; 
		       (\<forall> s t v. 
		        (\<forall> z. (z,s)\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			\<longrightarrow>
		        (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q))
                        \<and> 
                       (\<forall>s. (\<exists>z. (z,s)\<in>P) \<longrightarrow> (\<exists>z. (z,s)\<in>P')) \<rbrakk>
		   \<Longrightarrow> \<Turnstile> P e Q"
by (fastsimp)

text {*
The following proof of equivalence of the two rules is also due to Nipkow.
*}

lemma "((\<forall>s t. (\<forall>z. (z,s)\<in>P' \<longrightarrow> (z,t,v)\<in>Q') \<longrightarrow> (\<forall>z. (z,s)\<in>P \<longrightarrow> (z,t,v)\<in>Q)) \<and>
            (\<forall>s. (\<exists>z. (z,s)\<in>P) \<longrightarrow> (\<exists>z. (z,s)\<in>P')))
        = (\<forall>z s. (z,s)\<in>P \<longrightarrow> (\<forall>t. \<exists>z'. (z',s)\<in>P' \<and> ((z',t,v)\<in>Q' \<longrightarrow> (z,t,v)\<in>Q)))"
by blast

lemma Hbasic': "\<Turnstile> P e {(z,t,v). \<exists> s. (z,s) \<in> P \<and> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>}"
by (fastsimp)

(* sometimes can use as an [iff] rule? *)
lemma HExAll:  "(\<Turnstile> {(z,s).  \<exists> t. P t z s} e Q) =  (\<forall> t. \<Turnstile> {(z,s). P t z s} e Q)"
by (fastsimp)




(********************************************************************************)

subsection {* HNull *}

lemma HNull0: 
  "\<Turnstile> {(z, s). (z,tick s, RVal Nullref)\<in> Q} NULL Q"
by (auto elim: evalNull_cases)

lemmas HNull = HNull0 [THEN HSP]


(* Tests 
lemma "takesspace 0 NULL"
apply (simp add: takesspace_def)
apply (rule HNull)
apply (simp)
done
*)



(********************************************************************************)

subsection {* HInt *}

lemma HInt0: "\<Turnstile> {(z, s). (z,tick s,IVal i) \<in> Q} (expr.Int i) Q"
by (auto elim: evalInt_cases)

lemmas HInt = HInt0 [THEN HSP]


(* Test HInt *)
lemma "\<Turnstile> {(z, s). True}
	  (expr.Int 5)
         {(z, s, result). result = IVal 5}"
apply (rule HInt)
apply (simp)
done


(********************************************************************************)

subsection {* HVar *}


lemma HVar0: "\<Turnstile> {(z, s). (z, tick s, IVal (s<vn>)) \<in> Q}
		(expr.IVar vn) 
	        Q"
by (fastsimp elim: evalIVar_cases)

lemmas HVar = HVar0 [THEN HSP]

lemma HVarr0: "\<Turnstile> {(z, s). (z, tick s, RVal (s\<lfloor>vn\<rfloor>)) \<in> Q}
		(expr.RVar vn) 
	        Q"
by (fastsimp elim: evalRVar_cases)

lemmas HVarr = HVarr0 [THEN HSP]


(* Test HVar *)
lemma "\<Turnstile> {(z, s). s<vn>=z<vn>}
	  (expr.IVar vn)
         {(z, s, result). result = IVal (z<vn>)}"
apply (rule HVar)
apply (simp)
done

lemma "\<Turnstile> {(z, s). s\<lfloor>vn\<rfloor>=z\<lfloor>vn\<rfloor>}
	  (expr.RVar vn)
         {(z, s, result). result = RVal (z\<lfloor>vn\<rfloor>)}"
apply (rule HVarr)
apply (simp)
done



(********************************************************************************)

subsection {* HPrimop *}


lemma HPrimop0: 
  "\<Turnstile> {(z, s). (z,tickn 3 s, IVal (f (s<vn1>) (s<vn2>))) \<in> P}
      (expr.Primop f vn1 vn2)
      P"
by (fastsimp elim: evalPrimop_cases)

lemmas HPrimop = HPrimop0 [THEN HSP]


(* in this rule we know post state preserves vars; but preconds strong for PC *)
lemma HPrimopNothe: "\<Turnstile> {(z, s). (z,tickn 3 s, IVal (f (s<vn1>) (s<vn2>))) \<in> P}
                  (expr.Primop f vn1 vn2)
		   P"
by (fastsimp elim: evalPrimop_cases)

(* Test HPrimop *)

(* NB: assumption x1~=x2 is not used *)
lemma "x1 ~= x2 \<Longrightarrow>
	\<Turnstile> {(z, s). s<x1> = 2 \<and> s<x2> = 3}
	   (Primop (op+) x1 x2)
         {(z, s, result). result = IVal 5}"
apply (rule HPrimop)
apply (auto)
done


(* Ask Isabelle how long a primop takes: *)
lemma "takestime ?k (expr.Primop f v1 v2)"
apply (simp add: takestime_def)
apply (rule HPrimop)
apply (simp add: tickn_def)
apply (auto)
done

(*
lemma "takesspace ?k (expr.Primop f v1 v2)"
apply (simp add: takesspace_def)
apply (rule HPrimop)
apply (auto)
done
*)


(********************************************************************************)

subsection {* HIf *}


text {* HIf is the first rule which has nested sub-expressions.
  This combines an instance of strengthening pre-condition. 
   *}

lemma HIf : "\<lbrakk> \<Turnstile> P1 e1 Q;  \<Turnstile> P2 e2 Q; 
               P \<subseteq> {(z,s). (s<x>=1 \<longrightarrow> (z,tick s) \<in> P1)  \<and> 
			    (s<x>=0 \<longrightarrow> (z,tick s) \<in> P2)}
               \<rbrakk>
          \<Longrightarrow> \<Turnstile> P (IF x THEN e1 ELSE e2) Q"
by (unfold hoare_valid_def, fastsimp elim!: evalIf_cases)
(* NB: this proof doesn't use hoareE/hoareI *)

lemma HIf' : "\<lbrakk> P \<subseteq> {(z,s). (s<x>=1 \<longrightarrow> (z,tick s) \<in> P1)  \<and> 
			    (s<x>=0 \<longrightarrow> (z,tick s) \<in> P2)};
               \<Turnstile> P1 e1 Q;  \<Turnstile> P2 e2 Q
               \<rbrakk>
          \<Longrightarrow> \<Turnstile> P (IF x THEN e1 ELSE e2) Q"
by (unfold hoare_valid_def, fastsimp elim!: evalIf_cases)



(* Test HIf *)

lemma "\<Turnstile> {(z,s). s<b>=i \<and> (i=0 \<or> i=1) \<and> s<x1>=3 \<and> s<x2>=5}
        (IF b THEN IVar x1 ELSE IVar x2)
         {(z, s, result). \<exists> j. result = IVal j \<and> (j = 3 \<or> j =5)}"
apply (rule HIf)
apply (rule HVar, rule subset_refl)
apply (rule HVar, rule subset_refl)
apply (auto)
done

lemma "takestime ?k (IF b THEN IVar x ELSE IVar y)"
apply (simp add: takestime_def)
apply (rule HIf)
apply (rule HVar, rule subset_refl)
apply (rule HVar, rule subset_refl)
apply (simp add: tickn_def)
apply (auto)
done

(* This one is more of a challenge: takestimelt guesses right only
   because the longer branch comes first. *)
lemma "takestimelt ?k (IF b THEN (IF b' THEN IVar x ELSE IVar y) ELSE IVar z)"
apply (simp add: takestimelt_def)
apply (rule HIf)+
apply (rule HVar, rule subset_refl)
apply (rule HVar, rule subset_refl)
apply (rule subset_refl)
apply (rule HVar, rule subset_refl)
apply (auto)
oops (* INT PROBLEM *)





(********************************************************************************)

subsection {* HLet *}

text {* The version of HLet give seems the preferable version.  It
  reverses the premises to avoid (some) intermediate subgoals.  This does not
  require weakening before application.  Sub expressions are dealt
  with in reverse order for chaining (versions with "0" in name;
  versions without 0 are used by VCG tactic in that way).
   *}


lemma HLet0: "\<lbrakk> \<Turnstile> R e' Q;
	       \<Turnstile> P e {(z,s,v). \<exists> i. v=IVal i \<and> (z,ivarupdate (tick s) x i) \<in> R} \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLeti_cases)

lemma HLet: "\<lbrakk> \<Turnstile> P e {(z,s,v). \<exists> i. v=IVal i \<and> (z,ivarupdate (tick s) x i) \<in> R};
	      \<Turnstile> R e' Q \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLeti_cases)

lemma HLetr0: "\<lbrakk> \<Turnstile> R e' Q;
		\<Turnstile> P e {(z,s,v). \<exists> r. v=RVal r \<and> (z,rvarupdate (tick s) x r) \<in> R} \<rbrakk>
       \<Longrightarrow> \<Turnstile> P (LETR x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLetr_cases)

lemma HLetr: "\<lbrakk> \<Turnstile> P e {(z,s,v). \<exists> r. v=RVal r \<and> (z,rvarupdate (tick s) x r) \<in> R};
	        \<Turnstile> R e' Q \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LETR x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLetr_cases)


(* Test HLet *)

lemma "\<Turnstile> {(z, s). True}
	 LET x=(expr.Int 5) IN expr.IVar x END
         {(z, s, result). (result = IVal 5)}"
apply (rule HLet0)
apply (rule HVar, rule subset_refl)
apply (rule HInt)
apply (simp)
done

lemma "x~=y \<Longrightarrow> \<Turnstile> {(z, s). True}
	 LET x=(expr.Int 5); y=(expr.Int 7) IN  (Primop (op+) x y) END 
         {(z, s, result). (result = IVal 12)}"
apply (rule HLet0)
apply (rule HLet0)
apply (rule HPrimop, rule subset_refl)
apply (rule HInt, rule subset_refl)
apply (rule HInt, simp)
done

(********************************************************************************)

subsection {* HGetFi *}

lemma HGetFi0: "\<Turnstile> {(z, s). \<exists>  a obj. s\<lfloor>vn\<rfloor> = Ref a \<and> 
			            fmap_lookup (heap s) a = Some obj \<and> 
				    (z, tickn 2 s, IVal ((fst (snd obj) f))) \<in> Q}
	        (GetFi vn f)
		Q"
by (fastsimp elim: evalGetFi_cases)

lemmas HGetFi = HGetFi0 [THEN HSP]

lemma HGetFr0: "\<Turnstile> {(z, s). \<exists>  a obj. s\<lfloor>vn\<rfloor> = Ref a \<and> 
			            fmap_lookup (heap s) a = Some obj \<and> 
				    (z, tickn 2 s, RVal ((snd (snd obj)) f)) \<in> Q}
	        (GetFr vn f)
		Q"
by (fastsimp elim: evalGetFr_cases)

lemmas HGetFr = HGetFr0 [THEN HSP]


(*Tests
lemma "\<Turnstile> {(obj, s). s\<lfloor>vn\<rfloor> = Ref a \<and> 
		  fmap_lookup (heap s) a = Some obj}
	   (GetFi vn f)
         {(obj, s, result). result = IVal ((fst (snd obj)) f)}"
apply (rule HGetFi)
apply (auto)
done

lemma "\<Turnstile> {(obj, s). \<exists> a. s\<lfloor>vn\<rfloor> = Ref a \<and> 
		  fmap_lookup (heap s) a = Some obj}
	   (GetFr vn f)
         {(obj, s, result). result = RVal ((snd (snd obj)) f)}"
apply (rule HGetFr)
apply (auto)
done
*)

(* Ask Isabelle how long a GetF takes: *)
lemma "[| s\<lfloor>vn\<rfloor> = Ref a \<and> 
          fmap_lookup (heap s) a = Some obj \<and> obj=(cn,im,rm) |]
       ==> relTakestime s ?k (expr.GetFi vn f)"
apply (simp add: relTakestime_def)
apply (rule HGetFi)
apply (simp add: tickn_def)
apply (auto)
done

(*
lemma "[| s\<lfloor>vn\<rfloor> = Ref a \<and> 
          fmap_lookup (heap s) a = Some obj \<and> obj=(cn,im,rm) |]
       ==> relTakesspace s ?k (expr.GetFi vn f)"
apply (simp add: relTakesspace_def)
apply (rule HGetFi)
apply (simp add: tickn_def)
apply (auto)
done
*)

(********************************************************************************)

subsection {* HPutFi *}

lemma HPutFi0: 
  "\<Turnstile> {(z, s). \<exists> a obj i. 
		 (s\<lfloor>vn\<rfloor> = Ref a \<and> 
			 fmap_lookup (heap s) a = Some obj \<and> 
		         s<valv> = i \<and>
			 (z, tickn 3 (obj_ifieldupdate s a obj f i), IVal i) \<in> Q)}
	        (PutFi vn f valv)
		Q"
by (fastsimp elim: evalPutFi_cases)

lemmas HPutFi = HPutFi0 [THEN HSP]

lemma HPutFr0: 
  "\<Turnstile> {(z, s). \<exists> a obj r. 
		 (s\<lfloor>vn\<rfloor> = Ref a \<and> 
			 fmap_lookup (heap s) a = Some obj \<and> 
		         s\<lfloor>valv\<rfloor> = r \<and>
			 (z, tickn 3 (obj_rfieldupdate s a obj f r), RVal r) \<in> Q)}
	        (PutFr vn f valv)
		Q"
by (fastsimp elim: evalPutFr_cases)

lemmas HPutFr = HPutFr0 [THEN HSP]

lemma HPutFr1: 
  "\<forall> a obj r. \<Turnstile> {(z, s). 
		         (s\<lfloor>vn\<rfloor> = Ref a \<and> 
			 fmap_lookup (heap s) a = Some obj \<and> 
		         s\<lfloor>valv\<rfloor> = r \<and>
			 (z, tickn 3 (obj_rfieldupdate s a obj f r), RVal r) \<in> Q)}
	        (PutFr vn f valv)
		Q"
by (fastsimp elim: evalPutFr_cases)

lemmas HPutFr_ = HPutFr [THEN HSP]

(*Tests
lemma "\<Turnstile> {(z, s). \<exists> a cn fields. 
	s\<lfloor>vn\<rfloor> = Ref a \<and> fmap_lookup (heap s) a = Some (cn,fields) \<and> s<valv> = z}
	   (PutFi vn f valv)
         {(z, s, v) . v = IVal z}"
apply (rule HPutFi)
apply (auto)
done

lemma "\<Turnstile> {(z, s). \<exists> a cn fields. 
	s\<lfloor>vn\<rfloor> = Ref a \<and> fmap_lookup (heap s) a = Some (cn,fields) \<and> s\<lfloor>valv\<rfloor> = z}
	   (PutFr vn f valv)
         {(z, s, v) . v = RVal z}"
apply (rule HPutFr)
apply (auto)
done
*)
(* Ask Isabelle how long a PutF takes: *)
lemma "[| s\<lfloor>vn\<rfloor> = Ref a \<and> 
          fmap_lookup (heap s) a = Some obj \<and> obj=(cn,im,rm) |]
       ==> relTakestime s ?k (expr.PutFi vn f valv)"
apply (simp add: relTakestime_def)
apply (rule HPutFi)
apply (simp add: tickn_def)
apply (auto)
done

(* need {a \<in> fmap_dom (heap s) \<and>} to determine 0 space usage
lemma "[| s\<lfloor>vn\<rfloor> = Ref a \<and> a \<in> fmap_dom (heap s) \<and> 
          fmap_lookup (heap s) a = Some obj \<and> obj=(cn,im,rm) |]
       ==> relTakesspace s ?k (expr.PutFi vn f valv)"
apply (simp add: relTakesspace_def)
apply (rule HPutFi)
apply (auto)
apply (simp add: fmap_dom_def card_insert_if)
done
*)
(********************************************************************************)

subsection {* HNew *}

lemma HNew0: "\<Turnstile> {(z, s). \<exists> a. a = freshloc (fmap_dom (heap s)) \<and>
			(z, tick (newobj s a c), RVal (Ref a)) \<in> P}
	        (New c)
		P"
by (fastsimp elim: evalNew_cases)

lemmas HNew = HNew0 [THEN HSP]

(*
lemma "takesspace 1 (New c)"
apply (unfold takesspace_def)
apply (rule HNew)
apply (simp add: newobj_def)
apply (rule presubset_pred)
apply (subgoal_tac "freshloc (fmap_dom (heap s)) \<notin> (fmap_dom (heap s))")
apply (simp)
apply (rule freshloc, auto)
done
*)
(*Tests*)
lemma "\<Turnstile> {(z, s). freshloc (fmap_dom (heap s)) = a}
	   (New c)
         {(z, s, v) . v = RVal (Ref a)}"
apply (rule HNew)
apply (auto)
done

(* Ask Isabelle how long a New takes: *)
lemma "takestime ?k (expr.New c)"
apply (simp add: takestime_def)
apply (rule HNew)
apply (simp add: tickn_def)
apply (auto)
oops (* INT PROBLEM *)

lemma "takestime 1 (expr.New c)"
apply (simp add: takestime_def)
apply (rule HNew)
apply (simp add: tickn_def)
apply auto
done

(*
lemma "takesspace 1 (New c)"
apply (unfold takesspace_def)
apply (rule HNew)
apply (simp add: newobj_def)
apply (rule presubset_pred)
apply (subgoal_tac "freshloc (fmap_dom (heap s)) \<notin> (fmap_dom (heap s))")
apply (simp)
apply (rule freshloc, auto)
done
*)

(********************************************************************************)


subsection {* HCall --- expand function bodies *}

lemma HCall0: 
  "(\<forall> s'.\<Turnstile> {(z,s). (z, s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q) \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
by (fastsimp elim: evalCall_cases)

lemma HCall1: 
  "(\<Turnstile> {(z,s). \<exists> s'. (z, s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q) \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
by  (fastsimp elim: evalCall_cases)

(* tries to combine HCall and Adapt;
lemma HCall2: 
  "\<Turnstile> {(z,s). \<exists> s' z'. {(s'',v). (z',s'',v) \<in> Q'} \<subseteq> {(s',v). (z,s'',v) \<in> Q } \<and> (z',s) \<in> P \<and> s=tick (incrcallcount s')} (funtable fn) Q' \<Longrightarrow> 
   \<Turnstile> P (CALL fn) Q"
*)
(*
lemma HCall2: 
  "(\<forall> s'.\<Turnstile> P (funtable fn) Q) \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
sorry
*)
(* WRONG; just for testing propagation of assertions *)


lemma HCall5: 
  "\<lbrakk> (\<forall> s'.\<Turnstile> {(z,s). (z, s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q') ; Q'\<subseteq>Q \<rbrakk> \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule "HWC")
apply (rule "HCall0")
apply clarsimp
apply (drule_tac x="s'" in spec)
apply assumption
apply assumption
done

lemma HCall6: 
  "\<lbrakk> (\<forall> s'.\<Turnstile> {(z,s). (z, s')\<in>P' \<and> s=tick (incrcallcount s')} (funtable fn) Q') ; P\<subseteq>P' ; Q'\<subseteq>Q \<rbrakk> \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule "HSP")
apply (rule "HWC")
apply (rule "HCall0")
apply clarsimp
apply (drule_tac x="s'" in spec)
apply assumption
apply assumption
apply assumption
done

(*\<and> s=tick (incrcallcount s')*)
(* Rule used in the tactic w/ invariants to allow instantiating P' and Q' 
   with pre-post-conditions *)
lemma HCallAux: 
  "(\<Turnstile> {(z, s). (z, s') \<in> P' \<and> s = tick (incrcallcount s')} (funtable fn) Q' \<and> Q'\<subseteq>Q) \<Longrightarrow> \<Turnstile> P' (CALL fn) Q"
sorry

lemma HCallAux0: 
  "\<lbrakk> \<Turnstile> P (CALL fn) Q' ; (Q')\<subseteq>Q \<rbrakk> \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule "HWC")
apply fastsimp
apply assumption
done

(* combines call and adaptation rules *)
lemma HCallAux2: 
  "\<lbrakk>  \<Turnstile> P (CALL fn) Q' \<rbrakk> \<Longrightarrow> 
   \<Turnstile> {(z,s). \<exists> z'. {(s',v). (z',s',v) \<in> Q'} \<subseteq> {(s',v). (z,s',v) \<in> Q } \<and> (z',s) \<in> P}  (CALL fn) Q"
apply (rule HoareI)
apply (rule allI)+
apply (rule impI)
apply (rule allI)
apply simp
apply auto
done

lemma HCallAux1: 
  "(\<Turnstile> {(z, s). (z, s') \<in> (fun_preassn_table fn) \<and> s = tick (incrcallcount s')} (CALL fn) Q' \<and> (fun_postassn_table fn)\<subseteq>Q) \<Longrightarrow> \<Turnstile> (fun_preassn_table fn) (CALL fn) Q"
sorry

(* HWL was here; just for experimentation *)
lemma HCall9:
 "\<Turnstile> {(z,s). \<forall> t v.  (z,s) \<in> (fst (fun_assn_table fn)) \<longrightarrow> 
		    (z,t,v) \<in> (snd (fun_assn_table fn)) \<longrightarrow> (z,t,v) \<in> Q}
    (CALL fn) Q"
sorry

lemma HCall: "\<lbrakk> P \<subseteq> P'; 
	       (\<forall> s'.\<Turnstile> {(z,s). (z, s')\<in>P' \<and> s=tick (incrcallcount s')} (funtable fn) Q) \<rbrakk> \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
by (fastsimp elim: evalCall_cases)

(*Tests*)
lemma "     \<Turnstile> {(z,s) . \<Turnstile> {(z,s). \<exists> s'. (z, s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q}
	      (CALL fn)
              Q"
apply (rule HCall0)
oops (* done *)

lemma "takestime k (funtable fn) \<longrightarrow> takestime (Suc k) (CALL fn)"
apply (simp add: takestime_def)
apply (rule)
apply (rule HCall0)
apply (rule, erule HConseqProc)
apply (fastsimp)
done

(*
lemma "takesspace k (funtable fn) \<longrightarrow> takesspace k (CALL fn)"
apply (unfold takesspace_def)
apply (auto)
apply (rule HCall0)
apply (rule, erule HConseqProc)
apply (auto)
done
*)
(********************************************************************************)


subsection {* HCallMutRec --- sets of mutually recursive functions  *}

text {*
This uses the rule Nipkow has in his paper~\cite{Nipkow-CSL02}. It simultaneously
reasons over all functions and uses pre- and post-assertions indexed by function
name (we probably should grab them from the tables). *}

lemma HCallMutRec:
 "\<lbrakk> wf r;
   \<forall>fn s' pre. 
    (\<Union>fn'. {({(z, s). (z,s') \<in> (P fn') \<and> s=tick (incrcallcount s') \<and> 
                       ((fn',s'),(fn,pre)) \<in> r}, 
            CALL fn', 
            (Q fn'))})
     |\<Turnstile> { ({(z, s). (z,s) \<in> (P fn) \<and> s = pre}, funtable fn, (Q fn)) } \<rbrakk>
 \<Longrightarrow> |\<Turnstile> \<Union>fn. {(P fn, (CALL fn), Q fn)}"
apply (unfold hoare_valids_def)
apply (simp)
apply (rule allI)
apply (drule_tac x="a" in spec)
sorry

text {* 
An introduction rule for contextual validity. Not sure whether we need it, but it looks nice. 
*}

lemma HConjI:
 "\<forall>(P,e,Q) \<in> D. C |\<Turnstile> {(P,e,Q)} \<Longrightarrow> C |\<Turnstile> D"
apply (unfold cntxt_hoare_valids_def)
apply (rule impI)
apply (unfold hoare_valids_def)
apply auto
done

lemma HConjE:
 "\<lbrakk> C |\<Turnstile> D ; (P,e,Q) \<in> D \<rbrakk> \<Longrightarrow> C |\<Turnstile> {(P,e,Q)}"
apply (unfold cntxt_hoare_valids_def)
apply (rule impI)
apply simp
apply (unfold hoare_valids_def)
apply auto
done

(********************************************************************************)

subsection {* HInvoke *}


(* typing/total-correctness condition:
    mn \<in> dom (meths (classtable (fst obj1)))  *)

lemma HInvoke:
  "\<forall> obj a s_init. 
     \<Turnstile> {(z, s). s_init\<lfloor>vn1\<rfloor> = (Ref a) \<and> fmap_lookup (heap s_init) a = Some obj \<and> 
                s = newframe (incrinvokecount s_init) mn (Ref a) (s_init\<lfloor>vn2\<rfloor>) \<and>
                (z, s_init) \<in> P}
       (snd (meths (classtable (fst obj)) mn))
      {(z,s,v). \<exists> s'. s'=tickn 5 (s\<lparr> istore := istore s_init,
			            rstore := rstore s_init, 
			            framestack := framestack s_init\<rparr>)
		\<and> (z,s',v) \<in> Q} \<Longrightarrow>
  \<Turnstile> P (Invoke vn1 mn vn2) Q"
apply (unfold hoare_valid_def)
apply (clarify)
apply (erule evalInvoke_cases)
apply (erule_tac x="obj" in allE)
apply (erule_tac x="a" in allE)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe (incrinvokecount s) mn (Ref a) (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply (auto)
done

(* existentials in pre-cond rather than outer universals *)
lemma HInvoke':
  "\<forall> obj s_init.
     \<Turnstile> {(z, s). \<exists> a . 
                s_init\<lfloor>vn1\<rfloor> = (Ref a) \<and> fmap_lookup (heap s_init) a = Some obj \<and> 
                s = newframe (incrinvokecount s_init) mn (Ref a) (s_init\<lfloor>vn2\<rfloor>) \<and>
                (z, s_init) \<in> P}
       (snd (meths (classtable (fst obj)) mn))
      {(z,s,v). \<exists> s'. s'=tickn 5 (s\<lparr> istore := istore s_init,
			            rstore := rstore s_init, 
			            framestack := framestack s_init\<rparr>)
		\<and> (z,s',v) \<in> Q} \<Longrightarrow>
  \<Turnstile> P (Invoke vn1 mn vn2) Q"
apply (unfold hoare_valid_def)
apply (clarify)
apply (erule evalInvoke_cases)
apply (erule_tac x="obj" in allE)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe (incrinvokecount s) mn (Ref a) (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply auto
done

lemma HInvoke0:
      "\<Turnstile> {(z,s) . \<forall> obj a s_init.
                  (\<Turnstile> {(zz, ss). s_init\<lfloor>vn1\<rfloor> = (Ref a) \<and> s_init\<lless>a\<ggreater> = Some obj \<and>
                              s = newframe (incrinvokecount s_init) mn (Ref a) (s_init\<lfloor>vn2\<rfloor>) \<and>
                              (zz, s_init) = (z,s)}
                     (snd (meths (classtable (fst obj)) mn))
                     {(z, s, v). \<exists> s'. s'=tickn 5 (s\<lparr> istore := istore s_init,
			            rstore := rstore s_init, 
			            framestack := framestack s_init\<rparr>) \<and> 
                                 (z, s', v) \<in> Q})}
         (Invoke vn1 mn vn2)
         Q"
apply (unfold hoare_valid_def)
apply (clarify)
apply (erule evalInvoke_cases)
apply (erule_tac x="obj" in allE)
apply (erule_tac x="a" in allE)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe (incrinvokecount s) mn (Ref a) (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply (erule impE)
apply assumption
apply (erule_tac x="z" in allE)
apply (erule impE)
apply clarsimp
sorry

(* FIXME
lemma " (s<vn1> = Some (Ref a1) \<and> 
         fmap_lookup (heap s) a1 = Some (a, b) \<and> 
         s<vn2> = Some arg \<and> 
         relTakestime (newframe (incrinvokecount s) mn dummyvar dummybdy a1 arg) 
                      k 
                     (snd (the (meths (classtable a) mn))))
       \<longrightarrow> relTakestime s (k+5) (Invoke vn1 mn vn2)"
apply (simp add: relTakestime_def takestime_def)
apply (auto)
apply (rule HInvoke)
 FIXME: would be nice to do this proof using consequence rule & simp,
   not expanding hoare_valid !!
   For this step:  apply (erule HConseqProc)  to work, must make exprs
   match in assumption and conclusion.

apply (simp add: tickn_def incrcallcount_def hoare_valid_def )
done
*)

(*
lemma "(s<vn1> = Some (Ref a1) \<and> 
        fmap_lookup (heap s) a1 = Some (a, b) \<and> 
        s<vn2> = Some arg \<and> 
        relTakesspace (newframe (incrinvokecount s) mn dummyvar dummybdy a1 arg) 
                      k 
                      (snd (the (meths (classtable a) mn))))
       \<longrightarrow> relTakesspace s k (Invoke vn1 mn vn2)"
apply (simp add: relTakesspace_def takesspace_def)
apply(auto)
apply (rule HInvoke)
apply(simp add: tickn_def incrcallcount_def hoare_valid_def)
done
*)

(* DA: commented this out because it clutters the context and causes much confusion
   later (e.g. global declaration of n).  Please use locale, or keep this test
   example in a child theory file which is not part of main development. *)
(*
consts 
 l1 :: locn
 dummyarg :: vname
 n :: vname
 q1 :: vname
 two   :: vname
 three :: vname
 stat1 :: vname
 even  :: mname
 cn    :: cname
 fns   :: "fldname => val option "

constdefs evenBody :: expr
"evenBody ==  LET 
                two = (expr.Int (2::int)) ;
                q1 = Primop (% x y . if (x dvd y) then (1::int) else (0::int)) two  n
              IN
                Var q1
              END"

constdefs evenClassdef :: bool
 "evenClassdef == (snd (the (meths (classtable cn) even))) = evenBody"

lemma testInvoke:
"evenClassdef \<Longrightarrow>
 \<Turnstile> {(z,s) . s<three> = Some (IVal 3) & s<stat1> = Some (RVal l1) & fmap_lookup (heap s) l1 = Some (cn,fns) & 
               snd (the ((meths (classtable cn)) even)) = evenBody } 
 (Invoke stat1 even three)
 { (z,s,v). v = IVal 0}"

apply (rule HInvoke)
apply (simp)
apply (rule allI)+
apply (simp add: evenClassdef_def)
apply (simp add: evenBody_def)
oops
*)
(* FIXME: complete this 
apply (rule HLet)
apply (rule HLet)
apply (rule HVar)
apply (rule HPrimop)
apply (rule HSP, rule HInt)
apply (auto)
*)

(********************************************************************************)

subsection {* HInvokeStatic *}

(* typing condition:  mn \<in> dom (meths (classtable c))  *)

lemma HInvokeStatic:
  "\<forall> s_init. 
     \<Turnstile> {(z, s). s = newframe (incrinvokecount s_init) mn Nullref (s_init\<lfloor>vn2\<rfloor>) \<and>
                (z, s_init) \<in> P}
       (snd (meths (classtable c) mn))
      {(z,s,v). \<exists> s'. s'=tickn 4 (s\<lparr> istore := istore s_init,
			            rstore := rstore s_init, 
			            framestack := framestack s_init\<rparr>)
		\<and> (z,s',v) \<in> Q} \<Longrightarrow>
  \<Turnstile> P (InvokeStatic c mn vn2) Q"
apply (unfold hoare_valid_def)
apply (clarify)
apply (erule evalInvokeStatic_cases)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe (incrinvokecount s) mn Nullref (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply (auto)
done


(********************************************************************************)

subsection {* HRec, HRecWF --- "total" correctness rules for recursion *}

text {* From a suggestion of Robert Atkey: we have separate rules
  for recursion, meaning that we can re-use these for both call
  and invoke. *}

text {* First, a rule for mathematical induction, using a function 
  which maps a state to some natural number. *}

lemma fs:
 "(\<Turnstile> {(z,s). \<exists> n. (z,s)\<in> P \<and> (F s = n)} e Q) \<Longrightarrow> (\<Turnstile> P e Q)"
by (simp add: hoare_valid_def)

lemma HRec: 
 "\<lbrakk> \<Turnstile> {(z,s). (z,s)\<in> P \<and> F s = 0} e Q;
    \<forall> n. (\<Turnstile> {(z,s). (z,s)\<in> P \<and> (F s = n)} e Q) \<longrightarrow> 
         (\<Turnstile> {(z,s). (z,s)\<in> P \<and> (F s = Suc n)} e Q) \<rbrakk> \<Longrightarrow> 
   \<Turnstile> P e Q"
apply (rule fs, rule HExAll [THEN iffD2])
apply (rule allI)
apply (rule nat_induct)
apply (assumption)
apply (erule_tac x=na in allE)
apply (erule impE, assumption)
apply (assumption)
done


text {* Now comes the rule using well-founded induction. The formulation of HRecWF is 
  based on the rule given by Nipkow. *}

lemma HRecWF:
 "\<lbrakk> wf r;
    \<forall> s'. (\<Turnstile> {(z,s).(z,s)\<in> P \<and> (s,s')\<in> r} e Q) \<longrightarrow> \<Turnstile> {(z,s).(z,s)\<in> P \<and> s=s'} e Q \<rbrakk>
  \<Longrightarrow> \<Turnstile> P e Q"
apply (rule, rule)
apply (erule_tac a = "s" and 
       P = "\<lambda> s. \<forall> v t. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q)" in wf_induct)
apply (fastsimp)
done

lemma HRecWF':
 "\<lbrakk> \<forall> s'. (\<Turnstile> {(z,s).(z,s)\<in> P \<and> (s,s')\<in> r} e Q) \<longrightarrow> \<Turnstile> {(z,s).(z,s)\<in> P \<and> s=s'} e Q;
   wf r \<rbrakk> 
  \<Longrightarrow> \<Turnstile> P e Q"
apply (rule, rule)
apply (erule_tac a = "s" and 
         P = "\<lambda> s. \<forall> v t. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q)" in wf_induct)
apply (fastsimp)
done


(* Some particular instantiations of HRec: *)

lemma HRecClockWF: 
 "\<lbrakk> \<forall> s'. ((\<Turnstile> {(z,s).(z,s)\<in> P \<and> clock s < clock s'} e Q) \<longrightarrow> 
	   (\<Turnstile> {(z,s).(z,s)\<in> P \<and> s=s'} e Q)) \<rbrakk>
  \<Longrightarrow> \<Turnstile> P e Q"
oops
(*
apply (rule_tac r="inv_image less_than (clock::state\<Rightarrow>int)" in HRecWF)
apply (rule wf_inv_image, rule wf_less_than)
apply (simp add: inv_image_def)
done
*)


lemma HRecCallWF: 
 "\<lbrakk> \<forall> s'. ((\<Turnstile> {(z,s).(z,s)\<in> P \<and> callcount s  < callcount s'} e Q) \<longrightarrow> 
	   (\<Turnstile> {(z,s).(z,s)\<in> P \<and> s=s'} e Q)) \<rbrakk>
  \<Longrightarrow> \<Turnstile> P e Q"
oops
(*
apply (rule_tac r="inv_image less_than (clock::state\<Rightarrow>int)" in HRecWF)
apply (rule wf_inv_image, rule wf_less_than)
apply (simp add: inv_image_def)
done
*)

(********************************************************************************)
(* naked adaption *)

text {* 
A rule of adaptation, i.e. consequence rule in a flavour that is "left-constructive"
so as to be used in a VCG, where we want to pull assertions from right to left
through the program. 
See Kleyman thesis, Eq (3.21) p72 
*}

lemma HKleymanAdapt: 
 "\<lbrakk> \<Turnstile> P' e Q' \<rbrakk>
  \<Longrightarrow> 
  \<Turnstile> {(z,s). \<exists> z'. (z',s) \<in> P' \<and> (\<forall> s' v. (((z',s',v) \<in> Q') \<longrightarrow> ((z,s',v) \<in> Q)))} e Q"
apply (rule HoareI)
apply (rule allI)+
apply (rule impI)
apply (rule allI)
apply auto
apply (erule_tac x="t" in allE)
apply (erule_tac x="v" in allE)
apply (frule HoareE)
apply auto
done

text {* try to pull existential out of pre-cond *}

lemma HKleymanAdapt1: 
 "\<lbrakk> \<Turnstile> P' e Q' \<rbrakk>
  \<Longrightarrow> 
  \<Turnstile> {(z,s). (z',s) \<in> P' \<and> (\<forall> s' v.  (((z',s',v) \<in> Q') \<longrightarrow> ((z,s',v) \<in> Q)))} e Q"
apply (rule HoareI)
apply (rule allI)+
apply (rule impI)
apply (rule allI)
apply (rule impI)
apply (frule HoareE)
apply auto
done

text {* 
Combines CALL rule and a "left-constructive" version of adaptation.
This should give use all we need in the VCG.
*}

lemma HKleymanAdapt00: 
 "\<lbrakk> \<Turnstile> {(z,s). \<exists> s'. (z, s')\<in>P' \<and> s=tick (incrcallcount s')} (funtable fn) Q' \<rbrakk>
  \<Longrightarrow> 
  \<Turnstile> {(z,s). (z',s) \<in> P' \<and> (\<forall> s' v.  (((z',s',v) \<in> Q') \<longrightarrow> ((z,s',v) \<in> Q)))} (CALL fn) Q"
apply (rule HoareI)
apply (rule allI)+
apply (rule impI)
apply (rule allI)
apply (rule impI)
apply (frule HoareE)
apply (auto elim: evalCall_cases)
done

lemma HKleymanAdapt01: 
 "\<lbrakk> \<Turnstile> {(z,s). \<exists> s'. (z, s')\<in>P' \<and> s=tick (incrcallcount s')} (funtable fn) Q' \<rbrakk>
  \<Longrightarrow> 
  \<forall> z'. \<Turnstile> {(z,s). (z',s) \<in> P' \<and> (\<forall> s' v.  (((z',s',v) \<in> Q') \<longrightarrow> ((z,s',v) \<in> Q)))} (CALL fn) Q"
apply (rule allI)+
apply (rule HoareI)
apply (rule allI)+
apply (rule impI)
apply (rule allI)
apply (rule impI)
apply (frule HoareE)
apply (auto elim: evalCall_cases)
done

(*
lemma HKleymanAdapt2: 
 "\<forall> z'. \<Turnstile> {(z,s). \<forall> s' v. (z',s) \<in> P \<and> (((z',s',v) \<in> Q') \<longrightarrow> ((z,s',v) \<in> Q))} e Q 
  \<Longrightarrow> 
  \<Turnstile> P e Q"
sorry
*)

(********************************************************************************)
(* adaption w/ CALL *)

lemma HCallAdapt0: "\<lbrakk> \<Turnstile> P' (funtable fn) Q';
		     \<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule HCall0)
apply (rule allI)
apply (erule_tac x="s'" in allE)
apply (erule HConseqProc)
apply (auto)
done

text {*
Pretty much the Nipkow-style adaptation but with assimilated CALL rule
*}

lemma HCallAdapt00: "\<lbrakk> \<Turnstile> P' (funtable fn) Q';
		     \<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<and>
                     (\<forall> s. (\<exists> z. (z,s) \<in> P) \<longrightarrow> (\<exists> z. (z,s) \<in> P')) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule HCall0)
apply (rule allI)
apply (erule_tac x="s'" in allE)
apply (erule HConseqProc)
apply (auto)
done

(* weakening only *)
lemma HCallAdapt5: "\<lbrakk> \<Turnstile> {(z,s). (z, s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q';
		     \<forall> s t v. 
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q')
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
sorry


(* The "adaptation" rule for call, using table of assertions and without expanding body *)

lemma HCallAdapt: "\<lbrakk> fst (fun_assn_table fn) = P'; 
	             snd (fun_assn_table fn) = Q';
		     \<Turnstile> P' (funtable fn) Q';
		     \<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule HCall0)
apply (rule)
apply (erule_tac x="s'" in allE)
apply (erule HConseqProc)
apply (auto)
done

(* inlined tables, w/ unfolding of body *)
lemma HCallAdapt1: "\<Turnstile> (fun_preassn_table fn) (funtable fn) (fun_postassn_table fn) \<Longrightarrow> 
                    \<lbrakk>\<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> (fun_preassn_table fn) \<longrightarrow> (z,t,v)\<in> (fun_postassn_table fn))
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule HCall0)
apply (rule)
apply (erule_tac x="s'" in allE)
apply (erule HConseqProc)
(* apply  auto *)
(* TODO: fix; broken after changing pre-post-assn back to polymorphic version -- HWL *)
sorry

(* --- just experimentation in the rest of this section --- *)

(* separate tables, w/o unfolding of body *)
lemma HCallAdapt2: "\<lbrakk> fun_preassn_table fn = P'; 
	             fun_postassn_table fn = Q';
		     \<Turnstile> P' (CALL fn) Q';
		     \<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
sorry

(* separate tables, w/o unfolding of body *)
lemma HCallAdapt3: "\<lbrakk> \<Turnstile> (fun_preassn_table fn) (CALL fn) (fun_postassn_table fn);
		     \<forall> s t v. 
		         (\<forall> z. (z,tick (incrcallcount s))\<in> (fun_preassn_table fn) \<longrightarrow> (z,t,v)\<in> (fun_postassn_table fn))
			 \<longrightarrow>
		         (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
apply (rule HConseqProc)
apply blast
sorry

(* different structure of the tables
lemma HCallAdapt5: "\<lbrakk> pre \<subseteq> P; 
	              Q \<subseteq> post;
		      \<forall> s'. \<Turnstile> {(z,s). (z,s')\<in>P \<and> s=tick (incrcallcount s')} (funtable fn) Q \<rbrakk> 
	      \<Longrightarrow> \<Turnstile> P (CALL fn) Q"
by (rule HCall0, assumption)
*)
(*
apply (rule)
apply (erule HConseqProc)
apply (rule allI)+
apply (rule impI)
apply (rule allI)+
apply (rotate_tac 2)
apply (auto)
done
*)

(********************************************************************************)

subsection {* Collecting the rules together *}

text {* These rules are the ones for non-looping code: *}

lemmas HBasicRules = HNull HInt HVar HPrimop HIf 
        HLet HLetr
	HGetFi HGetFr HPutFi HPutFr HNew HCall HInvoke

text {* Adjust the rules known to the classical reasoners so that
  hoare\_valid\_def is never expanded automatically; we restrict to using
  our Hoare rules from now on. *}

declare HoareI [rule del] HoareE [rule del]

(* NB: This is an annoying rule to have present, it breaks some proofs for us *)
declare Collect_split [simp del]  
end
