(*  
   Defunct version: has example of using axiomatic classes for heavy
   overloading.  Not used because seems a bit fragile/risk of 
   potentially confusing type errors.
*)   

header {* The definition of Toy Grail *}

theory ToyGrailDef = Finmap + Main:

text {* 
 Toy Grail is an attempt to reduce Grail to an absolute minimum to
 study and design the bytecode logic of resources. 

  Simplifications include:
  \begin{itemize}
   \item only two types: integer and reference, distinguished in syntax
   \item much type information is lost
   \item no static fields 
   \item methods have only a single (reference) parameter
   \item there is a single syntactic category for terms (function bodies)
   \item we use total maps to model local variables and objects
   %%\item functions have a global namespace
   \end{itemize}

  We define a syntax for programs which includes invariants.
  Because of this, we begin the definition of Toy Grail
  with its machine model.

*}





(****************************************************************************
 *
 * Section 1: MACHINE MODEL
 *
 ****************************************************************************)

 
section {* Machine model *}

subsection {* Locations and references *}

text {*
  We model locations as natural numbers.  This is somewhat concrete,
  but avoids constructing an abstract type or
  assuming the properties we expect. *}

types
  locn = nat

datatype ref = Nullref | Ref locn  

text {* 
  Given a finite set of locations, we can always find a new fresh one.
  *}

constdefs
  freshloc  :: "locn set \<Rightarrow> locn"         
 "freshloc L == Suc (Max (insert 0 L))" 

lemma freshloc: "finite L \<Longrightarrow> freshloc L \<notin> L"
  apply (simp add: freshloc_def)
  apply (subgoal_tac "finite (insert 0 L)")
  apply (erule thin_rl)
  apply (drule Max)
  apply (auto)
  done

subsection {* Types and Names *}

text {* We declare abstract types for names, including some 
  reserved names.  A polymorphic type of variable names is handy later
  for overloading the lookup syntax, and similarly for fields;
  the polymorphism is restricted to a class of (denotations of)
  \emph{grail primitive types}. *}

classes grailprimtype < logic

arities 
  int :: grailprimtype
  ref :: grailprimtype

defaultsort grailprimtype
typedecl 'a name  (* FIXME: declare without using defaultsort? *)
typedecl 'a fldname
defaultsort type

types
  iname = "int name"         -- {* names of integer variables *}
  rname = "ref name"         -- {* names of reference (object) variables *}
  ifldname = "int fldname"   -- {* names of integer fields *}
  rfldname = "ref fldname"   -- {* names of reference fields *}

translations
  "iname"    <= (type)  "int name"
  "rname"    <= (type)  "ref name"
  "ifldname" <= (type)  "int fldname"
  "rfldname" <= (type)  "ref fldname"

consts
  self     :: rname -- {* the self object *}
  param    :: rname -- {* the method parameter *}
  dummyref :: rname -- {* dummy reference used for void assignments *}

axioms
  constdistinct: "distinct [self,param,dummyref]"

typedecl
  funame	-- {* names of functions *}
typedecl 
  cname		-- {* names of classes *}
typedecl
  mname		-- {* names of methods *}



subsection {* Heaps *}

text {* 
  The heap is a map from locations to objects.  
  Conceptually, objects consist of a class name together with a
  mapping of field names to values.  We use Burstall's
  idea~\cite{NipkowXX} of treating
  records (i.e. objects) on the heap as separate
  ``mini''-heaps themselves, so that a heap is split into
  three components: the object heap, the integer heap,
  and the reference heap.  The latter two heaps are
  modelled as total functions, which simplifies
  the formalization.  Of course, we must check
  that the semantics only examines
  the contents of heaps at non-empty locations.
  *}

types 
  oheap    = "locn \<leadsto>\<^sub>f cname"            -- {* object heap; a finite map *}
  iheap    = "ifldname \<Rightarrow> locn \<Rightarrow> int"  -- {* integer heap for each field *}
  rheap    = "rfldname \<Rightarrow> locn \<Rightarrow> ref"  -- {* reference heap for each field *}


text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "oheap"   <= (type)  "nat \<leadsto>\<^sub>f cname"
  "iheap"   <= (type)  "ifldname \<Rightarrow> nat \<Rightarrow> int"
  "rheap"   <= (type)  "rfldname \<Rightarrow> nat \<Rightarrow> ref"



subsection {* Stores *}

text {*
  Like heaps, stores are modelled as total functions for simplicity.
  If the source program is well-typed, it should never attempt to
  access an undefined part of the store. 

  A frame is a method name (which can
  be used for profiling), together with a store.
  For a small-step semantics we would also need 
  a return point and a place to store the return value.
  We do not need to include the class of the currently
  executing method because this can be retrieved from
  the @{text self} variable.  When executing a static
  method, @{text self} is set to null.
 *}

types
  istore   = "iname \<Rightarrow> int"	        -- {* an integer   store *}
  rstore   = "rname \<Rightarrow> ref"          -- {* a reference store *}
  store    = "istore \<times> rstore"
  frame    = "mname \<times> store"

text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "istore"  <= (type)  "iname \<Rightarrow> int"
  "rstore"  <= (type)  "rname \<Rightarrow> ref"
  "store"   <= (type)  "istore \<times> rstore"
  "frame"   <= (type)  "mname \<times> store"


subsection {* States *}

text {* The state contains several resource-aware components:
  \begin{itemize}
  \item maxstack
  \item callcount
  \item invokecount
  \item clock
  \end{itemize}
  These are all represented as integers for ease of reasoning
  in combination with integer program variables, because Isabelle
  doesn't handle the subtyping Nat<Int very well.
*}

record state =
	oheap       :: oheap		 -- {* object heap *}
	iheap	    :: iheap		 -- {* integer heap *}
	rheap	    :: rheap		 -- {* references heap *}
	istore      :: istore	         -- {* integer store on top of stack *}
	rstore      :: rstore	         -- {* ref store on top of stack     *}
        framestack  :: "frame list"      -- {* frame stack		     *}
        maxstack    :: int		 -- {* maximum depth of stack        *}  
        callcount   :: int		 -- {* count of CALL depth	     *}  
        invokecount :: int		 -- {* count of INVOKE depth	     *}  
	clock	    :: int		 -- {* instruction counter	     *}

translations
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int\<rparr>"
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       istore::istore, rstore::rstore, 
		       framestack:: frame list, 
		       maxstack::int, 
		       callcount::int, invokecount::int,
		       clock::int,\<dots>::'a \<rparr>"



subsection  {* Values and assertions *}

text {* Values and assertions are introduced here because
  the syntax of programs can include them.  *}

text {* A value is the result of evaluating an expression. 
  Values may be either integer values or reference values. 
  (We have to collapse the polymorphism here, unfortunately). *}

datatype val = IVal int | RVal ref

text {* Assertions take two forms: pre-conditions are predicates
  on pre-states; post-conditions are predicates on post-states
  and also the value which results from evaluating an
  expression.  Toy Grail exclusively treats side-effecting
  expressions.

  Additionally we include auxiliary variables related in the pre 
  and post condition.  Auxiliary variables can have any type. 
  *}

types
 'a preassn  = "('a \<times> state) set"
 'a postassn = "('a \<times> state \<times> val) set"

text {* Sets are slightly easier to deal with in Isabelle than
  predicates. *}

translations
  "a preassn"   <= (type)  "(a \<times> state) set"
  "a postassn"  <= (type)  "(a \<times> state \<times> val) set"




(****************************************************************************
 *
 * Section 2: SYNTAX
 *
 ****************************************************************************)


section {* Syntax of Toy Grail *}


text {* Simple expressions are the basic unit of execution,
  and denote values.  They are built from operators, constants,
  and previously computed values (names).  An expression may have a
  side effect, but there are no directly nested expressions.
  We model untyped expressions. 

  Compound expressions are built using let expressions.  
  A function body is a let expression.  A function ends with
  tail calls to other functions or method returns.  
  The natural thing to do is to have two syntactic categories
  of terms: one for simple expressions and one for compound
  expressions.  However this makes the whole development 
  more tedious; so instead we define a predicate
  @{text SimpleExpr} which restricts the operational
  semantics in the single case that matters (the
  binding in a let expressions).

  Expressions are augmented with logical annotations, for
  pre-conditions, post-conditions, and termination
  measures.  They are polymorphic in the type of auxiliary
  variables.
*}

datatype 'a expr =
   Int	    int			
 | IVar     iname		
 | Primop   "int \<Rightarrow> int \<Rightarrow> int" iname iname
 | Null	          		          ("NULL")
 | RVar     rname	
 | RPrimop   "ref \<Rightarrow> ref \<Rightarrow> int" rname rname
 | New      cname			  ("NEW _")
 | GetFi    rname ifldname	  	  ("_\<bullet>_" [65,1000] 65)
 | GetFr    rname rfldname	  	  ("_\<bullet>_" [65,1000] 65)
 | PutFi    rname ifldname iname	  ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | PutFr    rname rfldname rname	  ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | Invoke   rname mname rname             ("_\<bullet>_'(_')" [70,1000,65] 61) 
 | InvokeStatic cname mname rname         ("[class _]\<bullet>_'(_')" [70,1000,65] 61) 
 | Leti     iname "'a expr" "'a expr"        
 | Letr     rname "'a expr" "'a expr"
 | If_      iname "'a expr" "'a expr"	  ("(1IF _/ THEN _/ ELSE _)"  [0,0,0] 61)
 | Call     funame			  ("(CALL _)" [0] 60)
 | Pre      "'a preassn" "'a expr"	  ("(PRE _;/ _)" [0,0] 60)   (* REQUIRES *)
 | Post     "'a expr" "'a postassn"	  ("(_/; POST _)" [0,0] 60)  (* ENSURES *)
 | Measure  "(state * state) set" "'a expr" ("(MEASURE _; _)" [0,0] 60) (* VARIANT *)


text {* Notes:
  \begin{enumerate}
  \item Primitive operations (\texttt{RPrimop}) on references might
  include a test for null, and (perhaps) a pointer equality test.

  \item In method invocations, the first argument is either a
  variable pointing to an object (instance methods), or
  a class name (static/class methods).  The second argument
  is the method name, and the third argument is the
  actual parameter, a single reference variable.

  \item If we treated imperative Grail, we might instead use the traditional
  imperative language of commands together with non-side effecting
  expressions.  That would be considerably simpler;
  for example, types of pre-assertions and post-assertions become
  the same, and the syntax needs only one kind of assertion.
  Lets are broken into assignment and composition.
  But that treatment de-emphasises the functional view.
  \end{enumerate}
*}
  


text {* The following black magic is Isabelle gobbledygook to
  introduce some nice concrete syntax. *}

(* let syntax *)
nonterminals
 gletbind gletbinds

syntax
  "_Gbind"      :: "[id, 'a] => gletbind"                ("(2_ =/ _)" 10)
  "_Grefbind"   :: "[id, 'a] => gletbind"                ("(2rf _ =/ _)" 10)
  "_Gvoidbind"  :: "'a => gletbind"                      ("(2 '_ =/ _)" 10)
  ""            :: "gletbind => gletbinds"               ("_")
  "_gbinds"     :: "[gletbind, gletbinds] => gletbinds"  ("_;/ _")
  "_GLets"      :: "[gletbinds, 'inv expr] => 'inv expr" ("(LET (_)/ IN (_) END)" 10)

translations
  "_GLets (_gbinds b bs) e"  == "_GLets b (_GLets bs e)"
  "LET  v = e IN l END"      == "Leti v e l"
  "LET _ = e IN l END"       == "Letr dummyref e l"
  "LET rf v = e IN l END"    == "Letr v e l"


subsection {* Simple expressions *}

text {* Apart from being well-typed, a good program
  must only assign let-bound variables to simple expressions.  In
  particular, it must not have any non-tail calls, which can arise in
  the syntax above, for example, as @{text "LET v=CALL f IN e END"}.
  Programs that break this rule will get stuck in the operational
  semantics. *}

consts
  SimpleExpr :: "'a expr \<Rightarrow> bool"
primrec
  "SimpleExpr Null = True"
  "SimpleExpr (expr.Int i) = True"
  "SimpleExpr (IVar v) = True"
  "SimpleExpr (RVar v) = True"
  "SimpleExpr (Primop f u v) = True"
  "SimpleExpr (RPrimop f u v) = True"
  "SimpleExpr (New c) = True"
  "SimpleExpr (GetFi v f) = True"
  "SimpleExpr (GetFr v f) = True"
  "SimpleExpr (PutFi v f u) = True"
  "SimpleExpr (PutFr v f u) = True"
  "SimpleExpr (Invoke v m u) = True"
  "SimpleExpr (InvokeStatic c m u) = True"
  "SimpleExpr (Leti v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (Letr v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (If_ v e1 e2) = ((SimpleExpr e1) \<and> (SimpleExpr e2))"
  "SimpleExpr (Call f) = False"
  "SimpleExpr (Pre P e) = SimpleExpr e"
  "SimpleExpr (Post e Q) = SimpleExpr e"
  "SimpleExpr (Measure M e) = SimpleExpr e"


subsection {* Functions and methods *}

text {* A Grail function is notionally parameterised on
  a subset of the current variables in scope.  Variables
  not in the formal parameters should not be accessible
  in the body.  However, a new stack frame is not 
  allocated.  We model this simply by ignoring
  the formal parameters, so a Grail function is just
  its body, which is an @{text expr}.  

  A Grail method body is a collection of function bindings,
  together with a main body and a set of local variables used
  within.

  To simplify the representation for ToyGrail, we consider a
  global mapping from function names to function bodies (see later
  below, in class table).
  This means that we do not need to keep a record of the current
  program position (current method or current functions) in the state.

  Methods are parameterised on two things: first the
  self object (self) invoking the method, and then the single formal
  parameter.  Both parameter names are fixed, however, so
  are left implicit.

  When a method is invoked, we might want to know the size of the frame
  to calculate the size of the stack, which is why we record the 
  total set of local variables.
  (Although we do not initialize these).
 *}

types
  'inv methbody = "(iname set \<times> rname set) \<times> 'inv expr"

translations
  "inv methbody" <= (type) "(iname set \<times> rname set) \<times> inv expr"





subsection {* Programs *}

text {* The class table represents the program as a global parameter. 
   Rather than a single table, we split it into four tables,
  representing: functions (global), fields, and methods. *}


consts
  funtable      :: "funame \<Rightarrow> 'a expr"	         -- {* global function names *}
  ifieldtable   :: "cname \<Rightarrow> ifldname list"      -- {* int fields by class *}
  rfieldtable   :: "cname \<Rightarrow> rfldname list"      -- {* ref fields by class *}
  methtable     :: "cname \<Rightarrow> mname \<Rightarrow> 'a expr"   -- {* methods by class *}






(****************************************************************************
 *
 * Section 3: NATURAL SEMANTICS
 *
 ****************************************************************************)

section {* Big-step semantics *}

text {*
  We define a big-step semantics for Toy Grail.  We choose big-step mainly
  for similarity with the functional viewpoint.  With some minor additions
  to states, we could also define a small-step semantics.
  *}

subsection {* Functions for accessing the state *}

text {* We define some semantic accessor functions for accessing
  components of the state. We use an ugly Isabelle trick with 
  axiomatic type classes to overload the lookup syntax. *}

(* NB: Only priority of 1000 is high enough to override curried application in 
   writing "f s<x>".  Even then have ambiguous cases like  0 < s<n>.
   We might restrict the syntax to using ids only? *)

axclass statevar < type
consts get_var::   "state \<Rightarrow> ('a::get_var) name \<Rightarrow> 'a" ("_<_>" [1000,0] 1000)
consts set_var::   "state \<Rightarrow> ('a::put_thing) name \<Rightarrow> 'b \<Rightarrow> state" ("_<_:=_>" [1000,0] 1000)
consts get_field:: "state \<Rightarrow> ('a::put_field) fldname \<Rightarrow> 'a \<Rightarrow> state" ("_<_._>" [1000,0] 1000)
consts put_field:: "state \<Rightarrow> ('a::put_field) fldname \<Rightarrow> 'a \<Rightarrow> state" ("_<_._:=_>" [1000,0] 1000)

(* da: this is a bit of a mess: what was once syntax now becomes constants again, 
   just for sake of overloading syntax. *)
instance int::get_var ..
defs (overloaded)
get_ivar_def: "s<x::iname> \<equiv> istore s x"

instance ref::get_thing ..
defs (overloaded)
get_rvar_def: "s<x::rname> \<equiv> rstore s x"

(* da: adjusted concrete syntax of next one because \<guillemotleft> much easier to type! 
   We could use get_thing for this too, but it means generalising polymorphism in
   get_thing which could be confusing. *)

syntax  
 get_obj  ::      "[state, locn] \<Rightarrow> cname option"	("_\<guillemotleft>_\<guillemotright>" [1000,0] 1000)
translations
 "get_obj s l" == "fmap_lookup (oheap s) l"



subsection {* Functions for updating the state *}

text {* The state updating functions are treated as abstract operations
  on states.  This makes propositions and intermediate assertions
  more readable, at the expense of needing to add numerous
  simplification rules to deal with them. *}


constdefs
 tickn	     :: "int \<Rightarrow> state \<Rightarrow> state"
 "tickn n s \<equiv>  s \<lparr> clock := (clock s) + n \<rparr>"
syntax
 tick	     :: "state \<Rightarrow> state" -- {* a handy abbreviation *}
translations
 "tick s"    == "tickn 1 s"


text {*  Again, some Isabelle hackery for overloaded syntax for state update operations. *}

(* DA: watch polymorphism here; if updating value has wrong type, it won't rewrite *) 

axclass put_thing < type


instance int::put_thing ..
defs (overloaded)
ivarupdate_def: "s<x::iname:=val> \<equiv> (s \<lparr> istore := (istore s)(x := val) \<rparr>)"

instance ref::put_thing ..
defs (overloaded)
rvarupdate_def: "s<x::rname:=val> \<equiv> (s \<lparr> rstore := (rstore s)(x := val) \<rparr>)"


text {* 
  Object initialization simply sets an object with an empty map.  To
  do something more sensible we would need to record type information
  for fields (at least integer vs object).  Instead we may assume that
  the programmer correctly initializes fields immediately after making
  an object, as a condition for static correctness.
 *}

constdefs
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"


text {* Updating fields. *}
 
axclass put_field < type
consts 



constdefs
  obj_ifieldupdate :: "state \<Rightarrow> locn \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> state"
  "obj_ifieldupdate s a f rtv \<equiv> s \<lparr> iheap := (iheap s) (f:= (iheap s f)(a:=rtv)) \<rparr>"

  obj_rfieldupdate :: "state \<Rightarrow> locn \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> state"
  "obj_rfieldupdate s a f rtv \<equiv> s \<lparr> rheap := (rheap s) (f:= (rheap s f)(a:=rtv)) \<rparr>"

text {* A method invocation allocates a new frame on the frame stack. 
        This function adjusts the state accordingly, given a reference
        to the invoking object, and the parameter.
        The new store contains only bindings for the self object
	and the method parameter.  (We might also give initial
        values to the local variables for the method).
	Note that if we are invoking a static method, then the
	self variable will be set to null.
*}

constdefs
  newframe :: "state \<Rightarrow> mname \<Rightarrow> ref \<Rightarrow> ref \<Rightarrow> state"
  "newframe s m objref arg \<equiv> 
	s \<lparr> framestack := (m,(istore s, rstore s))#(framestack s),
	    istore := emptyi,
	    rstore := ((emptyr(self := objref))(param := arg)),
	    invokecount:= (invokecount s) + 1,
            maxstack := max ((int (length (framestack s))) + 1) (maxstack s) \<rparr>"

constdefs
  incrcallcount :: "state \<Rightarrow> state"
  "incrcallcount s \<equiv> s \<lparr> callcount := (callcount s)+1 \<rparr>"


subsection {* Evalution relation *}

consts
  evalexpr  :: "(state \<times> 'a expr \<times> val \<times> state) set"

syntax
 evalexpr_  :: "[state, 'a expr, val, state] \<Rightarrow> bool"      ("\<langle>_,_\<rangle> \<longrightarrow>e \<langle>_,_\<rangle>")

translations
 "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>" == "(s,e,v,s') : evalexpr"


inductive evalexpr intros
 evalNull:    "\<langle>s, expr.Null\<rangle>    \<longrightarrow>e  \<langle>RVal Nullref, tick s\<rangle>"

 evalInt:     "\<langle>s, expr.Int i\<rangle> \<longrightarrow>e \<langle>IVal i, tick s\<rangle>"

 evalIVar:    "\<langle>s, IVar v\<rangle>   \<longrightarrow>e  \<langle>IVal (s<v>), tick s\<rangle>"

 evalRVar:    "\<langle>s, RVar v\<rangle>   \<longrightarrow>e  \<langle>RVal (s<v>), tick s\<rangle>"

 evalPrimop:  "\<langle>s, Primop f vn1 vn2\<rangle>   \<longrightarrow>e  \<langle>IVal (f (s<vn1>) (s<vn2>)), tickn 3 s\<rangle>"

 evalRPrimop: "\<langle>s, RPrimop f vn1 vn2\<rangle>   \<longrightarrow>e  \<langle>IVal (f (s<vn1>) (s<vn2>)), tickn 3 s\<rangle>"

 evalInvoke: 
  "\<lbrakk> s<vn1> = Ref a; s\<guillemotleft>a\<guillemotright> = Some C;
     \<langle>newframe s mn (Ref a) (s<vn2>), methtable C mn\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
    \<langle>s , Invoke vn1 mn vn2\<rangle> \<longrightarrow>e   
    \<langle>rtv, tickn 5 (s'\<lparr>istore := istore s,  rstore := rstore s, 
	              framestack := (framestack s) \<rparr>) \<rangle>"

 evalInvokeStatic: 
  "\<lbrakk> \<langle>newframe s mn Nullref (s<vn2>), methtable C mn\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
    \<langle>s , InvokeStatic C mn vn2\<rangle> \<longrightarrow>e   
    \<langle>rtv, tickn 4 (s'\<lparr>istore := istore s,  rstore := rstore s, 
	              framestack := (framestack s) \<rparr>) \<rangle>"

 evalGetFi: "s<vn> = Ref a  \<Longrightarrow>  \<langle>s,  GetFi vn f\<rangle> \<longrightarrow>e \<langle>IVal (iheap s f a), tickn 2 s\<rangle>"

 evalGetFr: "s<vn> = Ref a  \<Longrightarrow>  \<langle>s,  GetFr vn f\<rangle> \<longrightarrow>e \<langle>RVal (rheap s f a), tickn 2 s\<rangle>"

 evalPutFi: "s<vn1> = Ref a 
	     \<Longrightarrow> 
             \<langle>s, PutFi vn1 f vn2\<rangle> \<longrightarrow>e \<langle> IVal (s<vn2>), tickn 3 (obj_ifieldupdate s a f (s<vn2>)) \<rangle>"


 evalPutFr: "s<vn1> = Ref a
            \<Longrightarrow>
            \<langle>s, PutFr vn1 f vn2\<rangle> \<longrightarrow>e \<langle> RVal (s<vn2>), tickn 3 (obj_rfieldupdate s a f (s<vn2>)) \<rangle>"

 evalNew:  "\<lbrakk> a = freshloc (fmap_dom (oheap s))  \<rbrakk>
            \<Longrightarrow>
            \<langle>s, New c\<rangle> \<longrightarrow>e \<langle>RVal (Ref a), tick (s \<lparr> oheap := (oheap s)(a \<mapsto>\<^sub>f c)\<rparr>) \<rangle>"

 evalCall: "\<lbrakk> \<langle>tick (incrcallcount s), funtable fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
            \<Longrightarrow>
            \<langle>s, Call fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>" 

 evalLeti: "\<lbrakk> \<langle>s, e\<rangle> \<longrightarrow>e \<langle>IVal i, s1\<rangle> ; 
             \<langle>ivarupdate (tick s1) vn i, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>;
	     SimpleExpr e \<rbrakk>
           \<Longrightarrow>
           \<langle>s, Leti vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

 evalLetr: "\<lbrakk> \<langle>s, e\<rangle> \<longrightarrow>e \<langle>RVal r, s1\<rangle> ; 
             \<langle>rvarupdate (tick s1) vn r, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>;
	     SimpleExpr e \<rbrakk>
           \<Longrightarrow>
           \<langle>s, Letr vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

 evalIf_True:  "\<lbrakk> s<v> = 1; \<langle>tick s, l1\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
                \<Longrightarrow>
                \<langle>s, If_ v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>"

 evalIf_False: "\<lbrakk> s<v> = 0; \<langle>tick s, l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
                \<Longrightarrow>
                \<langle>s, If_ v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv,s1\<rangle>"

 evalPre:  "\<lbrakk> (z,s) \<in> P; \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<rbrakk>    \<Longrightarrow> \<langle>s, Pre P e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"

 evalPost: "\<lbrakk> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>; (z,s',v) \<in> Q \<rbrakk> \<Longrightarrow> \<langle>s, Post e Q\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"
 
 evalMeasure: "\<lbrakk> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>; (s,s') \<in> M \<rbrakk> \<Longrightarrow> \<langle>s, Measure M e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"

text {*
  Notes:
  \begin{enumerate}
  \item Auxiliary variables in assertions play no role in program
  evalution, as expected.
    
  \item Put instructions ought to have no return value, but we do not
     include void type, so we return the value stored.

  \item To explain the resource counting in CALL: vn1 = 1, vn2 = 1,
  frame push/pop = 2, and the implicit return = 1.
  NB: we could break up the increment of ticks into before/after 
  CALL.


  \item Many rules omit type-checking premises.
  For example, in get field and put field rules, we should
  check that the object belongs to a class that indeed has
  the fields being accessed.

  \item
  Object initialization in the New rule simply sets an object
  component of the heap.
  The iheap and rheap are left alone.  For the logic (and
  by comparison with Grail), it might be better to have a
  New operator which also initialises the fields of an
  object.  However, if we admit in-place-update in objects,
  it seems difficult to avoid dealing with partly-defined
  data structures.

 \item The rule for evaluating Let contains an additional side
  condition that the expression being bound does not itself contain a
  CALL (non-tail position).  So evaluation gets stuck for non-Grail
  programs.  However, we might relax this and consider the more
  general semantics for a slightly richer language, since this
  side condition has not been needed in the logic so far.

  \item 
   We might tickn 2 in the If rules, but we leave it at one to
   allow for extra store/retrieve which may not be necessary in 
   some cases on the real machine.

  \end{enumerate}
*}
           

end
