(*  
   File:	ToyHLbasic.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyHLderived.thy,v 1.23 2003/06/24 11:56:25 da Exp $

   Derived rules of Hoare Logic for Toy Grail.

   These rules are derived from the basic rules, for use by the 
   VCG or interactively.
*)   

header {* Derived Rules *}

theory ToyHLderived = ToyHLbasic:

text {* 
  This section derives additional rules of the Hoare Logic for Toy Grail.

  These rules are proved sound using the basic rules only.  
  We use them in the VCG, or interactive proofs.
*}

subsection {* Further structural rules *}

(* strengthening precedent (SP) *)
lemma HSP :    "\<lbrakk> \<Turnstile> P' e Q; P \<subseteq> P' \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"   
by (erule HConseq, auto)

(* weakening consequent (WC) *)
lemma HWC :    "\<lbrakk> \<Turnstile> P e Q'; Q' \<subseteq>  Q \<rbrakk> \<Longrightarrow> \<Turnstile> P e Q"  
by (erule HConseq, auto)


text {* The canonical consequence rule @{term HConseq} is the one used by 
  Kleymann~\cite{kleymann98hoare}. 
  Nipkow~\cite{nipkow02:HoareLogics} uses a slightly different 
  version of  precondition strengthening  that takes care of
  the auxiliary state @{term z}: @{prop "\<forall>s. (\<exists>z. P z s) \<longrightarrow> (\<exists>z. P' z s)"}.
*}

lemma HConseqNipkow: "\<lbrakk> \<Turnstile> P' e Q'; 
		       (\<forall> s t v. 
		        (\<forall> z. (z,s)\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			\<longrightarrow>
		        (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q))
                        \<and> 
                        (\<forall>s. (\<exists>z. (z,s)\<in>P) \<longrightarrow> (\<exists>z. (z,s)\<in>P')) \<rbrakk>
		   \<Longrightarrow> \<Turnstile> P e Q"
by (erule HConseq, auto)

text {*
  As an aside, we have the following proof of equivalence of the two rules.
*}

lemma "((\<forall>s t. (\<forall>z. (z,s)\<in>P' \<longrightarrow> (z,t,v)\<in>Q') \<longrightarrow> (\<forall>z. (z,s)\<in>P \<longrightarrow> (z,t,v)\<in>Q)) \<and>
             (\<forall>s. (\<exists>z. (z,s)\<in>P) \<longrightarrow> (\<exists>z. (z,s)\<in>P')))
       = (\<forall>z s. (z,s)\<in>P \<longrightarrow> (\<forall>t. \<exists>z'. (z',s)\<in>P' \<and> ((z',t,v)\<in>Q' \<longrightarrow> (z,t,v)\<in>Q)))"
by (blast intro: HoareI elim: HoareE)

text {*
 This is the adaptation rule as given by Klemann, suitable for backwards
 proof.
*}

lemma assume_adapt: 
   "\<Turnstile> P e Q \<Longrightarrow> 
    \<Turnstile> {(z,s). (\<exists> z'.(z',s) \<in> P \<and>  (\<forall> t v. (z',t,v)\<in> Q \<longrightarrow> (z,t,v)\<in> Q'))} e Q'"
by (rule HConseq, auto)



subsection {* Handy rules for interactive use *}

lemma HLetI: "\<lbrakk> \<Turnstile> R e' Q;
	        \<Turnstile> P e {(z,s,v). (z,ivarupdate (tick s) x  (theival v)) \<in> R} \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
by (rule HLet, auto)


lemma HLetrI: "\<lbrakk> \<Turnstile> R e' Q;
	        \<Turnstile> P e {(z,s,v). (z,rvarupdate (tick s) x  (therval v)) \<in> R} \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET rf x=e IN e' END) Q"
by (rule HLetr, auto)

lemma HLetvI: "\<lbrakk> \<Turnstile> R e' Q; \<Turnstile> P e {(z,s,v). (z,s)\<in> R} \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET _=e IN e' END) Q"
by (rule HLetv, auto)


subsection {* An alternative Call rule *}




subsection {* Rules for recursive procedures *}

lemma HCallRec: "(\<Turnstile> P (CALL fn) Q \<Longrightarrow> \<Turnstile> (imagepre tickcall P) (funtable fn) Q) 
		\<Longrightarrow> \<Turnstile> P (CALL fn) Q"
sorry

lemma HCallRecinv: "(\<Turnstile> (invimagepre tickcall P) (CALL fn) Q \<Longrightarrow> \<Turnstile> P (funtable fn) Q)  \<Longrightarrow>
			 \<Turnstile> (invimagepre tickcall P) (CALL fn) Q"
sorry

lemma HInvokeRec:
 "\<lbrakk> \<forall> s''. (\<Turnstile> P (Invoke vn1 mn vn2) Q) \<longrightarrow> 
           (\<Turnstile> {(z,s'). \<exists> a C s. (z,s) \<in> P \<and> s=s'' \<and> s\<lfloor>vn1\<rfloor> = Ref a \<and> s\<guillemotleft>a\<guillemotright> = Some C \<and> 
                      s' = newframe s mn (Ref a) (s\<lfloor>vn2\<rfloor>) }
              (methtable C mn)
              (apsndofthree (\<lambda> s' . tickn 5 (oldframe s' s'')) ` Q)) \<rbrakk>
  \<Longrightarrow> \<Turnstile> P (Invoke vn1 mn vn2) Q"
sorry

lemma HInvokeStaticRec:
 "\<lbrakk> \<forall> s_init . (\<Turnstile> P (InvokeStatic C mn vn2) Q) \<longrightarrow> 
           (\<Turnstile> {(z,s). s = newframe s_init mn Nullref (s_init\<lfloor>vn2\<rfloor>) \<and> (z,s_init) \<in> P}
               (methtable C mn) 
              {(z,s,v). \<exists> s'. s'=tickn 4 (oldframe s s_init) \<and> (z,s',v) \<in> Q})  \<rbrakk>
  \<Longrightarrow> \<Turnstile> P (InvokeStatic C mn vn2) Q"
sorry

subsection {* Derived rules for annotated recursive procedures *}

text {* This next rule isn't derivable directly using rule HPre, because we
  ought to show that the pre-condition is preserved when the clock/callcount
  is advanced if we take the pre-condition to refer to the body of the
  call.  Instead this rule supposes that the pre condition refers to
  the state *before* the call.  *}

lemma HCallRecPrePost: 
  "\<lbrakk> funtable fn = (PRE P: POST Q: body); 
      \<Turnstile> P (CALL fn) Q \<Longrightarrow> \<Turnstile> (imagepre tickcall P) body Q;
     R\<subseteq>P; Q\<subseteq>S \<rbrakk>
  \<Longrightarrow> \<Turnstile> R (CALL fn) S"
apply (rule HSP, rule HWC)
apply (rule HCallRec)
apply (simp add: HPre0)  (* use fact that annotations are ignored *)
apply (rule HPost)
apply auto
done

text {* This rule treats the pre-condition as surrounding the function
  body, so it proves an invariant with the clock/callcount stepped
  backwards.  Somewhat ugly but works OK.
*}

lemma HCallRecPrePostinv: 
  "\<lbrakk> funtable fn = (PRE P: POST Q: body); 
     \<Turnstile> (invimagepre tickcall P) (CALL fn) Q \<Longrightarrow> \<Turnstile> P body Q;
     R\<subseteq> (invimagepre tickcall P); Q\<subseteq>S \<rbrakk>
  \<Longrightarrow> \<Turnstile> R (CALL fn) S"
apply (rule HSP, rule HWC)
apply (rule HCallRecinv)
apply simp
apply (rule HPre)
apply (rule HPost)
apply auto
done

text {*  Same as above but with adaptation built in. *}

lemma HCallRecPrePostInvAdapt: 
  "\<lbrakk> funtable fn = ((PRE P: POST Q: body)::'a expr); 
     \<Turnstile> (invimagepre tickcall P) (CALL fn) Q \<Longrightarrow> \<Turnstile> P body Q;
     R \<subseteq> {(z,s). (\<exists> z'.((z',tickcall s) \<in> P)
		\<and>  (\<forall> t v. (z',t,v)\<in> Q \<longrightarrow> (z,t,v)\<in> S))} \<rbrakk>
  \<Longrightarrow>  \<Turnstile> R ((CALL fn)::'a expr) S"
apply (rule HConseq)
apply (rule HCallRec)
apply simp
apply (rule HPre)
apply (rule HPost)
apply auto
done


end

