(*  
   File:	ToyHLbasic.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyHLbasicNat.thy,v 1.1 2003/06/08 13:09:15 da Exp $

   Basic rules of Hoare Logic for Toy Grail.

   We take this to be the definition of the logic: additional derived
   rules are used by the VCG.  
*)   

header {* Hoare logic for Toy Grail: Basic Rules *}
theory ToyHLbasicNat = ToyGrailLemmasNat:

text {* This following rule is an annoying rule to have present in
  the simplifier, it breaks some proofs, so we remove it *}

global declare Collect_split [simp del]  


subsection {* Hoare triples defined *}

constdefs 
  hoare_valid :: "'a preassn \<Rightarrow> 'a expr \<Rightarrow> 'a postassn \<Rightarrow> bool"  ("|= (1_)/ (_)/ (1_)" 50)

   "|= P e Q \<equiv> \<forall> s t v.  \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> 
				  (\<forall> z. (z, s) \<in>  P \<longrightarrow> (z, t, v) \<in> Q)"

syntax (xsymbols)
  hoare_valid :: "'a preassn \<Rightarrow> 'a expr \<Rightarrow> 'a postassn \<Rightarrow> bool" ("\<Turnstile> (1_)/ (_)/ (1_)" 50)


subsection {* Natural deduction for triples *}

text {* Here are some basic rules for introducing or eliminating a triple. 
  These are helpful for deriving the Hoare proof rules, but we do 
  \emph{not} use them when doing proofs with the Hoare rules! *}

lemma HoareI [intro]: 
  "(\<forall> s v t. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. ((z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q))) \<Longrightarrow> \<Turnstile> P e Q"
by (unfold hoare_valid_def, auto)

lemma HoareE [elim!]: "\<lbrakk> \<Turnstile> P e Q;  \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>;  (z,s)\<in> P \<rbrakk> \<Longrightarrow> (z,t,v)\<in> Q"
by (unfold hoare_valid_def, auto)

(* FIXME: is this one used?
lemma presubset_pred [intro]: "\<lbrakk> \<And>z s. P z s \<Longrightarrow> Q z s \<rbrakk> \<Longrightarrow> {(z,s). P z s} \<subseteq> {(z,s). Q z s}"
by auto  *)


text {* The following subsections define the basic rules. *}

subsection {* Structural rules *}

text {*  The following generalised consequence rule is due to
  Kleymann~\cite{kleymann98hoare}: *}

lemma HConseq: "\<lbrakk> \<Turnstile> P' e Q'; 
		       \<forall> s t v. 
		        (\<forall> z. (z,s)\<in> P' \<longrightarrow> (z,t,v)\<in> Q')
			\<longrightarrow>
		        (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q) \<rbrakk>
		   \<Longrightarrow> \<Turnstile> P e Q"
by (fastsimp)


text {* The following lemma is given by Kleymann to move
  existentials out of pre-conditions, which reduces the
  number of unknowns in automatic proofs. *}

lemma HExAll:  "(\<Turnstile> {(z,s).  \<exists> t. P t z s} e Q) =  (\<forall> t. \<Turnstile> {(z,s). P t z s} e Q)"
by (fastsimp)


subsection {* Rules for atomic expressions *}

lemma HNull:   "\<Turnstile> {(z, s). (z,tick s, RVal Nullref)\<in> Q} NULL Q"
by (auto elim: evalNull_cases)

lemma HInt:   "\<Turnstile> {(z, s). (z,tick s,IVal i) \<in> Q} (expr.Int i) Q"
by (auto elim: evalInt_cases)

lemma HVar: "\<Turnstile> {(z, s). (z, tick s, IVal (s<vn>)) \<in> Q}
		(expr.IVar vn) 
	        Q"
by (fastsimp elim: evalIVar_cases)

lemma HVarr: "\<Turnstile> {(z, s). (z, tick s, RVal (s\<lfloor>vn\<rfloor>)) \<in> Q}
		(expr.RVar vn) 
	        Q"
by (fastsimp elim: evalRVar_cases)

subsection {* HPrimop, HRPrimop *}

lemma HPrimop: 
  "\<Turnstile> {(z, s). (z,tickn 3 s, IVal (f (s<vn1>) (s<vn2>))) \<in> P}
      (expr.Primop f vn1 vn2)
      P"
by (fastsimp elim: evalPrimop_cases)

lemma HRPrimop: 
  "\<Turnstile> {(z, s). (z,tickn 3 s, IVal (f (s\<lfloor>vn1\<rfloor>) (s\<lfloor>vn2\<rfloor>))) \<in> P}
      (expr.RPrimop f vn1 vn2)
      P"
by (fastsimp elim: evalRPrimop_cases)

lemma HGetFi: "\<Turnstile> {(z, s). ( z, tickn 2 s, IVal (s<(theloc s\<lfloor>vn\<rfloor>)\<bullet>f>) ) \<in> Q}
	        (GetFi vn f)
		Q"
by (fastsimp elim: evalGetFi_cases)

lemma HGetFr: "\<Turnstile> {(z, s). ( z, tickn 2 s, RVal (s\<lfloor>(theloc s\<lfloor>vn\<rfloor>)\<diamondsuit>f\<rfloor>) ) \<in> Q}
	        (GetFr vn f)
		Q"
by (fastsimp elim: evalGetFr_cases)


lemma HPutFi: 
  "\<Turnstile> {(z, s). (z, tickn 3 (s<(theloc s\<lfloor>vn\<rfloor>)\<bullet>f := (s<valv>)>), IVal (s<valv>)) \<in> Q}
	      (PutFi vn f valv)
	      Q"
by (fastsimp elim: evalPutFi_cases)


lemma HPutFr: 
  "\<Turnstile> {(z, s). (z, tickn 3 (s\<lfloor>(theloc s\<lfloor>vn\<rfloor>)\<diamondsuit>f := (s\<lfloor>valv\<rfloor>)\<rfloor>), RVal (s\<lfloor>valv\<rfloor>)) \<in> Q}
	      (PutFr vn f valv)
	       Q"
by (fastsimp elim: evalPutFr_cases)

lemma HNew: "\<Turnstile> {(z, s). (z, tick (s \<lparr> oheap := (oheap s)((freshlocst s) \<mapsto>\<^sub>f c),
				  iheap := iheapflds ifldvals (freshlocst s) (iheap s) s,
			          rheap := rheapflds rfldvals (freshlocst s) (rheap s) s\<rparr>),
			    RVal (Ref (freshlocst s))) \<in> P}
	        (New c ifldvals rfldvals)
		P"
by (fastsimp elim: evalNew_cases)


subsection {* Conditionals *}

text {* HIf is the first rule which has nested sub-expressions.
  This combines an instance of strengthening pre-condition. 
   *}

lemma HIf : "\<lbrakk> \<Turnstile> P1 e1 Q;  \<Turnstile> P2 e2 Q; 
               P \<subseteq> {(z,s). (s<x>=1 \<longrightarrow> (z,tick s) \<in> P1)  \<and> 
			    (s<x>=0 \<longrightarrow> (z,tick s) \<in> P2)}
               \<rbrakk>
          \<Longrightarrow> \<Turnstile> P (IF x THEN e1 ELSE e2) Q"
by (unfold hoare_valid_def, fastsimp elim!: evalIf_cases)
(* NB: this proof doesn't use hoareE/hoareI *)


subsection {* Composition and assignment *}

text {* The rules for Let combine sequencing with assignment. *}

lemma HLet: "\<lbrakk> \<Turnstile> P e {(z,s,v). (z,ivarupdate (tick s) x (theival v)) \<in> R};
	       \<Turnstile> R e' Q \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLeti_cases)

lemma HLetr: "\<lbrakk> \<Turnstile> P e {(z,s,v). (z,rvarupdate (tick s) x (therval v)) \<in> R};
	        \<Turnstile> R e' Q \<rbrakk> 
       \<Longrightarrow> \<Turnstile> P (LET rf x=e IN e' END) Q"
by (simp add: hoare_valid_def, fastsimp elim: evalLetr_cases)

subsection {* Function calls *}

text {*  The basic rule for CALL unwinds the function body. To avoid
  existentials in the pre-condition, we use Isabelle's image operator. 
  The pre-condition is equivalent to: 
  \begin{quote}
  @{term [source] "{(z,s). \<exists> s'. (z, s') \<in> P \<and> s=tick (incrcallcount s')}"}
  \end{quote}
  (which is the version tested so far in examples).
  *}

constdefs 
  apsnd :: "['b \<Rightarrow> 'c, 'a \<times> 'b ] \<Rightarrow> 'a \<times> 'c"
  "apsnd \<equiv> \<lambda> f (a,b). (a, f b)"

declare apsnd_def [simp add]

constdefs 
  apsndofthree :: "['b \<Rightarrow> 'd, 'a \<times> 'b \<times> 'c ] \<Rightarrow> 'a \<times> 'd \<times> 'c"
  "apsndofthree \<equiv> \<lambda> f (a,b,c). (a, f b, c)"

declare apsndofthree_def [simp add]

lemma HCall:
    "\<Turnstile> (apsnd (tickn 1 o incrcallcount) ` P) (funtable fn) Q 
\<Longrightarrow> \<Turnstile> P (CALL fn) Q"
by (fastsimp elim: evalCall_cases)



(********************************************************************************)

subsection {* Method calls *}

text {* The rules for method invocation are the most delicate.  The operational
  semantics rule adjusts the state in two places: creating a new frame before
  evaluating the method body, and then restoring the old frame 
  afterwards.\footnote{With a small step semantics and a return instruction 
   these two steps could be separated.}
  The basic Hoare rule has the same structure, adjusting both the
  pre-condition and the post-condition in the premise.  
  Furthermore, because we use the trick of keeping the 
  old frame in the evaluation rule rather than the machine,
  this rule is really a rule scheme: hence the quantification
  over @{term s_init}.  
  *}
(* typing condition:  mn \<in> dom (meths (classtable c))  *)

lemma HInvokeStatic:
  "\<forall> s_init. 
     \<Turnstile> {(z, s). s = newframe s_init mn Nullref (s_init\<lfloor>vn2\<rfloor>) \<and> (z, s_init) \<in> P}
       (methtable C mn)
      {(z,s,v). \<exists> s'. s'=tickn 4 (oldframe s s_init) \<and> (z,s',v) \<in> Q} \<Longrightarrow>
  \<Turnstile> P (InvokeStatic C mn vn2) Q"
apply (unfold hoare_valid_def, clarify)
apply (erule evalInvokeStatic_cases)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe s mn Nullref (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply (auto)
done


text {* The rule @{text HInvoke} is slightly more complicated.  
  We additionally quantify globally over the location @{text a} of the 
  object, and use @{term "(the (s_init\<guillemotleft>a\<guillemotright>))"} to extract the class
  of the object --- this is a bit ugly, but writing 
  @{term "s_init\<guillemotleft>a\<guillemotright>=Some C"} in the pre-condition (inside
  quantifiers), we would have to show that this gives the same class 
  as the one in the @{text Invoke} rule.
*}

lemma HInvoke:
  "\<forall> a s_init. 
     \<Turnstile> {(z, s). s_init\<lfloor>vn1\<rfloor> = (Ref a) \<and> 
                s = newframe s_init mn (Ref a) (s_init\<lfloor>vn2\<rfloor>) \<and>
                (z, s_init) \<in> P}
      (methtable (the (s_init\<guillemotleft>a\<guillemotright>)) mn)
      {(z,s,v). \<exists> s'. s'=tickn 5 (oldframe s s_init) \<and> (z,s',v) \<in> Q} \<Longrightarrow>
  \<Turnstile> P (Invoke vn1 mn vn2) Q"
apply (unfold hoare_valid_def, clarify)
apply (erule evalInvoke_cases)
apply (erule_tac x="a" in allE)
apply (erule_tac x="s" in allE)
apply (erule_tac x="newframe s mn (Ref a) (s\<lfloor>vn2\<rfloor>)" in allE)
apply (erule_tac x="s'" in allE)
apply (erule_tac x="v" in allE)
apply fastsimp
done

text {*  
  Neither of these rules has been well tested in our examples yet and
  alternative formulations may be possible.
*}



(********************************************************************************)

subsection {* Trivial rules for annotations *}

text {* Because annotations are ignored by the
  operational semantics, we have these trivial rules.  
  They're used to derive further convenient rules later. *}


lemma HPre0: "(\<Turnstile> P (PRE R: e) Q) = (\<Turnstile> P e Q)"
by (auto, (fastsimp elim: evalPre_cases intro: evalPre)+)
lemma HPost0: "(\<Turnstile> P (POST R: e) Q) = (\<Turnstile> P e Q)"
by (auto, (fastsimp elim: evalPost_cases intro: evalPost)+)
lemma HMeasure0: "(\<Turnstile> P (MEASURE M: e) Q) = (\<Turnstile> P e Q)"
by (auto, (fastsimp elim: evalMeasure_cases intro: evalMeasure)+)



subsection {* Rules for annotations (in progress) *}

text {* Annotations are ignored by the operational semantics.
  Their intention as proof-hints is characterised by the following rules,
  which check that pre and post conditions are satisfied. *}

text {* The rule for pre-conditions checks the body under the 
  given pre-condition in the assertion. *}
  

lemma HPre:  "\<lbrakk> \<Turnstile> P1 e Q; P \<subseteq> P1  \<rbrakk> \<Longrightarrow> \<Turnstile> P (PRE P1: e) Q"
by (fastsimp elim: evalPre_cases)

text {* The rule for post-conditions checks the body under the
  strengthened requirement that both the original post condition
  and the post condition in the assertion must be satisfied:
  the idea here is to help chaining post-conditions backwards. *}

lemma HPost: "\<lbrakk> \<Turnstile> P e Q1; Q1 \<subseteq> Q \<rbrakk> \<Longrightarrow> \<Turnstile> P (POST Q1: e) Q"
by (fastsimp elim: evalPost_cases)

text {* The measure annotation is used to decorate an expression
  with a well-founded decreasing relation on states.  The relation
  decreases when the expression @{term e} is evaluated. *}

lemma HMeasure: "\<lbrakk> \<Turnstile> P e Q \<rbrakk> \<Longrightarrow> 
                  \<Turnstile> {(z,s). (z,s)\<in> P \<and> (\<exists> s'. (\<exists> v. (z,s',v)\<in> Q) \<and> (s,s')\<in> M)} (Measure M e) Q"
by (fastsimp elim: evalMeasure_cases)


text {* This is a new rule which hasn't been tested yet (previous formulation
  used tables for storing measures rather than syntax). *}






(********************************************************************************)


subsection {* HRecWF --- total correctness rule for recursion *}

text {* From a suggestion of Robert Atkey: we have a separate rule
  for recursion, meaning that we can re-use @{text HRecWF} to derive
  rules for both call and invoke.  The recursion rule
  uses induction along a well-founded relation \cite{Nipkow,Kleymann}. *}

lemma HRecWF:
 "\<lbrakk> wf r;
    \<forall> s'. (\<Turnstile> {(z,s).(z,s)\<in> P \<and> (s,s')\<in> r} e Q) \<longrightarrow> \<Turnstile> {(z,s).(z,s)\<in> P \<and> s=s'} e Q \<rbrakk>
  \<Longrightarrow> \<Turnstile> P e Q"
apply (rule, rule)
apply (erule_tac a = "s" and 
       P = "\<lambda> s. \<forall> v t. \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<longrightarrow> (\<forall> z. (z,s)\<in> P \<longrightarrow> (z,t,v)\<in> Q)" in wf_induct)
apply (fastsimp)
done



(********************************************************************************)

subsection {* Collecting the rules together *}

text {* The set @{text hoarebasics} are rules used for non-looping code;
  @{text hoareprocs} are the procedure rules.  These rule sets are
  for interactive use, the VCG uses a custom set. *}

lemmas hoarebasics = HNull HInt HVar HVarr HPrimop HRPrimop HIf 
		     HLet HLetr HGetFi HGetFr HPutFi HPutFr HNew 

lemmas hoareprocs  = HCall HInvoke HInvokeStatic



subsection {* Cleanup *}

text {* Finally, we adjust the rules known to the classical reasoners so that
  hoare\_valid\_def is never expanded automatically; we restrict to using
  the Hoare rules defined above from now on. *}

declare HoareI [rule del] HoareE [rule del]

end
