(*  
   File:	ToyGrailLemmas.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyGrailLemmasNat.thy,v 1.2 2003/06/12 10:25:58 da Exp $

   Useful properties, simplification rules for ToyGrail.

   NB: Most of the simplification rules merely characterise the state
   updaters used in the Hoare rules.  It might be more efficient (and certainly
   simpler) not to use these abstract constants at all.
*)   


header  {* Toy Grail Lemmas *}

theory ToyGrailLemmasNat = ToyGrailDefNat:

subsection {* Elimination rules for the operational semantics *}

inductive_cases evalNull_cases 	 : "(s, expr.Null, v, s') \<in> evalexpr"
inductive_cases evalInt_vases 	 : "(s, expr.Int i, v, s') \<in> evalexpr"
inductive_cases evalIVar_cases   : "(s, expr.IVar vn, v, s') \<in> evalexpr"
inductive_cases evalRVar_cases   : "(s, expr.RVar vn, v, s') \<in> evalexpr"
inductive_cases evalInt_cases  	 : "(s, expr.Int i, v, s') \<in> evalexpr"
inductive_cases evalPrimop_cases : "(s, expr.Primop f vn1 vn2, v, s') \<in> evalexpr"
inductive_cases evalRPrimop_cases: "(s, expr.RPrimop f vn1 vn2, v, s') \<in> evalexpr"
inductive_cases evalGetFi_cases	 : "(s, expr.GetFi vn f, v, s') \<in> evalexpr"
inductive_cases evalGetFr_cases	 : "(s, expr.GetFr vn f, v, s') \<in> evalexpr"
inductive_cases evalPutFi_cases	 : "(s, expr.PutFi vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalPutFr_cases	 : "(s, expr.PutFr vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalNew_cases  	 : "(s, New c ifs rfs, v, s') \<in> evalexpr"
inductive_cases evalIf_cases     : "(s, Ifg vn l1 l2, v, s') \<in> evalexpr"
inductive_cases evalLeti_cases 	 : "(s, Leti vn e ls, v, s') \<in> evalexpr"
inductive_cases evalLetr_cases 	 : "(s, Letr vn e ls, v, s') \<in> evalexpr"
inductive_cases evalCall_cases 	 : "(s, Call fn, v, s') \<in> evalexpr"
inductive_cases evalInvoke_cases : "(s, expr.Invoke vn1 mn vn2, v, s') \<in> evalexpr"
inductive_cases evalInvokeStatic_cases : "(s, expr.InvokeStatic c mn vn, v, s') \<in> evalexpr"
inductive_cases evalPre_cases    : "(s, Pre P e, v, s') \<in> evalexpr"
inductive_cases evalPost_cases    : "(s, Post P e, v, s') \<in> evalexpr"
inductive_cases evalMeasure_cases : "(s, Measure M e, v, s') \<in> evalexpr"

text {* We could declare these elimination rules to the automated
  solvers, but they might sometimes get expanded when we don't want it.  
  Instead, let's make a set.
  *}

lemmas eval_cases = evalNull_cases evalInt_cases evalIVar_cases evalRVar_cases
  evalPrimop_cases evalRPrimop_cases 
  evalGetFi_cases evalGetFr_cases
  evalPutFi_cases evalPutFr_cases 
  evalNew_cases evalIf_cases
  evalLeti_cases evalLetr_cases
  evalCall_cases evalInvoke_cases evalInvokeStatic_cases
  evalPre_cases evalPost_cases evalMeasure_cases



subsection {* Some rules for simplifying states *}

text {* To maintain the abstract constants for updating states
  rather than immediately expanding them, we need a bunch of  
  simplifications rules for simplifying state expressions.
*}
 
(* NB: The 8 state updaters are: 

   clock:  tickn 
   stores: ivarupdate rvarupdate
   heap:   obj_ifieldupdate obj_rfieldupdate
   frames: newframe incrcallcount 
*)


text {* First, collect together the definitions of semantic functions. *}

lemmas state_functions [simp] = 
   tickn_def ivarupdate_def rvarupdate_def 
   obj_ifieldupdate_def obj_rfieldupdate_def 
   newframe_def oldframe_def incrcallcount_def 
   newobj_def 


subsubsection {* Tick *} 

text {* We pull all ticks to the outside, and sum them. *}

lemma [simp]: "tickn i (tickn j s)          = tickn (i+j) s"                by simp
lemma [simp]: "ivarupdate (tickn i s) x v   = tickn i (ivarupdate s x v)"   by simp
lemma [simp]: "rvarupdate (tickn i s) x v   = tickn i (rvarupdate s x v)"   by simp
lemma [simp]: "obj_ifieldupdate (tickn i s) a f rtv = 
                tickn i (obj_ifieldupdate s a f rtv)"  		  by simp
lemma [simp]: "obj_rfieldupdate (tickn i s) a f rtv = 
                tickn i (obj_rfieldupdate s a f rtv)"  		  by simp
lemma [simp]: "newframe (tickn i s) m l r = tickn i (newframe s m l r)"   by simp
lemma [simp]: "incrcallcount (tickn i s) = tickn i (incrcallcount s)"	  by simp
lemma [simp]: "oldframe (tickn i s) s' = tickn i (oldframe s s')"	  by simp


lemma iheapflds [rule_format]:
  "istore s1 = istore s2 \<Longrightarrow> \<forall> ih. (iheapflds ifls l ih s1)  = (iheapflds ifls l ih s2)"
by (induct ifls, simp+)

lemma rheapflds [rule_format]:
  "rstore s1 = rstore s2 \<Longrightarrow> \<forall> ih. (rheapflds ifls l ih s1)  = (rheapflds ifls l ih s2)"
by (induct ifls, simp+)

lemma [simp]: "newobj (tickn i s) c ifls rfls  = tickn i (newobj s c ifls rfls)" 
apply (unfold newobj_def)
apply simp
apply (simp add: iheapflds [of s "tickn i s"])
apply (simp add: rheapflds [of s "tickn i s"])
done



subsection {* Projecting components of the state *}

subsubsection {* Projecting istore *}

lemma [simp]: "istore (tickn k s) = istore s"			 by simp
lemma [simp]: "istore (rvarupdate s v val) = istore s"		 by simp
lemma [simp]: "istore (obj_ifieldupdate s a f rtv) = istore s" 	 by simp
lemma [simp]: "istore (obj_rfieldupdate s a f rtv) = istore s" 	 by simp
lemma [simp]: "istore (incrcallcount s) = istore s"		 by simp

lemma [simp]: "istore (ivarupdate s v val) = (istore s)(v:=val)" by simp
lemma [simp]: "istore (newframe s m l r) = emptyi"		 by simp
lemma [simp]: "istore (oldframe s s') = istore s'"		 by simp
lemma [simp]: "istore (newobj s c ifls rfls)  = istore s"        by simp


subsubsection {* Projecting rstore *}

lemma [simp]: "rstore (tickn k s) = rstore s"			 by simp
lemma [simp]: "rstore (ivarupdate s v val) = rstore s"		 by simp
lemma [simp]: "rstore (obj_ifieldupdate s a f rtv) = rstore s"   by simp
lemma [simp]: "rstore (obj_rfieldupdate s a f rtv) = rstore s"   by simp
lemma [simp]: "rstore (incrcallcount s) = rstore s"		 by simp

lemma [simp]: "rstore (rvarupdate s v val) = (rstore s)(v:=val)"   by simp
lemma [simp]: "rstore (newframe s m l r) = emptyr(self:=l, param := r)"
 by simp
lemma [simp]: "rstore (oldframe s s') = rstore s'"		 by simp
lemma [simp]: "rstore (newobj s c ifls rfls)  = rstore s"        by simp




subsubsection {* Projecting (three components of) the heap *}

lemma [simp] :"oheap (tickn i s) = oheap s"	     by simp
lemma [simp]: "oheap (ivarupdate s x v) = oheap s"   by simp
lemma [simp]: "oheap (rvarupdate s x v) = oheap s"   by simp
lemma [simp]: "oheap (newframe s mn  a v) = oheap s" by simp
lemma [simp]: "oheap (oldframe s s') = oheap s"      by simp
lemma [simp]: "oheap (incrcallcount s) = oheap s"    by simp
lemma [simp]: "oheap (obj_ifieldupdate s a f rtv) = oheap s" by simp
lemma [simp]: "oheap (obj_rfieldupdate s a f rtv) = oheap s" by simp
lemma [simp]: "oheap (newobj s c ifls rfls) = (oheap s)((freshlocst s) \<mapsto>\<^sub>f c)" by simp


lemma [simp] :"iheap (tickn i s) = iheap s"	     by simp
lemma [simp]: "iheap (ivarupdate s x v) = iheap s"   by simp
lemma [simp]: "iheap (rvarupdate s x v) = iheap s"   by simp
lemma [simp]: "iheap (newframe s mn  a v) = iheap s" by simp
lemma [simp]: "iheap (oldframe s s') = iheap s"      by simp
lemma [simp]: "iheap (incrcallcount s) = iheap s"    by simp
lemma [simp]: "iheap (obj_rfieldupdate s a f rtv) = iheap s" by simp
lemma [simp]: "iheap (newobj s c ifls rfls) = iheapflds ifls (freshlocst s) (iheap s) s" 
by simp

lemma [simp] :"rheap (tickn i s) = rheap s"	     by simp
lemma [simp]: "rheap (ivarupdate s x v) = rheap s"   by simp
lemma [simp]: "rheap (rvarupdate s x v) = rheap s"   by simp
lemma [simp]: "rheap (newframe s mn  a v) = rheap s" by simp
lemma [simp]: "rheap (oldframe s s') = rheap s"      by simp
lemma [simp]: "rheap (incrcallcount s) = rheap s"    by simp
lemma [simp]: "rheap (obj_ifieldupdate s a f rtv) = rheap s" by simp
lemma [simp]: "rheap (newobj s c ifls rfls) = rheapflds rfls (freshlocst s) (rheap s) s" 
by simp

lemma [simp]: "iheap (obj_ifieldupdate s a f rtv) = (iheap s) (f:= (iheap s f)(a:=rtv))"
by simp

lemma [simp]: "rheap (obj_rfieldupdate s a f rtv) = (rheap s) (f:= (rheap s f)(a:=rtv))"
by simp



subsubsection {* Projecting the framestack *}

lemma [simp]: "framestack (tickn n s) = framestack s"		by simp
lemma [simp]: "framestack (ivarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (rvarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (obj_ifieldupdate s a f rtv) = framestack s" by simp
lemma [simp]: "framestack (obj_rfieldupdate s a f rtv) = framestack s" by simp
lemma [simp]: "framestack (incrcallcount s) = framestack s"     by simp

lemma [simp]: "framestack (newframe s m objadr arg) = 
			(m,(istore s, rstore s))#(framestack s)" by simp

lemma [simp]: "framestack (oldframe s' s) = framestack s"	 by simp

lemma [simp]: "framestack (newobj s c ifls rfls) = framestack s" by simp

subsubsection {* Projecting the maxstack *}

lemma [simp]: "maxstack (tickn n s) = maxstack s"		   by simp
lemma [simp]: "maxstack (ivarupdate s v x) = maxstack s"	   by simp
lemma [simp]: "maxstack (rvarupdate s v x) = maxstack s"	   by simp
lemma [simp]: "maxstack (obj_ifieldupdate s a f rtv) = maxstack s" by simp
lemma [simp]: "maxstack (obj_rfieldupdate s a f rtv) = maxstack s" by simp
lemma [simp]: "maxstack (incrcallcount s) = maxstack s"		   by simp

lemma [simp]: "maxstack (newframe s m adr arg) = 
		max (length (framestack s) + 1) (maxstack s)"    by simp

lemma [simp]: "maxstack (oldframe s' s) = maxstack s'"		   by simp

lemma [simp]: "maxstack (newobj s c ifls rfls) = maxstack s" by simp



subsubsection {* Projecting the clock *}

lemma [simp]: "clock (ivarupdate s v x) = clock s"   	     by simp
lemma [simp]: "clock (rvarupdate s v x) = clock s"           by simp     
lemma [simp]: "clock (obj_ifieldupdate s a f rtv) = clock s" by simp
lemma [simp]: "clock (obj_rfieldupdate s a f rtv) = clock s" by simp
lemma [simp]: "clock (incrcallcount s) = clock s"    	     by simp
lemma [simp]: "clock (newframe s mn  a v) = clock s"	     by simp     
lemma [simp]: "clock (oldframe s s') = clock s"	             by simp     

lemma [simp]: "clock (tickn i s) = i + (clock s)"	     by simp

lemma [simp]: "clock (newobj s c ifls rfls) = clock s" by simp

subsubsection {* Projecting the callcount *}

lemma [simp]: "callcount (tickn i s) = callcount s"		     by simp
lemma [simp]: "callcount (ivarupdate s v x) = callcount s"   	     by simp
lemma [simp]: "callcount (rvarupdate s v x) = callcount s"           by simp     
lemma [simp]: "callcount (obj_ifieldupdate s a f rtv) = callcount s" by simp
lemma [simp]: "callcount (obj_rfieldupdate s a f rtv) = callcount s" by simp
lemma [simp]: "callcount (incrcallcount s) = (callcount s) + 1"	     by simp
lemma [simp]: "callcount (newframe s mn  a v) = callcount s"	     by simp     
lemma [simp]: "callcount (oldframe s s') = callcount s"   	     by simp     
lemma [simp]: "callcount (newobj s c ifls rfls) = callcount s" by simp

subsubsection {* Projecting the invokecount *}

lemma [simp]: "invokecount (tickn i s) = invokecount s"		     by simp
lemma [simp]: "invokecount (ivarupdate s v x) = invokecount s"   	     by simp
lemma [simp]: "invokecount (rvarupdate s v x) = invokecount s"           by simp     
lemma [simp]: "invokecount (obj_ifieldupdate s a f rtv) = invokecount s" by simp
lemma [simp]: "invokecount (obj_rfieldupdate s a f rtv) = invokecount s" by simp
lemma [simp]: "invokecount (incrcallcount s) = invokecount s"	     by simp
lemma [simp]: "invokecount (newframe s mn  a v) = invokecount s + 1"	     by simp     
lemma [simp]: "invokecount (oldframe s s') = invokecount s"   	     by simp     
lemma [simp]: "invokecount (newobj s c ifls rfls) = invokecount s" by simp


subsection {* Injectivity of state functions *}

lemma tickn_inj: "inj (tickn i)"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

lemma incrcallcount_inj: "inj incrcallcount"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

(* Could do similarly for maxstack, invokecount, etc. *)


subsection {* Monotonicity of resource parameters *}

text {* Here are the functions which are used in the operational
  semantics to change state. *}

lemma callcount_mono  : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> callcount s <= callcount t"
 by (erule evalexpr.induct, simp_all)

lemma invokecount_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> invokecount s <= invokecount t"
 by (erule evalexpr.induct, simp_all)

lemma maxstack_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> maxstack s <= maxstack t"
 by (erule evalexpr.induct, simp_all)

(* clock_mono actually says that clock *increases* on each evaluation *)
lemma clock_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> clock s < clock t"
by (erule evalexpr.induct, simp_all)

lemma heap_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> (fmap_dom (oheap s) \<subseteq> fmap_dom (oheap t))"
by (erule evalexpr.induct, simp_all, auto)

text {* Now we remove the raw definitions of the state functions
  from the simplifier set. *}

declare state_functions [simp del]

subsection {* State/record simplifications *}

(*Lenb: Some lemmas which simplify states by contracting subsequent
        updates - the components are ordered in the order given in the
        definition of states*)

(* successive updates *)
lemma[simp]: "s\<lparr>callcount := A\<rparr>\<lparr>callcount := B\<rparr> = s\<lparr>callcount := B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>clock := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>istore := A\<rparr>\<lparr>istore := B\<rparr> = s\<lparr>istore :=B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>rstore := A\<rparr>\<lparr>rstore := B\<rparr> = s\<lparr>rstore :=B\<rparr>"  by(simp)

(* commutativity *)
lemma[simp]: "s\<lparr>istore := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>\<lparr>istore :=A\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>rstore := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>\<lparr>rstore :=A\<rparr>"  by(simp)



subsection {* Heap size and heap properties *}

text {* Some trivial heap properties *}

lemma [simp]: "finite (fmap_dom (heap s))"  by auto

lemma theloc_ref [simp]: "s\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> theloc s\<lfloor>x\<rfloor> = a"
by auto

lemma theival_int [simp]: "x = IVal i \<Longrightarrow> theival x = i"
by auto

lemma therval_ref [simp]: "x = RVal r \<Longrightarrow> therval x = r"
by auto

subsection {* Calculating the size of the heap *}

constdefs hpsize ::"state \<Rightarrow> int"
  "hpsize s == int (card (fmap_dom (oheap s)))"

declare hpsize_def [simp]

lemma hpsizeTick[simp]:"hpsize (tickn n s) = hpsize s" by simp
lemma hpsizeIvarUpd[simp]:"hpsize (s<x:=v>) = hpsize s" by simp
lemma hpsizeRvarUpd[simp]:"hpsize (s\<lfloor>x:=v\<rfloor>) = hpsize s" by simp
lemma hpsizeIfldUpd[simp]:"hpsize (s<a\<bullet>f:=v>) = hpsize s" by simp
lemma hpsizeRfkdUpd[simp]:"hpsize (s\<lfloor>a\<diamondsuit>f:=v\<rfloor>) = hpsize s" by simp
lemma hpsizeIncrCallCount[simp]:"hpsize (incrcallcount s) = hpsize s" by simp

lemma hpsizeInsert[simp]: "hpsize (newobj s c ifls rfls) = hpsize s + 1"
by (simp add: freshloc)

end
