(*  
   File:	$RCSfile: ToyGrailLemmasBD.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyGrailLemmasBD.thy,v 1.5 2003/07/09 20:58:43 a1hloidl Exp $

   Useful properties, simplification rules for ToyGrail.

   NB: Most of the simplification rules merely characterise the state
   updaters used in the Hoare rules.  It might be more efficient (and certainly
   simpler) not to use these abstract constants at all.
*)   


header  {* Toy Grail Lemmas *}

theory ToyGrailLemmasBD = ToyGrailDefBD:

subsection {* Elimination rules for the operational semantics *}

inductive_cases evalNull_cases 	 : "(s, expr.Null, v, s') \<in> evalexpr"
inductive_cases evalInt_vases 	 : "(s, expr.Int i, v, s') \<in> evalexpr"
inductive_cases evalIVar_cases   : "(s, expr.IVar vn, v, s') \<in> evalexpr"
inductive_cases evalRVar_cases   : "(s, expr.RVar vn, v, s') \<in> evalexpr"
inductive_cases evalInt_cases  	 : "(s, expr.Int i, v, s') \<in> evalexpr"
inductive_cases evalPrimop_cases : "(s, expr.Primop f vn1 vn2, v, s') \<in> evalexpr"
inductive_cases evalRPrimop_cases: "(s, expr.RPrimop f vn1 vn2, v, s') \<in> evalexpr"
inductive_cases evalGetFi_cases	 : "(s, expr.GetFi vn f, v, s') \<in> evalexpr"
inductive_cases evalGetFr_cases	 : "(s, expr.GetFr vn f, v, s') \<in> evalexpr"
inductive_cases evalPutFi_cases	 : "(s, expr.PutFi vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalPutFr_cases	 : "(s, expr.PutFr vn1 f vn2, v, s') \<in> evalexpr"
inductive_cases evalNew_cases  	 : "(s, New c ifs rfs, v, s') \<in> evalexpr"
inductive_cases evalIf_cases     : "(s, Ifg vn l1 l2, v, s') \<in> evalexpr"
inductive_cases evalLeti_cases 	 : "(s, Leti vn e ls, v, s') \<in> evalexpr"
inductive_cases evalLetr_cases 	 : "(s, Letr vn e ls, v, s') \<in> evalexpr"
inductive_cases evalLetv_cases 	 : "(s, Letv e ls, v, s') \<in> evalexpr"
inductive_cases evalCall_cases 	 : "(s, Call fn, v, s') \<in> evalexpr"
inductive_cases evalInvoke_cases : "(s, expr.Invoke vn1 mn vn2, v, s') \<in> evalexpr"
inductive_cases evalInvokeStatic_cases : "(s, expr.InvokeStatic c mn vn, v, s') \<in> evalexpr"
inductive_cases evalPre_cases    : "(s, Pre P e, v, s') \<in> evalexpr"
inductive_cases evalPost_cases    : "(s, Post P e, v, s') \<in> evalexpr"
inductive_cases evalMeasure_cases : "(s, Measure M e, v, s') \<in> evalexpr"

text {* We could declare these elimination rules to the automated
  solvers, but they might sometimes get expanded when we don't want it.  
  Instead, let's make a set.
  *}

lemmas eval_cases = evalNull_cases evalInt_cases evalIVar_cases evalRVar_cases
  evalPrimop_cases evalRPrimop_cases 
  evalGetFi_cases evalGetFr_cases
  evalPutFi_cases evalPutFr_cases 
  evalNew_cases evalIf_cases
  evalLeti_cases evalLetr_cases
  evalCall_cases evalInvoke_cases evalInvokeStatic_cases
  evalPre_cases evalPost_cases evalMeasure_cases



subsection {* Some rules for simplifying states *}

text {* To maintain the abstract constants for updating states
  rather than immediately expanding them, we need a bunch of  
  simplifications rules for simplifying state expressions.
*}
 
(* NB: The 8 state updaters are: 

   clock:  tickn 
   stores: ivarupdate rvarupdate
   heap:   obj_ifieldupdate obj_rfieldupdate
   frames: newframe incrcallcount 
*)


text {* First, collect together the definitions of semantic functions. *}

lemmas state_functions [simp] = 
   tickn_def ivarupdate_def rvarupdate_def 
   obj_ifieldupdate_def obj_rfieldupdate_def 
   newframe_def oldframe_def popframe_def incrcallcount_def 
   newobj_def 
(*    newframe_def oldframe_def popframe_def topframe_ist_def topframe_rst_def incrcallcount_def 
*)

subsubsection {* Tick *} 

text {* We pull all ticks to the outside, and sum them. *}

lemma [simp]: "tickn i (tickn j s)          = tickn (i+j) s"                by simp
lemma [simp]: "ivarupdate (tickn i s) x v   = tickn i (ivarupdate s x v)"   by simp
lemma [simp]: "rvarupdate (tickn i s) x v   = tickn i (rvarupdate s x v)"   by simp
lemma [simp]: "obj_ifieldupdate (tickn i s) a f rtv = 
                tickn i (obj_ifieldupdate s a f rtv)"  		  by simp
lemma [simp]: "obj_rfieldupdate (tickn i s) a f rtv = 
                tickn i (obj_rfieldupdate s a f rtv)"  		  by simp
lemma [simp]: "newframe (tickn i s) m l r = tickn i (newframe s m l r)"   by simp
lemma [simp]: "incrcallcount (tickn i s) = tickn i (incrcallcount s)"	  by simp
lemma [simp]: "oldframe (tickn i s) s' = tickn i (oldframe s s')"	  by simp
lemma [simp]: "popframe (tickn i s) = tickn i (popframe s)"	   by (simp add: popframe_def topframe_ist_def topframe_rst_def)


lemma iheapflds [rule_format]:
  "istore s1 = istore s2 \<Longrightarrow> \<forall> ih. (iheapflds ifls l ih s1)  = (iheapflds ifls l ih s2)"
by (induct ifls, simp+)

lemma rheapflds [rule_format]:
  "rstore s1 = rstore s2 \<Longrightarrow> \<forall> ih. (rheapflds ifls l ih s1)  = (rheapflds ifls l ih s2)"
by (induct ifls, simp+)

lemma [simp]: "newobj (tickn i s) c ifls rfls  = tickn i (newobj s c ifls rfls)" 
apply (unfold newobj_def)
apply simp
apply (simp add: iheapflds [of s "tickn i s"])
apply (simp add: rheapflds [of s "tickn i s"])
done



subsection {* Projecting components of the state *}

subsubsection {* Projecting istore *}

lemma [simp]: "istore (tickn k s) = istore s"			 by simp
lemma [simp]: "istore (rvarupdate s v val) = istore s"		 by simp
lemma [simp]: "istore (obj_ifieldupdate s a f rtv) = istore s" 	 by simp
lemma [simp]: "istore (obj_rfieldupdate s a f rtv) = istore s" 	 by simp
lemma [simp]: "istore (incrcallcount s) = istore s"		 by simp

lemma [simp]: "istore (ivarupdate s v val) = (istore s)(v:=val)" by simp
lemma [simp]: "istore (newframe s m l r) = emptyi"		 by simp
lemma [simp]: "istore (oldframe s s') = istore s'"		 by simp
lemma [simp]: "istore (popframe s) = fst (snd (hd (framestack s)))"		 by (simp add: popframe_def topframe_ist_def topframe_rst_def)
(*++ lemma [simp]: "istore (popframe s) = topframe_ist s"		 by (simp add: popframe_def topframe_ist_def topframe_rst_def) ++*)
lemma [simp]: "istore (newobj s c ifls rfls)  = istore s"        by simp


subsubsection {* Projecting rstore *}

lemma [simp]: "rstore (tickn k s) = rstore s"			 by simp
lemma [simp]: "rstore (ivarupdate s v val) = rstore s"		 by simp
lemma [simp]: "rstore (obj_ifieldupdate s a f rtv) = rstore s"   by simp
lemma [simp]: "rstore (obj_rfieldupdate s a f rtv) = rstore s"   by simp
lemma [simp]: "rstore (incrcallcount s) = rstore s"		 by simp

lemma [simp]: "rstore (rvarupdate s v val) = (rstore s)(v:=val)"   by simp
lemma [simp]: "rstore (newframe s m l r) = emptyr(self:=l, param := r)"
 by simp
lemma [simp]: "rstore (oldframe s s') = rstore s'"		 by simp
lemma [simp]: "rstore (popframe s) = snd (snd (hd (framestack s)))"		 by (simp add: popframe_def topframe_ist_def topframe_rst_def)
(*++ lemma [simp]: "rstore (popframe s) = topframe_rst s"		 by (simp add: popframe_def topframe_ist_def topframe_rst_def) ++*)
lemma [simp]: "rstore (newobj s c ifls rfls)  = rstore s"        by simp




subsubsection {* Projecting (three components of) the heap *}

lemma [simp] :"oheap (tickn i s) = oheap s"	     by simp
lemma [simp]: "oheap (ivarupdate s x v) = oheap s"   by simp
lemma [simp]: "oheap (rvarupdate s x v) = oheap s"   by simp
lemma [simp]: "oheap (newframe s mn  a v) = oheap s" by simp
lemma [simp]: "oheap (oldframe s s') = oheap s"      by simp
lemma [simp]: "oheap (popframe s) = oheap s"      by (simp add: popframe_def topframe_ist_def topframe_rst_def)
lemma [simp]: "oheap (incrcallcount s) = oheap s"    by simp
lemma [simp]: "oheap (obj_ifieldupdate s a f rtv) = oheap s" by simp
lemma [simp]: "oheap (obj_rfieldupdate s a f rtv) = oheap s" by simp
lemma [simp]: "oheap (newobj s c ifls rfls) = (oheap s)((freshlocst s) \<mapsto>\<^sub>f c)" by simp


lemma [simp] :"iheap (tickn i s) = iheap s"	     by simp
lemma [simp]: "iheap (ivarupdate s x v) = iheap s"   by simp
lemma [simp]: "iheap (rvarupdate s x v) = iheap s"   by simp
lemma [simp]: "iheap (newframe s mn  a v) = iheap s" by simp
lemma [simp]: "iheap (oldframe s s') = iheap s"      by simp
lemma [simp]: "iheap (popframe s) = iheap s"      by (simp add: popframe_def topframe_ist_def topframe_rst_def)
lemma [simp]: "iheap (incrcallcount s) = iheap s"    by simp
lemma [simp]: "iheap (obj_rfieldupdate s a f rtv) = iheap s" by simp
lemma [simp]: "iheap (newobj s c ifls rfls) = iheapflds ifls (freshlocst s) (iheap s) s" 
by simp

lemma [simp] :"rheap (tickn i s) = rheap s"	     by simp
lemma [simp]: "rheap (ivarupdate s x v) = rheap s"   by simp
lemma [simp]: "rheap (rvarupdate s x v) = rheap s"   by simp
lemma [simp]: "rheap (newframe s mn  a v) = rheap s" by simp
lemma [simp]: "rheap (oldframe s s') = rheap s"      by simp
lemma [simp]: "rheap (popframe s) = rheap s"      by (simp add: popframe_def topframe_ist_def topframe_rst_def)
lemma [simp]: "rheap (incrcallcount s) = rheap s"    by simp
lemma [simp]: "rheap (obj_ifieldupdate s a f rtv) = rheap s" by simp
lemma [simp]: "rheap (newobj s c ifls rfls) = rheapflds rfls (freshlocst s) (rheap s) s" 
by simp

lemma [simp]: "iheap (obj_ifieldupdate s a f rtv) = (iheap s) (f:= (iheap s f)(a:=rtv))"
by simp

lemma [simp]: "rheap (obj_rfieldupdate s a f rtv) = (rheap s) (f:= (rheap s f)(a:=rtv))"
by simp





subsubsection {* Projecting the framestack *}

lemma [simp]: "framestack (tickn n s) = framestack s"		by simp
lemma [simp]: "framestack (ivarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (rvarupdate s x v) = framestack s"    by simp
lemma [simp]: "framestack (obj_ifieldupdate s a f rtv) = framestack s" by simp
lemma [simp]: "framestack (obj_rfieldupdate s a f rtv) = framestack s" by simp
lemma [simp]: "framestack (incrcallcount s) = framestack s"     by simp

lemma [simp]: "framestack (newframe s m objadr arg) = 
			(m,(istore s, rstore s))#(framestack s)" by simp

lemma [simp]: "framestack (oldframe s' s) = framestack s"	 by simp
lemma [simp]: "framestack (popframe s) = tl (framestack s)"	 by (simp add: popframe_def topframe_ist_def topframe_rst_def)

lemma [simp]: "framestack (newobj s c ifls rfls) = framestack s" by simp

subsubsection {* Projecting the stackszs *}

lemma [simp]: "stackszs (tickn n s) = stackszs s"		   by simp
lemma [simp]: "stackszs (ivarupdate s v x) = stackszs s"	   by simp
lemma [simp]: "stackszs (rvarupdate s v x) = stackszs s"	   by simp
lemma [simp]: "stackszs (obj_ifieldupdate s a f rtv) = stackszs s" by simp
lemma [simp]: "stackszs (obj_rfieldupdate s a f rtv) = stackszs s" by simp
lemma [simp]: "stackszs (incrcallcount s) = stackszs s"		   by simp

lemma [simp]: "stackszs (newframe s m adr arg) = 
		((int (length (framestack s))) + 1)#(stackszs s)"    by simp

lemma [simp]: "stackszs (oldframe s' s) = stackszs s'"		   by simp
lemma [simp]: "stackszs (popframe s) = stackszs s"		   by (simp add: popframe_def topframe_ist_def topframe_rst_def)

lemma [simp]: "stackszs (newobj s c ifls rfls) = stackszs s" by simp

lemma [simp]: "maxstackfn (tickn n s) = maxstackfn s"		   by simp
lemma [simp]: "maxstackfn (ivarupdate s v x) = maxstackfn s"	   by simp
lemma [simp]: "maxstackfn (rvarupdate s v x) = maxstackfn s"	   by simp
lemma [simp]: "maxstackfn (obj_ifieldupdate s a f rtv) = maxstackfn s" by simp
lemma [simp]: "maxstackfn (obj_rfieldupdate s a f rtv) = maxstackfn s" by simp
lemma [simp]: "maxstackfn (incrcallcount s) = maxstackfn s"		   by simp

lemma [simp]: "maxstackfn (newframe s m adr arg) = 
		max ((int (length (framestack s))) + 1) (maxstackfn s)"    by simp

lemma [simp]: "maxstackfn (oldframe s' s) = maxstackfn s'"		   by simp
lemma [simp]: "maxstackfn (popframe s) = maxstackfn s"		   by (simp add: popframe_def topframe_ist_def topframe_rst_def)

lemma [simp]: "maxstackfn (newobj s c ifls rfls) = maxstackfn s" by simp



subsubsection {* Projecting the clock *}

lemma [simp]: "clock (ivarupdate s v x) = clock s"   	     by simp
lemma [simp]: "clock (rvarupdate s v x) = clock s"           by simp     
lemma [simp]: "clock (obj_ifieldupdate s a f rtv) = clock s" by simp
lemma [simp]: "clock (obj_rfieldupdate s a f rtv) = clock s" by simp
lemma [simp]: "clock (incrcallcount s) = clock s"    	     by simp
lemma [simp]: "clock (newframe s mn  a v) = clock s"	     by simp     
lemma [simp]: "clock (oldframe s s') = clock s"	             by simp     
lemma [simp]: "clock (popframe s) = clock s"	             by simp     

lemma [simp]: "clock (tickn i s) = i + (clock s)"	     by simp

lemma [simp]: "clock (newobj s c ifls rfls) = clock s" by simp

subsubsection {* Projecting the callcount *}

lemma [simp]: "callcount (tickn i s) = callcount s"		     by simp
lemma [simp]: "callcount (ivarupdate s v x) = callcount s"   	     by simp
lemma [simp]: "callcount (rvarupdate s v x) = callcount s"           by simp     
lemma [simp]: "callcount (obj_ifieldupdate s a f rtv) = callcount s" by simp
lemma [simp]: "callcount (obj_rfieldupdate s a f rtv) = callcount s" by simp
lemma [simp]: "callcount (incrcallcount s) = (callcount s) + 1"	     by simp
lemma [simp]: "callcount (newframe s mn  a v) = callcount s"	     by simp     
lemma [simp]: "callcount (oldframe s s') = callcount s"   	     by simp     
lemma [simp]: "callcount (popframe s) = callcount s"   	     by simp     
lemma [simp]: "callcount (newobj s c ifls rfls) = callcount s" by simp

subsubsection {* Projecting the invokecount *}

lemma [simp]: "invokecount (tickn i s) = invokecount s"		     by simp
lemma [simp]: "invokecount (ivarupdate s v x) = invokecount s"   	     by simp
lemma [simp]: "invokecount (rvarupdate s v x) = invokecount s"           by simp     
lemma [simp]: "invokecount (obj_ifieldupdate s a f rtv) = invokecount s" by simp
lemma [simp]: "invokecount (obj_rfieldupdate s a f rtv) = invokecount s" by simp
lemma [simp]: "invokecount (incrcallcount s) = invokecount s"	     by simp
lemma [simp]: "invokecount (newframe s mn  a v) = invokecount s + 1"	     by simp     
lemma [simp]: "invokecount (oldframe s s') = invokecount s"   	     by simp     
lemma [simp]: "invokecount (popframe s) = invokecount s"   	     by simp     
lemma [simp]: "invokecount (newobj s c ifls rfls) = invokecount s" by simp




subsection {* Injectivity of state functions *}

lemma tickn_inj: "inj (tickn i)"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

lemma incrcallcount_inj: "inj incrcallcount"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

lemma tickcall_inj: "inj tickcall"
apply (rule inj_onI)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply (simp add: incrcallcount_def tickn_def)
done

(* Could do similarly for stackszs, invokecount, etc. *)


subsection {* Monotonicity of resource parameters *}

text {* Here are the functions which are used in the operational
  semantics to change state. *}

lemma callcount_mono  : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> callcount s <= callcount t"
by (erule evalexpr.induct, simp_all)

lemma invokecount_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> invokecount s <= invokecount t"
by (erule evalexpr.induct, simp_all)

lemma stackszs_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> maxstackfn s <= maxstackfn t"
by (erule evalexpr.induct, simp_all)


(* clock_mono actually says that clock *increases* on each evaluation *)
lemma clock_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> clock s < clock t"
by (erule evalexpr.induct, simp_all)

lemma heap_mono : "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> (fmap_dom (oheap s) \<subseteq> fmap_dom (oheap t))"
by (erule evalexpr.induct, simp_all, auto)

text {* Now we remove the raw definitions of the state functions
  from the simplifier set. *}

declare state_functions [simp del]

subsection {* State/record simplifications *}

(*Lenb: Some lemmas which simplify states by contracting subsequent
        updates - the components are ordered in the order given in the
        definition of states*)

(* successive updates *)
lemma[simp]: "s\<lparr>callcount := A\<rparr>\<lparr>callcount := B\<rparr> = s\<lparr>callcount := B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>clock := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>istore := A\<rparr>\<lparr>istore := B\<rparr> = s\<lparr>istore :=B\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>rstore := A\<rparr>\<lparr>rstore := B\<rparr> = s\<lparr>rstore :=B\<rparr>"  by(simp)

(* commutativity *)
lemma[simp]: "s\<lparr>istore := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>\<lparr>istore :=A\<rparr>"  by(simp)
lemma[simp]: "s\<lparr>rstore := A\<rparr>\<lparr>clock := B\<rparr> = s\<lparr>clock := B\<rparr>\<lparr>rstore :=A\<rparr>"  by(simp)

subsection {* new popframe lemmas *}

(* big-step semantics: framestack just changes within an invoke step *)
lemma framestack_const: "\<forall> s s' v e.  
          \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<longrightarrow> framestack s' = framestack s"
apply (rule allI)+
apply (rule impI)
apply (erule evalexpr.induct)
apply (simp_all)
done

(*++
lemma istore_const: "\<forall> s s' cn mn vn v .  
          \<langle>s,InvokeStatic cn mn vn\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<longrightarrow> istore s' = istore s"
apply (rule allI)+
apply (rule impI)
apply (erule eval_cases)
apply clarsimp
apply (simp add: topframe_ist_def)
apply (insert framestack_const)
apply (erule_tac x="newframe s mn Nullref s\<lfloor>vn\<rfloor>" in allE)
apply (erule_tac x="s'a" in allE)
apply (erule_tac x="v" in allE)
apply (erule_tac x="methtable cn mn" in allE)
apply (clarsimp)
apply (drule mp)
apply auto
done

lemma rstore_const: "\<forall> s s' cn mn vn v .  
          \<langle>s,InvokeStatic cn mn vn\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<longrightarrow> rstore s' = rstore s"
apply (rule allI)+
apply (rule impI)
apply (erule eval_cases)
apply clarsimp
apply (simp add: topframe_rst_def)
apply (insert framestack_const)
apply (erule_tac x="newframe s mn Nullref s\<lfloor>vn\<rfloor>" in allE)
apply (erule_tac x="s'a" in allE)
apply (erule_tac x="v" in allE)
apply (erule_tac x="methtable cn mn" in allE)
apply (clarsimp)
apply (drule mp)
apply auto
done
++*)

(* should be used with invoke rule to link frame in post-state with rstore in pre-state*)
lemma topframe_rstore0: "\<forall> s s' v e mn r vn.  
          \<langle>newframe s mn r (s\<lfloor>vn\<rfloor>),e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> --> topframe_rst s' = rstore s"
apply (rule allI)+
apply (rule impI)
apply (simp_all add: topframe_rst_def framestack_const newframe_def)
done

(* first state has a frame specifying the second state on the top of its framestack *)
constdefs
  has_frame_of :: "mname \<Rightarrow> state \<Rightarrow> state \<Rightarrow> bool"
 "has_frame_of mn s' s == framestack s' = (mn,(istore s, rstore s))#(framestack s)"

(* declare has_frame_of_def[simp] *)

lemma  "\<And>s mn r vn. popframe (newframe s mn r (s\<lfloor>vn\<rfloor>)) = s\<lparr> invokecount := (invokecount s) + 1, stackszs := ((int (length (framestack s))) + 1)#(stackszs s) \<rparr>"
by (simp add: popframe_def newframe_def topframe_ist_def topframe_rst_def)

lemma  "\<And>s mn r vn. has_frame_of mn (newframe s mn r (s\<lfloor>vn\<rfloor>)) s"
by (simp add: popframe_def newframe_def topframe_ist_def topframe_rst_def has_frame_of_def topframe_rstore0)


lemma popframe_ist: "\<And> mn s s'. has_frame_of mn s' s \<longrightarrow>
                  istore (popframe s') = istore s"
by (simp add:  has_frame_of_def topframe_ist_def)

lemma popframe_rst: "\<And> mn s s'. has_frame_of mn s' s \<longrightarrow>
                  rstore (popframe s') = rstore s"
by (simp add:  has_frame_of_def topframe_rst_def)

lemma popframe1: "\<forall> s s' mn. has_frame_of mn s' s --> 
                             (popframe s') = (oldframe s' s)"
by (simp add:  has_frame_of_def popframe_def oldframe_def topframe_ist_def topframe_rst_def)

lemma topframe_rst: "\<forall> s s' mn. has_frame_of mn s' s --> 
                     topframe_rst s' = rstore s"
by  (simp add: topframe_rst_def has_frame_of_def)


lemma topframe_ist: "\<forall> s s' mn. has_frame_of mn s' s --> 
                     topframe_ist s' = istore s"
by  (simp add: topframe_ist_def has_frame_of_def)

lemma topframe_newframe_rst: "\<forall> s s' v e mn vn r. \<langle>newframe s mn r (s\<lfloor>vn\<rfloor>),e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<longrightarrow> topframe_rst s' = rstore s"
by  (simp add: framestack_const topframe_rst_def has_frame_of_def)

lemma popframe_oldframe: "\<forall> s s' mn vn r  v e.  
                     \<langle>(newframe s mn r (s\<lfloor>vn\<rfloor>)),InvokeStatic cn mn vn\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> -->
                     (popframe s') = (oldframe s' s)"
apply (rule allI)+
apply (rule impI)
apply (simp add: framestack_const popframe_def oldframe_def topframe_ist_def topframe_rst_def)
done

lemmas frame_lemmas = has_frame_of_def topframe_rst_def topframe_ist_def newframe_def popframe_def topframe_rstore0 framestack_const
lemmas frame_lemmas_no_defs = topframe_rstore0 framestack_const

(* like tickcall_add but for methods *)
lemma VngoqQIp_cha':
 "\<forall> s s' e v mn vn2 . \<langle>newframe s mn Nullref (s\<lfloor>vn2\<rfloor>), e\<rangle> \<longrightarrow>e \<langle>v, s'\<rangle> \<longrightarrow> topframe_rst s' = rstore s"
apply (rule allI)+
apply (rule impI)
apply (case_tac e)
apply (simp_all add: framestack_const popframe_def oldframe_def topframe_ist_def topframe_rst_def)
done

subsection {* Heap size and heap properties *}

text {* Some trivial heap properties *}

lemma [simp]: "finite (fmap_dom (heap s))"  by auto

lemma theloc_ref [simp]: "s\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> theloc s\<lfloor>x\<rfloor> = a"
by auto

lemma theival_int [simp]: "x = IVal i \<Longrightarrow> theival x = i"
by auto

lemma therval_ref [simp]: "x = RVal r \<Longrightarrow> therval x = r"
by auto

subsection {* Calculating the size of the heap *}

constdefs hpsize ::"state \<Rightarrow> int"
  "hpsize s == int (card (fmap_dom (oheap s)))"

declare hpsize_def [simp]

lemma hpsizeTick[simp]:"hpsize (tickn n s) = hpsize s" by simp
lemma hpsizeIvarUpd[simp]:"hpsize (s<x:=v>) = hpsize s" by simp
lemma hpsizeRvarUpd[simp]:"hpsize (s\<lfloor>x:=v\<rfloor>) = hpsize s" by simp
lemma hpsizeIfldUpd[simp]:"hpsize (s<a\<bullet>f:=v>) = hpsize s" by simp
lemma hpsizeRfkdUpd[simp]:"hpsize (s\<lfloor>a\<diamondsuit>f:=v\<rfloor>) = hpsize s" by simp
lemma hpsizeIncrCallCount[simp]:"hpsize (incrcallcount s) = hpsize s" by simp

lemma hpsizeInsert[simp]: "hpsize (newobj s c ifls rfls) = hpsize s + 1"
by (simp add: freshloc)

end
