(*  
   File:	ToyGrailLemmas.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyGrailLemmas.thy,v 1.19 2003/06/24 11:55:48 da Exp $

   Useful properties, simplification rules for ToyGrail.

   NB: Most of the simplification rules merely characterise the state
   updaters used in the Hoare rules.  It might be more efficient (and certainly
   simpler) not to use these abstract constants at all.
*)   


header  {* Toy Grail Lemmas *}

theory ToyGrailLemmas = ToyGrailDef:

subsection {* Elimination rules for the operational semantics *}

inductive_cases eval_cases: 
  "(s, expr.Null, v, s') \<in> evalexpr"
  "(s, expr.Int i, v, s') \<in> evalexpr"
  "(s, expr.IVar vn, v, s') \<in> evalexpr"
  "(s, expr.RVar vn, v, s') \<in> evalexpr"
  "(s, expr.Primop f vn1 vn2, v, s') \<in> evalexpr"
  "(s, expr.RPrimop f vn1 vn2, v, s') \<in> evalexpr"
  "(s, expr.GetFi vn f, v, s') \<in> evalexpr"
  "(s, expr.GetFr vn f, v, s') \<in> evalexpr"
  "(s, expr.PutFi vn1 f vn2, v, s') \<in> evalexpr"
  "(s, expr.PutFr vn1 f vn2, v, s') \<in> evalexpr"
  "(s, New c ifs rfs, v, s') \<in> evalexpr"
  "(s, Ifg vn l1 l2, v, s') \<in> evalexpr"
  "(s, Leti vn e ls, v, s') \<in> evalexpr"
  "(s, Letr vn e ls, v, s') \<in> evalexpr"
  "(s, Letv e ls, v, s') \<in> evalexpr"
  "(s, Call fn, v, s') \<in> evalexpr"
  "(s, expr.Invoke vn1 mn vn2, v, s') \<in> evalexpr"
  "(s, expr.InvokeStatic c mn vn, v, s') \<in> evalexpr"
  "(s, Pre P e, v, s') \<in> evalexpr"
  "(s, Post P e, v, s') \<in> evalexpr"
  "(s, Measure M e, v, s') \<in> evalexpr"



subsection {* Some rules for simplifying states *}

text {* To maintain the abstract constants for updating states
  rather than immediately expanding them, we need a bunch of  
  simplifications rules for simplifying state expressions.
*}
 
(* NB: The 8 state updaters are: 

   clock:  tickn 
   stores: ivarupdate rvarupdate
   heap:   obj_ifieldupdate obj_rfieldupdate
   frames: newframe incrcallcount 
*)


text {* First, collect together the definitions of semantic functions. *}

lemmas state_functions [simp] = 
   tickn_def ivarupdate_def rvarupdate_def 
   obj_ifieldupdate_def obj_rfieldupdate_def 
   newframe_def oldframe_def incrcallcount_def 
   newobj_def 


subsubsection {* Tick *} 

text {* We pull all ticks to the outside, and sum them. *}

lemma tick_simps [simp]: 
 "tickn i (tickn j s)          = tickn (i+j) s"                
 "ivarupdate (tickn i s) x v   = tickn i (ivarupdate s x v)"   
 "rvarupdate (tickn i s) y w   = tickn i (rvarupdate s y w)"   
 "obj_ifieldupdate (tickn i s) a f rtv = tickn i (obj_ifieldupdate s a f rtv)"
 "obj_rfieldupdate (tickn i s) a g rtvn = tickn i (obj_rfieldupdate s a g rtvn)"
 "newframe (tickn i s) m l r = tickn i (newframe s m l r)"   
 "incrcallcount (tickn i s) = tickn i (incrcallcount s)"
 "oldframe (tickn i s) s' = tickn i (oldframe s s')"
 "newobj (tickn i s) c ifls rfls  = tickn i (newobj s c ifls rfls)" 
by simp+



subsection {* Projecting components of the state *}

subsubsection {* Projecting istore, rstore *}

lemma istore_simps [simp]: 
  "istore (tickn k s) = istore s"
  "istore (rvarupdate s v val) = istore s"
  "istore (obj_ifieldupdate s a f itv) = istore s"
  "istore (obj_rfieldupdate s a g rtv) = istore s"
  "istore (incrcallcount s) = istore s"
  "istore (newframe s m l r) = emptyi"
  "istore (oldframe s s') = istore s'"
  "istore (newobj s c ifls rfls)  = istore s"
  "istore (ivarupdate s x vl) = (istore s)(x:=vl)"
by simp+

lemma rstore_simps [simp]: 
  "rstore (tickn k s) = rstore s"
  "rstore (ivarupdate s v val) = rstore s"
  "rstore (obj_ifieldupdate s a f itv) = rstore s"
  "rstore (obj_rfieldupdate s a g rtv) = rstore s"
  "rstore (incrcallcount s) = rstore s"
  "rstore (newframe s m l r) = emptyr(self:=l, param := r)"
  "rstore (oldframe s s') = rstore s'"
  "rstore (newobj s c ifls rfls)  = rstore s"
  "rstore (rvarupdate s x vl) = (rstore s)(x:=vl)"
by simp+


subsubsection {* Projecting (three components of) the heap *}

lemma oheap_simps [simp]:
  "oheap (tickn i s) = oheap s"	     
  "oheap (ivarupdate s x v) = oheap s"
  "oheap (rvarupdate s y w) = oheap s"
  "oheap (newframe s mn  a w) = oheap s"
  "oheap (oldframe s s') = oheap s"
  "oheap (incrcallcount s) = oheap s"
  "oheap (obj_ifieldupdate s l f itv) = oheap s"
  "oheap (obj_rfieldupdate s l g rtv) = oheap s"
  "oheap (newobj s c ifls rfls) = (oheap s)((freshlocst s) \<mapsto>\<^sub>f c)"
by simp+


lemma iheap_simps [simp]:
  "iheap (tickn i s) = iheap s"	     
  "iheap (ivarupdate s x v) = iheap s"
  "iheap (rvarupdate s y w) = iheap s"
  "iheap (newframe s mn  a w) = iheap s"
  "iheap (oldframe s s') = iheap s"
  "iheap (incrcallcount s) = iheap s"
  "iheap (obj_rfieldupdate s l g rtv) = iheap s"
  "iheap (obj_ifieldupdate s l f itv) = (iheap s) (f:= (iheap s f)(l:=itv))"
  "iheap (newobj s c ifls rfls) = iheapflds ifls (freshlocst s) (iheap s) (istore s)" 
by simp+

lemma rheap_simps [simp]:
  "rheap (tickn i s) = rheap s"	     
  "rheap (ivarupdate s x v) = rheap s"
  "rheap (rvarupdate s y w) = rheap s"
  "rheap (newframe s mn  a w) = rheap s"
  "rheap (oldframe s s') = rheap s"
  "rheap (incrcallcount s) = rheap s"
  "rheap (obj_ifieldupdate s l f itv) = (rheap s)"
  "rheap (obj_rfieldupdate s l g rtv) = (rheap s) (g:= (rheap s g)(l:=rtv))"
  "rheap (newobj s c ifls rfls) = rheapflds rfls (freshlocst s) (rheap s) (rstore s)" 
by simp+


subsubsection {* Projecting the framestack, maxstack, clock *}

lemma framestack_simps [simp]: 
  "framestack (tickn n s) = framestack s"
  "framestack (ivarupdate s x v) = framestack s"
  "framestack (rvarupdate s y w) = framestack s"
  "framestack (obj_ifieldupdate s a f rtv) = framestack s"
  "framestack (obj_rfieldupdate s a g itv) = framestack s"
  "framestack (incrcallcount s) = framestack s"
  "framestack (newframe s m objadr arg) = (m,(istore s, rstore s))#(framestack s)" 
  "framestack (oldframe s' s) = framestack s"
  "framestack (newobj s c ifls rfls) = framestack s"
by simp+

lemma maxstack_simps [simp]: 
  "maxstack (tickn n s) = maxstack s"
  "maxstack (ivarupdate s v x) = maxstack s"
  "maxstack (rvarupdate s w y) = maxstack s"
  "maxstack (obj_ifieldupdate s a f rtv) = maxstack s"
  "maxstack (obj_rfieldupdate s a g itv) = maxstack s"
  "maxstack (incrcallcount s) = maxstack s"
  "maxstack (newframe s m adr arg) = 
		max (int (length (framestack s)) + 1) (maxstack s)"
  "maxstack (oldframe s' s) = maxstack s'"
  "maxstack (newobj s c ifls rfls) = maxstack s"
by simp+

lemma clock_simps [simp]: 
  "clock (ivarupdate s v x) = clock s"
  "clock (rvarupdate s w y) = clock s"
  "clock (obj_ifieldupdate s a f rtv) = clock s"
  "clock (obj_rfieldupdate s a g itv) = clock s"
  "clock (incrcallcount s) = clock s"
  "clock (newframe s mn l arg) = clock s"
  "clock (oldframe s s') = clock s"
  "clock (tickn i s) = i + (clock s)"
  "clock (newobj s c ifls rfls) = clock s"
by simp+

subsubsection {* Projecting the callcount, invokecount *}

lemma callcount_simps [simp]:
  "callcount (tickn i s) = callcount s"
  "callcount (ivarupdate s v x) = callcount s"
  "callcount (rvarupdate s w y) = callcount s"
  "callcount (obj_ifieldupdate s a f rtv) = callcount s"
  "callcount (obj_rfieldupdate s a g itv) = callcount s"
  "callcount (incrcallcount s) = (callcount s) + 1"
  "callcount (newframe s mn l vn) = callcount s"
  "callcount (oldframe s s') = callcount s"
  "callcount (newobj s c ifls rfls) = callcount s"
by simp+

subsubsection {* Projecting the invokecount *}

lemma invokecount_simps [simp]: 
  "invokecount (tickn i s) = invokecount s"
  "invokecount (ivarupdate s v x) = invokecount s"
  "invokecount (rvarupdate s w y) = invokecount s"
  "invokecount (obj_ifieldupdate s a f rtv) = invokecount s"
  "invokecount (obj_rfieldupdate s a g itv) = invokecount s"
  "invokecount (incrcallcount s) = invokecount s"
  "invokecount (newframe s mn l vn) = invokecount s + 1"
  "invokecount (oldframe s s') = invokecount s"
  "invokecount (newobj s c ifls rfls) = invokecount s"
by simp+


subsubsection {* State-as-a-record simplifications *}

text {* Some lemmas which simplify states by contracting/reordering
  sequential updates.  The updates are ordered in the order given in the
  definition of the state record. *}

(* NB: A heap record has 10 components. *)

lemma state_record_simps [simp]:
(* successive updates *)
  "s\<lparr>iheap := ih1\<rparr>\<lparr>iheap := ih2\<rparr> = s\<lparr>iheap := ih2\<rparr>"
  "s\<lparr>rheap := rh1\<rparr>\<lparr>rheap := rh2\<rparr> = s\<lparr>rheap := rh2\<rparr>"
  "s\<lparr>oheap := oh1\<rparr>\<lparr>oheap := oh2\<rparr> = s\<lparr>oheap := oh2\<rparr>"
  "s\<lparr>istore := i1\<rparr>\<lparr>istore := i2\<rparr> = s\<lparr>istore :=i2\<rparr>"
  "s\<lparr>rstore := r1\<rparr>\<lparr>rstore := r2\<rparr> = s\<lparr>rstore :=r2\<rparr>"
  "s\<lparr>framestack := fs1\<rparr>\<lparr>framestack :=fs2\<rparr> = s\<lparr>framestack := fs2\<rparr>"
  "s\<lparr>maxstack := ms1\<rparr>\<lparr> maxstack := ms2\<rparr> = s\<lparr>maxstack := ms2\<rparr>"
  "s\<lparr>callcount := cc1\<rparr>\<lparr>callcount := cc2\<rparr> = s\<lparr>callcount := cc2\<rparr>"
  "s\<lparr>invokecount := ic1\<rparr>\<lparr>invokecount := ic2\<rparr> = s\<lparr>invokecount := ic2\<rparr>"
  "s\<lparr>clock := c1\<rparr>\<lparr>clock := c2\<rparr> = s\<lparr>clock := c2\<rparr>"
(* commutativity: rearrangement to standard order [cases should do this?] *)
(* for clock: *)
  "s\<lparr>clock := c\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>rstore :=r\<rparr> = s\<lparr>rstore := r\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>framestack :=fs\<rparr> = s\<lparr>framestack := fs\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>maxstack :=ms\<rparr> = s\<lparr>maxstack := ms\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>callcount := cc\<rparr> = s\<lparr>callcount :=cc\<rparr>\<lparr>clock := c\<rparr>"
  "s\<lparr>clock := c\<rparr>\<lparr>invokecount := ic\<rparr> = s\<lparr>invokecount :=ic\<rparr>\<lparr>clock := c\<rparr>"
(* for invokecount: *)
  "s\<lparr>invokecount := ic\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>rstore :=r\<rparr> = s\<lparr>rstore := r\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>framestack :=fs\<rparr> = s\<lparr>framestack := fs\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount := ic\<rparr>\<lparr>maxstack :=ms\<rparr> = s\<lparr>maxstack := ms\<rparr>\<lparr>invokecount := ic\<rparr>"
  "s\<lparr>invokecount :=ic\<rparr>\<lparr>callcount := cc\<rparr> = s\<lparr>callcount := cc\<rparr>\<lparr>invokecount := ic\<rparr>"
(* for callcount: *)
  "s\<lparr>callcount := cc\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>rstore :=r\<rparr> = s\<lparr>rstore := r\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>framestack :=fs\<rparr> = s\<lparr>framestack := fs\<rparr>\<lparr>callcount := cc\<rparr>"
  "s\<lparr>callcount := cc\<rparr>\<lparr>maxstack :=ms\<rparr> = s\<lparr>maxstack := ms\<rparr>\<lparr>callcount := cc\<rparr>"
(* for maxstack: *)
  "s\<lparr>maxstack := ms\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>maxstack := ms\<rparr>"
  "s\<lparr>maxstack := ms\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>maxstack := ms\<rparr>"
  "s\<lparr>maxstack := ms\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>maxstack := ms\<rparr>"
  "s\<lparr>maxstack := ms\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>maxstack := ms\<rparr>"
  "s\<lparr>maxstack := ms\<rparr>\<lparr>rstore :=r\<rparr> = s\<lparr>rstore := r\<rparr>\<lparr>maxstack := ms\<rparr>"
  "s\<lparr>maxstack := ms\<rparr>\<lparr>framestack :=fs\<rparr> = s\<lparr>framestack := fs\<rparr>\<lparr>maxstack := ms\<rparr>"
(* for framestack: *)
  "s\<lparr>framestack := fs\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>framestack := fs\<rparr>"
  "s\<lparr>framestack := fs\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>framestack := fs\<rparr>"
  "s\<lparr>framestack := fs\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>framestack := fs\<rparr>"
  "s\<lparr>framestack := fs\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>framestack := fs\<rparr>"
  "s\<lparr>framestack := fs\<rparr>\<lparr>rstore :=r\<rparr> = s\<lparr>rstore := r\<rparr>\<lparr>framestack := fs\<rparr>"
(* for rstore *)
  "s\<lparr>rstore := r\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>rstore := r\<rparr>"
  "s\<lparr>rstore := r\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>rstore := r\<rparr>"
  "s\<lparr>rstore := r\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>rstore := r\<rparr>"
  "s\<lparr>rstore := r\<rparr>\<lparr>istore :=i\<rparr> = s\<lparr>istore := i\<rparr>\<lparr>rstore := r\<rparr>"
(* for istore *)
  "s\<lparr>istore := i\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>istore := i\<rparr>"
  "s\<lparr>istore := i\<rparr>\<lparr>rheap :=rh\<rparr> = s\<lparr>rheap := rh\<rparr>\<lparr>istore := i\<rparr>"
  "s\<lparr>istore := i\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>istore := i\<rparr>"
(* for rheap *)
  "s\<lparr>rheap := rh\<rparr>\<lparr>iheap :=ih\<rparr> = s\<lparr>iheap := ih\<rparr>\<lparr>rheap := rh\<rparr>"
  "s\<lparr>rheap := rh\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>rheap := rh\<rparr>"
(* for iheap *)
  "s\<lparr>iheap := ih\<rparr>\<lparr>oheap :=oh\<rparr> = s\<lparr>oheap := oh\<rparr>\<lparr>iheap := ih\<rparr>"
by simp+




subsection {* Injectivity of state functions *}

lemma tickn_inj: "inj (tickn i)"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

lemma incrcallcount_inj: "inj incrcallcount"
apply (rule inj_onI, simp)
apply (rule_tac r=x in state.cases)
apply (rule_tac r=y in state.cases)
apply auto
done

(* Could do similarly for maxstack, invokecount, etc. *)


subsection {* Monotonicity of resource parameters *}

text {* Here are the functions which are used in the operational
  semantics to change state. *}

lemma resource_monos: 
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> callcount s <= callcount t"
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> invokecount s <= invokecount t"
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> maxstack s <= maxstack t"
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> clock s < clock t"  (* clock strictly increasing *)
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> (fmap_dom (oheap s) \<subseteq> fmap_dom (oheap t))"
by ((erule evalexpr.induct, simp_all)+, auto)


subsection {* Cleanup *}

text {* Now we remove the raw definitions of the state functions
  from the simplifier set. *}

declare state_functions [simp del]



subsection {* Parametric behaviour of op sems wrt resource parameters *}

lemma evalintro_cong: "\<lbrakk> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>; s=s'; v=v'; t=t' \<rbrakk> \<Longrightarrow> \<langle>s',e\<rangle> \<longrightarrow>e \<langle>v',t'\<rangle>"
by auto

lemma evalintro_cong2: "\<lbrakk> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>; s=s'; v=v' \<rbrakk> \<Longrightarrow> \<langle>s',e\<rangle> \<longrightarrow>e \<langle>v',t\<rangle>"
by auto



lemma plusone [simp]: "x + 1 + (i::int) = x + i + 1" by auto


(* FIXME: this theorem could be more abstract: prove for any function which
   commutes with other state transformers. *)

lemma clock_tickpos [rule_format]: 
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> (\<forall> s'. s= tickn i s'\<longrightarrow> (\<exists> t'. \<langle>s',e\<rangle> \<longrightarrow>e \<langle>v,t'\<rangle> \<and> t=tickn i t'))"
apply (erule evalexpr.induct)
prefer 12  (* If_True *)
apply (rule, drule_tac x="tick s'" in spec)
apply (rule, simp, erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalIf_True, simp+)
prefer 12  (* If_False *)
apply (rule, drule_tac x="tick s'" in spec)
apply (rule, simp, erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalIf_False, simp+)
prefer 12 (* Leti *)
apply (rule, drule_tac x="s'" in spec)
apply (rule, simp)
apply (erule exE, simp)
apply (drule_tac x="tick (t'<vn:=ia>)" in spec)
apply simp
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Letr *)
apply (rule, drule_tac x="s'" in spec, rule, simp)
apply (erule exE, drule_tac x="tick (t'\<lfloor>vn:=r\<rfloor>)" in spec, simp)
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Letv *)
apply (rule, drule_tac x="s'" in spec, rule, simp)
apply (erule exE, drule_tac x="t'" in spec, simp)
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Call *)
apply (rule, drule_tac x="tickcall s'" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Invoke *)
apply (rule, drule_tac x="(newframe s'a mn (Ref a) s'a\<lfloor>vn2\<rfloor>)" in spec, rule, simp)
apply (erule exE, rule_tac x="tickn 5 (oldframe t' s'a)" in exI, rule, simp)
apply (rule evalintro_cong, rule evalexpr.intros, fastsimp+)
prefer 12 (* InvokeStatic *)
apply (rule, drule_tac x="(newframe s'a mn Nullref s'a\<lfloor>vn2\<rfloor>)" in spec, rule, simp)
apply (erule exE, rule_tac x="tickn 4 (oldframe t' s'a)" in exI, rule, simp)
apply (rule evalintro_cong, rule evalexpr.intros, fastsimp+)
prefer 12 (* Pre *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Post *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Measure *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
apply (rule, rule, rule, rule, rule evalintro_cong, rule evalexpr.intros, simp+)+
done


lemma callcount_incpos [rule_format]: 
  "\<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle> \<Longrightarrow> 
	(\<forall> s'. s= incrcallcount s'\<longrightarrow> (\<exists> t'. \<langle>s',e\<rangle> \<longrightarrow>e \<langle>v,t'\<rangle> \<and> t=incrcallcount t'))"
apply (erule evalexpr.induct)
prefer 12  (* If_True *)
apply (rule, drule_tac x="tick s'" in spec)
apply (rule, simp, erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalIf_True, simp+)
prefer 12  (* If_False *)
apply (rule, drule_tac x="tick s'" in spec)
apply (rule, simp, erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalIf_False, simp+)
prefer 12 (* Leti *)
apply (rule, drule_tac x="s'" in spec)
apply (rule, simp)
apply (erule exE, simp)
apply (drule_tac x="tick (t'<vn:=i>)" in spec)
apply simp
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Letr *)
apply (rule, drule_tac x="s'" in spec, rule, simp)
apply (erule exE, drule_tac x="tick (t'\<lfloor>vn:=r\<rfloor>)" in spec, simp)
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Letv *)
apply (rule, drule_tac x="s'" in spec, rule, simp)
apply (erule exE, drule_tac x="t'" in spec, simp)
apply (erule exE, rule_tac x=t'a in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Call *)
apply (rule, drule_tac x="tickcall s'" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Invoke *)
apply (rule, drule_tac x="(newframe s'a mn (Ref a) s'a\<lfloor>vn2\<rfloor>)" in spec, rule, simp)
apply (erule exE, rule_tac x="tickn 5 (oldframe t' s'a)" in exI, rule, simp)
apply (rule evalintro_cong, rule evalexpr.intros, fastsimp+)
prefer 12 (* InvokeStatic *)
apply (rule, drule_tac x="(newframe s'a mn Nullref s'a\<lfloor>vn2\<rfloor>)" in spec, rule, simp)
apply (erule exE, rule_tac x="tickn 4 (oldframe t' s'a)" in exI, rule, simp)
apply (rule evalintro_cong, rule evalexpr.intros, fastsimp+)
prefer 12 (* Pre *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Post *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
prefer 12 (* Measure *)
apply (rule, drule_tac x="s'a" in spec, rule, simp)
apply (erule exE, rule_tac x=t' in exI)
apply (rule, simp, rule evalexpr.intros, fastsimp+)
apply (rule, rule, rule, rule, rule evalintro_cong, rule evalexpr.intros, simp+)+
done

lemma tickcall_pos: 
   "\<lbrakk> \<langle>s,e\<rangle> \<longrightarrow>e \<langle>v,t\<rangle>; s=tickcall s' \<rbrakk> \<Longrightarrow> \<exists> t'. \<langle>s',e\<rangle> \<longrightarrow>e \<langle>v,t'\<rangle> \<and> t=tickcall t'"
apply (drule clock_tickpos, simp)
apply (erule exE, clarify)
apply (drule callcount_incpos)
apply auto
done



subsection {* Trivial simplifications *}

lemma [simp]: "finite (fmap_dom (heap s))"  by auto

lemma thething_simps [simp]: 
  "s\<lfloor>z\<rfloor> = Ref a \<Longrightarrow> theloc s\<lfloor>z\<rfloor> = a"
  "x = IVal i \<Longrightarrow> theival x = i"
  "y = RVal r \<Longrightarrow> therval y = r"  
  by auto+

subsection {* Heap size simplifications *}

lemma hpsize_simps [simp]:
  "hpsize (tickn n s) = hpsize s"
  "hpsize (s<x:=v>) = hpsize s"
  "hpsize (s\<lfloor>y:=w\<rfloor>) = hpsize s"
  "hpsize (s<a\<bullet>f:=v>) = hpsize s"
  "hpsize (s\<lfloor>a\<diamondsuit>g:=w\<rfloor>) = hpsize s"
  "hpsize (incrcallcount s) = hpsize s"
  "hpsize (newobj s c ifls rfls) = hpsize s + 1"
by ((simp add: hpsize_def)+, simp add: freshloc)


end
