(*  
   File:	ToyFunGrailDef.thy
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: ToyFunGrailDef.thy,v 1.1 2003/06/08 13:13:48 da Exp $
 
   Functional version of semantics separating environments from states.
*)   

header {* The definition of Toy Grail *}

theory ToyGrailDef = Finmap + Main:

text {* 
 Toy Grail is an attempt to reduce Grail to an absolute minimum to
 study and design the bytecode logic of resources. 

  Simplifications include:
  \begin{itemize}
   \item only two types: integer and reference, distinguished in syntax
   \item much type information is lost
   \item no static fields 
   \item methods have only a single (reference) parameter
   \item there is a single syntactic category for terms (function bodies)
   \item we use total maps to model local variables and objects
   %%\item functions have a global namespace
   \end{itemize}

  We define a syntax for programs which includes invariants.
  Because of this, we begin the definition of Toy Grail
  with its machine model.

*}





(****************************************************************************
 *
 * Section 1: MACHINE MODEL
 *
 ****************************************************************************)

 
section {* Machine model *}

subsection {* Locations and references *}

text {*
  We model locations as natural numbers.  This is somewhat concrete,
  but avoids constructing an abstract type or
  assuming the properties we expect. *}

types
  locn = nat

datatype ref = Nullref | Ref locn  

text {* We define a partial projection on non-null locations. *}

consts theloc :: "ref \<Rightarrow> locn"
primrec
  "theloc (Ref a) = a"

text {* 
  Given a finite set of locations, we can always find a new fresh one.
  *}

constdefs
  freshloc  :: "locn set \<Rightarrow> locn"         
 "freshloc L == Suc (Max (insert 0 L))" 

lemma freshloc: "finite L \<Longrightarrow> freshloc L \<notin> L"
  apply (simp add: freshloc_def)
  apply (subgoal_tac "finite (insert 0 L)")
  apply (erule thin_rl)
  apply (drule Max)
  apply (auto)
  done

subsection {* Types and Names *}

text {* We declare abstract types for names, including some 
  reserved names. *}

typedecl
  iname		 --  {* names of integer variables *}
typedecl
  rname		 --  {* names of reference variables *}

consts
  self     :: rname -- {* the self object *}
  param    :: rname -- {* the method parameter *}
  dummyref :: rname -- {* dummy reference used for void assignments *}

axioms
  constdistinct: "distinct [self,param,dummyref]"
typedecl
  funame	-- {* names of functions *}
typedecl 
  cname		-- {* names of classes *}
typedecl
  ifldname	-- {* names of integer fields *}
typedecl
  rfldname	-- {* names of reference fields *}
typedecl
  mname		-- {* names of methods *}



subsection {* Heaps *}

text {* 
  The heap is a map from locations to objects.  
  Conceptually, objects consist of a class name together with a
  mapping of field names to values.  We use Burstall's
  idea~\cite{NipkowXX} of treating
  records (i.e. objects) on the heap as separate
  ``mini''-heaps themselves, so that a heap is split into
  three components: the object heap, the integer heap,
  and the reference heap.  The latter two heaps are
  modelled as total functions, which simplifies
  the formalization.  Of course, we must check
  that the semantics only examines
  the contents of heaps at non-empty locations.
  *}

types 
  oheap    = "locn \<leadsto>\<^sub>f cname"            -- {* object heap; a finite map *}
  iheap    = "ifldname \<Rightarrow> locn \<Rightarrow> int"  -- {* integer heap for each field *}
  rheap    = "rfldname \<Rightarrow> locn \<Rightarrow> ref"  -- {* reference heap for each field *}


text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "oheap"   <= (type)  "nat \<leadsto>\<^sub>f cname"
  "iheap"   <= (type)  "ifldname \<Rightarrow> nat \<Rightarrow> int"
  "rheap"   <= (type)  "rfldname \<Rightarrow> nat \<Rightarrow> ref"



subsection {* Stores *}

text {*
  Like heaps, stores are modelled as total functions for simplicity.
  If the source program is well-typed, it should never attempt to
  access an undefined part of the store. 

  A frame is a method name (which can
  be used for profiling), together with a store.
  For a small-step semantics we would also need 
  a return point and a place to store the return value.
  We do not need to include the class of the currently
  executing method because this can be retrieved from
  the @{text self} variable.  When executing a static
  method, @{text self} is set to null.
 *}

types
  istore   = "iname \<Rightarrow> int"	        -- {* an integer   store *}
  rstore   = "rname \<Rightarrow> ref"          -- {* a reference store *}


text {*
  The next translations are included to fold up type definitions in printing.
  *}

translations
  "istore"  <= (type)  "iname \<Rightarrow> int"
  "rstore"  <= (type)  "rname \<Rightarrow> ref"


subsection {* States *}

text {* The state contains several resource-aware components:
  \begin{itemize}
  \item maxstack
  \item callcount
  \item invokecount
  \item clock
  \end{itemize}
  These are all represented as integers for ease of reasoning
  in combination with integer program variables, because Isabelle
  doesn't handle the subtyping Nat<Int very well.
*}

record state =
	oheap       :: oheap		 -- {* object heap *}
	iheap	    :: iheap		 -- {* integer heap *}
	rheap	    :: rheap		 -- {* references heap *}
	stackdepth  :: int		 -- {* current depth of stack        *}  
        maxstack    :: int		 -- {* maximum depth of stack        *}  
        callcount   :: nat		 -- {* count of CALL depth	     *}  
        invokecount :: nat		 -- {* count of INVOKE depth	     *}  
	clock	    :: int		 -- {* instruction counter	     *}

record env = 
	istore      :: istore	         -- {* integer store on top of stack *}
	rstore      :: rstore	         -- {* ref store on top of stack     *}
        curmeth     :: mname             -- {* currently executing method *}

translations
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       stackdepth::int, maxstack::int, 
		       callcount::nat, invokecount::nat,
		       clock::int\<rparr>"
  "state" <= (type) "\<lparr> oheap::oheap, iheap::iheap, rheap::rheap,
		       stackdepth::int, maxstack::int, 
		       callcount::nat, invokecount::nat,
		       clock::int,\<dots>::'a \<rparr>"

  "env"   <= (type) "\<lparr> istore::istore, rstore::rstore, 
		       curmeth :: mname \<rparr>"

  "env"   <= (type) "\<lparr> istore::istore, rstore::rstore, 
		       curmeth :: mname,\<dots>::'a \<rparr>"




subsection  {* Values and assertions *}

text {* Values and assertions are introduced here because
  the syntax of programs can include them.  *}

text {* Values may be either integer values or reference values. *}

datatype val = IVal int | RVal ref
consts
  theival :: "val \<Rightarrow> int"
  therval :: "val \<Rightarrow> ref"
primrec
  "theival  (IVal i) = i"
primrec
  "therval  (RVal r) = r"


text {* Assertions are predicates on pre-states, result values,
  and post-states. Toy Grail exclusively treats side-effecting
  expressions.
*}

types
 'a preassn     = "('a \<times> state) set"
 'a postassn = "('a \<times> state \<times> val) set"

text {* Sets are slightly easier to deal with in Isabelle than
  predicates. *}

translations
  "a preassn"   <= (type)  "(a \<times> state) set"
  "a postassn"  <= (type)  "(a \<times> state \<times> val) set"




(****************************************************************************
 *
 * Section 2: SYNTAX
 *
 ****************************************************************************)


section {* Syntax of Toy Grail *}


text {* Simple expressions are the basic unit of execution,
  and denote values.  They are built from operators, constants,
  and previously computed values (names).  An expression may have a
  side effect, but there are no directly nested expressions.
  We model untyped expressions. 

  Compound expressions are built using let expressions.  
  A function body is a let expression.  A function ends with
  tail calls to other functions or method returns.  
  The natural thing to do is to have two syntactic categories
  of terms: one for simple expressions and one for compound
  expressions.  However this makes the whole development 
  more tedious; so instead we define a predicate
  @{text SimpleExpr} which restricts the operational
  semantics in the single case that matters (the
  binding in a let expressions).

  Expressions are augmented with logical annotations, for
  pre-conditions, post-conditions, and termination
  measures.  They are polymorphic in the type of auxiliary
  variables.
*}

datatype 'a expr =
   Int	    int			
 | IVar     iname		
 | Primop   "int \<Rightarrow> int \<Rightarrow> int" iname iname
 | Null	          		          ("NULL")
 | RVar     rname	
 | RPrimop   "ref \<Rightarrow> ref \<Rightarrow> int" rname rname
 | New      "cname" "(ifldname \<times> iname) list" "(rfldname \<times> rname) list" ("NEW <_> '(_,_')")
 | GetFi    rname ifldname	  	  ("_\<bullet>_" [65,1000] 65)
 | GetFr    rname rfldname	  	  ("_\<diamondsuit>_" [65,1000] 65)
 | PutFi    rname ifldname iname	  ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | PutFr    rname rfldname rname	  ("(2_\<diamondsuit>_ :=/ _)" [70,1000,65] 61)
 | Invoke   rname mname rname             ("_\<diamondsuit>_'(_')" [70,1000,65] 61) 
 | InvokeStatic cname mname rname         ("_\<bullet>_'(_')" [70,1000,65] 61) 
 | Leti     iname "'a expr" "'a expr"        
 | Letr     rname "'a expr" "'a expr"
 | Ifg      iname "'a expr" "'a expr"	  ("(1IF _/ THEN _/ ELSE _)"  [0,0,0] 61)
 | Call     funame			  ("(CALL _)" [0] 60)
 | Pre      "'a preassn"  "'a expr"	  ("(PRE _:/ _)"  [0,60] 67)   (* REQUIRES *)
 | Post     "'a postassn" "'a expr" 	  ("(POST _/: _)" [0,60] 66)   (* ENSURES *)
 | Measure  "(state * state) set" "'a expr" ("(MEASURE _: _)" [0,0] 60) (* VARIANT *)



text {* Notes:
  \begin{enumerate}
  \item Primitive operations (\texttt{RPrimop}) on references might
  include a test for null, and (perhaps) a pointer equality test.

  \item In method invocations, the first argument is either a
  variable pointing to an object (instance methods), or
  a class name (static/class methods).  The second argument
  is the method name, and the third argument is the
  actual parameter, a single reference variable.

  \item If we treated imperative Grail, we might instead use the traditional
  imperative language of commands together with non-side effecting
  expressions.  That would be considerably simpler;
  for example, types of pre-assertions and post-assertions become
  the same, and the syntax needs only one kind of assertion.
  Lets are broken into assignment and composition.
  But that treatment de-emphasises the functional view.
  \end{enumerate}
*}
  


text {* The following black magic is Isabelle gobbledygook to
  introduce some nice concrete syntax. *}

(* let syntax *)
nonterminals
 gletbind gletbinds

syntax
  "_Gbind"      :: "[id, 'a] => gletbind"                ("(2_ =/ _)" 10)
  "_Grefbind"   :: "[id, 'a] => gletbind"                ("(2rf _ =/ _)" 10)
  "_Gvoidbind"  :: "'a => gletbind"                      ("(2 '_ =/ _)" 10)
  ""            :: "gletbind => gletbinds"               ("_")
  "_gbinds"     :: "[gletbind, gletbinds] => gletbinds"  ("_;/ _")
  "_GLets"      :: "[gletbinds, 'a expr] => 'a expr"     ("(LET (_)/ IN (_) END)" 60)

translations
  "_GLets (_gbinds b bs) e"  == "_GLets b (_GLets bs e)"
  "LET  v = e IN l END"      == "Leti v e l"
  "LET _ = e IN l END"       == "Letr dummyref e l"
  "LET rf v = e IN l END"    == "Letr v e l"


subsection {* Functions and methods *}

text {* A Grail function is notionally parameterised on
  a subset of the current variables in scope.  Variables
  not in the formal parameters should not be accessible
  in the body.  However, a new stack frame is not 
  allocated.  We model this simply by ignoring
  the formal parameters, so a Grail function is just
  its body, which is an @{text expr}.  

  A Grail method body is a collection of function bindings,
  together with a main body and a set of local variables used
  within.

  To simplify the representation for ToyGrail, we consider a
  global mapping from function names to function bodies (see later
  below, in class table).
  This means that we do not need to keep a record of the current
  program position (current method or current functions) in the state.

  Methods are parameterised on two things: first the
  self object (self) invoking the method, and then the single formal
  parameter.  Both parameter names are fixed, however, so
  are left implicit.

  When a method is invoked, we might want to know the size of the frame
  to calculate the size of the stack, which is why we record the 
  total set of local variables.
  (Although we do not initialize these).
 *}

types
  'inv methbody = "(iname set \<times> rname set) \<times> 'inv expr"

translations
  "inv methbody" <= (type) "(iname set \<times> rname set) \<times> inv expr"





subsection {* Programs *}

text {* The class table represents the program as a global parameter. 
   Rather than a single table, we split it into four tables,
  representing: functions (global), fields, and methods. *}


consts
  funtable      :: "funame \<Rightarrow> 'a expr"	         -- {* global function names *}
  ifieldtable   :: "cname \<Rightarrow> ifldname list"      -- {* int fields by class *}
  rfieldtable   :: "cname \<Rightarrow> rfldname list"      -- {* ref fields by class *}
  methtable     :: "cname \<Rightarrow> mname \<Rightarrow> 'a expr"   -- {* methods by class *}






(****************************************************************************
 *
 * Section 3: NATURAL SEMANTICS
 *
 ****************************************************************************)

section {* Big-step semantics *}

text {*
  We define a big-step semantics for Toy Grail.  We choose big-step mainly
  for similarity with the functional viewpoint.  With some minor additions
  to states, we could also define a small-step semantics.
  *}

subsection {* Functions for accessing the state *}

text {* We define some accessor functions for accessing components
  of the state.  These are treated as syntactic abbreviations,
  rather than functions. *}

(* NB: only priority of 1000 is high enough to override curried application in 
  writing "f s<x>".  We might try unifying these projections using axiomatic
  type class hack -- see HOL/Bali --- but that's pretty ugly. 
  Another improvement might be to restrict to using identifiers in the
  syntax: certainly for field names and variable names. *)

syntax
 "_get_ivar"     :: "state \<Rightarrow> iname \<Rightarrow> int"		("_<_>" [1000,1000] 1000)
translations
 "_get_ivar s x" == "istore s x"

syntax
 "_get_rvar"     :: "state \<Rightarrow> rname \<Rightarrow> ref"		("_\<lfloor>_\<rfloor>" [1000,1000] 1000)
translations
 "_get_rvar s x" == "rstore s x"

(* da: adjusted concrete syntax of next one because \<guillemotleft> much easier to type! *)
syntax  
 "_get_obj"  ::      "[state, locn] \<Rightarrow> cname option"	("_\<guillemotleft>_\<guillemotright>" [1000,1000] 1000)
translations
 "_get_obj s l" == "fmap_lookup (oheap s) l"

syntax
 "_get_ifld" ::	  "[state, locn, ifldname] \<Rightarrow> int"      ("_<_\<bullet>_>" [1000,1000,1000] 1000)
translations
 "_get_ifld s loc inm" == "iheap s inm loc"

syntax
 "_get_rfld" ::	  "[state, locn, rfldname] \<Rightarrow> ref"      ("_\<lfloor>_\<diamondsuit>_\<rfloor>" [1000,1000,1000] 1000)
translations
 "_get_rfld s loc rnm" == "rheap s rnm loc"

syntax
 "freshlocst"  :: "state \<Rightarrow> locn"
translations
 "freshlocst s" == "freshloc (fmap_dom (oheap s))"


subsection {* Functions for updating the state *}

text {* The state updating functions are treated as abstract operations
  on states.  This makes propositions and intermediate assertions
  more readable, at the expense of needing to add numerous
  simplification rules to deal with them. *}


constdefs
 tickn	     :: "int \<Rightarrow> state \<Rightarrow> state"
 "tickn n s \<equiv>  s \<lparr> clock := (clock s) + n \<rparr>"
syntax
 tick	     :: "state \<Rightarrow> state" -- {* a handy abbreviation *}
translations
 "tick s"    == "tickn 1 s"

constdefs
  ivarupdate :: "env \<Rightarrow> iname \<Rightarrow> int \<Rightarrow> env"	("_<_:=_>" [1000,1000,0] 1000)
  "ivarupdate s v val \<equiv> (s \<lparr> istore := (istore s)(v := val) \<rparr>)"

constdefs
  rvarupdate :: "env \<Rightarrow> rname \<Rightarrow> ref \<Rightarrow> env"        ("_\<lfloor>_:=_\<rfloor>" [1000,1000,0] 1000)
  "rvarupdate s v val \<equiv> (s \<lparr> rstore := (rstore s)(v := val) \<rparr>)"


constdefs
  obj_ifieldupdate :: "state \<Rightarrow> locn \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> state"   ("_<_\<bullet>_:=_>" [1000,80,1000,80] 1000)
  "obj_ifieldupdate s a f rtv \<equiv> s \<lparr> iheap := (iheap s) (f:= (iheap s f)(a:=rtv)) \<rparr>"

  obj_rfieldupdate :: "state \<Rightarrow> locn \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> state"   ("_\<lfloor>_\<diamondsuit>_:=_\<rfloor>" [1000,80,1000,80] 1000)
  "obj_rfieldupdate s a f rtv \<equiv> s \<lparr> rheap := (rheap s) (f:= (rheap s f)(a:=rtv)) \<rparr>"



constdefs
  incrcallcount :: "state \<Rightarrow> state"
  "incrcallcount s \<equiv> s \<lparr> callcount := (callcount s)+1 \<rparr>"


text {* 
  Object initialization simply sets the class of an object.
  We might also update the fields with empty maps, defined
  below.  But it should be a guaranteed by type-checking that
  no uninitialised field is accessed.
 *}

constdefs
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"


text {* A method invocation allocates a new frame on the frame stack. 
        This function adjusts the state accordingly, given a reference
        to the invoking object, and the parameter.
        The new store contains only bindings for the self object
	and the method parameter.  (We might also give initial
        values to the local variables for the method).
	Note that if we are invoking a static method, then the
	self variable will be set to null.
*}

constdefs
  newframe_state :: "[state, mname] \<Rightarrow> state"
  "newframe_state s m \<equiv> 
	s \<lparr> invokecount:= (invokecount s) + 1,
            stackdepth := (stackdepth s) + 1\<rparr>"  (* FIXME: max?? *)

constdefs
  newframe_env :: "[env,ref,ref] \<Rightarrow> env"
  "newframe_env e objref arg \<equiv>
	  e \<lparr>  istore := emptyi,
	       rstore := ((emptyr(self := objref))(param := arg)) \<rparr>"

constdefs
  popframe :: "[state,state] \<Rightarrow> state"
 "popframe s oldstate \<equiv> 
	  s \<lparr> stackdepth := (stackdepth s) - 1,
	      maxstack := max (stackdepth s) (maxstack oldstate) \<rparr>"


text {* Values of booleans as integers *}

syntax
 grailbool :: "bool \<Rightarrow> int"
translations
 "grailbool b" == "if b then 1 else 0"



subsection {* Evalution relation *}

consts
  evalexpr  :: "(state \<times> env \<times> 'a expr \<times> val \<times> state) set"

syntax
 evalexpr_  :: "[state, env, 'a expr, val, state] \<Rightarrow> bool"      ("\<langle>_,_,_\<rangle> \<longrightarrow>e \<langle>_,_\<rangle>")

translations
 "\<langle>s,en,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>" == "(s,en,e,v,s') : evalexpr"

consts
  iheapflds :: "[(ifldname \<times> iname) list, locn, iheap, env] \<Rightarrow> iheap"
primrec
  "iheapflds [] a ihp env = ihp"
  "iheapflds (ii#iis) a ihp env = iheapflds iis a (ihp ((fst ii):=((ihp (fst ii))(a:= (env<(snd ii)>))))) env"

consts
  rheapflds :: "[(rfldname \<times> rname) list, locn, rheap, env] \<Rightarrow> rheap"
primrec
  "rheapflds [] a rhp env = rhp"
  "rheapflds (rr#rrs) a rhp env = rheapflds rrs a (rhp ((fst rr):=((rhp (fst rr))(a:=(env\<lfloor>(snd rr)\<rfloor>))))) env"


inductive evalexpr intros
 evalNull:    "\<langle>s, p, expr.Null\<rangle>   \<longrightarrow>e  \<langle>RVal Nullref, tick s\<rangle>"

 evalInt:     "\<langle>s, p, expr.Int i\<rangle>  \<longrightarrow>e  \<langle>IVal i, tick s\<rangle>"

 evalIVar:    "\<langle>s, p, IVar v\<rangle>      \<longrightarrow>e  \<langle>IVal (p<v>), tick s\<rangle>"

 evalRVar:    "\<langle>s, p, RVar v\<rangle>      \<longrightarrow>e  \<langle>RVal (p\<lfloor>v\<rfloor>), tick s\<rangle>"

 evalPrimop:  "\<langle>s, p, Primop f vn1 vn2\<rangle>  \<longrightarrow>e  \<langle>IVal (f (p<vn1>) (p<vn2>)), tickn 3 s\<rangle>"

 evalRPrimop: "\<langle>s, p, RPrimop f vn1 vn2\<rangle> \<longrightarrow>e  \<langle>IVal (f (p\<lfloor>vn1\<rfloor>) (p\<lfloor>vn2\<rfloor>)), tickn 3 s\<rangle>"

 evalGetFi:   "p\<lfloor>vn\<rfloor> = Ref a  \<Longrightarrow>  \<langle>s,p,GetFi vn f\<rangle> \<longrightarrow>e \<langle>IVal (s<a\<bullet>f>), tickn 2 s\<rangle>"

 evalGetFr:   "p\<lfloor>vn\<rfloor> = Ref a  \<Longrightarrow>  \<langle>s,p,GetFr vn f\<rangle> \<longrightarrow>e \<langle>RVal (s\<lfloor>a\<diamondsuit>f\<rfloor>), tickn 2 s\<rangle>"

 evalPutFi:   "p\<lfloor>vn1\<rfloor> = Ref a \<Longrightarrow> 
		\<langle>s,p,PutFi vn1 f vn2\<rangle> \<longrightarrow>e \<langle> IVal (p<vn2>), tickn 3 (s<a\<bullet>f := (p<vn2>)>) \<rangle>"

 evalPutFr:   "p\<lfloor>vn1\<rfloor> = Ref a \<Longrightarrow> 
                \<langle>s,p,PutFr vn1 f vn2\<rangle> \<longrightarrow>e \<langle> RVal (p\<lfloor>vn2\<rfloor>), tickn 3 (s\<lfloor>a\<diamondsuit>f := (p\<lfloor>vn2\<rfloor>)\<rfloor>) \<rangle>"

 evalNew:      "\<langle>s,p,New c ifldvals rfldvals\<rangle> \<longrightarrow>e 
		\<langle>RVal (Ref (freshlocst s)), 
		 tick (s \<lparr> oheap := (oheap s)((freshlocst s) \<mapsto>\<^sub>f c),
			   iheap := iheapflds ifldvals (freshlocst s) (iheap s) p,
			   rheap := rheapflds rfldvals (freshlocst s) (rheap s) p \<rparr>)\<rangle>"

 evalIf_True:  "\<lbrakk> p<v> = grailbool True; 
		  \<langle>tick s,p,l1\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk> \<Longrightarrow> \<langle>s,p,Ifg v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>"

 evalIf_False: "\<lbrakk> p<v> = grailbool False; 
		  \<langle>tick s,p,l2\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk> \<Longrightarrow> \<langle>s,p,Ifg v l1 l2\<rangle> \<longrightarrow>e \<langle>rtv,s1\<rangle>"

 evalLeti:     "\<lbrakk> \<langle>s,p,e\<rangle> \<longrightarrow>e \<langle>IVal i, s1\<rangle> ; \<langle>(tick s1),p<vn:=i>, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle> \<rbrakk>
	        \<Longrightarrow>
                \<langle>s,p,Leti vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

 evalLetr:     "\<lbrakk> \<langle>s,p,e\<rangle> \<longrightarrow>e \<langle>RVal r, s1\<rangle> ; \<langle>(tick s1),p\<lfloor>vn:=r\<rfloor>, ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle> \<rbrakk>
                \<Longrightarrow>
                \<langle>s,p,Letr vn e ls\<rangle> \<longrightarrow>e \<langle>rtv2, s2\<rangle>"

 evalCall:     "\<lbrakk> \<langle>tick (incrcallcount s),p, funtable fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle> \<rbrakk>
                \<Longrightarrow>
                \<langle>s,p,CALL fn\<rangle> \<longrightarrow>e \<langle>rtv, s1\<rangle>" 

 evalInvoke: 
  "\<lbrakk> p\<lfloor>vn1\<rfloor> = Ref a; s\<guillemotleft>a\<guillemotright> = Some C;
     \<langle>newframe_state s mn,
      newframe_env p (Ref a) (p\<lfloor>vn2\<rfloor>), methtable C mn\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
    \<langle>s,p,Invoke vn1 mn vn2\<rangle> \<longrightarrow>e \<langle>rtv, tickn 5 (popframe s' s)\<rangle>"

 evalInvokeStatic: 
  "\<lbrakk> \<langle>newframe_state s mn,
      newframe_env p Nullref (p\<lfloor>vn2\<rfloor>), methtable C mn\<rangle> \<longrightarrow>e \<langle>rtv, s'\<rangle> \<rbrakk>
 \<Longrightarrow>
   \<langle>s,p,InvokeStatic C mn vn2\<rangle> \<longrightarrow>e \<langle>rtv, tickn 4 (popframe s' s)\<rangle>"

 evalPre:     "\<lbrakk> \<langle>s,p,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<rbrakk> \<Longrightarrow> \<langle>s,p,Pre P e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"

 evalPost:    "\<lbrakk> \<langle>s,p,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<rbrakk> \<Longrightarrow> \<langle>s,p,Post Q e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"
 
 evalMeasure: "\<lbrakk> \<langle>s,p,e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle> \<rbrakk>   \<Longrightarrow> \<langle>s,p,Measure M e\<rangle> \<longrightarrow>e \<langle>v,s'\<rangle>"

text {*
  Notes:
  \begin{enumerate}
  \item Auxiliary variables in assertions play no role in program
  evalution, as expected.
    
  \item Put instructions ought to have no return value, but we do not
     include void type, so we return the value stored.

  \item To explain the resource counting in CALL: vn1 = 1, vn2 = 1,
  frame push/pop = 2, and the implicit return = 1.
  NB: we could break up the increment of ticks into before/after 
  CALL.


  \item Many rules omit type-checking premises (just like Grail\dots).
  For example, in get field and put field rules, we should
  check that the object belongs to a class that indeed has
  the fields being accessed.  

  \item
  Object initialization in the New rule simply sets an object
  component of the heap.
  The iheap and rheap are left alone.  For the logic (and
  by comparison with Grail), it might be better to have a
  New operator which also initialises the fields of an
  object.  However, if we admit in-place-update in objects,
  it seems difficult to avoid dealing with partly-defined
  data structures.

  \item 
   We might tickn 2 in the If rules, but we leave it at one to
   allow for extra store/retrieve which may not be necessary in 
   some cases on the real machine.

 \item The rule for evaluating Let once contained an additional side
  condition that the expression being bound does not itself contain a
  CALL (non-tail position).  So evaluation would get stuck for non-Grail
  programs.  However, there seems to be no harm in considering the
  more general semantics for a slightly richer language, since this
  side condition is not used in the logic.  (This restriction could
  therefore be considered as part of the type-checking phase for Toy Grail).

  \end{enumerate}
*}
           

end
