(* 
   Title:      GrailState.thy
   ID:         $Id: GrailState.thy,v 1.5 2003/05/16 18:37:18 da Exp $ 
   Authors:    Hans-Wolfgang Loidl, Lennart Beringer, David Aspinall
   Copyright:  University of Edinburgh, Ludwig-Maximilians Universitt, Mnchen

   Representation of machine state and resource counts.
   Auxiliary functions for manipulating the state.
*)

(*  TODO: 
      - doc/formatting improvements 
      - naming convention
 *)

header {* Grail abstract machine *}

theory GrailState = GrailAbsyn:

subsection {* Values *}

text {* First we define the \emph{run-time values}, which are 
  all of the non-object values.  These are the values that can be
  stored in local variables.  They include \emph{locations} which are
  abstract references to objects. *}

typedecl Loc

datatype RTVal  = 
                  rtVoid             -- "unit type"
                | rtInt int          -- "primitive type data"
                | rtNull             -- "polymorphic null value"
                | rtRef Loc          -- "proper references"
                | rtError            -- "abort value, e.g. type error"

consts
 the_Int    :: "RTVal \<Rightarrow> int"
 the_Ref    :: "RTVal \<Rightarrow> Loc"

primrec
 "the_Int (rtInt i) = i"
primrec
 "the_Ref (rtRef a) = a"


subsection {* Object and class representation *}

types 
  Fields = "Fldname \<leadsto> RTVal"   -- "field name, defining class, value"
  Obj    = "Cname * Fields"     -- "class instance with class name and fields"


text {* Classes are represented by records which contain field
  type information and class variables, together with a 
  list of method definitions.   Method definitions contain
  the bodies of Grail's local functions.  
  Class representations are stored in the method area. *}

record  Methd = 
          res  :: RTy           -- {* Result type *}
          args :: "Arg list"    -- {* Argument names and types *}
          funs :: "Fname \<leadsto> (Arg list * LetDecs)"  -- "Local functions"
          bdy  :: LetDecs


record  Class =
          super   :: "Cname"             -- "Name of super class"
          flds    :: "Fldname \<leadsto> RTy"    -- "Types of instance variables"
          clsvars :: "Fldname \<leadsto> RTVal"   -- "Class (static) variables"
          mthds   :: "Mname \<leadsto> Methd"    -- "Methods defined by the class."


subsection "State"

text {* The types below are the main components of a state. *}

types   
        classenv = "Cname \<leadsto> Class"   -- "Class environment"
        locals   = "Vname \<leadsto> RTVal"   -- "Local (stack) variables"
        heap     = "Loc \<leadsto> Obj"       -- "Map from locations to objects (should be finite)"

        Prog     = classenv           -- "A program denotes a class environment"


text {* For now, time and size are both measured in plain nats. 
        For time we could use a table mapping instructions to frequencies 
        instead. *}

types   Time = nat

(* da: later on maybe max heap size if we model GC *)

text {* The State contains the heap, method store, local variables,
  and the resource counters: a clock and the maximum frame size. *}

text {* FrameState is the portion of the state to save when entering a
        new method. *}

record FrameState =
          locals      :: locals        -- "local (stack) variables"
          CM          :: Mname         -- "currently executing method"
          CC          :: Cname         -- "class of currently executing method"
         
record  State = FrameState +
          classenv    :: classenv      -- "class and method definitions"
          heap        :: heap          -- "objects"
          clock       :: Time          -- "the elapsed time"
          curframes   :: nat           -- "current depth of frame stack"
          maxframes   :: nat           -- "maximum depth of frame stack so far"

constdefs
  get_framestate :: "State \<Rightarrow> FrameState"
  "get_framestate s \<equiv> FrameState.truncate s"

constdefs
  restore_framestate :: "State \<Rightarrow> FrameState \<Rightarrow> State"
  "restore_framestate s f \<equiv> 
        FrameState.extend f \<lparr> classenv=classenv s, heap=heap s, clock=clock s,
                              curframes=curframes s, maxframes=maxframes s \<rparr>"


subsection "Auxiliary functions to manipulate denotations"


(* da: dead? 
constdefs
  method :: "State \<Rightarrow> Cname \<Rightarrow> Mname \<Rightarrow> MethBody"
 "method s c m \<equiv> bdy (the (mthds (the ((methodStore s) c)) m))"
*)

constdefs
  field  :: "State \<Rightarrow> Loc \<Rightarrow> Fldname \<Rightarrow> RTVal"
 "field s loc f \<equiv> the ((snd (the ((heap s) loc))) f)"

constdefs
  init_vars:: "('a \<leadsto> 'b) \<Rightarrow> ('a \<leadsto> RTVal)"
 "init_vars m \<equiv> option_map (\<lambda>T. rtNull) o m"
  
constdefs
  get_local     :: "State \<Rightarrow> Vname \<Rightarrow> RTVal"    ("_<_>" [99,0] 99)
 "get_local s x  \<equiv> the ((locals s) x)"
  get_local_Int :: "State \<Rightarrow> Vname \<Rightarrow> int"    ("_<_>\<^sub>i" [99,0] 99)
 "get_local_Int s x  \<equiv> the_Int (the ((locals s) x))"
  get_local_Ref :: "State \<Rightarrow> Vname \<Rightarrow> Loc"    ("_<_>\<^sub>r" [99,0] 99)
 "get_local_Ref s x  \<equiv> the_Ref (the ((locals s) x))"

constdefs 
  upd_obj    :: "Loc \<Rightarrow> Fldname \<Rightarrow> RTVal \<Rightarrow> State \<Rightarrow> State"  
 "upd_obj a f v s \<equiv> let (c,fs) = the (heap s a) 
                    in  s (|heap := (heap s)(a |-> (c,fs (f |-> v)))|)"

constdefs
  get_obj       :: "State \<Rightarrow> Loc \<Rightarrow> Obj"
 "get_obj s a \<equiv> the (heap s a)"

constdefs
  obj_class     :: "State \<Rightarrow> Loc \<Rightarrow> Cname"
 "obj_class s a \<equiv> fst (get_obj s a)"

constdefs
  get_class     :: "State \<Rightarrow> Cname \<Rightarrow> Class"
 "get_class s cn \<equiv> the ((classenv s) cn)"

(* Basic access functions for fields *)
constdefs
  get_field     :: "State \<Rightarrow> Loc \<Rightarrow> Fldname \<Rightarrow> RTVal"
 "get_field s a fn \<equiv> the (snd (get_obj s a) fn)"

constdefs
  put_field     :: "State \<Rightarrow> Loc \<Rightarrow> Fldname \<Rightarrow> RTVal \<Rightarrow> State"
 "put_field s a fn v \<equiv> upd_obj a fn v s"

constdefs
  get_static     :: "State \<Rightarrow> Cname \<Rightarrow> Fldname \<Rightarrow> RTVal"
 "get_static s c fn \<equiv> the (clsvars (get_class s c) fn)"

constdefs
  put_static     :: "State \<Rightarrow> Cname \<Rightarrow> Fldname \<Rightarrow> RTVal \<Rightarrow>  State"
 "put_static s cn fn v \<equiv> 
    let oldClass =  the (classenv s cn);
        newClass = oldClass \<lparr> clsvars := (clsvars oldClass) (fn \<mapsto> v) \<rparr>
    in  s \<lparr> classenv := (classenv s) (cn \<mapsto> newClass) \<rparr>"

constdefs
  get_body :: "State \<Rightarrow> Loc \<Rightarrow> Mname \<Rightarrow> Methd"
 "get_body s a m \<equiv> let c = obj_class s a 
                   in (the ((mthds (the (classenv s c)))  m))"

text {* Get a function body from the function store: *}

constdefs
  get_funbody :: "State \<Rightarrow> Loc \<Rightarrow> Mname \<Rightarrow> Fname \<Rightarrow> LetDecs"
 "get_funbody s a mname fname \<equiv> 
    let mth = get_body s a mname
    in snd (the ((funs mth) fname))"

text {* The same thing bug using a Cname rather than an address: *}

constdefs
  get_funbody2 :: "State \<Rightarrow> Cname \<Rightarrow> Mname \<Rightarrow> Fname \<Rightarrow> LetDecs"
 "get_funbody2 s cname mname fname \<equiv> 
    let mth = the ((mthds (the (classenv s cname))) mname)
    in snd (the ((funs mth) fname))"


text {* The next function creates a new object on the heap.  
   [TODO: get class \& instance vars from class declaration,
   and set fields to initial values.] *}

constdefs 
  new_obj    :: "Loc => Cname => State => State" 
  (* "new_obj a C \<equiv> hupd (a \<mapsto> (C,init_vars (field C)))" *)
 "new_obj a c s \<equiv> s (| heap := (heap s)(a |-> (c, \<lambda> f. None)) |)"

text {* Given a classname $c$ and a methodname $m$, 
   return the body of this method. *}

constdefs
  get_method   :: "State => Cname => Mname => Methd"
 "get_method s c m \<equiv> the ((mthds (the (classenv s c))) m)"


text {* Local values (stack variables) update: add the value 
   $ s v$ to the local values of state $s$ under name 
  $x$. *}

constdefs
  lupd       :: "Vname \<Rightarrow> RTVal \<Rightarrow> State \<Rightarrow> State" 
 "lupd x sv s  \<equiv> s \<lparr> locals := ((locals s)(x \<mapsto> sv)) \<rparr>"


text {* Bind values to formal parameters and pass them on the stack: *}
consts
  initStack :: "Vname list * Arg list * State => State" 

recdef initStack "measure(\<lambda> (vs,as,s). size vs)" 
  "initStack ([], [], s) = s"
  "initStack ((v # vs), ((ARG tt aa) # as), s) =
        initStack (vs, as, lupd v (s<aa>) s)" 
(* FIXME da/lenb: the above seems to be wrong in case of aliasing between variable lists? *)




text {* The function which calculates an address for the new instruction
  will either return some fresh location, or will fail by returning
  null. [FIXME da: perhaps it should abort instead, 
	corresponding to raising an exception]  *}

constdefs
  newAddr :: "State => RTVal"
 "newAddr s \<equiv> (SOME r. (\<exists> a. r = rtRef a \<and> heap s a = None)  | r = rtNull)"


text {* Basic functions on heap and time components of the state. *}

constdefs
  tick :: "State \<Rightarrow> State"
 "tick s \<equiv> s \<lparr> clock := (clock s)+1 \<rparr>"

  elapsedTime :: "State \<Rightarrow> Time \<Rightarrow> bool"
 "elapsedTime s c \<equiv> clock s = c"

  inheritTime :: "State => State => State"
 "inheritTime s1 s2 \<equiv> s1 \<lparr> clock := (clock s2) \<rparr>"






subsection {* Initial values for State *}


consts
 rootClass :: "Cname"    --{* Root of class hierarchy (e.g. java.lang.Object) *}

consts
 mainMethod :: "Mname"   --{* Name of main method *}

consts 
 This :: "Vname"         --{* This pointer *}


text {* Initial settings for environments: *}

constdefs emptyLocals ::"locals"
"emptyLocals \<equiv> (\<lambda> v . None)"

constdefs emptyHeap :: "heap"
"emptyHeap \<equiv> (\<lambda> l . None)"

constdefs emptyCenv :: "classenv"
"emptyCenv \<equiv> (\<lambda> l . None)"

constdefs emptyFrameState :: "FrameState"
"emptyFrameState \<equiv> \<lparr> locals = emptyLocals, 
                     CM = mainMethod,
                     CC = rootClass  \<rparr>"

constdefs emptyState :: "State"
"emptyState \<equiv> FrameState.extend emptyFrameState 
              \<lparr> classenv = emptyCenv,
                heap = emptyHeap,
                clock = 0, 
                curframes = 1,
                maxframes = 1 \<rparr>"
end
