(* 
   Title:      GrailDynSem.thy
   ID:         $Id: GrailDynSem.thy,v 1.6 2003/02/24 19:56:29 da Exp $ 
   Authors:    Hans-Wolfgang Loidl, Lennart Beringer, David Aspinall
   Copyright:  University of Edinburgh, Ludwig-Maximilians Universitt, Mnchen

   Resource counting dynamic semantics for Grail.
*)

header {* Resource counting operational semantics *}

theory GrailDynSem = GrailState:

subsection "Evaluating primitive operations and tests"

consts evalBOP :: "BinOp *  RTVal * RTVal \<Rightarrow> RTVal"
recdef evalBOP "measure(\<lambda> (b,v1,v2). size v1)"
 "evalBOP (ADDop, rtInt n, rtInt m) = rtInt(n+m)"
 "evalBOP (SUBop, rtInt n, rtInt m) = rtInt(n-m)"
 "evalBOP (MULop, rtInt n, rtInt m) = rtInt(n*m)"
 "evalBOP (opn,   m,       n)       = rtError"

constdefs grailbool :: "bool \<Rightarrow> RTVal"
 "grailbool b \<equiv> rtInt(if b then 1 else 0)"

consts evalTest :: "Test * RTVal * RTVal \<Rightarrow> RTVal"
recdef evalTest "measure(\<lambda> (t,n,m). size n)"
 "evalTest (EQUALStest, rtInt n, rtInt m) = grailbool (n=m)"
 "evalTest (EQUALStest, rtRef n, rtRef m) = grailbool (n=m)"  
 "evalTest (EQUALStest, rtVoid,  rtVoid)  = grailbool True"
 "evalTest (LESStest,   rtInt n, rtInt m) = grailbool (n<m)"
 "evalTest (t,          n,       m)       = rtError"




subsection "Main evaluation functions"

subsubsection "Prototypes and syntax"

text {* main evaluation function *}

(* Evaluation function is actually a set; as in NanoJava
   One eval function per non-terminal *)

consts
 eval_PrimRes  :: "(PrimRes  * State * RTVal * State) set"
 eval_PrimOp   :: "(PrimOp   * State * RTVal * State) set"
 eval_Value    :: "(Value    * State * RTVal * State) set"
 eval_LetDecs  :: "(LetDecs  * State * RTVal * State) set"

(* prettier syntax; \<longrightarrow> is annotated with (first letter of) name of non-term *)
syntax
 eval_PrimRes  :: "[PrimRes,State,RTVal,State] \<Rightarrow> bool"   ("\<langle>_,_\<rangle> \<longrightarrow>\<^sub>p \<langle>_,_\<rangle>")
 eval_PrimOp   :: "[PrimOp,State,RTVal,State] \<Rightarrow> bool"    ("\<langle>_,_\<rangle> \<longrightarrow>\<^sub>o \<langle>_,_\<rangle>")
 eval_Value    :: "[Value,State,RTVal,State] \<Rightarrow> bool"     ("\<langle>_,_\<rangle> \<longrightarrow>\<^sub>v \<langle>_,_\<rangle>")
 eval_LetDecs  :: "[LetDecs,State,RTVal,State] \<Rightarrow> bool"   ("\<langle>_,_\<rangle> \<longrightarrow>\<^sub>l \<langle>_,_\<rangle>")

(* map syntax to evaluations *)
translations
 "\<langle>c,s\<rangle> \<longrightarrow>\<^sub>p \<langle>v,s'\<rangle>" == "(c,s,v,s') : eval_PrimRes"
 "\<langle>c,s\<rangle> \<longrightarrow>\<^sub>o \<langle>v,s'\<rangle>" == "(c,s,v,s') : eval_PrimOp"
 "\<langle>c,s\<rangle> \<longrightarrow>\<^sub>v \<langle>v,s'\<rangle>" == "(c,s,v,s') : eval_Value"
 "\<langle>c,s\<rangle> \<longrightarrow>\<^sub>l \<langle>v,s'\<rangle>" == "(c,s,v,s') : eval_LetDecs"


subsubsection "Evaluation function definition"

(* Expressions *)

(* da: Perhaps we may want lemmas for rewriting, with this form:
   \<lbrakk> x\<in>dom(locals s); X = (v,s) \<rbrakk> \<Longrightarrow>   (VARval x,s) \<longrightarrow>\<^sub>v X   *)


text {* Each value takes one clock tick to execute. *}

inductive eval_Value intros
 VARval:  "\<lbrakk> x\<in>dom(locals s); v = s<x> \<rbrakk> 
            \<Longrightarrow> 
           \<langle>VARval x,s\<rangle> \<longrightarrow>\<^sub>v \<langle>v,tick s\<rangle>"

 INTval:  "\<langle>INTval i,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtInt i,tick s\<rangle>"

 NULLval: "\<langle>NULLval str,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtNull,tick s\<rangle>"

text {* The remaining evaluation relations are defined together.  *}

inductive eval_PrimOp eval_PrimRes eval_LetDecs intros

(** PrimOp **)

 VALop: "\<lbrakk> \<langle>x,s\<rangle> \<longrightarrow>\<^sub>v \<langle>v,s'\<rangle> \<rbrakk> 
          \<Longrightarrow> 
         \<langle>VALop x,s\<rangle> \<longrightarrow>\<^sub>o \<langle>v,s'\<rangle>"

 BINop: "\<lbrakk> \<langle>v1,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv1,s1\<rangle> ; \<langle>v2,s1\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv2,s2\<rangle> \<rbrakk> 
        \<Longrightarrow>
          \<langle>BINop b v1 v2,s\<rangle> \<longrightarrow>\<^sub>o \<langle>evalBOP (b,rtv1,rtv2), tick s2\<rangle>"

 (* Objects *)
 NEWop: "\<lbrakk> newAddr s = rtRef l; c \<in> dom(classenv s) \<rbrakk>
        \<Longrightarrow>
        \<langle>NEWop c,s\<rangle> \<longrightarrow>\<^sub>o \<langle>rtRef l, tick (new_obj l c s)\<rangle>"

 (* NB: field type info ignored here *)
 GETFIELDop: "\<lbrakk> \<langle>VARval x,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtRef a,s0\<rangle> ;
                a \<in> dom (heap s0);
                fldname \<in> dom(snd (get_obj s0 a))
              \<rbrakk>
              \<Longrightarrow>
              \<langle>GETFIELDop x (FDESC fldtype cname fldname),s\<rangle> \<longrightarrow>\<^sub>o 
              \<langle>get_field s0 a fldname, tick s0\<rangle>"

 (* NB: objects currently grow with putfield; otherwise would have to
    initialize with exactly right size.  Also, ought to check that putfield
    accesses a field which is valid for the class.  Type-checking!
    (Currently we presume that this is ensured by gdf)
  *) 
 PUTFIELDop:"\<lbrakk> \<langle>VARval x,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtRef a,s0\<rangle>;
               \<langle>v,s0\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv,s1\<rangle>;
               a \<in> dom (heap s1);
               fldname \<in> dom(snd (get_obj s1 a))
             \<rbrakk>
             \<Longrightarrow>
                \<langle>PUTFIELDop x (FDESC fldtype cname fldname) v, s\<rangle> \<longrightarrow>\<^sub>o 
                \<langle>rtVoid, tick (put_field s1 a fldname rtv)\<rangle>"

 GETSTATICop:"\<lbrakk> fldname \<in> dom (clsvars (get_class s cname)) \<rbrakk>
               \<Longrightarrow>
               \<langle>GETSTATICop (FDESC fldtype cname fldname),s\<rangle> \<longrightarrow>\<^sub>o 
               \<langle>get_static s cname fldname, tick s\<rangle>"


 PUTSTATICop:"\<lbrakk> \<langle>v,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv,s1\<rangle>;
                fldname \<in> dom (clsvars (get_class s1 cname))
              \<rbrakk>
              \<Longrightarrow>
               \<langle>PUTSTATICop (FDESC fldtype cname fldname) v, s\<rangle> \<longrightarrow>\<^sub>o 
               \<langle>rtVoid, tick (put_static s1 cname fldname rtv)\<rangle>"

(* NB: typing info in MDESC ignored *)
(* NB: cost of invokes on real machine likely to be expensive because
   of allocation: here we just tick the clock for each Grail instruction. 
 *)

(* FIXME: should add check here that method is defined *)

 INVOKESTATICop: "\<lbrakk> 
                    mthd = get_method s cname mname ;
                    oldframe = get_framestate s ;
                    s1 = initStack (vs, args mthd, s) ; 
                    s2 = s1 \<lparr> CM := mname, CC := cname,
                              curframes := (curframes s1) + 1,
                              maxframes := max ((curframes s1) + 1) (maxframes s1) \<rparr> ;
                    \<langle>bdy mthd, s2\<rangle> \<longrightarrow>\<^sub>l \<langle>rtv, s3\<rangle> ;
                    s4 = restore_framestate s3 oldframe
                   \<rbrakk>
                \<Longrightarrow>
                \<langle>INVOKESTATICop (MDESC ty cname mname tys) vs, s\<rangle> \<longrightarrow>\<^sub>o 
                \<langle>rtv, tick s4\<rangle>"


(* Costs: 1 for fct call; 1 for conditional; rest summed up; here we know the
          branch to take; in a static analysis that has to be approximated *)


(** PrimRes **)

(* NB: We charge one tick for the terminal PrimRes because they correspond
   to return points from methods. *)

 OPres:   "\<lbrakk> \<langle>p,s\<rangle> \<longrightarrow>\<^sub>o \<langle>x,s'\<rangle> \<rbrakk> 
           \<Longrightarrow>
           \<langle>OPres p,s\<rangle> \<longrightarrow>\<^sub>p \<langle>x,tick s'\<rangle>"

 VOIDres: "\<langle>VOIDres,s\<rangle> \<longrightarrow>\<^sub>p \<langle>rtVoid,tick s'\<rangle>"

(* Invoking a Grail function (jump) *)

(* FIXME: should add check here that function is defined *)

 FUNres: "\<lbrakk>  b = get_funbody2 s (CC s) (CM s) f ; \<langle>b,s\<rangle> \<longrightarrow>\<^sub>l \<langle>x,s'\<rangle> \<rbrakk>
         \<Longrightarrow>
            \<langle>FUNres f xs,s\<rangle> \<longrightarrow>\<^sub>p \<langle>x,tick s'\<rangle>"



(** LetDecs **)

 VALdec: "\<lbrakk> \<langle>primop,s\<rangle> \<longrightarrow>\<^sub>o \<langle>rtv,s1\<rangle> ;
            \<langle>letdecs,lupd vname rtv s1\<rangle> \<longrightarrow>\<^sub>l \<langle>rtv2,s2\<rangle> \<rbrakk> 
          \<Longrightarrow>
          \<langle>VALdec vname primop letdecs,s\<rangle> \<longrightarrow>\<^sub>l \<langle>rtv2,tick s2\<rangle>"  

(* NB: we consider VOIDdec as free because there is no store of
  the value involved. But we should think exactly what this cost model
  is charging for --- at the moment it isn't \emph{quite} the number of 
  Grail executions executed (nor the number of JVM instructions). *)

 VOIDdec: "\<lbrakk> \<langle>primop,s\<rangle> \<longrightarrow>\<^sub>o \<langle>x,s1\<rangle>;  \<langle>letdecs,s1\<rangle> \<longrightarrow>\<^sub>l \<langle>rtv,s2\<rangle> \<rbrakk>
           \<Longrightarrow>
           \<langle>VOIDdec primop letdecs,s\<rangle> \<longrightarrow>\<^sub>l \<langle>rtv,s2\<rangle>"

 PRIMres: "\<lbrakk> \<langle>pres,s\<rangle> \<longrightarrow>\<^sub>p \<langle>v,s'\<rangle>\<rbrakk> 
           \<Longrightarrow>  
            \<langle>PRIMres pres,s\<rangle> \<longrightarrow>\<^sub>l \<langle>v,s'\<rangle>"

 CHOICEres_True: "\<lbrakk> \<langle>v1,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv1,s1\<rangle> ; \<langle>v2,s1\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv2,s2\<rangle> ; 
                    evalTest (t,rtv1,rtv2) = grailbool True; 
		    \<langle>p1,s2\<rangle> \<longrightarrow>\<^sub>p \<langle>x,s3\<rangle> \<rbrakk> 
                  \<Longrightarrow> 
                  \<langle>CHOICEres  v1 t v2 p1 p2,s\<rangle> \<longrightarrow>\<^sub>l \<langle>x, tick s3\<rangle>"

 CHOICEres_False: "\<lbrakk> \<langle>v1,s\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv1,s1\<rangle> ; \<langle>v2,s1\<rangle> \<longrightarrow>\<^sub>v \<langle>rtv2,s2\<rangle> ; 
                    evalTest (t,rtv1,rtv2) = grailbool False; 
		    \<langle>p2,s2\<rangle> \<longrightarrow>\<^sub>p \<langle>x,s3\<rangle> \<rbrakk> 
                  \<Longrightarrow> 
                  \<langle>CHOICEres  v1 t v2 p1 p2,s\<rangle> \<longrightarrow>\<^sub>l \<langle>x, tick s3\<rangle>"


end

