(* Grail abstract syntax *)

val required = () (* for mosmldep *)

datatype ClassDef = 
	 CDEF of Classdecl.class_access_flag list * string *
		 string option * string list * FieldDef list * MethodDef list
		 * layout list option
and FieldDef = FDEF of Classdecl.field_access_flag list * Ty * string
and MethodDef = MDEF of Classdecl.method_access_flag list * RTy * string *
    (Ty*Var) list * MethodBody
and MethodBody = MBODY of LetDec list * FunDec list * Result
and LetDec =
    VALdec of Var * PrimOp
  | VOIDdec of PrimOp
and FunDec = FDEC of string * (Ty*Var) list * FunBody
and FunBody = FUNbody
 of LetDec list * Result
and Result =
    PRIMres of PrimRes
  | CHOICEres of Value * Test * Value * PrimRes * PrimRes
  | CASEres of Var * int * int * (int * string * Var list) list
and PrimRes =
    VOIDres
  | OPres of PrimOp
  | FUNres of string * Var list
and PrimOp =
    VALop of Value
  | BINop of BinOp * Value * Value
  | NEWop of MethDesc * Value list
  | CHECKCASTop of string * Var
  | INSTANCEop of string * Var
  | INVOKESTATICop of MethDesc * Value list
  | INVOKEVIRTUALop of Var * MethDesc * Value list
  | INVOKEINTERFACEop of Var * MethDesc * Value list
  | INVOKESPECIALop of Var * MethDesc * Value list
  | GETFIELDop of Var * FieldDesc
  | PUTFIELDop of Var * FieldDesc * Value
  | GETSTATICop of FieldDesc
  | PUTSTATICop of FieldDesc * Value
  | MAKEop of Value * Value
  | GETop of Value * Value
  | SETop of Value * Value * Value
  | LENGTHop of Value
  | EMPTYop of Value * Ty
  | FTOIop of Value
  | ITOFop of Value
and MethDesc = MDESC of RTy * string * Ty list
and FieldDesc = FDESC of Ty * string
and Test = EQtest | NEtest | Ltest | LEtest | Gtest | GEtest
and BinOp = ADDop | SUBop | MULop | DIVop | MODop
	  | ANDop | ORop  | XORop | SHLop | SHRop | USHRop  (* bitwise logic *)

and Value =
    VARval of Var
  | BYTEval of int
  | SHORTval of int
  | INTval of int  (* MIGHT BE TOO SHORT *)
  | LONGval of int (* MIGHT BE TOO SHORT *)
  | CHARval of int
  | FLOATval of real
  | DOUBLEval of real (* MIGHT BE TOO SHORT *)
  | STRINGval of string
  | NULLval of string * string option 
	(* 2nd component is Camelot type, for theorem-proving *)

and Ty = (* many of these aren't accessible from Grail (yet) *)
    BYTEty    (* 8-bit signed int *)
  | SHORTty   (* 16-bit signed int *)
  | INTty     (* 32-bit signed int *)
  | LONGty    (* 64-bit signed int *)
  | CHARty    (* 16-bit unsigned Unicode character *)
  | FLOATty   (* float *)
  | DOUBLEty  (* extended float *)
  | BOOLEANty
  | REFty of string
  | ARRAYty of Ty

withtype Var = string
and RTy = Ty option
and layout = string * (string * int * (string * string) list) list
(* (typename, [(constructor name, tag, [(field name, field type)])]) *)
;

exception grailError of string;

(* ---------------- Variable and field descriptors for Isabelle output ---------------- *)

fun tyPrefix t = 
    case t of
	BYTEty =>    "b"
      | CHARty =>    "c"
      | DOUBLEty =>  "d"
      | FLOATty =>   "f"
      | INTty =>     "v"
      | LONGty =>    "l"
      | SHORTty =>   "s"
      | BOOLEANty => "z"
      | REFty _ =>   "r"
      | ARRAYty _ => "a"

(* Make these consistent with JVM type descriptors? *)

fun uc1char s = String.str (Char.toUpper (String.sub(s,0)))

fun ucTyPrefix t = uc1char (tyPrefix t)


(************ Printing abstract syntax ************)

val indent = ref "";
fun incIndent() = indent := (!indent) ^ "   ";
fun decIndent() = indent := String.extract(!indent, 3, NONE);


val ostream = ref TextIO.stdOut : TextIO.outstream ref
fun setOut x = ostream := x;
fun Pr x = TextIO.output(!ostream,x)
fun Prl s = TextIO.output(!ostream,s^"\n")
fun tab() = TextIO.output(!ostream, !indent)
fun nl(s) =  TextIO.output(!ostream, "\n" ^ (!indent) ^ s)

(* HWL was here *)
val reallyprint = ref true;
fun PrF os x = if (!reallyprint) then TextIO.output(os, x^" ") else ();
fun PrlF os x = if (!reallyprint) then TextIO.output(os, x^"\n") else ();
fun nlF os s =  if (!reallyprint) then TextIO.output(os, "\n" ^ (!indent) ^ s) else ();

fun tyToString BYTEty       = "byte"
  | tyToString SHORTty      = "short"
  | tyToString INTty        = "int"
  | tyToString LONGty       = "long"
  | tyToString CHARty       = "char"
  | tyToString FLOATty      = "float"
  | tyToString DOUBLEty     = "double"
  | tyToString BOOLEANty    = "boolean"
  | tyToString (ARRAYty(t)) = tyToString(t) ^ "[]"
  | tyToString (REFty(s))   = s

fun rtyToString (NONE) = "void"
  | rtyToString (SOME ty) = tyToString ty

fun itos n = 
    let
	val s = Int.toString (Int.abs n)
    in
	if (n<0) then "-" ^ s else s
    end

val prInt = Pr o itos

fun valToString w =
    case w of
	VARval v => v
      | BYTEval b  => itos b
      | SHORTval s => itos s
      | INTval n   => itos n   (* FIX: may be too short *)
      | LONGval l  => itos l  (* FIX: may be too short *)
      | CHARval c  => ( 
	String.str (Char.chr c) handle Chr => raise grailError "character too big")
      | FLOATval f =>
	let 
	    val s = Real.toString (Real.abs f)
	in
	    if (f<0.0) then "-" ^ s else s
	end
      | DOUBLEval d =>
	let 
	    val s = Real.toString (Real.abs d)
	in
	    if (d<0.0) then "-" ^ s else s
	end
      | STRINGval s => "\"" ^ s ^ "\"" 
       (* CAUTION:  string values can contain escape sequences.  We require that 
          escape sequences are converted into literals during code emission.
          Disassembly should perform the reverse translation. *)
      | NULLval(s,t) => 
	let in 
	    case t of NONE => "null[" ^ s ^ "]"
		    | SOME d => "null[" ^ s ^ "]{" ^ d ^ "}"
	end


fun listToString toString separator l =
case l of [] => ""
        | [h] => toString h
        | h::t => (toString h) ^ separator ^ (listToString toString separator t)

fun id x = x

fun vecToString toString l = (* eg [1,2,5] -> "(1, 2, 5)" *)
    "("
    ^ (listToString toString ", " l)
    ^ ")"

val strListToString = listToString (fn x => x)

fun prVec toString l =
       Pr (vecToString toString l)

val prVarVec = prVec id

val prIntVec = prVec Int.toString

val prTyVec = prVec tyToString

val prTyVarVec = prVec (fn (t,v) => (tyToString t ^ " " ^ v))

val prValVec = prVec valToString

fun nth([],_) = raise grailError "nth: list to short"
  | nth (h::_,0) = h
  | nth (_::t,n) = nth(t,n-1)

fun prClassDef (CDEF (flags, cname, super, intfs, fdefs, mdefs, layout)) = 
(
 app prCflag flags;
 Pr ("class " ^ cname ^ " ");
 prSuper super; prIntfs intfs;
 Pr " {";
 incIndent();
 app prFieldDef fdefs;
 app prMethodDef mdefs;
 decIndent();
 Prl "}"
 (* ; prLayout layout *)
)

and prLayout NONE = ()
  | prLayout (SOME l) = 
    let
	fun listToString toString separator l =
	    case l of [] => ""
		    | [h] => toString h
		    | h::t => (toString h) ^ separator ^ (listToString toString separator t)
			      
	fun prConLayout (C, tag, fields) = 
	    (
	     tab();
	     Pr C;
	     Pr ": $tag = ";
	     Pr (Int.toString tag);
	     Pr ", $fields = [";
	     Pr (listToString (fn (s,t) => "<" ^ s ^ ", " ^ t ^ ">") ", " fields);
	     Prl "]"
	    )

	fun pr1type (tname, l) =
	    (incIndent();
	     tab();
	     Pr "type ";
	     Prl tname;
	     incIndent ();
	     app prConLayout l;
	     decIndent ();
	     Pr "\n";
	     decIndent ()
	    )
    in
	(
	 Prl "\nlayout {";
	 app pr1type l;
	 Prl "}\n"
	)
    end
     
and prSuper (SOME cl) = Pr ("extends " ^ cl ^ " ")
  | prSuper (NONE) = ()
and prIntfs [] = ()
  | prIntfs ifs =
    (
      Pr "implements ";
      Pr (listToString id " " ifs)
    )



and prFieldDef (FDEF(flags, ty, name )) =
    (
     nl "field ";
     app prFflag flags;
     prTy ty;
     Pr " ";
     Prl name
    )

and prMethodDef (MDEF(flags, rty, name, params, mbody)) =
    (
     nl "method ";
     app prMflag flags;
     prRTy rty;
     Pr (" " ^ name ^ " ");
     prTyVarVec params;
     Pr " =";
     prMethodBody mbody
     )

and prMethodBody (MBODY(letDecs, funDecs, result)) =
    (
     nl "let";
     incIndent();
     app prLetDec letDecs;
     app prFunDec funDecs;
     decIndent();
     nl "in";
     prResult result;
     nl "end\n"
     )

and prLetDec (VALdec(id, p)) =
    ( nl("val " ^ id ^ " = "); prPrimOp p )
  | prLetDec (VOIDdec p) =
    ( nl "val () = "; prPrimOp p)

and prFunDec (FDEC (fname, params, fbody)) =
    (
     Pr "\n";
     nl("fun " ^ fname );
     prTyVarVec params;
     Pr " =";
     prFunBody fbody
)

and prFunDecList [] = ()  (* just to see if we can get round the bug *)
  | prFunDecList (h::t) = (prFunDec h; prFunDecList t)

and prFunBody (FUNbody(letdecs, result)) =
    if letdecs = nil then prResult result
    else
	(nl "let";
	 incIndent();
	 app prLetDec letdecs;
	 decIndent();
	 nl "in";
	 prResult result;
	 nl "end"
	 )

and prResult (PRIMres p) = (incIndent(); nl ""; prPrimRes p; decIndent())
  | prResult (CHOICEres (v, tst, w, p1, p2)) =
    (
     incIndent();
     nl "if "; prValue v; prTest tst; prValue w;
     nl "then "; prPrimRes p1;
     nl "else "; prPrimRes p2;
     decIndent()
     )
  | prResult (CASEres (v, low, high, cases)) =
    ( 
     incIndent();
     nl "case "; Pr v; Pr " in [";
     prInt low; Pr ".."; prInt high;
     Pr "] of";
     incIndent ();
     case cases of 
	 [] => Pr "(* Empty case statement *)"
       | h::t => (prCase "  " h;  app (prCase "| ") t);
     decIndent ();
     decIndent()
    )

and prCase s (n, name, args) =
    (
     nl s;
     prInt n;
     Pr " => ";
     Pr name;
     prVarVec args
)
     
and prPrimRes r = 
    case r of
	OPres p => prPrimOp p
      | VOIDres => Pr "()"
      | FUNres (name, args) => (Pr name; prVarVec args)
		
and prPrimOp p = 
    case p of 
	VALop v => prValue v
      | BINop(b, v, w) => (prBinOp b; prValue v; Pr " "; prValue w)
      | NEWop(m,v) => prNew (m, v)
      | CHECKCASTop(s,v) => Prl ("checkcast " ^ s ^ " " ^ v )
      | INSTANCEop(s,v) =>  Prl ("instance " ^ s ^ " " ^ v )
      | INVOKESTATICop(mdesc, args) =>
	( Pr "invokestatic "; prMethDesc mdesc; prValVec args )
      | INVOKEVIRTUALop(v, mdesc, args) =>
	( Pr ("invokevirtual " ^ v ^ " "); prMethDesc mdesc; prValVec args )
      | INVOKESPECIALop(v, mdesc, args) =>
	( Pr ("invokespecial " ^ v ^ " "); prMethDesc mdesc; prValVec args )
      | INVOKEINTERFACEop(v, mdesc, args) =>
	( Pr ("invokeinterface " ^ v ^ " "); prMethDesc mdesc; prValVec args )
      | GETFIELDop(v, fdesc) =>
	( Pr ("getfield " ^ v ^ " " ); prFieldDesc fdesc )
      | PUTFIELDop(v, fdesc, w) =>
	( Pr ("putfield " ^ v ^ " "); prFieldDesc fdesc; Pr " "; prValue w )
      | GETSTATICop(fdesc) =>
	( Pr "getstatic "; prFieldDesc fdesc )
      | PUTSTATICop(fdesc,w) =>
	( Pr "putstatic "; prFieldDesc fdesc; Pr " "; prValue w)
      | MAKEop(a, i) =>
	( Pr "make "; prValue a; Pr " "; prValue i)
      | SETop(aa, i, a) =>
	( Pr "set "; prValue aa; Pr " "; prValue i; Pr " "; prValue a)
      | LENGTHop(aa) =>
	( Pr "length "; prValue aa)
      | EMPTYop(a,ARRAYty(ty)) =>
	( Pr "empty "; prValue a; Pr " "; prTy ty)
      | GETop(a,i) =>
	( Pr "get "; prValue a; Pr " "; prValue i)
      | FTOIop(f) =>
	( Pr "ftoi "; prValue f)
      | ITOFop(i) =>
	( Pr "itof "; prValue i)
      | _ =>
	( raise grailError "Not implemented yet" )

(* Stupid structure due to compiler bug *)
and prNew (mdesc, vals) =
    (case mdesc of
	 MDESC(rty, name, types) => (Pr ("new <" ^ name);
				     prTyVec types;
				     Pr "> ";
				     prValVec vals
				     )
     )
and prMethDesc (MDESC(rty, name, types)) =
    (Pr  "<";
     prRTy rty;
     Pr (" " ^ name ^ " ");
     prTyVec types;
     Pr "> "
     )
and prFieldDesc (FDESC(ty, name)) =
    (Pr "<";
     prTy ty;
     Pr " ";
     Pr name;
     Pr ">"
     )

and prTest t = 
    let
	val s = 
	    case t of
		EQtest => " = "
	      | NEtest => " <> "
	      | Ltest  => " < "
	      | LEtest => " <= "
	      | Gtest  => " > "
	      | GEtest => " >= "
    in
	Pr s
    end

and prBinOp t =
    let val s = 
	    case t of
		ADDop => "add "
	      | SUBop => "sub "
	      | MULop => "mul "
	      | DIVop => "div "
	      | MODop => "mod "
	      | ANDop => "and "
	      | ORop  => "or "
	      | XORop => "xor "
	      | SHLop => "shl "
	      | SHRop => "shr "
	      | USHRop => "ushr "
    in
	Pr s
    end

and prValue v = Pr (valToString v)

and prTy t = Pr (tyToString t)

and prRTy r = Pr (rtyToString r)

and prCflag f = 
    let 
	val s = 
	    case f of
		Classdecl.C_ACCpublic =>     "public "
	      | Classdecl.C_ACCprivate =>    "private "
	      | Classdecl.C_ACCprotected =>  "protected "
	      | Classdecl.C_ACCstatic =>     "static "
	      | Classdecl.C_ACCfinal =>      "final "
	      | Classdecl.C_ACCinterface =>  "interface "
	      | _ =>                         ""
    in
	Pr s
    end

and prFflag f = 
    let 
	val s = 
	    case f of
		Classdecl.F_ACCpublic =>     "public "
	      | Classdecl.F_ACCprivate =>    "private "
	      | Classdecl.F_ACCprotected =>  "protected "
	      | Classdecl.F_ACCstatic =>     "static "
	      | Classdecl.F_ACCfinal =>      "final "
	      | Classdecl.F_ACCvolatile =>   "volatile "
	      | Classdecl.F_ACCtransient =>  "transient "
	      | Classdecl.F_ACCsynthetic =>  "synthetic "
	      | Classdecl.F_ACCenum =>       "enum"
    in
	Pr s
    end

and prMflag f = 
    let 
	val s = 
	    case f of
		Classdecl.M_ACCpublic       => "public "
	      | Classdecl.M_ACCprivate      => "private "
	      | Classdecl.M_ACCprotected    => "protected "
	      | Classdecl.M_ACCstatic       => "static "
	      | Classdecl.M_ACCfinal        => "final "
	      | Classdecl.M_ACCsynchronized => "synchronized "
	      | Classdecl.M_ACCbridge       => "bridge "
	      | Classdecl.M_ACCvarargs        => "varargs "
	      | Classdecl.M_ACCnative         => "native "
	      | Classdecl.M_ACCabstract       => "abstract "
	      | Classdecl.M_ACCstrictfp       => "strictfp "
	      | Classdecl.M_ACCsynthetic    => "synthetic"
    in
	Pr s
    end
;





(* Matthew's CoreGrail Abstract Syntax printing functions *)

val names = ref [];

(* da: really should record correct types here as well --- might as well accumulate Isabelle
   syntax, in fact *)
fun addName (n:string) = if (not (List.exists (fn x => x=n) (!names))) andalso
			    (not (n = "()")) andalso   (* da: avoid generating bad Isabelle syntax *)
			    (not (n = "<init>"))
			 then names := n::(!names) else ();

fun printNames [] = ()
  | printNames (h::t) = (Pr (h^" :: \"string\"\n"); printNames t)

fun printNamesF os [] = ()
  | printNamesF os (h::t) = (PrF os (h^" :: \"string\"\n"); printNamesF os t)


fun printCoreGrailPROG (MDEF (_, _, _, _, body)) =
(
  Prl "theory ACoreGrailProgram = CoreGrailAbsyn:\n";
  Pr "constdefs prog :: \"Prog\"\n\"prog ==";
  incIndent();
  nl "PROG (";
  incIndent(); nl "";
  printCoreGrailMBODY body;
  decIndent();
  nl ")";
  decIndent();
  Prl "\"";

  Prl "";
  Pr ("(* Move the next "^Int.toString((length (!names))+1)^" to before the \"constdefs prog...\" line *)\n");
  Pr "consts\n";
  printNames(!names);
  Prl "\nend"
)

and printCoreGrailMBODY (MBODY (letDecs, funDecs, result)) =
(
  Pr "MBODY";
  incIndent();

  (if letDecs=[] then nl "[]" else (
  nl "["; incIndent();
  printCoreGrailLetDecs letDecs;
  decIndent(); Pr "]"));

  (if funDecs=[] then nl "[]" else (
  nl "["; incIndent();
  printCoreGrailFunDecs funDecs;
  decIndent(); Pr "]"));

  nl ""; printCoreGrailResult result;

  decIndent()
)

and printCoreGrailLetDecs [] = ()
  | printCoreGrailLetDecs (h::[]) = printCoreGrailLetDec h
  | printCoreGrailLetDecs (h::t) = (printCoreGrailLetDec h; Pr ","; printCoreGrailLetDecs t)

and printCoreGrailLetDec (VALdec (name, primop)) =
( addName(name);
  nl ("(VALdec "^name^" ");
  printCoreGrailPrimOp primop;
  Pr ")"
)
  | printCoreGrailLetDec (VOIDdec (primop)) =
(
  Pr "(VOIDdec ";
  printCoreGrailPrimOp primop;
  Pr ")"
)

and printCoreGrailFunDecs [] = ()
  | printCoreGrailFunDecs (h::[]) = printCoreGrailFunDec h
  | printCoreGrailFunDecs (h::t) = (printCoreGrailFunDec h; Pr ","; printCoreGrailFunDecs t)

and printCoreGrailFunDec (FDEC (name, args, body)) =
( addName(name);
  nl ("(FDEC "^name);
  incIndent();

  (if args=[] then nl "[]" else (
  nl "["; incIndent();
  printCoreGrailFunArgs args;
  decIndent(); Pr "]"));

  printCoreGrailFunBody body;

  decIndent(); Pr ")"
)

and printCoreGrailFunArgs [] = ()
  | printCoreGrailFunArgs ((ty,v)::[]) = Pr ("("^(if (ty=INTty) then "INTty" else "(NOT SUPPORTED)")^","^v^")")
  | printCoreGrailFunArgs ((ty,v)::t) = (Pr ("("^(if (ty=INTty) then "INTty" else "(NOT SUPPORTED)")^","^v^")"); Pr ","; printCoreGrailFunArgs t)

and printCoreGrailFunBody (FUNbody (letDecs, result)) =
(
  nl "(FUNbody";
  incIndent();

  (if letDecs=[] then nl "[]" else (
  nl "["; incIndent();
  printCoreGrailLetDecs letDecs;
  decIndent(); nl "]"));

  (*nl "("; incIndent();*)
  nl "";
  printCoreGrailResult result;
  (*decIndent(); nl ")";*)
  decIndent(); Pr ")"
)

and printCoreGrailResult (PRIMres primres) =
(
  Pr "(PRIMres"; printCoreGrailPrimRes primres; Pr ")"
)
  | printCoreGrailResult (CHOICEres (v, test, v', primres, primres')) =
(
  Pr "(CHOICEres";
  incIndent();
  nl "(CONDhead ";
  printCoreGrailValue v; Pr " ";
  printCoreGrailTest test; Pr " ";
  printCoreGrailValue v';
  nl ")";
  nl "";
  printCoreGrailPrimRes primres;
  nl "";
  printCoreGrailPrimRes primres';
  decIndent();
  nl ")"
)
  | printCoreGrailResult (CASEres (name, low, high, cases)) =
    (
     Pr "CASEres not implemented in CoreGrail\n"
    )

and printCoreGrailPrimOp (VALop v) = (Pr "(VALop "; printCoreGrailValue v; Pr ")")
  | printCoreGrailPrimOp (BINop (b,v,v')) =
(
  Pr "(BINop ";
  printCoreGrailBinOp b; Pr " ";
  printCoreGrailValue v; Pr " ";
  printCoreGrailValue v';
  Pr ")"
)
  | printCoreGrailPrimOp (INVOKESTATICop (M,cs)) = Pr "INVOKESTATICop..."
  | printCoreGrailPrimOp _ = Pr "(NOT SUPPORTED)"

and printCoreGrailPrimRes (OPres p) = (Pr "(OPres "; printCoreGrailPrimOp p; Pr ")")
  | printCoreGrailPrimRes VOIDres = Pr "VOIDres"
  | printCoreGrailPrimRes (FUNres (s,args)) = (Pr ("(FUNres "^s^" ["); printCoreGrailArgNames args; Pr "])")

and printCoreGrailArgNames [] = ()
  | printCoreGrailArgNames (h::[]) = Pr h
  | printCoreGrailArgNames (h::t) = (Pr (h^","); printCoreGrailArgNames t)

and printCoreGrailValue (VARval v) = (addName v; Pr ("(VARval "^v^")"))
  | printCoreGrailValue (INTval i) = Pr ("(INTval "^Int.toString(i)^")")
  | printCoreGrailValue (STRINGval s) = Pr ("(STRINGval "^s^")")
  | printCoreGrailValue (NULLval (s,t)) = 
    (
     case t of NONE => Pr ("(NULLval "^s^")")
	     | SOME d => Pr ("(NULLval "^s^"){"^d^"}")
    )
  | printCoreGrailValue _ = Pr "(NOT SUPPORTED)"

and printCoreGrailTest EQtest = Pr "EQUALStest"
  | printCoreGrailTest Ltest = Pr "LESStest"
  | printCoreGrailTest _ = Pr "(NOT SUPPORTED)"

and printCoreGrailBinOp ADDop = Pr "ADDop"
  | printCoreGrailBinOp SUBop = Pr "SUBop"
  | printCoreGrailBinOp MULop = Pr "MULop"
  | printCoreGrailBinOp _ = Pr "(NOT SUPPORTED)"
;

