(* Wrapper for standalone version of gdf *)

exception toyError of string;
exception gdfError of string;

fun usage() = TextIO.print "Usage: gdf [-c] [-t [syntax]] [-d outputdir] file1 [file2 ...]\n\
\   -d dir       use dir as output directory\n\
\   -c           print class name\n\
\   -T           print stardate\n\
\   -t [syntax]  generate Isabelle .thy file as well [in given syntax]\n"

fun extend extn filename = (* Add extension if necessary *) (* Stolen from Camelot/src/Util.sml *)
   let 
       val e = Path.ext filename;
   in 
       if e = SOME extn 
       then 
	   filename 
       else 
	   Path.joinBaseExt {base=filename, ext=SOME extn}
   end

(* val doargs: string list -> string -> bool -> int option -> unit *)
fun doargs ("-h"::t) _ _ _ = usage ()
  | doargs ("-s"::t) dir pCN thySyntax' = (ToyGrailAbsyn.shut_up := true ; doargs t dir pCN thySyntax')
  | doargs ("-c"::t) dir _ thySyntax = doargs t dir true thySyntax
  | doargs ("-z"::t) _ _ _ = TextIO.print (Stardate.stardate())
  | doargs ("-v"::t) dir printClassName thySyntax' = let (* set verbose level *)
         in case t of
	    nil => usage() 
         | d::t' => 
            if (String.isPrefix "-" d) 
              then (ToyGrailAbsyn.verbose_level := ToyGrailAbsyn.VERB_MAX ; 
                    doargs t dir printClassName thySyntax')
              else case Int.fromString d of  
                       NONE => (ToyGrailAbsyn.verbose_level := ToyGrailAbsyn.VERB_MAX ; 
                                doargs t dir printClassName thySyntax') 
                     | (SOME v) => (ToyGrailAbsyn.verbose_level := v ; 
                                    doargs t' dir printClassName thySyntax')
         end

  | doargs ("-test"::t) dir printClassName thySyntax' = 
    (Compile.testing := true; doargs t dir printClassName thySyntax')

  | doargs ("-l"::t) dir printClassName (p,d,tag,thy,tFlavour)  = doargs t dir printClassName (p,d+10,tag,thy,tFlavour) 
     (* thySyntax encoding = (printCertificate?, dataLayout, tag_offset, logic *)

  | doargs ("-L"::t)  dir printClassName (p,d,tag,thy,tFlavour)  =  (* set data layout *)
    let in case t of
	 nil => usage() 
       | layoutStr::t' => 
	 let
             val l = case (Int.fromString layoutStr) of
				 NONE => 0
                               | (SOME x) => x
         in
           doargs t' dir printClassName (p,l,tag,thy,tFlavour) (* set layout *)
         end
    end

  | doargs ("-T"::t) dir printClassName (p,d,tag,thy,tFlavour)  = (* set tag offset *)
    let in case t of
	 nil => usage() 
       | tagStr::t' => 
	 let
             val tag' = case (Int.fromString tagStr) of
				 NONE => 0
                               | (SOME x) => x
         in
           doargs t' dir printClassName (p,d,tag',thy,tFlavour) (* set layout *)
         end
    end

  | doargs ("-F"::t) dir printClassName (p,d,tag,thy,tFlavour)  = (* set tactic flavour *)
    let in case t of
	 nil => usage() 
       | tagStr::t' => 
	 let
             val tag' = case (Int.fromString tagStr) of
				 NONE => 0
                               | (SOME x) => x
             val (tFlavour'', wrapper'') = (tag' mod 10, tag' div 10)
         in
           doargs t' dir printClassName (p,d,tag,thy,tFlavour'') (* set tactic flavour *)
         end
    end
  | doargs ("-C"::t) dir printClassName (p,d,tag,thy,tFlavour)  = doargs t  dir printClassName (true,d,tag,thy,tFlavour) 
     (* enable certificate generation by setting 100 field in thySyntax *)

  | doargs ("-t"::t) dir printClassName (p,d,tag,thy,tFlavour) =
    let 
    in case t of
	 nil => usage() 
       | z::args' =>
	        if (String.isPrefix "-" z) then doargs (z::args') dir printClassName (p,d,tag,6,tFlavour) (* 6 is default syntax *) 
		else
		    case z of
			(* default:1 toy:2 bcl:3 *)
			"toy" => doargs args' dir printClassName (p,d,tag, 2,tFlavour) (* use ToyGrail syntax *)
		      | "bcl" => doargs args' dir printClassName (p,d,tag, 3,tFlavour) (* use BytecodeLogic syntax *)
		      | "dal" => doargs args' dir printClassName (p,d,tag, 4,tFlavour) (* use DA Logic syntax *)
		      | "vcg" => doargs args' dir printClassName (p,d,tag, 5,tFlavour) (* use VCG Logic syntax *)
		      | "mrg" => doargs args' dir printClassName (p,d,tag, 6,tFlavour) (* use MRG Logic (final version!) *)
		      | _ => doargs (z::args') dir printClassName (p,d,tag, 6,tFlavour)
		    (*| _ => raise (gdfError (d ^ " is not a recognised logic format"))*)
    end
	 
  | doargs ("-d"::t) _ printClassName thySyntax = 
    let in case t of 
         nil => raise (gdfError "no directory specified for classfile output")
       | d::args' => 
	 	if not (Nonstdio.file_exists d) then raise (gdfError ("can't find directory " ^ d))
		else 
		   if not (FileSys.isDir d) then raise (gdfError (d ^ " is not a directory"))
		   else let 
        	        val dir = if (String.sub (d, (String.size d)-1)) = #"/" 
	 		          then d else d ^ "/"
			in
			    doargs args' dir printClassName thySyntax
			end
    end

  | doargs [] dir printClassName thySyntax =
    let 
	val cdef = MainParser.parseStdIn ()
(*LENBB11/12/2004		val cname = Compile.compile "std_in" cdef dir generateThy thySyntax NONE*)
			    (* NONE -> no rich type predicates *)
		val cname = Compile.compile "std_in" cdef dir thySyntax (Binarymap.mkDict (String.compare))
			    (* empty Dictionary -> no rich type predicates *)
    in
	if printClassName then 
	    TextIO.print (cname ^ "\n")
	else (* print a (not very) helpful message *)
		 TextIO.print ("Compiled " ^ "std_in" ^ "\n")
    end
  | doargs args dir printClassName thySyntax =
    let 
	fun compileFile filename = 
	    let 
		val cdef = MainParser.parse filename
(*LENBB11/12/2004		val cname = Compile.compile filename cdef dir generateThy thySyntax NONE*)
			    (* NONE -> no rich type predicates *)
		val cname = Compile.compile filename cdef dir thySyntax (Binarymap.mkDict (String.compare))
			    (* empty Dictionary -> no rich type predicates *)
	    in
		if printClassName then 
		   TextIO.print (cname ^ "\n")
		else (* print a (not very) helpful message *)
		   TextIO.print ("Compiled " ^ filename ^ "\n")
	    end
	in
	    app compileFile (map (extend "gr") args)
	end

val _ = doargs (CommandLine.arguments()) "" false  (false,0,0,6,4)
         (* default: don't print cert, data layout 0, tag offset 0, thySynt 6, tactic flavour 4 *)
	   handle gdfError s => TextIO.print ("Error: " ^ s ^"\n")
	   handle Compile.gdfError s => TextIO.print ("Error: " ^ s ^"\n");

