(* Careful: this file is shared by Camelot and gdf via symbolic links.
   The master copy is progs/Grail/gdf/src/Compile.sml *)

(* Should be a little careful here:  if we have nonsensical types
   then they'll just go through to JVML and will cause a verifiaction
   error.  On the other hand,  we don't want to check all the types
   at compile time.  Maybe we should save a list of all of the reference
   types we've seen and (optionally) print out a list. *)

(* Similarly,  we can blatantly apply a method requiring a float to an int
   and not get an error till verification time *)

(* NOTE:  there's partial support for the full range of JVM types here.
   However lots more remains to be done,  so be careful. *)


val testing = ref false

open GrailAbsyn
open Bytecode
open Classdecl
open Jvmtype

val debug = ref false
val lax = ref false
(* If true then don't check scope for local function args;  useful 
   when we've got variable consolidation turned off *)

exception gdfError of string;

type ident = string;

fun fst (x,y) = x
fun snd (x,y) = y

fun member x [] = false
  | member x (h::t) = x=h orelse member x t


fun pr x = TextIO.print (x^" ")
fun prl x = TextIO.print (x^"\n")

fun pr x = ()
fun prl x = ()


(* Don't want messages mixed with syntax going to stdout *)
(*
fun printToStdErr s = 
    if not (!ToyGrailAbsyn.shut_up) then
	TextIO.output(TextIO.stdErr, s)
else ()
*)
fun printToStdErr s = TextIO.output(TextIO.stdErr, s)


(* --------------------- Info about current location in file -------------------- *)
(* We don't use textual locations since we won't have this for stuff obtained from Camelot *)

val currentClass = ref ""
fun setCurrentClass s = currentClass := s

val currentMethod = ref ""
fun setCurrentMethod s = currentMethod := s

val currentFunction = ref ""
fun setCurrentFunction s = currentFunction := s

fun here () = "[class " ^ !currentClass
	      ^ ", method " ^ !currentMethod
	      ^ ", function " ^ !currentFunction
	      ^ "]"

fun localError s = raise gdfError (s ^ "\n" ^ here())


(* ---------- Recording reference types for debugging purposes ---------- *)

val refTypes = ref (Splayset.empty String.compare)
fun recordRefType t = (refTypes := (Splayset.add(!refTypes, t)))
fun refTypeUsed t = Splayset.member (!refTypes, t)
fun reportRefTypes cname =
    let
	val () = print ("Reference types used in " ^ cname ^ ":\n")
	val reftypes = Splayset.listItems (!refTypes)
    in case reftypes of
	   [] => print "NONE\n"
	 | _ => app (fn t => print (" " ^ t ^ "\n")) reftypes
    end



(* ---------------- Fiddling with the boolean type ---------------- *)


(* The following stuff is to fix the bool/int problem.  The JVM does have
   a boolean type (Z in descriptors),  but it really just treats it as
   int.  If you get a boolean field then you can immediately add 77 to it
   without the verifier complaining;  booleans obtained as method return
   values are treated similarly. [There may be some special treatment for
   boolean arrays.]  The stuff below converts booleans to ints.  Whenever
   we declare (perhaps implicitly) a value of type boolean it gets treated
   as being of type int.  However,  boolean types are preserved in descriptors.
   I think this does the right thing;  it makes the Grail type system rather
   odd,  but it's consistent with the behaviour allowed by the JVM.  *)

(* FIX:  now I'm not so sure;  if a method declares a value of type boolean,
   what should its type be in local function declarations:  int or boolean? *)

(* Can the JVM even distinguish between local variables of type int and boolean? YES. *)

fun debool BOOLEANty = INTty (* Used when we create a var of type boolean *)
  | debool t = t
	
fun tysEqual x y = debool x = debool y

fun rtysEqual (SOME x) (SOME y) = tysEqual x y
  | rtysEqual NONE NONE = true
  | rtysEqual _ _ = false



(* ------------------------------- Subtyping --------------------------------- *)

(* At present we more or less ignore type safety for method invocation,
   field access &c.  The following code is a crude approximation to the
   checks required.  It will at least detect obvious things like giving
   a method a float where an int is required.   The most important
   missing check is for type subsumption for reference types.  We
   previously weren't able to do this because we couldn't look at
   classfiles,  but maybe the extended version of sml-jvm will enable 
   as to do this now (at least optionally;  we maybe don't want to check 
   this kind  of thing twice when we're compiling from Camelot).  Other things
   that need checking are the rules for arrays and byte and char types etc. *)

fun subClass t1 t2 = true
(* INSERT CORRECT SUBTYPE CHECK HERE *)

fun subtype t1 t2 =
    case (t1,t2) of
	    (BOOLEANty, BOOLEANty) => true
	  | (BOOLEANty, INTty) => true
	  | (INTty, BOOLEANty) => true   (* AAAARGH!!!! *)
	  | (REFty c1, REFty c2)  => subClass c1 c2
	  | (ARRAYty u1, ARRAYty u2) => subtype u1 u2
	  | _ => t1 = t2  (* CAREFUL if more types are added *)



(* ---------------- Conversions from Grail representations to sml-jvm ---------------- *)

fun tyToJty INTty = Tint
  | tyToJty BOOLEANty = Tboolean
  | tyToJty FLOATty = Tfloat
  | tyToJty (REFty x) = Tclass (qnameToJClass x)
  | tyToJty (ARRAYty ty) = Tarray (tyToJty ty)
  | tyToJty CHARty = Tchar
  | tyToJty BYTEty = Tbyte
  | tyToJty SHORTty = Tshort
  | tyToJty LONGty = Tlong
  | tyToJty DOUBLEty = Tdouble

and rtyToJty (SOME t) = SOME (tyToJty t)
  | rtyToJty NONE = NONE

and qnameToJClass name = (pr ("# class " ^ name ^ "\n");
    if name ="" then class{pkgs = [], name = ""} else
    let val l = String.tokens (fn c => (c = #".")) name
	val n = List.length l
	val _ = app (fn x => (pr (x ^ "\n"))) l
	val _ = pr ((Int.toString n) ^"\n")
	val pkgs = List.take (l,n-1)
	val name' = List.nth (l,n-1)
	val _ = prl "@ Classname"
	val _ = pr "Packages: ["
	val _ = app pr pkgs;
	val _ = prl "]"
	val _ = prl ("Class:" ^ name')
    in class {pkgs = pkgs, name = name'}
    end)


(* ---------------------------- Local variables ---------------------------- *)

fun lookup  v (ref l) = let fun
    lookup' v [] = NONE
  | lookup' v ((w,p)::tail) = if v=w then SOME p else lookup' v tail
in
    lookup' v l
end

fun find v h = case lookup v h of
    NONE => raise gdfError ("Can't find variable/function " ^ v)
  | SOME p => p

(* Local Variables and labels for function declarations and calls *)


local
    (* locals *)
    val localsref = ref Localvar.freshLocals : Localvar.locals ref

    fun newLocal1 () =
        let val (locals, loc) = Localvar.nextVar1 (!localsref)
        in localsref := locals; loc
	end

    val registers: (ident * (Ty * Localvar.index)) list ref
	= ref []  (* (name, (type, varpos)) *)

    (* labels *)
    val nextlab = ref Label.freshLabels : Label.labels ref
    val (_,cs) = Label.newLabel(!nextlab)   (* Horrible *)
    val (_,ce) = Label.newLabel(!nextlab)   (* Horrible *)
    val funLabels: (ident * (RTy * (Ty*string) list * Label.label)) list ref
	= ref []

in
    val codeStart = ref cs
    val codeEnd = ref cs;  (* Both of these are initial values which will never be used *)

    fun resetLocals () = (localsref := Localvar.freshLocals; registers := [])
    fun numLocals() = List.length (!registers)

    fun lookupVar v = lookup v registers
    fun findVar v = find v registers

    fun declareVar v ty =
	case lookupVar v of
	    SOME (t1,_) =>
		if tysEqual t1 ty then ()
		else localError ( "Trying to reuse variable "
				     ^ v ^ " (of type " ^ tyToString t1
				     ^ ") with new type " ^ tyToString ty)
	  | NONE =>
		let
		    val _ = pr ("Declaring " ^ v ^ ": " ^ (tyToString ty ) ^ "\n")
		    val () = case ty of
				 REFty t =>
				 let
				     val () = if t = "bool"
					      then print "WHOA! Type bool found: did you mean boolean?\n"
					      else ()
				 in
				     recordRefType t
				 end
			       | _ => ()
		    val loc = newLocal1()
		in
		    registers := (v, (debool ty, loc))::(!registers)
		end
	
    fun dumpLocals () =
	let
	    val l = !registers
	    fun dl [] n = ()
	      | dl ((name,_)::t) n =
		(BasicIO.print ((Int.toString n) ^ ": " ^ name ^"\n");
		 dl t (n-1))
	in dl l (List.length l - 1)
	end

    fun getLocals () =
	let
	    fun get' [] = []
	      | get'  ((id,(ty,loc))::t) =
		{from=(!codeStart), thru=(!codeEnd), name=id, ty=tyToJty(ty), index=loc}::get'(t)
	in
	    LOCALVAR(get' (!registers))
	end



(* ------------------------------ Labels ---------------------------------- *)

    fun newLabel () = (* We need this to be globally visible so that
		         we can generate new labels for "if ..." *)
        let val (labels, lab) = Label.newLabel (!nextlab)
        in nextlab := labels; lab
	end

    fun resetLabels () =
	(nextlab := Label.freshLabels;
	 codeStart := newLabel();
	 codeEnd := newLabel();
	 funLabels := [])
	
    fun lookupFun f = lookup f funLabels
    fun findFun f = find f funLabels
    fun getFunLabels () = rev (!funLabels)
	
    fun declareFun name ty params =
	case lookupFun name of
	    SOME _ => raise gdfError ( "Trying to redeclare function " ^ name )
	  | NONE => let val _ = pr("Declaring function " ^ name ^ "\n")
			val loc = newLabel()
		    in
			funLabels := (name, (ty, params, loc))::(!funLabels)
		    end

end




(* ----------------- Scope checking and type inference for variables ---------------------- *)

fun numeric t = 
    case t of 
	BYTEty   => true
      | SHORTty  => true
      | INTty    => true
      | LONGty   => true 
      | FLOATty  => true
      | DOUBLEty => true
      |  _ => false

fun logical oper = 
    case oper of
	ANDop => true
      | ORop  => true
      | XORop => true
      | SHLop => true
      | SHRop => true
      | USHRop => true
      | _ =>  false

fun typeOfVar v =
    case lookupVar v of
	NONE => raise gdfError ("Can't infer type of variable " ^ v)
      | SOME (t,_) => t

fun typeOfValue w = 
    case w of
	VARval v      => typeOfVar v
      | BYTEval _     => BYTEty
      | SHORTval _    => SHORTty
      | INTval _      => INTty
      | LONGval _     => LONGty
      | CHARval _     => CHARty
      | FLOATval _    => FLOATty
      | DOUBLEval _   => DOUBLEty
      | STRINGval _   => REFty "java.lang.String"
      | NULLval (t,_) => REFty t

fun typeOfPrimop p =
    case p of 
	VALop v => SOME(typeOfValue v)
      | BINop(oper,v,w) =>
	let val t1 = typeOfValue v
	    val t2 = typeOfValue w
	in
	    if t1 <> t2 then 
		localError "Arithmetic/logical operator applied to mixed types\n"
	    else if logical oper andalso t1 <> INTty then
		localError "Logical operator applied to non-integral type"
	    else if numeric t1 then SOME t1
	    else localError "Arithmetic operator applied to non-numeric types\n"
    end
  | NEWop (MDESC(_,s,_),_) =>               SOME(REFty s)
  | CHECKCASTop (s,_) =>                    SOME(REFty s)
  | INSTANCEop  _     =>                    SOME INTty
  | GETFIELDop (_,FDESC (ty,_)) =>          SOME ty
  | GETSTATICop (FDESC (ty,_)) =>           SOME ty
  | INVOKESTATICop (MDESC(rty,_,_),_)    => rty
  | INVOKEVIRTUALop (_,MDESC(rty,_,_),_) => rty
  | INVOKESPECIALop (_,MDESC(rty,_,_),_) => rty
  | INVOKEINTERFACEop (_,MDESC(rty,_,_),_) => rty
  | PUTFIELDop _  =>                        NONE
  | PUTSTATICop _ =>                        NONE
  | MAKEop (a,_) => raise gdfError "The make operation is not allowed yet"
                                  (* SOME(ARRAYty(typeOfValue(a))) *)
  | GETop (aa,_) =>
    let in
     case typeOfValue(aa) of
	 ARRAYty(ty) => SOME(ty)
       | _ => (raise gdfError ("First argument of get must be of array type"))
    end
  | SETop (aa,_,_) => NONE
  | LENGTHop _     => SOME(INTty)
  | EMPTYop(v,t)   => SOME(ARRAYty(t))
  | FTOIop _       => SOME(INTty)
  | ITOFop _       => SOME(FLOATty)

fun checkDec p vars =
    let (* Check vars in primop;  also do some type consistency checks for descriptors *)
	fun chkvar v =
	    if member v vars orelse (!lax) then ()
	    else raise gdfError ("Unknown variable " ^ v ^ " in " ^ !currentFunction )
	   (* There were problems here before we had minimal Grail argument
              lists: hopefully this is OK now. *)

	fun chkval w =
	    case w of (VARval v) => chkvar v
	  | _ => ()

	fun chkvlist [] = ()
	  | chkvlist (h::t) = (chkval h; chkvlist t)

	fun chkValTy v ty = (* check that v has type which is a subtype of ty *)
	    let val vty = typeOfValue v in
		if subtype vty ty then ()
		else print ("WARNING: " ^ here ()
			    ^ "\n" ^ valToString v ^ " has type " ^ tyToString vty
			    ^ ", but should have a type which is a subtype of "
			    ^ tyToString ty ^ "\n")
	    end

	fun chkArgTys [] [] = ()
	  | chkArgTys (v::args) (ty::tys) = (chkValTy v ty; chkArgTys args tys)
	  | chkArgTys _ _ = localError "Argument length mismatch"

	(* We check putfield operations and invocations for type-consistency;
           maybe other things should be checked as well *)


	fun chk p = case p of
	    VALop w => chkval w
	  | BINop(_,v,w) => (chkval v; chkval w)
	  | NEWop(MDESC(_,s,_),vs) => (chkvlist vs)
	  | CHECKCASTop (_,v) => chkvar v
	  | INSTANCEop  (_,v) => chkvar v
	  | INVOKESTATICop (MDESC(_,_,argtys) ,l) => (chkArgTys l argtys; chkvlist l)
	  | INVOKEVIRTUALop (v,MDESC(_,_,argtys),l) => (chkvar v; chkArgTys l argtys; chkvlist l)
          (* We don't check that the type of v is compatible with the class to which the
             method belongs.  This can lead to verification-time type errors.  The alternatives
	     are either to perform compile-time subtype-checking,  or to require the class type
             to match the type of the object exactly.  The latter might cause problems for 
             dynamic dispatch. *)
	  | INVOKESPECIALop (v,MDESC(_,_,argtys),l) => (chkvar v; chkArgTys l argtys; chkvlist l)
	  | INVOKEINTERFACEop (v,MDESC(_,_,argtys),l) => (chkvar v; chkArgTys l argtys; chkvlist l)
	  | GETFIELDop  (v,FDESC(ty,_)) => chkvar v
	  | PUTFIELDop (v,FDESC(ty,_),w) => (chkValTy w ty; chkvar v; chkval w)
	  | GETSTATICop (FDESC(ty,_)) => ()
	  | PUTSTATICop (FDESC(ty,_),v) => (chkValTy v ty; chkval v)
          | MAKEop (v,i) => (chkval v; chkval i)
          | GETop (a,i) => (chkval a; chkval i)
          | SETop (a,i,v) => (chkval a; chkval i; chkval v)
          | LENGTHop a => chkval a
          | EMPTYop (v, _) => chkval v
          | ITOFop i => chkval i
          | FTOIop f => chkval f
    in
	chk p
    end


fun declareLetDecs [] vars = vars
  | declareLetDecs (h::t) vars =
    let in case h of VOIDdec(p) =>
	 (
	  checkDec p vars;
	  case typeOfPrimop p of
	      NONE => declareLetDecs t vars
	    | _ => raise gdfError ("Non-unit value assigned to () in " ^ !currentFunction )
	 )
   | VALdec (v,p) =>
	 (
	  checkDec p vars;
	  case typeOfPrimop p of
	      NONE => raise gdfError
		  ("You're trying to assign a void value to the variable " ^ v ^ " in " ^ (!currentFunction))
	    | SOME ty =>  (declareVar v ty; declareLetDecs t (v::vars))
	 )
    end
	

fun chkArg vars f v = 
    if member v vars then ()
    else localError ("Function " ^ f
		     ^ " applied to unknown variable " ^ v)

fun chkPrimres vars r =
    if (!lax) then ()
    (* If we have the -n Camelot flag set then we get dodgy arglists;
       this lets them through *)
    else case r of VOIDres => ()
		| OPres p => checkDec p vars
		| FUNres (f, args) => app (chkArg vars f) args
	
fun chkCaseRes vars low high cases =
    let fun chk l q = 
	    case l of [] => q
		    | (n, f, args)::t =>  
		      if n<>q then raise gdfError ("bad key (" 
						    ^ Int.toString n 
						    ^ ") in case statement")
		      else (app (chkArg vars f) args; chk t (q+1))
    in
	if 
	    high < low 
	then raise gdfError "Invalid interval in case statement"
	else 
	    if chk cases low <> high+1 then raise gdfError "Wrong number of cases in case statement"
	    else ()
    end


fun declareParams params = app (fn (t,v) => declareVar v t) params

fun declareFunDec (FDEC (fname, params, FUNbody (lets, result))) =
    let
	val () = setCurrentFunction fname
	val () = declareParams params
	val vars = declareLetDecs lets (map snd params)
    in
	case result of
	    PRIMres r => chkPrimres vars r
	  | CHOICEres (_,_,_, r1, r2) => (chkPrimres vars r1; chkPrimres vars r2)
	  | CASEres (_,low, high, cases) => chkCaseRes vars low high cases
    end

fun declareMdefVars flags params (MBODY(letdecs, fundecs, result)) =
    let
	val () = setCurrentFunction "<initial letdecs>";
        val params =
	    if member M_ACCstatic flags then params
	    else (REFty(!currentClass), "this")::params
		 (* instance method - fake declaration of "this" *)
	val () = declareParams params
	val vars = declareLetDecs letdecs (map snd params)
	val () = app declareFunDec fundecs
    in
	case result of
	    PRIMres r => chkPrimres vars r
	  | CHOICEres (_,_,_, r1, r2) => (chkPrimres vars r1; chkPrimres vars r2)
	  | CASEres (_, low, high, cases) => chkCaseRes vars low high cases
    end

fun declareFuns methty mbody =
    let val MBODY(_, fundecs,_) = mbody in
	app (fn FDEC(name, params, _)
	     => declareFun name methty params)
	fundecs
    end


(* ------------------ Check that types are used consistently ------------------------ *)


fun typeCheckFunCall fname params =
    let
	val (_,formalParams,_) = findFun fname
	val formalNames = map snd formalParams
    in
	if params <> formalNames then
	    raise gdfError ("Function " ^ fname ^ " is called with arguments " ^
			      vecToString id params ^
			      ", but requires arguments " ^ vecToString id formalNames)

(*	else if map typeOfValue params <> map fst formalParams then
	    raise gdfError ("INTERNAL ERROR: type mismatch in function call") *)
(* Doesn't work, but shouldn't happen *)
	else ()
    end
	


fun typeOfFunCall fname args = 
    (
     typeCheckFunCall fname args;
     case (lookupFun fname) of
	 NONE => (raise gdfError ("Type error: can't find function " ^ fname))
       | SOME(t, n, p) => t
    )

fun typeOfPrimRes (OPres p) = typeOfPrimop p
  | typeOfPrimRes VOIDres = NONE
  | typeOfPrimRes (FUNres (fname, params)) = typeOfFunCall fname params

fun typeOfResult (PRIMres p) = typeOfPrimRes p
  | typeOfResult (CHOICEres (v,tst,w, p1,p2)) =
    let val t = typeOfValue v
	val t' = typeOfValue w
	val () = if tysEqual t t' then ()
		 else raise gdfError ("Test arguments have different types (" ^
				      tyToString t ^ " and " ^ tyToString t' ^")")
	val () = case tst of
	    EQtest => ()
	  | NEtest => ()
	  | _ => if not (numeric t)
		     then
			 raise gdfError "Numerical test applied to non-numerical values"
		 else ()
	val t1 = typeOfPrimRes p1
	val t2 = typeOfPrimRes p2			

    in
	if rtysEqual t1 t2 then t1
	else raise gdfError ("Test branches have different types")
    end
  | typeOfResult (CASEres (v, low, high, cases)) =
    let in case cases of 
	       [] => raise gdfError "No cases in case statement"
	     | ((_, fname, args)::rest) => 
	       let 
		   val t0 = typeOfFunCall fname args
		   fun ok (_,f,a) = 
		       let val t1 = typeOfFunCall f a 
		       in if t1 = t0 then ()
			  else raise gdfError "cases have incomaptible return types"
		       end 
		   val () = app ok rest
	       in
		   t0
	       end
    end
    
fun typeCheckLetDec fname (VOIDdec p) =
    let val t = typeOfPrimop p in
    case t of
	NONE => ()
      | SOME ty =>
	raise gdfError ("In function " ^ fname ^ ", attempted to assign value of type " ^
			  tyToString ty ^ " to ()")
    end
  | typeCheckLetDec fname (VALdec (v, p)) =
    let
	val lty = typeOfVar v
	val ty = typeOfPrimop p (* Shouldn't be NONE by this stage *)
    in
	if rtysEqual (SOME lty) ty then ()
	else
	    raise gdfError ("In function " ^ fname ^
			    ", attempted to assign value of type "
			    ^ rtyToString(ty)
			    ^ " to variable " ^ v
			    ^ " of type " ^ tyToString lty)
    end


fun typeCheckReturnType expected fname actual =
    if rtysEqual actual expected then ()
    else raise gdfError("Function " ^ fname
 			^ " should return " ^ rtyToString(expected)
			^ ", but actually returns " ^ rtyToString(actual))

fun typeCheckFunBody rtype fname (FUNbody(lets, r)) =
       (app (typeCheckLetDec fname) lets;
	     typeCheckReturnType rtype fname (typeOfResult r) )

fun typeCheckFunDec rtype (FDEC(fname,_,fbody)) = typeCheckFunBody rtype fname fbody

fun typeCheckMbody rtype (MBODY(_, fundecs, result)) =
    let
	val actualrtype = typeOfResult result
    in
	if rtysEqual actualrtype rtype
	then app (typeCheckFunDec rtype) fundecs
	else raise gdfError ("Method should return " ^ rtyToString rtype ^
			     ", but actually returns " ^ rtyToString actualrtype)
    end



(* -------------------------------- Bytecode emission -------------------------------- *)


fun mdescToMethodref (MDESC(ty, name, argtypes)) =
    let val _ = prl ("## method " ^ name )
	val l = String.tokens (fn c => (c = #".")) name
	val n = List.length l
	val msig = (map tyToJty argtypes, rtyToJty ty)
	val name = List.nth(l,n-1)
	val (pkgs, cname) =
	    if n=1 then
		(TextIO.print ("WARNING: suspect method descriptor (" ^ name ^ ")\n");
		 ([],"")
		 )
	    else (List.take(l,n-2), List.nth(l,n-2))
	val _ = prl "@ Methodref"
	val _ = pr "Packages: ["
	val _ = app pr pkgs
	val _ = prl "]"
	val _ = prl ("Class: " ^ cname)
	val _ = prl ("Name: " ^ name)
    in
	{name=name, msig=msig, class = Jvmtype.class{pkgs=pkgs, name=cname}}
    end


fun fdescToFieldRef (FDESC(fty,name)) =
  let val l = String.tokens (fn c => (c = #".")) name
	val n = List.length l
	val ty = tyToJty fty
	val name = List.nth(l,n-1)
	val (pkgs, cname) =
	    if n=1 then
		(TextIO.print ("WARNING: suspect field descriptor (" ^ name ^ ")\n");
		 ([],"")
		 )
	    else (List.take(l,n-2), List.nth(l,n-2))
	(* val _ =BasicIO.print ("cname = "^cname^", field = "^ name ^"\n") *)
  in
      {name=name, ty=ty, class = class{pkgs=pkgs, name=cname}}
  end


fun loadVar v A = (
   case lookupVar v  of   (* should really use findVar here *)
     NONE => raise gdfError ("Eh? can't find variable " ^ v)
   | SOME (t,l) =>
     let in 
	 case t of
	     BOOLEANty => (Jiload l) :: A
           (* Probably never happens: t will have been converted to INTty
              by the call to typeOfPrimop in declareVar *)
           | BYTEty => raise gdfError "Unexpected BYTEty in loadVar"
	   | SHORTty => raise gdfError "Unexpected SHORTty in loadVar"
           | INTty => (Jiload l) :: A
	   | LONGty => raise gdfError "Unexpected LONGty in loadVar"
	   | CHARty => raise gdfError "Unexpected CHARty in loadVar"
	   | FLOATty => (Jfload l) :: A
	   | DOUBLEty => raise gdfError "Unexpected DOUBLEty in loadVar"
	   | REFty _ => (Jaload l) :: A
	   | ARRAYty _ => (Jaload l) :: A
     end
)


fun storeVar v A = (
   case lookupVar v of   (* should really use findVar here *)
     NONE => raise gdfError ("Eh? can't find variable " ^ v)
   | SOME (t,l) =>
     let in 
	 case t of
             BOOLEANty => (Jistore l) :: A
	   | BYTEty => raise gdfError "Unexpected BYTEty in storeVar"
	   | SHORTty => raise gdfError "Unexpected SHORTty in storeVar"
	   | INTty => (Jistore l) :: A
	   | LONGty => raise gdfError "Unexpected LONGty in storeVar"
	   | CHARty => raise gdfError "Unexpected CHARty in storeVar"
           | FLOATty => (Jfstore l) :: A
	   | DOUBLEty => raise gdfError "Unexpected DOUBLEty in storeVar"
	   | REFty _ => (Jastore l) :: A
	   | ARRAYty _ => (Jastore l) :: A
     end
)

local 
    fun loadInt n A =   (Jiconst (Int32.fromInt n)) :: A
    fun loadFloat r A = (Jfconst (Real32.fromReal r)) :: A
in
fun loadVal w A =
    case w of VARval v    => loadVar v A
	    | BYTEval b   => loadInt b A
	    | SHORTval s  => loadInt s A
	    | INTval i    => loadInt i A  (* FIX: May be too sort *)
	    | LONGval l   => loadInt l A  (* FIX: May be too sort *)
	    | CHARval c   => loadInt c A
	    | FLOATval r  => loadFloat r A
	    | DOUBLEval d => loadFloat d A (* FIX: May be too sort *)
	    | STRINGval s =>  (* quotes should have been removed by now *)
	      let in 
		  case String.fromString s of 
		      SOME x => (Jsconst x) :: A
		    | NONE => raise gdfError ("Error in string literal \"" ^ s ^ "\"")
	      end
	    | NULLval (_,t) => (* Save type marker for conversion into metadata *)
	      let 
		  val () = Metadata.saveMarker t
	      in 
		  (Jaconst_null) :: A
	      end
end
			       
fun loadArgs [] A = A
  | loadArgs (h::t) A = loadVar h (loadArgs t A)

fun loadArgVals [] A = A
  | loadArgVals (h::t) A = loadVal h (loadArgVals t A)

fun storeArgs l A =
let
    fun storeArgs' [] A = A
      | storeArgs' ((ty,name)::t) A = storeVar name (storeArgs' t A)
in
    storeArgs' (List.rev l) A
end


(* FIX: needs lots more stuff for other numeric types *)
fun compilePrimOp p A = 
    case p of
	VALop v => loadVal v A
      | BINop (oper, a, b) =>
	if typeOfValue a = INTty then
	    let 
		val opcode = 
		    case oper of 
			ADDop => Jiadd
		      | SUBop => Jisub
		      | MULop => Jimul
		      | DIVop => Jidiv
		      | MODop => Jirem
		      | ANDop => Jiand
		      | ORop  => Jior
		      | XORop => Jixor
		      | SHLop => Jishl
		      | SHRop => Jishr
		      | USHRop => Jiushr
	    in
		loadVal a (loadVal b (opcode:: A))
	    end
	else
	    let 
		val opcode = 
		    case oper of 
			ADDop => Jfadd
		      | SUBop => Jfsub
		      | MULop => Jfmul
		      | DIVop => Jfdiv
		      | MODop => Jfrem
		      | ANDop => Jiand
		      | ORop  => raise gdfError "ORop not available for floats"
		      | XORop => raise gdfError "XORop not available for floats"
		      | SHLop => raise gdfError "SHLop not available for floats"
		      | SHRop => raise gdfError "SHRop not available for floats"
		      | USHRop => raise gdfError "USHRop not available for floats"
	    in
		loadVal a (loadVal b (opcode:: A))
	    end
      | NEWop(MDESC(r,s,tl), vs) =>
	(Jnew (qnameToJClass s)) :: Jdup :: (loadArgVals vs
		(Jinvokespecial (mdescToMethodref (MDESC(r, s ^ ".<init>", tl))) :: A))
      | CHECKCASTop (s,v) =>
	loadVar v  ((Jcheckcast(CLASS (qnameToJClass s))) :: A)
      | INSTANCEop  (s,v) =>
        loadVar v (Jinstanceof(CLASS (qnameToJClass s)) :: A)
      | INVOKESTATICop (md, args) =>
        loadArgVals(args) (Jinvokestatic (mdescToMethodref md) :: A)
      | INVOKEVIRTUALop (inst, md, args) =>
	let val (_,loc) = findVar inst in
            (Jaload loc) :: (loadArgVals args (Jinvokevirtual (mdescToMethodref md) :: A))
	end
      | INVOKESPECIALop (inst, md, args) =>
	let val (_,loc) = findVar inst in
            (Jaload loc) :: (loadArgVals args (Jinvokespecial (mdescToMethodref md) :: A))
	end
      | INVOKEINTERFACEop (inst, md, args) =>
	let val (_,loc) = findVar inst in
            (Jaload loc) :: (loadArgVals args (Jinvokeinterface (mdescToMethodref md) :: A))
	end
      | GETFIELDop (inst ,fd) =>
	let val (_,loc) = findVar inst in
	    (Jaload loc) :: (Jgetfield (fdescToFieldRef fd) :: A)
	end
      | PUTFIELDop (inst, fd, input) =>
	let val (_,loc) = findVar inst in
	    (Jaload loc) :: (loadVal input (Jputfield (fdescToFieldRef fd) :: A))
	end
      | GETSTATICop fd => (Jgetstatic (fdescToFieldRef fd))::A
      | PUTSTATICop (fd, input) =>
	loadVal input (Jputstatic (fdescToFieldRef fd) :: A)
      | GETop(aa,i) =>
	(
	 case typeOfValue aa of
	     ARRAYty(INTty)     => loadVal aa (loadVal i (Jiaload :: A))
	   | ARRAYty(FLOATty)   => loadVal aa (loadVal i (Jfaload :: A))
	   | ARRAYty(REFty _)   => loadVal aa (loadVal i (Jaaload :: A))
	   | ARRAYty(ARRAYty _) => loadVal aa (loadVal i (Jaaload :: A))
	   | _ => (raise gdfError("First argument of get must be an array type"))
	)
      | SETop(aa,i,v) => (* Changed to return no value = no dup *)
	let in 
	    case typeOfValue aa of
		ARRAYty(INTty)     => loadVal aa (loadVal i (loadVal v (Jiastore :: A)))
	      | ARRAYty(FLOATty)   => loadVal aa (loadVal i (loadVal v (Jfastore :: A)))
	      | ARRAYty(REFty _)   => loadVal aa (loadVal i (loadVal v (Jaastore :: A)))
	      | ARRAYty(ARRAYty _) => loadVal aa (loadVal i (loadVal v (Jaastore :: A)))
	      | _ => raise gdfError("First argument of set must be an array type")
	end
      | LENGTHop(aa) => loadVal aa (Jarraylength :: A)
      | EMPTYop(i,ty) =>
        loadVal i ( (Jnewarray { elem = tyToJty ty, dim = 1 }) :: A)
      | MAKEop(v,i) =>
	loadVal v  ((Jnewarray { elem = tyToJty (typeOfValue v), dim = 1 }) :: A)
      | FTOIop f => loadVal f (Jf2i :: A)
      | ITOFop i => loadVal i (Ji2f :: A)


fun compileLetDec (VALdec (v,p)) A = compilePrimOp p (storeVar v A)
  | compileLetDec (VOIDdec p) A = compilePrimOp p A

fun compileLetDecs letdecs A =
    let fun comp [] A = A
	  | comp (h::t) A = compileLetDec h (comp t A)
    in
	comp letdecs A
end
(* This could be more tail-recursive; we don't usually have thousands
   of letDecs,  so it's probably fairly safe. *)


fun compilePrimRes (OPres p) A = compilePrimOp p (Jreturn :: A)
  | compilePrimRes VOIDres  A = Jreturn :: A
  | compilePrimRes (FUNres (fname, params)) A =
    let val (_,_,dest) = findFun fname
    in
	(Jgoto dest) :: A
    end

(* FIX: needs lots more stuff for other numeric types *)
fun compileTest INTty EQtest dest A = (Jif_icmpeq dest) :: A         (* EQtest *)
  | compileTest BOOLEANty EQtest dest A = (Jif_icmpeq dest) :: A
  | compileTest FLOATty EQtest dest A =
    Jfcmpl :: (Jifeq dest) :: A
    (* fcmpl & gcmpl only differ on NaN *)
  | compileTest _ EQtest dest A = (Jif_acmpeq dest) :: A

  | compileTest INTty NEtest dest A = (Jif_icmpne dest) :: A         (* NEtest *)
  | compileTest BOOLEANty NEtest dest A = (Jif_icmpne dest) :: A
  | compileTest FLOATty NEtest dest A =
    Jfcmpl :: (Jifne dest) :: A
    (* fcmpl & gcmpl only differ on NaN *)
  | compileTest _ NEtest dest A = (Jif_acmpne dest) :: A

  | compileTest INTty Ltest dest A =   (Jif_icmplt dest) :: A        (* Ltest *)
  | compileTest FLOATty Ltest dest A = Jfcmpl :: (Jiflt dest) :: A
  | compileTest t Ltest _ _ =
    raise gdfError("< may only be applied to numerical types (got " ^ tyToString t ^")")

  | compileTest INTty LEtest dest A =   (Jif_icmple dest) :: A       (* LEtest *)
  | compileTest FLOATty LEtest dest A = Jfcmpl :: (Jifle dest) :: A
  | compileTest _ LEtest _ _ =
    raise gdfError("<= may only be applied to numerical types")

  | compileTest INTty Gtest dest A =   (Jif_icmpgt dest) :: A        (* Gtest *)
  | compileTest FLOATty Gtest dest A = Jfcmpl :: (Jifgt dest) :: A
  | compileTest _ Gtest _ _ =
    raise gdfError("> may only be applied to numerical types")

  | compileTest INTty GEtest dest A =   (Jif_icmpge dest) :: A       (* GEtest *)
  | compileTest FLOATty GEtest dest A = Jfcmpl :: (Jifge dest) :: A
  | compileTest _ GEtest _ _ =
    raise gdfError(">= may only be applied to numerical types")

fun compileCaseRes (low, high, cases) A = 
    let 
	fun doCases c acc = 
	    case c of 
		[] => raise gdfError "Empty case list"
	      | [(_,fname,_)] => let 
		    val (_,_,dest) = findFun fname
		in 
		    (rev (dest::acc), dest) 
		end
	      | (_,fname,_)::t => let 
		    val (_,_,dest) = findFun fname
		in 
		    doCases t (dest::acc)
		end

	val (labels, default) = doCases cases []
    in
	Jtableswitch {default=default, 
		      offset = Int32.fromInt low, 
		      targets=Vector.fromList labels} :: A
    end


fun compileResult (PRIMres p) A = compilePrimRes p A
       | compileResult (CHOICEres (v, test, w, p1, p2)) A =
	     let val dest1 = newLabel()
	     in
		 loadVal v (
	          loadVal w (
                   compileTest (typeOfValue v) test dest1 (
		    compilePrimRes p2 ((Jlabel dest1) :: (compilePrimRes p1 A)))))
	     end
       | compileResult (CASEres (v, low, high, cases)) A =
	 loadVar v (compileCaseRes (low, high, cases) A)
	  
			
fun compileFunBody (FUNbody(letdecs, result)) A
		       = compileLetDecs letdecs (compileResult result A)

(* Now we're compiling a fun declaration.  At this point,
   the name of the function and its local variables have already
   been declared *)

fun compileFunDec (FDEC(fname, args, body)) A =
    let
	val (_,_,entry) = findFun fname
    in
	Jlabel(entry) :: (compileFunBody body A)
    end

fun compileFunDecs l A =
    let fun
	compileFuns [] A = A
      | compileFuns (h::t) A = compileFunDec h (compileFuns t A)
    in compileFuns l A end
(* Again,  could be more tail-recursive.  If this gets changed, watch out 
   that it doesn't mess up the order of the metadata null-markers. *)

fun compileMethodBody (MBODY(letdecs,fundecs,result)) =
	Jlabel(!codeStart) :: (
         compileLetDecs letdecs (
	  compileResult result (
	   compileFunDecs fundecs [Jlabel(!codeEnd)])))
		
fun compile_mdef (MDEF(flags, rettype, mname, params, mbody)) =
    let
	val () = setCurrentMethod mname
	val () = resetLocals ()
	val () = resetLabels ()
	val () = Metadata.resetMarkers ()  (* Camelot type markers for null values *)

	val () = declareMdefVars flags params mbody
	val () = declareFuns rettype mbody
	val () = typeCheckMbody rettype mbody
	val () = 
	    if !testing then 
		let
(*		    val f = FlowGraph.flowGraph mbody
		    val i = FlowGraph.dominators f
		    val () = FlowGraph.printGraph f
		    val ()  = FlowGraph.pidom i*)
		    val () = print ("(*---- " ^ mname ^" ----*)\n")
		    val () = FlowGraph.makeDefs mbody
		in
		    ()
		end
	    else ()
	val msig =(map (fn (t,v) => tyToJty t) params, rtyToJty rettype)
	val code = compileMethodBody mbody
	val attrs = [CODE{stack=Stackdepth.maxdepth code [],
			  locals=numLocals(),
			  code=code, hdls=[],
			  attrs=[getLocals()]}, 
		     Metadata.makeFunInfo findVar (getFunLabels()), 
		     Metadata.makeNullInfo()
		    ]
(*	val _ = TextIO.print (Int.toString(numLocals ()) ^ " local variables\n");*)
(*	val _ = dumpLocals();*)

    in
	{flags=flags, name=mname, msig=msig, attrs=attrs}
    end;


(* ---------------- Some stuff about initialisers and superclasses ---------------- *)

fun defInit super =  {class=super,
				name="<init>",
				msig=([]: jtype list, NONE: jtype option)}
			
fun init super = {flags = [M_ACCpublic],
		  name="<init>",
		  msig = ([]: jtype list, NONE: jtype option),
		  attrs = [CODE{stack=1, locals=1,
			  code=[Jaload (Localvar.fromInt 0),
				Jinvokespecial (defInit super), Jreturn],
				hdls=[]: exn_hdl list,
				attrs=[]: attribute list }] }

(* The type annotations above are there to prevent warnings
   about value polymorphism *)

(* Add <init> if there is no <init> already *)
fun addInit meths super =
    if List.exists isInit meths then meths
    else ((init super)::meths)
and isInit {name, ...} = name = "<init>"


(* -------------------------------- Main function -------------------------------- *)

infix 5 ::?
fun x::?l = case x of NONE => l | SOME y => y::l

fun compile filename cdef outputdir (printCert,dataLayout,tagOffset,thySyntax,tFlavour) richTypes =
(* outputdir = name of directory in which to write class files;  must end with "/" *)
let
    fun basename f = List.last (String.tokens (fn c => ( c = #"/")) f)
    val javaObject = SOME (class{pkgs=["java","lang"], name="Object"})
    val CDEF(flags, cname, super, interfaces, fdefs, mdefs, layout) = cdef

    val () = setCurrentClass cname
    val () = Metadata.resetTypes ()  (* map from type names to integers for metadata *)

    (* optionally, also generate a .thy file out of the grail code *)
    val generateThy = not (thySyntax=0)

    val () = if generateThy then
               let
	         val fname = GrailUtils.makeFullFilename outputdir cname "thy"
                 val thy = TextIO.openOut fname
	         val () = ToyGrailAbsyn.printGrailPROG thy cdef (printCert,dataLayout,tagOffset,thySyntax,tFlavour) 
	         val () = TextIO.closeOut thy
                 val () = printToStdErr ("Wrote "^fname^"\n")
	       in
		   () (* YUCK *)
	       end
	     else ()

    val superclass =
	case super of
	    NONE => javaObject
	  | SOME c => SOME (qnameToJClass c)
		
    val fields = map (fn FDEF(flags,ty,name)
		     => {flags=flags, name=name,
			 ty=tyToJty(ty), attrs=[]}) fdefs
		
		 (* What about initialising constant values? *)
	
    val cm = map compile_mdef mdefs

    val () = if (!debug) then reportRefTypes cname else ()

    val cl: class_decl = { major = Classdecl.major, minor = Classdecl.minor,
	flags = C_ACCsuper::C_ACCpublic::flags,
	this = class{pkgs=[], name=cname},
	super = superclass,
	ifcs = map qnameToJClass interfaces,
	fdecls = fields,
	mdecls = addInit cm (valOf superclass),
			   attrs=(Metadata.makeLayoutInfo layout)::?
				 [SRCFILE(GrailUtils.basename filename), 
				  Metadata.makeTypeInfo()]
			   };

    val cp = Constpool.create();

    (* val _ = TextIO.print ("Trying to write JVM file to " ^ (GrailUtils.makeFullFilename outputdir cname  "class")) *)

    val os = BinIO.openOut (GrailUtils.makeFullFilename outputdir cname  "class")
    fun emitword w = BinIO.output1 (os, w);
    val () = Classfile.emit emitword cp cl
    val () = BinIO.closeOut os

    val _ = if printCert
              then CertGenP.makeCert1 cname mdefs outputdir richTypes (printCert,dataLayout,tagOffset,thySyntax,tFlavour) 
              else () (* TextIO.print "Omitting certgen, as requested\n" *)
in
    cname
end

