Require StoreLemmas.
Require cgAbSyn.
Require Arith.
Require LtFacts.
Require EqNat.
Require Relations.
Require Zmisc.

Section EvalRules.

Hypothesis mt : MethTable.

(* The global state of the virtual machine *)
Record globalstate : Set := mkGState {
  ticks : nat;
  heap : nat
 }.

Definition addtick : globalstate -> globalstate := [g](mkGState (S (ticks g)) (heap g)).
Definition addticks : nat -> globalstate -> globalstate := [n,gs](mkGState (plus n (ticks gs)) (heap gs)).
Definition diffticks := [gs,gs':globalstate](minus (ticks gs') (ticks gs)).

Definition initstate := (mkGState O).

Definition resultprop := globalstate->nat->Prop.
Definition resultis : (nat->Prop)->resultprop := [p,g,n](p n).

(*************************************************************)
(* Some auxilary definitions *)
Definition semBinOp : synBinOp -> nat -> nat -> nat :=
  [op;n1;n2]
  Cases op of synEq    => if (beq_nat n1 n2) then (1) else (0)
            | synLt    => if (blt_nat n1 n2) then (1) else (0)
            | synPlus  => (plus n1 n2)
            | synMinus => (minus n1 n2)
            | synTimes => (mult n1 n2)
           end.

Definition get := (find Var nat beq_Var).
Definition evalVal : store->synVal->(option nat) :=
 [s;v]Cases v of (const c) => (Some nat c)
               | (var x)   => (get s x)
              end.

(*************************************************************)
(* Constructing a store from an argument list and an old store *)
Fixpoint mkStore [s:store;c:nat;args:(list synVal)] : (option store) :=
  Cases args of
    nil            => (Some ? emptyStore)
  | (cons v args') => Cases (evalVal s v) (mkStore s (S c) args') of
                        None     _         => (None store)
                      | _        None      => (None store)
                      | (Some n) (Some s') => (Some ? (cons (mkVar c, n) s'))
                      end
   end.

(*************************************************************)
(* Definition of the single step operational semantics *)
Inductive evalExp : store->(globalstate * synExp)->(globalstate * synExp)->Prop :=
  evalVar   : (s:store;x:Var;n:nat;gs:globalstate)
              (Instore s x n)->
              (evalExp s (gs, value (var x)) (addticks (1) gs, value (const n)))
| evalBinop : (s:store;v1,v2:synVal;n1,n2:nat;op:synBinOp;gs:globalstate)
              (evalVal s v1)=(Some nat n1) -> (evalVal s v2)=(Some nat n2) ->
              (evalExp s (gs, binop op v1 v2) (addticks (2) gs, value (const (semBinOp op n1 n2))))
| evalInvoke: (s,s':store;meth:method;mname:Mname;args:(list synVal);gs:globalstate)
              (InMethTable mt mname meth)->
              (mkStore s (0) args)=(Some store s')->
              (evalExp s (gs, invoke mname args) (addticks (1) gs, execute (Fst meth) s' (Snd meth)))
| evalExecuteFinal: (s,s':store;v:synVal;bbt:BBTable;n:nat;gs:globalstate)
                    (evalVal s' v)=(Some nat n)->
                    (evalExp s (gs, execute bbt s' (return v)) (gs, value (const n)))
| evalExecuteStep: (s,s',s'':store;stm',stm'':synStm;bbt:BBTable;gs,gs':globalstate)
                   (evalStm bbt ((s',stm'),gs) ((s'',stm''),gs'))->
                   (evalExp s (gs, execute bbt s' stm') (gs', execute bbt s'' stm''))

with evalStm : BBTable -> (store * synStm) * globalstate -> (store * synStm) * globalstate -> Prop :=
  evalGoto   : (bt:BBTable;lbl:BBname;target:synStm;s:store;gs:globalstate)
               (InBBTable bt lbl target)->
               (evalStm bt ((s, goto lbl), gs) ((s, target), addticks (1) gs))
| evalIf0    : (bt:BBTable;s:store;lbl1,lbl2:BBname;target:synStm;gs:globalstate)
               (InBBTable bt lbl2 target)->
               (evalStm bt ((s, cond (value (const (0))) lbl1 lbl2), gs) ((s, target), addticks (1) gs))
| evalIf1    : (bt:BBTable;s:store;lbl1,lbl2:BBname;target:synStm;gs:globalstate)
               (InBBTable bt lbl1 target)->
               (evalStm bt ((s, cond (value (const (1))) lbl1 lbl2), gs) ((s, target), addticks (1) gs))
| evalIf     : (bt:BBTable;s:store;e,e':synExp;lbl1,lbl2:BBname;gs,gs':globalstate)
               (evalExp s (gs, e) (gs', e'))->
               (evalStm bt ((s, cond e lbl1 lbl2), gs) ((s, cond e' lbl1 lbl2), gs'))
| evalAssign': (bt:BBTable;s:store;e,e':synExp;x:Var;cont:synStm;gs,gs':globalstate)
               (evalExp s (gs, e) (gs', e'))->
               (evalStm bt ((s, assign x e cont), gs) ((s, assign x e' cont), gs'))
| evalAssign : (bt:BBTable;s:store;n:nat;x:Var;cont:synStm;gs:globalstate)
               (evalStm bt ((s, assign x (value (const n)) cont), gs) ((update s x n, cont), addticks (1) gs)).

(*************************************************************)
(* Definition of a whole method to the final state satisfying a predicate *)
Inductive evalMethod : method -> globalstate -> store -> resultprop -> Prop :=
  evalFinal : (bt:BBTable;s:store;n:nat;gs:globalstate;pred:resultprop;v:synVal)
              (evalVal s v)=(Some nat n) -> (pred gs n) ->
              (evalMethod (bt, return v) gs s pred)
| evalStep  : (bt:BBTable;bl,bl':synStm;s,s':store;pred:resultprop;gs,gs':globalstate)
              (evalStm bt ((s,bl),gs) ((s',bl'),gs'))->
              (evalMethod (bt, bl') gs' s' pred)->
              (evalMethod (bt, bl) gs s pred).

Inductive evalExpBig : globalstate -> store -> synExp -> nat -> globalstate -> Prop :=
  evalExpFinal : (gs:globalstate;s:store;n:nat)(evalExpBig gs s (value (const n)) n gs)
| evalExpStep  : (gs,gs',gs'':globalstate;s:store;n:nat;e,e':synExp)
                 (evalExp s (gs, e) (gs', e'))->(evalExpBig gs' s e' n gs'')->(evalExpBig gs s e n gs'').

Inductive evalExpBigPred : globalstate -> store -> synExp -> resultprop -> Prop :=
  evalExpPredFinal : (gs:globalstate;s:store;pred:resultprop;n:nat)(pred gs n)->(evalExpBigPred gs s (value (const n)) pred)
| evalExpPredStep  : (gs,gs':globalstate;s:store;pred:resultprop;e,e':synExp)
                     (evalExp s (gs, e) (gs', e'))->(evalExpBigPred gs' s e' pred)->(evalExpBigPred gs s e pred).

End EvalRules.

Section EvalLemmas.

Hypothesis mt : MethTable.

(* Some useful lemmas *)
Lemma evalBinop' : (gs:globalstate;s:store;v1,v2:synVal;n1,n2,r:nat;op:synBinOp)
                   r=(semBinOp op n1 n2) ->
                   (evalVal s v1)=(Some nat n1) -> (evalVal s v2)=(Some nat n2) ->
                   (evalExpBig mt gs s (binop op v1 v2) r (addticks (2) gs)).
Intros. EApply evalExpStep.
Apply evalBinop. Apply H0. Apply H1.
Rewrite H. Apply evalExpFinal.
Save.

Lemma evalStepGoto : (pred:resultprop;s:store;blocks:BBTable;target:synStm;lbl:BBname;gs:globalstate)
                     (InBBTable blocks lbl target)->
                     (evalMethod mt (blocks, target) (addticks (1) gs) s pred)->
		     (evalMethod mt (blocks, goto lbl) gs s pred).
Intros. EApply evalStep. Apply evalGoto. Apply H. Apply H0.
Save.

Lemma evalStepsIf : (pred:resultprop;s:store;blocks:BBTable;e:synExp;lbl1,lbl2:BBname;n:nat;gs,gs':globalstate)
  (evalExpBig mt gs s e n gs')->
  (evalMethod mt (blocks, cond (value (const n)) lbl1 lbl2) gs' s pred)->
  (evalMethod mt (blocks, cond e lbl1 lbl2) gs s pred).
Intros.
Induction H.
Assumption.
EApply evalStep.
Apply evalIf.
Apply H.
Apply HrecH.
Apply H0.
Save.

Lemma evalStepIf1 : (pred:resultprop;s:store;blocks:BBTable;test:synExp;lbl1,lbl2:BBname;cont:synStm;gs,gs':globalstate)
 (evalExpBig mt gs s test (1) gs')->
 (InBBTable blocks lbl1 cont)->
 (evalMethod mt (blocks, cont) (addticks (1) gs') s pred)->
 (evalMethod mt (blocks, cond test lbl1 lbl2) gs s pred).
Intros.
EApply evalStepsIf.
Apply H.
EApply evalStep.
Apply evalIf1.
Apply H0.
Apply H1.
Save.

Lemma evalStepIf0 : (pred:resultprop;s:store;blocks:BBTable;test:synExp;lbl1,lbl2:BBname;cont:synStm;gs,gs':globalstate)
 (evalExpBig mt gs s test (0) gs')->
 (InBBTable blocks lbl2 cont)->
 (evalMethod mt (blocks, cont) (addticks (1) gs') s pred)->
 (evalMethod mt (blocks, cond test lbl1 lbl2) gs s pred).
Intros.
EApply evalStepsIf.
Apply H.
EApply evalStep.
Apply evalIf0.
Apply H0.
Apply H1.
Save.

Lemma evalStepsAssign :
  (pred:resultprop;s:store;blocks:BBTable;e:synExp;x:Var;cont:synStm;n:nat;gs,gs':globalstate)
  (evalExpBig mt gs s e n gs')->
  (evalMethod mt (blocks, assign x (value (const n)) cont) gs' s pred)->
  (evalMethod mt (blocks, assign x e cont) gs s pred).
Intros.
Induction H.
Assumption.
EApply evalStep.
Apply evalAssign'.
Apply H.
Apply HrecH.
Apply H0.
Save.

Lemma evalStepAssign :
 (pred:resultprop;s:store;blocks:BBTable;assignres:nat;exp:synExp;v:Var;cont:synStm;gs,gs':globalstate)
 (evalExpBig mt gs s exp assignres gs')->
 (evalMethod mt (blocks, cont) (addticks (1) gs') (update s v assignres) pred)->
 (evalMethod mt (blocks, assign v exp cont) gs s pred).
Intros.
EApply evalStepsAssign.
Apply H.
EApply evalStep.
Apply evalAssign.
Apply H0.
Save.

Lemma evalStepsAssignPred :
 (pred:resultprop;s:store;blocks:BBTable;assignpred:resultprop;exp:synExp;v:Var;cont:synStm;gs:globalstate)
 (evalExpBigPred mt gs s exp assignpred)->
 ((n:nat;gs':globalstate)(assignpred gs' n)->(evalMethod mt (blocks, cont) (addticks (1) gs') (update s v n) pred))->
 (evalMethod mt (blocks, assign v exp cont) gs s pred).
Intros.
Induction H.
EApply evalStepAssign. Apply evalExpFinal. Apply H0. Apply H.
EApply evalStep. Apply evalAssign'. Apply H. Apply HrecH. Assumption.
Save.

Lemma evalStepAssignConst :
 (pred:resultprop;s:store;v:Var;blocks:BBTable;c:nat;cont:synStm;gs:globalstate)
 (evalMethod mt (blocks, cont) (addticks (1) gs) (update s v c) pred)->
 (evalMethod mt (blocks,assign v (value (const c)) cont) gs s pred).
Intros. EApply evalStepAssign. Apply evalExpFinal. Apply H.
Save.

Lemma evalBinopLt1 : (s:store;n1,n2:nat;v1,v2:synVal;gs:globalstate)
                     (lt n1 n2) ->
                     (evalVal s v1)=(Some nat n1) -> (evalVal s v2)=(Some nat n2) ->
                     (evalExpBig mt gs s (binop synLt v1 v2) (1) (addticks (2) gs)).
Intros.
Apply evalBinop' with n1:=n1 n2:=n2. Simpl. Rewrite blt_nat_lt; Trivial.
Assumption. Assumption.
Save.

Lemma evalBinopLt0 : (s:store;n1,n2:nat;v1,v2:synVal;gs:globalstate)
                     (ge n1 n2) ->
                     (evalVal s v1)=(Some nat n1) -> (evalVal s v2)=(Some nat n2) ->
                     (evalExpBig mt gs s (binop synLt v1 v2) (0) (addticks (2) gs)).
Intros.
Apply evalBinop' with n1:=n1 n2:=n2. Simpl. Rewrite blt_nat_ge; Trivial.
Assumption. Assumption.
Save.

(* Why not?
Lemma evalInvokeMethod : (bb:BBTable;m:synStm;s,s':store;pred:nat->Prop)
 (evalMethod mt (bb,m) s' pred)->(evalExpBigPred mt s (execute bb s' m) pred).
Intros.
Induction H.
*)

Lemma evalInvokeMethod : (m:method;s,s':store;pred:resultprop;gs:globalstate)
 (evalMethod mt m gs s' pred)->(evalExpBigPred mt gs s (execute (Fst m) s' (Snd m)) pred).
Intros.
Induction H.
Simpl.
EApply evalExpPredStep.
Apply evalExecuteFinal. Apply H.
Apply evalExpPredFinal. Apply H0.
EApply evalExpPredStep.
Apply evalExecuteStep.
Apply H. Apply HrecH.
Save.

Lemma evalStepAssignInvoke :
 (pred:resultprop;s,s':store;blocks:BBTable;assignpred:resultprop;m:Mname;meth:method;v:Var;cont:synStm;args:(list synVal);gs:globalstate)
 (InMethTable mt m meth)->
 (mkStore s (0) args)=(Some store s')->
 (evalMethod mt meth (addticks (1) gs) s' assignpred)->
 ((n:nat;gs':globalstate)(assignpred gs' n)->(evalMethod mt (blocks, cont) (addticks (1) gs') (update s v n) pred))->
 (evalMethod mt (blocks, assign v (invoke m args) cont) gs s pred).
Intros.
EApply evalStep.
Apply evalAssign' with e':=(execute (Fst meth) s' (Snd meth)).
Apply evalInvoke. Assumption. Assumption.
EApply evalStepsAssignPred. Apply evalInvokeMethod. Apply H1. Apply H2.
Save.

Lemma strengthen : (m:method;s:store;P,Q:resultprop;gs:globalstate)
  ((x:nat;gs':globalstate)(P gs' x)->(Q gs' x))->(evalMethod mt m gs s P)->(evalMethod mt m gs s Q).
Intros. Induction H0.
EApply evalFinal. Apply H0. Apply H. Apply H1.
EApply evalStep. Apply H0. Apply HrecH0. Apply H.
Save.

Lemma addticks_combine : (gs:globalstate;i,j:nat)(addticks i (addticks j gs))=(addticks (plus i j) gs).
Intros. Unfold addticks. Simpl. Auto with arith.
Save.

Lemma ticks_addticks : (gs:globalstate;n:nat)(ticks (addticks n gs))=(plus n (ticks gs)).
Intros. Reflexivity.
Save.

End EvalLemmas.

Tactic Definition evalVal := Simpl; Reflexivity Orelse (Apply (lookup_find Var nat Var_eqdec beq_Var beq_Var_eq beq_Var_neq); FindInStore).
Tactic Definition evalBinop x y :=
  (Match Context With 
    [ |- (evalExpBig ? ? ? (binop ?1 ? ?) ?2 ?)] ->
      (Match (?1,?2) With [(synLt,(0))] -> Apply evalBinopLt0 with n1:=x n2:=y
                        | [(synLt,(1))] -> Apply evalBinopLt1 with n1:=x n2:=y
                        | [(?,?)] -> Apply evalBinop' with n1:=x n2:=y));
  [ Simpl; Reflexivity Orelse Assumption Orelse Auto with arith Orelse Idtac | evalVal | evalVal ].

Tactic Definition evalReturn x := Apply evalFinal with n:=x; [ evalVal | Idtac ].
Tactic Definition evalGoto := EApply evalStepGoto; [ FindInBBTable | Idtac ].
Tactic Definition evalGoto' H := EApply evalStepGoto; [ Apply H | Idtac ].
Tactic Definition evalIf1 := EApply evalStepIf1; [Idtac|Try FindInBBTable|Idtac].
Tactic Definition evalIf1' H := EApply evalStepIf1; [Idtac|Apply H|Idtac].
Tactic Definition evalIf0 := EApply evalStepIf0; [Idtac|Try FindInBBTable|Idtac].
Tactic Definition evalIf0' H := EApply evalStepIf0; [Idtac|Apply H|Idtac].
Tactic Definition evalAssign x := EApply evalStepAssign with assignres:=x;
                                  [ (Apply evalExpFinal)
                                    Orelse
                                    (EApply evalExpStep with e':=(value (const x)); [Apply evalVar;FindInStore|Apply evalExpFinal])
                                    Orelse
                                    Idtac
                                  | Idtac ].

Tactic Definition evalInvoke_1 have_meth s v0 meth_correct :=
  EApply evalStepAssignInvoke;
 [ Apply have_meth
 | Simpl;
   Rewrite (lookup_find_store s v#0 v0); [ Idtac | Try FindInStore ];
   Reflexivity
 | Apply meth_correct; Try FindInStore
 | Intros ].

Tactic Definition evalInvoke_2 have_meth s v0 v1 meth_correct :=
  EApply evalStepAssignInvoke;
 [ Apply have_meth
 | Simpl;
   Unfold get; Rewrite (lookup_find_store s v#0 v0); [ Idtac | Try FindInStore ];
   Unfold get; Rewrite (lookup_find_store s v#1 v1); [ Idtac | Try FindInStore ];
   Reflexivity
 | Apply meth_correct; Try FindInStore
 | Intros ].

Tactic Definition NormTicks := Unfold diffticks; Try (Repeat Rewrite addticks_combine; Try Rewrite ticks_addticks).
Tactic Definition NormTicksIn H := Unfold diffticks in H; Try (Repeat Rewrite addticks_combine in H; Try Rewrite ticks_addticks in H).

Tactic Definition SplitMany := (Split; [ SplitMany | SplitMany ]) Orelse Idtac.
Tactic Definition OpenRecord H := Repeat (Red in H); Decompose Record H; Clear H.
